//
// "$Id: cupsui.cpp,v 1.5 2005/06/17 20:54:21 mike Exp $"
//
//   UI class for the CUPS printer driver for Windows.
//
//   Copyright 2001-2005 by Easy Software Products.
//
//   These coded instructions, statements, and computer programs are the
//   property of Easy Software Products and are protected by Federal
//   copyright law.  Distribution and use rights are outlined in the file
//   "LICENSE.txt" which should have been included with this file.  If this
//   file is missing or damaged please contact Easy Software Products
//   at:
//
//       Attn: CUPS Licensing Information
//       Easy Software Products
//       44141 Airport View Drive, Suite 204
//       Hollywood, Maryland 20636 USA
//
//       Voice: (301) 373-9600
//       EMail: cups-info@cups.org
//         WWW: http://www.cups.org/
//
// Contents:
//
//   cupsUI::cupsUI()                 - Create a UI instance.
//   cupsUI::~cupsUI()                - Destroy a UI instance.
//   cupsUI::AddRef()                 - Increase the reference count.
//   cupsUI::Callback()               - Callback for our UI controls.
//   cupsUI::CommonUIProp()           - Manipulate the CUPS UI properties.
//   cupsUI::DeviceCapabilities()     - Query device capabilities.
//   cupsUI::DevicePropertySheets()   - Handle the device property sheet.
//   cupsUI::DevMode()                - Manipulate CUPS device mode data.
//   cupsUI::DevQueryPrintEx()        - See if a job is printable.
//   cupsUI::DocumentPropertySheets() - Handle the document property sheet.
//   cupsUI::DriverEvent()            - Handle driver events.
//   cupsUI::FontInstallerDlgProc()   - Provide font installation UI support.
//   cupsUI::GetInfo()                - Get driver information.
//   cupsUI::PrinterEvent()           - Handle printer events.
//   cupsUI::PublishDriverInterface() - Get the parent driver interface.
//   cupsUI::QueryColorProfile()      - Get the color profile associated with the printer.
//   cupsUI::QueryInterface()         - Return our UI interface.
//   cupsUI::Release()                - Decrease the reference count and delete as needed.
//   cupsUI::UpdateExternalFonts()    - Update the external fonts.
//   cupsUI::UpgradePrinter()         - Update device options in the registry.
//

//
// Include necessary headers...
//

#define CUPS_CLASS_PREFIX L"[cupsUI]"
#define CUPS_CLASS_NAME	cupsUI
#define CUPS_CLASS_ID	CLSID_OEMUI

#include "../common/driver.h"


class cupsUI : public IPrintOemUI	//// CUPS user interface module class
{
  protected:

  LONG			ref;		// Reference count
  IPrintOemDriverUI	*help;		// Pointer to driver help???

  public:

  STDMETHOD(QueryInterface)(THIS_ REFIID riid, LPVOID FAR *obj);
  STDMETHOD_(ULONG,AddRef)(THIS);
  STDMETHOD_(ULONG,Release)(THIS);

  STDMETHOD(PublishDriverInterface)(THIS_ IUnknown *pIUnknown);
  STDMETHOD(GetInfo)(THIS_ DWORD mode, PVOID buffer, DWORD size,
                     PDWORD needed);
  STDMETHOD(DevMode)(THIS_ DWORD mode, POEMDMPARAM param);
  STDMETHOD(CommonUIProp)(THIS_ DWORD mode, POEMCUIPPARAM param);
  STDMETHOD(DocumentPropertySheets)(THIS_ PPROPSHEETUI_INFO info, LPARAM param);
  STDMETHOD(DevicePropertySheets)(THIS_ PPROPSHEETUI_INFO info, LPARAM param);
  STDMETHOD(DevQueryPrintEx)(THIS_ POEMUIOBJ uiobj, PDEVQUERYPRINT_INFO info,
        		     PDEVMODE public_dm, PVOID cups_dm);
  STDMETHOD(DeviceCapabilities)(THIS_ POEMUIOBJ uiobj, HANDLE printer,
                                PWSTR device_name, WORD capability,
				PVOID output, PDEVMODE public_dm, PVOID cups_dm,   
				DWORD old, PDWORD result);
  STDMETHOD(UpgradePrinter)(THIS_ DWORD level, PBYTE info);
  STDMETHOD(PrinterEvent)(THIS_ PWSTR printer_name, INT driver_event,
        		  DWORD flags, LPARAM param);
  STDMETHOD(DriverEvent)(THIS_ DWORD driver_event, DWORD level,
        		 LPBYTE driver_info, LPARAM param);
  STDMETHOD(QueryColorProfile)(THIS_ HANDLE printer, POEMUIOBJ uiobj,
        		       PDEVMODE public_dm, PVOID cups_dm,
        		       ULONG reserved, VOID *profile_data,
        		       ULONG *cb_profile_data, FLONG *fl_profile_data);
  STDMETHOD(FontInstallerDlgProc)(THIS_ HWND window, UINT msg, WPARAM wparam,
        			  LPARAM lparam);
  STDMETHOD(UpdateExternalFonts)(THIS_ HANDLE printer, HANDLE heap,
        			 PWSTR cartridges);

  static LONG APIENTRY Callback(PCPSUICBPARAM pCallbackParam,
                                POEMCUIPPARAM pOEMUIParam);

  cupsUI();
  ~cupsUI();

};


#include "../common/driver.cpp"

//
// 'cupsUI::cupsUI()' - Create a UI instance.
//

cupsUI::cupsUI()
{
  cupsDebug(0, L"cupsUI::cupsUI()\n");

  // Initialize class data...
  ref  = 1;
  help = NULL;
}


//
// 'cupsUI::~cupsUI()' - Destroy a UI instance.
//

cupsUI::~cupsUI()
{
  cupsDebug(0, L"cupsUI::~cupsUI()\n");

  // Release helper interface...
  if (help)
  {
    help->Release();
    help = NULL;
  }
}


//
// 'cupsUI::AddRef()' - Increase the reference count.
//

ULONG __stdcall				// O - New reference count
cupsUI::AddRef()
{
  ULONG	temp;				// New reference count


  cupsDebug(4, L"cupsUI::AddRef()\n");

  temp = InterlockedIncrement(&ref);

  cupsDebug(-4, L"returning %ld...\n", ref);

  return (temp);
}


//
// 'cupsUI::Callback()' - Callback for our UI controls.
//

LONG APIENTRY				// O - Action taken
cupsUI::Callback(
    PCPSUICBPARAM data,			// I - Callback data
    POEMCUIPPARAM param)		// I - Parameter data
{
  cups_dev_t	*dm;			// CUPS device mode options


  cupsDebug(4, L"cupsUI::Callback(data=%p, param=%p)\n", data, param);

  dm = (cups_dev_t *)param->pOEMDM;

  switch (data->Reason)
  {
    case CPSUICB_REASON_APPLYNOW:
        // Copy options to device mode storage...
	lstrcpyW(dm->page_label, (LPWSTR)(param->pOEMOptItems[0].pSel));
	lstrcpyW(dm->job_billing, (LPWSTR)(param->pOEMOptItems[1].pSel));
        break;

    default:
        break;
  }

  cupsDebug(-4, L"returning CPSUICB_ACTION_NONE...\n");

  return (CPSUICB_ACTION_NONE);
}


//
// 'cupsUI::CommonUIProp()' - Manipulate the CUPS UI properties.
//

HRESULT __stdcall			// O - Status
cupsUI::CommonUIProp(
    DWORD         mode,			// I - Mode parameter
    POEMCUIPPARAM param)		// I - Parameter data
{
  cups_dev_t	*dm;			// CUPS device mode options
  POPTITEM	item;			// Current option item
  POPTTYPE	type;			// Option type
  POPTPARAM	tparam;			// Option type parameter


  cupsDebug(4, L"cupsUI::CommonUIProp(mode=%d, param=%p)\n", mode, param);

  // Only show a document properties UI...
  if (mode != OEMCUIP_DOCPROP)
  {
    cupsDebug(-4, L"returning E_NOTIMPL...\n");

    return (E_NOTIMPL);
  }

  // Make sure the items array is initialized...
  if (!param->pOEMOptItems)
  {
    // Return the number of options to create for the property UI...
    param->cOEMOptItems = 2;

    cupsDebug(-4, L"set cOEMOptItems to 2...\n");
    cupsDebug(-4, L"returning S_OK...\n");

    return (S_OK);
  }

  // Initialize the UI parameter data...
  cupsDebug(-4, L"initializing options...\n");

  param->OEMCUIPCallback = Callback;

  item = param->pOEMOptItems;
  dm   = (cups_dev_t *)(param->pOEMDM);

  // page-label option...
  memset(item, 0, sizeof(OPTITEM));

  item->cbSize     = sizeof(OPTITEM);
  item->DMPubID    = DMPUB_NONE;
  item->Level      = 1;
  item->Flags      = 0;
  item->pName      = L"Page Label";
  item->pSel       = (LPTSTR)HeapAlloc(param->hOEMHeap, HEAP_ZERO_MEMORY,
                                       sizeof(dm->page_label) /
                                       sizeof(dm->page_label[0]) *
                                       sizeof(WCHAR));
  type             =
  item->pOptType   = (POPTTYPE)HeapAlloc(param->hOEMHeap, HEAP_ZERO_MEMORY,
                                         sizeof(OPTTYPE));
  type->cbSize     = sizeof(OPTTYPE);
  type->Type       = TVOT_EDITBOX;
  type->Count      = 2;
  tparam           =
  type->pOptParam  = (POPTPARAM)HeapAlloc(param->hOEMHeap, HEAP_ZERO_MEMORY,
                                          2 * sizeof(OPTPARAM));
  tparam[0].cbSize = sizeof(OPTPARAM);
  tparam[1].cbSize = sizeof(OPTPARAM);
  tparam[1].IconID = sizeof(dm->page_label) / sizeof(dm->page_label[0]) - 1;

  lstrcpyW((LPWSTR)(item->pSel), dm->page_label);
  item ++;

  // job-billing option...
  memset(item, 0, sizeof(OPTITEM));

  item->cbSize     = sizeof(OPTITEM);
  item->DMPubID    = DMPUB_NONE;
  item->Level      = 1;
  item->Flags      = 0;
  item->pName      = L"Job Billing";
  item->pSel       = (LPTSTR)HeapAlloc(param->hOEMHeap, HEAP_ZERO_MEMORY,
                                       sizeof(dm->job_billing) /
                                       sizeof(dm->job_billing[0]) *
                                       sizeof(WCHAR));
  type             =
  item->pOptType   = (POPTTYPE)HeapAlloc(param->hOEMHeap, HEAP_ZERO_MEMORY,
                                         sizeof(OPTTYPE));
  type->cbSize     = sizeof(OPTTYPE);
  type->Type       = TVOT_EDITBOX;
  type->Count      = 2;
  tparam           =
  type->pOptParam  = (POPTPARAM)HeapAlloc(param->hOEMHeap, HEAP_ZERO_MEMORY,
                                          2 * sizeof(OPTPARAM));
  tparam[0].cbSize = sizeof(OPTPARAM);
  tparam[1].cbSize = sizeof(OPTPARAM);
  tparam[1].IconID = sizeof(dm->job_billing) / sizeof(dm->job_billing[0]) - 1;

  lstrcpyW((LPWSTR)(item->pSel), dm->job_billing);
  item ++;

  cupsDebug(-4, L"returning S_OK...\n");

  return (S_OK);
}


//
// 'cupsUI::DeviceCapabilities()' - Query device capabilities.
//

HRESULT __stdcall			// O - Status
cupsUI::DeviceCapabilities(POEMUIOBJ uiobj,
        		   HANDLE    printer,
        		   PWSTR     device_name,
        		   WORD      capability,
        		   PVOID     output,
        		   PDEVMODE  public_dm,
        		   PVOID     cups_cm,
        		   DWORD     old,
        		   DWORD     *result)
{
  cupsDebug(4, L"cupsUI::DeviceCapabilities()\n");
  cupsDebug(-4, L"returning E_NOTIMPL...\n");

  // NOT IMPLEMENTED
  return (E_NOTIMPL);
}


//
// 'cupsUI::DevicePropertySheets()' - Handle the device property sheet.
//

HRESULT __stdcall			// O - Status
cupsUI::DevicePropertySheets(
    PPROPSHEETUI_INFO info,		// I - Property sheet info
    LPARAM            param)		// I - Sheet parameter
{
  cupsDebug(4, L"cupsUI::DevicePropertySheets()\n");
  cupsDebug(-4, L"returning E_NOTIMPL...\n");

  // NOT IMPLEMENTED
  return (E_NOTIMPL);
}


//
// 'cupsUI::DevMode()' - Manipulate CUPS device mode data.
//

HRESULT __stdcall			// O - Status
cupsUI::DevMode(DWORD       mode,	// I - Mode parameter
                POEMDMPARAM param)	// I - Parameter data
{   
  HRESULT temp;				// Status


  cupsDebug(4, L"cupsUI::DevMode(mode=%d, param=%p)\n", mode, param);

  temp = cupsDevMode(mode, param);

  cupsDebug(-4, L"returning %d...\n", temp);

  return (temp);
}


//
// 'cupsUI::DevQueryPrintEx()' - See if a job is printable.
//

HRESULT __stdcall			// O - Status
cupsUI::DevQueryPrintEx(POEMUIOBJ           uiobj,
			PDEVQUERYPRINT_INFO pDQPInfo,
			PDEVMODE            public_dm,
			PVOID               cups_dm)
{
  cupsDebug(4, L"cupsUI::DevQueryPrintEx()\n");
  cupsDebug(-4, L"returning E_NOTIMPL...\n");

  // NOT IMPLEMENTED
  return (E_NOTIMPL);
}


//
// 'cupsUI::DocumentPropertySheets()' - Handle the document property sheet.
//

HRESULT __stdcall			// O - Status
cupsUI::DocumentPropertySheets(
    PPROPSHEETUI_INFO info,		// I - Property sheet info
    LPARAM            param)		// I - Sheet parameter
{
  cupsDebug(4, L"cupsUI::DocumentPropertySheets()\n");
  cupsDebug(-4, L"returning E_NOTIMPL...\n");

  // NOT IMPLEMENTED
  return (E_NOTIMPL);
}


//
// 'cupsUI::DriverEvent()' - Handle driver events.
//

HRESULT __stdcall			// O - Status
cupsUI::DriverEvent(DWORD  driver_event,
		    DWORD  level,
		    LPBYTE driver_info,
		    LPARAM param)
{
  cupsDebug(4, L"cupsUI::DriverEvent()\n");
  cupsDebug(-4, L"returning E_NOTIMPL...\n");

  // NOT IMPLEMENTED
  return (E_NOTIMPL);
}


//
// 'cupsUI::FontInstallerDlgProc()' - Provide font installation UI support.
//

HRESULT __stdcall			// O - Status
cupsUI::FontInstallerDlgProc(HWND   hWnd,
        		     UINT   usMsg,
        		     WPARAM wparam,
        		     LPARAM lparam) 
{
  cupsDebug(4, L"cupsUI::FontInstallerDlgProc()\n");
  cupsDebug(-4, L"returning E_NOTIMPL...\n");

  // NOT IMPLEMENTED
  return (E_NOTIMPL);
}


//
// 'cupsUI::GetInfo()' - Get driver information.
//

HRESULT __stdcall			// O - Status
cupsUI::GetInfo(DWORD  mode,		// I - Which info
		PVOID  buffer,		// I - Buffer
		DWORD  size,		// I - Size of buffer
		PDWORD needed)		// O - How much space is needed
{
  cupsDebug(4, L"cupsUI::GetInfo(mode=%d, buffer=%p, size=%d, needed=%p)\n",
            mode, buffer, size, needed);

  // Range check input
  if (!needed)
  {
    SetLastError(ERROR_INVALID_PARAMETER);

    cupsDebug(-4, L"returning E_FAIL (invalid parameter)...\n");

    return (E_FAIL);
  }

  if (mode != OEMGI_GETSIGNATURE && mode != OEMGI_GETVERSION)
  {
    SetLastError(ERROR_NOT_SUPPORTED);

    cupsDebug(-4, L"returning E_FAIL (not supported)...\n");

    return (E_FAIL);
  }

  // Set required buffer size and check if we have enough space...
  *needed = sizeof(DWORD);

  if (!buffer || size < sizeof(DWORD))
  {
    SetLastError(ERROR_INSUFFICIENT_BUFFER);

    cupsDebug(-4, L"returning E_FAIL (insufficient buffer)...\n");

    return (E_FAIL);
  }

  // Get the information requested...
  if (mode == OEMGI_GETSIGNATURE)
  {
    // Get the DLL signature...
    cupsDebug(0, L"OEMGI_GETSIGNATURE = %08x\n", CUPS_SIGNATURE);
    *(PDWORD)buffer = CUPS_SIGNATURE;
  }
  else
  {
    // Get the DLL version...
    cupsDebug(0, L"OEMGI_GETVERSION = %08x\n", CUPS_WINVERSION);
    *(PDWORD)buffer = CUPS_WINVERSION;
  }

  // Return with no errors...
  cupsDebug(-4, L"returning S_OK...\n");

  return (S_OK);
}


//
// 'cupsUI::PrinterEvent()' - Handle printer events.
//

HRESULT __stdcall			// O - Status
cupsUI::PrinterEvent(PWSTR  printer_name,
		     INT    driver_event,
		     DWORD  flags,
		     LPARAM param)
{
  cupsDebug(4, L"cupsUI::PrinterEvent()\n");
  cupsDebug(-4, L"returning E_NOTIMPL...\n");

  // NOT IMPLEMENTED
  return (E_NOTIMPL);
}


//
// 'cupsUI::PublishDriverInterface()' - Get the parent driver interface.
//

HRESULT __stdcall			// O - Status
cupsUI::PublishDriverInterface(IUnknown *pIUnknown)
					// I - Parent driver
{
  cupsDebug(4, L"cupsUI::PublishDriverInterface(pIUnknown=%p)\n", pIUnknown);

  if (!help)
  {
    // Get helper pointer...
    if (pIUnknown->QueryInterface(IID_IPrintOemDriverUI,
                                  (void **)&(help)) != S_OK)
    {
      // Failed, so force the helper pointer to NULL and return an error...
      help = NULL;

      cupsDebug(-4, L"returning E_FAIL...\n");

      return (E_FAIL);
    }
  }

  cupsDebug(-4, L"returning S_OK...\n");

  return (S_OK);
}


//
// 'cupsUI::QueryColorProfile()' - Get the color profile associated with the printer.
//

HRESULT __stdcall			// O - Status
cupsUI::QueryColorProfile(HANDLE    printer,
        		  POEMUIOBJ uiobj,
        		  PDEVMODE  public_dm,
        		  PVOID     cups_dm,
        		  ULONG     query_mode,
        		  VOID      *pvProfileData,
        		  ULONG     *pcbProfileData,
        		  FLONG     *pflProfileData)
{
  HRESULT result;


  cupsDebug(4, L"cupsUI::QueryColorProfile(printer=%d, uiobj=%p, public_dm=%p, "
            L"cups_dm=%p, query_mode=%lu, pvProfileData=%p, pcbProfileData=%p, "
            L"pflProfileData=%p)\n",
            printer, uiobj, public_dm, cups_dm, query_mode, pvProfileData,
            pcbProfileData, pflProfileData);

  // Range check input...
  if (!pcbProfileData)
  {
    SetLastError(ERROR_INVALID_PARAMETER);

    cupsDebug(-4, L"returning E_FAIL (invalid parameter)...\n");

    return (E_FAIL);
  }

  // Handle the query...
  if (query_mode == QCP_DEVICEPROFILE)
  {
    if (!pvProfileData)
    {
      *pcbProfileData = sizeof(CUPS_PROFILE);
      SetLastError(ERROR_INSUFFICIENT_BUFFER);

      cupsDebug(-4, L"returning E_FAIL (insufficient buffer)...\n");

      return (E_FAIL);
    }
    else if (*pcbProfileData < sizeof(CUPS_PROFILE))
    {
      *pcbProfileData = sizeof(CUPS_PROFILE);
      SetLastError(ERROR_INSUFFICIENT_BUFFER);

      cupsDebug(-4, L"returning E_FAIL (insufficient buffer)...\n");

      return (E_FAIL);
    }
    else
    {
      lstrcpynW((LPWSTR)pvProfileData, CUPS_PROFILE, *pcbProfileData);
      *pcbProfileData = sizeof(CUPS_PROFILE);
      *pflProfileData = QCP_PROFILEDISK;

      cupsDebug(-4, L"returning S_OK...\n");

      return (S_OK);
    }
  }
  else
  {
    cupsDebug(-4, L"returning E_FAIL...\n");

    return (E_FAIL);
  }
}


//
// 'cupsUI::QueryInterface()' - Return our UI interface.
//

HRESULT __stdcall			// O - S_OK on success, E_NOINTERFACE on error
cupsUI::QueryInterface(const IID &iid,	// I - Interface ID
                       void      **ppv)	// O - Pointer to class
{
  cupsDebug(4, L"cupsUI::QueryInterface(iid=%p, ppv=%p)\n", iid, ppv);

  if (iid == IID_IUnknown)
  {
    *ppv = static_cast<IUnknown*>(this);

    AddRef();

    cupsDebug(-4, L"returning IUnknown + S_OK...\n");

    return (S_OK);
  }
  else if (iid == IID_IPrintOemUI)
  {
    *ppv = static_cast<IPrintOemUI*>(this);

    AddRef();

    cupsDebug(-4, L"returning IPrintOemUI + S_OK...\n");

    return (S_OK);
  }
  else
  {
    *ppv = NULL;

    cupsDebug(-4, L"returning E_NOINTERFACE...\n");

    return (E_NOINTERFACE);
  }
}


//
// 'cupsUI::Release()' - Decrease the reference count and delete as needed.
//

ULONG __stdcall				// O - Reference count
cupsUI::Release() 
{
  cupsDebug(4, L"cupsUI::Release()\n");

  if (!InterlockedDecrement(&ref))
  {
    // Reference count is now 0, so delete it...
    delete this;

    cupsDebug(-4, L"returning 0...\n");

    return (0);
  }

  cupsDebug(-4, L"returning %ld...\n", ref);

  return (ref);
}


//
// 'cupsUI::UpdateExternalFonts()' - Update the external fonts.
//

HRESULT __stdcall			// O - Status
cupsUI::UpdateExternalFonts(HANDLE printer,
        		    HANDLE heap,
        		    PWSTR  cartridges)
{
  cupsDebug(4, L"cupsUI::UpdateExternalFonts()\n");
  cupsDebug(-4, L"returning E_NOIMPL...\n");

  // NOT IMPLEMENTED
  return (E_NOTIMPL);
}


//
// 'cupsUI::UpgradePrinter()' - Update device options in the registry.
//

HRESULT __stdcall			// O - Status
cupsUI::UpgradePrinter(DWORD level,
		       PBYTE pDriverUpgradeInfo)
{
  cupsDebug(4, L"cupsUI::UpgradePrinter()\n");
  cupsDebug(-4, L"returning E_NOIMPL...\n");

  // NOT IMPLEMENTED
  return (E_NOTIMPL);
}


//
// End of "$Id: cupsui.cpp,v 1.5 2005/06/17 20:54:21 mike Exp $".
//
