/*----------------------------------------------------------------------------

   libtunepimp -- The MusicBrainz tagging library.  
                  Let a thousand taggers bloom!
   
   Copyright (C) Robert Kaye 2003
   
   This file is part of libtunepimp.

   libtunepimp is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   libtunepimp is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with libtunepimp; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   $Id: filelookup.cpp,v 1.41 2005/06/21 03:50:52 robert Exp $

----------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "track.h"
#include "filelookup.h"
#include "tunepimp.h"
#include "metadata.h"

const int maxFieldLen = 1024;
const int maxIdLen = 64;
const int maxNumLen = 16;
const int maxNumTracks = 100;

int LookupFile::extractArtistList(musicbrainz_t o)
{
    TPArtistResult *artist;
    int             i, rel;
    char            temp[maxFieldLen], url[maxFieldLen];

    results.clear();
    for(i = 1;; i++)
    {
        mb_Select(o, MBS_Rewind);
        if (!mb_Select1(o, MBS_SelectLookupResult, i))
            break;

        artist = new TPArtistResult();
        rel = mb_GetResultInt(o, MBE_LookupGetRelevance);
        artist->setRelevance(rel);

        mb_Select(o, MBS_SelectLookupResultArtist);
        mb_GetResultData(o, MBE_ArtistGetArtistName, temp, maxFieldLen);
        artist->setName(temp);
        mb_GetResultData(o, MBE_ArtistGetArtistSortName, temp, maxFieldLen);
        artist->setSortName(temp);

        mb_GetResultData(o, MBE_ArtistGetArtistId, url, maxFieldLen);
        mb_GetIDFromURL(o, url, temp, maxIdLen);
        artist->setId(temp);

        results.push_back(artist);
    }

    return results.size();
}

int LookupFile::extractAlbumList(musicbrainz_t o)
{
    TPAlbumResult  *album;
    int             i, j, rel;
    char            temp[maxFieldLen], url[maxFieldLen];
    TPArtistResult  artist;

    results.clear();
    for(i = 1;; i++)
    {
        mb_Select(o, MBS_Rewind);
        if (!mb_Select1(o, MBS_SelectLookupResult, i))
            break;

        album = new TPAlbumResult();
        
        rel = mb_GetResultInt(o, MBE_LookupGetRelevance);
        album->setRelevance(rel);

        mb_Select(o, MBS_SelectLookupResultAlbum);

        mb_GetResultData(o, MBE_AlbumGetAlbumName, temp, maxFieldLen);
        album->setName(temp);
        album->setNonAlbum(strcmp(temp, TP_NONALBUMTRACKS_NAME) == 0);

        album->setNumCDIndexIds(mb_GetResultInt(o, MBE_AlbumGetNumCdindexIds));
        album->setNumTracks(mb_GetResultInt(o, MBE_AlbumGetNumTracks));

        mb_GetResultData(o, MBE_AlbumGetAlbumId, url, maxFieldLen);
        mb_GetIDFromURL(o, url, temp, maxIdLen);
        album->setId(temp);

        mb_GetResultData(o, MBE_AlbumGetAlbumArtistId, url, maxFieldLen);
        mb_GetIDFromURL(o, url, temp, maxIdLen);
        album->setVariousArtists(strcmp(MBI_VARIOUS_ARTIST_ID, temp) == 0);


        mb_GetResultData(o, MBE_AlbumGetAlbumType, url, maxFieldLen);
        mb_GetFragmentFromURL(o, url, temp, maxFieldLen);
        album->setType(convertToAlbumType(temp));

        mb_GetResultData(o, MBE_AlbumGetAlbumStatus, url, maxFieldLen);
        mb_GetFragmentFromURL(o, url, temp, maxFieldLen);
        album->setStatus(convertToAlbumStatus(temp));

        int numDates = mb_GetResultInt(o, MBE_AlbumGetNumReleaseDates);
        for(j = 1; j <= numDates; j++)
        {
            // Select the first release date
            if (mb_Select1(o, MBS_SelectReleaseDate, j))
            {
                // Pull back the release date and release country
                if (mb_GetResultData(o, MBE_ReleaseGetDate, temp, 256))
                {
                    int month = 0, day = 0, year = 0;

                    int nummatch = sscanf(temp, "%d-%d-%d", &year, &month, &day);
                    if (nummatch >= 1)
                    {
                        if (album->getReleaseYear() == 0 ||
                            album->getReleaseYear() > year ||
                           (album->getReleaseYear() == year && album->getReleaseMonth() > month) ||
                           (album->getReleaseYear() == year && album->getReleaseMonth() == month && album->getReleaseDay() > day))
                        {
                            album->setReleaseYear(year);
                            album->setReleaseMonth(month);
                            album->setReleaseDay(day);
                            mb_GetResultData(o, MBE_ReleaseGetCountry, temp, 256);
                            album->setReleaseCountry(temp);
                        }
                    }
                }
                mb_Select(o, MBS_Back);
            }
            else
                break;
        }

        // Extract the artist information
        mb_Select(o, MBS_SelectTrackArtist);

        mb_GetResultData(o, MBE_ArtistGetArtistId, url, maxFieldLen);
        mb_GetIDFromURL(o, url, temp, maxIdLen);
        artist.setId(string(temp));

        mb_GetResultData(o, MBE_ArtistGetArtistName, temp, maxFieldLen);
        artist.setName(string(temp));

        mb_GetResultData(o, MBE_ArtistGetArtistSortName, temp, maxFieldLen);
        artist.setSortName(string(temp));

        album->setArtist(artist);
        results.push_back(album);
    }

    return results.size();
}

int LookupFile::extractTrackList(musicbrainz_t o)
{
    TPAlbumTrackResult *albumTrack;
    TPArtistResult      artist;
    TPAlbumResult       album;
    int                 i, j, rel, trackNum;
    char                temp[maxFieldLen], url[maxFieldLen], trackURI[maxFieldLen];

    for(i = 1;; i++)
    {
        mb_Select(o, MBS_Rewind);
        if (!mb_Select1(o, MBS_SelectLookupResult, i))
            break;

        rel = mb_GetResultInt(o, MBE_LookupGetRelevance);

        // -------------------------------------------------------------
        // Get the track info
        // -------------------------------------------------------------
        albumTrack = new TPAlbumTrackResult();
        albumTrack->setRelevance(rel);

        mb_Select(o, MBS_SelectLookupResultTrack);
        mb_GetResultData(o, MBE_TrackGetTrackName, temp, maxFieldLen);
        albumTrack->setName(temp);

        mb_GetResultData(o, MBE_TrackGetTrackId, trackURI, maxFieldLen);
        mb_GetIDFromURL(o, trackURI, temp, maxIdLen);
        albumTrack->setId(temp);
        albumTrack->setDuration(mb_GetResultInt(o, MBE_TrackGetTrackDuration));
        albumTrack->setNumTRMIds(mb_GetResultInt(o, MBE_GetNumTrmids));

        // -------------------------------------------------------------
        // Extract the artist information
        // -------------------------------------------------------------
        mb_Select(o, MBS_SelectTrackArtist);

        mb_GetResultData(o, MBE_ArtistGetArtistId, url, maxFieldLen);
        mb_GetIDFromURL(o, url, temp, maxIdLen);
        artist.setId(string(temp));

        mb_GetResultData(o, MBE_ArtistGetArtistName, temp, maxFieldLen);
        artist.setName(string(temp));

        mb_GetResultData(o, MBE_ArtistGetArtistSortName, temp, maxFieldLen);
        artist.setSortName(string(temp));
        album.setArtist(artist);

        // -------------------------------------------------------------
        // Get the album info
        // -------------------------------------------------------------
        mb_Select(o, MBS_Rewind);
        mb_Select1(o, MBS_SelectLookupResult, i);
        mb_Select(o, MBS_SelectLookupResultAlbum);

        mb_GetResultData(o, MBE_AlbumGetAlbumName, temp, maxFieldLen);
        album.setName(temp);
        album.setNonAlbum(strcmp(temp, TP_NONALBUMTRACKS_NAME) == 0);

        mb_GetResultData(o, MBE_AlbumGetAlbumType, url, maxFieldLen);
        mb_GetFragmentFromURL(o, url, temp, 256);
        album.setType(convertToAlbumType(temp));

        mb_GetResultData(o, MBE_AlbumGetAlbumStatus, url, maxFieldLen);
        mb_GetFragmentFromURL(o, url, temp, 256);
        album.setStatus(convertToAlbumStatus(temp));

        int numDates = mb_GetResultInt(o, MBE_AlbumGetNumReleaseDates);
        for(j = 1; j <= numDates; j++)
        {
            // Select the first release date
            if (mb_Select1(o, MBS_SelectReleaseDate, j))
            {
                // Pull back the release date and release country
                if (mb_GetResultData(o, MBE_ReleaseGetDate, temp, 256))
                {
                    int month = 0, day = 0, year = 0;

                    int nummatch = sscanf(temp, "%d-%d-%d", &year, &month, &day);
                    if (nummatch >= 1)
                    {
                        if (album.getReleaseYear() == 0 ||
                            album.getReleaseYear() > year ||
                           (album.getReleaseYear() == year && album.getReleaseMonth() > month) ||
                           (album.getReleaseYear() == year && album.getReleaseMonth() == month && album.getReleaseDay() > day))
                        {
                            album.setReleaseYear(year);
                            album.setReleaseMonth(month);
                            album.setReleaseDay(day);
                            mb_GetResultData(o, MBE_ReleaseGetCountry, temp, 256);
                            album.setReleaseCountry(temp);
                        }
                    }
                }
                mb_Select(o, MBS_Back);
            }
            else
                break;
        }

        album.setNumCDIndexIds(mb_GetResultInt(o, MBE_AlbumGetNumCdindexIds));
        album.setNumTracks(mb_GetResultInt(o, MBE_AlbumGetNumTracks));

        mb_GetResultData(o, MBE_AlbumGetAlbumId, url, maxFieldLen);
        mb_GetIDFromURL(o, url, temp, maxIdLen);
        album.setId(temp);

        mb_GetResultData(o, MBE_AlbumGetAlbumArtistId, url, maxFieldLen);
        mb_GetIDFromURL(o, url, temp, maxIdLen);
        album.setVariousArtists(strcmp(MBI_VARIOUS_ARTIST_ID, temp) == 0);

        // Extract the track number
        trackNum = mb_GetOrdinalFromList(o, MBE_AlbumGetTrackList, trackURI);
        if (trackNum > 0 && trackNum < maxNumTracks)
           albumTrack->setTrackNum(trackNum);

        albumTrack->setArtist(artist);
        albumTrack->setAlbum(album);
        results.push_back(albumTrack);
    }

    return results.size();
}

void LookupFile::extractMatch(musicbrainz_t o)
{
    char name[maxFieldLen], id[maxIdLen], trackURI[maxFieldLen];
    int trackNum, duration;

    mb_Select(o, MBS_Rewind);

    mb_Select(o, MBS_SelectLookupResultArtist);
    mb_GetResultData(o, MBE_ArtistGetArtistName, name, maxFieldLen);
    data.artist = name;
    mb_GetResultData(o, MBE_ArtistGetArtistSortName, name, maxFieldLen);
    data.sortName = name;

    mb_GetResultData(o, MBE_ArtistGetArtistId, name, maxFieldLen);
    mb_GetIDFromURL(o, name, id, maxIdLen);
    data.artistId = id;
    
    mb_Select(o, MBS_Rewind);

    mb_Select(o, MBS_SelectLookupResultTrack);
    mb_GetResultData(o, MBE_TrackGetTrackName, name, maxFieldLen);
    data.track = name;

    mb_GetResultData(o, MBE_TrackGetTrackId, name, maxFieldLen);
    strcpy(trackURI, name);
    mb_GetIDFromURL(o, name, id, maxIdLen);
    data.trackId = id;

    duration = mb_GetResultInt(o, MBE_TrackGetTrackDuration);
    if (duration > 0)
        data.duration = duration;

    mb_Select(o, MBS_Rewind);

    mb_Select(o, MBS_SelectLookupResultAlbum);
    mb_GetResultData(o, MBE_AlbumGetAlbumId, name, maxFieldLen);
    mb_GetIDFromURL(o, name, id, maxIdLen);
    data.albumId = id;

    mb_GetResultData(o, MBE_AlbumGetAlbumName, name, maxFieldLen);
    data.album = name;

    // Extract the track number
    trackNum = mb_GetOrdinalFromList(o, MBE_AlbumGetTrackList, trackURI);
    if (trackNum > 0 && trackNum < maxNumTracks)
       data.trackNum = trackNum;
}

LookupStatus LookupFile::lookup(void)
{
    musicbrainz_t   o;
    Metadata        mdata;
    int             ret;
    char            trackNum[maxNumLen], duration[maxNumLen], temp[maxFieldLen], url[maxFieldLen];
    const char     *args[15];
    LookupStatus    status = eLookupError;

    // Nuke any old results
    results.clear();

    o = mb_New();
    mb_UseUTF8(o, 1);

    if (proxyServer.size() > 0 && proxyPort != 0)
        mb_SetProxy(o, (char *)proxyServer.c_str(), proxyPort);

    if (server.size() > 0 && port != 0)
        mb_SetServer(o, (char *)server.c_str(), port);

    mb_SetDebug(o, (int)pimp->context.getDebug());

    args[0] = trm.c_str();
    args[1] = data.artist.c_str();
    args[2] = data.album.c_str();
    args[3] = data.track.c_str();
    sprintf(trackNum, "%d", data.trackNum);
    args[4] = trackNum;
    sprintf(duration, "%ld", data.duration);
    args[5] = duration;
    args[6] = fileName.c_str();
    args[7] = data.artistId.c_str();
    args[8] = data.albumId.c_str();
    args[9] = data.trackId.c_str();
    args[10] = NULL;

    ret = mb_QueryWithArgs(o, MBQ_FileInfoLookup, (char **)args);
    if (!ret)
    {
        char error[maxFieldLen];

        mb_GetQueryError(o, error, maxFieldLen);
        err = string(error);
        status = eLookupError;
    }

    // If there is no further result, then bail.
    if (!mb_Select1(o, MBS_SelectLookupResult, 1))
    {
        extractMatch(o);
        status = eFound;
    }
    else
    {
        if (!mb_GetResultData(o, MBE_LookupGetType, url, maxFieldLen))
        {
            err= "Could not determine the type of lookup result.\n";
            status = eLookupError;
        }  
        else
        {
            mb_GetFragmentFromURL(o, url, temp, maxFieldLen);
            if (strcmp(temp, "ArtistResult") == 0)
            {
                extractArtistList(o);
                type = eArtistList;
                status = eFound;
            }
            else
            if (strcmp(temp, "AlbumResult") == 0)
            {
                extractAlbumList(o);
                type = eAlbumList;
                status = eFound;
            }
            else
            if (strcmp(temp, "AlbumTrackResult") == 0)
            {
                extractTrackList(o);
                type = eTrackList;
                status = eFound;
            }
            else
            {
                err= "Unknown query response.";
                status = eLookupError;
            }
        }
    }

    mb_Delete(o);

    return status;
}

FileLookupThread::FileLookupThread(TunePimp   *tunePimpArg,
                                   FileCache  *cacheArg) : Thread()
{
    tunePimp = tunePimpArg;
    cache = cacheArg;
    exitThread = false;
    sem = new Semaphore();
}

FileLookupThread::~FileLookupThread(void)
{
    exitThread = true;
    sem->signal();
    join();
    delete sem;
}

void FileLookupThread::wake(void)
{
    sem->signal();
}

void FileLookupThread::threadMain(void)
{
    string        fileName, status, trm;
    Track        *track;
    Metadata      data;
    LookupStatus  ret;

    for(; !exitThread;)
    {
        track = cache->getNextItem(eFileLookup);
        if (track == NULL)
        {
           sem->wait();
           continue;
        }

        track->lock();
        track->getFileName(fileName);
        status = "Looking up " + fileName;
        tunePimp->setStatus(status);

        LookupFile lookupFile(tunePimp);

        track->getServerMetadata(data);
        if (data.isEmpty())
            track->getLocalMetadata(data);

        track->getTRM(trm);
        lookupFile.setArgs(trm, fileName, data);

        track->unlock();
        ret = lookupFile.lookup();
        track->lock();

		if (track->getStatus() == eFileLookup)
		{
			if (ret == eLookupError)
			{
				string err;

				track->setStatus(eError);
				lookupFile.getError(err);
				track->setError(err);
			}
			else
			{
				TPResultType         type;
				vector<TPResult *>   result;
				Metadata             data;

				type = lookupFile.getResultType();
				lookupFile.getResults(result, data);

				track->setResults(type, result);
				if (type == eMatchedTrack)
				{
					track->setServerMetadata(data);
					track->setStatus(eRecognized);
				}
				else
				{
				    // If no items were returned, we need to prod the user for more data
				    if (result.size() == 0)
     					track->setStatus(eUnrecognized);
				    else
     					track->setStatus(eUserSelection);
				}
			}
		}
        track->unlock();

        tunePimp->wake(track);
        cache->release(track);
    }
}

