/*
 * Copyright (c) 2003 Benedikt Meurer (benedikt.meurer@unix-ag.uni-siegen.de)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif	/* !HAVE_CONFIG_H */

#include <gtk/gtk.h>

#include <libxfce4util/i18n.h>

#include "sheets_page.h"

static void sheets_page_class_init(SheetsPageClass *);
static void sheets_page_init(SheetsPage *);
static void sheets_page_finalize(GObject *);

static GtkWidgetClass *parent_class = NULL;

GtkType
sheets_page_get_type(void)
{
	static GtkType sheets_page_type = 0;

	if (!sheets_page_type) {
		static const GTypeInfo sheets_page_info = {
			sizeof(SheetsPageClass),
			NULL,
			NULL,
			(GClassInitFunc)sheets_page_class_init,
			NULL,
			NULL,
			sizeof(SheetsPage),
			0,
			(GInstanceInitFunc)sheets_page_init
		};

		sheets_page_type = g_type_register_static(GTK_TYPE_TABLE,
			"SheetsPage", &sheets_page_info, 0);
	}

	return(sheets_page_type);
}

static void
sheets_page_class_init(SheetsPageClass *klass)
{
	GObjectClass *gobject_class;

	gobject_class = G_OBJECT_CLASS(klass);
	gobject_class->finalize = sheets_page_finalize;

	parent_class = gtk_type_class(gtk_table_get_type());
}

static void
sheets_page_init(SheetsPage *pg)
{
	GtkWidget *label, *hbox;
	GtkSizeGroup *sg;

	gtk_container_set_border_width(GTK_CONTAINER(pg), 6);
	gtk_table_resize(GTK_TABLE(pg), 6, 2);
	gtk_table_set_col_spacings(GTK_TABLE(pg), 6);
	gtk_table_set_row_spacings(GTK_TABLE(pg), 4);
	gtk_table_set_homogeneous(GTK_TABLE(pg), FALSE);

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(hbox);
	gtk_table_attach(GTK_TABLE(pg), hbox, 0, 2, 0, 1,
		GTK_FILL, 0, 0, 0);
	
	label = gtk_label_new(_("Papersize:"));
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

	pg->papersize = gtk_combo_new();
	gtk_widget_show(pg->papersize);
	gtk_box_pack_start(GTK_BOX(hbox), pg->papersize, TRUE, TRUE, 0);

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(hbox);
	gtk_table_attach(GTK_TABLE(pg), hbox, 0, 2, 1, 2,
		GTK_FILL, 0, 0, 0);
	
	pg->portrait = gtk_radio_button_new_with_label(NULL, _("Portrait"));
	gtk_widget_show(pg->portrait);
	gtk_box_pack_start(GTK_BOX(hbox), pg->portrait, FALSE, FALSE, 0);

	pg->landscape = gtk_radio_button_new_with_label_from_widget(
		GTK_RADIO_BUTTON(pg->portrait), _("Landscape"));
	gtk_widget_show(pg->landscape);
	gtk_box_pack_start(GTK_BOX(hbox), pg->landscape, FALSE, FALSE, 0);

	/* empty row */
	label = gtk_label_new(" ");
	gtk_widget_show(label);
	gtk_table_attach(GTK_TABLE(pg), label, 0, 1, 2, 3,
		GTK_FILL, 0, 0, 0);
		
	sg = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);
	
	hbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(hbox);
	gtk_table_attach(GTK_TABLE(pg), hbox, 0, 1, 3, 4,
		GTK_FILL, 0, 0, 0);
	
	label = gtk_label_new(_("Columns:"));
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_size_group_add_widget(sg, label);

	pg->cols = gtk_spin_button_new_with_range(1.0, 10.0, 1.0);
	gtk_widget_show(pg->cols);
	gtk_box_pack_start(GTK_BOX(hbox), pg->cols, FALSE, FALSE, 0);

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(hbox);
	gtk_table_attach(GTK_TABLE(pg), hbox, 1, 2, 3, 4,
		GTK_FILL, 0, 0, 0);
	
	pg->fillcols = gtk_radio_button_new_with_label(NULL,
		_("Fill columns first"));
	gtk_widget_show(pg->fillcols);
	gtk_box_pack_start(GTK_BOX(hbox), pg->fillcols, FALSE, FALSE, 0);

	pg->fillrows = gtk_radio_button_new_with_label_from_widget(
		GTK_RADIO_BUTTON(pg->fillcols), _("Fill rows first"));
	gtk_widget_show(pg->fillrows);
	gtk_box_pack_start(GTK_BOX(hbox), pg->fillrows, FALSE, FALSE, 0);

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(hbox);
	gtk_table_attach(GTK_TABLE(pg), hbox, 0, 1, 4, 5,
		GTK_FILL, 0, 0, 0);
	
	label = gtk_label_new(_("Rows:"));
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_size_group_add_widget(sg, label);

	pg->rows = gtk_spin_button_new_with_range(1.0, 10.0, 1.0);
	gtk_widget_show(pg->rows);
	gtk_box_pack_start(GTK_BOX(hbox), pg->rows, FALSE, FALSE, 0);

	pg->borders = gtk_check_button_new_with_label(
		_("Print borders around columns"));
	gtk_widget_show(pg->borders);
	gtk_table_attach(GTK_TABLE(pg), pg->borders, 1, 2, 4, 5,
		GTK_EXPAND | GTK_FILL, 0, 0, 0);

	/* empty row */
	label = gtk_label_new(" ");
	gtk_widget_show(label);
	gtk_table_attach(GTK_TABLE(pg), label, 0, 1, 5, 6,
		GTK_FILL, 0, 0, 0);
		
	pg->reverse = gtk_check_button_new_with_label(
		_("Print pages in reverse order"));
	gtk_widget_show(pg->reverse);
	gtk_table_attach(GTK_TABLE(pg), pg->reverse, 0, 2, 6, 7,
		GTK_EXPAND | GTK_FILL, 0, 0, 0);
}

static void
sheets_page_finalize(GObject *object)
{
	g_return_if_fail(object != NULL);
	g_return_if_fail(SHEETS_IS_PAGE(object));
	
	G_OBJECT_CLASS(parent_class)->finalize(object);
}

GtkWidget *
sheets_page_new(void)
{
	SheetsPage *pg;
	GList *list;

	pg = SHEETS_PAGE(g_object_new(sheets_page_get_type(), NULL));

	if ((list = xfce_option_list(papersizes)) != NULL) {
		gtk_combo_set_popdown_strings(GTK_COMBO(pg->papersize),list);
		g_list_free(list);
	}

	return(GTK_WIDGET(pg));
}

void
sheets_page_set_settings(SheetsPage *pg, const XfPrintSettingsSheets *sheets)
{
	g_return_if_fail(pg != NULL);
	g_return_if_fail(sheets != NULL);

	if (sheets->landscape) {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pg->landscape),
			TRUE);
	}
	else {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pg->portrait),
			TRUE);
	}

	if (sheets->fillcols) {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pg->fillcols),
			TRUE);
	}
	else {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pg->fillrows),
			TRUE);
	}

	gtk_spin_button_set_value(GTK_SPIN_BUTTON(pg->cols), sheets->cols);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(pg->rows), sheets->rows);

	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(pg->papersize)->entry),
		xfce_option_alias(papersizes, sheets->papersize));

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pg->borders),
		sheets->borders);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pg->reverse),
		sheets->reverse);
}

void
sheets_page_get_settings(SheetsPage *pg, XfPrintSettingsSheets *sheets)
{
	g_return_if_fail(pg != NULL);
	g_return_if_fail(sheets != NULL);

	sheets->landscape = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
			pg->landscape));
	sheets->fillcols = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
			pg->fillcols));
	sheets->cols = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(
			pg->cols));
	sheets->rows = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(
			pg->rows));
	sheets->papersize = xfce_option_name(papersizes, gtk_entry_get_text(
			GTK_ENTRY(GTK_COMBO(pg->papersize)->entry)));
	sheets->borders = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
			pg->borders));
	sheets->reverse = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
			pg->reverse));
}

