#include <libxfcegui4/libxfcegui4.h>
#include <libxfcegui4/netk-screen.h>
#include <libxfcegui4/netk-window.h>
#include <libxfce4util/debug.h>

#include "ls_icon.h"
#include "ls_iconbox.h"

struct _LSIconBoxPrivate {
  NetkScreen 		*pScreen;
  GtkHBox     		*pHBox;
  GtkVBox		*pVBox;
  glong			lOrientation;
  glong			lSize;
  GHashTable		*pghtXIDIcon;
  gboolean		bUseColor;
  gint			nVisibleChilds;
  gboolean		bHideOnEmpty;
  
  struct _LSIconPrivateDisplay
  {
    LSDisplayIcon       sameWorkspace;
    LSDisplayIcon       otherWorkspace;
  } display;
  
  struct _LSIconBoxPrivateSignals {
    gulong		window_closed;
    gulong		window_opened;
  } signals;
};

static void 	ls_iconbox_init(LSIconBox *plsi);
static void 	ls_iconbox_class_init(LSIconBoxClass *plsic);
static void 	ls_iconbox_finalize(GObject *pgo);
static void	ls_iconbox_re_evaluate_visibility(LSIconBox *plsib);
/* CALLBACKS									*/
static void	cb_lsib_window_closed(NetkScreen *pScreen, 
    		  NetkWindow *pWin, void *pVoid);
static void	cb_lsib_window_opened(NetkScreen *pScreen, 
    		  NetkWindow *pWin, void *pVoid);
static void	cb_lsib_xid_destroy(gpointer lXID);
static void     cb_lsib_icon_hide(GtkWidget *pWidget, gpointer pUserData);
static void	cb_lsib_icon_show(GtkWidget *pWidget, gpointer pUserData);

static gpointer parent_class;

/* from nek-xutils.h								*/
static int xid_equal(gconstpointer v1, gconstpointer v2)
{
    return *((const gulong *)v1) == *((const gulong *)v2);
}

static guint xid_hash(gconstpointer v)
{
    gulong val = *(const gulong *)v;

    /* I'm not sure this works so well. */
#if G_SIZEOF_LONG > 4
    return (guint) (val ^ (val >> 32));
#else
    return val;
#endif
}

/********************************************************************************/
/* GLIB OBJECT FUNCTIONS							*/
/********************************************************************************/
GType ls_iconbox_get_type()
{
  static GType ls_iconbox_type = 0;

  if(!ls_iconbox_type)
  {
    static const GTypeInfo ls_iconbox_info = {
      sizeof(LSIconBoxClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc)ls_iconbox_class_init,
      NULL,				/* class_finalize */
      NULL,				/* class_data */
      sizeof(LSIconBox),
      0,				/* n_preallocs */
      (GInstanceInitFunc)ls_iconbox_init
    };

    ls_iconbox_type = g_type_register_static(GTK_TYPE_EVENT_BOX, "LSIconBox", 
      &ls_iconbox_info, 0);
  }

  return ls_iconbox_type;
}

static void ls_iconbox_init(LSIconBox *plsi) 
{
}

static void ls_iconbox_class_init(LSIconBoxClass *plsic) 
{
  GObjectClass *object_class = G_OBJECT_CLASS(plsic);
  parent_class = g_type_class_peek_parent(plsic);
  object_class->finalize = ls_iconbox_finalize;
}

static void ls_iconbox_finalize(GObject *pgo) 
{
  LSIconBox *plsib;
  
  g_return_if_fail(LS_IS_ICONBOX(pgo));
 
  plsib = LS_ICONBOX(pgo);

  g_signal_handler_disconnect(G_OBJECT(plsib->priv->pScreen),
    plsib->priv->signals.window_closed);
  g_signal_handler_disconnect(G_OBJECT(plsib->priv->pScreen),
    plsib->priv->signals.window_opened);

  if(plsib->priv->lOrientation == LS_ICONBOX_HORIZONTAL)
  {
    g_object_unref(plsib->priv->pVBox);
  }
  else
  {
    g_object_unref(plsib->priv->pHBox);
  }

  g_hash_table_destroy(plsib->priv->pghtXIDIcon);

  G_OBJECT_CLASS(parent_class)->finalize(pgo);
  
  return;
}

/********************************************************************************/
/* CONSTRUCTORS									*/
/********************************************************************************/
LSIconBox *ls_iconbox_new(NetkScreen *pScreen, glong lOrientation) 
{
  LSIconBox	*plsib;
  GList		*pglWin;
  
  plsib 			= LS_ICONBOX(g_object_new(LS_TYPE_ICONBOX, NULL)); 
  plsib->priv 			= (LSIconBoxPrivate *)
    				    g_malloc(sizeof(LSIconBoxPrivate));
  plsib->priv->pScreen 		= pScreen;
  plsib->priv->pVBox		= GTK_VBOX(gtk_vbox_new(TRUE, 0));
  plsib->priv->pHBox		= GTK_HBOX(gtk_hbox_new(TRUE, 0));
  plsib->priv->lOrientation	= lOrientation;
  plsib->priv->pghtXIDIcon	= g_hash_table_new_full(xid_hash, 
      				    xid_equal, cb_lsib_xid_destroy, NULL);
  plsib->priv->nVisibleChilds	= 0;
  plsib->priv->bHideOnEmpty	= TRUE;
  /* default settings								*/
  plsib->priv->bUseColor	= FALSE;
  plsib->priv->display.otherWorkspace.minimized		= TRUE;
  plsib->priv->display.otherWorkspace.notMinimized	= TRUE;
  plsib->priv->display.otherWorkspace.skipTasklist	= FALSE;
  plsib->priv->display.sameWorkspace.minimized		= TRUE;
  plsib->priv->display.sameWorkspace.notMinimized	= TRUE;
  plsib->priv->display.sameWorkspace.skipTasklist	= FALSE;
  
  gtk_container_set_border_width(GTK_CONTAINER(plsib), 0);
  gtk_container_set_border_width(GTK_CONTAINER(plsib->priv->pHBox), 0);
  gtk_container_set_border_width(GTK_CONTAINER(plsib->priv->pVBox), 0);

  if(plsib->priv->lOrientation == LS_ICONBOX_HORIZONTAL)
  {
    gtk_container_add(GTK_CONTAINER(plsib), GTK_WIDGET(plsib->priv->pHBox));
    ls_iconbox_set_size(plsib, LS_TINY);
    gtk_widget_show(GTK_WIDGET(plsib->priv->pHBox));
    g_object_ref(plsib->priv->pVBox);
  }
  else
  {
    gtk_container_add(GTK_CONTAINER(plsib), GTK_WIDGET(plsib->priv->pVBox));
    ls_iconbox_set_size(plsib, LS_TINY);
    gtk_widget_show(GTK_WIDGET(plsib->priv->pVBox));
    g_object_ref(plsib->priv->pHBox);
  }
  
  plsib->priv->signals.window_closed =
    g_signal_connect_object (G_OBJECT(plsib->priv->pScreen), "window_closed", 
    G_CALLBACK(cb_lsib_window_closed), plsib, 0);
  plsib->priv->signals.window_opened =
    g_signal_connect_object (G_OBJECT(plsib->priv->pScreen), "window_opened", 
    G_CALLBACK(cb_lsib_window_opened), plsib, 0);

  pglWin = netk_screen_get_windows (pScreen);
  while (pglWin != NULL) {
    cb_lsib_window_opened(pScreen, pglWin->data, plsib);
    pglWin = pglWin->next;
  }
  g_list_free(pglWin);

  return plsib;
}

/*******************************************************************************/
/* FUNCTIONS                                                                   */
/*******************************************************************************/
void ls_iconbox_set_hide_on_empty(LSIconBox *plsib, gboolean bHide)
{
  if(plsib->priv->bHideOnEmpty==bHide)
    return;

  plsib->priv->bHideOnEmpty=bHide;
  ls_iconbox_re_evaluate_visibility(plsib);
}

GList *ls_iconbox_get_children(LSIconBox *plsib)
{
  g_return_val_if_fail(plsib!=NULL, (GList *)NULL);
  g_return_val_if_fail(LS_IS_ICONBOX(plsib), (GList *)NULL);

  if(plsib->priv->lOrientation == LS_ICONBOX_HORIZONTAL)
  {
    return gtk_container_get_children(GTK_CONTAINER(plsib->priv->pHBox));
  }
  else
  {
    return gtk_container_get_children(GTK_CONTAINER(plsib->priv->pVBox));
  }
}

void ls_iconbox_set_size(LSIconBox *plsib, glong lSize)
{
  /* GList *pgl; */
  
  plsib->priv->lSize = lSize;
  
  if(plsib->priv->lOrientation == LS_ICONBOX_HORIZONTAL)
  {
    gtk_widget_set_size_request(GTK_WIDGET(plsib), 
      -1, lsIconSize[lSize] + 2*lsBorderWidth);
    gtk_widget_set_size_request(GTK_WIDGET(plsib->priv->pHBox), 
      -1, lsIconSize[lSize] + 2*lsBorderWidth);
    /*
      pgl = gtk_container_get_children(GTK_CONTAINER(plsib->priv->pHBox));
     */
  }
  else
  {
    gtk_widget_set_size_request(GTK_WIDGET(plsib), 
      lsIconSize[lSize] + 2*lsBorderWidth, -1);
    gtk_widget_set_size_request(GTK_WIDGET(plsib->priv->pVBox), 
      lsIconSize[lSize] + 2*lsBorderWidth, -1);
    /* 
      pgl = gtk_container_get_children(GTK_CONTAINER(plsib->priv->pHBox));
     */
  }

  /*
  while(pgl)
  {
    if(LS_IS_ICON(pgl->data))
      gtk_widget_set_size_request(GTK_WIDGET(pgl->data), lsIconSize[lSize], 
        lsIconSize[lSize]);
    pgl = pgl->next;
  }
  */
}

void ls_iconbox_add_icon(LSIconBox *plsib, LSIcon *plsi)
{
  /* add ref, should be removed in ls_iconbox_remove_icon			*/
  g_object_ref(plsi);

  if(plsib->priv->lOrientation == LS_ICONBOX_HORIZONTAL) 
  {
    gtk_box_pack_start(GTK_BOX(plsib->priv->pHBox), 
      GTK_WIDGET(plsi), FALSE, FALSE, 0);
  }
  else
  {
    gtk_box_pack_start(GTK_BOX(plsib->priv->pVBox), 
      GTK_WIDGET(plsi), FALSE, FALSE, 0);
  }

  if(plsib->priv->lOrientation == LS_ICONBOX_HORIZONTAL) 
  {
    gtk_widget_queue_resize(GTK_WIDGET(plsib->priv->pHBox));
  }
  else
  {
    gtk_widget_queue_resize(GTK_WIDGET(plsib->priv->pVBox));
  }
  
  ls_iconbox_re_evaluate_visibility(plsib); 

#ifdef DEBUG
  ls_iconbox_dump_windows(plsib);
#endif
  return;
}

void ls_iconbox_re_evaluate_visibility(LSIconBox *plsib)
{
  gboolean bVisible;
  
  g_object_get(plsib, "visible", &bVisible, NULL);

  if(plsib->priv->nVisibleChilds>0 && 
     bVisible==FALSE &&
     plsib->priv->bHideOnEmpty==TRUE
  ) {
    gtk_widget_show(GTK_WIDGET(plsib));
    DBG("gtk_widget_show");
  } 
  else if (
      plsib->priv->bHideOnEmpty==FALSE
  ) {
    gtk_widget_show(GTK_WIDGET(plsib));
    DBG("gtk_widget_show");
  }

  if(plsib->priv->nVisibleChilds==0 && 
     bVisible==TRUE &&
     plsib->priv->bHideOnEmpty==TRUE
  ) {
    gtk_widget_hide(GTK_WIDGET(plsib));
    DBG("gtk_widget_hide");
  }

  return;
}

void ls_iconbox_remove_icon(LSIconBox *plsib, LSIcon *plsi)
{
  /* hide the icon before remove 						*/
  /* this is needed becores some applications do not close their xwindow 	*/
  /* correctly and we need to hide the icon to make the autohide on empty 	*/
  /* work correctly 								*/
  gtk_widget_hide(GTK_WIDGET(plsi)); 
  
  if(plsib->priv->lOrientation == LS_ICONBOX_HORIZONTAL) 
  {
    gtk_container_remove(GTK_CONTAINER(plsib->priv->pHBox), 
      GTK_WIDGET(plsi));
  }
  else
  {
    gtk_container_remove(GTK_CONTAINER(plsib->priv->pVBox), 
      GTK_WIDGET(plsi));
  }

  if(plsib->priv->lOrientation == LS_ICONBOX_HORIZONTAL) 
  {
    gtk_widget_queue_resize(GTK_WIDGET(plsib->priv->pHBox));
  }
  else
  {
    gtk_widget_queue_resize(GTK_WIDGET(plsib->priv->pVBox));
  }
  
  ls_iconbox_dump_windows(plsib);

  /* remove ref added in ls_iconbox_add_icon					*/
  g_object_unref(plsi);
  
  return;
}

void ls_iconbox_gtkcb_dump_window(GtkWidget *pData, gpointer pUserData) 
{
}

void ls_iconbox_dump_windows (LSIconBox *plsib)
{
}

void ls_iconbox_gtkcb_set_orientation(GtkWidget *pData, gpointer pUserData) 
{
  LSIconBox	*pIconBox;
  LSIcon	*pIcon;
  
  g_return_if_fail(LS_IS_ICON(pData));
  g_return_if_fail(pUserData!=NULL);
  g_return_if_fail(LS_IS_ICONBOX(pUserData));

  pIcon		= LS_ICON(pData);
  pIconBox	= LS_ICONBOX(pUserData);

  gtk_widget_ref(GTK_WIDGET(pIcon));
  if(pIconBox->priv->lOrientation == LS_ICONBOX_VERTICAL)
  {
    gtk_container_remove(GTK_CONTAINER(pIconBox->priv->pHBox), GTK_WIDGET(pIcon));
    gtk_box_pack_start(GTK_BOX(pIconBox->priv->pVBox), GTK_WIDGET(pIcon),
      FALSE, FALSE, 0);
  }
  else
  {
    gtk_container_remove(GTK_CONTAINER(pIconBox->priv->pVBox), GTK_WIDGET(pIcon));
    gtk_box_pack_start(GTK_BOX(pIconBox->priv->pHBox), GTK_WIDGET(pIcon),
      FALSE, FALSE, 0);
  }
  gtk_widget_unref(GTK_WIDGET(pIcon));

  ls_icon_re_evaluate_visibility(pIcon);
  
  return;
}

void ls_iconbox_set_orientation(LSIconBox *plsib, glong lOrientation)
{
  if(plsib->priv->lOrientation == lOrientation)
    return;

  plsib->priv->lOrientation = lOrientation;
  
  if(lOrientation == LS_ICONBOX_VERTICAL) 
  {
    gtk_container_foreach(GTK_CONTAINER(plsib->priv->pHBox), 
      (GtkCallback)ls_iconbox_gtkcb_set_orientation, 
      (gpointer)plsib);

    g_object_ref(plsib->priv->pHBox);
    gtk_widget_hide(GTK_WIDGET(plsib->priv->pHBox));
    gtk_container_remove(GTK_CONTAINER(plsib), GTK_WIDGET(plsib->priv->pHBox));
    ls_iconbox_set_size(plsib, plsib->priv->lSize);
    gtk_container_add(GTK_CONTAINER(plsib), GTK_WIDGET(plsib->priv->pVBox));
    gtk_widget_show(GTK_WIDGET(plsib->priv->pVBox));
    g_object_unref(plsib->priv->pVBox);
  }
  else
  {
    gtk_container_foreach(GTK_CONTAINER(plsib->priv->pVBox), 
      (GtkCallback)ls_iconbox_gtkcb_set_orientation, 
      (gpointer)plsib);
    
    g_object_ref(plsib->priv->pVBox);
    gtk_widget_hide(GTK_WIDGET(plsib->priv->pVBox));
    gtk_container_remove(GTK_CONTAINER(plsib), GTK_WIDGET(plsib->priv->pVBox));
    ls_iconbox_set_size(plsib, plsib->priv->lSize);
    gtk_container_add(GTK_CONTAINER(plsib), GTK_WIDGET(plsib->priv->pHBox));
    gtk_widget_show(GTK_WIDGET(plsib->priv->pHBox));
    g_object_unref(plsib->priv->pHBox);
  }

  return;
}

NetkScreen *ls_iconbox_get_screen(LSIconBox *plsib)
{
  return plsib->priv->pScreen;
}

void ls_iconbox_set_use_color(
  LSIconBox 	*plsib, 
  gint		nUseColor
) {
  GList *pglChild;
  GList *pglIter;

  if(nUseColor!=0&&nUseColor!=1)
    return;

  plsib->priv->bUseColor = nUseColor;

  pglChild = pglIter = ls_iconbox_get_children(plsib);

  if(pglIter==NULL)
    DBG("pglIter==NULL");

  while(pglIter!=NULL)
  {
    if(LS_IS_ICON(pglIter->data))
    {
      ls_icon_set_use_color(LS_ICON(pglIter->data), nUseColor);
    }
    else
    {
      g_warning("!LS_IS_ICON(pglIter->data)");
    }
    pglIter = pglIter->next;
  }

  g_list_free(pglChild);
      
  return;
}
void ls_iconbox_set_show_icon(
  LSIconBox *plsib, 
  gint nMin, 
  gint nMax,
  gint nSkip, 
  gint nOWSMin, 
  gint nOWSMax, 
  gint nOWSSkip
) {
  GList *pglChild;
  GList *pglIter;
  
  if(nOWSMin==0||nOWSMin==1)
    plsib->priv->display.otherWorkspace.minimized		= nOWSMin;
  if(nOWSMax==0||nOWSMax==1)
    plsib->priv->display.otherWorkspace.notMinimized		= nOWSMax;
  if(nOWSSkip==0||nOWSSkip==1)
    plsib->priv->display.otherWorkspace.skipTasklist		= nOWSSkip;
  if(nMin==0||nMin==1)
    plsib->priv->display.sameWorkspace.minimized		= nMin;
  if(nMax==0||nMax==1)
    plsib->priv->display.sameWorkspace.notMinimized		= nMax;
  if(nSkip==0||nSkip==1)
    plsib->priv->display.sameWorkspace.skipTasklist		= nSkip;

  pglChild = pglIter = ls_iconbox_get_children(plsib);

  if(pglIter==NULL)
    DBG("pglIter==NULL");

  while(pglIter!=NULL)
  {
    if(LS_IS_ICON(pglIter->data))
    {
      ls_icon_set_show_same_workspace(LS_ICON(pglIter->data), 
	&(plsib->priv->display.sameWorkspace));
      ls_icon_set_show_other_workspace(LS_ICON(pglIter->data), 
	&(plsib->priv->display.otherWorkspace));
    }
    else
    {
      DBG("!LS_IS_ICON(pglIter->data)");
    }
    pglIter = pglIter->next;
  }

  g_list_free(pglChild);
      
  return;
}

/*******************************************************************************/
/* CALLBACKS                                                                   */
/*******************************************************************************/
static void cb_lsib_window_opened (
  NetkScreen *pScreen,
  NetkWindow *pWin,
  void *pVoid
) {
  LSIconBox	*plsib;
  LSIcon 	*plsi;
  glong		*plXID;
  
  g_return_if_fail(pVoid!=NULL);
  g_return_if_fail(LS_IS_ICONBOX(pVoid));

  DBG("XID:%ld", netk_window_get_xid(pWin));

  plXID		= g_malloc(sizeof(glong));
  *plXID	= netk_window_get_xid(pWin);
  plsi 		= ls_icon_new(pWin, pScreen);
  plsib		= LS_ICONBOX(pVoid);

  g_signal_connect_object(G_OBJECT(plsi), "hide", 
    G_CALLBACK(cb_lsib_icon_hide), plsib, 0);
  g_signal_connect_object(G_OBJECT(plsi), "show", 
    G_CALLBACK(cb_lsib_icon_show), plsib, 0);

  g_hash_table_insert(plsib->priv->pghtXIDIcon, plXID, plsi);
  ls_icon_set_parent(plsi, plsib);
  ls_icon_set_show_same_workspace(plsi, 
    &(plsib->priv->display.sameWorkspace));
  ls_icon_set_show_other_workspace(plsi, 
    &(plsib->priv->display.otherWorkspace));
  ls_icon_set_use_color(plsi, plsib->priv->bUseColor);
  ls_iconbox_add_icon(plsib, plsi);
  ls_icon_re_evaluate_visibility(plsi);
  
  return;
}

static void cb_lsib_window_closed (
  NetkScreen *pScreen,
  NetkWindow *pWin,
  void *pVoid
) {
  LSIconBox 	*plsib;
  LSIcon	*plsi;
  gulong    	lXID;
  
  g_return_if_fail(pVoid!=NULL);
  g_return_if_fail(LS_IS_ICONBOX(pVoid));
  
  DBG("XID:%ld", netk_window_get_xid(pWin));
  
  plsib		= LS_ICONBOX(pVoid);
  lXID		= netk_window_get_xid(pWin);
  plsi		= g_hash_table_lookup(plsib->priv->pghtXIDIcon, &lXID);
  
  g_hash_table_remove(plsib->priv->pghtXIDIcon, &lXID);
  ls_iconbox_remove_icon(plsib, plsi);

  return;
}

static void cb_lsib_xid_destroy(gpointer plXID)
{
  g_free((glong *)plXID);
  return;
}

static void cb_lsib_icon_hide(GtkWidget *pWidget, gpointer pUserData)
{
  g_return_if_fail(LS_IS_ICONBOX(pUserData));

  
  LS_ICONBOX(pUserData)->priv->nVisibleChilds--;
  
  DBG("nVisibleChilds:%d", LS_ICONBOX(pUserData)->priv->nVisibleChilds);
  
  ls_iconbox_re_evaluate_visibility(LS_ICONBOX(pUserData));
  
  return;
}

static void cb_lsib_icon_show(GtkWidget *pWidget, gpointer pUserData)
{
  g_return_if_fail(LS_IS_ICONBOX(pUserData));
  
  LS_ICONBOX(pUserData)->priv->nVisibleChilds++;
  
  DBG("nVisibleChilds:%d", LS_ICONBOX(pUserData)->priv->nVisibleChilds);
  
  ls_iconbox_re_evaluate_visibility(LS_ICONBOX(pUserData));
  
  return;
}

