/*
 *   kdeprintfax - a small fax utility
 *   Copyright (C) 2001  Michael Goffioul
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "faxab.h"

#include <qlabel.h>
#include <qlayout.h>
#include <kpushbutton.h>
#include <klistbox.h>
#include <klocale.h>
#include <kabc/stdaddressbook.h>
#include <kmessagebox.h>
#include <kiconloader.h>
#include <krun.h>
#include <kstdguiitem.h>
#include <kglobal.h>
#include <kconfig.h>

FaxAB::FaxAB(QWidget *parent, const char *name)
: KDialog(parent, name, true)
{
	m_name = new KListBox(this);
	m_fax = new KListBox(this);
	QLabel	*m_namelabel = new QLabel(i18n("Entries:"), this);
	QLabel	*m_faxlabel = new QLabel(i18n("Fax number:"), this);
	m_ok = new KPushButton(KStdGuiItem::ok(), this);
	QPushButton	*m_cancel = new KPushButton(KStdGuiItem::cancel(), this);
	QPushButton	*m_ab = new KPushButton(KGuiItem(i18n("&Edit Addressbook"), "contents"), this);
	connect(m_ok, SIGNAL(clicked()), SLOT(accept()));
	connect(m_cancel, SIGNAL(clicked()), SLOT(reject()));
	connect(m_name, SIGNAL(highlighted(const QString&)), SLOT(slotSelected(const QString&)));
	connect(m_ab, SIGNAL(clicked()), SLOT(slotEditAb()));
	m_ok->setDefault(true);

	QVBoxLayout	*l0 = new QVBoxLayout(this, 10, 10);
	QGridLayout	*l1 = new QGridLayout(0, 2, 3, 0, 0);
	l0->addLayout(l1, 1);
	l1->setRowStretch(1, 1);
	l1->addColSpacing(1, 10);
	l1->addWidget(m_namelabel, 0, 0);
	l1->addWidget(m_faxlabel, 0, 2);
	l1->addWidget(m_name, 1, 0);
	l1->addWidget(m_fax, 1, 2);
	QHBoxLayout	*l2 = new QHBoxLayout(0, 0, 10);
	l0->addLayout(l2, 0);
	l2->addWidget(m_ab, 0);
	l2->addStretch(1);
	l2->addWidget(m_ok, 0);
	l2->addWidget(m_cancel, 0);

	KConfig *conf = KGlobal::config();
	conf->setGroup( "General" );
	QSize defsize( 400, 200 );
	resize( conf->readSizeEntry( "ABSize", &defsize ) );

	initialize();
	connect(KABC::StdAddressBook::self(), SIGNAL(addressBookChanged(AddressBook*)), SLOT(slotAbChanged(AddressBook*)));
}

FaxAB::~FaxAB()
{
	KConfig *conf = KGlobal::config();
	conf->setGroup( "General" );
	conf->writeEntry( "ABSize", size() );
}

void FaxAB::initialize()
{
	m_entries.clear();
	m_name->clear();
	m_fax->clear();

	KABC::AddressBook	*bk = KABC::StdAddressBook::self();
	for (KABC::AddressBook::Iterator it=bk->begin(); it!=bk->end(); ++it)
	{
		KABC::PhoneNumber::List	numbers = (*it).phoneNumbers();
		QStringList	filteredNumbers;
		for (QValueList<KABC::PhoneNumber>::Iterator nit=numbers.begin(); nit!=numbers.end(); ++nit)
		{
			if (((*nit).type() & KABC::PhoneNumber::Fax) && !(*nit).number().isEmpty())
				filteredNumbers << (*nit).number();
		}
		if (filteredNumbers.count() > 0)
		{
			filteredNumbers.prepend((*it).organization());
			if ( !( *it ).formattedName().isEmpty() )
				m_entries[(*it).formattedName()] = filteredNumbers;
			else
			{
				QString key = ( *it ).familyName();
				if ( !( *it ).givenName().isEmpty() )
				{
					if ( !key.isEmpty() )
						key.append( " " );
					key.append( ( *it ).givenName() );
				}
				m_entries[ key ] = filteredNumbers;
			}
		}
	}

	if (m_entries.count() > 0)
	{
		for (QMap<QString,QStringList>::ConstIterator it=m_entries.begin(); it!=m_entries.end(); ++it)
			m_name->insertItem(SmallIcon("abentry"), it.key());
		m_name->sort();
		m_name->setCurrentItem(0);
		m_ok->setEnabled(true);
	}
	else
		m_ok->setDisabled(true);
}

void FaxAB::slotEditAb()
{
	KRun::runCommand("kaddressbook");
}

void FaxAB::slotAbChanged(AddressBook*)
{
	initialize();
}

void FaxAB::slotSelected(const QString& text)
{
	if (m_entries.contains(text))
	{
		QStringList	l = m_entries[text];
		m_fax->clear();
		QStringList::ConstIterator	it = l.begin();
		++it;	// skip organization name
		for (; it!=l.end(); ++it)
			m_fax->insertItem(SmallIcon("kdeprintfax"), *it);
		m_fax->sort();
		m_fax->setCurrentItem(0);
	}
}

bool FaxAB::getEntry(QString& number, QString& name, QString& enterprise, QWidget *parent)
{
	FaxAB	kab(parent);
	if (!kab.isValid())
	{
		KMessageBox::error(parent, i18n("No fax number found in your address book."));
		return false;
	}
	if (kab.exec())
	{
		number = kab.m_fax->currentText();
		name = kab.m_name->currentText();
		if (kab.m_entries.contains(name))
		{
			enterprise = kab.m_entries[name][0];
		}
		return true;
	}

	return false;
}

bool FaxAB::getEntryByNumber(const QString& number, QString& name, QString& enterprise)
{
	KABC::AddressBook *bk = KABC::StdAddressBook::self();
	for (KABC::AddressBook::Iterator it=bk->begin(); it!=bk->end(); ++it)
	{
		KABC::PhoneNumber::List	numbers = (*it).phoneNumbers();
		QStringList	filteredNumbers;
		for (QValueList<KABC::PhoneNumber>::Iterator nit=numbers.begin(); nit!=numbers.end(); ++nit)
		{
			if (((*nit).type() & KABC::PhoneNumber::Fax) )
			{
				QString strippedNumber;
				for (uint i = 0; i < (*nit).number().length(); ++i)
					if ((*nit).number()[i].isDigit())
						strippedNumber.append((*nit).number()[i]);

				if ( strippedNumber == number)
				{
					enterprise = (*it).organization();
					name = (*it).formattedName();
					return true;
				}
			}
		}
	}

	return false;
}

bool FaxAB::isValid()
{
	return true;
	//return (m_name->count() > 0);
}

#include "faxab.moc"
