/*
 * Test statistics on mail folders
 *
 * Copyright (C) 2005--2013  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"
#include <wibble/tests.h>
#include <wibble/exception.h>
#include <buffy/mailfolder.h>
#include <buffy/mailfolder/maildir.h>
#include <buffy/mailfolder/mailbox.h>
#include <sys/stat.h>  // mkdir
#include <sys/types.h>
#include <cstdio>
#include <unistd.h>

namespace tut {
using namespace std;
using namespace buffy;
using namespace wibble::tests;

struct buffy_folder_test {
	buffy_folder_test() {
		//fprintf(stderr, "Start\n");
		//unlink("test.tdb");
	}
	~buffy_folder_test() {
		//fprintf(stderr, "End\n");
		//unlink("test.tdb");
	}
};
typedef test_group<buffy_folder_test> tg;
typedef tg::object to;
tg folder_tg("buffy_folder_test");

template<> template<>
void to::test<1> ()
{
	try {
		std::vector<string> dirs;
		dirs.push_back("mbox");
		dirs.push_back("mbox/");
		for (std::vector<string>::const_iterator i = dirs.begin();
				i != dirs.end(); ++i)
		{
			// An empty database should return empty sets, but not fail
			std::vector<MailFolder> folders = MailFolder::enumerateFolders(*i);

			ensure_equals(folders.size(), 2u);

			MailFolder empty = folders[0].name() == "empty.mbox" ? folders[0] : folders[1];
			MailFolder test = folders[0].name() == "empty.mbox" ? folders[1] : folders[0];

			ensure_equals(empty.path(), "mbox/empty.mbox");
			ensure_equals(empty.name(), "empty.mbox");
			ensure_equals(empty.type(), "Mailbox");
			ensure_equals(test.path(), "mbox/test.mbox");
			ensure_equals(test.name(), "test.mbox");
			ensure_equals(test.type(), "Mailbox");

			empty.updateStatistics();

			ensure_equals(empty.getMsgTotal(), 0);
			ensure_equals(empty.getMsgUnread(), 0);
			ensure_equals(empty.getMsgNew(), 0);
			ensure_equals(empty.getMsgFlagged(), 0);
			ensure_equals(empty.changed(), false);

			test.updateStatistics();

			ensure_equals(test.getMsgTotal(), 3);
			ensure_equals(test.getMsgUnread(), 0);
			ensure_equals(test.getMsgNew(), 0);
			ensure_equals(test.getMsgFlagged(), 1);
			ensure_equals(test.changed(), false);
		}
	} catch (std::exception& e) {
		cerr << e.what() << endl;
		throw;
	}
}

template<> template<>
void to::test<2> ()
{
	try {
		std::vector<string> dirs;
		dirs.push_back("mboxgz");
		dirs.push_back("mboxgz/");
		for (std::vector<string>::const_iterator i = dirs.begin();
				i != dirs.end(); ++i)
		{
			// An empty database should return empty sets, but not fail
			std::vector<MailFolder> folders = MailFolder::enumerateFolders(*i);

			ensure_equals(folders.size(), 2u);

			MailFolder empty = folders[0].name() == "empty.mbox.gz" ? folders[0] : folders[1];
			MailFolder test = folders[0].name() == "empty.mbox.gz" ? folders[1] : folders[0];

			ensure_equals(empty.path(), "mboxgz/empty.mbox.gz");
			ensure_equals(empty.name(), "empty.mbox.gz");
			ensure_equals(empty.type(), "Mailbox");
			ensure_equals(test.path(), "mboxgz/test.mbox.gz");
			ensure_equals(test.name(), "test.mbox.gz");
			ensure_equals(test.type(), "Mailbox");

			empty.updateStatistics();

			ensure_equals(empty.getMsgTotal(), 0);
			ensure_equals(empty.getMsgUnread(), 0);
			ensure_equals(empty.getMsgNew(), 0);
			ensure_equals(empty.getMsgFlagged(), 0);
			ensure_equals(empty.changed(), false);

			test.updateStatistics();

			ensure_equals(test.getMsgTotal(), 3);
			ensure_equals(test.getMsgUnread(), 0);
			ensure_equals(test.getMsgNew(), 0);
			ensure_equals(test.getMsgFlagged(), 1);
			ensure_equals(test.changed(), false);
		}
	} catch (std::exception& e) {
		cerr << e.what() << endl;
		throw;
	}
}

template<> template<>
void to::test<3> ()
{
	try {
		std::vector<string> dirs;
		dirs.push_back("maildir");
		dirs.push_back("maildir/");
		for (std::vector<string>::const_iterator i = dirs.begin();
				i != dirs.end(); ++i)
		{
			// An empty database should return empty sets, but not fail
			std::vector<MailFolder> folders = MailFolder::enumerateFolders(*i);

			ensure_equals(folders.size(), 4u);

			MailFolder root;
			MailFolder empty;
			MailFolder test;
			MailFolder loop1;
			int missed = 0;
			for (int j = 0; j < 4; ++j)
			{
				//cerr << *i << " NAME: " << folders[j].name() << endl;
				if (folders[j].name() == "maildir")
					root = folders[j];
				else if (folders[j].name() == "maildir.empty")
					empty = folders[j];
				else if (folders[j].name() == "maildir.test")
					test = folders[j];
				else if (folders[j].name() == "maildir.loop1")
					loop1 = folders[j];
				else
					++missed;
			}
			ensure_equals(missed, 0);
			ensure(root.valid());
			ensure(empty.valid());
			ensure(test.valid());
			ensure(loop1.valid());

			ensure_equals(root.path(), "maildir");
			ensure_equals(root.name(), "maildir");
			ensure_equals(root.type(), "Maildir");
			ensure_equals(empty.path(), "maildir/empty");
			ensure_equals(empty.name(), "maildir.empty");
			ensure_equals(empty.type(), "Maildir");
			ensure_equals(test.path(), "maildir/test");
			ensure_equals(test.name(), "maildir.test");
			ensure_equals(test.type(), "Maildir");
			ensure_equals(loop1.path(), "maildir/loop1");
			ensure_equals(loop1.name(), "maildir.loop1");
			ensure_equals(loop1.type(), "Maildir");

			root.updateStatistics();

			ensure_equals(root.getMsgTotal(), 0);
			ensure_equals(root.getMsgUnread(), 0);
			ensure_equals(root.getMsgNew(), 0);
			ensure_equals(root.getMsgFlagged(), 0);
			ensure_equals(root.changed(), false);

			empty.updateStatistics();

			ensure_equals(empty.getMsgTotal(), 0);
			ensure_equals(empty.getMsgUnread(), 0);
			ensure_equals(empty.getMsgNew(), 0);
			ensure_equals(empty.getMsgFlagged(), 0);
			ensure_equals(empty.changed(), false);

			test.updateStatistics();

			ensure_equals(test.getMsgTotal(), 3);
			ensure_equals(test.getMsgUnread(), 0);
			ensure_equals(test.getMsgNew(), 0);
			ensure_equals(test.getMsgFlagged(), 1);
			ensure_equals(test.changed(), false);

			loop1.updateStatistics();

			ensure_equals(loop1.getMsgTotal(), 0);
			ensure_equals(loop1.getMsgUnread(), 0);
			ensure_equals(loop1.getMsgNew(), 0);
			ensure_equals(loop1.getMsgFlagged(), 0);
			ensure_equals(loop1.changed(), false);
		}
	} catch (std::exception& e) {
		cerr << e.what() << endl;
		throw;
	}
}

template<> template<>
void to::test<4> ()
{
	// Test that it is ok to remove a mail folder when its MailFolder
	// exists
	try {
		// Create an empty mailbox
		FILE* out = fopen("empty.mbox", "wt");
		if (!out)
			throw wibble::exception::System("creating file empty.mbox");
		fclose(out);

		MailFolder mf(new mailfolder::Mailbox("empty.mbox"));

		mf.updateStatistics();

		ensure_equals(mf.getMsgTotal(), 0);
		ensure_equals(mf.getMsgUnread(), 0);
		ensure_equals(mf.getMsgNew(), 0);
		ensure_equals(mf.getMsgFlagged(), 0);

		if (unlink("empty.mbox") == -1)
			throw wibble::exception::System("Deleting file empty.mbox");

		ensure_equals(mf.changed(), true);

		mf.updateStatistics();

		ensure_equals(mf.getMsgTotal(), 0);
		ensure_equals(mf.getMsgUnread(), 0);
		ensure_equals(mf.getMsgNew(), 0);
		ensure_equals(mf.getMsgFlagged(), 0);
	} catch (std::exception& e) {
		cerr << e.what() << endl;
		throw;
	}
}

template<> template<>
void to::test<5> ()
{
	// Test that it is ok to remove a mail folder when its MailFolder
	// exists
	try {
		// Create an empty maildir
		if (mkdir("empty", 0755) == -1)
			throw wibble::exception::System("Creating directory empty");
		if (mkdir("empty/cur", 0755) == -1)
			throw wibble::exception::System("Creating directory empty/cur");
		if (mkdir("empty/new", 0755) == -1)
			throw wibble::exception::System("Creating directory empty/new");

		MailFolder mf(new mailfolder::Maildir("empty"));

		mf.updateStatistics();

		ensure_equals(mf.getMsgTotal(), 0);
		ensure_equals(mf.getMsgUnread(), 0);
		ensure_equals(mf.getMsgNew(), 0);
		ensure_equals(mf.getMsgFlagged(), 0);

		if (rmdir("empty/cur") == -1)
			throw wibble::exception::System("Deleting directory empty/cur");
		if (rmdir("empty/new") == -1)
			throw wibble::exception::System("Deleting directory empty/new");
		if (rmdir("empty") == -1)
			throw wibble::exception::System("Deleting directory empty");

		ensure_equals(mf.changed(), true);

		mf.updateStatistics();

		ensure_equals(mf.getMsgTotal(), 0);
		ensure_equals(mf.getMsgUnread(), 0);
		ensure_equals(mf.getMsgNew(), 0);
		ensure_equals(mf.getMsgFlagged(), 0);
	} catch (std::exception& e) {
		cerr << e.what() << endl;
		throw;
	}
}

}

// vim:set ts=4 sw=4:
