﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

#nullable disable

using System;
using Microsoft.CodeAnalysis;
using Microsoft.VisualStudio.IntegrationTest.Utilities;
using Microsoft.VisualStudio.IntegrationTest.Utilities.Common;
using Microsoft.VisualStudio.IntegrationTest.Utilities.Input;
using Roslyn.Test.Utilities;

namespace Roslyn.VisualStudio.IntegrationTests.VisualBasic
{
    public class BasicErrorListCommon : AbstractEditorTest
    {
        public BasicErrorListCommon(VisualStudioInstanceFactory instanceFactory, string templateName)
            : base(instanceFactory, nameof(BasicErrorListCommon), templateName)
        {
        }

        protected override string LanguageName => LanguageNames.VisualBasic;

        public virtual void ErrorList()
        {
            VisualStudio.Editor.SetText(@"
Module Module1

    Function Good() As P
        Return Nothing
    End Function

    Sub Main()
        Goo()
    End Sub

End Module
");
            VisualStudio.ErrorList.ShowErrorList();
            var expectedContents = new[] {
                new ErrorListItem(
                    severity: "Error",
                    description: "Type 'P' is not defined.",
                    project: "TestProj",
                    fileName: "Class1.vb",
                    line: 4,
                    column: 24),
                new ErrorListItem(
                    severity: "Error",
                    description: "'Goo' is not declared. It may be inaccessible due to its protection level.",
                    project: "TestProj",
                    fileName: "Class1.vb",
                    line: 9,
                    column: 9)
            };
            var actualContents = VisualStudio.ErrorList.GetErrorListContents();
            AssertEx.EqualOrDiff(
                string.Join<ErrorListItem>(Environment.NewLine, expectedContents),
                string.Join<ErrorListItem>(Environment.NewLine, actualContents));

            VisualStudio.ErrorList.NavigateToErrorListItem(0);
            VisualStudio.Editor.Verify.CaretPosition(43);
            VisualStudio.SolutionExplorer.BuildSolution();
            VisualStudio.ErrorList.ShowErrorList();
            actualContents = VisualStudio.ErrorList.GetErrorListContents();
            AssertEx.EqualOrDiff(
                string.Join<ErrorListItem>(Environment.NewLine, expectedContents),
                string.Join<ErrorListItem>(Environment.NewLine, actualContents));
        }

        public virtual void ErrorsDuringMethodBodyEditing()
        {
            VisualStudio.Editor.SetText(@"
Namespace N
    Class C
        Private F As Integer
        Sub S()
             ' Comment
        End Sub
    End Class
End Namespace
");
            VisualStudio.Editor.PlaceCaret(" Comment", charsOffset: -2);
            VisualStudio.SendKeys.Send("F = 0");
            VisualStudio.ErrorList.ShowErrorList();
            var expectedContents = new ErrorListItem[] { };
            var actualContents = VisualStudio.ErrorList.GetErrorListContents();
            AssertEx.EqualOrDiff(
                string.Join<ErrorListItem>(Environment.NewLine, expectedContents),
                string.Join<ErrorListItem>(Environment.NewLine, actualContents));

            VisualStudio.Editor.Activate();
            VisualStudio.Editor.PlaceCaret("F = 0 ' Comment", charsOffset: -1);
            VisualStudio.Editor.SendKeys("F");
            VisualStudio.ErrorList.ShowErrorList();
            expectedContents = new ErrorListItem[] {
                new ErrorListItem(
                    severity: "Error",
                    description: "'FF' is not declared. It may be inaccessible due to its protection level.",
                    project: "TestProj",
                    fileName: "Class1.vb",
                    line: 6,
                    column: 13)
            };
            actualContents = VisualStudio.ErrorList.GetErrorListContents();
            AssertEx.EqualOrDiff(
                string.Join<ErrorListItem>(Environment.NewLine, expectedContents),
                string.Join<ErrorListItem>(Environment.NewLine, actualContents));

            VisualStudio.Editor.Activate();
            VisualStudio.Editor.PlaceCaret("FF = 0 ' Comment", charsOffset: -1);
            VisualStudio.Editor.SendKeys(VirtualKey.Delete);
            VisualStudio.ErrorList.ShowErrorList();
            expectedContents = new ErrorListItem[] { };
            actualContents = VisualStudio.ErrorList.GetErrorListContents();
            AssertEx.EqualOrDiff(
                string.Join<ErrorListItem>(Environment.NewLine, expectedContents),
                string.Join<ErrorListItem>(Environment.NewLine, actualContents));
        }
    }
}
