/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     element_2d.c                                                   */
/*                                                                          */
/*                                                                          */
/* description:  routines on elements that depend on the dimension in 2d    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"

/*--------------------------------------------------------------------------*/
/*  world_to_coord_2d():  return -1 if inside, otherwise index of lambda<0  */
/*--------------------------------------------------------------------------*/

int world_to_coord_2d(const EL_INFO *el_info, const REAL *xy,
		      REAL_B lambda)
{
  FUNCNAME("world_to_coord_2d");
  REAL  edge[2][DIM_OF_WORLD], x[DIM_OF_WORLD];
  REAL  x0, det, det0, det1, adet, lmin;
  int   i, j, k;

#if (DIM_OF_WORLD != 2)
  ERROR_EXIT("not yet for DIM_OF_WORLD != 2\n");
#endif

  DEBUG_TEST_FLAG(FILL_COORDS, el_info);
  DEBUG_TEST_EXIT(!el_info->mesh->parametric || el_info->mesh->parametric->use_reference_mesh, "You must enable the use_reference_mesh entry in the PARAMETRIC structure to use this function on the reference mesh. Use parametric->worldto_coord() to access the parametric mesh\n");

/*  wir haben das gleichungssystem zu loesen: */
/*       ( q1x q2x )  (lambda1)     (qx)      */
/*       ( q1y q2y )  (lambda2)  =  (qy)      */
/*      mit qi=pi-p3, q=xy-p3                 */

  for (j=0; j<DIM_OF_WORLD; j++) {
    x0 = el_info->coord[2][j];
    x[j] = xy[j] - x0;
    for (i=0; i < 2; i++)
      edge[i][j] = el_info->coord[i][j] - x0;
  }

  det  = edge[0][0] * edge[1][1] - edge[0][1] * edge[1][0]; 
  det0 =       x[0] * edge[1][1] -       x[1] * edge[1][0]; 
  det1 = edge[0][0] * x[1]       - edge[0][1] * x[0]; 

  adet = ABS(det);

  if (adet < 1.E-20) {
    ERROR_EXIT("det = %le; abort\n", det);
    return(-2);
  }

  lambda[0] = det0 / det;
  lambda[1] = det1 / det;
  lambda[2] = 1.0 - lambda[0] - lambda[1];

  k = -1;
  lmin = 0.0;
  j = 0;
  for (i = 0; i <= 2; i++) {
    if ((lambda[i]*adet) < -1.E-15) {
      if (lambda[i] < lmin) {
	k = i;
	lmin = lambda[i];
      }
      j++;
    }
  }

  return(k);
}

/*--------------------------------------------------------------------------*/
/*  transform local coordinates l to world coordinates; if w is non nil     */
/*  store them at w otherwise return a pointer to some local static         */
/*  area containing world coordintes                                        */
/*--------------------------------------------------------------------------*/

const REAL *coord_to_world_2d(const EL_INFO *el_info, const REAL *l, REAL_D w)
{
  FUNCNAME("coord_to_world_2d");
  static REAL_D world;
  REAL         *ret;
  int           i;

#if DIM_OF_WORLD < 2
  ERROR_EXIT("Does not make sense for DIM_OF_WORLD == 1!\n");
#endif

  DEBUG_TEST_FLAG(FILL_COORDS, el_info);
  DEBUG_TEST_EXIT(!el_info->mesh->parametric || el_info->mesh->parametric->use_reference_mesh, "You must enable the use_reference_mesh entry in the PARAMETRIC structure to use this function on the reference mesh. Use parametric->coord_to_world() to access the parametric mesh\n");

  ret = w ? w : world;

  for(i = 0; i < DIM_OF_WORLD; i++)
    ret[i] = el_info->coord[0][i] * l[0]
           + el_info->coord[1][i] * l[1]
           + el_info->coord[2][i] * l[2];

  return((const REAL *) ret);
}

/*--------------------------------------------------------------------------*/
/*  compute volume of an element                                            */
/*--------------------------------------------------------------------------*/

REAL el_det_2d(const EL_INFO *el_info)
{
  FUNCNAME("el_det_2d");
  REAL_D      e1, e2;
  REAL        det;
  const REAL *v0;
  int         i;
#if DIM_OF_WORLD == 3
  REAL_D      n;
#endif

  DEBUG_TEST_FLAG(FILL_COORDS, el_info);
  DEBUG_TEST_EXIT(!el_info->mesh->parametric, "not for parametric meshes\n");
  DEBUG_TEST_EXIT(!el_info->mesh->parametric || el_info->mesh->parametric->use_reference_mesh, "You must enable the use_reference_mesh entry in the PARAMETRIC structure to use this function on the reference mesh. Use parametric->det() to access the parametric mesh\n");

#if DIM_OF_WORLD < 2
  ERROR_EXIT("Does not make sense for DIM_OF_WORLD == 1!\n");
#endif
  
  v0 = el_info->coord[0];
  for (i = 0; i < DIM_OF_WORLD; i++) {
    e1[i] = el_info->coord[1][i] - v0[i];
    e2[i] = el_info->coord[2][i] - v0[i];
  }

#if DIM_OF_WORLD==2
  det = WEDGE_DOW(e1, e2);
  det = ABS(det);
#else
  WEDGE_DOW(e1, e2, n);
  det = NORM_DOW(n);
#endif

  return(det);
}

REAL el_volume_2d(const EL_INFO *el_info)
{
  return 0.5 * el_det_2d(el_info);
}

/*--------------------------------------------------------------------------*/
/*  compute gradients of basis functions on element; return the absolute    */
/*  value of the determinant from the transformation to the reference       */
/*  element                                                                 */
/*                                                                          */
/*  Notice: for dim < DIM_OF_WORLD grd_lam will contain tangential          */
/*  derivatives of the barycentric coordinates!                             */
/*--------------------------------------------------------------------------*/

REAL el_grd_lambda_2d(const EL_INFO *el_info,
		      REAL grd_lam[N_LAMBDA][DIM_OF_WORLD])
{
  FUNCNAME("el_grd_lambda_2d");
  int         i, j;
  REAL        e1[DIM_OF_WORLD], e2[DIM_OF_WORLD], det, adet;
  const REAL  *v0;
  REAL        a11, a12, a21, a22;
#if DIM_OF_WORLD == 3
  REAL        n[DIM_OF_WORLD];
  REAL        a13, a23;
#endif

#if DIM_OF_WORLD < 2
  ERROR_EXIT("Does not make sense for DIM_OF_WORLD == 1!\n");
#endif

  DEBUG_TEST_FLAG(FILL_COORDS, el_info);
  DEBUG_TEST_EXIT(!el_info->mesh->parametric || el_info->mesh->parametric->use_reference_mesh, "You must enable the use_reference_mesh entry in the PARAMETRIC structure to use this function on the reference mesh. Use parametric->grd_lambda() to access the parametric mesh\n");

  v0 = el_info->coord[0];
  for (i = 0; i < DIM_OF_WORLD; i++) {
    e1[i] = el_info->coord[1][i] - v0[i];
    e2[i] = el_info->coord[2][i] - v0[i];
  }

#if DIM_OF_WORLD == 2
  det = WEDGE_DOW(e1, e2);
  adet = ABS(det);
#else
  WEDGE_DOW(e1, e2, n);
  det = NRM2_DOW(n);
  adet = sqrt(det);
#endif

  if (adet < 1.0E-25) 
  {
    MSG("abs(det) = %lf\n", adet);

    for (i = 0; i < N_LAMBDA; i++)
      for (j = 0; j < DIM_OF_WORLD; j++)
	grd_lam[i][j] = 0.0;
  }
  else
  {
    det = 1.0 / det;
#if DIM_OF_WORLD == 2
    a11 =  e2[1] * det;
    a21 = -e2[0] * det;
    a12 = -e1[1] * det;
    a22 =  e1[0] * det;

    grd_lam[1][0] = a11;
    grd_lam[1][1] = a21;
    grd_lam[2][0] = a12;
    grd_lam[2][1] = a22;
    grd_lam[0][0] = - grd_lam[1][0] - grd_lam[2][0];
    grd_lam[0][1] = - grd_lam[1][1] - grd_lam[2][1];
#else
    a11 = (e2[1]* n[2] - e2[2]* n[1]) * det;
    a12 = (e2[2]* n[0] - e2[0]* n[2]) * det;
    a13 = (e2[0]* n[1] - e2[1]* n[0]) * det;
    a21 = (e1[2]* n[1] - e1[1]* n[2]) * det;
    a22 = (e1[0]* n[2] - e1[2]* n[0]) * det;
    a23 = (e1[1]* n[0] - e1[0]* n[1]) * det;

    grd_lam[1][0] = a11;
    grd_lam[1][1] = a12;
    grd_lam[1][2] = a13;
    grd_lam[2][0] = a21;
    grd_lam[2][1] = a22;
    grd_lam[2][2] = a23;
    grd_lam[0][0] = -grd_lam[1][0] - grd_lam[2][0];
    grd_lam[0][1] = -grd_lam[1][1] - grd_lam[2][1];
    grd_lam[0][2] = -grd_lam[1][2] - grd_lam[2][2];
#endif
  }

  return(adet);
}


/*--------------------------------------------------------------------------*/
/*  calculate a normal of an edge of a triangle with coordinates            */
/*  coord; return the absolute value of the determinant from the            */
/*  transformation to the reference element                                 */
/*--------------------------------------------------------------------------*/

REAL get_wall_normal_2d(const EL_INFO *el_info, int i0, REAL *normal)
{
  FUNCNAME("get_face_normal_2d");
#if DIM_OF_WORLD == 2
  static int   ind[5] = {0, 1, 2, 0, 1};
  REAL         det;
  int          i1 = ind[i0+1], i2 = ind[i0+2];
  const REAL_D *coord = el_info->coord;

  DEBUG_TEST_FLAG(FILL_COORDS, el_info);

  normal[0] = coord[i2][1] - coord[i1][1];
  normal[1] = coord[i1][0] - coord[i2][0];

  det = sqrt(SQR(normal[0])+SQR(normal[1]));
  TEST_EXIT(det > 1.e-30, "det = 0 on face %d\n", i0);

  normal[0] /= det;
  normal[1] /= det;
  return(det);
#else 
  ERROR_EXIT("not implemented for DIM_OF_WORLD = %d in 2d\n", DIM_OF_WORLD);
  return 0.0;
#endif

}

/*--------------------------------------------------------------------------*/
/* orient the vertices of walls                                             */
/* used by estimator for the jumps => same quadrature nodes from both sides!*/
/*--------------------------------------------------------------------------*/

int *sorted_wall_indices_2d(int wall, int permno)
{
  static int sorted[3][2][2] = {
    {{1,2}, {2,1}}, {{2,0},{0,2}}, {{0,1}, {1,0}}};

  return sorted[wall][permno];
}

int wall_orientation_2d(const EL *el, int wall, int **vec)
{
  int    no;
  int    *vof = vertex_of_edge_2d[wall];
  DOF    **dof = el->dof;

  if (dof[vof[0]][0] < dof[vof[1]][0])
    no = 0;
  else
    no = 1;

  if (vec) {
    *vec = sorted_wall_indices_2d(wall, no);
  }

  return no;
}

int *sort_wall_indices_2d(const EL *el, int wall, int *vec)
{
  int    i, *val  = nil;

  (void)wall_orientation_2d(el, wall, &val);

  if (vec)
  {
    for (i = 0; i < 2; i++)
      vec[i] = val[i];
    return vec;
  }
  else
    return val;
}
