#!/bin/sh
#
# Copyright (c) 2011, 2014 Nicira, Inc.
# Copyright (c) 2007, 2009 Javier Fernandez-Sanguino <jfs@debian.org>
#
# This is free software; you may redistribute it and/or modify
# it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2,
# or (at your option) any later version.
#
# This is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License with
# the Debian operating system, in /usr/share/common-licenses/GPL;  if
# not, write to the Free Software Foundation, Inc., 59 Temple Place,
# Suite 330, Boston, MA 02111-1307 USA
#
# Short-Description: Simple OpenFlow controller for testing
# Description:       This controller enables OpenFlow switches that connect to
#                    it to act as MAC-learning Ethernet switches.

PATH=/usr/local/sbin:/usr/local/bin:/sbin:/bin:/usr/sbin:/usr/bin

DAEMON=/usr/bin/ovs-testcontroller # Introduce the server's location here
NAME=ovs-testcontroller         # Introduce the short server's name here
DESC=ovs-testcontroller         # Introduce a short description here
LOGDIR=/var/log/openvswitch	# Log directory to use

PIDFILE=/var/run/openvswitch/$NAME.pid

test -x $DAEMON || exit 0

. /lib/lsb/init-functions

# Default options, these can be overriden by the information
# at /etc/openvswitch-testcontroller/openvswitch-testcontroller.default
DAEMON_OPTS=""          # Additional options given to the server

DODTIME=10              # Time to wait for the server to die, in seconds
                        # If this value is set too low you might not
                        # let some servers to die gracefully and
                        # 'restart' will not work

LOGFILE=$LOGDIR/$NAME.log  # Server logfile
#DAEMONUSER=            # User to run the daemons as. If this value
                        # is set start-stop-daemon will chuid the server

# Include defaults if available
default=/etc/openvswitch-testcontroller/openvswitch-testcontroller.default
if [ -f $default ] ; then
    . $default
fi

# Check that the user exists (if we set a user)
# Does the user exist?
if [ -n "$DAEMONUSER" ] ; then
    if getent passwd | grep -q "^$DAEMONUSER:"; then
        # Obtain the uid and gid
        DAEMONUID=`getent passwd |grep "^$DAEMONUSER:" | awk -F : '{print $3}'`
        DAEMONGID=`getent passwd |grep "^$DAEMONUSER:" | awk -F : '{print $4}'`
    else
        log_failure_msg "The user $DAEMONUSER, required to run $NAME does not exist."
        exit 1
    fi
fi


set -e

running_pid() {
# Check if a given process pid's cmdline matches a given name
    pid=$1
    name=$2
    [ -z "$pid" ] && return 1
    [ ! -d /proc/$pid ] &&  return 1
    cmd=`cat /proc/$pid/cmdline | tr "\000" "\n"|head -n 1 |cut -d : -f 1`
    # Is this the expected server
    [ "$cmd" != "$name" ] &&  return 1
    return 0
}

running() {
# Check if the process is running looking at /proc
# (works for all users)

    # No pidfile, probably no daemon present
    [ ! -f "$PIDFILE" ] && return 1
    pid=`cat $PIDFILE`
    running_pid $pid $DAEMON || return 1
    return 0
}

start_server() {
    if [ -z "$LISTEN" ]; then
        echo "$default: No connection methods configured, controller disabled" >&2
        exit 0
    fi

    if [ ! -d /var/run/openvswitch ]; then
        install -d -m 755 -o root -g root /var/run/openvswitch
    fi

    SSL_OPTS=
    case $LISTEN in
        *ssl*)
            : ${PRIVKEY:=/etc/openvswitch-testcontroller/privkey.pem}
            : ${CERT:=/etc/openvswitch-testcontroller/cert.pem}
            : ${CACERT:=/etc/openvswitch-testcontroller/cacert.pem}
            if test ! -e "$PRIVKEY" || test ! -e "$CERT" ||
                test ! -e "$CACERT"; then
                if test ! -e "$PRIVKEY"; then
                    echo "$PRIVKEY: private key missing" >&2
                fi
                if test ! -e "$CERT"; then
                    echo "$CERT: certificate for private key missing" >&2
                fi
                if test ! -e "$CACERT"; then
                    echo "$CACERT: CA certificate missing" >&2
                fi
                exit 1
            fi
            SSL_OPTS="--private-key=$PRIVKEY --certificate=$CERT --ca-cert=$CACERT"
            ;;
    esac

# Start the process using the wrapper
        if [ -z "$DAEMONUSER" ] ; then
            start-stop-daemon --start --pidfile $PIDFILE \
                        --exec $DAEMON -- --detach --pidfile=$PIDFILE \
                        $LISTEN $DAEMON_OPTS $SSL_OPTS
            errcode=$?
        else
# if we are using a daemonuser then change the user id
            start-stop-daemon --start --quiet --pidfile $PIDFILE \
                        --chuid $DAEMONUSER --exec $DAEMON -- \
                        --detach --pidfile=$PIDFILE $LISTEN $DAEMON_OPTS \
                        $SSL_OPTS
            errcode=$?
        fi
        return $errcode
}

stop_server() {
# Stop the process using the wrapper
        if [ -z "$DAEMONUSER" ] ; then
            start-stop-daemon --stop --quiet --pidfile $PIDFILE \
                        --exec $DAEMON
            errcode=$?
        else
# if we are using a daemonuser then look for process that match
            start-stop-daemon --stop --quiet --pidfile $PIDFILE \
                        --user $DAEMONUSER --exec $DAEMON
            errcode=$?
        fi

        return $errcode
}

reload_server() {
    [ ! -f "$PIDFILE" ] && return 1
    pid=`cat $PIDFILE` # This is the daemon's pid
    # Send a SIGHUP
    kill -1 $pid
    return $?
}

force_stop() {
# Force the process to die killing it manually
    [ ! -e "$PIDFILE" ] && return
    if running ; then
        kill -15 $pid
        # Is it really dead?
        sleep "$DODTIME"
        if running ; then
            kill -9 $pid
            sleep "$DODTIME"
            if running ; then
                echo "Cannot kill $NAME (pid=$pid)!"
                exit 1
            fi
        fi
    fi
    rm -f $PIDFILE
}


case "$1" in
  start)
        log_daemon_msg "Starting $DESC " "$NAME"
        # Check if it's running first
        if running ;  then
            log_progress_msg "apparently already running"
            log_end_msg 0
            exit 0
        fi
        if start_server && running ;  then
            # It's ok, the server started and is running
            log_end_msg 0
        else
            # Either we could not start it or it is not running
            # after we did
            # NOTE: Some servers might die some time after they start,
            # this code does not try to detect this and might give
            # a false positive (use 'status' for that)
            log_end_msg 1
        fi
        ;;
  stop)
        log_daemon_msg "Stopping $DESC" "$NAME"
        if running ; then
            # Only stop the server if we see it running
            stop_server
            log_end_msg $?
        else
            # If it's not running don't do anything
            log_progress_msg "apparently not running"
            log_end_msg 0
            exit 0
        fi
        ;;
  force-stop)
        # First try to stop gracefully the program
        $0 stop
        if running; then
            # If it's still running try to kill it more forcefully
            log_daemon_msg "Stopping (force) $DESC" "$NAME"
            force_stop
            log_end_msg $?
        fi
        ;;
  restart|force-reload)
        log_daemon_msg "Restarting $DESC" "$NAME"
        if running; then
            stop_server
            # Wait some sensible amount, some server need this.
            [ -n "$DODTIME" ] && sleep $DODTIME
        fi
        start_server
        running
        log_end_msg $?
        ;;
  status)

        log_daemon_msg "Checking status of $DESC" "$NAME"
        if running ;  then
            log_progress_msg "running"
            log_end_msg 0
        else
            log_progress_msg "apparently not running"
            log_end_msg 1
            exit 1
        fi
        ;;
  # Use this if the daemon cannot reload
  reload)
        log_warning_msg "Reloading $NAME daemon: not implemented, as the daemon"
        log_warning_msg "cannot re-read the config file (use restart)."
        ;;
  *)
        N=/etc/openvswitch-testcontroller/openvswitch-testcontroller.init
        echo "Usage: $N {start|stop|force-stop|restart|force-reload|status}" >&2
        exit 1
        ;;
esac

exit 0
