/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 *   2020      Ashleigh Newman-Jones <ashnewman-jones@hotmail.co.uk>
 */

#define SIMDE_TEST_X86_AVX512_INSN madd

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/madd.h>

static int
test_simde_mm_mask_madd_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int16_t src[8];
    const simde__mmask8 k;
    const int16_t a[8];
    const int16_t b[8];
    const int32_t r[4];
  } test_vec[] = {
    { {  INT16_C( 21813),  INT16_C( 32222),  INT16_C(  7623),  INT16_C( 12885), -INT16_C( 31767), -INT16_C( 17157), -INT16_C( 26419),  INT16_C( 27124) },
      UINT8_C( 87),
      {  INT16_C(  4964),  INT16_C( 13070),  INT16_C( 11795), -INT16_C( 30925), -INT16_C(  4013), -INT16_C( 24845),  INT16_C( 17494), -INT16_C( 21549) },
      {  INT16_C( 20514),  INT16_C( 16243), -INT16_C( 23131),  INT16_C( 10280), -INT16_C(  6752),  INT16_C( 14582),  INT16_C( 24537),  INT16_C( 15760) },
      {  INT32_C(   314127506), -INT32_C(   590739145), -INT32_C(   335194014),  INT32_C(  1777637581) } },
    { { -INT16_C( 24973), -INT16_C( 31120), -INT16_C( 23603),  INT16_C(  8205),  INT16_C(   404), -INT16_C(  5442), -INT16_C( 28347),  INT16_C( 26517) },
      UINT8_C(226),
      { -INT16_C( 23032), -INT16_C( 21113), -INT16_C( 20274), -INT16_C( 19635), -INT16_C( 31066),  INT16_C(  1420), -INT16_C( 13802), -INT16_C( 19336) },
      { -INT16_C(   198), -INT16_C(  8575), -INT16_C( 24052),  INT16_C(  3442),  INT16_C( 23648), -INT16_C(  3502), -INT16_C( 17935), -INT16_C(  1324) },
      { -INT32_C(  2039439757),  INT32_C(   420046578), -INT32_C(   356646508),  INT32_C(  1737855301) } },
    { {  INT16_C( 23391),  INT16_C( 11943), -INT16_C(  2805), -INT16_C( 19999),  INT16_C( 28283), -INT16_C( 28233),  INT16_C( 12088),  INT16_C( 29253) },
      UINT8_C( 46),
      {  INT16_C( 20679),  INT16_C( 26939),  INT16_C( 18626),  INT16_C(  7881), -INT16_C( 17509),  INT16_C( 21520),  INT16_C(  2703), -INT16_C(  5196) },
      { -INT16_C(  7503), -INT16_C( 22794), -INT16_C( 22333),  INT16_C( 12577), -INT16_C( 19873), -INT16_C( 29079), -INT16_C(  8968), -INT16_C( 16451) },
      {  INT32_C(   782719839), -INT32_C(   316855121), -INT32_C(   277823723),  INT32_C(    61238892) } },
    { { -INT16_C(  2004), -INT16_C(  4312), -INT16_C(  3776), -INT16_C(  9459),  INT16_C(  7597),  INT16_C( 15408), -INT16_C(  7129), -INT16_C(  9945) },
      UINT8_C(198),
      {  INT16_C( 32542), -INT16_C( 14711), -INT16_C( 17503),  INT16_C( 21285), -INT16_C( 19676),  INT16_C(    75),  INT16_C(  2672),  INT16_C( 26669) },
      {  INT16_C(  7218),  INT16_C(  9385), -INT16_C( 31703),  INT16_C( 18385),  INT16_C(  3508), -INT16_C( 26514),  INT16_C( 18229),  INT16_C( 21342) },
      { -INT32_C(   282527700),  INT32_C(   946222334), -INT32_C(    71011958), -INT32_C(   651697113) } },
    { { -INT16_C(  5945),  INT16_C( 26649),  INT16_C( 16035), -INT16_C( 14405),  INT16_C(  2033),  INT16_C( 25288), -INT16_C(  2799),  INT16_C( 17610) },
      UINT8_C( 17),
      {  INT16_C( 26739), -INT16_C(  1990), -INT16_C( 32455),  INT16_C( 18092),  INT16_C( 17904),  INT16_C( 14203), -INT16_C( 12637), -INT16_C( 29698) },
      {  INT16_C( 26343),  INT16_C(  9518), -INT16_C(  2526),  INT16_C( 10519),  INT16_C( 31166), -INT16_C( 19654),  INT16_C( 32323), -INT16_C( 18492) },
      {  INT32_C(   685444657), -INT32_C(   944030045),  INT32_C(  1657276401),  INT32_C(  1154151697) } },
    { { -INT16_C(   282),  INT16_C(  8111),  INT16_C( 23424),  INT16_C( 28774), -INT16_C(  7776),  INT16_C( 17575), -INT16_C( 22864), -INT16_C( 26673) },
      UINT8_C( 12),
      { -INT16_C( 16898), -INT16_C(  3026),  INT16_C( 22484),  INT16_C( 19890),  INT16_C( 26002),  INT16_C(  4240),  INT16_C( 18217),  INT16_C( 10231) },
      {  INT16_C(  5878),  INT16_C( 21159),  INT16_C(  6012),  INT16_C( 24306),  INT16_C( 14015),  INT16_C( 25870), -INT16_C( 23290),  INT16_C(  1137) },
      {  INT32_C(   531627750),  INT32_C(  1885756288),  INT32_C(   474106830), -INT32_C(   412641283) } },
    { { -INT16_C( 24478),  INT16_C( 14072), -INT16_C( 21769), -INT16_C( 30333),  INT16_C(  5135),  INT16_C( 14490), -INT16_C( 28325),  INT16_C( 21087) },
      UINT8_C(167),
      { -INT16_C( 23545),  INT16_C(  7716), -INT16_C( 32106), -INT16_C( 12835),  INT16_C( 17040),  INT16_C( 13779), -INT16_C( 10316),  INT16_C( 21656) },
      { -INT16_C( 12593),  INT16_C( 31051), -INT16_C( 10926),  INT16_C( 26248), -INT16_C( 16273),  INT16_C(   193),  INT16_C(  4895),  INT16_C(  9895) },
      {  INT32_C(   536091701),  INT32_C(    13897076), -INT32_C(   274632573),  INT32_C(  1381994843) } },
    { { -INT16_C( 13385),  INT16_C( 20037),  INT16_C(  8781), -INT16_C(  8933), -INT16_C(  4507),  INT16_C(  6419), -INT16_C( 21563), -INT16_C( 27539) },
      UINT8_C(121),
      {  INT16_C(  3512), -INT16_C( 29237),  INT16_C( 12693),  INT16_C( 22012), -INT16_C(   781),  INT16_C(  1652), -INT16_C( 25692),  INT16_C( 28606) },
      {  INT16_C(  3296),  INT16_C(   701), -INT16_C( 26073),  INT16_C(  5479), -INT16_C( 32595),  INT16_C( 22746),  INT16_C( 28397), -INT16_C( 22830) },
      { -INT32_C(     8919585), -INT32_C(   585424307),  INT32_C(   420736613), -INT32_C(  1382650704) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i src = simde_mm_loadu_epi16(test_vec[i].src);
    simde__m128i a = simde_mm_loadu_epi16(test_vec[i].a);
    simde__m128i b = simde_mm_loadu_epi16(test_vec[i].b);
    simde__m128i r = simde_mm_mask_madd_epi16(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x4(r, simde_mm_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm_maskz_madd_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask8 k;
    const int16_t a[8];
    const int16_t b[8];
    const int32_t r[4];
  } test_vec[] = {
    { UINT8_C( 77),
      {  INT16_C( 19733),  INT16_C( 18694), -INT16_C(  2158), -INT16_C( 12949), -INT16_C( 22664), -INT16_C(  3805),  INT16_C( 30779),  INT16_C(  1600) },
      {  INT16_C( 11918),  INT16_C(  7268),  INT16_C( 32199),  INT16_C( 14806), -INT16_C( 14152), -INT16_C( 23986),  INT16_C( 16941),  INT16_C( 17135) },
      {  INT32_C(   371045886),  INT32_C(           0),  INT32_C(   412007658),  INT32_C(   548843039) } },
    { UINT8_C(143),
      { -INT16_C( 29707), -INT16_C(  5087), -INT16_C(  4361), -INT16_C( 24987),  INT16_C( 22033), -INT16_C( 30247), -INT16_C(  8041), -INT16_C( 15080) },
      {  INT16_C( 13380), -INT16_C( 15988), -INT16_C( 15094), -INT16_C( 11398),  INT16_C(  7187),  INT16_C( 21760),  INT16_C( 17163),  INT16_C(   228) },
      { -INT32_C(   316148704),  INT32_C(   350626760), -INT32_C(   499823549), -INT32_C(   141445923) } },
    { UINT8_C(206),
      { -INT16_C(  5115), -INT16_C(  3131),  INT16_C( 25681), -INT16_C( 22524), -INT16_C( 29123),  INT16_C(  7487),  INT16_C(  1190), -INT16_C(  9631) },
      {  INT16_C(  9105),  INT16_C( 22244), -INT16_C( 18531), -INT16_C( 18070), -INT16_C( 16456), -INT16_C(  1084), -INT16_C( 15196), -INT16_C( 22071) },
      {  INT32_C(           0), -INT32_C(    68885931),  INT32_C(   471132180),  INT32_C(   194482561) } },
    { UINT8_C(176),
      { -INT16_C( 25201), -INT16_C(  3326), -INT16_C( 21855),  INT16_C( 12080),  INT16_C( 20201), -INT16_C(  4651), -INT16_C( 20561), -INT16_C( 11650) },
      { -INT16_C( 10860),  INT16_C( 19311),  INT16_C( 10303), -INT16_C(   509), -INT16_C(   276), -INT16_C( 20318),  INT16_C( 19656),  INT16_C( 22369) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) } },
    { UINT8_C(233),
      {  INT16_C( 19043),  INT16_C(  3466), -INT16_C( 17798), -INT16_C( 14090), -INT16_C(  7281),  INT16_C( 16248),  INT16_C( 19042),  INT16_C( 14291) },
      {  INT16_C(  7866), -INT16_C(  7562),  INT16_C( 29730),  INT16_C(  8399),  INT16_C( 32535),  INT16_C( 25576),  INT16_C( 16352),  INT16_C( 17228) },
      {  INT32_C(   123582346),  INT32_C(           0),  INT32_C(           0),  INT32_C(   557580132) } },
    { UINT8_C(137),
      {  INT16_C( 20694), -INT16_C( 28668), -INT16_C( 13242),  INT16_C( 10784),  INT16_C( 24388), -INT16_C( 28788), -INT16_C( 15566),  INT16_C( 20553) },
      {  INT16_C( 11065), -INT16_C( 21134), -INT16_C( 27654),  INT16_C( 31428),  INT16_C( 10107), -INT16_C( 17574), -INT16_C( 24973),  INT16_C( 19012) },
      {  INT32_C(   834848622),  INT32_C(           0),  INT32_C(           0),  INT32_C(   779483354) } },
    { UINT8_C(238),
      { -INT16_C(  9656),  INT16_C(  5429),  INT16_C( 24570),  INT16_C( 22873), -INT16_C(  5909), -INT16_C( 20853), -INT16_C(  9167),  INT16_C( 24039) },
      { -INT16_C( 27570), -INT16_C(  7849), -INT16_C( 11943), -INT16_C( 32675),  INT16_C(  6188), -INT16_C( 13580),  INT16_C( 15964), -INT16_C( 23112) },
      {  INT32_C(           0), -INT32_C(  1040814785),  INT32_C(   246618848), -INT32_C(   701931356) } },
    { UINT8_C( 24),
      { -INT16_C( 17683),  INT16_C( 19475),  INT16_C( 27667), -INT16_C(   969), -INT16_C(  6664), -INT16_C( 11219), -INT16_C( 30004),  INT16_C( 24866) },
      {  INT16_C(  1250), -INT16_C( 19526),  INT16_C( 14945),  INT16_C( 31199), -INT16_C( 22226),  INT16_C( 27861),  INT16_C( 31330),  INT16_C( 20357) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   433828158) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i a = simde_x_mm_loadu_epi16(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi16(test_vec[i].b);
    simde__m128i r = simde_mm_maskz_madd_epi16(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x4(r, simde_x_mm_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_mask_madd_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int16_t src[16];
    const simde__mmask8 k;
    const int16_t a[16];
    const int16_t b[16];
    const int32_t r[8];
  } test_vec[] = {
    { {  INT16_C( 16074), -INT16_C( 23692), -INT16_C( 14496),  INT16_C( 27418), -INT16_C( 12637),  INT16_C( 14730),  INT16_C( 24645),  INT16_C(  3541),
        -INT16_C( 19839),  INT16_C(  7216),  INT16_C( 21625), -INT16_C(  4144),  INT16_C( 21389),  INT16_C( 15792),  INT16_C(  9810),  INT16_C(  7279) },
      UINT8_C(100),
      { -INT16_C( 16412), -INT16_C( 21564),  INT16_C( 12250), -INT16_C( 22449), -INT16_C( 30535),  INT16_C(  6637), -INT16_C(  1187),  INT16_C(  4251),
        -INT16_C( 18645), -INT16_C( 32631),  INT16_C( 30855), -INT16_C(  9715),  INT16_C( 18984),  INT16_C( 20012),  INT16_C( 18874), -INT16_C( 24910) },
      {  INT16_C( 30472), -INT16_C(  7607), -INT16_C( 26458),  INT16_C( 24714),  INT16_C( 30752),  INT16_C( 32377),  INT16_C(  5235), -INT16_C( 24946),
         INT16_C(  6091),  INT16_C( 21022),  INT16_C( 11151), -INT16_C( 18643),  INT16_C( 22902),  INT16_C( 12294), -INT16_C( 18270), -INT16_C( 21554) },
      { -INT32_C(  1552662838),  INT32_C(  1796917088), -INT32_C(   724126171),  INT32_C(   232087621),  INT32_C(   472953473),  INT32_C(   525180850),  INT32_C(   680799096),  INT32_C(   477046354) } },
    { {  INT16_C(  5935), -INT16_C( 10611),  INT16_C(  6320), -INT16_C( 12234), -INT16_C( 20592),  INT16_C(   846), -INT16_C(  9020), -INT16_C( 28767),
        -INT16_C( 16141), -INT16_C( 32030),  INT16_C(  4075),  INT16_C( 24890),  INT16_C( 16488),  INT16_C(  2961),  INT16_C( 24568),  INT16_C( 10422) },
      UINT8_C(119),
      { -INT16_C(   445),  INT16_C( 23335), -INT16_C(  2252), -INT16_C(  7189), -INT16_C(  4538),  INT16_C(  8871),  INT16_C( 14224),  INT16_C( 20502),
        -INT16_C( 26599),  INT16_C( 10299), -INT16_C( 25134),  INT16_C(  4752), -INT16_C( 25810), -INT16_C( 29173),  INT16_C( 13137), -INT16_C( 27387) },
      {  INT16_C( 11313),  INT16_C( 26096), -INT16_C(  9181),  INT16_C( 26952), -INT16_C(  3894),  INT16_C( 23180), -INT16_C( 24025),  INT16_C( 16554),
        -INT16_C(  6598),  INT16_C(  3432), -INT16_C(  1917), -INT16_C( 20193),  INT16_C( 10900), -INT16_C(  6849),  INT16_C( 17501), -INT16_C( 29062) },
      {  INT32_C(   603915875), -INT32_C(   173082316),  INT32_C(   223300752), -INT32_C(  1885217596),  INT32_C(   210846370), -INT32_C(    47775258), -INT32_C(    81523123),  INT32_C(   683040760) } },
    { {  INT16_C( 27504), -INT16_C( 27405),  INT16_C( 15431),  INT16_C(  4605), -INT16_C( 30420),  INT16_C( 21356),  INT16_C(  5675),  INT16_C( 26259),
        -INT16_C(  1028),  INT16_C( 32627), -INT16_C( 27917), -INT16_C( 30927),  INT16_C( 28861),  INT16_C(  6765), -INT16_C(  6219),  INT16_C(  9641) },
      UINT8_C( 82),
      { -INT16_C( 18020), -INT16_C( 10087), -INT16_C( 21577),  INT16_C( 16388),  INT16_C( 22295),  INT16_C( 11628), -INT16_C( 11542), -INT16_C(  6870),
        -INT16_C( 22203), -INT16_C( 10279),  INT16_C( 24794),  INT16_C( 19348), -INT16_C( 20531), -INT16_C( 19200),  INT16_C(  9560), -INT16_C(  3065) },
      { -INT16_C( 24097), -INT16_C( 26931), -INT16_C( 11956),  INT16_C( 25558),  INT16_C( 16937),  INT16_C(  5008), -INT16_C( 17900),  INT16_C( 23033),
        -INT16_C( 11676),  INT16_C( 15921), -INT16_C( 15054),  INT16_C(   137), -INT16_C( 30348), -INT16_C( 13131), -INT16_C( 17233), -INT16_C( 28991) },
      { -INT32_C(  1795986576),  INT32_C(   676819116),  INT32_C(  1399621932),  INT32_C(  1720915499),  INT32_C(    95590269), -INT32_C(  2026794253),  INT32_C(   875189988),  INT32_C(   631891893) } },
    { { -INT16_C( 29091), -INT16_C( 22236), -INT16_C(  1441), -INT16_C( 30708), -INT16_C( 25283),  INT16_C( 20892), -INT16_C( 27305), -INT16_C( 17493),
        -INT16_C(  9113), -INT16_C( 26118), -INT16_C( 31839),  INT16_C(  5785),  INT16_C( 19981), -INT16_C( 17182), -INT16_C( 23797),  INT16_C( 26698) },
      UINT8_C( 49),
      {  INT16_C(  4718),  INT16_C( 26769),  INT16_C(  6430), -INT16_C( 17499), -INT16_C(  2123),  INT16_C( 18963), -INT16_C( 12638),  INT16_C( 32433),
         INT16_C( 19400),  INT16_C( 19487),  INT16_C( 13796),  INT16_C( 13145),  INT16_C(  5400), -INT16_C( 17602), -INT16_C( 22945), -INT16_C( 12819) },
      {  INT16_C( 32440), -INT16_C( 10443), -INT16_C(  9321),  INT16_C( 19858), -INT16_C( 23086),  INT16_C( 29847),  INT16_C( 18804),  INT16_C( 15602),
         INT16_C(  4500),  INT16_C( 30856), -INT16_C(  7865),  INT16_C( 24491), -INT16_C(  5642),  INT16_C( 21786),  INT16_C(  1936),  INT16_C( 18466) },
      { -INT32_C(   126496747), -INT32_C(  2012415393),  INT32_C(  1369218365), -INT32_C(  1146383017),  INT32_C(   688590872),  INT32_C(   213428655), -INT32_C(  1126019571),  INT32_C(  1749721867) } },
    { {  INT16_C( 22661),  INT16_C(  7455), -INT16_C( 19917),  INT16_C(  1386),  INT16_C(   343), -INT16_C( 13447),  INT16_C( 27466), -INT16_C(  8696),
        -INT16_C( 28548), -INT16_C( 15529),  INT16_C(   626),  INT16_C( 26658),  INT16_C( 15852),  INT16_C( 31934), -INT16_C(  8124), -INT16_C( 13628) },
      UINT8_C( 56),
      { -INT16_C(  6172), -INT16_C( 27029),  INT16_C( 28753),  INT16_C( 21229), -INT16_C( 17943),  INT16_C( 21661),  INT16_C( 31681),  INT16_C( 20945),
        -INT16_C( 27438), -INT16_C( 10813),  INT16_C( 11447), -INT16_C(  2879),  INT16_C( 15850), -INT16_C( 13768),  INT16_C(   513), -INT16_C(  6909) },
      {  INT16_C( 28393),  INT16_C( 14971),  INT16_C( 27103), -INT16_C( 14195),  INT16_C( 10786), -INT16_C(  7395), -INT16_C(  4443),  INT16_C( 30772),
        -INT16_C(  1918),  INT16_C( 14669),  INT16_C(  3620),  INT16_C(  3629),  INT16_C( 26187),  INT16_C( 19672), -INT16_C(  9368),  INT16_C( 21042) },
      {  INT32_C(   488593541),  INT32_C(    90878515), -INT32_C(   881262249),  INT32_C(   503760857), -INT32_C(   105989813),  INT32_C(    30990249),  INT32_C(  2092842476), -INT32_C(   893067196) } },
    { { -INT16_C( 21174),  INT16_C( 10636),  INT16_C(  6422),  INT16_C( 14577),  INT16_C(  3651), -INT16_C(  5861),  INT16_C( 20732),  INT16_C( 32609),
        -INT16_C( 20920),  INT16_C( 27832), -INT16_C(  6468),  INT16_C(  1914),  INT16_C( 21068), -INT16_C( 19373), -INT16_C( 31442),  INT16_C( 30726) },
      UINT8_C( 51),
      { -INT16_C( 24173), -INT16_C( 21431), -INT16_C( 32110), -INT16_C( 24080), -INT16_C(  9827), -INT16_C(  4707),  INT16_C(  7226), -INT16_C(  6091),
        -INT16_C( 24107), -INT16_C( 17500), -INT16_C( 21733),  INT16_C( 28167), -INT16_C( 17410), -INT16_C( 31588),  INT16_C(  5314),  INT16_C( 21943) },
      {  INT16_C(   181),  INT16_C( 18177), -INT16_C(  3710),  INT16_C(  8424), -INT16_C( 31030),  INT16_C(  1037),  INT16_C( 17314),  INT16_C( 30700),
        -INT16_C( 28444),  INT16_C(    50),  INT16_C( 14651),  INT16_C( 14958),  INT16_C(  2805), -INT16_C( 18498),  INT16_C( 29982), -INT16_C( 11508) },
      { -INT32_C(   393926600), -INT32_C(    83721820), -INT32_C(   384102845),  INT32_C(  2137084156),  INT32_C(   684824508),  INT32_C(   102911803), -INT32_C(  1269607860),  INT32_C(  2013693230) } },
    { {  INT16_C(  3445), -INT16_C(  2022),  INT16_C(  1023), -INT16_C( 14056),  INT16_C(  9609),  INT16_C( 11214), -INT16_C( 17816),  INT16_C( 19875),
        -INT16_C( 10933), -INT16_C( 31155), -INT16_C( 17649),  INT16_C(  1216),  INT16_C( 32453), -INT16_C(  7237), -INT16_C( 14349),  INT16_C( 27062) },
      UINT8_C(212),
      {  INT16_C( 25040), -INT16_C( 11309), -INT16_C( 25223), -INT16_C( 24996), -INT16_C( 30613),  INT16_C(  9479),  INT16_C( 21547),  INT16_C(   112),
        -INT16_C(  2143),  INT16_C( 23567),  INT16_C(  5047),  INT16_C( 13857),  INT16_C(  1230), -INT16_C( 27351), -INT16_C( 27974), -INT16_C( 30102) },
      {  INT16_C( 15859),  INT16_C( 27742), -INT16_C( 17702),  INT16_C( 17675),  INT16_C(  4674),  INT16_C( 28011), -INT16_C(  9370),  INT16_C(  1902),
         INT16_C( 32210), -INT16_C( 30109), -INT16_C( 31599),  INT16_C( 24512), -INT16_C(  5752),  INT16_C( 17141),  INT16_C( 24444),  INT16_C( 28620) },
      { -INT32_C(   132510347), -INT32_C(   921172993),  INT32_C(   122431107),  INT32_C(  1302575720), -INT32_C(   778604833),  INT32_C(    79739663), -INT32_C(   475898451), -INT32_C(  1545315696) } },
    { {  INT16_C( 10908),  INT16_C( 30684), -INT16_C(  6171),  INT16_C( 10172),  INT16_C( 10233),  INT16_C( 24469),  INT16_C(   771), -INT16_C( 10906),
        -INT16_C( 13952),  INT16_C(  4447),  INT16_C(  8013), -INT16_C( 10895),  INT16_C( 26121), -INT16_C( 31465), -INT16_C(  7227),  INT16_C( 25076) },
      UINT8_C( 14),
      { -INT16_C( 10032), -INT16_C( 18445),  INT16_C(  6805), -INT16_C( 17232),  INT16_C(  4015), -INT16_C( 19777), -INT16_C( 27275),  INT16_C( 15923),
         INT16_C( 17652),  INT16_C(  5259),  INT16_C( 24757),  INT16_C(  6941), -INT16_C( 23945),  INT16_C( 23520),  INT16_C( 17046),  INT16_C( 26473) },
      {  INT16_C( 23578), -INT16_C( 20706), -INT16_C( 12426),  INT16_C(  9836),  INT16_C( 11230),  INT16_C( 21720),  INT16_C(  3008), -INT16_C( 19054),
         INT16_C(  7760),  INT16_C(  1481), -INT16_C(  6530), -INT16_C(  2527),  INT16_C(   392),  INT16_C(  7761), -INT16_C( 17853),  INT16_C( 24197) },
      {  INT32_C(  2010917532), -INT32_C(   254052882), -INT32_C(   384467990), -INT32_C(   385440042),  INT32_C(   291490176), -INT32_C(   714006707), -INT32_C(  2062064119),  INT32_C(  1643439045) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i src = simde_x_mm256_loadu_epi16(test_vec[i].src);
    simde__m256i a = simde_x_mm256_loadu_epi16(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi16(test_vec[i].b);
    simde__m256i r = simde_mm256_mask_madd_epi16(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x8(r, simde_x_mm256_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_maskz_madd_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask8 k;
    const int16_t a[16];
    const int16_t b[16];
    const int32_t r[8];
  } test_vec[] = {
    { UINT8_C(217),
      { -INT16_C( 16454), -INT16_C(  5745), -INT16_C( 10179),  INT16_C( 22530),  INT16_C( 26767), -INT16_C( 17839), -INT16_C( 21048), -INT16_C(   156),
         INT16_C(  7387),  INT16_C(  4894), -INT16_C(  9375),  INT16_C( 29558),  INT16_C(  5931), -INT16_C( 28270),  INT16_C(  6388), -INT16_C( 20885) },
      { -INT16_C(  1321),  INT16_C(  5271), -INT16_C( 26158),  INT16_C( 24940), -INT16_C( 17151), -INT16_C( 14053), -INT16_C( 32661),  INT16_C( 18121),
        -INT16_C(  6244), -INT16_C(   679), -INT16_C( 12350), -INT16_C(  4495),  INT16_C(   998), -INT16_C(  9601), -INT16_C(  5605), -INT16_C(  3448) },
      { -INT32_C(     8546161),  INT32_C(           0),  INT32_C(           0),  INT32_C(   684621852), -INT32_C(    49447454),  INT32_C(           0),  INT32_C(   277339408),  INT32_C(    36206740) } },
    { UINT8_C(228),
      {  INT16_C(  1568), -INT16_C( 17993),  INT16_C(  6259),  INT16_C( 12475), -INT16_C( 31692), -INT16_C( 19301), -INT16_C(  7859),  INT16_C( 13392),
         INT16_C( 20026),  INT16_C(  2807), -INT16_C(  6721), -INT16_C( 15632), -INT16_C( 13468),  INT16_C( 20445), -INT16_C( 12461),  INT16_C( 29491) },
      { -INT16_C(  5419),  INT16_C( 18477), -INT16_C(  6141),  INT16_C( 14201),  INT16_C(  5228), -INT16_C( 17685),  INT16_C( 15350),  INT16_C( 12526),
        -INT16_C(  6775),  INT16_C( 18490),  INT16_C( 11210),  INT16_C( 12042), -INT16_C(  6154),  INT16_C( 18814), -INT16_C( 20042), -INT16_C( 29507) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(   175652409),  INT32_C(           0),  INT32_C(           0), -INT32_C(   263582954),  INT32_C(   467534302), -INT32_C(   620447575) } },
    { UINT8_C(156),
      { -INT16_C( 11030), -INT16_C( 11617), -INT16_C( 10675),  INT16_C( 25150), -INT16_C(  1855), -INT16_C(   936), -INT16_C( 30489), -INT16_C( 13178),
        -INT16_C( 12605), -INT16_C(  4457), -INT16_C( 14631), -INT16_C( 16156),  INT16_C( 11588), -INT16_C(  2697),  INT16_C(  1002), -INT16_C( 11119) },
      {  INT16_C( 12503),  INT16_C(  9638), -INT16_C(  6906), -INT16_C( 14457), -INT16_C(  8227), -INT16_C( 15164),  INT16_C( 19047),  INT16_C( 10897),
         INT16_C( 10264), -INT16_C(  3816), -INT16_C(   786),  INT16_C( 12978),  INT16_C( 10538),  INT16_C(  5159), -INT16_C( 18132),  INT16_C(  1001) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(    29454589), -INT32_C(   724324649), -INT32_C(   112369808),  INT32_C(           0),  INT32_C(           0), -INT32_C(    29298383) } },
    { UINT8_C(233),
      {  INT16_C( 10383),  INT16_C( 29936), -INT16_C( 18513), -INT16_C( 29102),  INT16_C(  5755), -INT16_C( 14858),  INT16_C(  8359), -INT16_C( 12322),
        -INT16_C( 12487),  INT16_C( 13757), -INT16_C(  4223), -INT16_C( 21921),  INT16_C( 29719), -INT16_C( 12074), -INT16_C(  9635), -INT16_C(  4935) },
      { -INT16_C( 22270), -INT16_C( 19871), -INT16_C( 19615), -INT16_C(  9152),  INT16_C( 14025),  INT16_C( 29090), -INT16_C( 32681), -INT16_C( 28608),
        -INT16_C(   433), -INT16_C( 11835),  INT16_C(  9709),  INT16_C(  1147),  INT16_C( 21145), -INT16_C(  2348), -INT16_C( 29140),  INT16_C( 12002) },
      { -INT32_C(   826087666),  INT32_C(           0),  INT32_C(           0),  INT32_C(    79327297),  INT32_C(           0), -INT32_C(    66144494),  INT32_C(   656758007),  INT32_C(   221534030) } },
    { UINT8_C( 55),
      { -INT16_C(  8125), -INT16_C(  2408),  INT16_C( 29985),  INT16_C( 22464),  INT16_C( 12567), -INT16_C( 26706),  INT16_C( 15985),  INT16_C( 28646),
        -INT16_C( 18684),  INT16_C( 10589),  INT16_C( 24883), -INT16_C( 31294), -INT16_C( 18378), -INT16_C( 15183), -INT16_C(  8294), -INT16_C(  8453) },
      { -INT16_C( 27456), -INT16_C(  7724), -INT16_C( 27639),  INT16_C(  8248), -INT16_C(  6203),  INT16_C( 14263), -INT16_C( 25307),  INT16_C( 10662),
         INT16_C(   853), -INT16_C( 30638),  INT16_C(  5221), -INT16_C( 25843), -INT16_C( 16692),  INT16_C( 26463),  INT16_C( 23197),  INT16_C( 23877) },
      {  INT32_C(   241679392), -INT32_C(   643472343), -INT32_C(   458860779),  INT32_C(           0), -INT32_C(   340363234),  INT32_C(   938644985),  INT32_C(           0),  INT32_C(           0) } },
    { UINT8_C(238),
      {  INT16_C( 15897), -INT16_C( 20745),  INT16_C(  6007),  INT16_C( 24179), -INT16_C( 21810),  INT16_C( 27779), -INT16_C( 21167),  INT16_C( 21697),
         INT16_C( 18943),  INT16_C(  5305),  INT16_C( 21590),  INT16_C(  5344),  INT16_C( 18355),  INT16_C(  3761),  INT16_C(  3980), -INT16_C( 22788) },
      { -INT16_C(  2995), -INT16_C( 15276), -INT16_C( 14581), -INT16_C(  9694), -INT16_C( 22926), -INT16_C( 15546),  INT16_C(  1875),  INT16_C( 21015),
        -INT16_C( 11952), -INT16_C( 22938),  INT16_C( 18213), -INT16_C(  9798),  INT16_C( 27534),  INT16_C(  7143), -INT16_C(  7302), -INT16_C( 14143) },
      {  INT32_C(           0), -INT32_C(   321979293),  INT32_C(    68163726),  INT32_C(   416274330),  INT32_C(           0),  INT32_C(   340858158),  INT32_C(   532251393),  INT32_C(   293228724) } },
    { UINT8_C(215),
      { -INT16_C( 29675), -INT16_C(  8989), -INT16_C( 16977),  INT16_C( 21838),  INT16_C(  4355),  INT16_C(  2728), -INT16_C(  1495), -INT16_C(  1446),
         INT16_C(    97), -INT16_C( 22497), -INT16_C(  1862),  INT16_C(  9526),  INT16_C( 20959), -INT16_C( 15456),  INT16_C( 26642),  INT16_C( 10138) },
      {  INT16_C( 32244), -INT16_C( 23804),  INT16_C( 21050),  INT16_C( 15864), -INT16_C( 24476), -INT16_C( 29369), -INT16_C( 24165), -INT16_C(   889),
        -INT16_C( 22879),  INT16_C( 23460), -INT16_C(  9569),  INT16_C( 32385),  INT16_C(  8492),  INT16_C( 15937), -INT16_C(  9079),  INT16_C( 32102) },
      { -INT32_C(   742866544), -INT32_C(    10927818), -INT32_C(   186711612),  INT32_C(           0), -INT32_C(   529998883),  INT32_C(           0), -INT32_C(    68338444),  INT32_C(    83567358) } },
    { UINT8_C( 89),
      {  INT16_C(  8554), -INT16_C( 17260), -INT16_C( 12007), -INT16_C( 17888), -INT16_C( 21223), -INT16_C( 17835),  INT16_C( 20788), -INT16_C(  9380),
        -INT16_C( 18443), -INT16_C( 12422), -INT16_C(  1992),  INT16_C( 23035),  INT16_C( 14906),  INT16_C(  5858),  INT16_C( 24736),  INT16_C(  2671) },
      {  INT16_C(   897), -INT16_C( 25914), -INT16_C(  6187), -INT16_C(  4524), -INT16_C( 22124), -INT16_C( 13912),  INT16_C(  1274), -INT16_C(  4188),
         INT16_C(  7868), -INT16_C(  2881), -INT16_C( 17898),  INT16_C( 20558),  INT16_C( 12532), -INT16_C( 27546), -INT16_C( 10608),  INT16_C(  4510) },
      {  INT32_C(   454948578),  INT32_C(           0),  INT32_C(           0),  INT32_C(    65767352), -INT32_C(   109321742),  INT32_C(           0),  INT32_C(    25437524),  INT32_C(           0) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_mm256_loadu_epi16(test_vec[i].a);
    simde__m256i b = simde_mm256_loadu_epi16(test_vec[i].b);
    simde__m256i r = simde_mm256_maskz_madd_epi16(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x8(r, simde_mm256_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_madd_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int16_t a[32];
    const int16_t b[32];
    const int32_t r[16];
  } test_vec[] = {
    { {  INT16_C( 28000),  INT16_C( 31877), -INT16_C( 10728),  INT16_C( 28413), -INT16_C(  7321), -INT16_C( 24759),  INT16_C( 27401),  INT16_C(  4695),
        -INT16_C(  5593), -INT16_C(  3334), -INT16_C( 10804),  INT16_C( 22250),  INT16_C( 24880), -INT16_C( 25050),  INT16_C(  4800),  INT16_C(  8447),
        -INT16_C( 31616), -INT16_C( 26467), -INT16_C( 26021), -INT16_C( 15866),  INT16_C( 20349), -INT16_C( 31135), -INT16_C( 18245), -INT16_C(  7528),
        -INT16_C( 27997),  INT16_C( 28629), -INT16_C( 16537), -INT16_C( 26427), -INT16_C(  5344), -INT16_C(  7882),  INT16_C( 13821),  INT16_C( 32001) },
      { -INT16_C( 24903),  INT16_C(  5142),  INT16_C(  7224), -INT16_C( 18986),  INT16_C( 14188),  INT16_C( 10043), -INT16_C( 11280), -INT16_C( 27895),
        -INT16_C(  8602), -INT16_C( 13054), -INT16_C( 14435), -INT16_C( 16795), -INT16_C( 25678), -INT16_C( 20577), -INT16_C( 24368), -INT16_C( 30163),
         INT16_C( 17215),  INT16_C( 30622),  INT16_C( 30047), -INT16_C( 13523),  INT16_C( 26796), -INT16_C( 25358), -INT16_C(   964), -INT16_C( 24017),
         INT16_C( 12762),  INT16_C( 30831), -INT16_C( 10760), -INT16_C( 21962), -INT16_C( 10896),  INT16_C( 16730), -INT16_C( 30859), -INT16_C( 19253) },
      { -INT32_C(   533372466), -INT32_C(   616948290), -INT32_C(   352524985), -INT32_C(   440050305),  INT32_C(    91633022), -INT32_C(   217733010), -INT32_C(   123414790), -INT32_C(   371753261),
        -INT32_C(  1354741914), -INT32_C(   567297069),  INT32_C(  1334793134),  INT32_C(   198388156),  INT32_C(   525362985),  INT32_C(   758327894), -INT32_C(    73637636), -INT32_C(  1042617492) } },
    { {  INT16_C( 27082),  INT16_C( 10540),  INT16_C( 23006), -INT16_C( 29707), -INT16_C(  6207), -INT16_C(   729),  INT16_C( 22499), -INT16_C( 16737),
         INT16_C(  3976), -INT16_C( 32458),  INT16_C( 27876),  INT16_C( 21547), -INT16_C( 31423), -INT16_C( 18795),  INT16_C( 24588), -INT16_C( 10645),
        -INT16_C( 26678), -INT16_C( 22528), -INT16_C(  2576), -INT16_C( 20173),  INT16_C( 23516), -INT16_C( 16209),  INT16_C( 20146),  INT16_C( 14974),
        -INT16_C( 19363),  INT16_C( 16827), -INT16_C(  6368),  INT16_C( 24982),  INT16_C( 11116),  INT16_C( 30999), -INT16_C( 32116),  INT16_C( 22095) },
      {  INT16_C( 20249),  INT16_C(  2558),  INT16_C( 12868),  INT16_C(  8635),  INT16_C( 27277),  INT16_C( 16353),  INT16_C( 24504),  INT16_C(  5753),
         INT16_C( 13587),  INT16_C( 13143), -INT16_C(  4836), -INT16_C( 30572), -INT16_C( 21735), -INT16_C( 23295),  INT16_C( 20782),  INT16_C( 18427),
        -INT16_C(  1632), -INT16_C(  6831),  INT16_C(  3115), -INT16_C( 18426), -INT16_C(  6282),  INT16_C( 12023),  INT16_C( 28998),  INT16_C( 22852),
        -INT16_C( 25434), -INT16_C( 15732),  INT16_C(  8329), -INT16_C( 23990),  INT16_C( 19659), -INT16_C(  1721),  INT16_C( 17053),  INT16_C( 15681) },
      {  INT32_C(   575344738),  INT32_C(    39521263), -INT32_C(   181229676),  INT32_C(   455027535), -INT32_C(   372573582), -INT32_C(   793543220),  INT32_C(  1120808430),  INT32_C(   314832401),
         INT32_C(   197427264),  INT32_C(   363683458), -INT32_C(   342608319),  INT32_C(   926379556),  INT32_C(   227756178), -INT32_C(   652357252),  INT32_C(   165180165), -INT32_C(   201202453) } },
    { { -INT16_C( 28100),  INT16_C( 26402),  INT16_C( 10398),  INT16_C(  5152),  INT16_C(  5903),  INT16_C( 21826), -INT16_C( 30840),  INT16_C( 11950),
         INT16_C( 14883), -INT16_C( 21264),  INT16_C( 15194),  INT16_C(  9807), -INT16_C( 27001),  INT16_C(  9247),  INT16_C( 24793),  INT16_C(  5473),
        -INT16_C( 31502), -INT16_C( 28548), -INT16_C( 25428), -INT16_C( 17244), -INT16_C(  6220),  INT16_C( 15377), -INT16_C( 16274), -INT16_C( 28309),
         INT16_C( 23546),  INT16_C( 21821), -INT16_C( 29546),  INT16_C(  7547), -INT16_C( 26077), -INT16_C(   959), -INT16_C( 23557), -INT16_C(  4847) },
      { -INT16_C( 29401), -INT16_C( 11394),  INT16_C(  8746), -INT16_C(  8561), -INT16_C( 24311),  INT16_C( 30490), -INT16_C( 31391),  INT16_C( 23304),
         INT16_C( 18145),  INT16_C( 30640),  INT16_C( 11218), -INT16_C(  2667), -INT16_C( 10554), -INT16_C( 15887),  INT16_C(   633), -INT16_C( 24402),
         INT16_C( 11408), -INT16_C( 17804),  INT16_C(   847),  INT16_C( 22680), -INT16_C( 19804),  INT16_C(  1488), -INT16_C( 10184),  INT16_C(  6497),
         INT16_C(  4382), -INT16_C(  3696),  INT16_C(  9533),  INT16_C(   998), -INT16_C(  9988),  INT16_C( 30148),  INT16_C( 29402),  INT16_C( 27158) },
      {  INT32_C(   525343712),  INT32_C(    46834636),  INT32_C(   521966907),  INT32_C(  1246581240), -INT32_C(   381476925),  INT32_C(   144291023),  INT32_C(   138061465), -INT32_C(   117858177),
         INT32_C(   148893776), -INT32_C(   412631436),  INT32_C(   146061856), -INT32_C(    18189157),  INT32_C(    22528156), -INT32_C(   274130112),  INT32_C(   231545144), -INT32_C(   824257740) } },
    { { -INT16_C( 30049), -INT16_C(  4572), -INT16_C( 17267),  INT16_C( 12870),  INT16_C(  5743), -INT16_C( 22729), -INT16_C( 26385),  INT16_C(  3520),
         INT16_C( 20650), -INT16_C(  6146), -INT16_C(  6794),  INT16_C( 29418), -INT16_C( 20803), -INT16_C( 26649), -INT16_C(   736), -INT16_C( 16638),
         INT16_C(  9863),  INT16_C(  5549), -INT16_C(  2845),  INT16_C( 21063),  INT16_C( 32266), -INT16_C(  1543), -INT16_C( 18153), -INT16_C( 16121),
         INT16_C(  1289),  INT16_C( 32680), -INT16_C( 27926), -INT16_C( 22543), -INT16_C(  9920),  INT16_C( 24639),  INT16_C( 16854),  INT16_C( 24096) },
      { -INT16_C( 12953),  INT16_C( 19059), -INT16_C( 17727), -INT16_C( 13156), -INT16_C( 27336),  INT16_C( 20421), -INT16_C( 13234),  INT16_C( 22544),
        -INT16_C( 18222), -INT16_C( 17193), -INT16_C( 14006), -INT16_C( 30108), -INT16_C( 23646),  INT16_C( 30955),  INT16_C(  3044),  INT16_C( 19414),
         INT16_C( 18904), -INT16_C( 25962),  INT16_C( 12803),  INT16_C( 15462),  INT16_C( 11208),  INT16_C(  5771), -INT16_C( 25352), -INT16_C( 13714),
         INT16_C( 18004), -INT16_C( 24698), -INT16_C(  5617), -INT16_C( 20183),  INT16_C(  5261),  INT16_C( 28969),  INT16_C(    31), -INT16_C(  1859) },
      {  INT32_C(   302086949),  INT32_C(   136774389), -INT32_C(   621139557),  INT32_C(   428533970), -INT32_C(   270616122), -INT32_C(   790560380), -INT32_C(   333012057), -INT32_C(   325250516),
         INT32_C(    42387014),  INT32_C(   289251571),  INT32_C(   352732675),  INT32_C(   681298250), -INT32_C(   783923484),  INT32_C(   611845711),  INT32_C(   661578071), -INT32_C(    44271990) } },
    { {  INT16_C( 21321),  INT16_C( 19858), -INT16_C(  1915),  INT16_C( 19849),  INT16_C(  5155),  INT16_C(  7012), -INT16_C( 11600),  INT16_C(  1509),
         INT16_C( 27672),  INT16_C( 10148), -INT16_C( 12970), -INT16_C(  6952),  INT16_C(   738),  INT16_C(   341),  INT16_C(  4610),  INT16_C( 19449),
        -INT16_C( 29851), -INT16_C(  5224),  INT16_C(  8579), -INT16_C( 22728), -INT16_C( 25546), -INT16_C(  6462), -INT16_C( 22417), -INT16_C( 30741),
        -INT16_C( 28908),  INT16_C( 27311), -INT16_C( 30883),  INT16_C( 16206), -INT16_C( 23415), -INT16_C( 29888),  INT16_C( 15030),  INT16_C(  7383) },
      {  INT16_C( 28613),  INT16_C( 18695),  INT16_C( 16273), -INT16_C( 14352), -INT16_C( 19748),  INT16_C( 19373), -INT16_C( 26278),  INT16_C( 28370),
        -INT16_C( 32472), -INT16_C( 31271),  INT16_C(  9993), -INT16_C( 27964),  INT16_C(  1483), -INT16_C( 32226), -INT16_C(  2753),  INT16_C(  1182),
        -INT16_C( 23196), -INT16_C(  2739),  INT16_C( 15844), -INT16_C( 16196),  INT16_C( 27376),  INT16_C( 18955), -INT16_C(  8701),  INT16_C( 11193),
        -INT16_C( 28065),  INT16_C( 26801),  INT16_C( 30137), -INT16_C( 31237),  INT16_C(  6522), -INT16_C( 18169), -INT16_C( 23282),  INT16_C( 29374) },
      {  INT32_C(   981303083), -INT32_C(   316035643),  INT32_C(    34042536),  INT32_C(   347635130), -INT32_C(  1215903292),  INT32_C(    64796518), -INT32_C(     9894612),  INT32_C(    10297388),
         INT32_C(   706732332),  INT32_C(   504028364), -INT32_C(   821834506), -INT32_C(   149033696),  INT32_C(  1543265131), -INT32_C(  1436947793),  INT32_C(   390322442), -INT32_C(   133060218) } },
    { {  INT16_C(  2890),  INT16_C( 11880),  INT16_C(  9289),  INT16_C( 14831), -INT16_C(  1394), -INT16_C( 28285),  INT16_C( 15576),  INT16_C( 14525),
         INT16_C( 28366), -INT16_C( 30560), -INT16_C( 25629),  INT16_C( 24077),  INT16_C(  5300), -INT16_C( 15849), -INT16_C( 10823),  INT16_C(   821),
        -INT16_C( 25119),  INT16_C( 10801),  INT16_C(  8385),  INT16_C( 20579), -INT16_C(  6629), -INT16_C(  3103), -INT16_C( 25053), -INT16_C(  3797),
        -INT16_C( 13300), -INT16_C(  3975), -INT16_C( 31129),  INT16_C(  7246),  INT16_C( 26010),  INT16_C( 21470),  INT16_C(  4923),  INT16_C(  7254) },
      { -INT16_C( 30544),  INT16_C( 29254), -INT16_C( 22104), -INT16_C( 15422), -INT16_C( 23665), -INT16_C( 19785), -INT16_C(  7614),  INT16_C( 20132),
         INT16_C(  7598),  INT16_C(  5694), -INT16_C( 29532),  INT16_C( 15922),  INT16_C(  4338),  INT16_C( 11666), -INT16_C(  6108), -INT16_C( 11191),
        -INT16_C( 28816),  INT16_C(  6470),  INT16_C(  2104), -INT16_C( 14372), -INT16_C( 27732), -INT16_C(  4486),  INT16_C(  7798),  INT16_C(  9276),
         INT16_C( 31547), -INT16_C(  8390),  INT16_C( 27655), -INT16_C(  1762), -INT16_C( 20355), -INT16_C( 24282),  INT16_C( 28568),  INT16_C(  2421) },
      {  INT32_C(   259265360), -INT32_C(   434047738),  INT32_C(   592607735),  INT32_C(   173821636),  INT32_C(    41516228),  INT32_C(  1140229622), -INT32_C(   161903034),  INT32_C(    56919073),
         INT32_C(   793711574), -INT32_C(   278119348),  INT32_C(   197755486), -INT32_C(   230584266), -INT32_C(   386224850), -INT32_C(   873639947), -INT32_C(  1050768090),  INT32_C(   158202198) } },
    { { -INT16_C( 17154),  INT16_C( 13858), -INT16_C(   316),  INT16_C( 28926),  INT16_C( 30866),  INT16_C(  2142), -INT16_C( 25706), -INT16_C( 11988),
         INT16_C( 26390),  INT16_C(  7601), -INT16_C( 12333),  INT16_C( 20503),  INT16_C( 15743),  INT16_C(  6129),  INT16_C( 26541), -INT16_C( 21728),
         INT16_C( 16931), -INT16_C(  6174), -INT16_C(  8127), -INT16_C( 11432), -INT16_C( 18856), -INT16_C(  4389),  INT16_C(  1873),  INT16_C( 26559),
         INT16_C( 28782),  INT16_C( 17029), -INT16_C( 25537), -INT16_C( 16750), -INT16_C( 31527), -INT16_C( 31018), -INT16_C(  2325),  INT16_C(  3634) },
      {  INT16_C(  5177),  INT16_C( 31477),  INT16_C( 19956),  INT16_C( 19533),  INT16_C( 10244),  INT16_C( 21818), -INT16_C(  1745), -INT16_C( 24899),
         INT16_C( 17002), -INT16_C( 22048),  INT16_C( 29406), -INT16_C( 18584),  INT16_C( 16118), -INT16_C(  7874),  INT16_C( 28724),  INT16_C( 28143),
        -INT16_C(  6780),  INT16_C( 30951),  INT16_C( 13362),  INT16_C( 14020), -INT16_C(   420), -INT16_C( 29556),  INT16_C( 18935),  INT16_C( 24874),
         INT16_C(  2699),  INT16_C( 26891),  INT16_C( 29564),  INT16_C( 29472),  INT16_C( 24241), -INT16_C(  6828),  INT16_C( 17614),  INT16_C( 21075) },
      {  INT32_C(   347402008),  INT32_C(   558705462),  INT32_C(   362925460),  INT32_C(   343346182),  INT32_C(   281095932), -INT32_C(   743691950),  INT32_C(   205485928),  INT32_C(   150872580),
        -INT32_C(   305883654), -INT32_C(   268869614),  INT32_C(   137640804),  INT32_C(   696093821),  INT32_C(   535609457), -INT32_C(  1248631868), -INT32_C(   552455103),  INT32_C(    35634000) } },
    { {  INT16_C( 14889),  INT16_C( 23498), -INT16_C( 29073), -INT16_C( 13422),  INT16_C(  7820), -INT16_C( 31657), -INT16_C( 32409), -INT16_C(  3355),
        -INT16_C(  3957),  INT16_C(  2139),  INT16_C( 31587),  INT16_C(  5243), -INT16_C( 12326), -INT16_C( 22278),  INT16_C( 19731),  INT16_C( 15611),
        -INT16_C( 14969), -INT16_C(  2408),  INT16_C( 10836), -INT16_C(  7998),  INT16_C(  6472), -INT16_C( 20636),  INT16_C( 19099),  INT16_C(  9889),
        -INT16_C(   966), -INT16_C( 25042), -INT16_C( 22153),  INT16_C( 20914), -INT16_C( 21383), -INT16_C( 29446), -INT16_C(  2567), -INT16_C( 32311) },
      {  INT16_C( 25018),  INT16_C(  3703),  INT16_C( 14731), -INT16_C( 11281),  INT16_C( 21331), -INT16_C(  4478),  INT16_C(  9117), -INT16_C( 10220),
         INT16_C( 17183), -INT16_C( 27018),  INT16_C( 10476),  INT16_C( 26088), -INT16_C(  7467), -INT16_C( 12558), -INT16_C( 17449), -INT16_C( 28337),
        -INT16_C( 14564), -INT16_C( 22624), -INT16_C( 28928),  INT16_C( 21370), -INT16_C(   798), -INT16_C( 32703),  INT16_C( 22047),  INT16_C( 15960),
        -INT16_C( 12647), -INT16_C( 31276), -INT16_C( 17162), -INT16_C( 13333), -INT16_C(  8802),  INT16_C( 30106), -INT16_C(  5736), -INT16_C( 19449) },
      {  INT32_C(   459506096), -INT32_C(   276860781),  INT32_C(   308568466), -INT32_C(   261184753), -INT32_C(   125784633),  INT32_C(   467684796),  INT32_C(   371805366), -INT32_C(   786655126),
         INT32_C(   272487108), -INT32_C(   484381068),  INT32_C(   669694452),  INT32_C(   578904093),  INT32_C(   795430594),  INT32_C(   101343424), -INT32_C(   698288110),  INT32_C(   643140951) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi16(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi16(test_vec[i].b);
    simde__m512i r = simde_mm512_madd_epi16(a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_mask_madd_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int16_t src[32];
    const simde__mmask16 k;
    const int16_t a[32];
    const int16_t b[32];
    const int32_t r[16];
  } test_vec[] = {
    { { -INT16_C( 24666), -INT16_C( 29655),  INT16_C( 25716), -INT16_C(  4900),  INT16_C( 25090), -INT16_C(  2830),  INT16_C( 11305), -INT16_C( 29651),
        -INT16_C( 19186),  INT16_C(   399), -INT16_C( 21850), -INT16_C( 23497), -INT16_C(  8300),  INT16_C(  7570), -INT16_C( 14461),  INT16_C( 10749),
         INT16_C( 10086), -INT16_C(  9547), -INT16_C( 28277), -INT16_C( 29242), -INT16_C( 18188),  INT16_C(  7553), -INT16_C( 20764), -INT16_C(  3159),
         INT16_C( 14435),  INT16_C(  2548),  INT16_C( 11234),  INT16_C( 30637),  INT16_C( 16394), -INT16_C( 29292), -INT16_C( 28153),  INT16_C( 28087) },
      UINT16_C(27833),
      {  INT16_C( 17479),  INT16_C(  3582), -INT16_C(  3375),  INT16_C( 21189), -INT16_C( 22257), -INT16_C( 18432),  INT16_C( 25500), -INT16_C( 28432),
        -INT16_C( 11667),  INT16_C(  6844), -INT16_C( 14775), -INT16_C(  8614),  INT16_C( 24916),  INT16_C(  2928),  INT16_C( 10702),  INT16_C(  5495),
         INT16_C( 30061),  INT16_C( 15906), -INT16_C(  6297),  INT16_C( 30352), -INT16_C( 28527),  INT16_C( 11566),  INT16_C(  7923),  INT16_C( 24766),
         INT16_C( 31473),  INT16_C( 14971), -INT16_C( 10944), -INT16_C( 27624), -INT16_C( 30665),  INT16_C(  1439),  INT16_C(  6065),  INT16_C(  7707) },
      {  INT16_C( 15756), -INT16_C(  2980), -INT16_C(  5083), -INT16_C( 18838), -INT16_C( 26244),  INT16_C( 28899), -INT16_C( 24137), -INT16_C( 22320),
         INT16_C( 19227),  INT16_C( 23779), -INT16_C(  1247),  INT16_C( 22768), -INT16_C( 28540),  INT16_C( 13661),  INT16_C( 30887),  INT16_C( 13140),
        -INT16_C( 20298), -INT16_C(  9433), -INT16_C( 28003),  INT16_C(  6545),  INT16_C( 29739), -INT16_C(  7543),  INT16_C( 23062),  INT16_C( 12683),
         INT16_C( 28325), -INT16_C( 14707),  INT16_C( 32361), -INT16_C(  4834),  INT16_C( 31758), -INT16_C( 19165),  INT16_C( 30708), -INT16_C( 21784) },
      {  INT32_C(   264724764), -INT32_C(   321100684), -INT32_C(   185441790),  INT32_C(    19108740), -INT32_C(    61577933), -INT32_C(   177699127),  INT32_C(   496164756),  INT32_C(   402756974),
        -INT32_C(   625662106), -INT32_C(  1916366453), -INT32_C(   935606791),  INT32_C(   496827404),  INT32_C(   167000163), -INT32_C(   220624368), -INT32_C(  1001437505),  INT32_C(  1840747015) } },
    { {  INT16_C(  4135), -INT16_C( 15227),  INT16_C(  5794), -INT16_C( 12834),  INT16_C( 26507), -INT16_C( 24145),  INT16_C( 15041),  INT16_C( 26578),
         INT16_C( 24744),  INT16_C(  4653),  INT16_C( 19678), -INT16_C(  4865),  INT16_C(  8904), -INT16_C( 17247), -INT16_C( 30311), -INT16_C( 16025),
        -INT16_C(  4967),  INT16_C( 15237),  INT16_C( 25347), -INT16_C( 29176), -INT16_C( 18229), -INT16_C( 29649),  INT16_C(   498), -INT16_C( 25613),
         INT16_C(  8545),  INT16_C( 16301), -INT16_C( 21395),  INT16_C( 13611), -INT16_C( 13105),  INT16_C( 26865),  INT16_C( 22614), -INT16_C(  4311) },
      UINT16_C(44869),
      {  INT16_C( 18475),  INT16_C( 13074), -INT16_C(  8746),  INT16_C(  1515), -INT16_C(  8598),  INT16_C( 23814),  INT16_C( 26745),  INT16_C(  9854),
        -INT16_C(  5209), -INT16_C( 11310), -INT16_C( 24288),  INT16_C(  4767), -INT16_C(  2806),  INT16_C( 13162), -INT16_C( 20507),  INT16_C(  4322),
        -INT16_C(  2569), -INT16_C( 12989),  INT16_C( 12242),  INT16_C( 15570), -INT16_C(  9971), -INT16_C( 31078),  INT16_C(  6209), -INT16_C(  5972),
         INT16_C( 32260),  INT16_C(  9403),  INT16_C( 23328),  INT16_C( 10806), -INT16_C( 24240),  INT16_C( 13661),  INT16_C( 16464),  INT16_C( 18501) },
      { -INT16_C( 30411),  INT16_C(  1813), -INT16_C(  5960), -INT16_C( 15036), -INT16_C(  8511),  INT16_C(   587), -INT16_C(  2058), -INT16_C(  1302),
        -INT16_C( 22923), -INT16_C( 27361),  INT16_C( 21761),  INT16_C( 20927),  INT16_C(  7670),  INT16_C( 18311), -INT16_C( 13219), -INT16_C( 28017),
        -INT16_C( 23467),  INT16_C(  3481), -INT16_C(  8820),  INT16_C( 19922),  INT16_C(  7611), -INT16_C( 19889),  INT16_C( 14868), -INT16_C( 30036),
        -INT16_C( 13344), -INT16_C(  7905), -INT16_C(  8415),  INT16_C(  5938), -INT16_C( 17924),  INT16_C( 22878), -INT16_C(  4730), -INT16_C(  9237) },
      { -INT32_C(   538140063), -INT32_C(   841083230),  INT32_C(    87156396),  INT32_C(  1741830849),  INT32_C(   304963752), -INT32_C(   318812962),  INT32_C(   219487362), -INT32_C(  1050179175),
         INT32_C(    15072014),  INT32_C(   202211100),  INT32_C(   542221061),  INT32_C(   271690404),  INT32_C(  1068310881), -INT32_C(   132139092),  INT32_C(  1760677071), -INT32_C(   248768457) } },
    { { -INT16_C( 31598),  INT16_C(  7913), -INT16_C( 17566),  INT16_C(  7532), -INT16_C( 17447), -INT16_C(  4657),  INT16_C( 31989), -INT16_C( 10889),
        -INT16_C( 26809),  INT16_C( 26806), -INT16_C(  5770),  INT16_C( 29312), -INT16_C(  8542),  INT16_C( 10443), -INT16_C( 18740),  INT16_C( 24068),
        -INT16_C(  4806), -INT16_C( 25476), -INT16_C(  5976), -INT16_C( 32326), -INT16_C( 30300), -INT16_C( 26257), -INT16_C(  6651),  INT16_C( 19823),
         INT16_C(  9597), -INT16_C(  3147),  INT16_C( 13582), -INT16_C( 20123),  INT16_C( 12308), -INT16_C(  7975), -INT16_C(  8730),  INT16_C(  8510) },
      UINT16_C(47818),
      {  INT16_C( 29629),  INT16_C( 30627),  INT16_C( 18420),  INT16_C( 25345),  INT16_C(  1760),  INT16_C( 20298), -INT16_C( 14509),  INT16_C(  2421),
        -INT16_C( 31813),  INT16_C(  8254),  INT16_C( 21044),  INT16_C(  3665),  INT16_C( 14130),  INT16_C( 28907), -INT16_C( 18856),  INT16_C(  5675),
        -INT16_C( 12759),  INT16_C(  7565), -INT16_C( 29163), -INT16_C(  2687), -INT16_C( 13419), -INT16_C(  6075), -INT16_C( 17774),  INT16_C( 19953),
         INT16_C( 12349),  INT16_C( 29294), -INT16_C( 16510), -INT16_C( 19072),  INT16_C( 27638),  INT16_C( 20261),  INT16_C( 20513),  INT16_C( 19045) },
      { -INT16_C(  3554),  INT16_C( 13160), -INT16_C(  5759),  INT16_C(  5673),  INT16_C( 28340),  INT16_C( 18174), -INT16_C(  4056),  INT16_C( 26004),
         INT16_C(   544), -INT16_C( 23849),  INT16_C( 22465), -INT16_C( 18601),  INT16_C( 32195), -INT16_C(  7162),  INT16_C( 27597), -INT16_C(  5073),
        -INT16_C( 26786), -INT16_C(  8417),  INT16_C( 18560),  INT16_C( 13557), -INT16_C(  3146), -INT16_C(  8582),  INT16_C(  3811),  INT16_C(   836),
         INT16_C(  6928), -INT16_C( 11866), -INT16_C(   653),  INT16_C( 13961), -INT16_C( 28806),  INT16_C( 18458),  INT16_C( 18939),  INT16_C( 22836) },
      {  INT32_C(   518620306),  INT32_C(    37701405), -INT32_C(   305153063),  INT32_C(   121804188),  INT32_C(  1756796743),  INT32_C(  1921050998),  INT32_C(   247883416), -INT32_C(   549158307),
        -INT32_C(  1669534406), -INT32_C(   577692939), -INT32_C(  1720743516), -INT32_C(    51056006), -INT32_C(   262048732), -INT32_C(   255483162), -INT32_C(   522637292),  INT32_C(   823407327) } },
    { {  INT16_C( 21472),  INT16_C( 24632),  INT16_C( 11676),  INT16_C( 21140),  INT16_C(  3872),  INT16_C(  1073),  INT16_C( 29981),  INT16_C( 11783),
        -INT16_C( 21104),  INT16_C(  1023), -INT16_C( 30549),  INT16_C(  9529),  INT16_C( 21528),  INT16_C(  4973), -INT16_C( 24163),  INT16_C( 32364),
        -INT16_C( 23307), -INT16_C( 28194),  INT16_C( 29649), -INT16_C(  3613),  INT16_C(  5250), -INT16_C( 24587), -INT16_C(   631),  INT16_C(  6861),
        -INT16_C( 12886),  INT16_C( 21789),  INT16_C( 22357),  INT16_C( 28027), -INT16_C(  5973),  INT16_C( 18560), -INT16_C(  4982),  INT16_C( 32710) },
      UINT16_C(42384),
      {  INT16_C( 24848), -INT16_C(  3304), -INT16_C( 26029),  INT16_C( 18440), -INT16_C( 28359),  INT16_C(  1861), -INT16_C(  3925), -INT16_C( 13868),
         INT16_C( 10565), -INT16_C( 16352), -INT16_C( 13417),  INT16_C(  6057),  INT16_C( 13075), -INT16_C(  9724), -INT16_C( 27470), -INT16_C( 15745),
        -INT16_C( 26634),  INT16_C( 18869), -INT16_C( 17103),  INT16_C( 27281), -INT16_C( 10417), -INT16_C(  1423),  INT16_C( 17863),  INT16_C(  3267),
        -INT16_C(  7313),  INT16_C(  1741),  INT16_C( 30382), -INT16_C( 15843),  INT16_C(  8617),  INT16_C( 23452),  INT16_C(  7094), -INT16_C( 21475) },
      { -INT16_C( 11598), -INT16_C(  7179), -INT16_C( 31088), -INT16_C(  8371), -INT16_C( 16547),  INT16_C(  9433), -INT16_C( 25340),  INT16_C( 29489),
        -INT16_C(   384),  INT16_C( 12153), -INT16_C( 26764),  INT16_C(  7665), -INT16_C( 29256),  INT16_C( 28280), -INT16_C( 27224),  INT16_C( 23066),
         INT16_C(  3943), -INT16_C(  2243), -INT16_C( 30058), -INT16_C(  3114), -INT16_C( 20407),  INT16_C( 19992),  INT16_C( 18765), -INT16_C( 12863),
         INT16_C( 15175), -INT16_C( 17412), -INT16_C(  4654), -INT16_C( 29992),  INT16_C( 20602),  INT16_C(  8953),  INT16_C(  5093),  INT16_C( 19580) },
      {  INT32_C(  1614304224),  INT32_C(  1385442716),  INT32_C(    70324000),  INT32_C(   772240669), -INT32_C(   202782816),  INT32_C(   624527531),  INT32_C(   325932056),  INT32_C(   384669110),
        -INT32_C(   147341029), -INT32_C(   236751919),  INT32_C(   184131103),  INT32_C(   449707401),  INT32_C(  1428016554),  INT32_C(   333765428),  INT32_C(  1216407723), -INT32_C(   384350758) } },
    { { -INT16_C( 18141), -INT16_C( 18108),  INT16_C(  6724), -INT16_C( 29268), -INT16_C( 15158),  INT16_C(  6107), -INT16_C( 25331),  INT16_C( 21733),
        -INT16_C(  7720), -INT16_C( 22001), -INT16_C(  6193),  INT16_C( 18740),  INT16_C( 11575),  INT16_C(  7276), -INT16_C(  6079),  INT16_C( 25705),
        -INT16_C( 21086), -INT16_C(  6627), -INT16_C( 13881), -INT16_C( 28045),  INT16_C( 20366), -INT16_C( 25687), -INT16_C( 28948), -INT16_C( 15120),
        -INT16_C(   144),  INT16_C( 16238), -INT16_C( 23833),  INT16_C(  7816), -INT16_C(  2864),  INT16_C(  4411), -INT16_C( 23331),  INT16_C( 32629) },
      UINT16_C(37457),
      {  INT16_C(  6245), -INT16_C( 10149), -INT16_C(  5718),  INT16_C( 21543),  INT16_C(  4997),  INT16_C( 30178),  INT16_C( 21207),  INT16_C( 17780),
         INT16_C( 23441),  INT16_C(  6888), -INT16_C( 18310), -INT16_C( 19186), -INT16_C(  5175),  INT16_C( 15961), -INT16_C( 21910), -INT16_C( 12336),
         INT16_C( 11202),  INT16_C( 28072), -INT16_C( 12523), -INT16_C( 25919), -INT16_C( 23581), -INT16_C( 17905), -INT16_C( 31754), -INT16_C( 30976),
        -INT16_C(  5921),  INT16_C( 22945), -INT16_C( 20320),  INT16_C( 26894),  INT16_C( 26523),  INT16_C(  1703),  INT16_C( 30481), -INT16_C( 11307) },
      {  INT16_C( 32162), -INT16_C( 18624),  INT16_C(   333),  INT16_C( 12369),  INT16_C( 24741), -INT16_C( 25622), -INT16_C(  5404), -INT16_C( 15582),
        -INT16_C( 15150),  INT16_C( 29212),  INT16_C( 10868),  INT16_C(  4059), -INT16_C( 32111), -INT16_C( 24043), -INT16_C(  5127), -INT16_C( 25483),
        -INT16_C( 18840), -INT16_C( 19117), -INT16_C( 23113),  INT16_C( 23781), -INT16_C( 12283), -INT16_C(  5641),  INT16_C(  6842), -INT16_C( 29268),
        -INT16_C( 14114),  INT16_C( 21247), -INT16_C(  9230), -INT16_C( 31903),  INT16_C( 30557),  INT16_C( 22309), -INT16_C( 25758), -INT16_C( 13581) },
      {  INT32_C(   389866666), -INT32_C(  1918100924),  INT32_C(   400278730),  INT32_C(  1424334093), -INT32_C(   153918894),  INT32_C(  1228203983), -INT32_C(   217575898),  INT32_C(  1684662337),
        -INT32_C(   434262622), -INT32_C(   326935640), -INT32_C(  1683402866), -INT32_C(   990867732),  INT32_C(   571081409),  INT32_C(   512271079),  INT32_C(   289141968), -INT32_C(   631569231) } },
    { {  INT16_C( 18001),  INT16_C(  2176),  INT16_C( 26091), -INT16_C(  3739),  INT16_C( 23605), -INT16_C(  3878), -INT16_C( 30858),  INT16_C( 21629),
         INT16_C( 31823),  INT16_C( 17062),  INT16_C(  2135), -INT16_C( 19003), -INT16_C(  5249), -INT16_C(  7924), -INT16_C(   122), -INT16_C( 10325),
         INT16_C( 11077),  INT16_C( 12767),  INT16_C( 17553), -INT16_C( 14814), -INT16_C(   863),  INT16_C(  6070),  INT16_C( 13187), -INT16_C( 11412),
         INT16_C(  4784),  INT16_C(  1813), -INT16_C(  9702), -INT16_C( 26180), -INT16_C( 14139),  INT16_C( 19322),  INT16_C(  9927),  INT16_C(  3362) },
      UINT16_C(  593),
      { -INT16_C(  7618),  INT16_C( 24646), -INT16_C(  6231),  INT16_C( 24412), -INT16_C(  7937),  INT16_C( 27539),  INT16_C( 17331), -INT16_C( 14211),
        -INT16_C( 26550),  INT16_C(  1954),  INT16_C( 26673), -INT16_C( 21297), -INT16_C( 26701), -INT16_C( 10542),  INT16_C(  9124), -INT16_C(  7464),
         INT16_C(  7686), -INT16_C( 20670), -INT16_C( 25082),  INT16_C(  1294), -INT16_C( 24194),  INT16_C( 12656), -INT16_C(  4636),  INT16_C( 12281),
        -INT16_C( 25467), -INT16_C( 18634),  INT16_C(  1284), -INT16_C( 18589),  INT16_C( 13724),  INT16_C( 16525),  INT16_C( 25944),  INT16_C( 24098) },
      {  INT16_C( 25732), -INT16_C( 30195),  INT16_C(  7171), -INT16_C( 32369), -INT16_C(    67), -INT16_C( 23885), -INT16_C( 21268),  INT16_C( 29393),
         INT16_C(  1864),  INT16_C( 19497), -INT16_C( 29684), -INT16_C( 22268), -INT16_C( 28223),  INT16_C(  6633),  INT16_C(  3319),  INT16_C( 31608),
        -INT16_C( 31376),  INT16_C( 29445), -INT16_C( 27487),  INT16_C( 24565), -INT16_C( 22381),  INT16_C( 32513), -INT16_C( 11692), -INT16_C( 25103),
         INT16_C(  6873), -INT16_C(  6679), -INT16_C(  4698),  INT16_C( 26510),  INT16_C( 30847),  INT16_C( 30337), -INT16_C(  1660), -INT16_C(  2831) },
      { -INT32_C(   940212346), -INT32_C(   245013013), -INT32_C(   254125003),  INT32_C(  1417512822), -INT32_C(    11392062), -INT32_C(  1245378473),  INT32_C(   683657237), -INT32_C(   676593786),
         INT32_C(   836709189),  INT32_C(   721216044),  INT32_C(   397868193), -INT32_C(   747883645),  INT32_C(   118821552), -INT32_C(  1715676646),  INT32_C(  1266337989),  INT32_C(   220341959) } },
    { { -INT16_C(  2434),  INT16_C(  8296),  INT16_C( 23946),  INT16_C(  7551), -INT16_C( 32763),  INT16_C( 22940), -INT16_C( 29102),  INT16_C( 11254),
        -INT16_C(  8024),  INT16_C( 20240), -INT16_C( 24627),  INT16_C( 19638),  INT16_C( 14103), -INT16_C( 25662), -INT16_C( 19664), -INT16_C( 20593),
        -INT16_C(  2135),  INT16_C( 13263),  INT16_C( 20052),  INT16_C( 22864), -INT16_C(  4658),  INT16_C(  8371), -INT16_C( 22149),  INT16_C(  9035),
         INT16_C( 23433),  INT16_C( 22386),  INT16_C( 10746),  INT16_C(  4515),  INT16_C( 26208), -INT16_C( 28244),  INT16_C( 15385), -INT16_C( 15552) },
      UINT16_C( 3891),
      { -INT16_C( 30474),  INT16_C( 18269),  INT16_C( 11233), -INT16_C( 27596), -INT16_C( 20661), -INT16_C( 27074), -INT16_C( 14382),  INT16_C( 17905),
        -INT16_C(  5090), -INT16_C( 15762), -INT16_C( 12547), -INT16_C( 21976),  INT16_C( 16735), -INT16_C( 24602),  INT16_C(  6404), -INT16_C(  1106),
         INT16_C(  2977), -INT16_C( 31934),  INT16_C( 30262), -INT16_C( 32489),  INT16_C( 21797), -INT16_C(  2281),  INT16_C(  2333),  INT16_C( 15164),
        -INT16_C( 21771), -INT16_C(  3331),  INT16_C(  9593), -INT16_C( 10084), -INT16_C( 32153),  INT16_C( 27512),  INT16_C(  9884),  INT16_C( 15718) },
      { -INT16_C( 22478),  INT16_C( 26816), -INT16_C( 10210),  INT16_C( 17386),  INT16_C(   301),  INT16_C( 19003),  INT16_C( 30474), -INT16_C(   122),
        -INT16_C( 31966), -INT16_C( 25614), -INT16_C( 29015),  INT16_C(  4211), -INT16_C(  5359), -INT16_C( 21125), -INT16_C(  7662),  INT16_C( 17642),
        -INT16_C( 21622), -INT16_C( 22100), -INT16_C( 27005), -INT16_C( 20244),  INT16_C( 10136), -INT16_C( 23813), -INT16_C( 32353), -INT16_C( 15966),
        -INT16_C( 27644), -INT16_C( 21156), -INT16_C( 12510),  INT16_C( 13245),  INT16_C( 14779), -INT16_C( 12832), -INT16_C( 13541), -INT16_C( 23279) },
      {  INT32_C(  1174896076), -INT32_C(   594472986),  INT32_C(  1503428613),  INT32_C(   737578578),  INT32_C(   566434808),  INT32_C(   271510269), -INT32_C(  1681770729), -INT32_C(  1349536976),
         INT32_C(   641372706), -INT32_C(   159517994),  INT32_C(   275251845), -INT32_C(   317587973),  INT32_C(  1467112329),  INT32_C(   295905786), -INT32_C(  1850972576), -INT32_C(  1019200487) } },
    { { -INT16_C( 17034), -INT16_C(  1714),  INT16_C( 15188), -INT16_C(  4951), -INT16_C( 23454),  INT16_C(   398),  INT16_C( 12325),  INT16_C( 10946),
         INT16_C(  7876), -INT16_C(  6185), -INT16_C( 27154), -INT16_C( 22246), -INT16_C(  1074), -INT16_C(  5770), -INT16_C( 30778),  INT16_C( 15502),
        -INT16_C(  8892), -INT16_C( 26571), -INT16_C(  8680),  INT16_C( 31364),  INT16_C(  4995), -INT16_C( 22404),  INT16_C( 15939),  INT16_C(  2258),
        -INT16_C( 21923),  INT16_C( 19439),  INT16_C(  2367),  INT16_C(  3572),  INT16_C( 27140), -INT16_C( 13578), -INT16_C( 31503),  INT16_C( 13574) },
      UINT16_C(15201),
      {  INT16_C( 31182),  INT16_C( 21018), -INT16_C( 25100),  INT16_C( 28773), -INT16_C( 22203),  INT16_C(  6318),  INT16_C(  2993), -INT16_C( 24382),
         INT16_C(   342),  INT16_C( 19113), -INT16_C( 20978),  INT16_C(  1204), -INT16_C( 23176),  INT16_C( 32648), -INT16_C(  5413), -INT16_C( 22086),
        -INT16_C( 11165),  INT16_C( 22523),  INT16_C( 24945), -INT16_C( 18489),  INT16_C( 30218), -INT16_C( 17457), -INT16_C( 28287), -INT16_C( 10149),
         INT16_C(  1170), -INT16_C( 24542), -INT16_C( 10318),  INT16_C( 11172),  INT16_C( 11388),  INT16_C( 22442),  INT16_C( 25622),  INT16_C( 31232) },
      { -INT16_C(   967), -INT16_C( 21807), -INT16_C( 26275),  INT16_C( 26465),  INT16_C( 12303), -INT16_C( 28638),  INT16_C( 32193),  INT16_C( 21352),
        -INT16_C( 29823),  INT16_C( 13555), -INT16_C( 26782), -INT16_C(  8609),  INT16_C(  2500), -INT16_C(  9674),  INT16_C( 13933), -INT16_C( 22956),
         INT16_C(  9778), -INT16_C( 28847), -INT16_C( 19777), -INT16_C( 12554),  INT16_C(  6371), -INT16_C( 23458), -INT16_C( 14443),  INT16_C(  6136),
        -INT16_C(  5294), -INT16_C( 19381), -INT16_C( 21885),  INT16_C( 18322), -INT16_C( 14157),  INT16_C(  8225),  INT16_C( 30463),  INT16_C( 12743) },
      { -INT32_C(   488492520), -INT32_C(   324453548),  INT32_C(    26125410),  INT32_C(   717369381), -INT32_C(   405332284),  INT32_C(   551467560), -INT32_C(   373776752),  INT32_C(  1015973830),
        -INT32_C(   758892351), -INT32_C(   261226359), -INT32_C(  1468263549),  INT32_C(   346274877),  INT32_C(   469454522),  INT32_C(   430502814), -INT32_C(   889820668),  INT32_C(   889619697) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi16(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi16(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi16(test_vec[i].b);
    simde__m512i r = simde_mm512_mask_madd_epi16(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_maskz_madd_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask16 k;
    const int16_t a[32];
    const int16_t b[32];
    const int32_t r[16];
  } test_vec[] = {
    { UINT16_C(49133),
      { -INT16_C( 12436),  INT16_C(  8143),  INT16_C(  3509), -INT16_C(  8650),  INT16_C( 19172), -INT16_C( 17007), -INT16_C( 26437),  INT16_C( 27681),
        -INT16_C(  6686), -INT16_C( 11200), -INT16_C( 18005), -INT16_C(  6237), -INT16_C( 26567), -INT16_C(  3727),  INT16_C( 24180), -INT16_C(  7760),
         INT16_C( 32558), -INT16_C(  7424),  INT16_C( 14220),  INT16_C( 29121),  INT16_C( 21121),  INT16_C( 15406),  INT16_C( 20459), -INT16_C( 12887),
        -INT16_C(  5836), -INT16_C(  8031),  INT16_C( 17826), -INT16_C(  9273),  INT16_C( 14557),  INT16_C( 20940),  INT16_C( 31894), -INT16_C( 15310) },
      {  INT16_C( 13308), -INT16_C( 30553),  INT16_C( 26986), -INT16_C(  5127),  INT16_C( 10171), -INT16_C( 23000), -INT16_C( 11913), -INT16_C( 21645),
         INT16_C(  5562),  INT16_C( 23947),  INT16_C( 21082),  INT16_C( 14136),  INT16_C(  1418),  INT16_C(  8584), -INT16_C( 17535),  INT16_C( 32229),
        -INT16_C( 29202),  INT16_C( 22534), -INT16_C(    10), -INT16_C( 20157),  INT16_C( 27431), -INT16_C( 25000), -INT16_C( 13508), -INT16_C(  2231),
        -INT16_C( 10784),  INT16_C( 14932), -INT16_C( 29657), -INT16_C( 19855), -INT16_C(  1391),  INT16_C(  5075), -INT16_C( 18251), -INT16_C( 23664) },
      { -INT32_C(   414291367),  INT32_C(           0),  INT32_C(   586159412), -INT32_C(   284211264),  INT32_C(           0), -INT32_C(   467747642), -INT32_C(    69664574), -INT32_C(   674093340),
        -INT32_C(  1118051132), -INT32_C(   587134197),  INT32_C(   194220151), -INT32_C(   247609275), -INT32_C(    56983468), -INT32_C(   344550267),  INT32_C(           0), -INT32_C(   219801554) } },
    { UINT16_C(38469),
      {  INT16_C( 15355),  INT16_C( 16022), -INT16_C( 16915),  INT16_C( 17834), -INT16_C(  6565), -INT16_C( 23536), -INT16_C(  3619),  INT16_C( 12665),
        -INT16_C( 24277), -INT16_C( 25154),  INT16_C( 20307),  INT16_C(  9879),  INT16_C( 19554), -INT16_C(  3106),  INT16_C(  9455), -INT16_C(  5495),
         INT16_C(  8031),  INT16_C( 19496), -INT16_C( 11556),  INT16_C( 14225), -INT16_C( 23879), -INT16_C( 26916),  INT16_C( 21907), -INT16_C( 16696),
        -INT16_C( 30986),  INT16_C( 18779), -INT16_C(  3371),  INT16_C( 14447),  INT16_C( 20030),  INT16_C( 11563), -INT16_C( 19342), -INT16_C( 12009) },
      {  INT16_C( 16596), -INT16_C( 20450), -INT16_C( 20718), -INT16_C( 13336), -INT16_C( 15279), -INT16_C(  7070),  INT16_C( 10777),  INT16_C(  4259),
        -INT16_C(   336), -INT16_C( 31399), -INT16_C( 13839),  INT16_C( 12221), -INT16_C(  6121), -INT16_C( 30371),  INT16_C( 29853),  INT16_C( 29018),
         INT16_C( 30900), -INT16_C( 14559),  INT16_C(  2344),  INT16_C( 31122), -INT16_C(  2867), -INT16_C(  6306),  INT16_C(   286), -INT16_C( 12553),
         INT16_C( 20735), -INT16_C(  4012),  INT16_C(  4377),  INT16_C( 12320),  INT16_C( 32250), -INT16_C( 26695),  INT16_C(  5361), -INT16_C( 23032) },
      { -INT32_C(    72818320),  INT32_C(           0),  INT32_C(   266706155),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(    25357708),  INT32_C(           0),
         INT32_C(           0),  INT32_C(   415623186),  INT32_C(   238193389),  INT32_C(           0), -INT32_C(   717836058),  INT32_C(           0),  INT32_C(           0),  INT32_C(   172898826) } },
    { UINT16_C(10636),
      { -INT16_C( 19347), -INT16_C(   205),  INT16_C(    46), -INT16_C( 29452),  INT16_C(  4839), -INT16_C(  8563), -INT16_C( 29471),  INT16_C( 13615),
         INT16_C( 18557), -INT16_C( 25274),  INT16_C( 16505),  INT16_C( 12826),  INT16_C(  3031), -INT16_C(  8378), -INT16_C( 11343),  INT16_C(  7689),
         INT16_C( 15495), -INT16_C( 19170),  INT16_C(  4668),  INT16_C(  9281), -INT16_C( 12764),  INT16_C(  1282),  INT16_C( 12635), -INT16_C( 10182),
        -INT16_C( 32390), -INT16_C(  3211), -INT16_C( 28735), -INT16_C( 26331),  INT16_C( 27802),  INT16_C( 19576), -INT16_C( 32449), -INT16_C( 14742) },
      { -INT16_C( 30531), -INT16_C(  1412), -INT16_C( 16998), -INT16_C( 16610),  INT16_C(  8332), -INT16_C(  6204), -INT16_C(   174), -INT16_C( 13121),
         INT16_C( 13440),  INT16_C( 16831), -INT16_C(  6973),  INT16_C( 24026),  INT16_C( 21328), -INT16_C( 28759),  INT16_C(  5332), -INT16_C( 28074),
        -INT16_C( 11620),  INT16_C( 14220), -INT16_C( 21873),  INT16_C(  7158), -INT16_C( 17718),  INT16_C(  7170), -INT16_C( 15943),  INT16_C( 14824),
        -INT16_C( 22539), -INT16_C( 18309),  INT16_C( 21900), -INT16_C(  9194), -INT16_C( 16472),  INT16_C( 32108), -INT16_C( 15661),  INT16_C( 28687) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(    93443400), -INT32_C(   173514461),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   276341862),
        -INT32_C(   452649300),  INT32_C(           0),  INT32_C(           0), -INT32_C(   352377773),  INT32_C(           0), -INT32_C(   387209286),  INT32_C(           0),  INT32_C(           0) } },
    { UINT16_C(39828),
      {  INT16_C(  9127), -INT16_C( 25275),  INT16_C(  3903),  INT16_C( 16727),  INT16_C(  4396),  INT16_C(  5123), -INT16_C(  1974), -INT16_C( 14916),
         INT16_C( 18609), -INT16_C( 14565), -INT16_C( 15580), -INT16_C( 28538),  INT16_C( 23104),  INT16_C( 20306), -INT16_C(  6454),  INT16_C( 29162),
         INT16_C( 12042),  INT16_C( 18702),  INT16_C( 25919),  INT16_C( 27530), -INT16_C( 29322), -INT16_C( 16001),  INT16_C( 15238),  INT16_C( 14214),
        -INT16_C( 24189), -INT16_C( 22274), -INT16_C( 31643), -INT16_C( 23240), -INT16_C( 29730), -INT16_C( 22283), -INT16_C(  8335),  INT16_C( 31513) },
      {  INT16_C(  9999),  INT16_C( 20164),  INT16_C( 20365),  INT16_C(   953),  INT16_C( 14556),  INT16_C( 25284),  INT16_C( 19316), -INT16_C(  2151),
        -INT16_C( 26644),  INT16_C( 20895), -INT16_C( 10212), -INT16_C(  1289), -INT16_C(  5021), -INT16_C( 11101), -INT16_C( 17205), -INT16_C(  9648),
         INT16_C(  5348),  INT16_C( 28968), -INT16_C(  7837),  INT16_C( 16500),  INT16_C( 14618), -INT16_C( 29022),  INT16_C( 15492),  INT16_C( 28805),
         INT16_C(  9683), -INT16_C(  4158), -INT16_C( 17923),  INT16_C( 24810), -INT16_C( 29275),  INT16_C( 28724), -INT16_C( 31671),  INT16_C( 11595) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(   193518108),  INT32_C(           0), -INT32_C(   800153871),  INT32_C(           0),  INT32_C(           0), -INT32_C(   170313906),
         INT32_C(   606160152),  INT32_C(   251117797),  INT32_C(           0),  INT32_C(   645501366), -INT32_C(   141606795),  INT32_C(           0),  INT32_C(           0),  INT32_C(   629371020) } },
    { UINT16_C(29593),
      { -INT16_C(   866),  INT16_C(  4949),  INT16_C( 28476), -INT16_C(  8372), -INT16_C( 12035), -INT16_C( 32229), -INT16_C(  4544),  INT16_C(   679),
        -INT16_C( 23330), -INT16_C( 14149),  INT16_C( 24580),  INT16_C( 14677), -INT16_C( 24879),  INT16_C(  7357),  INT16_C( 22220),  INT16_C( 27279),
        -INT16_C(  7085), -INT16_C( 28803), -INT16_C( 13997),  INT16_C( 20590), -INT16_C( 30311), -INT16_C(  9517),  INT16_C( 31352),  INT16_C( 22236),
        -INT16_C( 26593),  INT16_C(  8990),  INT16_C( 29688), -INT16_C( 13988),  INT16_C(  6673), -INT16_C(  8731),  INT16_C( 30064), -INT16_C( 15544) },
      { -INT16_C( 15015), -INT16_C( 21165), -INT16_C( 15985),  INT16_C( 10493), -INT16_C( 12213), -INT16_C( 15614), -INT16_C(  8373),  INT16_C( 27161),
         INT16_C( 14199),  INT16_C( 28557), -INT16_C(  5462), -INT16_C( 17607),  INT16_C(  7684),  INT16_C( 29849), -INT16_C(  7789), -INT16_C(  4808),
        -INT16_C( 29786),  INT16_C( 13722), -INT16_C( 26804), -INT16_C( 26786),  INT16_C( 24680), -INT16_C( 19622),  INT16_C( 29503), -INT16_C( 18915),
        -INT16_C( 21846),  INT16_C( 21542),  INT16_C( 24468), -INT16_C( 26608), -INT16_C( 22147),  INT16_C(  4365),  INT16_C( 17802),  INT16_C( 12542) },
      { -INT32_C(    91742595),  INT32_C(           0),  INT32_C(           0),  INT32_C(    56489231), -INT32_C(   735315663),  INT32_C(           0),  INT32_C(           0), -INT32_C(   304229012),
        -INT32_C(   184200956), -INT32_C(   176348152),  INT32_C(           0),  INT32_C(           0),  INT32_C(   774613258),  INT32_C(  1098598688), -INT32_C(   185897746),  INT32_C(           0) } },
    { UINT16_C(39120),
      {  INT16_C(  7270), -INT16_C( 15313), -INT16_C( 26700),  INT16_C(  3620),  INT16_C( 25674),  INT16_C( 26498),  INT16_C( 11290),  INT16_C( 16402),
        -INT16_C( 22911), -INT16_C( 28257),  INT16_C(  7487),  INT16_C( 19514), -INT16_C( 15314),  INT16_C( 11409),  INT16_C( 25076),  INT16_C( 23236),
        -INT16_C(  3203),  INT16_C( 12574),  INT16_C( 17291), -INT16_C( 10944), -INT16_C( 15705), -INT16_C( 16067),  INT16_C( 20462),  INT16_C( 28418),
        -INT16_C( 24075),  INT16_C( 13312),  INT16_C( 15038), -INT16_C(  4992),  INT16_C(  4606), -INT16_C(  3304), -INT16_C(  9102), -INT16_C(  4019) },
      {  INT16_C( 27856),  INT16_C( 23329),  INT16_C( 25007),  INT16_C( 22064),  INT16_C( 27939),  INT16_C(  4631),  INT16_C(  6588), -INT16_C( 19839),
        -INT16_C( 32069),  INT16_C( 31206),  INT16_C( 26556), -INT16_C( 17562),  INT16_C( 32376), -INT16_C(  5202), -INT16_C(  1189),  INT16_C( 11227),
        -INT16_C(   921),  INT16_C(  5766), -INT16_C( 18850), -INT16_C( 32404), -INT16_C( 31708), -INT16_C(  8045),  INT16_C(  5533),  INT16_C( 22674),
         INT16_C( 31127),  INT16_C( 21458),  INT16_C( 14560),  INT16_C( 22542), -INT16_C( 17226),  INT16_C(  4419),  INT16_C(  7864),  INT16_C(  7996) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   147055083),  INT32_C(           0), -INT32_C(   555155682),  INT32_C(   231055208),
         INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(   757565978), -INT32_C(   463733629),  INT32_C(           0),  INT32_C(           0), -INT32_C(   103714052) } },
    { UINT16_C(49691),
      {  INT16_C( 31030), -INT16_C( 23943), -INT16_C( 25094), -INT16_C( 29146), -INT16_C( 15235),  INT16_C(  4259),  INT16_C( 14876), -INT16_C(  4471),
         INT16_C( 27021), -INT16_C( 25562), -INT16_C(  8767),  INT16_C(  1368),  INT16_C(  4334),  INT16_C( 11043),  INT16_C( 15920),  INT16_C( 26349),
         INT16_C( 26295), -INT16_C( 19960),  INT16_C( 12035), -INT16_C( 32448), -INT16_C(  7181),  INT16_C(  3985),  INT16_C(  6685), -INT16_C( 21762),
         INT16_C(  9347),  INT16_C( 17478), -INT16_C( 24831), -INT16_C(  4023),  INT16_C( 28079), -INT16_C(  8421),  INT16_C(  2219),  INT16_C( 25413) },
      {  INT16_C( 20079),  INT16_C( 29205),  INT16_C( 21885),  INT16_C( 28915), -INT16_C( 31688),  INT16_C( 21887),  INT16_C( 32158),  INT16_C(  8703),
         INT16_C( 18082), -INT16_C( 23706), -INT16_C( 20507), -INT16_C( 27501), -INT16_C( 20964), -INT16_C( 14220), -INT16_C( 17993),  INT16_C(  9771),
         INT16_C( 16391), -INT16_C( 31592), -INT16_C( 29547), -INT16_C( 12812),  INT16_C( 29712), -INT16_C( 20702),  INT16_C(  8689), -INT16_C( 27696),
         INT16_C( 13927),  INT16_C( 19511), -INT16_C( 13594),  INT16_C(   737),  INT16_C( 21881),  INT16_C( 12490), -INT16_C(  2802),  INT16_C(  5718) },
      { -INT32_C(    76203945), -INT32_C(  1391938780),  INT32_C(           0),  INT32_C(   439471295),  INT32_C(  1094566494),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),
         INT32_C(           0),  INT32_C(    60125631),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(   509218309),  INT32_C(   139093896) } },
    { UINT16_C(60981),
      { -INT16_C( 13670), -INT16_C( 28806), -INT16_C( 29801), -INT16_C( 18173), -INT16_C(  3014),  INT16_C(  2779),  INT16_C( 17032), -INT16_C( 16575),
         INT16_C( 10127),  INT16_C( 28809),  INT16_C(   553), -INT16_C(  2875), -INT16_C( 11470), -INT16_C( 30487),  INT16_C(  8169), -INT16_C( 31625),
        -INT16_C(  3607), -INT16_C( 32493),  INT16_C(  5756), -INT16_C( 18886),  INT16_C(  5386), -INT16_C( 27967),  INT16_C(   600), -INT16_C(  6319),
        -INT16_C(  9431),  INT16_C( 21079),  INT16_C(  7389),  INT16_C(  4166),  INT16_C( 12527), -INT16_C(  9832),  INT16_C(  3919),  INT16_C( 14429) },
      {  INT16_C( 28673),  INT16_C( 32185), -INT16_C(  2938), -INT16_C( 28620), -INT16_C(  2807),  INT16_C( 24867),  INT16_C( 29943),  INT16_C(  8264),
        -INT16_C( 24753),  INT16_C( 11634), -INT16_C( 17989), -INT16_C( 21699), -INT16_C( 10775),  INT16_C( 14468), -INT16_C(  7707), -INT16_C(  6544),
         INT16_C( 10833), -INT16_C( 10397), -INT16_C( 26850),  INT16_C( 10087), -INT16_C( 30068), -INT16_C( 31863), -INT16_C( 11777),  INT16_C( 20131),
         INT16_C(  5745),  INT16_C( 11387), -INT16_C( 18225), -INT16_C( 18217),  INT16_C( 23438),  INT16_C( 29680),  INT16_C( 24636), -INT16_C( 29351) },
      { -INT32_C(  1319081020),  INT32_C(           0),  INT32_C(    77565691),  INT32_C(           0),  INT32_C(    84490275),  INT32_C(    52436708),  INT32_C(           0),  INT32_C(           0),
         INT32_C(           0), -INT32_C(   345051682),  INT32_C(   729166273), -INT32_C(   134273989),  INT32_C(           0), -INT32_C(   210556547),  INT32_C(     1794066), -INT32_C(   326957095) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi16(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi16(test_vec[i].b);
    simde__m512i r = simde_mm512_maskz_madd_epi16(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_mask_madd_epi16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_maskz_madd_epi16)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_mask_madd_epi16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_maskz_madd_epi16)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_madd_epi16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_madd_epi16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_madd_epi16)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
