// SPDX-License-Identifier: GPL-2.0-or-later
// SPDX-FileCopyrightText: 2024 Konrad Twardowski

// A date/time-based trigger

#pragma once

#include "../plugins.h"

#include <QDateTimeEdit>

class DateTimeEdit final: public QDateTimeEdit {
public:
	explicit DateTimeEdit();
	virtual ~DateTimeEdit() = default;
private:
	Q_DISABLE_COPY(DateTimeEdit)
	void onLineEditSelectionChange();
};

class DateTimeTriggerBase: public Trigger {
public:
	explicit DateTimeTriggerBase(const QString &text, const QString &iconName, const QString &id);
	virtual ~DateTimeTriggerBase();
	virtual bool canActivateAction() override;
	static QString formatSecsTo(const seconds &secsTo, const bool lcd = false);
	QTime getSelectedTime() const { return m_dateTimeEdit->time(); }
	QDateTime getSelectedDateTime() const { return m_dateTimeEdit->dateTime(); }
	void setSelectedDateTime(const QDateTime &dateTime) { m_dateTimeEdit->setDateTime(dateTime); }
	virtual void initContainerWidget() final override;
	virtual void initDateTimeWidget(QDateTimeEdit *dateTimeEdit) = 0;
	static QString longDateTimeFormat();
	static QLocale longDateTimeLocale();
	virtual void readConfig() final override;
	virtual void writeConfig() final override;
	virtual void setState(const State state) override;
protected:
	QDateTime m_endDateTime;
	virtual QDateTime calcEndTime(const QDateTime &now);
	void setSelectedTime(const QTime &time) { m_dateTimeEdit->setTime(time); }
	void updateProgressWidgets(const seconds &secsTo, const seconds &totalSecs);
	virtual void updateStatus();
private:
	Q_DISABLE_COPY(DateTimeTriggerBase)
	seconds m_totalSecs = 0s;
	std::shared_ptr<DateTimeEdit> m_dateTimeEdit;
	seconds createStatus(const QDateTime &now);
};

class DateTimeTrigger final: public DateTimeTriggerBase {
public:
	explicit DateTimeTrigger();

// TODO: rename #api
	virtual QString getStringOption() override;

	void setSelectedDateTimeAdjusted(const QDateTime &dateTime);
	virtual void setStringOption(const QString &option) override;
	virtual void initDateTimeWidget(QDateTimeEdit *dateTimeEdit) override;
private:
	Q_DISABLE_COPY(DateTimeTrigger)
};

class NoDelayTrigger final: public Trigger {
public:
	explicit NoDelayTrigger();
	virtual bool canActivateAction() override { return true; }
	virtual void initContainerWidget() override { }
private:
	Q_DISABLE_COPY(NoDelayTrigger)
};

class TimeFromNowTrigger final: public DateTimeTriggerBase {
public:
	explicit TimeFromNowTrigger();
	virtual QString getStringOption() override;
	virtual void setStringOption(const QString &option) override;
	virtual void initDateTimeWidget(QDateTimeEdit *dateTimeEdit) override;
protected:
	virtual QDateTime calcEndTime(const QDateTime &now) override;
private:
	Q_DISABLE_COPY(TimeFromNowTrigger)
};
