/****************************************************************************** 
 *
 * File:        midiqueue.c
 * Version:     $Id: midiqueue.c,v 1.9 1995/07/12 02:27:09 burgaard Exp $
 *              $Version: 0.2$
 *
 * Project:     MIDI/Sequencer library.
 *              Roland MIDI-401 Device driver.
 * Authors:     Kim Burgaard, <burgaard@daimi.aau.dk>
 * Copyrights:  Copyright (c) 1994, 1995 Kim Burgaard.
 *
 *      This package is free software; you can redistribute it and/or modify it
 *      under the terms of the GNU General Public License as published by the
 *      Free Software Foundation; either version 2, or (at your option) any
 *      later version.
 *
 *      This package is distributed in the hope that it will be useful, but
 *      WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *      Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License along
 *      with this program; see the file COPYING. If not, write to the Free
 *      Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************************************
 *
 * Minor queue algorith optimization by ip@nessie.mcc.ac.uk (Ian Pellfreeman).
 *
 * Compile and see ../doc/libmidifile.tex for documentation on this file.
 *
 * The queue algorithm is optimized according to the following assumptions:
 *
 * + Most often the operations on the queue are monotoneous, i.e. either a series of
 *   insertions or a series of removals.
 *
 * + Insertions most often happens to be sorted with respect to time. There is actually
 *   only problems when events are merged from for example two MIDI tracks or to
 *   recordings.
 *
 * + Each time one starts to put a new series on the queue, the current field must be
 *   cleared.
 *
 * + It is then assumed that there is a very high probability that the next event should
 *   be inserted right after the previous - or at least some few events further down.
 *   So instead of scanning the whole queue everytime an insertions is made, we first
 *   check if we can use the previous inserted event as an offset.
 *
 ******************************************************************************/

/*** INCLUDES & DEFINES *******************************************************/

#include <stddef.h>
#include <errno.h>

#include "miditypes.h"
#include "midiqueue.h"

/*** MIDI-401 DEVICE MIDI EVENT QUEUE ******************************************/

long midi_queue_put_least(midi_queue * queue, midi_event * event)
{
  midi_event * tempe;
  
  if (!queue || !event) return -EINVAL;

  if (queue->count <= 0) /* FIRST EVER */
    {
      queue->head = queue->tail = event;
      event->prev = event->next = NULL;
      queue->count = 0;
    }
  else if (event->time > queue->tail->time) /* BIGGEST */
    {
      queue->tail->next = event;
      event->next = NULL;
      event->prev = queue->tail;
      queue->tail = event;
    }
  else if (event->time < queue->head->time) /* SMALLEST */
    {
      queue->head->prev = event;
      event->prev = NULL;
      event->next = queue->head;
      queue->head = event;
    }
  else /* IN BETWEEN */
    {
      tempe = queue->current;
      while (tempe && (event->time <= tempe->time)) tempe = tempe->prev;
      if (!tempe) tempe = queue->head;
      while (tempe && (event->time > tempe->time)) tempe = tempe->next;
      if (!tempe)
	{
	  queue->tail->next = event;
	  event->next = NULL;
	  event->prev = queue->tail;
	  queue->tail = event;
	}
      else
	{
	  event->next = tempe;
	  event->prev = tempe->prev;
	  tempe->prev = event;
	  if (event->prev) event->prev->next = event;
	  else queue->head = event;
	};
    };
  queue->current = event;
  return ++queue->count;
}

long midi_queue_put_least_event(midi_queue *q, byte tp, long tm, word sz, byte *dt)
{
  midi_event *event = midi_event_alloc();
  if (!event) return -ENOMEM;

  event->type = tp;
  event->time = tm;
  event->size = sz;
  event->data = dt;

  return midi_queue_put_least(q, event);
}

long midi_queue_put(midi_queue *queue, midi_event *event)
{
  midi_event * tempe;
  
  if (!queue || !event) return -EINVAL;

  if (queue->count <= 0) /* FIRST EVER */
    {
      queue->head = queue->tail = event;
      event->prev = event->next = NULL;
      queue->count = 0;
    }
  else if (event->time >= queue->tail->time) /* BIGGEST */
    {
      queue->tail->next = event;
      event->next = NULL;
      event->prev = queue->tail;
      queue->tail = event;
    }
  else if (event->time < queue->head->time) /* SMALLEST */
    {
      queue->head->prev = event;
      event->prev = NULL;
      event->next = queue->head;
      queue->head = event;
    }
  else /* IN BETWEEN */
    {
      tempe = queue->current;
      while (tempe && (event->time < tempe->time)) tempe = tempe->prev;
      if (!tempe) tempe = queue->head;
      while (tempe && (event->time >= tempe->time)) tempe = tempe->next;
      if (!tempe)
	{
	  queue->tail->next = event;
	  event->next = NULL;
	  event->prev = queue->tail;
	  queue->tail = event;
	}
      else
	{
	  event->next = tempe;
	  event->prev = tempe->prev;
	  tempe->prev = event;
	  if (event->prev) event->prev->next = event;
	  else queue->head = event;
	};
    };
  queue->current = event;
  return ++queue->count;
}


long midi_queue_put_event(midi_queue *q, byte tp, long tm, word sz, byte *dt)
{
  midi_event *event = midi_event_alloc();
  if (!event) return -ENOMEM;
    
  event->type = tp;
  event->time = tm;
  event->size = sz;
  event->data = dt;
    
  return midi_queue_put(q, event);
}

long midi_queue_at_put(midi_queue *queue, midi_event *atevent, midi_event *event)
{
  if (!queue || !event) return -EINVAL;
  else if (!queue->count || !atevent || atevent->time != event->time || atevent == queue->tail)
    return midi_queue_put(queue, event);
  else if (atevent == queue->head) return midi_queue_put_least(queue, event);
    
  event->prev = atevent->prev;
  event->next = atevent;
  atevent->prev = event;
  if (event->prev) event->prev->next = event;

  queue->current = event;
  return ++queue->count;
}

long midi_queue_at_put_event(midi_queue *q, midi_event *atevent, byte tp, long tm, word sz, byte *dt)
{
  midi_event * event = midi_event_alloc();
  if (!event) return -ENOMEM;
    
  event->type = tp;
  event->time = tm;
  event->size = sz;
  event->data = dt;
    
  return midi_queue_at_put(q, atevent, event);
}

midi_event * midi_queue_get(midi_queue *queue)
{
  midi_event * tmp;

  if (!queue) return NULL;
  else if (queue->count > 1)
    {
      tmp = queue->head;
      queue->head = queue->head->next;
      queue->head->prev = NULL;
      queue->count--;
      if (tmp == queue->current) queue->current = queue->head;
      return tmp;      
    }
  else if (queue->count == 1)
    {
      tmp = queue->head;
      queue->head = queue->tail = queue->current = NULL;
      queue->count = 0;
      return tmp;
    };

  queue->count = 0;
  return queue->head = queue->tail = queue->current = NULL;
}

midi_event * midi_queue_get_highest(midi_queue *queue)
{
  midi_event * tmp;

  if (!queue) return NULL;
  else if (queue->count > 1)
    {
      tmp = queue->tail;
      queue->tail = queue->tail->prev;
      queue->tail->next = NULL;
      queue->count--;
      if (tmp == queue->current) queue->current = queue->tail;
      return tmp;
    }
  else if (queue->count == 1)
    {
      tmp = queue->head;
      queue->head = queue->tail = queue->current = NULL;
      queue->count = 0;
      return tmp;
    };

  queue->count = 0;
  return queue->head = queue->tail = queue->current = NULL;
}

inline void midi_queue_remove(midi_queue *queue)
{
  midi_event_free(midi_queue_get(queue));
}
inline void midi_queue_remove_highest(midi_queue *queue)
{
  midi_event_free(midi_queue_get_highest(queue));
}
void midi_queue_remove_event(midi_queue *queue, midi_event *event)
{
  if (!queue) return;
  else if (event == queue->head) midi_queue_remove(queue);
  else if (event == queue->tail) midi_queue_remove_highest(queue);
  else /* We trust it really is a member of this queue - its too expensive to check... */
    {
      if (event->next) event->next->prev = event->prev;
      if (event->prev) event->prev->next = event->next;
      event->prev = event->next = NULL;
      midi_event_free(event);
    };
}

inline midi_event * midi_queue_peek(midi_queue *queue)
{
  return (queue) ? queue->head : NULL;
}
inline midi_event * midi_queue_peek_highest(midi_queue *queue)
{
  return (queue) ? queue->tail : NULL;
}

void midi_queue_swap(midi_queue *queue, midi_event *a, midi_event *b)
{
  static midi_event t;
    
  if (!queue || !a || !b) return;
    
  if (a == queue->head) queue->head = b;
  if (a == queue->tail) queue->tail = b;
    
  if (b == queue->head) queue->head = a;
  if (b == queue->tail) queue->tail = a;
    
  if (a->next) a->next->prev = b;
  if (a->prev) a->prev->next = b;
    
  if (b->next) b->next->prev = a;
  if (b->prev) b->prev->next = a;
    
  t.next = a->next;
  t.prev = a->prev;
    
  a->prev = b->prev;
  a->next = b->next;
    
  b->prev = t.prev;
  b->next = t.next;
}

inline void midi_queue_reset(midi_queue *queue)
{
  if (!queue) return;
  queue->head = queue->tail = queue->current = NULL;
  queue->count = 0;
}

void midi_queue_flush(midi_queue *queue)
{
  midi_event * tmp;
    
  if (!queue) return;
    
  queue->current = queue->head;
  while (queue->current)
    {
      tmp = queue->current;
      queue->current = queue->current->next;
      midi_event_free(tmp);
    };
  midi_queue_reset(queue);
}

/*** End of File **************************************************************/
