/****************************************************************************** 
 *
 * File:        mpu401.h
 * Version:     $Id: mpu401.h,v 1.11 1995/07/27 05:01:32 burgaard Exp $
 *
 * Purpose:     Roland MPU-401 MIDI Interface Device Driver.
 *
 *
 * Project:     Roland MPU-401 Device driver.
 * Authors:     Kim Burgaard, <burgaard@daimi.aau.dk>
 * Copyrights:  Copyright (c) 1994 Kim Burgaard.
 *
 *      This package is free software; you can redistribute it and/or modify it
 *      under the terms of the GNU General Public License as published by the
 *      Free Software Foundation; either version 2, or (at your option) any
 *      later version.
 *
 *      This package is distributed in the hope that it will be useful, but
 *      WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *      Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License along
 *      with this program; see the file COPYING. If not, write to the Free
 *      Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************************************/

/*** INCLUDES & DEFINES *******************************************************/

#ifndef __MPU401_H
#define __MPU401_H

#include <linux/module.h>
#include <linux/version.h>

#include "miditypes.h"
#include "midiqueue.h"

/* MPU status bits */
#define MPU_DRR 0x40
#define MPU_DSR 0x80

/* #define DEVELOP 1 */

/*** CONSTANTS ****************************************************************/

/* Keep a log of the last 256 commands executed in the driver.
 * Usefull for a detailed inspection, when problems occur. */
#ifdef DEVELOP
#define MPU_ACTION_SIZE 256 /* store 256 last actions = 2048 bytes */
enum MPU_ACTION
{
  MPU_ACT_none,
  MPU_ACT_total_event_cnt,
  MPU_ACT_clear_run,
  MPU_ACT_clear_run_play,
  MPU_ACT_msdelay,
  MPU_ACT_printkevent,
  MPU_ACT_data_error,
  MPU_ACT_can_store,
  MPU_ACT_kmalloc,
  MPU_ACT_kfree_s,
  MPU_ACT_wait_DRR,
  MPU_ACT_wait_DSR,
  MPU_ACT_rd_dta,
  MPU_ACT_wr_dta,
  MPU_ACT_wr_cmd,
  MPU_ACT_RcDta,
  MPU_ACT_SnCmd,
  MPU_ACT_RqDta,
  MPU_ACT_SnDta,
  MPU_ACT_SnCmdDta,
  MPU_ACT_disable_irq,
  MPU_ACT_enable_irq,
  MPU_ACT_sleep,
  MPU_ACT_wake_up,
  MPU_ACT_midi_in,
  MPU_ACT_midi_out,
  MPU_ACT_store_control_event,
  MPU_ACT_control_out,
  MPU_ACT_metronome,
  MPU_ACT_irqhandler,
  MPU_ACT_request_irq,
  MPU_ACT_free_irq,
  MPU_ACT_request_ioports,
  MPU_ACT_release_ioports,
  MPU_ACT_send_sysex,
  MPU_ACT_flush_queues,
  MPU_ACT_reset_driver,
  MPU_ACT_reset,
  MPU_ACT_detect,
  MPU_ACT_ioctl_guest,
  MPU_ACT_ioctl,
  MPU_ACT_open,
  MPU_ACT_release,
  MPU_ACT_register_device,
  MPU_ACT_unregister_device,
  MPU_ACT_end_read,
  MPU_ACT_close_read,
  MPU_ACT_get_conf
};

enum MPU_ACTION_FLAGS
{
  MPU_FLG_IRQ = 0x01,
  MPU_FLG_FAK = 0x02,
  MPU_FLG_DEM = 0x04,
  MPU_FLG_END = 0x08,
  MPU_FLG_DEL = 0x10,
  MPU_FLG_REC = 0x20,
  MPU_FLG_SYN = 0x40
};

const char * mpu_action_names[] =
{
  "none",
  "total_event_cnt",
  "clear_run",
  "clear_run_play",
  "msdelay",
  "printkevent",
  "data_error",
  "can_store",
  "kmalloc",
  "kfree_s",
  "wait_DRR",
  "wait_DSR",
  "rd_dta",
  "wr_dta",
  "wr_cmd",
  "RcDta",
  "SnCmd",
  "RqDta",
  "SnDta",
  "SnCmdDta",
  "disable_irq",
  "enable_irq",
  "sleep",
  "wake_up",
  "midi_in",
  "midi_out",
  "store_control_event",
  "control_out",
  "metronome",
  "irqhandler",
  "request_irq",
  "free_irq",
  "request_ioports",
  "release_ioports",
  "send_sysex",
  "flush_queues",
  "reset_driver",
  "reset",
  "detect",
  "ioctl_guest",
  "ioctl",
  "open",
  "release",
  "register_device",
  "unregister_device",
  "end_read",
  "close_read",
  "get_conf"
};
#endif

/* error flags */
enum MPU_ERROR
{
  MPU_ERR_NONE,
  MPU_ERR_DSR,     /* DSR time out */
  MPU_ERR_DRR,     /* DRR time out */
  MPU_ERR_NULL,    /* NULL pointer unexpected */
  MPU_ERR_DEV_MSG, /* Unknown device message */
  MPU_ERR_Q_OFLO,  /* Queue overflow -- maximum event count reached */
  MPU_ERR_MEM,     /* Out of kernel space memory */
  MPU_ERR_DELAY,   /* mpu_msdelay called from interrupt */
  MPU_ERR_CMD,     /* Invalid MPU-401 Command (MPU_SET_IMMEDIATE) */
  MPU_ERR_UNKNOWN  /* Uncatagorized error */
};

#ifdef DEVELOP
const char * mpu_error_names[] =
{
  "none",
  "DSR time out",
  "DRR time out",
  "NULL pointer",  
  "Unknown device msg",
  "Queue overflow",
  "Out of memory",
  "msdelay from irq",
  "Invalid MPU cmd",
  "Unknown error"
};
#endif

/* status flags */
enum MPU_STATE
{
  MPU_STA_NONE    = 0x0000,
  MPU_STA_OPEN    = 0x0001,
  MPU_STA_PLAY    = 0x0100,
  MPU_STA_REC     = 0x0200,
  MPU_STA_OVERDUB = MPU_STA_PLAY|MPU_STA_REC,
  MPU_STA_STEPREC = 0x1000, /* store MIDI IN data also when not in RECORD mode */

  MPU_STA_IS_PLAY    = MPU_STA_OPEN|MPU_STA_PLAY,
  MPU_STA_IS_REC     = MPU_STA_OPEN|MPU_STA_REC,
  MPU_STA_IS_OVERDUB = MPU_STA_OPEN|MPU_STA_OVERDUB,
  MPU_STA_IS_STEPREC = MPU_STA_OPEN|MPU_STA_STEPREC
};

/* debug verbosity levels */
enum MPU_DEBUG
{
  MPU_DBG_NONE,
  MPU_DBG_SPARSE,
  MPU_DBG_VERBOSE,
  MPU_DBG_ALL
};

/* blocking messages */
enum MPU_BLOCK
{
  MPU_BLK_NONE       = 0x0000,
  MPU_BLK_VOICE      = 0x0001, /* unblocking caused by voice queue fatigue */
  MPU_BLK_CONTROL    = 0x0002, /* unblocking caused by control queue fatigue */
  MPU_BLK_RECORD     = 0x0004, /* unblocking caused by record queue ``overrun'' */
  MPU_BLK_DEMAND     = 0x0010, /* unblocking on demand */
  MPU_BLK_DELAY      = 0x0020, /* unblocking on delay */
  MPU_BLK_END        = 0x0040, /* unblocking on end */
  MPU_BLK_SYNC       = 0x0080  /* unblocking on sync (once per beat = denominator) */
};

/* MPU Track assignment */

enum MPU_TRACK
{
  MPU_TRK_VOICE     = 0x00, /* active track */
  MPU_TRK_IMMEDIATE = 0x01, /* immediate data */
  MPU_TRK_CONTROL   = 0x02, /* immediate data, voice events in control queue */
  MPU_TRK_METRONOME = 0x03, /* immediate data */
  MPU_TRK_TIMER     = 0x07  /* active track */
};

/*** TYPE DEFINITIONS *********************************************************/

#ifdef DEVELOP
typedef struct mpu_action_t
{
   long queue[MPU_ACTION_SIZE];
   long head;
   long tail;
   long count;
} mpu_action_t;
#endif

typedef struct mpu_version_t
{
  byte version;
  byte subversion;
  char revision[11];
} mpu_version_t;

typedef struct mpu_hardw_t
{
  byte irq;
  word ioport;
  word dtp;			/* data port */
  word cmp;			/* command port */
  word stp;			/* status port */

  unsigned long real_irq_cnt;
  unsigned long fake_irq_cnt;
  char mutex;
  char fake;

  long timeout_drr;
  long timeout_dsr;
  unsigned long loops_drr;
  unsigned long loops_dsr;
} mpu_hardw_t;

typedef struct mpu_queue_t
{
  midi_queue voice;
  midi_queue record;
  midi_queue control;

  long voice_time;
  long record_time;
  long control_time;

  unsigned long kmallocs;
  unsigned long kfrees;

  unsigned long voice_cnt;
  unsigned long sysex_cnt;
  unsigned long meta_cnt;
  unsigned long nop_cnt;
  unsigned long voice_hits;
} mpu_queue_t;

typedef struct mpu_demand_t
{
  word voice;
  word control;
  word record;
  char sync;
  enum MPU_BLOCK message;
} mpu_demand_t;

typedef struct mpu_block_t
{
  struct wait_queue *record;
  struct wait_queue *play_end;
  struct wait_queue *play_delay;
  struct wait_queue *demand;
  struct wait_queue *sync;
  mpu_demand_t bias;
} mpu_block_t;

/* Persistent (until next MPU_RESET that is) MPU switches */
typedef struct mpu_inisw_t
{
  char all_notes_off;		/* MPU default: ON,  driver default: ON  */
  char no_real_time;		/* MPU default: OFF, driver default: OFF */
  char all_thru;		/* MPU default: ON,  driver default: ON  */
  char dummy_time;		/* MPU default: OFF, driver default: ON  */
  char mode_mess;		/* MPU default: OFF, driver default: OFF */
  char SysEx_thru;		/* MPU default: OFF, driver default: ON  */
  char common2host;		/* MPU default: OFF, driver default: OFF */
  char realtime2host;		/* MPU default: OFF, driver default: OFF */
  char uart_mode;		/* MPU default: OFF, driver default: OFF */
} mpu_inisw_t;

/* MPU Channel reference tables */
typedef struct mpu_chtbl_t
{
  byte a_ch;			/* default: 0x00 = monitor channel 1 */
  byte b_ch;			/* default: 0x01 = monitor channel 2 */
  byte c_ch;			/* default: 0x02 = monitor channel 3 */
  byte d_ch;			/* default: 0x03 = monitor channel 4 */
} mpu_chtbl_t;

/* MPU run time switches */
typedef struct mpu_swtch_t
{
  char int_clk;		/* default: ON */
  char fsk_clk;		/* default: OFF */
  char mid_clk;		/* default: OFF */

  char metronome;	/* MPU hw metro. Don't use, it's not available on all models */
  char bender;		/* default: ON */
  char midi_thru;	/* default: ON */
  char step_rec;	/* default: OFF */
  char send_meas;	/* default: ON */
  char conductor;	/* default: ON */
  char realtime;	/* default: OFF */
  char fsk_to_midi;	/* default: OFF (OFF = FSK2INT) */
  char clk2host;	/* Always OFF */
  char SysEx2host;	/* default: ON */

  char ch_ref_a;        /* default: OFF */
  char ch_ref_b;	/* default: OFF */
  char ch_ref_c;	/* default: OFF */
  char ch_ref_d;	/* default: OFF */
} mpu_swtch_t;

typedef struct mpu_metrosetup_t
{
  byte numerator;               /* default: 4 (metronome beats per measure) */
  byte denominator;             /* default: 4 (midi clocks per metronome beat) */
  byte channel;			/* default is channel 10 */
  byte click_key;		/* default is key #33 = GS metro click */
  byte click_velo;		/* default is 100 */
  byte acc_click_key;		/* default is key #33 GS metro click */
  byte acc_click_velo;		/* default is 127 */
  byte meas_click_key;		/* default is key #34 GS metro bell */
  byte meas_click_velo;		/* default is 127 */
} mpu_metrosetup_t;

typedef struct mpu_metro_t
{
  mpu_metrosetup_t setup;
  long time;
  long subtime;                 /* same as time, except it is reset when a TIME SIG is send */
  char status;
  byte timebase;                /* default: 120 */
  byte bpm;                     /* default: 100 */
  long mysectempo;              /* default: 60000 = 100 bpm */
  byte rel_bpm;                 /* default: 0x40 (1/1 ratio) */
  byte grad_bpm;                /* default: 0x00 (immediate) */
  byte midi_metro;              /* default: 24 = once per 1/4 note */
  byte int_clk2host;            /* default: 240 = twice per beat */
} mpu_metro_t;

typedef struct mpu_track_t
{
  word channelmap;		/* default: 0xffff = accept input from all channels */
  char send_play_cnt;           /* default: ON */
} mpu_track_t;

/* running voice and control messages */
typedef struct mpu_runmsg_t
{
  byte play;
  byte record;

  byte control_data[512]; 
  word control_size;
  char control;
} mpu_runmsg_t;

/* This structure is used for passing information from the driver to applications 
 * i.e. it's read only... */
typedef struct mpu_status_t
{
  mpu_hardw_t hardware;

  mpu_version_t mpu_ver;
  mpu_version_t drv_ver;

  unsigned long status;
  enum MPU_DEBUG debug;
  enum MPU_ERROR error;

  mpu_queue_t queues;
  mpu_block_t blocking;

  mpu_metro_t metronome;
  mpu_inisw_t init_switches;
  mpu_swtch_t switches;
} mpu_status_t;

/*** ENUMERATIONS *************************************************************/

/* Bitmaps for different sequencer modes */
enum MPU_MODE
{
  MPU_MOD_STOP_MIDI   = 1,	/* %xxxx01 */
  MPU_MOD_START_MIDI  = 1<<1,	/* %xxxx10 */
  MPU_MOD_CONT_MIDI   = (1<<1) | 1, /* %xxxx11 */

  MPU_MOD_STOP_PLAY   = 1<<2,	/* %xx01xx */
  MPU_MOD_START_PLAY  = 1<<3,	/* %xx10xx */

  MPU_MOD_STOP_REC    = 1<<4,	/* %01xxxx */
  MPU_MOD_START_REC   = 1<<5,	/* %10xxxx */
  MPU_MOD_STANDBY_REC = MPU_MOD_START_REC
};

/* These are the commands issued for start/stop/continue of sequencing */
enum MPU_SEQUENCE
{
  MPU_SEQ_START_ALL     = MPU_MOD_START_MIDI|MPU_MOD_START_PLAY|MPU_MOD_START_REC,
  MPU_SEQ_STOP_ALL      = MPU_MOD_STOP_MIDI|MPU_MOD_STOP_PLAY|MPU_MOD_STOP_REC,

  MPU_SEQ_START_PLAY    = MPU_MOD_START_MIDI|MPU_MOD_START_PLAY,
  MPU_SEQ_CONT_PLAY     = MPU_MOD_CONT_MIDI|MPU_MOD_START_PLAY,
  MPU_SEQ_STOP_PLAY     = MPU_MOD_STOP_MIDI|MPU_MOD_STOP_PLAY,

  MPU_SEQ_STANDBY_REC   = MPU_MOD_STANDBY_REC,
  MPU_SEQ_START_REC     = MPU_MOD_START_MIDI|MPU_MOD_START_REC,
  MPU_SEQ_CONT_REC      = MPU_MOD_CONT_MIDI|MPU_MOD_START_REC,
  MPU_SEQ_STOP_REC      = MPU_MOD_STOP_MIDI|MPU_MOD_STOP_REC,

  MPU_SEQ_START_OVERDUB = MPU_SEQ_START_ALL,
  MPU_SEQ_CONT_OVERDUB  = MPU_MOD_CONT_MIDI|MPU_MOD_START_PLAY|MPU_MOD_START_REC,
  MPU_SEQ_STOP_OVERDUB  = MPU_SEQ_STOP_ALL
};

/* All switches below can *ONLY* be escaped with an MPU_CMD_RESET!!! */
enum MPU_INIT_SWITCH
{
  MPU_INI_NOTES_OFF_ALL  = 0x30,
  MPU_INI_NO_REAL_TIME   = 0x32,
  MPU_INI_ALL_THRU_OFF   = 0x33,
  MPU_INI_TIMING_BYTE_ON = 0x34, /* Send dummy leading time byte (0x00) when in 
				  * DATA_IN STOP mode */
  MPU_INI_MODE_MESS_ON   = 0x35, /* Don't mask MIDI IN to HOST */
  MPU_INI_EXL_THRU_ON    = 0x37, /* Send all exclusive msg from IN to OUT */
  MPU_INI_COMMON_TO_HOST = 0x38, /* Send System Common msg to HOST */
  MPU_INI_REAL_TIME      = 0x39, /* Send REAL TIME msg to HOST */

  /* Set MPU in dumb UART MODE, only recognizing MPU_CMD_RESET (0xFF)
   * Note: When issuing an MPU_CMD_RESET to escape UART mode, *NO* ACK will be
   * sent from MPU! */
  MPU_INI_UART_MODE      = 0x3F
};

/* These COMMANDs can be issued to SET MPU states, modes etc. */
enum MPU_SET
{
  MPU_SET_TIME_BASE     = 0xC2,

  /* SET_TIME_BASE *MUST* be added with 0x00 thru 0x06,
   * Selecting one of the following clock resolutions:
   *
   * 0: Time base  48 per beat
   * 1: Time base  72 per beat
   * 2: Time base  96 per beat
   * 3: Time base 120 per beat
   * 4: Time base 144 per beat
   * 5: Time base 168 per beat
   * 6: Time base 192 per beat */

  /* The following values *MUST* be or'd with desired MIDI CHANNEL */
  MPU_SET_CH_REF_TBL_A  = 0x40,
  MPU_SET_CH_REF_TBL_B  = 0x50,
  MPU_SET_CH_REF_TBL_C  = 0x60,
  MPU_SET_CH_REF_TBL_D  = 0x70,

  /* METRONOME actions */
  MPU_SET_METRO_ON      = 0x83, /* Without accents */
  MPU_SET_METRO_OFF     = 0x84,
  MPU_SET_METRO_ACC_ON  = 0x85, /* (!) With accents */

  /* The following *MUST* be followed by additional DATA to MPU ("args") */
  MPU_SET_IMMEDIATE_FIRST = 0xE0,
  MPU_SET_TEMPO	          = 0xE0,
  MPU_SET_REL_TEMPO	  = 0xE1,
  MPU_SET_GRADUATION      = 0xE3, /* Set tempo graduation rate */

  /* The following are used to set misc. CLOCK conversions */
  MPU_SET_MIDI_METRO     = 0xE4, /* Set MIDI CLOCKs per METRONOME BEEP */
  MPU_SET_METRO_MEAS     = 0xE6, /* Set Beat per measure */
  MPU_SET_INT_HOST       = 0xE7, /* Set INTERNAL CLOCKS per CLOCK TO HOST */
  MPU_SET_TRACK_ACTIVE   = 0xEC, /* followed by bitmaped byte */
  MPU_SET_SEND_PLAY_CNT  = 0xED,
  MPU_SET_CH_1_8         = 0xEE, /* Bitmap of accepted channels 1 thru 8 */
  MPU_SET_CH_9_16        = 0xEF, /* Bitmap of accepted channels 9 thru 16 */
  MPU_SET_IMMEDIATE_LAST = 0xEF
};

/* FSK are only implemented on "true" MPU-IPC / MPU-401 interfaces!
 * I suggest not to use this feature unless *absolutely* needed!!! */
enum MPU_CLOCK
{
  MPU_CLK_INT_CLOCK    = 0x80,
  MPU_CLK_FSK_CLOCK    = 0x81,	/* (!) Frequency Shifted Key - TAPE SYNC */
  MPU_CLK_MIDI_CLOCK   = 0x82,

  MPU_CLK_FSK_TO_INT   = 0x92,	/* Resolution is INTERNAL TIMEBASE */
  MPU_CLK_FSK_TO_MIDI  = 0x93	/* Resolution is MIDI (24 per beat) */
};

/* switches are values for OFF, value + 1 is ON */
enum MPU_SWITCH
{
  MPU_SWI_BENDER        = 0x86,
  MPU_SWI_MIDI_THRU     = 0x88,
  MPU_SWI_DATA_IN_STOP  = 0x8A,
  MPU_SWI_SEND_MEAS_END = 0x8C,
  MPU_SWI_CONDUCTOR     = 0x8E,

  MPU_SWI_REAL_TIME_AFF = 0x90,

  MPU_SWI_CLK_TO_HOST   = 0x94,
  MPU_SWI_EXL_TO_HOST   = 0x96,

  MPU_SWI_CHAN_REF_A    = 0x98, /* Channel reference table A */
  MPU_SWI_CHAN_REF_B    = 0x9A, /* Channel reference table B */
  MPU_SWI_CHAN_REF_C    = 0x9C, /* Channel reference table C */
  MPU_SWI_CHAN_REF_D    = 0x9E  /* Channel reference table D */
};

/* This is the DATA that the HOST can receive from the MPU, */
enum MPU_MESSAGE
{
  MPU_MSG_MAX_TIME      = 0xEF,

  MPU_MSG_TIME_OVERFLOW = 0xF8,
  MPU_MSG_CONDUCTOR_REQ = 0xF9,
  MPU_MSG_DATA_END	 = 0xFC,
  MPU_MSG_ALL_END	 = 0xFC,
  MPU_MSG_CLOCK_TO_HOST = 0xFD,
  MPU_MSG_ACKNOWLEDGE   = 0xFE,
  MPU_MSG_MIDI_SYS_MSG  = 0xFF
};

/* This is the DATA that the HOST can receive during recording */
enum MPU_MARK
{
  MPU_MRK_NO_OP    = 0xF8,
  MPU_MRK_MEASURE  = 0xF9,
  MPU_MRK_DATA_END = 0xFC
};

/* Signal MPU that HOST WANT TO SEND DATA */
enum MPU_SEND
{
  /* SND_DATA (0xD0) *MUST* be or'd with MPU_TRACK[n], 1 <= n <= 8
   * May NOT be issued on active tracks!!!
   * e.g. MPU_SND_DATA|MPU_TRACK[5] signals MPU that DATA will be sent
   * to TRACK number 5 by HOST */
  MPU_SND_DATA  = 0xD0,
  MPU_SND_SYSEX = 0xDF
};

/* These REQUESTs can either be send/received to/from MPU */
enum MPU_REQUEST
{
  /* The following can only be SEND to MPU */
  MPU_REQ_PLAY_CNT = 0xA0,

  /* REQ_PLAY_CNT (0xA0) *MUST* by or'd with MPU_TRACK[n], 1 <= n <= 8
   *
   * i.e. MPU_REQ_PLAY_CNT|MPU_TRACK[3] request play counter for MPU TRACK 3
   * (do NOT confuse tracks with MIDI channels !!!) */

  MPU_REQ_REC_CNT    = 0xA0,	/* RECORD counter is cleared after request! */
  MPU_REQ_VERSION    = 0xAC,
  MPU_REQ_REVISION   = 0xAD,
  MPU_REQ_TEMPO      = 0xAF,

  /* The following can only be REVEICED from MPU */
  MPU_REQ_TRK        = 0xF0,

  /* REQ_TRK (0xF0) send by MPU is or'd with TRACK number 0x00 thru 0x07
   * (internally TRACK id's are zerobased!)
   *
   * i.e. MPU_REQ_TRK == 0xF3 request TRACK DATA for MPU TRACK 3
   * (do NOT confuse tracks with MIDI channels !!!) */

  MPU_REQ_CONDUCTOR  = 0xF9
};

/* MPU Clear commands. Normally used by MPU_Reset() */
enum MPU_CLEAR
{
  MPU_CLR_REL_TEMP = 0xB1,
  MPU_CLR_PLAY_CNT = 0xB8,
  MPU_CLR_PLAY_MAP = 0xB9,
  MPU_CLR_REC_CNT  = 0xBA,
  MPU_CLR_RESET    = 0xFF
};

#endif /* __MPU401_H */

/*** End of File **************************************************************/
