/* 
 * yppasswdd
 * Copyright 1994 Olaf Kirch, <okir@monad.swb.de>
 *
 * This program is covered by the GNU General Public License, version 2.
 * It is provided in the hope that it is useful. However, the author
 * disclaims ALL WARRANTIES, expressed or implied. See the GPL for details.
 */

#include <sys/types.h>
#include <sys/errno.h>
#include <sys/stat.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <pwd.h>
#ifdef SHADOWPWD
#include <shadow.h>
#endif

#include <syslog.h>
#include <stdio.h>
#include <string.h>

#include <rpc/rpc.h>
#include <rpc/pmap_clnt.h>
#include "yppasswd.h"


/* How often to retry locking the passwd file...
 */
#define MAX_RETRIES 5

#ifdef DEBUG
# define RPC_SVC_FG
# define _PATH_PASSWD    	"/tmp/passwd"
# define _PATH_OLDPASSWD 	"/tmp/passwd.OLD"
# define _PATH_PTMP  	 	"/tmp/ptmp"
#else
# ifndef _PATH_PASSWD
#  define _PATH_PASSWD   	"/etc/passwd"
# endif
# ifndef _PATH_OLDPASSWD
#  define _PATH_OLDPASSWD 	"/etc/passwd.OLD"
# endif
# ifdef SHADOWPWD
#  ifndef _PATH_SHADOW
#   define _PATH_SHADOW		"/etc/shadow"
#  endif
# endif
# ifndef _PATH_PTMP
#  define _PATH_PTMP	 	"/etc/ptmp"
# endif
#endif

#define xprt_addr(xprt)	(svc_getcaller(xprt)->sin_addr)
#define xprt_port(xprt)	ntohs(svc_getcaller(xprt)->sin_port)
void reaper( int sig );

/*===============================================================*
 * The /etc/passwd update handler
 *===============================================================*/
int *
yppasswdproc_pwupdate_1(yppasswd *yppw, struct svc_req *rqstp)
{
    struct xpasswd *newpw;	/* passwd struct passed by the client */
    struct passwd *pw;		/* passwd struct obtained from getpwent() */
    int		retries, gotit, fd, c;
    FILE	*oldfp, *newfp;
    static int	res;
    char	logbuf[255];

    res = 1;

    sprintf( logbuf, "update %.12s (uid=%d) from host %s",
			    yppw->newpw.pw_name,
			    yppw->newpw.pw_uid,
			    inet_ntoa(xprt_addr(rqstp->rq_xprt)));

    /* Lock the passwd file. We retry several times. Maybe it would be
     * better to just return an error condition and have the client reattempt
     * instead? This procedure is already slow enough...
     */
    retries = 0;
    while ((fd = open(_PATH_PTMP, O_CREAT|O_WRONLY|O_EXCL)) < 0
      && errno == EEXIST && retries < MAX_RETRIES) {
        sleep (1);
        retries++;
    }

    if (retries == MAX_RETRIES) {
        syslog ( LOG_NOTICE, "%s failed", logbuf );
        syslog ( LOG_NOTICE, "password file locked" );
        return &res;
    }

    if (fd < 0 || (newfp = fdopen(fd, "w")) == NULL) {
        syslog ( LOG_ERR, "%s failed", logbuf );
        syslog ( LOG_ERR, "Can't create %s. %m", _PATH_PTMP );
        close (fd);
        return &res;
    }

    /* Open the passwd file for reading. We can't use getpwent and friends
     * here, because they go through the YP maps, too.
     */
    if ((oldfp = fopen(_PATH_PASSWD, "r")) == NULL) {
        syslog ( LOG_ERR, "%s failed", logbuf );
        syslog ( LOG_ERR, "Can't open %s: %m", _PATH_PASSWD );
        fclose (newfp);
        return &res;
    }

    gotit = 0;
    newpw = &yppw->newpw;

    /*
     * Loop over all passwd entries
     */
    while((pw = fgetpwent(oldfp)) != NULL) {

        /* check if this is the uid we want to change. A few
         * sanity checks added for consistency.
         */
        if (newpw->pw_uid == pw->pw_uid && newpw->pw_gid == pw->pw_gid
         && !strcmp(newpw->pw_name, pw->pw_name) && !gotit) {

            /* Check the password.
             */
            if (strcmp(crypt(yppw->oldpass, pw->pw_passwd), pw->pw_passwd)) {
        	syslog ( LOG_ERR, "%s rejected", logbuf );
                syslog ( LOG_ERR, "Invalid password." );
                sleep(1);
                break;
            }

            /* set the new passwd
             */
            pw->pw_passwd = newpw->pw_passwd;
            gotit++;
        }

        /* write the passwd entry to /etc/ptmp
         */
        if (putpwent(pw, newfp) < 0) {
            syslog ( LOG_ERR, "%s failed", logbuf );
            syslog ( LOG_ERR, "Error while writing new password file: %m" );
            break;
        }
        /* fflush (newfp); */
    }
    fclose (newfp);
    fclose (oldfp);

    /* Check if we dropped out of the loop because of an error.
     * If so, return an error to the client.
     */
    if (pw != NULL) {
        unlink (_PATH_PTMP);
        return (&res);
    }

    /* Check whether we actually changed anything
     */
    if (!gotit) {
        syslog ( LOG_ERR, "%s failed", logbuf );
        syslog ( LOG_ERR, "User not in password file." );
        unlink (_PATH_PTMP);
        return (&res);
    }

    unlink (_PATH_OLDPASSWD);
    link (_PATH_PASSWD, _PATH_OLDPASSWD);
    unlink (_PATH_PASSWD);
    link (_PATH_PTMP, _PATH_PASSWD);
    unlink (_PATH_PTMP);
    chmod (_PATH_PASSWD, 0644);

    /* Fork off process to rebuild NIS passwd.* maps. If the fork
     * fails, restore old passwd file and return an error.
     */
    if ((c = fork()) < 0) {
    	unlink( _PATH_PASSWD );
    	link( _PATH_OLDPASSWD, _PATH_PASSWD );
    	syslog( LOG_ERR, "%s failed", logbuf );
    	syslog( LOG_ERR, "Couldn't fork map update process: %m" );
    	return (&res);
    }
    if (c == 0) {
    	execlp(MAP_UPDATE, MAP_UPDATE0, NULL);
    	exit(1);
    }

    syslog ( LOG_INFO, "%s successful. Password changed", logbuf );
    res = 0;

    return (&res);
}

#ifdef SHADOWPWD
/*===============================================================*
 * The /etc/shadow update handler
 *===============================================================*/
int *
yppasswdproc_spwupdate_1(yppasswd *yppw, struct svc_req *rqstp)
{
    struct xpasswd *newpw;	/* passwd struct passed by the client */
    struct spwd *spw;		/* passwd struct obtained from spw_locate */
    static int	res;		/* return value */
    int		retries;	/* number of retries to lock shadow file */
    int		c;
    char	logbuf[255];

    res = 1;

    sprintf( logbuf, "update %.12s (uid=%d) from host %s",
			    yppw->newpw.pw_name,
			    yppw->newpw.pw_uid,
			    inet_ntoa(xprt_addr(rqstp->rq_xprt)));

    /* Lock the passwd file. We retry several times. Maybe it would be
     * better to just return an error condition and have the client reattempt
     * instead? This procedure is already slow enough...
     */
    retries = 0;
    while (!spw_lock() && retries < MAX_RETRIES) {
        sleep (1);
        retries++;
    }

    if (retries == MAX_RETRIES) {
        syslog ( LOG_NOTICE, "%s failed", logbuf );
        syslog ( LOG_NOTICE, "password file locked" );
        return &res;
    }


    if (!spw_open(O_RDWR)) {
        syslog ( LOG_ERR, "%s failed", logbuf );
        syslog ( LOG_ERR, "Can't open %s: %m", _PATH_SHADOW);
        spw_unlock();
        return &res;
    }

    newpw = &yppw->newpw;

    /*
     * Get old shadow password entry
     */
    if (!(spw=spw_locate(newpw->pw_name))) {
        syslog ( LOG_ERR, "%s failed", logbuf );
        syslog ( LOG_ERR, "User not in password file." );
        spw_close();
	spw_unlock();
        return (&res);
    }

    /*
     * Check the password.
     */
    if (strcmp(crypt(yppw->oldpass, spw->sp_pwdp), spw->sp_pwdp)) {
	syslog ( LOG_ERR, "%s rejected", logbuf );
	syslog ( LOG_ERR, "Invalid password." );
	spw_close();
	spw_unlock();
	return (&res);
    }

    spw->sp_pwdp=newpw->pw_passwd;

    if (!spw_update(spw)) {
	syslog ( LOG_ERR, "%s failed", logbuf );
	syslog ( LOG_ERR, "Error while updating %s",  _PATH_SHADOW);
	spw_close();
	spw_unlock();
	return (&res);
    }

    spw_close();
    spw_unlock();


    /* Fork off process to rebuild NIS passwd.* maps. If the fork
     * fails, restore old passwd file and return an error.
     */
    if ((c = fork()) < 0) {
    	syslog( LOG_ERR, "%s failed", logbuf );
    	syslog( LOG_ERR, "Couldn't fork map update process: %m" );
    	return (&res);
    }
    if (c == 0) {
    	execlp(MAP_UPDATE, MAP_UPDATE0, NULL);
    	exit(1);
    }

    syslog ( LOG_INFO, "%s successful. Password changed", logbuf );
    res = 0;

    return (&res);
}
#endif
