/* -*- c -*- */
/*
 * Copyright (C) 1992-94 by Gustaf Neumann, Stefan Nusser
 *
 *      Wirtschaftsuniversitaet Wien,
 *      Abteilung fuer Wirtschaftsinformatik
 *      Augasse 2-6,
 *      A-1090 Vienna, Austria
 *      neumann@@wu-wien.ac.at, nusser@@wu-wien.ac.at
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appears in all copies and that both that
 * copyright notice and this permission notice appear in all supporting
 * documentation.  This software is provided "as is" without expressed or
 * implied warranty.
 *
 * Date: Dec 24 1994
 * Author: Gustaf Neumann
 * Version: 1.0.8
 */

#include <X11/IntrinsicP.h>
#include "wafe_quarks.h"

extern Boolean wafeMWMrunning(
#if NeedFunctionPrototypes
    Widget
#endif
);


#undef PIXMAPCACHETRACE
 /*
    * This file contains a new converter procedure and xpm support procedures:
    *
    *    A new StringToBitmap Converter, which is also able to convert
    *    Pixmaps (suffix .xpm). First it searches for a valid BitmapFile
    *    under the given filename; if the file format is invalid, it
    *    tries to interpret it as a pixmap file.
    */ 

#ifdef XPM
#include <xpm.h>
#include <X11/extensions/shape.h>
#endif
#include <X11/Xos.h>
#include <X11/Xmu/Xmu.h>


#ifdef XPM
 /* Create a linked list, which keeps track of all created pixmaps,
  * so that we can destroy Colorcells, if they are not longer needed.
  * Only the changePixmap command will use this feature, so pixmaps which
  * are set with setValues are still cached as usual by Xt!
  * Note that setting a pixmap with setValues therefore implies proctically
  * loosing these colorcells for the application's lifetime since Xt does
  * not see to free these!
  */
typedef struct _pixmapCache
     {
     Pixmap                pm;              /* Pixmap                */
     Pixmap                mask;            /* shape mask            */
     XpmAttributes        *attribPtr;       /* Colorcells, etc       */
     XrmQuark              fileName;        /* The filename..        */
     int                   count;           /* references            */
     struct _pixmapCache  *next;
     }  pixmapCache, *pixmapCachePtr;

static pixmapCachePtr pixmapCacheRoot = NULL;


 /* this is the per widget information about used pixmaps */
typedef struct _xpmList
     {
     XrmQuark         resName;  
     XrmQuark         fileName;
     Display         *dpy;
     Colormap         cmap;
     struct _xpmList *next;
     } xpmList, *xpmListPtr;
#endif /* XPM */

#ifdef MOTIF
typedef struct 
     {
     Pixmap    pixmap;
     Screen   *screen;
     } pixrec;
#endif /* MOTIF */

/* provide additional arguments for the pixmap converter: 
 */
static XtConvertArgRec xpmConvertArgsPixmap[] = {
  {XtWidgetBaseOffset, (XtPointer) XtOffsetOf(WidgetRec, core.window), 
   sizeof(Window)},
  {XtImmediate, (XtPointer)True, sizeof(Boolean)}
};


static XtConvertArgRec xpmConvertArgsBitmap[] = {
  {XtWidgetBaseOffset, (XtPointer) XtOffsetOf(WidgetRec, core.window), 
   sizeof(Window)}
};
   

/* try hard to locate bitmap or pixmap file */
static char *
locateBitmap(dpy, name, toPixmap)
Display *dpy;
char * name;
Boolean toPixmap;
    {
    char *fn;

    /* is the name enough? */
    if (access(name, R_OK) == 0)
	return name;
    else
	{
	if (!(fn = XtResolvePathname(dpy, "bitmaps", name, "", 
				     wafeFileSearchPath, NULL, 0, NULL)))
	    {
	    XtStringConversionWarning(name, toPixmap ? XtRPixmap : XtRBitmap);
	    wafeWarn("convert", "Wafe's current FILESEARCHPATH is <%s>", 
		     wafeFileSearchPath,NULL,NULL);  
	    }
	return fn;
	}
    }


#ifdef XPM
static void 
setPixmap(widget, resource, qResName, pm, mask, width, height)
Widget        widget;
String        resource;
XrmQuark      qResName;
Pixmap        pm, mask;
unsigned int  width, height;
    {
    Display      *dpy = XtDisplay(widget);
    Window        window, root;
    Pixel         bpix;
    int           x,y;
    unsigned int  w,h,b,d;

#ifdef ATHENA
    /* XtVaSetValues fails nowadays on Label bitmaps with XtUnspecifiedPixmap*/
    if (pm == XtUnspecifiedPixmap) 
	return;
#endif
#ifdef MOTIF
    if (!pm) 
	pm = XmUNSPECIFIED_PIXMAP;
#endif

    if (qResName != qiconPixmap)
	{
	XtVaSetValues(widget, resource, pm, NULL);

/*	if ((pm != XtUnspecifiedPixmap) && (window = XtWindow(widget))) */
	if ((window = XtWindow(widget)))
	    if (mask)
		{
		int xOff=0, yOff=0;
		XGetGeometry(dpy,window,&root,&x,&y,&w,&h,&b,&d);
		if (w<width || h<height)
		    {
		    wafeWarn("changePixmap",
			"Shape mask for %s of %s larger than window; ignored", 
			resource, wafeWidgetToName(widget), NULL);
		    mask = (Pixmap) NULL;
		    } 
		else 
		    {
#ifdef MOTIF
		    unsigned char alignment = 0;
		    Dimension margin_left,margin_right,margin_width,
		       margin_height,margin_top,
		       highlight_thickness, shadow_thickness,
		       wOffset, hOffset;
		
		    XtVaGetValues(widget,
				  XmNalignment,          &alignment,
				  XmNmarginLeft,         &margin_left,
				  XmNmarginRight,        &margin_right,
				  XmNmarginWidth,        &margin_width,
				  XmNmarginHeight,       &margin_height,
				  XmNmarginTop,          &margin_top,
				  XmNhighlightThickness, &highlight_thickness,
				  XmNshadowThickness,    &shadow_thickness,
				  NULL);

		    wOffset = margin_width  +
			highlight_thickness + shadow_thickness;
		    hOffset = margin_height +
			highlight_thickness + shadow_thickness;
		    
		    if (alignment == XmALIGNMENT_BEGINNING)
			{
			xOff = wOffset + margin_left;
			yOff = hOffset + margin_top;
			}
		    else if (alignment == XmALIGNMENT_END)
			{
			if (w>width)  xOff = (w - width ) - 
			    (wOffset + margin_right);
			if (h>height) yOff = (h - height) -
			    (hOffset + margin_top);
			}
		    else /* (alignment == XmALIGNMENT_CENTER) */
			{
			if (w>width)  xOff = (w - width ) / 2;
			if (h>height) yOff = (h - height) / 2;
			}
#else
		    if (w>width)  xOff = (w - width ) / 2;
		    if (h>height) yOff = (h - height) / 2;
#endif /* MOTIF */
		    }
		XShapeCombineMask(dpy, window, ShapeBounding, xOff,yOff, 
				  mask, ShapeSet);
		XClearWindow(dpy,window);
		}
	    else
		{
		XShapeCombineMask(dpy, window, ShapeBounding, 0,0, 
				  mask, ShapeSet);
		}
	else
	    if (mask)
		wafeWarn("changePixmap",
			 "Cannot set shape mask for unrealized widget %s", 
			 wafeWidgetToName(widget), NULL, NULL);

	return;
	}

    /* icon pixmaps are more complicate to handle:
       0) ignore calls to Unspecified Pixmaps
       1) get or create the icon window
       2) reconfigure it.
       3) under mwm and friends the rules for handling icons
          are different, since the icons are created withing a 
          surrounding frame
     */

    if (pm == (Pixmap)NULL 
	|| pm == None
	|| pm == ParentRelative
	|| pm == XtUnspecifiedPixmap
#ifdef MOTIF
	|| pm == XmUNSPECIFIED_PIXMAP
#endif
	) return;

    root = RootWindow(dpy, DefaultScreen(dpy));
    bpix = XBlackPixelOfScreen(DefaultScreenOfDisplay(dpy));

    XtVaGetValues(widget, XtNiconWindow, &window, NULL);
    if (!window)
	{
	window = XCreateSimpleWindow(dpy, root, 0, 0, 
				     width, height, 1, 
				     bpix, bpix);
	XtVaSetValues(widget, XtNiconWindow, window, NULL);
	}
    else 
	{
	XWindowChanges values;
	
	values.width  = width;
	values.height = height;

	/* reconfigure uglifies icon under MWM */
	if (wafeMWMrunning(widget))
	    {
	    XIconSize *size_list;
	    int count = 0;

	    XGetGeometry(dpy,window,&root,&x,&y,&w,&h,&b,&d);
	    XGetIconSizes(dpy, root, &size_list, &count);
	    if (count>0) 
		{
		if (values.width  > size_list->max_width)
		    values.width  = size_list->max_width;
		if (values.height > size_list->max_height)
		    values.height = size_list->max_height;

		values.x = (size_list->max_width  - values.width ) / 2 + 6;
		values.y = (size_list->max_height - values.height) / 2 + 6;

		XFree((char *)size_list);
		XReconfigureWMWindow(dpy,window, DefaultScreen(dpy),
				     CWX|CWY|CWWidth|CWHeight, &values);
		}
	    }
	else
	    XReconfigureWMWindow(dpy,window, DefaultScreen(dpy),
				 CWWidth|CWHeight, &values);
	}

    XSetWindowBackgroundPixmap(dpy, window, pm);
    XShapeCombineMask(dpy, window, ShapeBounding, 0, 0, mask, ShapeSet);
    XClearWindow(dpy,window);
    }



#ifdef PIXMAPCACHETRACE 
static void showPixmapCache(string)
String string;
    {
    pixmapCachePtr   pptr;
    fprintf(stderr,"Pixmap cache: %s\n",string);
    for (pptr=pixmapCacheRoot;  pptr != NULL; pptr=pptr->next)
	fprintf(stderr,"   %s (%x): count=%d\n",
		XrmQuarkToString(pptr->fileName),
		(int)pptr->pm,
		pptr->count
		);
    fprintf(stderr,"------------\n");
    }
#endif


/* Cleanup last settings:
 * Search the list of pixmaps to find out whether the specified pixmap's 
 * resources are to be freed - which is the case when it was created by
 * changePixmap
 */
static Pixmap 
freeInPixmapCache(dpy,cmap,qFileName)
Display    *dpy;
Colormap    cmap;
XrmQuark    qFileName;
    {
    pixmapCachePtr   pptr, lastPtr = NULL;
    Pixmap          toDestroy = (Pixmap) NULL;

    DBUG_ENTER("freeInPixmapCache");

#ifdef PIXMAPCACHETRACE 
    showPixmapCache("before Free");
#endif

    for (pptr=pixmapCacheRoot;  pptr != NULL; lastPtr=pptr, pptr=pptr->next)
	if (pptr->fileName == qFileName)
	    {
	    pptr->count--;
	    if (pptr->count < 1)
		{

		/* destroy Pixmap and deallocate resources */
		DBUG_PRINT("xpm",
			   ("deallocating pixmap %s (%x), %d colors", 
			    XrmQuarkToString(pptr->fileName), 
			    pptr->pm,
			    pptr->attribPtr->npixels));

		toDestroy = pptr->pm;
		if (pptr->attribPtr->depth > 1) 
		    XFreeColors(dpy, cmap, 
				pptr->attribPtr->pixels, 
				pptr->attribPtr->npixels, 0);
		if (pptr->mask) 
		    XFreePixmap(dpy,pptr->mask);

		XpmFreeAttributes(pptr->attribPtr);

		if (pptr == pixmapCacheRoot)
		    pixmapCacheRoot = pptr->next;
		else
		    lastPtr->next = pptr->next;
		XtFree((char *)pptr);
		}
	    break;
	    }
    
#ifdef PIXMAPCACHETRACE 
    showPixmapCache("after Free");
#endif
    DBUG_RETURN(toDestroy);
    }


static String
xpmErrorReason(xpmReturnCode)
int            xpmReturnCode;
    {
    char *reason;
    
    switch (xpmReturnCode) 
	{
    case XpmColorError: 
	reason = "Color Error"; break;
    case XpmOpenFailed: 
	reason = "Open Failed"; break;
    case XpmFileInvalid: 
	reason = "File Invalid"; break;
    case XpmNoMemory: 
	reason = "No Memory"; break;
    case XpmColorFailed: 
	reason = "Color Failed"; break;
    default:
	reason = "Unknown Reason";
	}
    return reason;
    }


static void
explainXpmError(immediateData, string, xpmReturnCode)
Boolean immediateData;
String string;
int xpmReturnCode;
    {
    if (immediateData) 
	wafeWarn("convertPixmap", 
		 "Couldn't convert data to pixmap (%s)", 
		 xpmErrorReason(xpmReturnCode), NULL, NULL);
    else 
	wafeWarn("convertPixmap", 
		 "Couldn't convert '%s' to pixmap (%s)", 
		 string, xpmErrorReason(xpmReturnCode), NULL);
    }



static int
xpmToPixmap(immediateData, dpy, d, string, pixmap, mask, attribPtr)
Boolean        immediateData;
Display       *dpy;
Window         d;
Pixmap        *pixmap;
Pixmap        *mask;
String         string;
XpmAttributes *attribPtr;
    {
    int    retcode;

    retcode = (immediateData ? XpmCreatePixmapFromBuffer : XpmReadFileToPixmap)
	(dpy, d, string, pixmap, mask, attribPtr);
    /* fprintf(stderr,"xpmToPixmap returns %d, pixmap=%ld, mask=%ld\n",
	    retcode,*pixmap,*mask);  */
    return retcode;
    }

#ifdef MOTIF
static int
xpmToImage(immediateData, dpy, d, string, image, mask, attribPtr)
Boolean        immediateData;
Display       *dpy;
Window         d;
XImage       **image;
XImage       **mask;
String         string;
XpmAttributes *attribPtr;
    {
    int    retcode;

    retcode = (immediateData ? XpmCreateImageFromBuffer : XpmReadFileToImage)
	(dpy, string, image, mask, attribPtr);
    /* fprintf(stderr,"xpmToImage returns %d, ximage=%p\n",
	    retcode,*image); */
    return retcode;
    }
#endif


static Boolean 
xpmImmediateData(fname,fn)
String  fname;
String *fn;
    {
    char *p = fname;
    
    while( *p == '\n' || *p == ' ' || *p == '\t') p++;
    if (strncmp(p,"/* XPM */",9) == 0)
	{
	*fn = p;
	return True;
	}
    else
	return False;
    }


static pixmapCachePtr
getThroughPixmapCache(immediateData,dpy,fname,qFileName,pixmapReturn)
Boolean       immediateData;
Display      *dpy;
String        fname;
XrmQuark      qFileName;
Pixmap       *pixmapReturn;
    {
    pixmapCachePtr  pptr;
    char           *fn;
    int             xpmReturnCode,
                    screenNum  = DefaultScreen(dpy);
    unsigned int    depth      = DefaultDepth(dpy, screenNum);
    XpmAttributes  *attribPtr; 
    Pixmap          pixmap,mask;

    DBUG_ENTER("getThroughPixmapCache");

    /* is the pixmap already in the cache? */
    for (pptr = pixmapCacheRoot; pptr != NULL; pptr = pptr->next)
	{
	if (qFileName == pptr->fileName)
	    {
	    DBUG_PRINT("xpm", ("Pixmap %s (%x) found in cache",
			       fname, pptr->pm));
	    pptr->count ++;
	    DBUG_RETURN(pptr);
	    }
	}

    /* If the new pixmap's filename is None etc, 
     * don't create new entry, ut use shortcut
     */
    if (!strcmp(fname, wafe_NONE))
	{
	*pixmapReturn = None;
	DBUG_RETURN((pixmapCachePtr)NULL);
	}
    else 
    if (!strcmp(fname, "Unspecified"))
	{
	*pixmapReturn = XtUnspecifiedPixmap;
	DBUG_RETURN((pixmapCachePtr)NULL);
	}
    else
    if (!strcmp(fname, "ParentRelative"))
	{
	*pixmapReturn = ParentRelative;
	DBUG_RETURN((pixmapCachePtr)NULL);
	}

    if (immediateData)
	fn = fname;
    else /* we have to look for the file */ if
	(!(fn = locateBitmap(dpy, fname, True)))
	{ 
	*pixmapReturn = XtUnspecifiedPixmap;
	DBUG_RETURN((pixmapCachePtr)NULL);
	}

    /* Now convert the new Pixmap-File, create a new element of the list 
     * and fill it with information.
     */
    attribPtr = (XpmAttributes *)XtCalloc(sizeof(XpmAttributes), sizeof(char));
    attribPtr->valuemask = XpmReturnPixels | XpmDepth | XpmReturnInfos;
    attribPtr->depth     = depth;
    /* fprintf(stderr, "changePixmap: calling xpmreadfile\n"); */

    xpmReturnCode = xpmToPixmap(immediateData, 
				dpy, RootWindow(dpy, screenNum), 
				fn, &pixmap, &mask, attribPtr);

    /* fprintf(stderr, "changePixmap: xpmreadfile returned %d\n",returnVal); */
    if (fn != fname) 
	XtFree(fn);

    if (xpmReturnCode == XpmSuccess)
	{
	pptr = (pixmapCachePtr)XtMalloc(sizeof(pixmapCache));
	pptr->next      = pixmapCacheRoot;
	pptr->pm        = pixmap;
	pptr->mask      = mask;
	pptr->count     = 1;
	pptr->fileName  = qFileName;
	pptr->attribPtr = attribPtr;
	pixmapCacheRoot = pptr;
	DBUG_PRINT("xpm", ("%d colorcells are used for new pixmap %s (%x)", 
			   pptr->attribPtr->npixels, fname, pixmap));
	DBUG_RETURN(pptr);
	}
    else
	{
	wafeSetError("Couldn't convert %s to pixmap (%s)",
		     immediateData ? "immediate data" : fname,
		     xpmErrorReason(xpmReturnCode), NULL);

	XpmFreeAttributes(attribPtr);
	*pixmapReturn = XtUnspecifiedPixmap;
	DBUG_RETURN((pixmapCachePtr)NULL);
	}
    }     


static void
freeXpmList(ptr)
char *ptr;
    {
    xpmListPtr p = (xpmListPtr)ptr, nextPtr;
    Pixmap     toDestroy;

    while (p) 
	{
	nextPtr = p->next;
	if ((toDestroy = freeInPixmapCache(p->dpy,p->cmap,p->fileName)))
	    XFreePixmap(p->dpy, toDestroy); 
	XtFree((char *)p);
	p = nextPtr;
	}
    }


int
wafeChangePixmap(w,resource,fname)
Widget     w;
String     resource;
String     fname;
    {
    Colormap         cmap;
    Pixmap           toDestroy = (Pixmap)NULL,
                     pixmapReturn;
    Display         *dpy       = XtDisplay(w);
    XrmQuark         qResName  = XrmStringToQuark(resource), 
                     qFileName;
    MMattribListPtr *al        = NULL;
    xpmListPtr       pixPtr    = 
	(xpmListPtr)wafeMMgetValue(w, qxpmList, &al, True);
    xpmListPtr       tpixPtr;
    pixmapCachePtr   pptr;    
    Boolean          immediateData = xpmImmediateData(fname,&fname);

    qFileName = XrmStringToQuark(fname);
    
    if (!wafeGetQTypeOfAttribute(XtClass(w),ParentWidget(w),qResName,NULL))
	return wafeSetError("widget %s has no resource %s",
			    wafeWidgetToName(w), resource, NULL);

    for (tpixPtr = pixPtr; tpixPtr != NULL; tpixPtr = tpixPtr->next)
	if (qResName == tpixPtr->resName) /* it has already a xpm */
	    break;
    
    XtVaGetValues(w, XtNcolormap, &cmap, NULL);

    if (tpixPtr)
	{
	/* we have already a xpm for this widget & resource */
	if (qFileName == tpixPtr->fileName)
	    {
	    DBUG_PRINT("xpm", ("You are about to set the same pixmap!"));
	    return TCL_OK;
	    }
	else
	    toDestroy = freeInPixmapCache(dpy,cmap,tpixPtr->fileName);
	}
    else
	{
	/* we have no xpm for this widget & resource */
	tpixPtr = (xpmListPtr) XtMalloc(sizeof(xpmList));
	tpixPtr->next     = pixPtr;
	tpixPtr->resName  = qResName;
	tpixPtr->dpy      = dpy;
	tpixPtr->cmap     = cmap;
	if (!pixPtr) 
	    wafeMMreplace(NULL, al, qxpmList, (char *)tpixPtr, freeXpmList);
	}
    tpixPtr->fileName = qFileName;    

#ifdef PIXMAPCACHETRACE 
    showPixmapCache("Before");
#endif
    /* now try to obtain the pixmap by using the pixmap cache */
    if ((pptr = getThroughPixmapCache(immediateData, 
				      dpy,fname,qFileName,&pixmapReturn))) 
	setPixmap(w, resource, qResName, pptr->pm, pptr->mask, 
		  pptr->attribPtr->width, pptr->attribPtr->height);
    else 
	setPixmap(w, resource, qResName, pixmapReturn, (Pixmap)NULL, 0, 0);

    if (toDestroy) 
	{
	DBUG_PRINT("xpm", ("old pixmap (%x) will be destroyed now", 
			   (int)toDestroy));
	XFreePixmap(dpy, toDestroy); 
	}

#ifdef PIXMAPCACHETRACE 
    showPixmapCache("After");
#endif
    if (wafeInterpreter->result && *(wafeInterpreter->result)) 
      return TCL_ERROR;
    else 
      return TCL_OK;
    }     

# ifdef MOTIF
static String 
getMaskName(immediateData,name)
Boolean immediateData;
String name;
    {
    char *dot = NULL;
    int oldLength = strlen(name);
    char *mask_name = XtMalloc(oldLength+3);
    
    if (!immediateData)
	dot = (char*)strrchr(name,'.');

    if (dot)
	{
	char *ext = XtNewString(dot);
	int dotOffset = dot-name;
	strncpy(mask_name,name,dotOffset);
	strcpy(mask_name+dotOffset,"_m");
	strcpy(mask_name+dotOffset+2,ext);
	XtFree(ext);
	}
    else
	{
	strncpy(mask_name,name,oldLength);
	strcpy(mask_name+oldLength,"_m");
	}
    /* fprintf(stderr,"mask_name = <%s>\n",mask_name);*/
    return mask_name;
    }
# endif /* MOTIF */
#endif /* XPM */


#ifdef MOTIF
Boolean
wafeInstallImage(dpy,string,name,quiet)
Display *dpy;
String string;
String name;
Boolean quiet;
    {
    XImage *image = NULL;
    XImage *mask  = NULL;
    int screenNum  = DefaultScreen(dpy);
    Boolean  installed = False;
    Boolean  result;
#ifdef XPM
    int xpmReturnCode;
    Boolean immediateData = xpmImmediateData(string,&string);
    XpmAttributes  attrib;
#else
    Boolean immediateData = False;
#endif
    String fn;

    /* if no name is given, use the file name or immediate data */
    if (!name)
	name = string;

    installed = (Tcl_GetVar2(wafeInterpreter,"_imageCache",name,
			     TCL_GLOBAL_ONLY) != NULL);

    /* fprintf(stderr, "installed=%d <%s>\n",installed,name);*/

    if (installed)
	return False;       /* the image is already installed */
    
    fn = string;
    if (!immediateData)
	{
	/* we have to look for the file */
	if (!(fn = locateBitmap(dpy, string, True)))
	    return False;   /* the image was not found */
	}

    if (!immediateData)   
	/* we have currently no program to process xbm immediate */
	{
	unsigned int     width, height;
	int              xhot, yhot;
	unsigned char   *data;

	if (XmuReadBitmapDataFromFile(fn, &width, &height, &data,
				      &xhot, &yhot) == BitmapSuccess)
	    {
	    image = (XImage *)XtMalloc(sizeof(XImage));
	    image->height           = height;
	    image->width            = width;
	    image->depth            = 1;
	    image->bits_per_pixel   = 1;
	    image->xoffset          = 0;
	    image->format           = XYBitmap;
	    image->data             = data;
	    image->byte_order       = LSBFirst;
	    image->bitmap_unit      = 8;
	    image->bitmap_bit_order = LSBFirst;
	    image->bitmap_pad       = 8;
	    image->bytes_per_line   = (width+7)/8;
	    }
	}
    
# ifdef XPM    
    if (!image)
	{
	attrib.valuemask = XpmDepth | XpmReturnInfos;
	attrib.depth = DefaultDepthOfScreen(DefaultScreenOfDisplay(dpy));

	xpmReturnCode = xpmToImage(immediateData, 
				   dpy, RootWindow(dpy, screenNum), 
				   fn, &image, &mask, &attrib); 
	if (!image && !quiet)
	    explainXpmError(immediateData,fn,xpmReturnCode);
	}
# endif

    if (fn != string) 
	XtFree(fn);
    
    if (image) 
	{
	if (!(result = XmInstallImage(image, name)))
	    {
	    /* fprintf(stderr, "   destroying image\n");*/
	    XDestroyImage(image);
	    }
	else 
	    {
	    Tcl_SetVar2(wafeInterpreter,"_imageCache",name,
			wafe_ONE, TCL_GLOBAL_ONLY);
	    if (mask)
		{
# ifdef XPM
		char *mask_name = getMaskName(immediateData,name);

		if (XmInstallImage(mask, mask_name))
		    Tcl_SetVar2(wafeInterpreter,"_imageCache",mask_name,
				wafe_ONE, TCL_GLOBAL_ONLY);		
		else
		    XDestroyImage(mask);

		XtFree(mask_name);
# endif /* XPM */
		}
    	    }
	}
    else
	result = False;
    
    return result;
    }
#endif /* MOTIF */



#ifdef MOTIF
static Pixmap
getPixmapViaXmImageCache(dpy,name,depth)
Display *dpy;
char   *name;
int     depth;
    {
    Pixmap  pixmap;
    Pixel   foreground;
    Pixel   background;
    Screen *screen    = DefaultScreenOfDisplay(dpy);

# ifdef XPM
    (void) xpmImmediateData(name,&name);
# endif

    if (wafeCurrentWidget)
	{
	/*
	   fprintf(stderr,
	   "getting foreground and background from currentwidget %s\n",
	   XtName(wafeCurrentWidget));
	   */
	XtVaGetValues(wafeCurrentWidget,
		      XtNforeground,&foreground,
		      XtNbackground,&background,
		      NULL);
	}
    else
	{
/*
#ifdef MOTIF20
	return XmDELAYED_PIXMAP;
#endif	
*/
	foreground = XBlackPixelOfScreen(screen);
	background = XWhitePixelOfScreen(screen);
	}

    /* fprintf(stderr,"foreground =%ld, background =%ld\n",
       foreground,background);
       */

    wafeInstallImage(dpy,name,NULL,True);
# ifdef MOTIF12
    pixmap = XmGetPixmapByDepth(screen, name, foreground, background, depth);
# else
    pixmap = XmGetPixmap(screen, name, foreground, background);
# endif
    if (pixmap != XmUNSPECIFIED_PIXMAP)
	{
	Tcl_SetVar2(wafeInterpreter,"_imageCache",name,
		    wafe_ONE, TCL_GLOBAL_ONLY);
	}

# ifdef NEVER
# ifdef XPM
	{
	char *mask_name = getMaskName(immediateData,name);
	Pixmap mask = XmGetPixmapByDepth(screen, mask_name, 
					 foreground, background, depth);
	
	XtFree(mask_name);
	}
# endif /* XPM */
# endif /* NEVER */
    return pixmap;
    }
#endif /* MOTIF */



#define	CONVERSION_DONE(type, value) \
	{							\
	    if (toVal->addr != NULL) {				\
		if (toVal->size < sizeof(type)) {		\
		    toVal->size = sizeof(type);			\
		    return False;				\
		}						\
		*(type*)(toVal->addr) = (value);		\
	    }							\
	    else {						\
		static type static_val;				\
		static_val = (value);				\
		toVal->addr = (XtPointer)&static_val;		\
	    }							\
	    toVal->size = sizeof(type);				\
	    return True;					\
	}

#ifdef MOTIF
static void
destroyPixrec(pr)
pixrec *pr;
    {          
    XmDestroyPixmap(pr->screen,pr->pixmap);
    XtFree((char*)pr);
    }
#endif

static Boolean
CvtStringToPixmapOrBitmap(dpy, args, num_args, fromVal, toVal,
			  converter_data)
Display *dpy;
XrmValuePtr args;
Cardinal *num_args;
XrmValuePtr fromVal;
XrmValuePtr toVal;
XtPointer *converter_data;
    {
    static Pixmap    pixmap;               /* static for cvt magic */
    String           fname;
    int              screenNum = DefaultScreen(dpy);
    int              depth     = DefaultDepth(dpy, screenNum);
    String           fn = NULL;
    Boolean          toPixmap = *num_args>1;
    Boolean          needPixmap = True;
    Screen          *screen     = DefaultScreenOfDisplay(dpy);
#ifndef MOTIF
    unsigned char   *data;
    unsigned int     width, height;
    int              xhot, yhot;
    int              returnValue;
    Boolean          immediateData;
#endif

    pixmap     = (Pixmap) NULL;
    fn = fname = (String) fromVal->addr;

    DBUG_PRINT("xpm",("Trying to convert '%s' to %s",
		      fname,toPixmap ? XtRPixmap : XtRBitmap));

    if (strcmp(fname, wafe_NONE) == 0) 
	{
        pixmap = None;
	needPixmap = False;
	}
    else
    if (strcmp(fname, "ParentRelative") == 0 && toPixmap) 
	{
        pixmap = ParentRelative;
	needPixmap = False;
	}
    else
    if (strcmp(fname, "Unspecified") == 0 && toPixmap) 
	{
        pixmap = XtUnspecifiedPixmap;
	needPixmap = False;
	}
#ifdef MOTIF
    if (strcmp(fname, "unspecified_pixmap") == 0) 
	{
	pixmap = XmUNSPECIFIED_PIXMAP;
	needPixmap = False;
	}
# ifdef MOTIF12
    else
    if ((pixmap = getPixmapViaXmImageCache(dpy,fname, depth)) 
	== XmUNSPECIFIED_PIXMAP)
	pixmap = None;
    else
	needPixmap = False;
# endif /* MOTIF12 */

    
    if (needPixmap) 
	{
	XtStringConversionWarning(fname, toPixmap ? XtRPixmap : XtRBitmap);
	pixmap = XmUNSPECIFIED_PIXMAP;
	}
    else 
	{
	pixrec *pr = (pixrec *)XtMalloc(sizeof(pixrec));
	pr->pixmap = pixmap;
	pr->screen = screen;
	wafeMMreplace(NULL, wafeCurrentAttribList, 
		      wafeCurrentAttrib, 
		      (char *)pr, destroyPixrec);
	}

    if (fname != fn) 
	XtFree(fn);
   
    CONVERSION_DONE(Pixmap,pixmap);

#else /* ------------------------ no MOTIF --------------------------- */

# ifdef XPM
    immediateData = xpmImmediateData(fname,&fname);
# else
    immediateData = False;  /* we have currently only a xpm-buffer to image */
# endif
    if (!immediateData && needPixmap)
	{
	/* we have to locate the image */
	if (!(fn = locateBitmap(dpy, fname, toPixmap)))
	    return False;
	}
    /* we have a valid filename fn */
    
    if (needPixmap) 	
	{
	/* try to read file as bitmap file... */
	if (XmuReadBitmapDataFromFile(fn, &width, &height, &data,
				      &xhot, &yhot) == BitmapSuccess)
	    {
	    pixmap = XCreatePixmapFromBitmapData(dpy,
			RootWindowOfScreen(screen),
			(char *) data, width, height, 1, 0, 
			depth);
	    XFree((char *)data);
	    needPixmap = False;
	    }
	}

#ifdef XPM
    if (needPixmap) 
	{
	Pixmap           mask;
	Window           coreWin;
	XpmAttributes    attrib;

	attrib.valuemask = XpmReturnPixels | XpmDepth | XpmReturnInfos;
	attrib.depth = DefaultDepthOfScreen(screen);
	coreWin = *((Window *) args[0].addr);

	if ((returnValue = xpmToPixmap(immediateData,
				      dpy, coreWin, fn, &pixmap, &mask, 
				      &attrib) ) != XpmSuccess)
	    {
	    explainXpmError(immediateData,fn,returnValue);
	    XtStringConversionWarning(fname, toPixmap ? XtRPixmap : XtRBitmap);
	    }
	else 
	    {
	    /*
	       fprintf(stderr,"pixmap=%p, width=%d, height=%d, addr=%p\n",
	       pixmap, attrib.width,attrib.height,toVal->addr);
	     */
	    needPixmap = False;
	    if (wafeCurrentAttrib != qiconPixmap)
		{
		/* if toVal->addr is set we assume that the value comes
		   from a resource file, in which case we can't assume that
		   the core fields of the widget are already initialized; 
		   coreWin will be wrong; Wafe never sets toVal->addr */
		
		if (coreWin && !toVal->addr) 
		    {
		    XShapeCombineMask(dpy, coreWin, ShapeBounding,
				      0, 0, mask, ShapeSet);
		    /* fprintf(stderr,"converter set mask\n");*/
		    }
		}
	    else if (wafeCurrentWidget)
		XtVaSetValues(wafeCurrentWidget,XtNiconMask,mask,NULL);
	    }
	}
#endif /* XPM */

    if (fname != fn) 
	XtFree(fn);

    if (needPixmap) 
	{
	XtStringConversionWarning(fname, toPixmap ? XtRPixmap : XtRBitmap);
	return False;    
	}
    else 
	{
	CONVERSION_DONE(Pixmap,pixmap);
	}
#endif /* no MOTIF */
    }


/* to keep knowledge about xpmConvertArgsPixmap etc local ...
 */
void
wafeRegisterXpmTypeConverter(resourceType, toPixmap)
_Xconst String resourceType;
Boolean toPixmap;
    { 
    if (toPixmap)
	XtSetTypeConverter(XtRString, 
			   resourceType, 
			   (XtTypeConverter)CvtStringToPixmapOrBitmap, 
			   xpmConvertArgsPixmap, 
			   2,
			   XtCacheByDisplay,
			   NULL);
    else
	XtSetTypeConverter(XtRString, 
			   resourceType, 
			   (XtTypeConverter)CvtStringToPixmapOrBitmap, 
			   xpmConvertArgsBitmap, 
			   1, 
			   XtCacheByDisplay, 
			   NULL);
    }

