/*
 * Copyright (C) 1992-94 by Gustaf Neumann, Stefan Nusser
 *
 *      Wirtschaftsuniversitaet Wien,
 *      Abteilung fuer Wirtschaftsinformatik
 *      Augasse 2-6,
 *      A-1090 Vienna, Austria
 *      neumann@@wu-wien.ac.at, nusser@@wu-wien.ac.at
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appears in all copies and that both that
 * copyright notice and this permission notice appear in all supporting
 * documentation.  This software is provided "as is" without expressed or
 * implied warranty.
 *
 * Date: Dec 24 1994
 * Author: Gustaf Neumann
 * Version: 1.0.8
 */

#define WAFESTRING_C

#include "wafe.h"

#undef WSDEBUG

#ifdef WSDEBUG
# define WSPRINT(fmtargs) printf fmtargs
# define WSREPORT(type) \
    fprintf(stderr,"%s: length=%d, max=%d %p <%s>\n", \
	    type,ws->length,ws->max,ws->buffer,ws->buffer)
#else
# define WSPRINT(fmtargs)
# define WSREPORT(type)
#endif


typedef enum { mustQuote, mustEscape, canCopy } quoting;
typedef enum { n, s, e, d } tclCharCode;

   
void
wafeStringInit(ws)
wafeString ws;
    {
    ws->length = 0;
    ws->buffer = ws->sBuffer;
    ws->max    = WAFESTRING_MIN;
    *ws->buffer = '\0';  /* when the buffer is used immediatedly after init */
    }

void
wafeStringClear(ws)
wafeString ws;
    {
    if (ws->buffer != ws->sBuffer) 
	{
	DBUG_PRINT("wafeString",("free size = %d",ws->max));
	WSPRINT(("freeing buffer in ws=%p\n",ws));
	XtFree(ws->buffer);
	}

    ws->length = 0;
    ws->buffer = ws->sBuffer;
    ws->max    = WAFESTRING_MIN;
    }

wafeString
wafeStringNew()
    {
    wafeString ws = (wafeString)XtMalloc(sizeof(wafeStringStruct));
    wafeStringInit(ws);
    return ws;
    }

void
wafeStringFree(ws)
wafeString ws;
    {
    wafeStringClear(ws);
    XtFree((char *)ws);
    }

/* wafeStringToString returns an XtMalloced buffer; don't call 
   wafeStringClear on ws! */
static String
wafeStringToString(ws)
wafeString ws;
    {
    return (ws->buffer == ws->sBuffer) ?
	XtNewString(ws->buffer) : ws->buffer;
    }


static String
wafeStringCheckAlloc(ws, size)
wafeString ws;
int size;
    {
    int desiredSize = size + ws->length;
    if (ws->max <= desiredSize)  /* we have to grow */
	{
	if ( desiredSize > (ws->max + WAFESTRING_SLACK) )
	    ws->max = desiredSize += WAFESTRING_SLACK;
	else
	    ws->max += WAFESTRING_SLACK;

	if (ws->buffer == ws->sBuffer) 
	    {
	    DBUG_PRINT("wafeString",("malloc, new size = %d",ws->max));
	    WSPRINT(("First malloc in ws=%p, size=%d\n",ws,ws->max));
	    ws->buffer = XtMalloc(ws->max);
	    memcpy(ws->buffer,ws->sBuffer,ws->length);
	    }
	else
	    {
	    WSPRINT(("realloc in ws=%p\n",ws));
	    DBUG_PRINT("wafeString",("realloc, new size = %d",ws->max));
	    ws->buffer = XtRealloc(ws->buffer,ws->max);
	    }
	}
    return ws->buffer + ws->length;
    }

void
wafeStringAppend(ws, string)
wafeString ws;
String string;
    {
    if (string) 
	{
	int stringLength = strlen(string);
	String next      = wafeStringCheckAlloc(ws,stringLength);
	memcpy(next, string, stringLength);
	ws->length += stringLength;
	next[stringLength] = 0;
	}
    WSREPORT("stringAppend");
    }

void
wafeStringAppendN(ws, string, stringLength)
wafeString ws;
String string;
int stringLength;
    {
    String next = wafeStringCheckAlloc(ws,stringLength+1);
    strncpy(next,string,stringLength);
    ws->length += stringLength;
    next[stringLength] = 0;
    WSREPORT("stringAppendN");
    }

static _Xconst tclCharCode charClass[] = {
       /* 0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f */
/* 00 */  n, n, n, n, n, n, n, n, n, n, s, n, n, n, n, n, 
/* 10 */  n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, 
/* 20 */  s, n, d, n, e, n, n, n, n, n, n, n, n, n, n, n, 
/* 30 */  n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, 
/* 40 */  n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, 
/* 50 */  n, n, n, n, n, n, n, n, n, n, n, e, d, n, n, n, 
/* 60 */  n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, 
/* 70 */  n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, 
/* 80 */  n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, 
/* 90 */  n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, 
/* a0 */  n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, 
/* b0 */  n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, 
/* c0 */  n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, 
/* d0 */  n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, 
/* e0 */  n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, 
/* f0 */  n, n, n, n, n, n, n, n, n, n, n, n, n, n, n, n

/*   "=0x22, {=0x7b, }=0x7d, $=0x24 [=0x5b \=0x5c  " "=20 \n=0a*/ 
/* n == none
:  s == space (need quotes in list)
   e == escape (need backslash)
   d == double escape (need 2 backslashes in lists)
 */
};

void
wafeStringAppendEscaped(ws, string, count)
wafeString ws;
String string;
int count;
    {
    char *p = string;
    char *next;
    int  freeBytes, lastFree;

    if (!string || count==0)
      return;

    /* the first optimistic guess is that nothing must be escaped;
       we know that check alloc preallocates space to prevent excessive
       reallocs.
     */
    if (count>0) 
	next = wafeStringCheckAlloc(ws,count);
    else
	next = wafeStringCheckAlloc(ws,strlen(string));
    
    lastFree = freeBytes = ws->max - ws->length;
    if (p) 
	{
	while (*p && count) 
	    {
	    if (freeBytes < 3) 
		{
		ws->length += lastFree - freeBytes;
		next = wafeStringCheckAlloc(ws,10);
		lastFree = freeBytes = ws->max - ws->length;
		}
	    if (charClass[*p & 255] > s)
		{
		*next++ = '\\';
		freeBytes--;
		}
	    *next++ = *p++;
	    freeBytes--;
	    if (count>0) count--;
	    }
	}
    *next = '\0';
    ws->length += lastFree - freeBytes;
    WSREPORT("eString");
    }

static void
wafeStringAppendEscapedInList(ws, string, count)
wafeString ws;
String string;
int count;
    {
    char *p = string;
    char *next;
    int  freeBytes, lastFree;

    if (!string || count==0)
      return;

    /* the first optimistic guess is that nothing must be escaped;
       we know that check alloc preallocates space to prevent excessive
       reallocs.
     */
    if (count>0) 
	next = wafeStringCheckAlloc(ws,count);
    else
	next = wafeStringCheckAlloc(ws,strlen(string));
    
    lastFree = freeBytes = ws->max - ws->length;
    if (p) 
	{
	while (*p && count) 
	    {
	    if (freeBytes < 5) 
		{
		ws->length += lastFree - freeBytes;
		next = wafeStringCheckAlloc(ws,10);
		lastFree = freeBytes = ws->max - ws->length;
		}
	    switch (charClass[*p & 255]) 
		{
	    case d:
		*next++ = '\\';
		*next++ = '\\';
		freeBytes -= 2;
	    case e:
		*next++ = '\\';
		freeBytes--;
	    case n:
	    case s:
		*next++ = *p++;
		freeBytes--;
		if (count>0) count--;
		}
	    }
	}
    *next = '\0';
    ws->length += lastFree - freeBytes;
    WSREPORT("eStringInList");
    }


void
wafeStringAppendLong(ws, l)
wafeString ws;
long l;
    {
    char *next = wafeStringCheckAlloc(ws,LONG_AS_STRING);
    sprintf(next, "%ld", l);
    ws->length += strlen(next);
    WSREPORT("long");
    }

void
wafeStringAppendInt(ws, i)
wafeString ws;
int i;
    {
    char *next = wafeStringCheckAlloc(ws,INT_AS_STRING);
    sprintf(next, "%d", i);
    ws->length += strlen(next);
    WSREPORT("int");
    }

void
wafeStringAppendFloat(ws, f)
wafeString ws;
double f;
    {
    char *next = wafeStringCheckAlloc(ws,FLOAT_AS_STRING);
    sprintf(next, "%f", f);
    ws->length += strlen(next);
    WSREPORT("float");
    }

void
wafeStringAppendShort(ws, snum)
wafeString ws;
short snum;
    {
    char *next = wafeStringCheckAlloc(ws,LONG_AS_STRING);
    sprintf(next, "%hd", snum);
    ws->length += strlen(next);
    WSREPORT("short");
    }

void
wafeStringAppendPointer(ws, p)
wafeString ws;
XtPointer p;
    {
    char *next = wafeStringCheckAlloc(ws,LONG_AS_STRING);
    sprintf(next, "%p", p);
    ws->length += strlen(next);
    WSREPORT("pointer");
    }


void
wafeStringAppendChar(ws, c)
wafeString ws;
char c;
    {
    char *next = wafeStringCheckAlloc(ws,2);
    *next++ = c;
    *next   = '\0';
    ws->length ++;
    WSREPORT("char");
    }
    

void 
wafeStringAppendListItemEscaped(ws,string)
wafeString ws;
String string;
    {
    int stringLength = string ? strlen(string) : 0;

    if (stringLength == 0) 
	wafeStringAppend(ws,"{} ");
    else
	{
	quoting quote;

	if (*string  == '"' || *string == '{')
	    quote = mustQuote;
	else
	    {
	    char *ptr;
	    quote = canCopy;
	    for ( ptr = string ; *ptr; ptr++) 
		switch (charClass[*ptr & 255])
		    {
		case s:
		    ptr = wafe_NULL; /* stop loop */
		    quote = mustQuote;
		    break;
	        case e:
		case d:
		    quote = mustEscape;
		case n:
		    break;
		    }
	    }

	switch (quote) 
	    {
	case mustQuote:
	    wafeStringAppend(ws,"\\\"");
	    wafeStringAppendEscapedInList(ws,string,stringLength);
	    wafeStringAppend(ws,"\\\" ");
	    break;
	case mustEscape:
	    wafeStringAppendEscaped(ws, string, stringLength);
	    wafeStringAppendChar(ws,' ');
	    break;
	case canCopy:
	    wafeStringAppendN(ws, string, stringLength);
	    wafeStringAppendChar(ws,' ');
	    }
	}
    }
    


void
wafeStringAppendListEscaped(ws, num, stringArray)
wafeString ws;
Cardinal   num;
String    *stringArray;
    {
    if (num>0) 
	{
	Cardinal i;
	for (i=0; i<num; i++)
	    wafeStringAppendListItemEscaped(ws,stringArray[i]);

	ws->length--;
	ws->buffer[ws->length] = 0;
	}
    }

void
wafeStringAppendWidgetList(ws, num, widgetArray)
wafeString ws;
Cardinal   num;
WidgetList widgetArray;
    {
    if (num>0) 
	{
	Cardinal i;
	char     *output,*buffer;

	buffer = output = wafeStringCheckAlloc(ws,num * (LONG_AS_STRING+1));
	for (i=0; i<num; i++)
	    {
	    sprintf(output, "%ld ", (long)widgetArray[i]);
	    output += strlen(output);
	    }
	ws->length += output-buffer-1;
	ws->buffer[ws->length] = 0;
	}
    }

void
wafeStringAppendIntList(ws, num, intArray)
wafeString ws;
Cardinal   num;
int       *intArray;
    {
    if (num>0) 
	{
	Cardinal i;
	char     *output,*buffer;

	buffer = output = wafeStringCheckAlloc(ws,num * (INT_AS_STRING+1));
	for (i=0; i<num; i++)
	    {
	    sprintf(output, "%d ", intArray[i]);
	    output += strlen(output);
	    }
	ws->length += output-buffer-1;
	ws->buffer[ws->length] = 0;
	}
    }

String
wafeCvtIntsToList(num, intArray)
Cardinal    num;
int        *intArray;
    {
    wafeStringStruct ws_s;
    wafeString ws = &ws_s;

    wafeStringInit(ws);
    wafeStringAppendIntList(ws, num, intArray);
    return wafeStringToString(ws);
    }

String
wafeCvtWidgetListToList(num, widgetArray)
Cardinal   num;
WidgetList widgetArray;
    {
    wafeStringStruct ws_s;
    wafeString ws = &ws_s;

    wafeStringInit(ws);
    wafeStringAppendWidgetList(ws, num, widgetArray);
    return wafeStringToString(ws);
    }
