/*
 * Copyright (C) 1992 by Gustaf Neumann, Stefan Nusser
 *
 *      Wirtschaftsuniversitaet Wien,
 *      Abteilung fuer Wirtschaftsinformatik
 *      Augasse 2-6,
 *      A-1090 Vienna, Austria
 *      neumann@wu-wien.ac.at, nusser@wu-wien.ac.at
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appears in all copies and that both that
 * copyright notice and this permission notice appear in all supporting
 * documentation.  This software is provided "as is" without expressed or
 * implied warranty.
 *
 * Date: Mon, Apr 13 1992
 * Author: Stefan Nusser
 * Version: 0.91
 */

/*
 *                 WAFE.C 
 *
 *                 This file contains the main function as well as some
 *                 utility procedures, which are used in several places:
 *
 *                 getQTypeOfAttribute
 *                 convert
 *                 name2Widget
 *                 convert
 *                 killCallbackProc
 *                 signalHandler
 *                 main
 */


#define MAIN

#include <stdio.h>
#include <sys/types.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#ifdef ATHENA
#include <X11/Xaw/AsciiText.h>
#endif

#include <unistd.h>
#include <limits.h>
#include <signal.h>

#ifndef STREAMS
#include <sys/socket.h>
#else
#include <sys/stream.h>
#include <stropts.h>
#include <fcntl.h>
#endif


#include "wafe.h"                             
#include "version.h"


/*
 * The following functions simplify error handling - they all are
 * invoked under certain circumstances and print an error message
 *
 */


void 
convError (tclCommand, argc, tclArg, type)
char * tclCommand;
char * argc;
char * tclArg;
char * type;
    {
    fprintf(stderr,"Wafe(%s): could not convert argument %s <%s> to type %s\n",
	    tclCommand, argc, tclArg, type);
    }

void
argcError (tclCommand, quant, tclArgs, argc)
char *tclCommand;
char *quant;
int   tclArgs;
int   argc;
    {
    fprintf(stderr,"Wafe(%s): %s%d arguments expected, %d arguments received\n",
            tclCommand, quant, tclArgs, argc - 1);
    }

void
noVarCompError (tclCommand, name, comp)
char *tclCommand;
char *name;
char *comp;
    {
    fprintf(stderr,"Wafe(%s): no component %s of structure %s\n",
            tclCommand, comp, name);
     }



/***********************************************************************************
  *  The following functions maintain the global lists of ResourceLists, one for 
  *  normal widgets and one for constraint resources. They look up the
  *  resources registered for a specified widget class and - if they are queried for
  *  the first time - append the new resourceList.
  *  In each appended resourceList, the string name of the ressource will be substituted
  *  by a quark representation.
  ***********************************************************************************/

QTypeList
resList2QTypeList(rList, rNum, addResPtr, addNum)
XtResourceList  rList;
int   rNum;
addResInfo *addResPtr;
Cardinal   addNum;
    {
    int i;
    QTypeList  returnList, ptr;

    returnList = ptr = 
	(QTypeList) XtMalloc((rNum + addNum) * sizeof(QType) + 1);

    for (i = 0; i < rNum; i++)
	{
	ptr->qName = XrmStringToQuark(rList[i].resource_name);
	ptr->type = XtNewString(rList[i].resource_type);
	ptr->qType = XrmStringToQuark(ptr->type);
	ptr++;
	}

    for (i = 0; i < addNum; i++)
	{
	ptr->qName = XrmStringToQuark(addResPtr[i].resource_name);
	ptr->type = addResPtr[i].resource_type;
	ptr->qType = XrmStringToQuark(ptr->type);
	ptr++;
	}
    XtFree((char *)rList);
    return(returnList);
    }


Cardinal
getResOfClass(wClass, qList)
WidgetClass wClass;
QTypeList  *qList;
    {
    Cardinal  num;
    RESINFOPTR  ptr, lptr;
    XtResourceList resListPtr;
    addResInfo *addResPtr;
    Cardinal    addNum;
    Boolean     disposeAddResPtr = False;
    

    if (resourceList)  /* there are allready some entries in the resource list */
	{
	for (lptr = resourceList; 
	     lptr != NULL; 
	     ptr = lptr, lptr = lptr->next) 
	    if (lptr->wClass == wClass)
		{
		*qList = lptr->resources;
		return(lptr->num);
		}
	ptr->next = (RESINFOPTR) XtMalloc(sizeof(RESINFO));
	ptr = ptr->next;
	}
    else
	ptr = resourceList = (RESINFOPTR) XtMalloc(sizeof(RESINFO));
        
    ptr->next = NULL;
    ptr->wClass = wClass;

    XtInitializeWidgetClass(wClass);  /* Just in case it is not yet done... */
    XtGetResourceList(wClass, &resListPtr, &num);



#ifdef ATHENA
    /* fetch ATHENA AsciiText's subpart resources here! */

    if (wClass == asciiTextWidgetClass)      /* check asciiText's subresources  */
	{
	XtResourceList rList[2], rl;
	Cardinal       rNum[2], *num;
	addResInfo    *ptr;
	int           i,j;
	
	XtInitializeWidgetClass(asciiSinkObjectClass);
	XtInitializeWidgetClass(asciiSrcObjectClass);

	XtGetResourceList(asciiSinkObjectClass, &rList[0], &rNum[0]);
	XtGetResourceList(asciiSrcObjectClass, &rList[1], &rNum[1]);

	addNum = rNum[0] + rNum[1];
	addResPtr = (addResInfo *)XtMalloc( (addNum+1)*sizeof(addResInfo));

	for(i=0, ptr=addResPtr; i<2; i++) 
	    for(j=0, rl=rList[i], j=rNum[i]; j>0; j--, rl++, ptr++)
		{
		ptr->resource_name = XtNewString(rl->resource_name);
		ptr->resource_type = XtNewString(rl->resource_type);
		}
        /* *addResPtr  would be wasted, since the values are copied */
	disposeAddResPtr = True;
	}
    else
	{
	addResPtr = Nil(addResInfo);
	addNum = 0;
	}

#else
    addResPtr = getAddRes(wClass, &addNum);
#endif
    *qList = 
	ptr->resources = 
	resList2QTypeList(resListPtr, num, addResPtr, addNum);

    if (disposeAddResPtr) 
	XtFree((char *) addResPtr);

    ptr->num = num + addNum;
    return(ptr->num);
    } 

Cardinal
getConOfClass(wClass, qList)
WidgetClass wClass;
QTypeList  *qList;
    {
    Cardinal  num;
    RESINFOPTR  ptr, lptr;
    XtResourceList conListPtr;

    if (constraintList)  /* there are allready some entries in the resource list */
	{
	for (lptr = constraintList; lptr != NULL; ptr = lptr, lptr = lptr->next) 
	    if (lptr->wClass == wClass)
		{
		*qList = lptr->resources;
		return(lptr->num);
		}
	ptr->next = (RESINFOPTR) XtMalloc(sizeof(RESINFO));
	ptr = ptr->next;
	}
    else
	ptr = constraintList = (RESINFOPTR) XtMalloc(sizeof(RESINFO));
        
    ptr->next = NULL;
    ptr->wClass = wClass;

    XtGetConstraintResourceList(wClass, &conListPtr, &num);

    *qList = ptr->resources = resList2QTypeList(conListPtr, num, NULL,0);
    ptr->num = num;
    return(num);
    } 



/*****************************************************************************
 *    FUNCTION:  getQTypeOfAttribute 
 *                                  
 *    Arguments: Widgetclass, parent widget and Quark of attribute to be queried.
 *    Returns:   Type of the attribute as a Quark or NULL.                     
 *                                                                      
 *    Used by    getValues, convert 
 *
 * Vielleicht wre es besser, eine Struktur mit einem String und einem Quark
 * zu retournieren, um ein etwaiges Zurckumwandeln zu ersparen. 
 * Dringlichkeitsstufe: 0
 *****************************************************************************/

XrmQuark
getQTypeOfAttribute(wClass, parent, qAttribute)
WidgetClass     wClass;
Widget          parent;
XrmQuark        qAttribute;
    {
    QTypeList    resList, conList;
    int          i, count;

 /* Check standard resources of this class */
    count = getResOfClass(wClass, &resList);  

/*
    for (i = 0; i < count; i++)
	if (qAttribute == resList[i].qName)
	    return(resList[i].qType);
 */        

    for (i = 0; i < count; i++, resList++)
	if (qAttribute == resList->qName)
	    return(resList->qType);
          
    if (XtIsConstraint(parent))         /* check Constraints if there are any... */
	{
	count = getConOfClass(XtClass(parent), &resList);

	for (i = 0; i < count; i++, resList++)
	    if (qAttribute == resList->qName)
		return(resList->qType);
	}

    return((XrmQuark)NULL); 
    }

void
addToEndOfWidgetList(wl,w,info)
WidgetListPtr *wl;
Widget         w;
XtPointer      info;
    {
    WidgetListPtr *ptr=wl;
    while (*ptr)
	ptr = &((*ptr)->next);
    *ptr = (WidgetListPtr)XtMalloc(sizeof(WidgetListItem));
    (*ptr)->widget = w;
    (*ptr)->next = NULL;
    (*ptr)->info = info;
/*
    fprintf(stderr,"add to end of widgetlist *ptr %p, widget= %p, tree=%p info <%s>\n",
	    *ptr,(*ptr)->widget,widgetTrees,(char*)((*ptr)->info));
 */
    }

/*
 * when a topLevel widget (typically on a different display) is 
 * deleted, it has to be removed from widgetTrees, which is 
 * used primarily by name2widget
 */
void
deleteFromWidgetList(wl,w)
WidgetListPtr *wl;
Widget         w;
    {
    WidgetListPtr *ptr=wl;
    WidgetListPtr tmpPtr;
    while (*ptr && (*ptr)->widget != w)
	ptr = &((*ptr)->next);
    if (!*ptr) 
	return;

    XtFree((*ptr)->info);  /* let us assume, info is always mallocated */
    tmpPtr = (*ptr)->next;     
    XtFree((char *)(*ptr));
    *ptr = tmpPtr;
    }






MMattribListPtr *
MMgetAttribList(w)
Widget w; 
    {
    WidgetListPtr   *ptr = &MMwidgetListHead;
    WidgetListPtr   ptrTmp;

    while (*ptr) 
	{
	if ((*ptr)->widget == w) 
	    {
/*
	    fprintf(stderr,"MMgetAttribList: found al %p for widget %p\n",
		    (*ptr)->info,w);
 */
	    return ((MMattribListPtr*)&((*ptr)->info));
	    }
	ptr = &(*ptr)->next;
	}

     /* not found; add new widgets to the front fo the list */
     
    ptrTmp = MMwidgetListHead;
    MMwidgetListHead = (WidgetListPtr)XtMalloc(sizeof(WidgetListItem));
/*
    fprintf(stderr,"MMgetAttribList: new widgetlist at %p\n",MMwidgetListHead);
 */
    MMwidgetListHead->next = ptrTmp;
    MMwidgetListHead->widget = w;
    MMwidgetListHead->info = NULL;
    return((MMattribListPtr*)&(MMwidgetListHead->info));
    }


MMsetAttribList(w,al)
Widget w; 
MMattribListPtr al;
    {
    WidgetListPtr   *ptr, p, ptrTmp;
/*
    ptr = &MMwidgetListHead;
    if (al == NULL) 
	while ((*ptr)->info && (*ptr)->next)
	    ptr = &((*ptr)->next);
*/
    ptrTmp = MMwidgetListHead;
    MMwidgetListHead = (WidgetListPtr)XtMalloc(sizeof(WidgetListItem));
    MMwidgetListHead->next = ptrTmp;
    MMwidgetListHead->widget = w;
    MMwidgetListHead->info = (XtPointer)al;
/*
    fprintf(stderr,"MMset: setting al %p for widget %p\n",al,w);
*/
    }

#ifdef WIDGETLISTTRACE
void
printAttributeList(al)
MMattribListPtr al;
    {
    
    while(al)
	{
	fprintf(stderr,"     al = %p\n",al);
	fprintf(stderr,"          attribute = %s\n",
		XrmQuarkToString(al->attribute));
	fprintf(stderr,"          value     = %p <%s>\n",
		al->value, al->value);
	fprintf(stderr,"          next      = %p\n",
		al->next);

	al = al->next;
	}
    }

void
printWidgetList(wl)
WidgetListPtr   wl;
    {
   
    while(wl)
	{
	fprintf(stderr,"  wl = %p\n",wl);
	fprintf(stderr,"       widget = %p %s\n",
		wl->widget, XtName(wl->widget));
	fprintf(stderr,"       info   = %p \n",
		wl->info);
	fprintf(stderr,"       next   = %p \n",
		wl->next);
        printAttributeList(wl->info);
	wl = wl->next;
	fprintf(stderr,"\n");
	}
    }
#endif


void
MMreplace(al,descriptor,value,freeFunc)
MMattribListPtr *al;
XrmQuark         descriptor;
char            *value;
freeProc         freeFunc;
    {
    MMattribListPtr   *ptr = al;
    while(*ptr) 
	{
	if ((*ptr)->attribute == descriptor) 
	    {
	    (*freeFunc)((*ptr)->value);
/*
	    fprintf(stderr,"MMreplace: freeing value %p for <%s> setting %p\n",
		    (*ptr)->value,XrmQuarkToString((*ptr)->attribute),value);
 */
	    (*ptr)->value = value;
	    return;
	    }
	ptr = &((*ptr)->next);
	}

    *ptr = (MMattribListPtr)XtMalloc(sizeof(MMattribList));
    (*ptr)->next = NULL;                    
    (*ptr)->attribute = descriptor;
/*
    fprintf(stderr,"MMreplace: new attriblistelement %p for %s in al %p, value=%p\n",
	    *ptr,XrmQuarkToString((*ptr)->attribute),al,value);
 */
    (*ptr)->value = value;
    }
     

/*
 * MMfreeGarbage(widget)
 * frees grabage frames and data 
 * when a widget is deleted
 * gustaf Sat Jul 11 15:09:12 MET DST 1992
 */
Widget
MMfreeGarbage(w)
Widget      w;
    {
    WidgetListPtr    *ptr = &MMwidgetListHead;
    WidgetListPtr     ptrTmp;
    MMattribListPtr   al,alTmp;
    int               count = 0;

    if (XtIsComposite(w)) 
	{
	int numChildren;
	Widget *children;

	XtVaGetValues(w, XtNnumChildren, &numChildren,
		      XtNchildren,    &children,    NULL);
 
	while(numChildren) 
	    {
/*
	    fprintf(stderr,"    clearing for %p <%s> child of %p <%s>\n",
		    *children,XtName(*children),w,XtName(w));
 */
	    MMfreeGarbage(*children);
	    numChildren--;
	    children++;        
	    }

	if (XtIsApplicationShell(w))
	    deleteFromWidgetList(&widgetTrees,w);
	}

/*
    fprintf(stderr,"----------after recursion %p %p\n",MMwidgetListHead,*ptr);
    printWidgetList(*ptr);
 */

    while (*ptr) 
	{
/*
       fprintf(stderr,"garbage: widget to delete %p <%s>, actual widget %p <%s>\n",
	       (*ptr)->widget,XtName((*ptr)->widget),XtName(w));
 */
	if ((*ptr)->widget == w) 
	    {
	    al = (MMattribListPtr)(*ptr)->info;

	    while(al) 
		{
/*
		fprintf(stderr,"garbage: freeing value %p\n",
			al->value, al->attribute);
		fprintf(stderr,"garbage: freeing attribute list %p\n",
			al);
 */
		(getFreeFunc(XrmQuarkToString(al->attribute)))(al->value);
		alTmp = al->next;
		XtFree((char*)al);
		al = alTmp;
		}

/*	 fprintf(stderr,"garbage: freeing widgetlist %p\n", *ptr);*/
	    ptrTmp = (*ptr)->next; 
	    XtFree((char*)*ptr);
	    *ptr = ptrTmp;
	    break;
	    }
	ptr = &(*ptr)->next;
	count++;
	}
/*
    fprintf(stderr,"counting %d frames\n", count);
 */
    return w;
    }




/*****************************************************************************
 *    FUNCTION:  convert                                                    
 *               Takes attribute/value pairs, both of them strings, and     
 *               constructs a proper X11 Arglist by converting the string   
 *               values to the requested data type.                         
 *                                                                          
 *    Arguments: parent widget                                              
 *               widgetClass of widget                                      
 *               argv and argc (containing the attribute/value pairs)       
 *               A Pointer to an Arglist and an int for passing the result  
 *                                                                          
 *    Returns:   1 on success, 0 on error                                   
 *                                                                          
 *    Used by:   setValues, tclWidgetCommand                                
 *    Uses:      getTypeOfAttribute                                         
 *****************************************************************************/

int
convert(parent, wClass, argv, argc, args, numArgs, w)
Widget        parent;    
WidgetClass   wClass;    
char        **argv;      
int           argc;      
ArgList      *args;                /* RETURN */
int          *numArgs;             /* RETURN */
Widget        w;
    {
    int               pairNum;  
    ArgList           argPtr;   
    XrmValue          input;    
    XrmValue          output;   
    XrmQuark          Qtype,Qname;
    int               i;
    int               count = 0; 

    DBUG_ENTER("convert");

    if (argc == 0)
	{
	*numArgs = 0;
	*args =NULL;
	currentStrInfo = Nil(MMattribList);
	DBUG_RETURN (1);
	}

    if (argc % 2)
	{
	fprintf(stderr, "Wafe(convert): Wrong # of args\n");
	DBUG_RETURN (0);
	}
    
    argPtr = *args = (ArgList)XtMalloc((argc/2)*sizeof(Arg));

    currentStrInfo = w ? (*(MMgetAttribList(w))) : Nil(MMattribList);
/*
    fprintf(stderr,"convert: widget %p, currentStrInfo = %p\n",
	    w, currentStrInfo);
 */   
    for (pairNum = 1; pairNum <= argc/2; pairNum++)
	{
	Qname = XrmStringToQuark(*argv); 
	Qtype = getQTypeOfAttribute(wClass, parent, Qname);

	if(Qtype)
	    { 
	    DBUG_PRINT("conv", ("N: %s, T: %s", 
				*argv, XrmQuarkToString(Qtype))); 
	    
	    argPtr->name = *argv;         /* Den Namen uebernehmen.. */  
	    currentAttrib = Qname;        /* Remember quark */
	    input.addr =   *(++argv);     /* Den Wert konvertieren.. */
	    input.size =   strlen(*argv)+1;
	    output.addr =  NULL;

	    DBUG_PRINT("conv", ("Input: String %s, size %d", 
				input.addr, input.size));
	    
	    IFCONVERTANDSTORE(parent, "String",input, 
			      XrmQuarkToString(Qtype), output)
		{
		DBUG_PRINT("conv", ("Konv OK, Gr: %u\n", output.size));
		
		if (Qtype == qString) 
		    {
		    argPtr->value = (XtArgVal)XtNewString(output.addr);
		    MMreplace(&currentStrInfo,currentAttrib,
			      (char *)(argPtr->value),XtFree);
		    }
		else 
		if (output.size == sizeof(long))
		    argPtr->value = (XtArgVal) *(long *)output.addr;
		else
		if (output.size == sizeof(short))
		    argPtr->value = (XtArgVal) *(short *)output.addr;
		else
                if (output.size == sizeof(double))
		    argPtr->value = (XtArgVal) (double *)output.addr; 
		else 
                if (output.size == sizeof(char))
		    argPtr->value = (XtArgVal) *(char *)output.addr;
		else
                if (output.size == sizeof(XtArgVal)) 
		    argPtr->value = *(XtArgVal *)output.addr;

		argPtr++; 
		count++; 
		argv++;
		}
             else 
		 {
                 argv++;
		 }
	    }
	else 
	    {
	    fprintf(stderr, "Wafe(convert):Attribute unknown: %s\n", *argv);
	    argv+=2; 
	    }
	}     

    *numArgs = count;
    DBUG_RETURN (1); 
    }

 
/*****************************************************************************
 *    FUNCTION:  name2Widget                                               
 *                                                                          
 *    Arguments: string containing an instance's name                       
 *    Returns:   Widget on success, otherwise NULL                          
 *                                                                          
 *    Used by:   nearly all tcl-commands dealing with widgets               
 *****************************************************************************/


Widget
name2Widget(name)
char  *name;
    {
    Widget result;
    char  starname[300];
    WidgetListPtr ptr = widgetTrees;
    
    if (!name)
	return(NULL);

    *starname = '*';
    strcpy(&starname[1], name);

    while (ptr) 
	{
	DBUG_PRINT("name2widget", ("Trying: %s under %s", 
				   starname, XtName(ptr->widget)));
	if (result = XtNameToWidget(ptr->widget, starname))
	    return result;
	
	DBUG_PRINT("name2widget", ("Trying: %s under %s", 
				   name,XtName(ptr->widget)));
	if (result = XtNameToWidget(ptr->widget, name))
	    return result;

	DBUG_PRINT("name2widget", ("Checking topLevel <%s>",
				   (char*)(ptr->info)));
	if (!strcmp((char*)(ptr->info), name))
	    return ptr->widget;
	
	ptr = ptr->next;
	}

    fprintf(stderr, "Wafe(name2Widget): %s not found!\n", name);
    return (NULL);
    }



/****************************************************************************
 *    FUNCTION:  name2Sink, name2Source
 *                                                                          
 *    Arguments: string containing an instance's name
 *    Returns:   source/sink widget of named text widget
 *                                                                          
 *    Used by:   some of the generated commands in XawGen as converter proc
 *****************************************************************************/

Widget 
name2Sink(string)
char * string;
    {
    Widget localVar1, sink;

    if (!(localVar1 = name2Widget(string)))
	{
	convError("name2Sink","1",string,"Widget");
	return(NULL);
	}

    XtVaGetValues(localVar1, XtNtextSink, &sink, NULL);
    return(sink);
    }

Widget 
name2Source(string)
char * string;
    {
    Widget localVar1, source;

    if (!(localVar1 = name2Widget(string)))
	{
	convError("name2Source","1",string,"Widget");
	return(NULL);
	}

    XtVaGetValues(localVar1, XtNtextSource, &source, NULL);
    return(source);
    }


char *
intsToTcl(num, intArray)
int   num;
int **intArray;
    {
    int    count;
    char   strArray[INT_ARRAY_IN_BYTES];
    char  *ptr;

    if (INT_ARRAY_IN_BYTES < (num*(sizeof(char *) + INT_AS_STRING)))
	{
	fprintf(stderr, 
		"Wafe(intsToTcl): Maximum number of list elements exceeded\n");
	return(NULL);
	}

    for(count = 0, ptr = strArray + num*sizeof(char *); 
	count < num; 
	count++, ptr += INT_AS_STRING)
	{
	*((char **)(strArray + count*sizeof(char *))) = ptr;
	sprintf(ptr, "%d", intArray[count]);  
	}
    return(Tcl_Merge(num, (char **)strArray));
    }
          
char *
widgetListToList(num, widgetArray)
int   num;
WidgetList widgetArray;
    {
    int     i;
    char   *strArray[500];

    for (i=0; i<num; i++)
	{
	strArray[i] = XtName(widgetArray[i]);
	}
    return(Tcl_Merge(num, strArray));
    }
          
               
   
/*****************************************************************************
 *    FUNCTION:  signalHandler                                             
 *               If wafe receives a signal
 *               it is forwarded to the application                
 *                                                                         
 *    Arguments: signal number                                             
 *    Returns:   nothing                                                   
 *                                                                         
 *    Used by:   main                                                      
 *****************************************************************************/

void
signalHandler(sig)
int   sig;
    {
    DBUG_ENTER ("signalHandler");
    DBUG_PRINT("signal", ("Signal %d received", sig));
    if (sig != SIGCLD) 
	{
	fprintf(stderr, "Wafe(main): Signal %d received!\n", sig);
        kill(pid, sig);
     	fprintf(stderr, "Wafe(main): forwarding it to application!\n");
        exit(0);
	}
    DBUG_VOID_RETURN;
    }

/*****************************************************************************
 *    FUNCTION:  XIOHandler                                             
 *               If wafe receives a XIOError
 *               SIGTERM is sent to the application
 *                                                                         
 *    Arguments: display, error event
 *    Returns:   nothing                                                   
 *                                                                         
 *    Used by:   main                                                      
 *****************************************************************************/

int
IOErrorHandler(dpy,xe)
Display *dpy;
XErrorEvent *xe;
    {
    char buffer[50];
    
    DBUG_ENTER ("IOErrorHandler");

    if (tclHandlers[XIOERR])
	{
	sprintf(buffer,"%s %s",tclHandlers[XIOERR],DisplayString(dpy));
	Tcl_Eval(interpreter, buffer, 0, NULL);
	}

    fprintf(stderr, "Wafe(main): XIOError on %s, terminating child!\n",
	    DisplayString(dpy));

    if (pid) 
	kill(pid, SIGTERM);
    else 
	exit(0);
    
    DBUG_RETURN (-1);
    }


/*****************************************************************************
 *    FUNCTION: iSocketPair
 *            Use socketpair on BSD or ioctl on Sys 5 or pipes
 *                                                                          
 *    Arguments: array of file descriptors
 * 
 *               parentRd childWr childRd parentWr
 *
 *    Returns:  0 if success, otherwise -1                         
 *                                                                          
 *    Used by:  main
 *****************************************************************************/
#ifdef PIPES
int
iSocketPair(fd,onlyOne)
int      fd[4],onlyOne;
    {
    if (pipe(&fd[0])<0) return(-1);
    if (onlyOne) return(0);
    if (pipe(&fd[2])<0) return(-1);
    return(0);
    }
#else
#ifdef STREAMS
#define SPX_DEVICE "/dev/spx"
int
iSocketPair(fd, onlyOne)
int      fd[4], onlyOne;
    {
    struct strfdinsert  ins;
    queue_t                  *pointer;

    if ((fd[0] = open(SPX_DEVICE, O_RDWR)) < 0)
	return(-1);

    if ((fd[1] = open(SPX_DEVICE, O_RDWR)) < 0)
	{
	close(fd[0]);
	return(-1);
	}

    ins.ctlbuf.buf = (char *) &pointer;
    ins.ctlbuf.maxlen = sizeof(queue_t *);		
    ins.ctlbuf.len = sizeof(queue_t *);		

    ins.databuf.buf = (char *) 0;
    ins.databuf.maxlen = 0;
    ins.databuf.len = -1;

    ins.fildes = fd[1];
    ins.flags = 0;
    ins.offset = 0;

    if (ioctl(fd[0], I_FDINSERT, (char *) &ins) < 0)
	{
	close(fd[0]);
	close(fd[1]);
	return(-1);
	}

    fd[2]=fd[1];
    fd[3]=fd[0];
    return(0);
    }

#else

int
iSocketPair(fd, onlyOne)
int      fd[4], onlyOne;
    {
    if (socketpair(AF_UNIX, SOCK_STREAM, 0, fd) == -1)
	return(-1);
    else
	{
	fd[2]=fd[1];
	fd[3]=fd[0];
	return(0);
	} 
    }
#endif
#endif


/*****************************************************************************
 *    FUNCTION:  MAIN                                
 *               Command Line parsing, starts Xt Application, registers     
 *               addditional tcl-commands  and according                   
 *               to mode establishes the socket connection or reads in a   
 *               file or reads from stdin.                                 
 *                                                                         
 *    Arguments: argc and argv                                             
 *    Returns:   nothing                                                   
 *                                                                         
 *****************************************************************************/

void
main(argc, argv, env)
int    argc;
char **argv;
char **env;
    {
    int                    i;
    Boolean                promptMode = True;
    Boolean                execMode = False;
    Boolean                opt_p = False;
    int                    args = 0; 
    int                    cargc = 0;
    char                 **cargv;    
    char                 **cargvCount;
    char                   file[50];
    char                  *charPtr;
    char                  *execString;

    int                    sp[4];  
    int                    ep[4];  
    int                    cp[4];

    static XtActionsRec    tclAction[] = {
	{"exec",  execActionProc},
	};

    DBUG_ENTER ("main");
    DBUG_PROCESS (argv[0]);

    qString = XrmStringToQuark("String");
    qCallback = XrmStringToQuark("Callback");
    qPointer = XrmStringToQuark("Pointer");
    qInt = XrmStringToQuark("Int"); 
    qDimension = XrmStringToQuark("Dimension"); 
    qCardinal = XrmStringToQuark("Cardinal"); 
    qPosition = XrmStringToQuark("Position"); 
    qFloat = XrmStringToQuark("Float"); 
    qDouble = XrmStringToQuark("Double"); 
    qBoolean = XrmStringToQuark("Boolean"); 
    qWidgetList = XrmStringToQuark("WidgetList"); 

    strHead = &STRINFOroot;
    cargvCount = cargv = (char **)XtMalloc((argc * sizeof(char *)) + 1);

     /* In standard mode, the class will be derived from the application's name,
       * in file mode from the filename and in direct  mode it is defined to be Wafe
       */

    strcpy(class, argv[0]);
    application[0] = '\0';

     /*
      * Argument parsing loop - build new argument list without wafe's 
      * own args
      */ 

    while (args < argc)
	{
	if ((argv[args][0] == '-') 
	    && (argv[args][1] == '-') 
	    && (strlen(argv[args]) > 2))
	    {
	    /***  Debug - Mode  ***/
	    if (argv[args][2] == 'D')
                {                             
                DBUG_PUSH (&(argv[args][3]));  
                args++; 
	        }                             
	    else
                             /***  Prompt - Mode  ***/ 
	    if (argv[args][2] == 'n')            /* Turn prompt mode off */
	        {                              
                promptMode = False;               
                args++;
                }
	    else
                             /***  Version  ***/ 
            if (argv[args][2] == 'v')            /* Print Version */
	        {                              
		fprintf(stderr, "\nThis is Wafe, version %s\n\n", VERSION);
                fprintf(stderr, "Copyright (C) 1992 by Gustaf Neumann, Stefan Nusser\n");
                fprintf(stderr, "University of Economics and Business Administration\n"); 
                fprintf(stderr, "Vienna, Austria\n\n");
                exit(0);
                }
	    else
                             /***  Input - Mode  ***/
	    if (argv[args][2] == 'i')            /* Enter input mode  */
	        {                            
                inputMode  = True;            
                args++;
                }
	    else
                             /***  Execute - Mode  ***/
	    if (argv[args][2] == 'e')            /* Enter execute mode  */
	        {      
		int     comLength = 0;

		args++;
                execMode  = True;     
		fileMode = True;
		execString = argv[args];

		if (execString == NULL)
		    {
		    fprintf(stderr, "Wafe: Option +e needs an additional argument\n");
		    exit(-1);
		    }

                strcpy(class, "Wafe");
                }
	    else
                             /***  Direct - Mode  ***/
            if (argv[args][2] == 'd')            /* Enter direct mode */
	        {                           
                directMode = True;

                toClient = fileno(stdout);
                fromClient = fileno(stdin);
                errorClient = -1;

                strcpy(class, "Wafe");
                args++; 
	        }
	    else 
                             /*** Change Prompt Character  ***/
            if (argv[args][2] == 'c')          
	        {                             
                promptChar = argv[args][3];
                fprintf(stderr, "PromptChar changed to %c\n", promptChar);
                args++;
	        }
	    else
	    if (argv[args][2] == 'p')
	        {
		opt_p = True;
                strcpy(class, argv[args+1]);
                strcpy(application, argv[args+1]);
                args += 2;
	        }
            else
                             /***  File - Mode  ***/
            if (argv[args][2] == 'f')        /* Enter file mode */
                {
                if (argv[args][3] == 'i')    /* For convenience: --fi == --f --i */
		    {
                     inputMode = True;
		     }

                fileMode = True;
                strcpy(file, argv[args+1]);   /* Next Arg = File to execute */
                strcpy(class, file);

                args +=2;
                }
	    else
		{
                fprintf(stderr, "Wafe(main): Unknown Argument, ignoring!\n");
                args++; 
                }
	    }
	else 
	    {
	    *cargvCount = XtNewString(argv[args]);
	    cargc++;
	    args++;
	    cargvCount++;
	    }
	}

    *cargvCount = NULL;

    /* DEBUG - Check that everybody has got his args */

    DBUG_PRINT("start", ("Arguments at startup (argc = %d):", argc));
    for (i = 0; argv[i] != NULL; i++)
	DBUG_PRINT("start",("argv[%d] = %s\n", i, argv[i]));

    DBUG_PRINT("start", ("Xt's + app's args (cargc = %d):", cargc));
    for (i = 0; cargv[i] != NULL; i++)
	DBUG_PRINT("start",("cargv[%d] = %s\n", i, cargv[i]));

     /*
       * If the program's name contains a path specification cut it off...
       */
    
    if (charPtr = strrchr(class, '/'))
	strcpy(class, ++charPtr);

   /*
     *  We will need this information when we execute the application process ...
     */

    if (!application[0]) 
	strcpy(application, class); 

     /*
       * If the name contains a dot, which is not allowed in Xt's class names, substitute it
       * with a dash (keep this in mind, it will possibly affect the name of the AD file!
       */

    while(charPtr = strrchr(class, '.'))
	*charPtr = '-';
         
     /* Just because argv[0] will appear in the titlebar of the 
      * top level window - if we were in file mode the title would
      * always be wafe (which is probably not desired).
      */


    cargv[0] = XtNewString(class);

    /* 
      *  Now create class: Capitalized X, second letter capitalized, 
      *  rest remains unchanged. Under this name, the app-defaults file
      *  will be searched! 
      */

    if ((class[0] == 'x') || (class[0] == 'X'))
	{
	class[0] = 'X';
	class[1] = toupper(class[1]);
	}
    else
	class[0] = toupper(class[0]);

     /* Must initialize here, so that Xt can strip off its arguments */

    topLevel = XtVaAppInitialize(&appContext, class, NULL, 0, 
				 &cargc, cargv, NULL, NULL); 
/*
    XtAppSetErrorHandler(appContext,IOErrorHandler);
 */
    XSetIOErrorHandler(IOErrorHandler);


    /* the default widget tree is under topLevel */
    addToEndOfWidgetList(&widgetTrees,topLevel,"topLevel");
    

    DBUG_PRINT("start", ("App's args (cargc = %d):", cargc)); 

    for(i = 0; cargv[i] != NULL; i++)
	DBUG_PRINT("start",("cargv[%d] = %s\n", i, cargv[i])); 


    if ((!directMode) && (!fileMode))     /* => standard mode            */
	{     

	 /*
          *   Main socketpair, will be connected to the client's stdin
          *   and stdout channels.
          */

	if (iSocketPair(sp,0) == -1)
	    {
	    fprintf(stderr, "Wafe(main): Couldn't create socket, aborting\n");
	    exit(-1);
	    }
/*
printf("sp %d %d %d %d\n",sp[0],sp[1],sp[2],sp[3]);
*/

	 /*
          *   This pair is used to facilitate an additional communication channel.
          */

	if (iSocketPair(cp,1) == -1)
	    {
	    fprintf(stderr, "Wafe(main): Couldn't create socket, aborting\n");
	    exit(-1);
	    }
/*
 printf("cp %d %d %d %d\n",cp[0],cp[1],cp[2],cp[3]);
*/

	comClient = cp[1];  /* Communication endpoint for client  */
	comWafe = cp[0];    /* Communication endpoint for wafe    */
/*
          close(cp[2]);
          close(cp[3]);
*/
	 /*
          *  Used to catch client's error messages - wafe will never write on
          *  this channel!
          */

	if (iSocketPair(ep,1) == -1)
	    {
	    fprintf(stderr, "Wafe(main): Couldn't create socket for app's stderr\n");
	    exit(-1);
	    }
/*
 printf("ep %d %d %d %d\n",ep[0],ep[1],ep[2],ep[3]);
*/
	pid = fork();
    
	if (!pid)                       /* Child-process               */
	    {
	    dup2(sp[1], fileno(stdout));
	    dup2(sp[2], fileno(stdin));
	    dup2(ep[1], fileno(stderr));

	    close(ep[0]);              /* Those 2 are for ..          */
	    close(ep[1]);              /* getting stderr-messages.    */

	    close(sp[1]);              /* For regular communication ..*/
	    close(sp[0]);              /* via stdin and stdout!       */

	    if (!opt_p && (*application != 'x') && (*application != 'X'))
		{
		fprintf(stderr, "Wafe(main): Wrong File was started, aborting\n");
		sleep(1);
		exit(-1);
		}

	    cargv[0] = opt_p ? application : application+1;

	    execvp(cargv[0], cargv);   /* execute the client-program  */  
   
	    fprintf(stderr, "Wafe(main): Error in exec-call to <%s>, aborting\n",
		    cargv[0]);  
	    exit(-1);
	    }

	if (pid == -1)
	    {
	    fprintf(stderr, "Wafe(main): Couldn't fork!\n");
	    DBUG_VOID_RETURN;
	    }

/*
          close(sp[1]);
          close(ep[1]);
*/

          /* Those are the four main communication-channels:
           *
           * 1) errorClient   is the client's diagnostic output,
           * 2) toClient      leads to client's stdin and
           * 3) fromClient    is client's stdout 
           * 4) wafeCom       additional communication channel
           */

	errorClient = ep[0];
	toClient = sp[3];
	fromClient = sp[0];

          /* If one of the following signals is caught, wafe exits and 
           * the same signal is sent to the application
           */

	signal(SIGTERM, signalHandler);  
	signal(SIGQUIT, signalHandler);
	signal(SIGINT, signalHandler);
	signal(SIGHUP, signalHandler);
	signal(SIGPIPE, signalHandler);
	signal(SIGCLD, signalHandler);

	}

     /* Now check for invalid combination of modes */

    if ((execMode && inputMode) || (execMode && directMode))
	{
	fprintf(stderr, "Wafe: Invalid combination of modes\n");
	exit(-1);
	}

    if ((directMode) && (inputMode))
	{
	fprintf(stderr, 
	   "Wafe: Invalid combination direct-mode and input-mode, aborting\n");
	exit(-1);
	}

    if ((directMode) && (fileMode))
	{
	fprintf(stderr, 
	   "Wafe: Invalid combination direct-mode and file-mode, aborting\n");
	exit(-1);
	}

    /* Now register input-handlers according to mode */

    if (!fileMode)
	{
/*
 printf("registering inputhandler fromClient %d\n",fromClient);
*/
	XtAppAddInput(appContext, fromClient, (XtPointer)XtInputReadMask,
		      getInput, (XtPointer)(int)promptMode); /* to make gcc happy */


	if (!directMode)
	    { 
/*
 printf("registering inputhandler errorCLient %d\n",errorClient);
*/
	    XtAppAddInput(appContext, errorClient,
			  (XtPointer)XtInputReadMask, stderrHandlerProc, 
			  (XtPointer)NULL); 

/*
 printf("registering inputhandler comWafe %d\n",comWafe);
*/
	    XtAppAddInput(appContext, comWafe, (XtPointer)XtInputReadMask,
			  communicationHandlerProc, NULL); 
	    }

	}

      
    if ((inputMode) && (!execMode))
	XtAppAddInput(appContext, fileno(stdin), (XtPointer)XtInputReadMask,
		      stdinHandlerProc, NULL); 

    XtAppAddActions(appContext, tclAction, 1);
    XtVaGetApplicationResources(topLevel, &app_data, resources,
				XtNumber(resources), NULL);

    interpreter = Tcl_CreateInterp();

    /* initialize handlers */
    for (i=0; i<HANDLERMAX; tclHandlers[i]=NULL, i++); 
    packages[0] = '\0';    

{  COM_PROTOTYPE(Tcl_ListCmd)
   Tcl_CreateCommand(interpreter, "llist", Tcl_ListCmd, NULL, NULL); 
}

  /*
      * The following include files contain the prototypings as well as the
      * Tcl_createCommand statements for the automagically gernerated functions
      */

    callbackQuarkInitialize();
    Initialize_XtGen();
    Initialize_MiscGen();
    Initialize_XawGen();
    Initialize_XawR5Gen();
    Initialize_XmGen();
    Initialize_XmGraphGen();
    Initialize_AtGen();
    Initialize_XcGen();
    Initialize_rddGen();
   
/*  using non-caching converter for names to widget to avoid dangling references
 *  thanx Christian Holzbaur
 */
    XtSetTypeConverter(XtRString, XtRWidget, 
        (XtTypeConverter)CvtStringToWidget, 
	NULL, 0, XtCacheNone, NULL);

    XtSetTypeConverter(XtRString, XtRCallback, 
	(XtTypeConverter)CvtStringToCallback, 
	NULL, 0, XtCacheNone, NULL);

    XtSetTypeConverter(XtRString, XtRPointer, 
	(XtTypeConverter)CvtStringToPointer, 
	NULL, 0, XtCacheNone, NULL);

#ifdef PRER5
    Tcl_SetVar(interpreter, "XVERSION", "R4", TCL_GLOBAL_ONLY);
#else
    Tcl_SetVar(interpreter, "XVERSION", "R5", TCL_GLOBAL_ONLY);
#endif
    Tcl_SetVar(interpreter, "WAFEVERSION", VERSION, TCL_GLOBAL_ONLY);
    Tcl_SetVar(interpreter, "PACKAGES", packages, TCL_GLOBAL_ONLY);


    if (execMode)
        {
	if(Tcl_Eval(interpreter, execString, 0, NULL) == TCL_ERROR)
	    fprintf(stderr, "Wafe(execute mode): %s\n", interpreter->result);
	}

    if ((fileMode) && (!execMode))
        {
	char    intBuffer[20];

	Tcl_SetVar(interpreter, "ARGV", Tcl_Merge(cargc, cargv), 
		   TCL_GLOBAL_ONLY);

	sprintf(intBuffer, "%d", cargc);
	Tcl_SetVar(interpreter, "ARGC",  intBuffer, TCL_GLOBAL_ONLY);

	for (i = 0; env[i] != NULL; i++);
          Tcl_SetVar(interpreter, "ENV", Tcl_Merge(i, env), TCL_GLOBAL_ONLY);

	if (Tcl_EvalFile(interpreter, file) == TCL_ERROR)
	    {
	    fprintf(stderr, "Wafe(file evaluation): %s\n", 
		    interpreter->result);
	    exit(-1);
	    }
	}

    if (strcmp(app_data.init_com, "noCom"))
        {
	fprintf(stderr, "Wafe: Start mit %s\n", app_data.init_com);
	SEND(app_data.init_com);
	}

#ifdef RDD
    rddAppMainLoop(appContext);
#else
    XtAppMainLoop(appContext);
#endif

    DBUG_VOID_RETURN;
    }

