D B U G C Program Debugging Package by This document introduces a
macro  based  C  debugging  package which has proven to be a very
flexible and useful tool for debugging, testing,  and  porting  C
programs.

All of the features of the package can  be  enabled  or  disabled
dynamically  at  execution time.  This means that production pro-
grams will run normally when debugging is not enabled, and  elim-
inates the need to maintain two separate versions of a program.

Many of the things easily accomplished with  conventional  debug-
ging tools, such as symbolic debuggers, are difficult or impossi-
ble with this package, and vice versa.  Thus the  package  should
be  thought of as a replacement or substitute for other debugging
tools, but simply as a useful  to  the  program  development  and
maintenance environment.

INTRODUCTION

Almost every program development environment worthy of  the  name
provides some sort of debugging facility.  Usually this takes the
form of a program which is capable of  controlling  execution  of
other  programs and examining the internal state of other execut-
ing programs.  These types of programs will  be  referred  to  as
external  debuggers since the debugger is not part of the execut-
ing program.  Examples of this type of debugger include  the  and
debuggers  provided with the UNIX is a trademark of AT&T Bell La-
boratories.  operating system.

One of the problems associated with developing programs in an en-
vironment with good external debuggers is that developed programs
tend to have little or no internal instrumentation.  This is usu-
ally  not  a  problem  for the developer since he is, or at least
should be, intimately familiar with  the  internal  organization,
data  structures, and control flow of the program being debugged.
It is a serious problem for maintenance programmers, who are  un-
likely  to  have  such  familiarity  with the program being main-
tained, modified, or ported to another environment.  It is also a
problem,  even for the developer, when the program is moved to an
environment with a primitive or unfamiliar debugger, or  even  no
debugger.

On the other hand, is an example of an  internal  debugger.   Be-
cause  it requires internal instrumentation of a program, and its
usage does not depend on any special capabilities of  the  execu-
tion  environment, it is always available and will execute in any
environment that the program itself will execute  in.   In  addi-
tion,  since it is a complete package with a specific user inter-
face, all programs which use it will be provided with similar de-
bugging  capabilities.   This is in sharp contrast to other forms
of internal instrumentation where each developer has  their  own,
usually less capable, form of internal debugger.  In summary, be-
cause is an internal  debugger  it  provides  consistency  across
operating  environments,  and  because  it  is  available  to all
developers it provides consistency across  all  programs  in  the
same environment.

The package imposes only a slight speed penalty on executing pro-
grams,  typically  much  less  than 10 percent, and a modest size
penalty, typically 10 to 20 percent.  By defining  a  specific  C
preprocessor  symbol both of these can be reduced to zero with no
changes required to the source code.

The following list is a quick summary of the capabilities of  the
package.  Each capability can be individually enabled or disabled
at the time a program is invoked by  specifying  the  appropriate
command  line  arguments.  Execution trace showing function level
control flow in a semi-graphically manner  using  indentation  to
indicate  nesting depth.  Output the values of all, or any subset
of, key internal variables.  Limit actions to a specific  set  of
named  functions.   Limit  function  trace to a specified nesting
depth.  Label each output line with source  file  name  and  line
number.   Label  each  output  line with name of current process.
Push or pop internal debugging  state  to  allow  execution  with
built in debugging defaults.  Redirect the debug output stream to
standard output (stdout) or a named  file.   The  default  output
stream  is standard error (stderr).  The redirection mechanism is
completely independent of  normal  command  line  redirection  to
avoid output conflicts.

PRIMITIVE DEBUGGING TECHNIQUES

Internal instrumentation is already a familiar  concept  to  most
programmers,  since  it  is usually the first debugging technique
learned.   Typically,  "print statements"  are  inserted  in  the
source code at interesting points, the code is recompiled and ex-
ecuted, and the resulting output is examined  in  an  attempt  to
determine where the problem is.

The procedure is iterative, with each iteration yielding more and
more   output,  and  hopefully  the  source  of  the  problem  is
discovered before the output becomes too large to  deal  with  or
previously  inserted  statements need to be removed.  Figure 1 is
an example of this type of primitive debugging  technique.   #in-
clude <stdio.h>

main (argc, argv) int argc; char *argv[]; {
    printf ("argv[0] = %d\n", argv[0]);
    /*
     *  Rest of program
     */
    printf ("== done ==\n"); }
                          Figure 1
               Primitive Debugging Technique

Eventually, and usually after at least  several  iterations,  the
problem  will  be  found and corrected.  At this point, the newly
inserted print statements must be dealt with.  One obvious  solu-
tion  is  to simply delete them all.  Beginners usually do this a
few times until they have to repeat the entire process every time
a  new  bug  pops  up.   The  second  most obvious solution is to
somehow disable the output, either through the source  code  com-
ment  facility, creation of a debug variable to be switched on or
off, or by using the C preprocessor.  Figure 2 is an  example  of
all three techniques.  #include <stdio.h>

int debug = 0;

main (argc, argv) int argc; char *argv[]; {
    /* printf ("argv = %x\n", argv) */
    if (debug) printf ("argv[0] = %d\n", argv[0]);
    /*
     *  Rest of program
     */ #ifdef DEBUG
    printf ("== done ==\n"); #endif }
                          Figure 2
                  Debug Disable Techniques

Each technique has its advantages and disadvantages with  respect
to dynamic vs static activation, source code overhead, recompila-
tion  requirements,  ease  of  use,  program  readability,   etc.
Overuse  of  the  preprocessor solution quickly leads to problems
with source code readability and  maintainability  when  multiple
symbols are to be defined or undefined based on specific types of
debug desired.  The source code can be made slightly  more  read-
able by suitable indentation of the arguments to match the inden-
tation of the code, but not all C preprocessors allow this.   The
only  requirement  for the standard C preprocessor is for the '#'
character to appear in the first column, but even this seems like
an  arbitrary and unreasonable restriction.  Figure 3 is an exam-
ple of this usage.  #include <stdio.h>

main (argc, argv) int argc; char *argv[]; { #   ifdef DEBUG
    printf ("argv[0] = %d\n", argv[0]); #   endif
    /*
     *  Rest of program
     */ #   ifdef DEBUG
    printf ("== done ==\n"); #   endif }
                          Figure 3
              More Readable Preprocessor Usage

FUNCTION TRACE EXAMPLE

We will start off learning about the capabilities of the  package
by  using a simple minded program which computes the factorial of
a number.  In order to  better  demonstrate  the  function  trace
mechanism,  this program is implemented recursively.  Figure 4 is
the main function for this factorial program.  #include <stdio.h>
/* User programs should use <local/dbug.h> */ #include "dbug.h"

main (argc, argv) int argc; char *argv[]; {
    register int result, ix;
    extern int factorial (), atoi ();

    DBUG_ENTER ("main");
    DBUG_PROCESS (argv[0]);
    for (ix = 1; ix  <  argc  &&  argv[ix][0]  ==  '-';  ix++)  {
        switch    (argv[ix][1])    {                  case   '#':
                DBUG_PUSH                       (&(argv[ix][2]));
                break;         }
    }
    for  (;  ix  <  argc;  ix++)  {          DBUG_PRINT  ("args",
("argv[%d]  =  %s",  ix,  argv[ix]));          result = factorial
(atoi (argv[ix]));         printf ("%d\n", result);
    }
    DBUG_RETURN (0); }
                          Figure 4
                 Factorial Program Mainline

The function is responsible for processing any command  line  op-
tion  arguments  and then computing and printing the factorial of
each non-option argument.  First of all, notice that all  of  the
debugger functions are implemented via preprocessor macros.  This
does not detract from the readability of the code and makes disa-
bling  all  debug compilation trivial (a single preprocessor sym-
bol, forces the macro expansions to be null).   Also  notice  the
inclusion  of the header file from the local header file directo-
ry.  (The version included here is the test version in  the  dbug
source  distribution  directory).  This file contains all the de-
finitions for the debugger macros, which all have the form

The macro informs that debugger that we have entered the function
named  It must be the very first "executable" line in a function,
after all declarations and before any other executable line.  The
macro  is  generally used only once per program to inform the de-
bugger what name the program was invoked with.  The  macro  modi-
fies  the current debugger state by saving the previous state and
setting a new state based on the control string passed as its ar-
gument.   The  macro is used to print the values of each argument
for which a factorial is to be computed.  The macro tells the de-
bugger  that the end of the current function has been reached and
returns a value to the calling function.   All  of  these  macros
will  be  fully explained in subsequent sections.  To use the de-
bugger, the factorial program is invoked with a command  line  of
the  form:  factorial  -#d:t 1 2 3 The function recognizes the "-
#d:t" string as a debugger control string,  and  passes  the  de-
bugger  arguments ("d:t") to the runtime support routines via the
macro.  This particular string enables output from the macro with
the 'd' flag and enables function tracing with the 't' flag.  The
factorial function is then called three times, with the arguments
"1",  "2", and "3".  Note that the DBUG_PRINT takes exactly argu-
ments, with the second argument (a  format  string  and  list  of
printable values) enclosed in parenthesis.  Debug control strings
consist of a header, the "-#", followed by a colon separated list
of  debugger arguments.  Each debugger argument is a single char-
acter flag followed by an optional comma separated list of  argu-
ments  specific  to  the  given flag.  Some examples are: -#d:t:o
-#d,in,out:f,main:F:L Note that previously enabled  debugger  ac-
tions can be disabled by the control string "-#".

The definition of the factorial function, symbolized as "N!",  is
given  by:  N!  =  N  * N-1 * ... 2 * 1 Figure 5 is the factorial
function which implements this algorithm recursively.  Note  that
this  is  not necessarily the best way to do factorials and error
conditions are ignored completely.  #include  <stdio.h>  /*  User
programs should use <local/dbug.h> */ #include "dbug.h"

int factorial (value) register int value; {
    DBUG_ENTER ("factorial");
    DBUG_PRINT ("find", ("find %d factorial", value));
    if (value > 1) {
        value *= factorial (value - 1);
    }
    DBUG_PRINT ("result", ("result is %d", value));
    DBUG_RETURN (value); }
                          Figure 5
                     Factorial Function

One advantage (some may not consider it so) to using the  package
is  that it strongly encourages fully structured coding with only
one entry and one exit point in  each  function.   Multiple  exit
points,  such as early returns to escape a loop, may be used, but
each such point requires the use of an appropriate or macro.

To build the factorial program on a system, compile and link with
the  command:  cc  -o  factorial main.c factorial.c -ldbug The "-
ldbug" argument tells the loader to link in the  runtime  support
modules  for the package.  Executing the factorial program with a
command of the form: factorial 1 2 3 4  5  generates  the  output
shown in figure 6.  1 2 6 24 120
                          Figure 6
                    factorial 1 2 3 4 5

Function level tracing is enabled by passing the debugger the 't'
flag in the debug control string.  Figure 7 is the output result-
ing from the command "factorial -#t:o 3 2".  |   >factorial |   |
>factorial  |    |   <factorial |   <factorial 2 |   >factorial |
|   >factorial |   |   |   >factorial |   |    |    <factorial  |
|   <factorial |   <factorial 6 <main
                          Figure 7
                    factorial -#t:o 3 2

Each entry to or return from a function is indicated by  '>'  for
the entry point and '<' for the exit point, connected by vertical
bars to allow matching points to be easily found  when  separated
by large distances.

This trace output indicates that there was  an  initial  call  to
factorial  from main (to compute 2!), followed by a single recur-
sive call to factorial to compute 1!.  The main program then out-
put  the  result  for  2! and called the factorial function again
with the second argument, 3.  Factorial called itself recursively
to compute 2! and 1!, then returned control to main, which output
the value for 3! and exited.

Note that there is no matching entry point "main>" for the return
point  "<main" because at the time the macro was reached in main,
tracing was not enabled yet.  It was only after the macro was ex-
ecuting that tracing became enabled.  This implies that the argu-
ment list should be processed as early as possible since all code
preceding the first call to is essentially invisible to (this can
be worked around by inserting a temporary immediately  after  the
macro.

One last note, the trace output normally comes out on  the  stan-
dard error.  Since the factorial program prints its result on the
standard output, there is the possibility of the  output  on  the
terminal being scrambled if the two streams are not synchronized.
Thus the debugger is told to write its  output  on  the  standard
output instead, via the 'o' flag character.  Note that no 'o' im-
plies the default (standard error), a 'o' with no arguments means
standard  output, and a 'o' with an argument means used the named
file.  I.E, "factorial -#t:o,logfile 3 2" would write  the  trace
output in "logfile".  Because of implementation details, programs
usually run faster when writing to  stdout  rather  than  stderr,
though this is not a prime consideration in this example.

USE OF DBUG_PRINT MACRO

The mechanism used to produce "printf" style output is the macro.

To allow selection of output from specific macros, the first  ar-
gument to every macro is a keyword.  When this keyword appears in
the argument list of the 'd' flag in a debug control  string,  as
in  "-#d,keyword1,keyword2,...:t",  output from the corresponding
macro is enabled.  The default when there is no 'd' flag  in  the
control string is to enable output from all macros.

Typically, a program will be run once, with  no  keywords  speci-
fied,  to determine what keywords are significant for the current
problem (the keywords are printed  in  the  macro  output  line).
Then the program will be run again, with the desired keywords, to
examine only specific areas of interest.

The second argument to a macro is a standard printf style  format
string  and  one  or  more  arguments  to  print, all enclosed in
parenthesis so that they collectively become a single macro argu-
ment.   This is how variable numbers of printf arguments are sup-
ported.  Also note that no explicit newline is  required  at  the
end  of  the  format  string.  As a matter of style, two or three
small macros are preferable to a single macro with a huge  format
string.  Figure 8 shows the output for default tracing and debug.
|   args: argv[2] = 3 |   >factorial |   |   find:  find  3  fac-
torial  |    |    >factorial |   |   |   find: find 2 factorial |
|   |   >factorial |   |   |   |   find: find 1 factorial  |    |
|    |    result:  result  is  1 |   |   |   <factorial |   |   |
result: result is 2 |   |   <factorial |   |   result: result  is
6 |   <factorial 6 <main
                          Figure 8
                    factorial -#d:t:o 3

The output from the macro is indented to match the  trace  output
for  the  function  in which the macro occurs.  When debugging is
enabled, but not trace, the output starts  at  the  left  margin,
without indentation.

To demonstrate selection of specific macros for output, figure  9
shows  the  result when the factorial program is invoked with the
debug control string "-#d,result:o".  factorial:  result:  result
is  1 factorial: result: result is 2 factorial: result: result is
6 factorial: result: result is 24 24
                          Figure 9
                  factorial -#d,result:o 4

It is sometimes desirable to restrict debugging and trace actions
to  a  specific  function  or  list of functions.  This is accom-
plished with the 'f' flag character in the debug control  string.
Figure  10  is  the output of the factorial program when run with
the control string "-#d:f,factorial:F:L:o".  The 'F' flag enables
printing  of the source file name and the 'L' flag enables print-
ing of the source file line number.
   factorial.c:     9: factorial: find: find 3 factorial
   factorial.c:     9: factorial: find: find 2 factorial
   factorial.c:     9: factorial: find: find 1 factorial
   factorial.c:    13: factorial: result: result is 1
   factorial.c:    13: factorial: result: result is 2
   factorial.c:    13: factorial: result: result is 6 6
                         Figure 10
             factorial -#d:f,factorial:F:L:o 3

The output in figure 10 shows that the "find" macro  is  in  file
"factorial.c"  at  source line 8 and the "result" macro is in the
same file at source line 12.

SUMMARY OF MACROS

This section summarizes the usage of all currently defined macros
in the package.  The macros definitions are found in the user in-
clude file from the standard include directory.

Used to tell the runtime support module the name of the  function
being  entered.  The argument must be of type "pointer to charac-
ter".  The DBUG_ENTER macro must precede all executable lines  in
the function just entered, and must come after all local declara-
tions.  Each DBUG_ENTER macro must have a matching DBUG_RETURN or
DBUG_VOID_RETURN  macro  at the function exit points.  DBUG_ENTER
macros used without a matching  DBUG_RETURN  or  DBUG_VOID_RETURN
macro  will  cause warning messages from the package runtime sup-
port module.  EX: DBUG_ENTER ("main"); Used at each exit point of
a function containing a DBUG_ENTER macro at the entry point.  The
argument is the value to return.  Functions which return no value
(void)  should use the DBUG_VOID_RETURN macro.  It is an error to
have a DBUG_RETURN or DBUG_VOID_RETURN macro in a function  which
has  no matching DBUG_ENTER macro, and the compiler will complain
if    the    macros     are     actually     used     (expanded).
EX: DBUG_RETURN (value);
EX: DBUG_VOID_RETURN; Used to name the current process being exe-
cuted.  A typical argument for this macro is "argv[0]", though it
will   be   perfectly    happy    with    any    other    string.
EX: DBUG_PROCESS (argv[0]);  Sets a new debugger state by pushing
the current state onto an internal stack and setting up  the  new
state  using  the  debug control string passed as the macro argu-
ment.  The most common usage is to set the state specified  by  a
debug control string retrieved from the argument list.  Note that
the leading "-#" in a debug control string specified as a command
line  argument must be passed as part of the macro argument.  The
proper usage is to pass a pointer to the first character the "-#"
string.  EX: DBUG_PUSH ((argv[i][2]));
EX: DBUG_PUSH ("d:t");
EX: DBUG_PUSH (""); Restores the previous debugger state by  pop-
ping  the state stack.  Attempting to pop more states than pushed
will be ignored and no warning will be given.  The DBUG_POP macro
has  no  arguments.  EX: DBUG_POP (); The DBUG_FILE macro is used
to do explicit I/O on the debug output stream.  It is used in the
same  manner as the symbols "stdout" and "stderr" in the standard
I/O package.  EX: fprintf (DBUG_FILE, "Doing my own  I/O!0);  The
DBUG_EXECUTE  macro is used to execute any arbitrary C code.  The
first argument is the debug keyword, used to trigger execution of
the  code  specified  as the second argument.  This macro must be
used cautiously because, like the DBUG_PRINT  macro,  it  is  au-
tomatically  selected by default whenever the 'd' flag has no ar-
gument    list    (I.E.,    a    "-#d:t"     control     string).
EX: DBUG_EXECUTE ("abort", abort ()); These macros, where N is in
the range 2-5, are currently obsolete and will be  removed  in  a
future release.  Use the new DBUG_PRINT macro.  Used to do print-
ing via the "fprintf"  library  function  on  the  current  debug
stream,  DBUG_FILE.   The  first argument is a debug keyword, the
second is a format string and the  corresponding  argument  list.
Note  that  the format string and argument list are all one macro
argument      and      be      enclosed      in      parenthesis.
EX: DBUG_PRINT ("eof", ("end of file found"));
EX: DBUG_PRINT ("type", ("type is %x", type));
EX: DBUG_PRINT ("stp", ("%x -> %s", stp, stp -> name));  Used  in
place of the setjmp() function to first save the current debugger
state and then execute the standard setjmp call.  This allows the
debugger to restore its state when the DBUG_LONGJMP macro is used
to invoke the standard longjmp() call.  Currently  all  instances
of  DBUG_SETJMP  must  occur  within the same function and at the
same function  nesting  level.   EX: DBUG_SETJMP (env);  Used  in
place of the longjmp() function to first restore the previous de-
bugger state at the time of the last DBUG_SETJMP and then execute
the   standard   longjmp()   call.    Note   that  currently  all
DBUG_LONGJMP macros restore the state at the  time  of  the  last
DBUG_SETJMP.    It   would   be  possible  to  maintain  separate
DBUG_SETJMP and DBUG_LONGJMP pairs by having the debugger runtime
support module use the first argument to differentiate the pairs.
EX: DBUG_LONGJMP (env,val);

DEBUG CONTROL STRING

The debug control string is used to set the state of the debugger
via  the  macro.  This section summarizes the currently available
debugger options and the flag characters which enable or  disable
them.   Argument lists enclosed in '[' and ']' are optional.  En-
able output from macros with specified keywords.  A null list  of
keywords  implies  that  all  keywords  are  selected.  Delay for
specified time after each output line, to let output drain.  Time
is  given in tenths of a second (value of 10 is one second).  De-
fault is zero.  Limit debugger actions to the specified  list  of
functions.   A  null list of functions implies that all functions
are selected.  Mark each debugger output line with  the  name  of
the source file containing the macro causing the output.  Turn on
machine independent profiling.  A profiling data collection file,
named  dbugmon.out,  will  be  written  for postprocessing by the
"analyze" program.  The accuracy of this  feature  is  relatively
unknown  at  this  time.  Mark each debugger output line with the
source file line number of the macro causing  the  output.   Mark
each  debugger  output  line  with  the  current function nesting
depth.  Sequentially number each debugger output line starting at
1.  This is useful for reference purposes when debugger output is
interspersed with program output.  Redirect the  debugger  output
stream  to  the  specified  file.   The  default output stream is
stderr.  A null argument list causes output to be  redirected  to
stdout.   Limit  debugger  actions to the specified processes.  A
null list implies all processes.  This is  useful  for  processes
which  run  child processes.  Note that each debugger output line
can be marked with the name of the current process  via  the  'P'
flag.   The  process  name  must match the argument passed to the
macro.  Mark each debugger output  line  with  the  name  of  the
current process.  Most useful when used with a process which runs
child processes that are also  being  debugged.   Note  that  the
parent process must arrange for the debugger control string to be
passed to the child processes.  Used in conjunction with the mac-
ro  to  reset  the  current indentation level back to zero.  Most
useful with macros used to temporarily alter the debugger  state.
Enable  function control flow tracing.  The maximum nesting depth
is specified by N, and defaults to 200.  HINTS AND MISCELLANEOUS

One of the most useful capabilities of the package is to  compare
the  executions of a given program in two different environments.
This is typically done by executing the program in  the  environ-
ment  where it behaves properly and saving the debugger output in
a reference file.  The program is then run with identical  inputs
in  the  environment  where it misbehaves and the output is again
captured in a reference file.  The two reference files  can  then
be  differentially  compared to determine exactly where execution
of the two processes diverges.

A related usage is regression testing where the  execution  of  a
current  version  is compared against executions of previous ver-
sions.  This is most useful when there are only minor changes.

It is not difficult to modify an existing compiler  to  implement
some  of  the functionality of the package automatically, without
source code changes to the program being debugged.  In fact, such
changes  were implemented in a version of the Portable C Compiler
by the author in less than a day.  However, it  is  strongly  en-
couraged  that  all  newly developed code continue to use the de-
bugger macros for the portability  reasons  noted  earlier.   The
modified  compiler  should be used only for testing existing pro-
grams.

CAVEATS

The package works best with programs which  have  "line oriented"
output,  such as text processors, general purpose utilities, etc.
It can be interfaced with screen oriented programs such as visual
editors  by  redefining  the  appropriate  macros to call special
functions for displaying the debugger results.  Of  course,  this
caveat  is not applicable if the debugger output is simply dumped
into a file for post-execution examination.

Programs which use memory allocation functions  other  than  will
usually  have problems using the standard package.  The most com-
mon problem is multiply allocated memory.













