/*
** RasterPBM.c
**	Import module for importing PBM, PGM or PPM format images into Xew.
**
** Copyright 1994 by Markku Savela and
**	Technical Research Centre of Finland
*/
#include <stdio.h>
#if SYSV_INCLUDES
#	include <memory.h>
#	include <malloc.h>
#else
#if ANSI_INCLUDES
#	include <stddef.h>
#	include <stdlib.h>
#else
char *malloc();
void free();
#endif
#endif
#include <string.h>
#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xew/RasterP.h>
#include "ImageTools.h"
#include "RasterPBM.h"

/*
** Mapping from PBM format ID to Xew internal image class
*/
typedef struct FormatMapping
    {
	XeImageClass class;
	int raw;	/* Non-zero, if RAW PBM (non-ASCII) format */
	int channels;	/* Number of channels */
    } FormatMapping;

static FormatMapping format[] = /* Index by numeric value after 'P' */
    {
	{ XeImageClass_UNKNOWN,1,1 },	/* P0 not used */
	{ XeImageClass_BILEVEL,0,1 },	/* P1 = Ascii bitmap */
	{ XeImageClass_GRAYSCALE,0,1 },	/* P2 = Ascii greymap */
	{ XeImageClass_FULLCOLOR,0,3 },	/* P3 = Ascii pixmap */
	{ XeImageClass_BILEVEL,1,1 },	/* P4 = Raw bitmap */
	{ XeImageClass_GRAYSCALE,1,1 },	/* P5 = Raw greymap */
	{ XeImageClass_FULLCOLOR,1,3 },	/* P6 = Raw pixmap */
    };

typedef struct ImageLoad
    {
	XeDataContent content;	/* Input data descriptor */
	int length, width, height, maxv;
	FormatMapping fmt;	/* Input format specification */
	unsigned char *p, *end;	/* Current input data to parse */
	XeRawImage *raw;	/* The RAW image structure being filled */
	unsigned char buffer[4000];
    } ImageLoad;

#if NeedFunctionPrototypes
static unsigned char *get_more_input(ImageLoad *);
static int getbyte(ImageLoad *);
static int getint(ImageLoad *);
static int getbit(ImageLoad *);
static unsigned char *get_raw_data(ImageLoad *, int *);
static void loadpbm(ImageLoad *);
static void loadpgm(ImageLoad *);
static void loadppm(ImageLoad *);
#endif


static void RasterImportWarning(w, msg)
XeRasterWidget w;
char *msg;
    {
	XeWidgetWarningMsg
		((Widget)w, "rasterImportWarning", msg, (String *)NULL, 0);
    }

#define ENDFILE(s,image) \
	((s) == (image)->end && (s = get_more_input(image)) == (image)->end)

/*
** get_more_input
**	must only be called, if current buffer is empty
**	(e.g. image->p == image->end).
*/
static unsigned char *get_more_input(image)
ImageLoad *image;
    {
	int size;

	if (image->content.type == XeDataContentType_STRING)
		return image->end;
	size = fread(image->buffer, 1, sizeof(image->buffer),
		     image->content.source.stream);
	image->end = image->p = image->buffer;
	if (size > 0)
		image->end += size;
	return image->p;
    }

/*
** getbyte
**	a single byte from the input stream
*/
static int getbyte(image)
ImageLoad *image;
    {
	unsigned char *s = image->p;

	if (ENDFILE(s,image))
		return 0;
	image->p++;
	return *s;
    }

static int getint(image)
ImageLoad *image;
    {
	int c, n;
	unsigned char *s = image->p;
	/*
	** Just skip all non-numeric data until one of '0'..'9'
	** or end of content is reached.
	*/
	while (1)
	    {
		if (ENDFILE(s,image))
		    {
			c = '0';  /* Fake '0' forever at end */
			break;
		    }
		else if ((c = *s++) == '#')
			while (!ENDFILE(s,image) && (c = *s++) != '\n');
		else if (c >= '0' && c <= '9') 
			break;
	    }
	n = 0;
	do
		n = n * 10 + c - '0';
	while (!ENDFILE(s,image) && (c = *s++) >= '0' && c <= '9');
	image->p = s;
	return n;
    }

static int getbit(image)
ImageLoad *image;
    {
	int c;
	unsigned char *s = image->p;
	/*
	** Just skip all non-numeric data until one of '0'..'9'
	** or end of content is reached.
	*/
	while (1)
	    {
		if (ENDFILE(s,image))
		    {
			c = '0';  /* Fake '0' forever at end */
			break;
		    }
		else if ((c = *s++) == '#')
			while (!ENDFILE(s,image) && (c = *s++) != '\n');
		else if (c >= '0' && c <= '9') 
			break;
	    }
	image->p = s;
	return c != '0';
    }

/*
** get_raw_data
**	assumes that all remaining input bytes are binary image data to
**	be copied as is into the XeRawImage.
**
**	Because, currently every use of this function is followed by
**	some transformation of the data, this function only returns a
**	pointer to the source data (which already *may* be in the image
**	structure, but may not (if STRING content).
*/
static unsigned char *get_raw_data(image, size)
ImageLoad *image;
int *size;
    {
	unsigned char *s;
	int n = image->end - image->p;	/* Buffered bytes */
	int space = *size;		/* Available space */
	
	if (image->content.type != XeDataContentType_STRING)
	    {
		s = image->raw->data;
		/*
		** First, move any possible buffered input
		*/
		if (n > space)
			n = space;	/* content is too long! */
		if (n > 0)
		    {
			memcpy(s, image->p, n);
			s += n;
			space -= n;
		    }
		(void)fread((char *)s, 1, space, image->content.source.stream);
		/* ... ignore read errors, image will be random data,
		   but if the input file is corrupt, there is not much
		   else that can be done at this point. --msa */
		s = image->raw->data;
	    }
	else
	    {
		if (space > n)
			*size = n;	/* content is too short! */
		s = image->p;
	    }
	return s;
    }

static void loadpbm(image)
ImageLoad *image;
    {
	unsigned char *row, *ptr;
	int i, j, size;

	image->raw->samples_per_pixel = 1;
	image->raw->bits_per_sample = 1;
	image->raw->bits_per_component = 1;
	image->raw->bytes_per_line = (image->raw->width + 7) / 8;
	size = image->raw->height * image->raw->bytes_per_line;
	if ((image->raw->data = (unsigned char *)malloc(size)) == NULL)
		return;
	image->raw->alloc_data = True;
	row = image->raw->data;
	if (!image->fmt.raw)
	    {
		int pix, m;

		for (i = 0; i < image->raw->height;
		     i++, row += image->raw->bytes_per_line)
		    {
			for (pix=j=0, ptr=row, m=0x80; j < image->raw->width;
			     j++)
			    {
				if (!getbit(image))
					pix |= m;
				if ((m >>= 1) == 0)
				    {
					*ptr++ = pix;
					m = 0x80;
					pix = 0;
				    }
			    }
			if (m)
				*ptr = pix;
		    }
	    }
	else
	    {
		ptr = get_raw_data(image, &size);
		for (i = 0; i < size; i++)
			row[i] = ~ ptr[i];
	    }
	image->raw->channel[0].addr = image->raw->data;
	image->raw->channel[0].line = image->raw->bytes_per_line;
	image->raw->channel[0].inc = 1;
	image->raw->channel[0].w = image->raw->width;
	image->raw->channel[0].h = image->raw->height;
    }


static void loadpgm(image)
ImageLoad *image;
    {
	int i, shift, size;
	unsigned char scale[256], *p, *s;
	
	image->raw->samples_per_pixel = 1;
	image->raw->bits_per_sample = 8;
	image->raw->bits_per_component = 8;
	image->raw->bytes_per_line = image->raw->width;

	/* if maxv>255, keep dropping bits until it's reasonable */
	shift = 0;
	while (image->maxv > 255)
	    {
		image->maxv = image->maxv >> 1;
		shift++;
	    }
	for (i = 0; i <= image->maxv; i++)
		scale[i] = (i * 255) / image->maxv;
	size = image->raw->height * image->raw->bytes_per_line;
	if ((image->raw->data = (unsigned char *)malloc(size)) == NULL)
		return;
	p = image->raw->data;
	image->raw->alloc_data = True;
	if (image->fmt.raw)
	    {
		s = get_raw_data(image, &size);
		for (i = 0; i < size; i++)
			p[i] = scale[s[i]];
	    }
	else
	    {
		for (i = 0; i < size; i++)
			p[i] = scale[(getint(image) >> shift) & 255];
	    }
	image->raw->channel[0].addr = p;
	image->raw->channel[0].line = image->raw->bytes_per_line;
	image->raw->channel[0].inc = 1;
	image->raw->channel[0].w = image->raw->width;
	image->raw->channel[0].h = image->raw->height;
    }

static void loadppm(image)
ImageLoad *image;
    {
	unsigned char scale[256], *p, *s;
	int i, shift, size;
	
	image->raw->samples_per_pixel = 3;
	image->raw->bits_per_sample = 8;
	image->raw->bits_per_component = 8;
	image->raw->bytes_per_line = 3 * image->raw->width;
	size = image->raw->height * image->raw->bytes_per_line;
	if ((image->raw->data = (unsigned char *)malloc(size)) == NULL)
		return;
	p = image->raw->data;
	image->raw->alloc_data = True;
	/* if maxv>255, keep dropping bits until it's reasonable */
	shift = 0;
	while (image->maxv > 255)
	    {
		image->maxv = image->maxv >> 1;
		shift++;
	    }
	for (i = 0; i <= image->maxv; i++)
		scale[i] = (i * 255) / image->maxv;
	if (image->fmt.raw)
	    {
		s = get_raw_data(image, &size);
		for (i = 0; i < size; i++)
			p[i] = scale[s[i]];
	    }
	else
	    {
		for (i = 0; i < size; i++)
			p[i] = scale[(getint(image) >> shift) & 255];
	    }
	for (i = 0; i < 3; i++)
	    {
		image->raw->channel[i].addr = p + i;
		image->raw->channel[i].w = image->raw->width;
		image->raw->channel[i].h = image->raw->height;
		image->raw->channel[i].inc = 3;
		image->raw->channel[i].line = image->raw->bytes_per_line;
	    }
    }

/*
** XeImport_PBM
*/
XeRawImage *XeImport_PBM(w)
XeRasterWidget w;
    {
	ImageLoad image;
	int c1, c2;

	image.raw = NULL;
	_XeOpenContent((XeBasicWidget)w, &image.content);
	if (image.content.source.string == NULL)
	    {
		RasterImportWarning(w, "Empty content");
		goto import_failed;
	    }
	if (image.content.type == XeDataContentType_STRING)
	    {
		image.p = image.content.source.string;
		image.end = image.p + image.content.length;
	    }
	else
		image.p = image.end = image.buffer;
	c1 = getbyte(&image);
	c2 = getbyte(&image);
	if (c1 != 'P' || c2 < '1' || c2 > '6')
	    {
		RasterImportWarning(w, "Not a known PBM format");
		goto import_failed;
	    }
	image.fmt = format[c2 - '0'];
	/*
	** read in header information
	*/
	image.width = getint(&image);
	image.height = getint(&image);
	/*
	** if we're not reading a bitmap, read the 'max value'
	*/
	if (image.fmt.class != XeImageClass_BILEVEL)
		image.maxv = getint(&image);
	else
		image.maxv = 1;
	if (image.width < 1 || image.height < 1 || image.maxv < 1)
	    {
		RasterImportWarning(w, "Bad PBM header information");
		goto import_failed;
	    }
	image.raw = _XeCreateRawImage(image.fmt.channels);
	image.raw->class = image.fmt.class;
	image.raw->width = image.width;
	image.raw->height = image.height;
	switch (image.fmt.class)
	    {
	    default:
	    case XeImageClass_BILEVEL:
		loadpbm(&image);
		break;
	    case XeImageClass_GRAYSCALE:
		loadpgm(&image);
		break;
	    case XeImageClass_FULLCOLOR:
		loadppm(&image);
		break;
	    }
	if (image.raw->data == NULL)
	    {
		RasterImportWarning(w, "Out of memory at PBM import");
		_XeDestroyRawImage(image.raw);
		image.raw = NULL;
	    }
    import_failed:
	_XeCloseContent(&image.content);
	return image.raw;
    }  
