/*
 * Copyright(c) 1992 Bell Communications Research, Inc. (Bellcore)
 *                        All rights reserved
 * Permission to use, copy, modify and distribute this material for
 * any purpose and without fee is hereby granted, provided that the
 * above copyright notice and this permission notice appear in all
 * copies, and that the name of Bellcore not be used in advertising
 * or publicity pertaining to this material without the specific,
 * prior written permission of an authorized representative of
 * Bellcore.
 *
 * BELLCORE MAKES NO REPRESENTATIONS AND EXTENDS NO WARRANTIES, EX-
 * PRESS OR IMPLIED, WITH RESPECT TO THE SOFTWARE, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR ANY PARTICULAR PURPOSE, AND THE WARRANTY AGAINST IN-
 * FRINGEMENT OF PATENTS OR OTHER INTELLECTUAL PROPERTY RIGHTS.  THE
 * SOFTWARE IS PROVIDED "AS IS", AND IN NO EVENT SHALL BELLCORE OR
 * ANY OF ITS AFFILIATES BE LIABLE FOR ANY DAMAGES, INCLUDING ANY
 * LOST PROFITS OR OTHER INCIDENTAL OR CONSEQUENTIAL DAMAGES RELAT-
 * ING TO THE SOFTWARE.
 *
 */

/*
 * Actions.c created by Andrew Lister (7 August, 1995)
 */

#include <Xm/Xm.h>
#include <Xm/XmP.h>
#if XmVersion > 1001
#include <Xm/DrawP.h>
#endif
#include <Xbae/MatrixP.h>
#include <Xbae/Clip.h>
#include <Xbae/Draw.h>
#include <Xbae/Actions.h>
#include <Xbae/Utils.h>
#include <Xbae/ScrollMgr.h>
#include <X11/cursorfont.h>

#ifndef XlibSpecificationRelease
#define XrmPermStringToQuark XrmStringToQuark
#endif

#if !defined(DRAW_RESIZE_LINE) && !defined(DRAW_RESIZE_SHADOW)
/* One of DRAW_RESIZE_LINE and DRAW_RESIZE_SHADOW must be defined. */
#define DRAW_RESIZE_SHADOW
#endif
    
typedef struct {
    XbaeMatrixWidget mw;
    GC gc;
    GC clip_gc;
    int column;
    int startx;
    int lastx;
    int currentx;
    int y, height;
    short *columnWidths;
    Boolean grabbed;
    Boolean haveVSB;
} XbaeMatrixResizeColumnStruct;

static int DoubleClick P(( XbaeMatrixWidget mw, XEvent *event, int row,
			   int column ));
static void DrawSlideColumn P(( XbaeMatrixWidget, int x ));
static void SlideColumn P(( Widget w, XbaeMatrixResizeColumnStruct *resizeData,
			    XEvent *event, Boolean *cont ));


/* ARGSUSED */
void
xbaeDefaultActionACT(w, event, params, nparams)
Widget w;
XEvent *event;
String *params;
Cardinal *nparams;
{
    XbaeMatrixWidget mw;
    int x, y;
    int row, column;
    CellType cell;

    /*
     * Get Matrix widget and make sure it is a Matrix subclass.
     * w could be Matrix, or the Clip or textField children of Matrix
     */
    if (XtIsSubclass(w, xbaeMatrixWidgetClass))
	mw = (XbaeMatrixWidget) w;
    else if (XtIsSubclass(XtParent(w), xbaeMatrixWidgetClass))
	mw = (XbaeMatrixWidget) XtParent(w);
    else {
	XtAppWarningMsg(
	    XtWidgetToApplicationContext(w),
	    "defaultActionACT", "badWidget", "XbaeMatrix",
	    "XbaeMatrix: Bad widget passed to DefaultAction action",
	    NULL, 0);
	return;
    }

    if( !mw->matrix.default_action_callback )
	return;

    if (!xbaeEventToXY(mw, event, &x, &y, &cell))
	return;

    if (!xbaeXYToRowCol(mw, &x, &y, &row, &column, cell))
	return;

    if( DoubleClick( mw, event, row, column ) )
    {
	XbaeMatrixDefaultActionCallbackStruct call_data;

	call_data.reason = XbaeDefaultActionReason;
	call_data.row = row;
	call_data.column = column;
	call_data.event = event;

	XtCallCallbackList((Widget)mw, mw->matrix.default_action_callback,
			   (XtPointer) &call_data);
      
    }
}

static void
DrawSlideColumn( mw, x )
XbaeMatrixWidget mw;
int x;
{
#ifdef DRAW_RESIZE_SHADOW
    /* These values derived through that age-old process
     * of what looks good to me */
#define SHADOW_WIDTH 2
#define RESIZE_COLUMN_LINE_WIDTH 4
    Dimension width = RESIZE_COLUMN_LINE_WIDTH;
    Dimension shadow_width = SHADOW_WIDTH;
#endif
    Dimension height;
    Window win;
    Display *display = XtDisplay( mw );
    int column = xbaeXtoCol( mw, x - COLUMN_LABEL_OFFSET( mw ) );
    int top, bottom;
    int adjusted_x;
    int y;
#ifdef DRAW_RESIZE_LINE
    GC clipgc = mw->matrix.draw_clip_gc;
    GC gc = mw->matrix.draw_gc;
#endif

    /*
     * If the column being resized is a fixed one then we don't need to
     * bother with the clip region
     */
    if( column < mw->matrix.fixed_columns )
    {
	y = ROW_LABEL_OFFSET(mw);
	height = VISIBLE_HEIGHT(mw) + FIXED_ROW_HEIGHT(mw) +
	    TRAILING_FIXED_ROW_HEIGHT(mw);
	win = XtWindow( mw );

#ifdef DRAW_RESIZE_LINE
	XDrawLine( display, win, gc, x, y, x, y + height );
#endif
#ifdef DRAW_RESIZE_SHADOW
	DRAW_SHADOW(display, win,
		    mw->matrix.resize_top_shadow_gc,
		    mw->matrix.resize_bottom_shadow_gc,
		    shadow_width, x, y, width, height, XmSHADOW_OUT );
#endif
	return;
    }

    /*
     * Similarly for trailingFixedColumns - beware going off the clip child
     * here also
     */
    if( column >= TRAILING_HORIZ_ORIGIN(mw) ||
	x >= ClipChild(mw)->core.x + ClipChild(mw)->core.width )
    {
	y = ROW_LABEL_OFFSET(mw);
	height = VISIBLE_HEIGHT(mw) + FIXED_ROW_HEIGHT(mw) +
	    TRAILING_FIXED_ROW_HEIGHT(mw);
	win = XtWindow( mw );

#ifdef DRAW_RESIZE_LINE
	XDrawLine( display, win, gc, x, y, x, y + height );
#endif
#ifdef DRAW_RESIZE_SHADOW
	DRAW_SHADOW(display, win,
		    mw->matrix.resize_top_shadow_gc,
		    mw->matrix.resize_bottom_shadow_gc,
		    shadow_width, x, y, width, height, XmSHADOW_OUT );
#endif
	return;
    }
    xbaeGetVisibleRows( mw, &top, &bottom );
    /*
     * we need all non-fixed rows, so add 1 to bottom
     * to include the last one as the return values
     * are inclusive
     */
    bottom += 1;

    /*
     * The area between top and bottom rows are the non fixed rows.  They
     * fall on the ClipChild
     */
    y = 0; /* relative to clip */

    height = ROW_HEIGHT( mw ) * ( bottom - top );

    /*
     * If we are on the clip, the x location is offset by the
     * fixed column width and label offset
     */
    adjusted_x = x - FIXED_COLUMN_LABEL_OFFSET(mw);

    win = XtWindow( ClipChild(mw) );

#ifdef DRAW_RESIZE_LINE
    XDrawLine( display, win, gc, adjusted_x, y, adjusted_x, y + height );
#endif
#ifdef DRAW_RESIZE_SHADOW
    DRAW_SHADOW(display, win,
		mw->matrix.resize_top_shadow_gc,
		mw->matrix.resize_bottom_shadow_gc,
		shadow_width, adjusted_x, y, width, height, XmSHADOW_OUT );
#endif
    /*
     * Now draw the line (or shadow) on the non clipped region - that is
     * the fixed and trailingFixed rows.  First, do the leading rows.
     */

    y = ROW_LABEL_OFFSET(mw);
    height = FIXED_ROW_HEIGHT( mw );
    win = XtWindow( mw );

#ifdef DRAW_RESIZE_LINE
    XDrawLine( display, win, clipgc, x, y, x, y + height );
#endif
#ifdef DRAW_RESIZE_SHADOW
    DRAW_SHADOW(display, win,
		mw->matrix.cell_top_shadow_clip_gc,
		mw->matrix.cell_bottom_shadow_clip_gc,
		shadow_width, x, y, width, height, XmSHADOW_OUT );
#endif
    /*
     * The trailingFixedRows
     */
    if( mw->matrix.trailing_fixed_rows )
    {
	y = TRAILING_FIXED_ROW_LABEL_OFFSET(mw);
	height = TRAILING_FIXED_ROW_HEIGHT(mw);
	xbaeSetClipMask(mw, CLIP_TRAILING_FIXED_ROWS);

#ifdef DRAW_RESIZE_LINE
	XDrawLine( display, win, clipgc, x, y, x, y + height );
#endif
#ifdef DRAW_RESIZE_SHADOW
	DRAW_SHADOW(display, win,
		    mw->matrix.cell_top_shadow_clip_gc,
		    mw->matrix.cell_bottom_shadow_clip_gc,
		    shadow_width, x, y, width, height, XmSHADOW_OUT );
#endif
	xbaeSetClipMask( mw, CLIP_NONE );			 
    }
}


static void
SlideColumn( w, rd, event, cont )
Widget w;
XbaeMatrixResizeColumnStruct *rd;
XEvent *event;
Boolean *cont;
{
    XMotionEvent *motionEvent;
    Boolean relayout = False;
    int numCharacters;
    int i;
    
    if( event->type == ButtonRelease )
    {
	DrawSlideColumn( rd->mw, rd->lastx );
	XUngrabPointer( XtDisplay( w ), CurrentTime );
	rd->grabbed = False;
	/*
	 * Remanage the VSB if we unmapped it earlier
	 */
	if( rd->haveVSB )
	    XtManageChild( VertScrollChild( rd->mw ) );

	if (rd->mw->matrix.resize_column_callback != NULL)
	{
	    XbaeMatrixResizeColumnCallbackStruct call_data;
	    
	    call_data.reason = XbaeResizeColumnReason;
	    call_data.which = rd->column - 1;
	    call_data.columns = rd->mw->matrix.columns;
	    call_data.column_widths  = rd->columnWidths;
	    call_data.event = event;
	    XtCallCallbackList ((Widget)rd->mw,
				rd->mw->matrix.resize_column_callback,
				(XtPointer) &call_data);	    
	}

	for( i = 0; i < rd->mw->matrix.columns; i++ )
	    if( rd->columnWidths[ i ] != rd->mw->matrix.column_widths[ i ] )
	    {
		/* Make sure everything is handled correctly with SetValues */
		XtVaSetValues( ( Widget )rd->mw, XmNcolumnWidths,
			       rd->columnWidths, NULL );
		break;
	    }
	/*
	 * If maxColumnLengths are set and we have resized the column to
	 * larger, reset the corresponding maxColumnLength
	 */
	if( rd->mw->matrix.column_max_lengths &&
	    rd->columnWidths[ rd->column - 1 ] >
	    rd->mw->matrix.column_max_lengths[ rd->column - 1 ] )
	    rd->mw->matrix.column_max_lengths[ rd->column - 1 ] =
		rd->columnWidths[ rd->column - 1 ];
	XtFree( ( char * )rd->columnWidths );
	return;
    }

    if( event->type != MotionNotify ) /* Double check! */
	return;

    motionEvent = ( XMotionEvent * )event;

    if( rd->currentx - motionEvent->x > FONT_WIDTH( rd->mw ) )
    {
	/* If we're only one character wide, we cannae get any smaller */
	if (rd->columnWidths[ rd->column - 1 ] == 1)
	    return;	
	/*
	 * Moved left a full character - update the column widths and force
	 * a redisplay
	 */	
	numCharacters = ( rd->currentx - motionEvent->x ) /
	    FONT_WIDTH( rd->mw );
	if( numCharacters >= rd->columnWidths[ rd->column - 1 ] )
	    /* Must keep a column at least one character wide */
	    numCharacters = rd->columnWidths[ rd->column - 1 ] - 1;
		
	rd->columnWidths[ rd->column - 1 ] -= numCharacters;
	rd->currentx -= numCharacters * FONT_WIDTH( rd->mw );
	relayout = True;
    }	
    
    if( motionEvent->x - rd->currentx > FONT_WIDTH( rd->mw ) )
    {
	/*
	 * Moved right a full character - update the column widths and force
	 * a redisplay
	 */
	numCharacters = ( motionEvent->x - rd->currentx ) /
	    FONT_WIDTH( rd->mw );
	rd->columnWidths[ rd->column - 1 ] += numCharacters;
	rd->currentx += numCharacters * FONT_WIDTH( rd->mw );
	relayout = True;
    }

    if( relayout )
    {
	/* Draw the marker line in the new location */
	if( rd->lastx != rd->currentx )
	{
	    DrawSlideColumn( rd->mw, rd->currentx );
	    DrawSlideColumn( rd->mw, rd->lastx );

	    rd->lastx = rd->currentx;
	}
    }
}

/* ARGSUSED */
void
xbaeResizeColumnsACT(w, event, params, nparams)
Widget w;
XEvent *event;
String *params;
Cardinal *nparams;
{
    XbaeMatrixWidget mw;
    int x, y;
    int eventx;
    int i;
    int row, column;
    CellType cell;
    static Cursor cursor;
    XbaeMatrixResizeColumnStruct resizeData;
    XGCValues values;
#ifdef DRAW_RESIZE_LINE
    XGCValues save;
#endif
    unsigned long gcmask, event_mask;
    Display *display = XtDisplay(w);
#define FUZZ_FACTOR	3
    int	fuzzy = FUZZ_FACTOR;
#undef FUZZ_FACTOR
    
    /*
     * Get Matrix widget and make sure it is a Matrix subclass.
     * w could be Matrix, or the Clip or textField children of Matrix
     */
    if (XtIsSubclass(w, xbaeMatrixWidgetClass))
	mw = (XbaeMatrixWidget) w;
    else if (XtIsSubclass(XtParent(w), xbaeMatrixWidgetClass))
	mw = (XbaeMatrixWidget) XtParent(w);
    else {
	XtAppWarningMsg(
	    XtWidgetToApplicationContext(w),
	    "resizeColumnsACT", "badWidget", "XbaeMatrix",
	    "XbaeMatrix: Bad widget passed to ResizeColumns action",
	    NULL, 0);
	return;
    }
    
    /*
     * If we won't allow dynamic column resize, leave.
     */
    if (!mw->matrix.allow_column_resize)
	return;
    
    if (!xbaeEventToXY(mw, event, &x, &y, &cell))
	return;
    
    eventx = x;
    
    if (!xbaeXYToRowCol(mw, &x, &y, &row, &column, cell))
	return;

    /*
     * Calculate if the x and y of the middle button event is on
     * a column border.  Allow the width of the shadow to be the
     * allowed delta.  x is modified in xbaeXYToRowCol() to be
     * the x distance from the cell's border
     */
    if (mw->matrix.cell_shadow_thickness > fuzzy )
	fuzzy = mw->matrix.cell_shadow_thickness;
    
    if ( x > fuzzy && COLUMN_WIDTH(mw, column) - x > fuzzy )
	return;
    
    /*
     * Looks like we hit a column border, determine the column that is
     * intended to be resized
     */
    if( (COLUMN_WIDTH(mw, column) - x) <= fuzzy )
	column++;
    
    /* Can't adjust the origin or should you be able to?? */
    if( column == 0 )
	return;
    
    /*
     * Make it here and it's time to start the fun stuff!
     */
    
    /* Create the left / right cursor */
    if( !cursor )
	cursor = XCreateFontCursor( display, XC_sb_h_double_arrow );
    
    /* Commit any edit in progress and unmap the text field -
       it's just bad luck */
    (*((XbaeMatrixWidgetClass) XtClass(mw))->matrix_class.commit_edit)
	(mw, True);

    /*
     * Say goodbye to the Vertical ScrollBar -> it only gets in the way!
     */
    if( ( resizeData.haveVSB = XtIsManaged( VertScrollChild( mw ) ) &&
	  ( (mw->matrix.scrollbar_placement == XmTOP_RIGHT ) ||
	    (mw->matrix.scrollbar_placement == XmBOTTOM_RIGHT ) ) ) )
        XtUnmanageChild( VertScrollChild( mw ) );
    /*
     * Flush the commit events out to the server.
     * Otherwise, our changes to the GCs below have
     * a bad effect.
     */
    XSync(display, True);
    
    event_mask = PointerMotionMask | ButtonReleaseMask;
    XtAddEventHandler( w, event_mask,
		       True, ( XtEventHandler )SlideColumn,
		       ( XtPointer )&resizeData );
    
    XGrabPointer( display, XtWindow(w), True, event_mask,
		  GrabModeAsync, GrabModeAsync, XtWindow( ( Widget )mw ),
		  cursor, CurrentTime );
    
    /* Copy the columnWidth array */
    resizeData.columnWidths =
	( short * )XtMalloc( mw->matrix.columns * sizeof( short ) );
    for( i = 0; i < mw->matrix.columns; i++ )
	resizeData.columnWidths[ i ] = mw->matrix.column_widths[ i ];
    resizeData.grabbed = True;
    resizeData.mw = mw;
    resizeData.column = column;
    resizeData.startx = resizeData.currentx = resizeData.lastx =
	event->xbutton.x;

    gcmask = GCForeground | GCBackground | GCFunction;
    values.function = GXxor;
#ifdef DRAW_RESIZE_LINE
    XGetGCValues( display, mw->matrix.draw_gc, gcmask, &save );
    values.foreground = values.background = save.background;

    XChangeGC(display, mw->matrix.draw_gc, gcmask, &values);
    XChangeGC(display, mw->matrix.draw_clip_gc, gcmask, &values);
#endif
#ifdef DRAW_RESIZE_SHADOW
    XSetFunction(display, mw->matrix.cell_top_shadow_clip_gc, GXxor);
    XSetFunction(display, mw->matrix.cell_bottom_shadow_clip_gc, GXxor);
#endif
    
    DrawSlideColumn( mw, resizeData.currentx );
    
    while( resizeData.grabbed )
	XtAppProcessEvent( XtWidgetToApplicationContext( w ), XtIMAll );
    
    XtRemoveEventHandler( w, event_mask, True,
			  ( XtEventHandler )SlideColumn,
			  ( XtPointer )&resizeData );

#ifdef DRAW_RESIZE_LINE
    XSetFunction(display, mw->matrix.draw_gc, GXcopy);
    XSetFunction(display, mw->matrix.draw_clip_gc, GXcopy);
#endif
#ifdef DRAW_RESIZE_SHADOW
    XSetFunction(display, mw->matrix.cell_top_shadow_clip_gc, GXcopy);
    XSetFunction(display, mw->matrix.cell_bottom_shadow_clip_gc, GXcopy);
#endif

}

/*
 * Action to process a drag out
 */
/* ARGSUSED */
void
xbaeProcessDragACT(w, event, params, nparams)
Widget w;
XEvent *event;
String *params;
Cardinal *nparams;
{
    XbaeMatrixWidget mw;
    int x, y;
    int row, column;
    CellType cell;
    XbaeMatrixProcessDragCallbackStruct call_data;

    /*
     * Get Matrix widget and make sure it is a Matrix subclass.
     * w could be Matrix, or the Clip or textField children of Matrix
     */
    if (XtIsSubclass(w, xbaeMatrixWidgetClass))
	mw = (XbaeMatrixWidget) w;
    else if (XtIsSubclass(XtParent(w), xbaeMatrixWidgetClass))
	mw = (XbaeMatrixWidget) XtParent(w);
    else {
	XtAppWarningMsg(
	    XtWidgetToApplicationContext(w),
	    "processDragACT", "badWidget", "XbaeMatrix",
	    "XbaeMatrix: Bad widget passed to ProcessDrag action",
	    NULL, 0);
	return;
    }

    if( !mw->matrix.process_drag_callback )
	return;

    if (!xbaeEventToXY(mw, event, &x, &y, &cell))
	return;

    if (!xbaeXYToRowCol(mw, &x, &y, &row, &column, cell))
	return;

    call_data.reason = XbaeProcessDragReason;
    call_data.row = row;
    call_data.column = column;
    if (mw->matrix.draw_cell_callback)
    {
	Pixel bgcolor, fgcolor;
	int width, height, depth;
	
	call_data.type = xbaeGetDrawCellValue(
	    mw, row, column, &call_data.string, &call_data.pixmap,
	    &call_data.mask, &width, &height, &bgcolor, &fgcolor, &depth );
    }
    else
	call_data.string = mw->matrix.cells ?
	    mw->matrix.cells[row][column] : "";
    
    call_data.num_params = *nparams;
    call_data.params = params;
    call_data.event = event;

    XtCallCallbackList((Widget)mw, mw->matrix.process_drag_callback,
		       (XtPointer) &call_data);
}


/*
 * Action to edit a non-fixed cell.
 */
/* ARGSUSED */
void
xbaeEditCellACT(w, event, params, nparams)
Widget w;
XEvent *event;
String *params;
Cardinal *nparams;
{
    XbaeMatrixWidget mw;
    int row, column;
    XrmQuark q;
    static XrmQuark QPointer, QLeft, QRight, QUp, QDown;
    static Boolean haveQuarks = False;

    /*
     * Get static quarks for the parms we understand
     */
    if (!haveQuarks)
    {
	QPointer = XrmPermStringToQuark("Pointer");
	QLeft = XrmPermStringToQuark("Left");
	QRight = XrmPermStringToQuark("Right");
	QUp = XrmPermStringToQuark("Up");
	QDown = XrmPermStringToQuark("Down");
	haveQuarks = True;
    }

    /*
     * Get Matrix widget and make sure it is a Matrix subclass.
     * w could be Matrix, or the Clip or textField children of Matrix
     */
    if (XtIsSubclass(w, xbaeMatrixWidgetClass))
	mw = (XbaeMatrixWidget) w;
    else if (XtIsSubclass(XtParent(w), xbaeMatrixWidgetClass))
	mw = (XbaeMatrixWidget) XtParent(w);
    else
    {
	XtAppWarningMsg(
	    XtWidgetToApplicationContext(w),
	    "editCellACT", "badWidget", "XbaeMatrix",
	    "XbaeMatrix: Bad widget passed to EditCell action",
	    NULL, 0);
	return;
    }

    /*
     * Make sure we have a single parm
     */
    if (*nparams != 1)
    {
	XtAppWarningMsg(
	    XtWidgetToApplicationContext(w),
	    "editCellACT", "badParms", "XbaeMatrix",
	    "XbaeMatrix: Wrong number of parameters passed to EditCell action, needs 1",
	    NULL, 0);
	return;
    }

    /*
     * Initialize row/column to the current position
     */
    row = mw->matrix.current_row;
    column = mw->matrix.current_column;

    /*
     * Quarkify the string param
     */
    q = XrmStringToQuark(params[0]);

    /*
     * If we aren't currently editing, then the only kind of traversal that
     * makes sense is pointer.
     */
    if (!XtIsManaged(TextChild(mw)) && q != QPointer)
	return;

    if (q == QPointer)
    {
	int x, y;

	/*
	 * Get the x,y point coordinate relative to the Clip window.
	 * Return if this event did not occur in the Clip subwindow
	 * (since we can only edit non-fixed cells).
	 */
	switch (event->type)
	{
	case ButtonPress:
	case ButtonRelease:
	    x = event->xbutton.x;
	    y = event->xbutton.y;
	    break;
	case KeyPress:
	case KeyRelease:
	    x = event->xkey.x;
	    y = event->xkey.y;
	    break;
	case MotionNotify:
	    x = event->xmotion.x;
	    y = event->xmotion.y;
	    break;
	default:
	    return;
	}

	if (event->xbutton.subwindow == XtWindow(ClipChild(mw)))
	{
	    x -= FIXED_COLUMN_LABEL_OFFSET(mw);
	    y -= FIXED_ROW_LABEL_OFFSET(mw);
	}
	else if (event->xbutton.window != XtWindow(ClipChild(mw)))
	    return;

	/*
	 * Convert the point to a row,column. If it does not pick a valid
	 * cell, then return.
	 */
	if (!xbaeXYToRowCol(mw, &x, &y, &row, &column, NonFixedCell))
	    return;
    }
    else if (q == QRight)
    {
	/*
	 * If we are in the lower right corner, stay there.
	 * Otherwise move over a column, if we move off the right edge,
	 * then move down a row and back to the first non-fixed column.
	 */
	if (mw->matrix.current_row != TRAILING_VERT_ORIGIN(mw) - 1 ||
	    mw->matrix.current_column != TRAILING_HORIZ_ORIGIN(mw) - 1)
	{

	    column++;

	    if (column >= TRAILING_HORIZ_ORIGIN(mw))
	    {
		column = mw->matrix.fixed_columns;
		row++;
	    }
	}
    }
    else if (q == QLeft)
    {
	/*
	 * If we are in the upper left corner, stay there.
	 * Otherwise move back a column, if we move before the fixed columns,
	 * then move up a row and over to the last column.
	 */
	if (mw->matrix.current_row != mw->matrix.fixed_rows ||
	    mw->matrix.current_column !=
	    mw->matrix.fixed_columns)
	{

	    column--;

	    if (column < (int) mw->matrix.fixed_columns)
	    {
		column = TRAILING_HORIZ_ORIGIN(mw) - 1;
		row--;
	    }
	}
    }
    else if (q == QDown)
    {
	row++;

	if (row >= TRAILING_VERT_ORIGIN(mw))
	    row = mw->matrix.fixed_rows;
    }
    else if (q == QUp)
    {
	row--;

	if (row < (int) mw->matrix.fixed_rows)
	    row = TRAILING_VERT_ORIGIN(mw) - 1;
    }

    /*
     * Call the traverseCellCallback to allow the application to
     * perform custom traversal.
     */
    if (mw->matrix.traverse_cell_callback)
    {
	XbaeMatrixTraverseCellCallbackStruct call_data;

	call_data.reason = XbaeTraverseCellReason;
	call_data.row = mw->matrix.current_row;
	call_data.column = mw->matrix.current_column;
	call_data.next_row = row;
	call_data.next_column = column;
	call_data.fixed_rows = mw->matrix.fixed_rows;
	call_data.fixed_columns = mw->matrix.fixed_columns;
	call_data.trailing_fixed_rows = mw->matrix.trailing_fixed_rows;
	call_data.trailing_fixed_columns = mw->matrix.trailing_fixed_columns;
	call_data.num_rows = mw->matrix.rows;
	call_data.num_columns = mw->matrix.columns;
	call_data.param = params[0];
	call_data.qparam = q;

	XtCallCallbackList((Widget) mw, mw->matrix.traverse_cell_callback,
			   (XtPointer) & call_data);

	row = call_data.next_row;
	column = call_data.next_column;
    }

    /*
     * Attempt to edit the new cell using the edit_cell method.
     * If we are editing a cell based on pointer position, we always
     * call edit_cell.	Otherwise, we must be editing a new cell to
     * call edit_cell.
     */
    if (q == QPointer || (row != mw->matrix.current_row ||
			  column != mw->matrix.current_column))
	(*((XbaeMatrixWidgetClass) XtClass(mw))->matrix_class.edit_cell)
	    (mw, row, column);

    /*
     * Traverse to the textField
     */
    ( void )XmProcessTraversal(TextChild(mw), XmTRAVERSE_CURRENT);
}

/*
 * Action to unmap the textField and discard any edits made
 */
/* ARGSUSED */
void
xbaeCancelEditACT(w, event, params, nparams)
Widget w;
XEvent *event;
String *params;
Cardinal *nparams;
{
    XbaeMatrixWidget mw;
    Boolean unmap;

    /*
     * Get Matrix widget and make sure it is a Matrix subclass.
     * w could be Matrix, or the Clip or textField children of Matrix
     */
    if (XtIsSubclass(w, xbaeMatrixWidgetClass))
	mw = (XbaeMatrixWidget) w;
    else if (XtIsSubclass(XtParent(w), xbaeMatrixWidgetClass))
	mw = (XbaeMatrixWidget) XtParent(w);
    else
    {
	XtAppWarningMsg(
	    XtWidgetToApplicationContext(w),
	    "cancelEditACT", "badWidget", "XbaeMatrix",
	    "XbaeMatrix: Bad widget passed to CancelEdit action",
	    NULL, 0);
	return;
    }

    /*
     * Make sure we have a single param
     */
    if (*nparams != 1)
    {
	XtAppWarningMsg(
	    XtWidgetToApplicationContext(w),
	    "cancelEditACT", "badParms", "XbaeMatrix",
	    "XbaeMatrix: Wrong number of parameters passed to CancelEdit action, needs 1",
	    NULL, 0);
	return;
    }

    /*
     * Validate our param
     */
    if (!strcmp(params[0], "True"))
	unmap = True;
    else if (!strcmp(params[0], "False"))
	unmap = False;
    else
    {
	XtAppWarningMsg(
	    XtWidgetToApplicationContext(w),
	    "cancelEditACT", "badParm", "XbaeMatrix",
	    "XbaeMatrix: Invalid parameter passed to CancelEdit action, must be True or False",
	    NULL, 0);
	return;
    }
    /*
     * Call the cancel_edit method
     */
    (*((XbaeMatrixWidgetClass) XtClass(mw))->matrix_class.cancel_edit)
	(mw, unmap);
}

/*
 * Action save any edits made and unmap the textField if params[0] is True
 */
/* ARGSUSED */
void
xbaeCommitEditACT(w, event, params, nparams)
Widget w;
XEvent *event;
String *params;
Cardinal *nparams;
{
    XbaeMatrixWidget mw;
    Boolean unmap;

    /*
     * Get Matrix widget and make sure it is a Matrix subclass.
     * w could be Matrix, or the Clip or textField children of Matrix
     */
    if (XtIsSubclass(w, xbaeMatrixWidgetClass))
	mw = (XbaeMatrixWidget) w;
    else if (XtIsSubclass(XtParent(w), xbaeMatrixWidgetClass))
	mw = (XbaeMatrixWidget) XtParent(w);
    else
    {
	XtAppWarningMsg(
	    XtWidgetToApplicationContext(w),
	    "commitEditACT", "badWidget", "XbaeMatrix",
	    "XbaeMatrix: Bad widget passed to CommitEdit action",
	    NULL, 0);
	return;
    }

    /*
     * Make sure we have a single param
     */
    if (*nparams != 1)
    {
	XtAppWarningMsg(
	    XtWidgetToApplicationContext(w),
	    "commitEditACT", "badParms", "XbaeMatrix",
	    "XbaeMatrix: Wrong number of parameters passed to CommitEdit action, needs 1",
	    NULL, 0);
	return;
    }

    /*
     * Validate our param
     */
    if (!strcmp(params[0], "True"))
	unmap = True;
    else if (!strcmp(params[0], "False"))
	unmap = False;
    else
    {
	XtAppWarningMsg(
	    XtWidgetToApplicationContext(w),
	    "commitEditACT", "badParm", "XbaeMatrix",
	    "XbaeMatrix: Invalid parameter passed to CommitEdit action, must be True or False",
	    NULL, 0);
	return;
    }

    (void) (*((XbaeMatrixWidgetClass) XtClass(mw))->matrix_class.commit_edit)
	(mw, unmap);
}

/*
 * Convert the coordinates in an event to be relative to the Clip
 * window or the Matrix window.  Set the cell to indicate which one.
 * Used by some actions.
 */
/* ARGSUSED */
Boolean
xbaeEventToXY(mw, event, x, y, cell)
XbaeMatrixWidget mw;
XEvent *event;
int *x, *y;
CellType *cell;
{
    switch (event->type)
    {
    case ButtonPress:
    case ButtonRelease:
	*x = event->xbutton.x;
	*y = event->xbutton.y;
	break;
    case KeyPress:
    case KeyRelease:
	*x = event->xkey.x;
	*y = event->xkey.y;
	break;
    case MotionNotify:
	*x = event->xmotion.x;
	*y = event->xmotion.y;
	break;
    default:
	return False;
    }

    if (event->xbutton.subwindow == XtWindow(ClipChild(mw)))
    {
	*cell = NonFixedCell;
	*x -= FIXED_COLUMN_LABEL_OFFSET(mw);
	*y -= FIXED_ROW_LABEL_OFFSET(mw);
    }
    else if (event->xbutton.window == XtWindow(mw))
	*cell = FixedCell;
    else if (event->xbutton.window == XtWindow(ClipChild(mw)))
	*cell = NonFixedCell;
    else if (event->xbutton.window == XtWindow(TextChild(mw)))
    {
	Position tx, ty;

	*cell = NonFixedCell;
	XtVaGetValues(TextChild(mw),
		      XmNx, &tx,
		      XmNy, &ty,
		      NULL);
	*x += tx;
	*y += ty;
    }
    else
	return False;

    return True;
}

static int
DoubleClick(mw, event, row, column)
XbaeMatrixWidget mw;
XEvent *event;
int row;
int column;
{
    /* A double click in this instance is two clicks in the
       same cell in a time period < double_click_interval */
    Time current_time;
    unsigned long delta;
    static int ret = 0;

    if (event->type == ButtonRelease)
    {
	/* If the button is released, store the current location and time -
	   next time through, if it's a button press event, we check for
	   double click */
	mw->matrix.last_row = row;
	mw->matrix.last_column = column;
	if( ret )		/* just had a double click */
	    mw->matrix.last_click_time = ( Time )0;
	else
	    mw->matrix.last_click_time = event->xbutton.time;
	ret = 0;
	return ret;
    }

    current_time = event->xbutton.time;
    delta = current_time - mw->matrix.last_click_time;

    if (row == mw->matrix.last_row && column == mw->matrix.last_column &&
	delta < ( unsigned long )mw->matrix.double_click_interval )
	ret = 1;
    else
	ret = 0;

    return ret;
}

/*ARGSUSED*/
void
xbaeHandleClick(w, mw, event, cont)
Widget w;
XbaeMatrixWidget mw;
XEvent *event;
Boolean *cont;
{
    /* if we have a double click and a callback - break out! */
    if (event->type != ButtonPress && event->type != ButtonRelease )
	return;
    if (mw->matrix.default_action_callback &&
	DoubleClick(mw, event, mw->matrix.current_row,
		    mw->matrix.current_column))
    {
	XbaeMatrixDefaultActionCallbackStruct call_data;

	call_data.reason = XbaeDefaultActionReason;
	call_data.row = mw->matrix.current_row;
	call_data.column = mw->matrix.current_column;
	call_data.event = event;

	XtCallCallbackList((Widget)mw, mw->matrix.default_action_callback,
			   (XtPointer) &call_data);
    }
}

/* ARGSUSED */
void
xbaeSelectCellACT(w, event, params, nparams)
Widget w;
XEvent *event;
String *params;
Cardinal *nparams;
{
    XbaeMatrixWidget mw;
    int x, y;
    int row, column;
    CellType cell;
    XbaeMatrixSelectCellCallbackStruct call_data;

    /*
     * Get Matrix widget and make sure it is a Matrix subclass.
     * w could be Matrix, or the Clip or textField children of Matrix
     */
    if (XtIsSubclass(w, xbaeMatrixWidgetClass))
	mw = (XbaeMatrixWidget) w;
    else if (XtIsSubclass(XtParent(w), xbaeMatrixWidgetClass))
	mw = (XbaeMatrixWidget) XtParent(w);
    else
    {
	XtAppWarningMsg(
	    XtWidgetToApplicationContext(w),
	    "xbaeSelectCellACT", "badWidget", "XbaeMatrix",
	    "XbaeMatrix: Bad widget passed to SelectCell action",
	    NULL, 0);
	return;
    }

    /*
     * If we don't have a selectCellCallback, then return now
     */
    if (!mw->matrix.select_cell_callback)
	return;

    if (!xbaeEventToXY(mw, event, &x, &y, &cell))
	return;

    /*
     * Convert the point to a row,column. If it does not pick a valid
     * cell, then return.
     */
    if (!xbaeXYToRowCol(mw, &x, &y, &row, &column, cell))
	return;

    /*
     * Call our select_cell callbacks
     */
    call_data.reason = XbaeSelectCellReason;
    call_data.row = row;
    call_data.column = column;
    call_data.selected_cells = mw->matrix.selected_cells;
    call_data.cells = mw->matrix.cells;
    call_data.num_params = *nparams;
    call_data.params = params;
    call_data.event = event;

    XtCallCallbackList((Widget) mw, mw->matrix.select_cell_callback,
		       (XtPointer) &call_data);
}


/* ARGSUSED */
void
xbaeTraverseNextACT(w, event, params, nparams)
Widget w;
XEvent *event;
String *params;
Cardinal *nparams;
{
    XbaeMatrixWidget mw;

    /*
     * Get Matrix widget and make sure it is a Matrix subclass.
     * w should be the textField widget.
     */
    if (XtIsSubclass(XtParent(w), xbaeMatrixWidgetClass))
	mw = (XbaeMatrixWidget) XtParent(w);
    else
    {
	XtAppWarningMsg(
	    XtWidgetToApplicationContext(w),
	    "traverseNextACT", "badWidget", "XbaeMatrix",
	    "XbaeMatrix: Bad widget passed to TraverseNext action",
	    NULL, 0);
	return;
    }

    /*
     * Set the traversing direction flag.  XmProcessTraversal may traverse
     * to the Clip widget. If it does, then we will see this flag in
     * the Clip focusCallback, TraverseInCB, and we will continue to traverse
     * on out of the mw.  yuck!
     */
    mw->matrix.traversing = XmTRAVERSE_NEXT_TAB_GROUP;
    ( void )XmProcessTraversal(TextChild(mw), XmTRAVERSE_NEXT_TAB_GROUP);
    mw->matrix.traversing = NOT_TRAVERSING;
}

/* ARGSUSED */
void
xbaeTraversePrevACT(w, event, params, nparams)
Widget w;
XEvent *event;
String *params;
Cardinal *nparams;
{
    XbaeMatrixWidget mw;

    /*
     * Get Matrix widget and make sure it is a Matrix subclass.
     * w should be the textField widget.
     */
    if (XtIsSubclass(XtParent(w), xbaeMatrixWidgetClass))
	mw = (XbaeMatrixWidget) XtParent(w);
    else
    {
	XtAppWarningMsg(
	    XtWidgetToApplicationContext(w),
	    "traversePrevACT", "badWidget", "XbaeMatrix",
	    "XbaeMatrix: Bad widget passed to TraversePrev action",
	    NULL, 0);
	return;
    }

    /*
     * Set the traversing direction flag.  XmProcessTraversal may traverse
     * to the Clip widget. If it does, then we will see this flag in
     * the Clip focusCallback, TraverseInCB, and we will continue to traverse
     * on out of the mw.  yuck!
     */
    mw->matrix.traversing = ( int )XmTRAVERSE_PREV_TAB_GROUP;
    ( void )XmProcessTraversal(TextChild(mw), XmTRAVERSE_PREV_TAB_GROUP);
    mw->matrix.traversing = NOT_TRAVERSING;
}

