# tools.tcl --
#
#	Creates settings for external tools
#
# Copyright (c) 2000-2002 Zveno Pty Ltd
#
# $Id: tools.tcl,v 1.2 2002/05/23 10:00:47 balls Exp $

package provide tools 1.0

catch {package require xslt}

namespace eval tools {
    namespace export init getconfig
}

# tools::init --
#
#	Setup configuration for tools
#
# Arguments:
#	cfgfilename	Configuration document
#
# Results:
#	Configuration doc parsed

proc tools::init cfgfilename {
    variable cfgfile
    variable cfgdom

    set ch [open $cfgfilename]
    set cfgdata [read $ch]
    close $ch

    set cfgdom [dom::libxml2::parse $cfgdata -baseuri file::/$cfgfilename]

    set cfgfile $cfgfilename

    # Directories

    variable tools_dir [file join [pwd] [file dirname [file dirname $cfgfilename]]]
    variable libdir [file join $tools_dir lib]
    variable bindir [file dirname [file dirname [file dirname [info script]]]]

    switch $::tcl_platform(platform) {
	windows {
	    variable system_dir C:/Progra~1/Javasoft
	    variable PathSep \\
	    variable ClassPathSep \;
	    variable omnimark_dir D:/Progra~2/Omnimark
	}
	unix {
	    variable system_dir /usr/local
	    variable PathSep /
	    variable ClassPathSep :
	    variable omnimark_dir [file join $tools_dir omnimark5.3]
	}
	macintosh {
	    variable PathSep :
	}
    }

    variable javaNode [lindex [dom::libxml2::selectNode $cfgdom /configuration/java] 0]
    variable java [dom::libxml2::node stringValue $javaNode]
    variable javamemflag [dom::libxml2::element getAttribute $javaNode memflag]

    # Default XSLT processor
    variable dflt_xsltNode [lindex [dom::libxml2::selectNode $cfgdom /configuration/default/xslt] 0]
    variable dflt_xslt [dom::libxml2::node stringValue $dflt_xsltNode]

    # Default XML WF parser
    variable dflt_xmlWFNode [lindex [dom::libxml2::selectNode $cfgdom /configuration/default/xml/nonvalid] 0]
    variable dflt_xmlwf [dom::libxml2::node stringValue $dflt_xmlWFNode]

    # Default XML valid parser
    variable dflt_xmlVNode [lindex [dom::libxml2::selectNode $cfgdom /configuration/default/xml/valid] 0]
    variable dflt_xmlvalid [dom::libxml2::node stringValue $dflt_xmlVNode]

    return {}
}

# tools::getconfig --
#
#	Return the DOM tree for the config doc
#
# Arguments:
#	None
#
# Results:
#	DOM node token

proc tools::getconfig {} {
    variable cfgdom
    return $cfgdom
}

# tools::FormatParams --
#
#	Massage parameters for input to XSLT processors
#
# Arguments:
#	args	name/value pairs
#
# Results:
#	Returns name=value string

proc tools::FormatParams args {
    set result {}
    foreach {key value} $args {
	if {[string match -* $key]} {
	    continue
	}
	lappend result $key=$value
    }
    return $result
}

# tools::xslt --
#
#	Perform a transformation
#
# Arguments:
#	src	Source XML document
#	ssheet	XSL stylesheet
#	dst	Filename for result
#	args	parameters
#
# Results:
#	Transformation occurs or an error

proc tools::xslt {src ssheet dst args} {
    variable dflt_xslt
    eval $dflt_xslt [list $src $ssheet $dst] $args
}

proc tools::xsltprocFormatParams args {
    set result {}
    foreach {name value} $args {
	switch -glob -- $name {
	    -* {}
	    *path {
		lappend result -param $name $value
	    }
	    default {
		lappend result -param $name '$value'
	    }
	}
    }

    return $result
}

proc tools::xsltproc {src ssheet dst args} {
    variable cfgdom
    variable libdir
    variable bindir

    array set opts {-async 0}
    array set opts $args

    set xsltprocNode [lindex [dom::libxml2::selectNode $cfgdom /configuration/xsltproc] 0]
    set xsltproc [file join [pwd] $bindir [dom::libxml2::node stringValue $xsltprocNode]]

    if {![file exists $xsltproc]} {
	set xsltproc [dom::libxml2::node stringValue $xsltprocNode]
    }

    if {$opts(-async)} {
	return [open "|$xsltproc -o $dst [eval xsltprocFormatParams $args] $ssheet $src"]
    } else {
	eval exec $xsltproc -o [list $dst] [eval xsltprocFormatParams $args] [list $ssheet $src]
    }
}

# builtin XSLT processor can't do asynchronous processing -
# need an external process for that.

proc tools::xsltbuiltin {src ssheet dst args} {
    variable builtinCache

    array set params $args
    foreach opt [array names params -*] {
	unset params($opt)
    }

    if {![info exists builtinCache($ssheet)]} {
	set ch [open $ssheet]
	set ssheetXML [read $ch]
	close $ch
	set ssheetdoc [dom::libxml2::parse $ssheetXML -baseuri file://$ssheet]
	set builtinCache($ssheet) [xslt::compile $ssheetdoc]
    }

    set ch [open $src]
    set docXML [read $ch]
    close $ch
    set sourceDoc [dom::libxml2::parse $docXML -baseuri file://$src]

    set ::tools::builtinXSLTmessages {}

    if {[catch {::xslt::transform $builtinCache($ssheet) $sourceDoc -messagecommand [namespace code [list xsltbuiltin:messages]] [array get params]} result]} {
	return -code error $result
    } else {
	set ch [open $dst w]
	puts $ch [dom::libxml2::serialize $result -method [xslt::cget $builtinCache($ssheet) -method]]
	close $ch
    }

    return $::tools::builtinXSLTmessages
}

proc tools::xsltbuiltin:messages args {
    eval append ::tools::builtinXSLTmessages $args
}

proc tools::xt {src ssheet dst args} {
    variable java
    variable xt_classpath
    variable xt_flags
    global env

    array set opts {-async 0}
    array set opts $args

    set env(CLASSPATH) $xt_classpath
    if {$opts(-async)} {
	return [open "|$java $xt_flags com.jclark.xsl.sax.Driver $src $ssheet $dst [eval FormatParams $args]" r]
    } else {
	eval exec [list $java] $xt_flags com.jclark.xsl.sax.Driver [list $src $ssheet $dst] [eval FormatParams $args]
    }
}

proc tools::saxon {src ssheet dst args} {
    variable java
    variable saxon_dir
    variable saxon_classpath
    variable saxon_flags
    global env

    array set opts {-async 0}
    array set opts $args

    set env(CLASSPATH) $saxon_classpath
    if {$opts(-async)} {
	set result [open "|$java $saxon_flags com.icl.saxon.StyleSheet -o $dst $src $ssheet [eval FormatParams $args]" r]
    } else {
	set result [eval exec [list $java] $saxon_flags com.icl.saxon.StyleSheet -o [list $dst] [list $src $ssheet] [eval FormatParams $args]]
    }

    return $result
}

proc tools::xalan {src ssheet dst args} {
    variable java
    variable xalan_classpath
    variable xalan_flags
    global env

    array set opts {-async 0}
    array set opts $args

    set params {}
    foreach {name value} $args {
	if {[string match -* $name]} {
	    continue
	}
	lappend params -PARAM $name $value
    }

    set env(CLASSPATH) $xalan_classpath
    if {$opts(-async)} {
	return [open "|$java $xalan_flags org.apache.xalan.xslt.Process -IN $src -XSL $ssheet -OUT $dst $params" r]
    } else {
	eval exec [list $java] $xalan_flags org.apache.xalan.xslt.Process [list -IN $src -XSL $ssheet -OUT $dst] $params
    }
}

# tools::wellformed --
#
#	Non-validating parse of an XML document
#
# Arguments:
#	src	Source XML document
#	args	parameters
#
# Results:
#	Well-formedness reported

proc tools::wellformed {src args} {
    variable dflt_xmlwf
    eval $dflt_xmlwf [list $src] -validate 0 $args
}

# tools::validate --
#
#	Perform a validation
#
# Arguments:
#	src	Source XML document
#	args	parameters
#
# Results:
#	Validity reported

proc tools::validate {src args} {
    variable dflt_xmlvalid
    eval $dflt_xmlvalid [list $src] -validate 1 $args
}

proc tools::wfbuiltin {src args} {

    set ch [open $src]
    set xml [read $ch]
    close $ch

    set doc [dom::libxml2::parse $xml -baseuri file://$src]
    # dom::libxml2::destroy $doc

    return {}
}

proc tools::validbuiltin {src args} {

    set ch [open $src]
    set xml [read $ch]
    close $ch

    set msg {}

    if {[catch {dom::libxml2::parse $xml -baseuri file://$src} doc]} {
	return $doc
    }

    catch {dom::libxml2::validate $doc} msg

    # dom::libxml2::destroy $doc

    return $msg
}

proc tools::xmllint {src args} {
    variable cfgdom
    variable tools_dir
    variable libdir
    variable bindir

    array set opts {
	-async 0
	-validate 0
    }
    array set opts $args

    set xmllintNode [lindex [dom::libxml2::selectNode $cfgdom /configuration/xmllint] 0]
    set xmllint [file join [pwd] $bindir [dom::libxml2::node stringValue $xmllintNode]]

    if {![file exists $xmllint]} {
	set xmllint [dom::libxml2::node stringValue $xmllintNode]
    }

    set validarg {}
    if {$opts(-validate)} {
	set validarg --valid
    }

    set pwd [pwd]
    cd [file dirname $src]

    if {$opts(-async)} {
	cd $pwd
	return [open "|$xmllint $validarg --noout [list [file tail $src]]" r]
    } else {
	set result [eval exec $xmllint $validarg --noout [list [file tail $src]]]
    }

    cd $pwd

    return $result
}

proc tools::fop {fo pdf args} {
    variable java
    variable fop_classpath
    variable fop_flags
    global env

    array set opts {-async 0}
    array set opts $args

    set env(CLASSPATH) $fop_classpath
    if {$opts(-async)} {
	return [open "|$java $fop_flags org.apache.fop.apps.CommandLine $fo $pdf" r]
    } else {
	exec $java $fop_flags org.apache.fop.apps.CommandLine $fo $pdf
    }
}

proc tools::batik {svg args} {
    variable java
    variable batik_dir

    array set opts {-async 0}
    array set opts $args

    if {$opts(-async)} {
	return [open "|$java -jar [file join $batik_dir batik-rasterizer.jar] -d [file dirname $svg] -m image/jpeg $svg" r]
    } else {
	exec $java -jar [file join $batik_dir batik-rasterizer.jar] -d [file dirname $svg] -m image/jpeg $svg
    }
}

proc tools::upcast {rtf xml dir args} {
    variable java
    variable upcast_dir
    variable upcast_classpath
    variable upcast_flags
    variable PathSep
    global env

    array set opts {
	-async 0
	-images on
    }
    array set opts $args

    # Make sure directory has path separator at the end
    if {![regexp $PathSep\$ $dir]} {
	append dir $PathSep
    }

    switch -- $opts(-images) {
	1 -
	yes -
	on {
	    set images -on
	}
	0 -
	no -
	off {
	    set images -off
	}
	default {
	    return -code error "invalid value for \"-images\""
	}
    }

    # Steve: 9/2/01: Removed images switch due to Windoze problem.

    set env(CLASSPATH) $upcast_classpath
    set env(DISPLAY) pbook:0
    if {$opts(-async)} {
	return [open "|$java $upcast_flags rtf2xml $rtf $xml $dir" r]
    } else {
	eval exec $java $upcast_flags rtf2xml [list $rtf $xml $dir]
    }
}

