# testnode.tcl --
#
# Test the TGraph node object's commands:
#
# Copyright (c) 2000-2003, JYL Software Inc.
# 
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
# 
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE, EVEN IF
# JYL SOFTWARE INC. IS MADE AWARE OF THE POSSIBILITY OF SUCH DAMAGE.

# Require the tcltest package:

package require tcltest

# Load some utilities:

source [file join [file dirname [info script]] util.tcl]
    
# The tests:

# Test coverage panel:
#
# $n kind
# $n vertexcount
# $n set
# $n add
# $n get
# $n setnode
# $n addnode
# $n getvertex
# $n movevertex
# $n detachvertex
# $n renamevertex
# $n vertextype
# $n vertexrank
# $n vertexname
# $n exists
# $n parent
# $n parentcount
# $n occurrencecount
# $n parentrank
# $n isroot
# $n detach
# 
# Not tested:
#
# $n rankinparent
# $n nameinparent
# $n storage
# $n dispose
# $n isvalid
# $n method
# $n call
# $n foreach
# $n id

# Test that [$n storage] returns the same storage as the one that contains
# the root.

tcltest::test node-1.0.1 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]

    if {![string equal [$n storage] $x]} {
	deleteAndError $x "\[$n storage\] and the storage are different"
    }
    $x delete
    set empty
} ""

# Test that enclosed nodes are removed when a storage is closed.

tcltest::test node-1.0.2 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    $x delete
    if {![string equal [info commands $x] ""] || [$n isvalid]} {
	error "Entities $n and/or $x were not deleted when $x was closed"
    }
    set empty
} ""

# Test the isroot operation:

tcltest::test node-1.0.4 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    if {![$n isroot]} {
	deleteAndError $x "Node $n is not root of storage $x"
    }
    set n1 [$n addnode frob last]
    if {[$n1 isroot]} {
	deleteAndError $x \
	    "Node $n1 is incorrectly marked as root instead of $n, storage $x"
    }
    $x close
    if {![string equal [info commands $x] ""] ||
	[$n isvalid] ||
	[$n1 isvalid]} {
	deleteAndError $x "Commands not deleted at close"
    }
    set empty
} ""

# Test the isroot operation:

tcltest::test node-1.0.9 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    if {![$n isroot]} {
	deleteAndError $x \
		"Isroot confusion, should be 1, got [$n isroot]"
    }
    set n1 [$n addnode frob last]
    if {[$n1 isroot]} {
	deleteAndError $x \
		"Isroot confusion, should be 0, got [$n1 isroot]"
    }
    $x delete
    set empty

} ""

tcltest::test node-1.0.10 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    if {![$n isroot]} {
	deleteAndError $x \
		"Isroot confusion, should be 1, got [$n isroot]"
    }
    set n1 [$n addnode frob last]
    if {[$n1 isroot]} {
	deleteAndError $x \
		"Isroot confusion, should be 0, got [$n1 isroot]"
    }

    # Make $n1 a parent of $n, see if [$n isroot] is still true (shouldn't be).

    $n1 add frob last $n -node
    if {![$n isroot]} {
	deleteAndError $x \
		"Isroot confusion, should be 1, got [$n isroot]"
    }

    # But $n is still root, and $n1 is not.

    if {[$n1 isroot] || ![$n isroot]} {
	deleteAndError $x "Isroot confusion"
    }

    $x delete
    set empty

} ""

# New nodes should have zero vertices initially.

tcltest::test node-1.1.0 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    if {[$n vertexcount] != 0} {
	deleteAndError $x \
		"initial vertex count for node $n is [$n vertexcount], not 0"
    }
    $x delete
    set empty
} ""

# The "kind" attribute of a node is "node"

tcltest::test node-1.2.0 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    if {![string equal node [$n kind]]} {
	deleteAndError $x \
		"node kind of $n is [$n kind], not \"node\""
    }
    $x delete
    set empty
} ""

# Add an integer vertex to a node, and see if it is there.

tcltest::test node-1.3.0 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    $n add frob last 12
    if {[$n vertexcount] != 1} {
	deleteAndError $x \
	   "after adding one vertex the vertexcount is [$n vertexcount], not 1"
    }
    set v [$n get frob]
    if {$v != 12} {
	deleteAndError $x \
	   "set vertex frob to 12, and got back $v"
    }
    $x delete
    set empty
} ""

# Add ten integer vertices to a node, and see if they're all there.

tcltest::test node-1.3.1 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	$n add frob$i last $i
    }
    for {set i 0} {$i < 10} {incr i} {
	set v [$n get frob$i]
	if {$v != $i} {
	    deleteAndError $x \
		    "set vertex frob$i to $i and got back $v"
	}
    }
    $x delete
    set empty
} ""

# Create ten integer vertices, then set all ten of them to a string

tcltest::test node-1.3.2 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	$n add frob$i last $i
    }
    for {set i 0} {$i < 10} {incr i} {
	$n set frob$i abcde$i
    }
    for {set i 0} {$i < 10} {incr i} {
	set v [$n get frob$i]
	if {![string equal $v abcde$i]} {
	    deleteAndError $x \
		    "set vertex frob$i to abcde$i and got back $v"
	}
    }
    $x delete
    set empty
} ""

# Create string vertices and see if they are all there.

tcltest::test node-1.3.3 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	$n add frob$i last abcde$i
    }
    for {set i 0} {$i < 10} {incr i} {
	set v [$n get frob$i]
	if {![string equal $v abcde$i]} {
	    deleteAndError $x \
		    "set vertex frob$i to abcde$i and got back $v"
	}
    }
    $x delete
    set empty
} ""

# Test vertex types:

tcltest::test node-1.3.4 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	if {$i < 3} {
	    $n add frob$i last $i
	} elseif {$i < 6} {
	    $n add frob$i last abcde$i
	} else {
	    $n add frob$i last $i.0
	}
    }
    for {set i 0} {$i < 10} {incr i} {
	set type [$n vertextype frob$i]
	if {$i < 3} {
	    if {![string equal $type int]} {
		deleteAndError $x \
			"set frob$i to integer value but type is $type"
	    }
	} elseif {$i < 6} {
	    if {![string equal $type string]} {
		deleteAndError $x \
			"set frob$i to string value but type is $type"
	    }
	} else {
	    if {![string equal $type double]} {
		deleteAndError $x \
			"set frob$i to double value but type is $type"
	    }
	}
    }
    $x delete
    set empty
} ""

# Test setting vertices:

tcltest::test node-1.3.5 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	if {$i < 3} {
	    $n add frob$i last $i
	} elseif {$i < 6} {
	    $n add frob$i last abcde$i
	} else {
	    $n add frob$i last $i.0
	}
    }
    for {set i 0} {$i < 10} {incr i} {
	set type [$n vertextype frob$i]
	if {$i < 3} {
	    if {![string equal $type int]} {
		deleteAndError $x \
			"set frob$i to integer value but type is $type"
	    }
	} elseif {$i < 6} {
	    if {![string equal $type string]} {
		deleteAndError $x \
			"set frob$i to string value but type is $type"
	    }
	} else {
	    if {![string equal $type double]} {
		deleteAndError $x \
			"set frob$i to double value but type is $type"
	    }
	}
    }
    for {set i 0} {$i < 10} {incr i} {
	$n set frob$i 23 -double
	set type [$n vertextype frob$i]
	if {![string equal $type double]} {
	    deleteAndError $x \
		    "set frob$i 23 double but type is $type"
	}
    }
    $x delete
    set empty
} ""

# Test the addnode operation.

tcltest::test node-1.4.0 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	$n addnode frob$i last
    }
    for {set i 0} {$i < 10} {incr i} {
	set n1 [$n get frob$i]
	set type [$n vertextype frob$i]
	set p [$n1 parent]
	if {![string equal $p $n] || ![string equal $type node]} {
	    deleteAndError $x \
		    "set frob$i to node and got $type, $n1"
	}
    }
    $x delete
    set empty
} ""

tcltest::test node-1.4.1 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	$n add frob$i last $i
    }
    for {set i 0} {$i < 10} {incr i} {
	$n addnode frob$i first
    }
    if {[$n vertexcount] != 20} {
	deleteAndError $x \
		"node $n should have 20 vertices, but has [$n vertexcount]"
    }
    for {set i 0} {$i < 10} {incr i} {
	set n1 [$n get frob$i]
	set type [$n vertextype frob$i]
	set p [$n1 parent]
	if {![string equal $p $n] || ![string equal $type node]} {
	    deleteAndError $x \
		    "set frob$i to node and got $type, $n1"
	}
    }
    $x delete
    set empty
} ""

# Test the setnode operation:

tcltest::test node-1.4.1 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	$n add frob$i last $i
    }
    for {set i 0} {$i < 10} {incr i} {
	if {![string equal [$n vertextype frob$i] int]} {
	    deleteAndError $x \
		    "set frob$i to integer and got type [$n vertextype frob$i]"
	}
    }
    for {set i 0} {$i < 10} {incr i} {
	$n setnode frob$i
    }
    for {set i 0} {$i < 10} {incr i} {
	set n1 [$n get frob$i]
	set type [$n vertextype frob$i]
	set p [$n1 parent]
	if {![string equal $p $n] || ![string equal $type node]} {
	    deleteAndError $x \
		    "set frob$i to node and got type $type, $n1"
	}
    }
    $x delete
    set empty
} ""

# Test vertex deletion for scalar vertices:

tcltest::test node-1.5.0 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	$n add frob$i last $i
    }
    if {[$n vertexcount] != 10} {
	deleteAndError $x \
		"Added ten vertices but vertex count is [$n vertexcount]"
    }
    for {set i 0} {$i < 10} {incr i} {
	$n detachvertex frob$i
    }
    if {[$n vertexcount] != 0} {
	deleteAndError $x \
		"Deleted ten vertices but vertex count is [$n vertexcount]"
    }
    $x delete
    set empty
} ""

tcltest::test node-1.5.1 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	$n add frob$i last abcde$i
    }
    if {[$n vertexcount] != 10} {
	deleteAndError $x \
		"Added ten vertices but vertex count is [$n vertexcount]"
    }
    for {set i 0} {$i < 10} {incr i} {
	$n detachvertex frob$i
    }
    if {[$n vertexcount] != 0} {
	deleteAndError $x \
		"Deleted ten vertices but vertex count is [$n vertexcount]"
    }
    $x delete
    set empty
} ""

tcltest::test node-1.5.2 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	$n add frob$i last $i.0
    }
    if {[$n vertexcount] != 10} {
	deleteAndError $x \
		"Added ten vertices but vertex count is [$n vertexcount]"
    }
    for {set i 0} {$i < 10} {incr i} {
	$n detachvertex frob$i
    }
    if {[$n vertexcount] != 0} {
	deleteAndError $x \
		"Deleted ten vertices but vertex count is [$n vertexcount]"
    }
    $x delete
    set empty
} ""

# Test vertex detach with node values:

tcltest::test node-1.5.3 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	$n addnode frob$i last
    }
    if {[$n vertexcount] != 10} {
	deleteAndError $x \
		"Added ten vertices but vertex count is [$n vertexcount]"
    }
    for {set i 0} {$i < 10} {incr i} {
	$n detachvertex frob$i
    }
    if {[$n vertexcount] != 0} {
	deleteAndError $x \
		"Deleted ten vertices but vertex count is [$n vertexcount]"
    }
    $x delete
    set empty
} ""

# Test vertex detach with non-empty node values:

tcltest::test node-1.5.4 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	set n1 [$n addnode frob$i last]
	for {set j 0} {$j < 10} {incr j} {
	    $n1 add jay$i last $i
	}
    }
    if {[$n vertexcount] != 10} {
	deleteAndError $x \
		"Added ten vertices but vertex count is [$n vertexcount]"
    }
    for {set i 0} {$i < 10} {incr i} {
	$n detachvertex frob$i
    }
    if {[$n vertexcount] != 0} {
	deleteAndError $x \
		"Deleted ten vertices but vertex count is [$n vertexcount]"
    }
    $x delete
    set empty
} ""

# Test vertex detach but with non-empty node-valued vertices.

tcltest::test node-1.5.5 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	set n1 [$n addnode frob$i last]
	for {set j 0} {$j < 10} {incr j} {
	    $n1 addnode jay$i last
	}
    }
    if {[$n vertexcount] != 10} {
	deleteAndError $x \
		"Added ten vertices but vertex count is [$n vertexcount]"
    }
    for {set i 0} {$i < 10} {incr i} {
	$n detachvertex frob$i
    }
    if {[$n vertexcount] != 0} {
	deleteAndError $x \
		"Deleted ten vertices but vertex count is [$n vertexcount]"
    }
    if {![$n1 isvalid] || ![$n1 isdetached]} {
	deleteAndError $x \
		"Deleted all vertices but recursive node $n1 invalid"
    }
    $x delete
    set empty
} ""

# Test the movevertex operation, within a node:

tcltest::test node-1.6.0 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	$n add frob$i last $i
    }
    if {[$n vertexcount] != 10} {
	deleteAndError $x \
		"Expecting vertex count of 10, got [$n vertexcount]"
    }
    if {[$n vertexrank frob3] != 4} {
	deleteAndError $x \
		"Vertex frob3 should be rank 4, is [$n vertexrank frob3]"
    }
    set v [$n getvertex frob3]
    for {set i 0} {$i < 9} {incr i} {
	$n movevertex $v at [expr $i + 1]
	if {[$n vertexrank frob3] != ($i + 1)} {
	    set j [expr $i + 1]
	    deleteAndError $x \
		    "Vertex frob3 should be rank $j, is [$n vertexrank frob3]"
	}
    }
    for {set i 10} {$i > 0} {incr i -1} {
	$n movevertex $v at $i
	if {[$n vertexrank frob3] != $i} {
	    deleteAndError $x \
		    "Vertex frob3 should be rank $i, is [$n vertexrank frob3]"
	}
    }
    $x delete
    set empty
} ""

# Test the movevertex operation between two nodes:

tcltest::test node-1.6.1 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	$n add frob$i last $i
    }
    if {[$n vertexcount] != 10} {
	deleteAndError $x \
		"Expecting vertex count of 10, got [$n vertexcount]"
    }
    if {[$n vertexrank frob3] != 4} {
	deleteAndError $x \
		"Vertex frob3 should be rank 4, is [$n vertexrank frob3]"
    }
    set v [$n getvertex frob3]
    set n1 [$n addnode barbar last]
    for {set i 0} {$i < 10} {incr i} {
	$n1 add foo$i last $i
    }
    if {[$n1 vertexcount] != 10} {
	deleteAndError $x \
		"Expecting vertex count of 10, got [$n1 vertexcount]"
    }
    if {![string equal [$v node] $n]} {
	deleteAndError $x \
		"Vertex frob3 should be in $n, actually is in [$v node]"
    }
    for {set i 0} {$i < 11} {incr i} {
	$n1 movevertex $v at [expr $i + 1]
	if {[$n1 vertexrank frob3] != ($i + 1)} {
	    set c1 [expr $i + 1]
	    set c2 [$n1 vertexrank frob3]
	    deleteAndError $x \
		    "Vertex frob3 expected at rank $c1, got $c2"
	}
    }
    if {([$n vertexcount] != 10) || ([$n1 vertexcount] != 11)} {
	set c1 [$n vertexcount]
	set c2 [$n1 vertexcount]
	deleteAndError $x \
		"Vertex count confusion, expected 10 and 11, got $c1 and $c2"
    }
    $x delete
    set empty
} ""

# Test the add operation with a node value; see that the occurrence count
# increases. Also tests the parentcount and occurrencecount operations.

tcltest::test node-1.6.3 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	$n add frob$i last $i
    }
    if {[$n vertexcount] != 10} {
	deleteAndError $x \
		"Expecting vertex count of 10, got [$n vertexcount]"
    }
    if {[$n vertexrank frob3] != 4} {
	deleteAndError $x \
		"Vertex frob3 should be rank 4, is [$n vertexrank frob3]"
    }
    set v [$n getvertex frob3]
    set n1 [$n setnode frob3]
    if {[$n1 parentcount] != 1} {
	deleteAndError $x \
		"Parent count confusion, expected 1, got [$n1 parentcount]"
    }
    set n2 [$n addnode foofoo last]
    for {set i 0} {$i < 10} {incr i} {
	$n2 add foo$i last $n1 -node
	if {[$n1 occurrencecount] != ($i + 2)} {
	    set c1 [expr $i + 2]
	    set c2 [$n1 occurrencecount]
	    deleteAndError $x \
		    "Occurrence count confusion, expected $c1, got $c2"
	}
    }
    if {[$n1 occurrencecount $n2] != 10} {
	set c1 [$n1 occurrencecount $n2]
	deleteAndError $x \
		"Occurrence count confusion, expected 10, got $c1"
    }
    if {[$n1 occurrencecount $n] != 1} {
	set c1 [$n1 occurrencecount $n]
	deleteAndError $x \
		"Occurrence count confusion, expected 1, got $c1"
    }
    if {[$n1 parentcount] != 2} {
	deleteAndError $x \
		"Parent count confusion, expected 2, got [$n1 parentcount]"
    }
    $x delete
    set empty
} ""

# Test the parentrank operation:

tcltest::test node-1.6.4 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	set nodes($i) [$n addnode foo last]
    }
    set c [$nodes(0) addnode frob last]
    for {set i 1} {$i < 10} {incr i} {
	$nodes($i) add frob last $c -node
    }
    if {[$c parentcount] != 10} {
	deleteAndError $x \
		"Parent count confusion, expected 10, got [$c parentcount]"
    }
    for {set i 0} {$i < 10} {incr i} {
	set r [$c parentrank $nodes($i)]
	if {$r != ($i + 1)} {
	    deleteAndError $x \
		    "Parent rank confusion, expected [expr $i + 1], got $r"
	}
    }
    for {set i 0} {$i < 10} {incr i} {
	$nodes($i) detach
	set nodes($i) null
	if {$i < 9} {
	    set remaining [expr 9 - $i]
	    set cnt [$c parentcount]
	    if {$cnt != $remaining} {
		deleteAndError $x \
			"Parent count confusion, expected $remaining got $cnt"
	    }
	}
    }
    $x delete
    set empty
} ""
    
# Test the rankinparent and nameinparent operations.

tcltest::test node-1.6.5 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	set nodes($i) [$n addnode foo last]
	for {set j 0} {$j < 10} {incr j} {
	    if {$i == $j} {
		$nodes($i) add frob$i last $n -node
	    } else {
		$nodes($i) add frob$i last $i
	    }
	}
    }

    # TODO: test the rankinparent and nameinparent operations:

    $x delete
    set empty
} ""
    
# Test exists operation:

tcltest::test node-1.7.0 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	$n add frob$i last $i
    }
    if {[$n exists foo]} {
	deleteAndError $x \
		"Vertex foo exists when it shouldn't"
    }
    if {![$n exists frob3]} {
	deleteAndError $x \
		"Vertex frob3 doesn't exist when it should"
    }
    $n detachvertex frob3
    if {[$n exists frob3]} {
	deleteAndError $x \
		"Vertex frob3 exists when it shouldn't"
    }
    $x delete
    set empty
} ""

# Test the vertextype, vertexrank and vertexname operations.

tcltest::test node-1.7.1 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	$n add frob$i last $i
    }
    for {set i 1} {$i < 11} {incr i} {
	set nm [$n vertexname $i]
	if {![string equal $nm frob[expr $i - 1]]} {
	    set en frob[expr $i - 1]
	    deleteAndError $x \
		    "Vertex name confusion, expected $en, got $nm"
	}
	if {![string equal int [$n vertextype $nm]]} {
	    set tp [$n vertextype $nm]
	    deleteAndError $x \
		    "Vertex type confusion, expected int, got $tp"
	}
	if {![string equal int [$n vertextype $i]]} {
	    set tp [$n vertextype $i]
	    deleteAndError $x \
		    "Vertex type confusion, expected int, got $tp"
	}
    }
    for {set i 0} {$i < 10} {incr i} {
	$n set frob$i abcde$i
	set nm [$n vertexname [expr $i + 1]]
	if {![string equal $nm frob$i]} {
	    deleteAndError $x \
		    "Vertex name confusion, expected frob$i, got $nm"
	}
	set tp [$n vertextype frob$i]
	if {![string equal $tp string]} {
	    deleteAndError $x \
		    "Vertex type confusion, expected string, got $tp"
	}
	if {![string equal [$n vertextype [expr $i + 1]] string]} {
	    set tp [$n vertextype [expr $i + 1]]
	    deleteAndError $x \
		    "Vertex type confusion, expected string, got $tp"
	}
    }
    for {set i 0} {$i < 10} {incr i} {
	set r [$n vertexrank frob$i]
	if {$r != ($i + 1)} {
	    set c [expr $i + 1]
	    deleteAndError $x \
		    "Vertex rank confusion, expected $c, got, $r"
	}
    }
    $x delete
    set empty
} ""

# Test the renamevertex operation

tcltest::test node-1.7.2 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	$n add frob$i last $i
    }
    for {set i 0} {$i < 10} {incr i} {
	$n renamevertex frob$i foo
    }
    for {set i 1} {$i < 11} {incr i} {
	$n renamevertex foo(1) hello$i
    }
    for {set i 1} {$i < 11} {incr i} {
	if {![$n exists hello$i]} {
	    deleteAndError $x \
		    "Vertex name confusion, expected hello$i to exist"
	}
    }
    for {set i 1} {$i < 11} {incr i} {
	$n renamevertex $i hello
    }
    for {set i 1} {$i < 11} {incr i} {
	if {![$n exists hello($i)]} {
	    deleteAndError $x \
		    "Vertex name confusion, expected hello($i) to exist"
	}
    }
    $x delete
    set empty
} ""

# Test the movevertex operation for moving a node around:

tcltest::test node-1.6.2 {node tests} {
    showActive
    set empty ""
    set x [ensureFreshStorage foo.db]
    set n [$x root]
    for {set i 0} {$i < 10} {incr i} {
	$n add frob$i last $i
    }
    if {[$n vertexcount] != 10} {
	deleteAndError $x \
		"Expecting vertex count of 10, got [$n vertexcount]"
    }
    if {[$n vertexrank frob3] != 4} {
	deleteAndError $x \
		"Vertex frob3 should be rank 4, is [$n vertexrank frob3]"
    }
    set v [$n getvertex frob3]
    set n1 [$n setnode frob3]
    for {set i 0} {$i < 10} {incr i} {
	$n1 add foo$i last $i.0
    }
    if {[$n1 vertexcount] != 10} {
	deleteAndError $x \
		"Expecting vertex count of 10, got [$n1 vertexcount]"
    }
    if {![string equal [$v node] $n]} {
	deleteAndError $x \
		"Vertex frob3 should be in $n, actually is in [$v node]"
    }
    for {set i 0} {$i < 10} {incr i} {
	$n movevertex $v at [expr $i + 1]
	if {[$n vertexrank frob3] != ($i + 1)} {
	    set c1 [expr $i + 1]
	    set c2 [$n vertexrank frob3]
	    deleteAndError $x \
		    "Vertex frob3 expected at rank $c1, got $c2"
	}
	if {![string equal $n [$n1 parent]]} {
	    deleteAndError $x \
		    "Node value of frob3 lost parent, $n vs. [$n1 parent]"
	}
	if {[$n1 parentcount] != 1} {
	    deleteAndError $x \
		    "Parent count confusion, expected 1, got [$n1 parentcount]"
	}
	if {[$n1 vertexcount] != 10} {
	    set c [$n1 vertexcount]
	    deleteAndError $x \
		    "Vertex count confusion, expected 10, got $c"
	}
    }
    $x delete
    set empty
} ""
