/**
 * PyNodeVisitor.cpp --
 *
 *	This file contains the node-visitor defintion.
 *
 *	Authors: Mike Krimerman.
 *		 hemkond@yahoo.com
 *
 * Copyright (c) 2003, Mike Krimerman.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE, EVEN IF
 * MIKE KRIMERMAN IS MADE AWARE OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "e4py.h"
#include <new>

/** Helpers */

/** PyNodeVisitor_New.
 * A helper function for object creation from C.
 */
PyObject* PyNodeVisitor_New()
{
	PyNodeVisitor* result = PyObject_NEW(PyNodeVisitor, &PyNodeVisitorType);
	if (result)
		new(&result->visitor) e4_NodeVisitor();		//Placement new
	return (PyObject*)result;
}

/** PyNodeVisitor_FromNodeVisitor.
 * A helper function for object creation from C.
 */
PyObject* PyNodeVisitor_FromNodeVisitor(e4_NodeVisitor visitor)
{
	PyNodeVisitor* result = PyObject_NEW(PyNodeVisitor, &PyNodeVisitorType);
	if (result)
		new(&result->visitor) e4_NodeVisitor(visitor);		//Placement new
	return (PyObject*)result;
}

/** PyNodeVisitor_AsNodeVisitor.
 *
 */
e4_NodeVisitor& PyNodeVisitor_AsNodeVisitor(PyObject* self)
{
	return ((PyNodeVisitor*)self)->visitor;
}

/** Methods */

/** PyNodeVisitor_Set.
 *
 */
static PyObject* PyNodeVisitor_Set(PyNodeVisitor* self, PyObject* args)
{
	bool success = false;
	switch (PyTuple_Size(args)) {
	case 1: {
		PyObject *obj;
		if (!PyArg_ParseTuple(args, "O", &obj))
			return 0;
		
		if (PyStorage_Check(obj)) {
			success = self->visitor.SetStorage(PyStorage_AsStorage(obj));
		} else if (PyNode_Check(obj)) {
			success = self->visitor.SetNode(PyNode_AsNode(obj));
		} else if (PyVertex_Check(obj)) {
			success = self->visitor.SetVertex(PyVertex_AsVertex(obj));
		} else {
			PyErr_SetString(PyExc_TypeError, ErrInvalidArgs);
			return 0;
		}
		break;
	}
	case 2: {
		PyObject *obj;
		int dc;
		if (!PyArg_ParseTuple(args, "O!i", &PyStorageType, &obj, &dc))
			return 0;
		success = self->visitor.SetStorage(PyStorage_AsStorage(obj), e4_DetachChoice(dc));
		break;
	}
	default:
		PyErr_SetString(PyExc_TypeError, ErrInvalidArgs);
		break;
	}

	if (!success) {
		PyErr_SetString(e4pyExc_APIFailure, "Set: Cannot set visitor");
		return 0;
	}

	Py_INCREF(Py_None);
	return Py_None;
}

/** PyNodeVisitor_Next.
 *
 */
static PyObject* PyNodeVisitor_Next(PyNodeVisitor* self)
{
	e4_Node node;
	if (!self->visitor.NextNode(node)) {
		PyErr_SetString(e4pyExc_APIFailure, "Next: No more nodes to visit");
		return 0;
	}

	return PyNode_FromNode(node);
}

/** PyNodeVisitor_CurrentAndAdvance.
 *
 */
static PyObject* PyNodeVisitor_CurrentAndAdvance(PyNodeVisitor* self)
{
	e4_Node node;
	if (!self->visitor.CurrentNodeAndAdvance(node)) {
		PyErr_SetString(e4pyExc_APIFailure, "CurrentAndAdvance: No more nodes");
		return 0;
	}

	return PyNode_FromNode(node);
}


/** Method doc */

static char doc_Set[] = "\
Set(storage[, detach-choise])\n\
Visit all attached nodes defined in the storage with an optional detached state (detached, attached, both).\n\
Set(node)\n\
Visit all nodes of the same kind defined in the storage containing the node (according to nodes attached state).\n\
Set(vertex)\n\
Visit all attached nodes and the root node defined in the storage containing the vertex.";

static char doc_Next[] = "\
Next() -> Vertex\n\
Advances to the next vertex to be visited and returns the vertex";

static char doc_CurrentAndAdvance[] = "\
CurrentAndAdvance() -> Vertex\n\
Returns the vertex currently being visited, advances to the vertex node to be visited.";

/** Method list */
static PyMethodDef PyNodeVisitorMethods[] = {
	{"Set", (PyCFunction)PyNodeVisitor_Set, METH_VARARGS, doc_Set},
	{"Next", (PyCFunction)PyNodeVisitor_Next, METH_NOARGS, doc_Next},
	{"CurrentAndAdvance", (PyCFunction)PyNodeVisitor_CurrentAndAdvance, METH_NOARGS, doc_CurrentAndAdvance},
	{0, 0, 0, 0}
};


/** Attributes */

/** PyNodeVisitor_get_current.
 *
 */
static PyObject* PyNodeVisitor_get_current(PyNodeVisitor* self, void *)
{
	e4_Node node;
	if (!self->visitor.CurrentNode(node)) {
		PyErr_SetString(e4pyExc_APIFailure, "current: No current node");
		return 0;
	}
	return PyNode_FromNode(node);
}

/** PyNodeVisitor_get_done.
 *
 */
static PyObject* PyNodeVisitor_get_done(PyNodeVisitor* self, void *)
{
	return PyInt_FromLong(self->visitor.IsDone());
}

/** PyNodeVisitor_get_valid.
 *
 */
static PyObject* PyNodeVisitor_get_valid(PyNodeVisitor* self, void *)
{
	return PyInt_FromLong(self->visitor.IsValid());
}


/** Attribute doc */

static char doc_current[] = "\
Returns the currently visited node.";

static char doc_done[] = "\
Returns true if there are no more nodes to visit.";

static char doc_valid[] = "\
Returns true if visitor has a current node being visited.";

/** Attribute list */
static PyGetSetDef PyNodeVisitorGetsets[] = {
    {"current", (getter)PyNodeVisitor_get_current, (setter)NULL, doc_current},
    {"done", (getter)PyNodeVisitor_get_done, (setter)NULL, doc_done},
    {"valid", (getter)PyNodeVisitor_get_valid, (setter)NULL, doc_valid},
    {NULL}
};


/** NodeVisitor access */

/** PyNodeVisitor_new.
 *
 */
/*static*/
PyObject* PyNodeVisitor_new(PyObject* o, PyObject* args)
{
	PyNodeVisitor* result = NULL;
	switch (PyTuple_Size(args)) {
	case 0:
		result = (PyNodeVisitor*)PyNodeVisitor_New();
		break;
	case 1: {
		PyObject *obj;
		if (!PyArg_ParseTuple(args, "O", &obj))
			return 0;
		
		if (PyNodeVisitor_Check(obj)) {
			result = (PyNodeVisitor*)PyNodeVisitor_FromNodeVisitor(PyNodeVisitor_AsNodeVisitor(obj));
		} else if (PyStorage_Check(obj)) {
			result = (PyNodeVisitor*)PyNodeVisitor_New();
			result->visitor = e4_NodeVisitor(PyStorage_AsStorage(obj));
		} else if (PyNode_Check(obj)) {
			result = (PyNodeVisitor*)PyNodeVisitor_New();
			result->visitor = e4_NodeVisitor(PyNode_AsNode(obj));
		} else if (PyVertex_Check(obj)) {
			result = (PyNodeVisitor*)PyNodeVisitor_New();
			result->visitor = e4_NodeVisitor(PyVertex_AsVertex(obj));
		} else {
			PyErr_SetString(PyExc_TypeError, ErrInvalidArgs);
			return 0;
		}
		break;
	}
	case 2: {
		PyObject *obj;
		int dc;
		if (!PyArg_ParseTuple(args, "O!i", &PyStorageType, &obj, &dc))
			return 0;
		result = (PyNodeVisitor*)PyNodeVisitor_New();
		result->visitor = e4_NodeVisitor(PyStorage_AsStorage(obj), e4_DetachChoice(dc));
		break;
	}
	default:
		PyErr_SetString(PyExc_TypeError, ErrInvalidArgs);
		break;
	}
	return (PyObject*)result;
}

/** PyNodeVisitor_dealloc.
 *
 */
static void PyNodeVisitor_dealloc(PyNodeVisitor *self)
{
	self->visitor.~e4_NodeVisitor();		//"placement" dtor
	PyObject_DEL(self);
}

/** PyNodeVisitor_compare.
 *
 */
static int PyNodeVisitor_compare(PyNodeVisitor *self, PyObject *rhs)
{
	if (!PyNodeVisitor_Check(rhs))
		return -1;
	return self->visitor == PyNodeVisitor_AsNodeVisitor(rhs) ? 0 : 1;
}

/** PyNodeVisitor_iter.
 *
 */
static PyObject* PyNodeVisitor_iter(PyNodeVisitor *self)
{
	Py_INCREF(self);
	return (PyObject*)self;
}

/** PyNodeVisitor_iternext.
 *
 */
static PyObject* PyNodeVisitor_iternext(PyNodeVisitor *self)
{
	e4_Node node;
	if (!self->visitor.CurrentNodeAndAdvance(node)) {
		PyErr_SetString(PyExc_StopIteration , "next: No more nodes");
		return 0;
	}

	return PyNode_FromNode(node);
}

/** Type doc */
static char doc_PyNodeVisitor[] = "\
The NodeVisitor class enables a program to iterate over all nodes defined in a storage.";

/** NodeVisitor type */
PyTypeObject PyNodeVisitorType = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,
	"NodeVisitor",
	sizeof(PyNodeVisitor),		/* tp_basicsize */
	0,					/* tp_itemsize */
	(destructor)PyNodeVisitor_dealloc,	/* tp_dealloc */
	0,//(printfunc)PyNodeVisitor_print, /* tp_print */
	0,					/* tp_getattr */
	0,					/* tp_setattr */
	(cmpfunc)PyNodeVisitor_compare,	/* tp_compare */
	(reprfunc)0,		/* tp_repr */
	0,					/* tp_as_number */
	0,					/* tp_as_sequence */
	0,		/* tp_as_mapping */
    0,					/* tp_hash */
    0,					/* tp_call */
    0,					/* tp_str */
    PyObject_GenericGetAttr,	/* tp_getattro */
    PyObject_GenericSetAttr,	/* tp_setattro */
    0,					/* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /* tp_flags */
    doc_PyNodeVisitor,	/* tp_doc */
    0,					/* tp_traverse */
    0,					/* tp_clear */
    0,					/* tp_richcompare */
    0,					/* tp_weaklistoffset */
    (getiterfunc)PyNodeVisitor_iter,		/* tp_iter */
    (iternextfunc)PyNodeVisitor_iternext,	/* tp_iternext */
    PyNodeVisitorMethods,		/* tp_methods */
    0,					/* tp_members */
    PyNodeVisitorGetsets,		/* tp_getset */
};

