
/*
 * bltPictureJpg.c --
 *
 * This module implements JPEG file format conversion routines for
 * the picture image type in the BLT toolkit.
 *
 *	Copyright 2003-2005 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * The JPEG reader/writer is adapted from jdatasrc.c and jdatadst.c
 * in the Independent JPEG Group (version 6b) library distribution.
 *
 *	The authors make NO WARRANTY or representation, either express
 *	or implied, with respect to this software, its quality,
 *	accuracy, merchantability, or fitness for a particular
 *	purpose.  This software is provided "AS IS", and you, its
 *	user, assume the entire risk as to its quality and accuracy.
 *
 *	This software is copyright (C) 1991-1998, Thomas G. Lane.  All
 *	Rights Reserved except as specified below.
 *
 *	Permission is hereby granted to use, copy, modify, and
 *	distribute this software (or portions thereof) for any
 *	purpose, without fee, subject to these conditions: (1) If any
 *	part of the source code for this software is distributed, then
 *	this README file must be included, with this copyright and
 *	no-warranty notice unaltered; and any additions, deletions, or
 *	changes to the original files must be clearly indicated in
 *	accompanying documentation.  (2) If only executable code is
 *	distributed, then the accompanying documentation must state
 *	that "this software is based in part on the work of the
 *	Independent JPEG Group".  (3) Permission for use of this
 *	software is granted only if the user accepts full
 *	responsibility for any undesirable consequences; the authors
 *	accept NO LIABILITY for damages of any kind.
 *
 */

#include "bltInt.h"

#ifdef HAVE_LIBJPEG

#include <bltSink.h>
#include <bltHash.h>
#include "bltPicture.h"
#include "bltPictureFormats.h"

#include <X11/Xutil.h>

typedef struct Blt_PictureStruct Picture;

#undef HAVE_STDLIB_H
#undef EXTERN
#ifdef WIN32
#define XMD_H	1
#endif
#include "jpeglib.h"
#include "jerror.h"
#include <setjmp.h>

#define JPG_BUF_SIZE  4096	/* choose an efficiently fwrite'able size */

typedef struct {
    struct jpeg_source_mgr pub;	/* public fields */

    Blt_DataSink data;		/* Collects the converted format. */
} JpgReader;

typedef struct {
    struct jpeg_destination_mgr pub; /* public fields */
    Blt_DataSink data;		/* Target stream */
    JOCTET *bytes;		/* start of buffer */
} JpgWriter;

typedef struct {
    struct jpeg_error_mgr pub;	/* "public" fields */
    jmp_buf jmpbuf;
    Tcl_DString dString;
} JpgErrorHandler;


static void
JpgErrorProc(j_common_ptr commPtr)
{
    JpgErrorHandler *errorPtr = (JpgErrorHandler *)commPtr->err;

    (*errorPtr->pub.output_message) (commPtr);
    longjmp(errorPtr->jmpbuf, 1);
}

static void
JpgMessageProc(j_common_ptr commPtr)
{
    JpgErrorHandler *errorPtr = (JpgErrorHandler *)commPtr->err;
    char mesg[JMSG_LENGTH_MAX];

    /* Create the message and append it into the dynamic string. */
    (*errorPtr->pub.format_message) (commPtr, mesg);
    Tcl_DStringAppend(&errorPtr->dString, " ", -1);
    Tcl_DStringAppend(&errorPtr->dString, mesg, -1);
}

static void
JpgInitSource(j_decompress_ptr commPtr)
{
    JpgReader *readerPtr = (JpgReader *)commPtr->src;

    readerPtr->pub.next_input_byte = Blt_SinkBuffer(readerPtr->data);
    readerPtr->pub.bytes_in_buffer = Blt_SinkMark(readerPtr->data);
}

static int
JpgFillInputBuffer(j_decompress_ptr commPtr)
{
    JpgReader *readerPtr = (JpgReader *)commPtr->src; 
    static unsigned char eoi[2] = { 0xFF, JPEG_EOI };

    /* Insert a fake EOI marker */
    readerPtr->pub.next_input_byte = eoi;
    readerPtr->pub.bytes_in_buffer = 2;
    return TRUE;
}

static void
JpgSkipInputData(j_decompress_ptr commPtr, long nBytes)
{
    if (nBytes > 0) {
	JpgReader *readerPtr = (JpgReader *)commPtr->src; 

	assert((readerPtr->pub.next_input_byte + nBytes) < 
	    (Blt_SinkBuffer(readerPtr->data) + Blt_SinkMark(readerPtr->data)));
	readerPtr->pub.next_input_byte += (size_t)nBytes;
	readerPtr->pub.bytes_in_buffer -= (size_t)nBytes;
    }
}

static void
JpgTermSource (j_decompress_ptr commPtr)
{
    /* Nothing to do. */
}

static void
JpgSetSourceFromSink(j_decompress_ptr commPtr, Blt_DataSink data)
{
    JpgReader *readerPtr;
    
    /* The source object is made permanent so that a series of JPEG
     * images can be read from the same file by calling jpeg_stdio_src
     * only before the first one.  (If we discarded the buffer at the
     * end of one image, we'd likely lose the start of the next one.)
     * This makes it unsafe to use this manager and a different source
     * manager serially with the same JPEG object.  Caveat programmer.
     */
    if (commPtr->src == NULL) {	/* first time for this JPEG object? */
	commPtr->src = (struct jpeg_source_mgr *)
	    (*commPtr->mem->alloc_small) ((j_common_ptr)commPtr, 
		JPOOL_PERMANENT, sizeof(JpgReader));
	readerPtr = (JpgReader *)commPtr->src;
    }
    readerPtr = (JpgReader *)commPtr->src;
    readerPtr->data = data;
    readerPtr->pub.init_source = JpgInitSource;
    readerPtr->pub.fill_input_buffer = JpgFillInputBuffer;
    readerPtr->pub.skip_input_data = JpgSkipInputData;
    /* use default method */
    readerPtr->pub.resync_to_restart = jpeg_resync_to_restart; 
    readerPtr->pub.term_source = JpgTermSource;
}

static void
JpgInitDestination (j_compress_ptr commPtr)
{
    JpgWriter *writerPtr = (JpgWriter *)commPtr->dest;

    writerPtr->bytes = (JOCTET *)(*commPtr->mem->alloc_small) 
	((j_common_ptr) commPtr, JPOOL_IMAGE, JPG_BUF_SIZE * sizeof(JOCTET));
    writerPtr->pub.next_output_byte = writerPtr->bytes;
    writerPtr->pub.free_in_buffer = JPG_BUF_SIZE;
}

static int
JpgEmptyOutputBuffer(j_compress_ptr commPtr)
{
    JpgWriter *writerPtr = (JpgWriter *)commPtr->dest;

    if (!Blt_SinkAppendData(writerPtr->data, writerPtr->bytes, JPG_BUF_SIZE)) {
	ERREXIT(commPtr, 10);
    }
    writerPtr->pub.next_output_byte = writerPtr->bytes;
    writerPtr->pub.free_in_buffer = JPG_BUF_SIZE;
    return TRUE;
}

static void
JpgTermDestination (j_compress_ptr commPtr)
{
    JpgWriter *writerPtr = (JpgWriter *)commPtr->dest;
    size_t nBytes = JPG_BUF_SIZE - writerPtr->pub.free_in_buffer;
    
    /* Write any data remaining in the buffer */
    if (nBytes > 0) {
	if (!Blt_SinkAppendData(writerPtr->data, writerPtr->bytes, nBytes)) {
	    ERREXIT(commPtr, 10);
	}
    }
}

static void
JpgSetDestinationToSink(j_compress_ptr commPtr, Blt_DataSink data)
{
    JpgWriter *writerPtr;
    
    /* The destination object is made permanent so that multiple JPEG
     * images can be written to the same file without re-executing
     * jpeg_stdio_dest.  This makes it dangerous to use this manager
     * and a different destination manager serially with the same JPEG
     * object, because their private object sizes may be different.
     * Caveat programmer.
     */
    if (commPtr->dest == NULL) {	/* first time for this JPEG object? */
	commPtr->dest = (struct jpeg_destination_mgr *)
	    (*commPtr->mem->alloc_small) ((j_common_ptr)commPtr, 
			JPOOL_PERMANENT, sizeof(JpgWriter));
    }
    writerPtr = (JpgWriter *)commPtr->dest;
    writerPtr->pub.init_destination = JpgInitDestination;
    writerPtr->pub.empty_output_buffer = JpgEmptyOutputBuffer;
    writerPtr->pub.term_destination = JpgTermDestination;
    writerPtr->data = data;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_IsJpg --
 *
 *      Attempts to parse a JPG file header.
 *
 * Results:
 *      Returns 1 is the header is JPG and 0 otherwise.  Note that
 *      the validity of the header contents is not checked here.  That's
 *	done in Blt_JpgToPicture.
 *
 *----------------------------------------------------------------------
 */
int
Blt_IsJpg(Blt_DataSink data)
{
    JpgErrorHandler error;
    struct jpeg_decompress_struct cinfo;
    int bool;

    /* Step 1: allocate and initialize JPEG decompression object */
    
    /* We set up the normal JPEG error routines, then override error_exit. */
    cinfo.dct_method = JDCT_IFAST;
    cinfo.err = jpeg_std_error(&error.pub);
    error.pub.error_exit = JpgErrorProc;
    error.pub.output_message = JpgMessageProc;
    
    /* Initialize possible error message */
    bool = FALSE;
    Tcl_DStringInit(&error.dString);
    if (setjmp(error.jmpbuf)) {
	goto done;
    }
    jpeg_create_decompress(&cinfo);
    JpgSetSourceFromSink(&cinfo, data);
    bool = (jpeg_read_header(&cinfo, TRUE) == JPEG_HEADER_OK);
 done:
    jpeg_destroy_decompress(&cinfo);
    Tcl_DStringFree(&error.dString);
    return bool;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_JpgToPicture --
 *
 *      Reads a JPEG file and converts it into a picture.
 *
 * Results:
 *      The picture is returned.  If an error occured, such
 *	as the designated file could not be opened, NULL is returned.
 *
 *----------------------------------------------------------------------
 */
Blt_Picture
Blt_JpgToPicture(
    Tcl_Interp *interp, 	/* Interpreter to report errors back to. */
    char *fileName,		/* Name of file used to fill the data sink.  */
    Blt_DataSink data,		/* Contents of the above file. */
    Blt_PictureImportSwitches *switchesPtr)
{

    JSAMPLE **rows;
    JpgErrorHandler error;
    Picture *destPtr;
    int samplesPerRow;
    struct jpeg_decompress_struct cinfo;
    unsigned int width, height;

    Tcl_SetErrorCode(interp, "", (char *)NULL);
    destPtr = NULL;

    /* Step 1: allocate and initialize JPEG decompression object */

    /* We set up the normal JPEG error routines, then override
     * error_exit. */
    cinfo.dct_method = (switchesPtr->fast) ? JDCT_IFAST : JDCT_ISLOW;
    cinfo.err = jpeg_std_error(&error.pub);
    error.pub.error_exit = JpgErrorProc;
    error.pub.output_message = JpgMessageProc;

    /* Initialize possible error message */
    Tcl_DStringInit(&error.dString);
    Tcl_DStringAppend(&error.dString, "error reading \"", -1);
    Tcl_DStringAppend(&error.dString, fileName, -1);
    Tcl_DStringAppend(&error.dString, "\": ", -1);

    if (setjmp(error.jmpbuf)) {
	jpeg_destroy_decompress(&cinfo);
	Tcl_DStringResult(interp, &error.dString);
	return NULL;
    }
    jpeg_create_decompress(&cinfo);
    JpgSetSourceFromSink(&cinfo, data);

    jpeg_read_header(&cinfo, TRUE);	/* Step 3: read file parameters */

    jpeg_start_decompress(&cinfo);	/* Step 5: Start decompressor */
    width = cinfo.output_width;
    height = cinfo.output_height;
    if ((width < 1) || (height < 1)) {
	Tcl_AppendResult(interp, "error reading \"", fileName, 
		"\": bad JPEG image size", (char *)NULL);
	return NULL;
    }
    /* JSAMPLEs per row in output buffer */
    samplesPerRow = width * cinfo.output_components;

    /* Make a one-row-high sample array that will go away when done
     * with image */
    rows = (*cinfo.mem->alloc_sarray)((j_common_ptr)&cinfo, JPOOL_IMAGE, 
	samplesPerRow, 1);
    destPtr = Blt_CreatePicture(width, height);
    if (cinfo.output_components == 1) {
	Pix32 *destRowPtr;

	destRowPtr = destPtr->bits;
	while (cinfo.output_scanline < height) {
	    JSAMPLE *bp;
	    Pix32 *dp;
	    int i;

	    dp = destRowPtr;
	    jpeg_read_scanlines(&cinfo, rows, 1);
	    bp = rows[0];
	    for (i = 0; i < (int)width; i++) {
		dp->Red = dp->Green = dp->Blue = *bp++;
		dp->Alpha = ALPHA_OPAQUE;
		dp++;
	    }
	    destRowPtr += destPtr->pixelsPerRow;
	}
    } else {
	Pix32 *destRowPtr;

	destRowPtr = destPtr->bits;
	while (cinfo.output_scanline < height) {
	    JSAMPLE *bp;
	    Pix32 *dp;
	    int i;
	    
	    dp = destRowPtr;
	    jpeg_read_scanlines(&cinfo, rows, 1);
	    bp = rows[0];
	    for (i = 0; i < (int)width; i++) {
		dp->Red = *bp++;
		dp->Green = *bp++;
		dp->Blue = *bp++;
		dp->Alpha = ALPHA_OPAQUE;
		dp++;
	    }
	    destRowPtr += destPtr->pixelsPerRow;
	}
	destPtr->flags |= BLT_PICTURE_COLOR;
    }
    jpeg_finish_decompress(&cinfo);
    jpeg_destroy_decompress(&cinfo);

    if (error.pub.num_warnings > 0) {
	Tcl_SetErrorCode(interp, "PICTURE", "JPG_READ_WARNINGS", 
		Tcl_DStringValue(&error.dString), (char *)NULL);
    }
    Tcl_DStringFree(&error.dString);
    return destPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_PictureToJpg --
 *
 *      Writes a JPEG format image to the provided data buffer.
 *
 * Results:
 *      A standard Tcl result.  If an error occured, TCL_ERROR is
 *	returned and an error message will be place in the interpreter
 *	result. Otherwise, the data sink will contain the binary
 *	output of the image.
 *
 * Side Effects:
 *	Memory is allocated for the data sink.
 *
 *----------------------------------------------------------------------
 */
int
Blt_PictureToJpg(
    Tcl_Interp *interp, 	/* Interpreter to report errors back to. */
    Blt_Picture picture,	/* Picture source. */
    Blt_DataSink dest,		/* Destination buffer to contain the
				 * JPEG image.  */
    Blt_PictureExportSwitches *switchesPtr)
{
    JpgErrorHandler error;
    int result, nColors;
    struct jpeg_compress_struct cinfo;
    Picture *srcPtr;

    Tcl_SetErrorCode(interp, "", (char *)NULL);
    if (switchesPtr->quality == 0) {
	switchesPtr->quality = 75; /* Default quality setting. */
    } else if (switchesPtr->quality > 100) {
	switchesPtr->quality = 100; /* Maximum quality setting. */
    }
    if (switchesPtr->smoothing > 100) {
	switchesPtr->smoothing = 100; /* Maximum smoothing setting. */
    }

    result = TCL_ERROR;
    srcPtr = picture;

    /* Step 1: allocate and initialize JPEG compression object */
    cinfo.err = jpeg_std_error(&error.pub);
    error.pub.error_exit = JpgErrorProc;
    error.pub.output_message = JpgMessageProc;

    /* Initialize possible error message */
    Tcl_DStringInit(&error.dString);
    Tcl_DStringAppend(&error.dString, "error writing jpg: ", -1);

    if (setjmp(error.jmpbuf)) {
	/* Transfer the error message to the interpreter result. */
	Tcl_DStringResult(interp, &error.dString);
	goto bad;
    }

    /* Now we can initialize the JPEG compression object. */
    jpeg_create_compress(&cinfo);
    
    /* Step 2: specify data destination (eg, a file) */
    
    JpgSetDestinationToSink(&cinfo, dest);
    
    /* Step 3: set parameters for compression */
    
    /* First we supply a description of the input image.
     * Four fields of the cinfo struct must be filled in:
     */
    cinfo.image_width = srcPtr->width;
    cinfo.image_height = srcPtr->height;

    if (!Blt_PictureIsOpaque(srcPtr)) {
	Blt_Picture background;

	/* Blend picture with solid color background. */
	background = Blt_CreatePicture(srcPtr->width, srcPtr->height);
	Blt_BlankPicture(background, &switchesPtr->bg); 
	Blt_BlendPictures(background, srcPtr);
	srcPtr = background;
    }
    nColors = Blt_QueryColors(srcPtr, (Blt_HashTable *)NULL);
    if (Blt_PictureIsColor(srcPtr)) {
	cinfo.input_components = 3; /* # of color components per pixel */
	cinfo.in_color_space = JCS_RGB;	/* colorspace of input image */
    } else {
	cinfo.input_components = 1; /* # of color components per pixel */
	cinfo.in_color_space = JCS_GRAYSCALE; /* colorspace of input image */
    }	
    jpeg_set_defaults(&cinfo);

    /* 
     * Now you can set any non-default parameters you wish to.  Here
     * we just illustrate the use of quality (quantization table)
     * scaling:
     */

    /* limit to baseline-JPEG values */
    jpeg_set_quality(&cinfo, switchesPtr->quality, TRUE);
    if (switchesPtr->flags & PICTURE_PROGRESSIVE) {
	jpeg_simple_progression(&cinfo);
    }
    if (switchesPtr->smoothing > 0) {
	cinfo.smoothing_factor = switchesPtr->smoothing;
    }
    /* Step 4: Start compressor */

    jpeg_start_compress(&cinfo, TRUE);

    /* Step 5: while (scan lines remain to be written) */
    /*           jpeg_write_scanlines(...); */
    {
	int y;
	int row_stride;
	JSAMPLE *destRow;
	Pix32 *srcRowPtr;
	JSAMPROW row_pointer[1];	/* pointer to JSAMPLE row[s] */

	/* JSAMPLEs per row in image_buffer */
	row_stride = srcPtr->width * cinfo.input_components;
	destRow = Blt_Malloc(sizeof(JSAMPLE) * row_stride);
	assert(destRow);

	srcRowPtr = srcPtr->bits;
	if (cinfo.input_components == 3) {
	    for (y = 0; y < srcPtr->height; y++) {
		Pix32 *sp, *send;
		JSAMPLE *dp;
		
		dp = destRow;
		for (sp = srcRowPtr, send = sp + srcPtr->width; sp < send; 
		     sp++) {
		    dp[0] = sp->Red;
		    dp[1] = sp->Green;
		    dp[2] = sp->Blue;
		    dp += 3;
		}
		row_pointer[0] = destRow;
		jpeg_write_scanlines(&cinfo, row_pointer, 1);
		srcRowPtr += srcPtr->pixelsPerRow;
	    }
	} else {
	    for (y = 0; y < srcPtr->height; y++) {
		Pix32 *sp, *send;
		JSAMPLE *dp;
		
		dp = destRow;
		for (sp = srcRowPtr, send = sp + srcPtr->width; sp < send; 
		     sp++) {
		    *dp++ = sp->Red;
		}
		row_pointer[0] = destRow;
		jpeg_write_scanlines(&cinfo, row_pointer, 1);
		srcRowPtr += srcPtr->pixelsPerRow;
	    }
	}
	Blt_Free(destRow);
    }
    /* Step 6: Finish compression */
    jpeg_finish_compress(&cinfo);
    result = TCL_OK;
 bad:
    /* Step 7: release JPEG compression object */
    jpeg_destroy_compress(&cinfo);

    if (error.pub.num_warnings > 0) {
	Tcl_SetErrorCode(interp, "PICTURE", "JPG_WRITE_WARNINGS", 
		Tcl_DStringValue(&error.dString), (char *)NULL);
    }
    Tcl_DStringFree(&error.dString);
    if (srcPtr != picture) {
	Blt_FreePicture(srcPtr);
    }
    return result;
}

#endif /* HAVE_LIBJPEG */

