
/*
 * bltInit.c --
 *
 * This module initials the BLT toolkit, registering its commands
 * with the Tcl/Tk interpreter.
 *
 *	Copyright 1991-2004 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "bltInt.h"
#include "bltNsUtil.h"

#define EXACT 1

#ifndef BLT_LIBRARY
#ifdef WIN32
#define BLT_LIBRARY  "c:/Program Files/Tcl/lib/blt"##BLT_VERSION
#else
#define BLT_LIBRARY "unknown"
#endif
#endif

double bltNaN;
Tcl_Obj *bltEmptyStringObjPtr;

extern void Blt_RegisterPictureImageType(void);
extern void Blt_RegisterEpsCanvasItem(void);

static Tcl_MathProc MinMathProc, MaxMathProc;
static char libPath[1024] =
{
    BLT_LIBRARY
};

/*
 * Script to set the BLT library path in the variable global "blt_library"
 *
 * Checks the usual locations for a file (bltGraph.pro) from the BLT
 * library.  The places searched in order are
 *
 *	$BLT_LIBRARY
 *	$BLT_LIBRARY/blt2.4
 *      $BLT_LIBRARY/..
 *      $BLT_LIBRARY/../blt2.4
 *	$blt_libPath
 *	$blt_libPath/blt2.4
 *      $blt_libPath/..
 *      $blt_libPath/../blt2.4
 *	$tcl_library
 *	$tcl_library/blt2.4
 *      $tcl_library/..
 *      $tcl_library/../blt2.4
 *	$env(TCL_LIBRARY)
 *	$env(TCL_LIBRARY)/blt2.4
 *      $env(TCL_LIBRARY)/..
 *      $env(TCL_LIBRARY)/../blt2.4
 *
 *  The Tcl variable "blt_library" is set to the discovered path.
 *  If the file wasn't found, no error is returned.  The actual
 *  usage of $blt_library is purposely deferred so that it can be
 *  set from within a script.
 */

/* FIXME: Change this to a namespace procedure in 3.0 */

static char initScript[] =
{"\n\
global blt_library blt_libPath blt_version tcl_library env\n\
set blt_library {}\n\
set path {}\n\
foreach var { env(BLT_LIBRARY) blt_libPath tcl_library env(TCL_LIBRARY) } { \n\
    if { ![info exists $var] } { \n\
        continue \n\
    } \n\
    set path [set $var] \n\
    if { [file readable [file join $path bltGraph.pro]] } { \n\
        set blt_library $path\n\
        break \n\
    } \n\
    set path [file join $path blt$blt_version ] \n\
    if { [file readable [file join $path bltGraph.pro]] } { \n\
        set blt_library $path\n\
        break \n\
    } \n\
    set path [file dirname [set $var]] \n\
    if { [file readable [file join $path bltGraph.pro]] } { \n\
        set blt_library $path\n\
        break \n\
    } \n\
    set path [file join $path blt$blt_version ] \n\
    if { [file readable [file join $path bltGraph.pro]] } { \n\
        set blt_library $path\n\
        break \n\
    } \n\
} \n\
if { $blt_library != \"\" } { \n\
    global auto_path \n\
    lappend auto_path $blt_library \n\
}\n\
unset var path\n\
\n"
};


static Tcl_AppInitProc *tclCmds[] =
{
#ifndef NO_BGEXEC
    Blt_BgexecInit,
#endif
#ifndef NO_DEBUG
    Blt_DebugInit,
#endif
#ifndef NO_WATCH
    Blt_WatchInit,
#endif
#ifndef NO_VECTOR
    Blt_VectorInit,
#endif
#ifndef NO_SPLINE
    Blt_SplineInit,
#endif
#ifndef NO_TREE
    Blt_TreeInit,
#endif
#ifndef NO_DATATABLE
    Blt_DataTableInit,
#endif
#ifndef NO_DDE
    Blt_DdeInit,
#endif
#ifndef NO_CRC32
    Blt_Crc32Init,
#endif
#ifndef NO_CSV
    Blt_CsvInit,
#endif
    (Tcl_AppInitProc *) NULL
};

#ifndef TCL_ONLY
static Tcl_AppInitProc *tkCmds[] =
{
    Blt_BgStyleInit,
#ifndef NO_GRAPH
    Blt_GraphInit,
#endif
#ifndef NO_PICTURE
    Blt_PictureInit,
#endif
#ifndef NO_TABLE
    Blt_TableInit,
#endif
#ifndef NO_TABSET
    Blt_TabsetInit,
#endif
#ifndef NO_TABNOTEBOOK
    Blt_TabnotebookInit,
#endif
#ifndef NO_HTEXT
    Blt_HtextInit,
#endif
#ifndef NO_BUSY
    Blt_BusyInit,
#endif
#ifndef NO_WINOP
    Blt_WinopInit,
#endif
#ifndef NO_BITMAP
    Blt_BitmapInit,
#endif
#ifndef NO_DRAGDROP
    Blt_DragDropInit,
#endif
#ifndef NO_DND
    Blt_DndInit,
#endif
#ifndef NO_CONTAINER
    Blt_ContainerInit,
#endif
#ifndef NO_BELL
    Blt_BeepInit,
#endif
#ifndef NO_CUTBUFFER
    Blt_CutbufferInit,
#endif
#ifndef NO_PRINTER
    Blt_PrinterInit,
#endif
#ifndef NO_TILEFRAME
    Blt_FrameInit,
#endif
#ifndef NO_TILEBUTTON
    Blt_ButtonInit,
#endif
#ifndef NO_TILESCROLLBAR
    Blt_ScrollbarInit,
#endif
#ifndef NO_TREEVIEW
    Blt_TreeViewInit,
#endif
#if (BLT_MAJOR_VERSION > 3)
#ifndef NO_MOUNTAIN
    Blt_MountainInit,
#endif
#endif
#ifndef NO_TED
    Blt_TedInit,
#endif
    (Tcl_AppInitProc *) NULL
};
#endif /* TCL_ONLY */

#ifdef WIN32
/*
 *----------------------------------------------------------------------
 *
 * DllMain --
 *
 *	This wrapper function is used by Windows to invoke the
 *	initialization code for the DLL.
 *
 * Results:
 *	Returns TRUE;
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
BOOL APIENTRY
DllMain(
    HINSTANCE hInst,		/* Library instance handle. */
    DWORD reason,		/* Reason this function is being called. */
    LPVOID reserved)		/* Not used. */
{
    return TRUE;
}

BOOL APIENTRY
DllEntryPoint(hInst, reason, reserved)
    HINSTANCE hInst;            /* Library instance handle. */
    DWORD reason;               /* Reason this function is being called. */
    LPVOID reserved;            /* Not used. */
{
    return DllMain(hInst, reason, reserved);
}
#endif /* WIN32 */


static double
MakeNaN(void)
{
    union DoubleValue {
	unsigned int words[2];
	double value;
    } result;

#ifdef WORDS_BIGENDIAN
    result.words[0] = 0x7ff80000;
    result.words[1] = 0x00000000;
#else
    result.words[0] = 0x00000000;
    result.words[1] = 0x7ff80000;
#endif
    return result.value;
}

#if 0
#ifdef __BORLANDC__
static double
MakeNaN(void)
{
    union DoubleValue {
	unsigned int words[2];
	double value;
    } result;

#ifdef WORDS_BIGENDIAN
    result.words[0] = 0x00000000;
    result.words[1] = 0x7ff80000;
#else
    result.words[0] = 0x7ff80000;
    result.words[1] = 0x00000000;
#endif
    return real.value;
}
#endif /* __BORLANDC__ */

#ifdef _MSC_VER
static double
MakeNaN(void)
{
    return sqrt(-1.0);	/* Generate IEEE 754 Quiet Not-A-Number. */
}
#endif /* _MSC_VER */

#if !defined(__BORLANDC__) && !defined(_MSC_VER)

static double
MakeNaN(void)
{
    union DoubleValue {
	unsigned char bytes[sizeof(double)];
	double value;
    } result;
    int i, j;
    static unsigned char quietNaN[] = {
	0x7f, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
    };
#ifdef WORDS_BIGENDIAN
    for (i = j = 0; i < sizeof(double); i++, j++) {
	result.bytes[i] = quietNaN[j];
    }
#else 
    for (i = 0, j = sizeof(double) - 1; i < sizeof(double); i++, j--) {
	result.bytes[i] = quietNaN[j];
    }
#endif
    return result.value;
}

#ifdef notdef
static double
MakeNaN(void)
{
    return 0.0 / 0.0;		/* Generate IEEE 754 Not-A-Number. */
}
#endif

#endif /* !__BORLANDC__  && !_MSC_VER */
#endif /*0*/

/* ARGSUSED */
static int
MinMathProc(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tcl_Value *argsPtr,
    Tcl_Value *resultPtr)
{
    Tcl_Value *op1Ptr, *op2Ptr;

    op1Ptr = argsPtr, op2Ptr = argsPtr + 1;
    if ((op1Ptr->type == TCL_INT) && (op2Ptr->type == TCL_INT)) {
	resultPtr->intValue = MIN(op1Ptr->intValue, op2Ptr->intValue);
	resultPtr->type = TCL_INT;
    } else {
	double a, b;

	a = (op1Ptr->type == TCL_INT) 
	    ? (double)op1Ptr->intValue : op1Ptr->doubleValue;
	b = (op2Ptr->type == TCL_INT)
	    ? (double)op2Ptr->intValue : op2Ptr->doubleValue;
	resultPtr->doubleValue = MIN(a, b);
	resultPtr->type = TCL_DOUBLE;
    }
    return TCL_OK;
}

/*ARGSUSED*/
static int
MaxMathProc(
    ClientData clientData,	/* Not Used. */
    Tcl_Interp *interp,
    Tcl_Value *argsPtr,
    Tcl_Value *resultPtr)
{
    Tcl_Value *op1Ptr, *op2Ptr;

    op1Ptr = argsPtr, op2Ptr = argsPtr + 1;
    if ((op1Ptr->type == TCL_INT) && (op2Ptr->type == TCL_INT)) {
	resultPtr->intValue = MAX(op1Ptr->intValue, op2Ptr->intValue);
	resultPtr->type = TCL_INT;
    } else {
	double a, b;

	a = (op1Ptr->type == TCL_INT)
	    ? (double)op1Ptr->intValue : op1Ptr->doubleValue;
	b = (op2Ptr->type == TCL_INT)
	    ? (double)op2Ptr->intValue : op2Ptr->doubleValue;
	resultPtr->doubleValue = MAX(a, b);
	resultPtr->type = TCL_DOUBLE;
    }
    return TCL_OK;
}

static int
SetLibraryPath(Tcl_Interp *interp)
{
    Tcl_DString dString;
    CONST char *value;

    Tcl_DStringInit(&dString);
    Tcl_DStringAppend(&dString, libPath, -1);
#ifdef WIN32
    {
	HKEY key;
	DWORD result;
#ifndef BLT_REGISTRY_KEY
#define BLT_REGISTRY_KEY "Software\\BLT\\" BLT_VERSION "\\" TCL_VERSION
#endif
	result = RegOpenKeyEx(
	      HKEY_LOCAL_MACHINE, /* Parent key. */
	      BLT_REGISTRY_KEY,	/* Path to sub-key. */
	      0,		/* Reserved. */
	      KEY_READ,		/* Security access mask. */
	      &key);		/* Resulting key.*/

	if (result == ERROR_SUCCESS) {
	    DWORD size;

	    /* Query once to get the size of the string needed */
	    result = RegQueryValueEx(key, "BLT_LIBRARY", NULL, NULL, NULL, 
		     &size);
	    if (result == ERROR_SUCCESS) {
		Tcl_DStringSetLength(&dString, size);
		/* And again to collect the string. */
		RegQueryValueEx(key, "BLT_LIBRARY", NULL, NULL,
				(LPBYTE)Tcl_DStringValue(&dString), &size);
		RegCloseKey(key);
	    }
	}
    }
#endif /* WIN32 */
    value = Tcl_SetVar(interp, "blt_libPath", Tcl_DStringValue(&dString),
	TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG);
    Tcl_DStringFree(&dString);
    if (value == NULL) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*LINTLIBRARY*/
EXPORT int
Blt_Init(Tcl_Interp *interp)	/* Interpreter to add extra commands */
{
    long flags;

#define BLT_THREAD_KEY		"BLT Initialized"
#define BLT_TCL_CMDS		(1<<0)
#define BLT_TK_CMDS		(1<<1)
#ifdef USE_TCL_STUBS
    if (Tcl_InitStubs(interp, TCL_VERSION, 1) == NULL) {
	return TCL_ERROR;
    };
#endif
    Blt_AllocInit(NULL, NULL, NULL);
    flags = (long)Tcl_GetAssocData(interp, BLT_THREAD_KEY, NULL);
    if ((flags & BLT_TCL_CMDS) == 0) {
	Tcl_AppInitProc **p;
	Tcl_Namespace *nsPtr;
	Tcl_ValueType args[2];

	/*
	 * Check that the versions of Tcl that have been loaded are
	 * the same ones that BLT was compiled against.
	 */
	if (Tcl_PkgRequire(interp, "Tcl", TCL_VERSION, EXACT) == NULL) {
	    return TCL_ERROR;
	}
	/* Set the "blt_version", "blt_patchLevel", and "blt_libPath" Tcl
	 * variables. We'll use them in the following script. */
	if ((Tcl_SetVar(interp, "blt_version", BLT_VERSION, 
			TCL_GLOBAL_ONLY) == NULL) ||
	    (Tcl_SetVar(interp, "blt_patchLevel", BLT_PATCH_LEVEL, 
			TCL_GLOBAL_ONLY) == NULL)) {
	    return TCL_ERROR;
	}
	if (SetLibraryPath(interp) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (Tcl_Eval(interp, initScript) != TCL_OK) {
	    return TCL_ERROR;
	}
	nsPtr = Tcl_CreateNamespace(interp, "blt", NULL, NULL);
	if (nsPtr == NULL) {
	    return TCL_ERROR;
	}
	/* Initialize the BLT commands that only require Tcl. */
	for (p = tclCmds; *p != NULL; p++) {
	    if ((**p) (interp) != TCL_OK) {
		Tcl_DeleteNamespace(nsPtr);
		return TCL_ERROR;
	    }
	}
	args[0] = args[1] = TCL_EITHER;
	Tcl_CreateMathFunc(interp, "min", 2, args, MinMathProc, (ClientData)0);
	Tcl_CreateMathFunc(interp, "max", 2, args, MaxMathProc, (ClientData)0);
	Blt_RegisterArrayObj(interp);
	bltEmptyStringObjPtr = Tcl_NewStringObj("", -1);
	Tcl_IncrRefCount(bltEmptyStringObjPtr);
	bltNaN = MakeNaN();
	if (Tcl_PkgProvide(interp, "BLT", BLT_VERSION) != TCL_OK) {
	    return TCL_ERROR;
	}
	Tcl_SetAssocData(interp, BLT_THREAD_KEY, NULL, 
		(ClientData)(flags | BLT_TCL_CMDS));
    }
#ifndef TCL_ONLY
    if ((flags & BLT_TK_CMDS) == 0) {
	Tcl_AppInitProc **p;
	Tcl_Namespace *nsPtr;

#if (TCL_VERSION_NUMBER >= _VERSION(8,1,0)) 
#ifdef USE_TK_STUBS
	if (Tk_InitStubs(interp, TK_VERSION, 1) == NULL) {
	    return TCL_ERROR;
	};
#endif
	if (Tcl_PkgPresent(interp, "Tk", TK_VERSION, EXACT) == NULL) {
	    return TCL_OK;
	} 
#else
	if (Tcl_PkgRequire(interp, "Tk", TK_VERSION, EXACT) == NULL) {
	    Tcl_ResetResult(interp);
	    return TCL_OK;
	} 
#endif
	nsPtr = Tcl_CreateNamespace(interp, "blt::tile", NULL, 
			    (Tcl_NamespaceDeleteProc *) NULL);
	if (nsPtr == NULL) {
	    return TCL_ERROR;
	}
	nsPtr = Tcl_FindNamespace(interp, "blt", (Tcl_Namespace *)NULL, 
		TCL_LEAVE_ERR_MSG);
	if (nsPtr == NULL) {
	    return TCL_ERROR;
	}
	/* Initialize the BLT commands that only use Tk too. */
	for (p = tkCmds; *p != NULL; p++) {
	    if ((**p) (interp) != TCL_OK) {
		Tcl_DeleteNamespace(nsPtr);
		return TCL_ERROR;
	    }
	}
#ifdef HAVE_X86_ASM
	Blt_CheckForMMX();
#endif /* HAVE_X86_ASM */
	Blt_RegisterPictureImageType();
	Blt_RegisterEpsCanvasItem();
	Tcl_SetAssocData(interp, BLT_THREAD_KEY, NULL, 
		(ClientData)(flags | BLT_TK_CMDS));
    }
#endif
    return TCL_OK;
}

/*LINTLIBRARY*/
EXPORT int
Blt_SafeInit(Tcl_Interp *interp) /* Interpreter to add extra commands */
{
    return Blt_Init(interp);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_InitCmd --
 *
 *      Given the name of a command, return a pointer to the
 *      clientData field of the command.
 *
 * Results:
 *      A standard TCL result. If the command is found, TCL_OK
 *	is returned and clientDataPtr points to the clientData
 *	field of the command (if the clientDataPtr in not NULL).
 *
 * Side effects:
 *      If the command is found, clientDataPtr is set to the address
 *	of the clientData of the command.  If not found, an error
 *	message is left in interp->result.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
int
Blt_InitCmd(Tcl_Interp *interp, char *nsName, Blt_InitCmdSpec *specPtr)
{
    char *cmdPath;
    Tcl_DString dString;
    Tcl_Command cmdToken;
    Tcl_Namespace *nsPtr;

    Tcl_DStringInit(&dString);
    if (nsName != NULL) {
	Tcl_DStringAppend(&dString, nsName, -1);
    }
    Tcl_DStringAppend(&dString, "::", -1);
    Tcl_DStringAppend(&dString, specPtr->name, -1);

    cmdPath = Tcl_DStringValue(&dString);
    cmdToken = Tcl_FindCommand(interp, cmdPath, (Tcl_Namespace *)NULL, 0);
    if (cmdToken != NULL) {
	Tcl_DStringFree(&dString);
	return TCL_OK;		/* Assume command was already initialized */
    }
    cmdToken = Tcl_CreateObjCommand(interp, cmdPath, 
	(Tcl_ObjCmdProc *)specPtr->cmdProc, specPtr->clientData, 
		specPtr->cmdDeleteProc);
    Tcl_DStringFree(&dString);

    nsPtr = Tcl_FindNamespace(interp, nsName, (Tcl_Namespace *)NULL,
	      TCL_LEAVE_ERR_MSG);
    if (nsPtr == NULL) {
	return TCL_ERROR;
    }
    if (Tcl_Export(interp, nsPtr, specPtr->name, FALSE) != TCL_OK) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_InitCmds --
 *
 *      Given the name of a command, return a pointer to the
 *      clientData field of the command.
 *
 * Results:
 *      A standard TCL result. If the command is found, TCL_OK is
 *      returned and clientDataPtr points to the clientData field of
 *      the command (if the clientDataPtr in not NULL).
 *
 * Side effects:
 *      If the command is found, clientDataPtr is set to the address
 *      of the clientData of the command.  If not found, an error
 *      message is left in interp->result.
 *
 *----------------------------------------------------------------------
 */
int
Blt_InitCmds(
    Tcl_Interp *interp, 
    char *nsName, 
    Blt_InitCmdSpec *specs, 
    int nCmds)
{
    Blt_InitCmdSpec *sp, *endPtr;

    for (sp = specs, endPtr = specs + nCmds; sp < endPtr; sp++) {
	if (Blt_InitCmd(interp, nsName, sp) != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * Tcl_AppInit --
 *
 *	This procedure performs application-specific initialization.
 *	Most applications, especially those that incorporate additional
 *	packages, will have their own version of this procedure.
 *
 * Results:
 *	Returns a standard Tcl completion code, and leaves an error
 *	message in interp->result if an error occurs.
 *
 * Side effects:
 *	Depends on the startup script.
 *
 *----------------------------------------------------------------------
 */

int
Blt_AppInit(Tcl_Interp *interp)	/* Interpreter for application. */
{
#ifdef TCLLIBPATH
    /* 
     * It seems that some distributions of Tcl don't compile-in a
     * default location of the library.  This causes Tcl_Init to fail
     * if bltwish and bltsh are moved to another directory. The
     * workaround is to set the magic variable "tclDefaultLibrary".
     */
    Tcl_SetVar(interp, "tclDefaultLibrary", TCLLIBPATH, TCL_GLOBAL_ONLY);
#endif /* TCLLIBPATH */
    if (Tcl_Init(interp) == TCL_ERROR) {
	return TCL_ERROR;
    }
#ifndef TCL_ONLY
    if (Tk_Init(interp) == TCL_ERROR) {
	return TCL_ERROR;
    }
#endif /* TCL_ONLY */
    /*
     * Call the init procedures for included packages.  Each call should
     * look like this:
     *
     * if (Mod_Init(interp) == TCL_ERROR) {
     *     return TCL_ERROR;
     * }
     *
     * where "Mod" is the name of the module.
     */
    if (Blt_Init(interp) == TCL_ERROR) {
	return TCL_ERROR;
    }
    Tcl_StaticPackage(interp, "BLT", Blt_Init, Blt_SafeInit);
    /*

     * Call Tcl_CreateCommand for application-specific commands, if
     * they weren't already created by the init procedures called above.
     */

    /*
     * Specify a user-specific startup file to invoke if the application
     * is run interactively.  Typically the startup file is "~/.apprc"
     * where "app" is the name of the application.  If this line is deleted
     * then no user-specific startup file will be run under any conditions.
     */

    Tcl_SetVar(interp, "tcl_rcFileName", "~/.wishrc", TCL_GLOBAL_ONLY);
    return TCL_OK;
}
