#include "bltInt.h"

#ifndef linux
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif /* HAVE_MALLOC_H */
#endif

/*
 *	Memory allocation/deallocation in BLT is performed via the
 *	global variables bltMallocPtr, bltFreePtr, and bltReallocPtr.
 *	By default, they point to the same routines that Tcl uses.
 *	The routine Blt_AllocInit allows you to specify your own
 *	memory allocation/deallocation routines for BLT on a
 *	library-wide basis.
 */
#ifdef WIN32

#ifdef __GNUC__
#ifndef USE_TCL_STUBS
extern char *Tcl_Alloc _ANSI_ARGS_((unsigned int size));
extern void Tcl_Free _ANSI_ARGS_((char * ptr));
extern char *Tcl_Realloc _ANSI_ARGS_((char *ptr, unsigned int size));
#endif
#endif /*__GNUC__*/

#else

#if (TCL_VERSION_NUMBER >= _VERSION(8,1,0)) 
#ifndef USE_TCL_STUBS
EXTERN Blt_MallocProc TclpAlloc;
EXTERN Blt_FreeProc TclpFree;
EXTERN Blt_ReallocProc TclpRealloc;
#endif
#else
#if !HAVE_DECL_FREE
extern void free (void *);
#endif
#endif /* >= 8.1.0 */
#endif /* WIN32 */

Blt_MallocProc *bltMallocPtr;
Blt_ReallocProc *bltReallocPtr;
Blt_FreeProc *bltFreePtr;

void *
Blt_Calloc(unsigned int n, size_t size)
{
    void *ptr;
    size_t nBytes;

    nBytes = n * size;
    ptr = Blt_Malloc(nBytes);
    if (ptr != NULL) {
	memset(ptr, 0, nBytes);
    }
    return ptr;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_Strdup --
 *
 *      Create a copy of the string from heap storage.
 *
 * Results:
 *      Returns a pointer to the need string copy.
 *
 *----------------------------------------------------------------------
 */
char *
Blt_Strdup(CONST char *string)
{
    size_t size;
    char *ptr;

    size = strlen(string) + 1;
    ptr = Blt_Malloc(size * sizeof(char));
    if (ptr != NULL) {
	strcpy(ptr, string);
    }
    return ptr;
}


void
Blt_AllocInit(
    Blt_MallocProc *mallocProc,
    Blt_ReallocProc *reallocProc,
    Blt_FreeProc *freeProc)
{
    Blt_MallocProc *defMallocProc;
    Blt_FreeProc *defFreeProc;
    Blt_ReallocProc *defReallocProc;
    
    /* 
     * Try to use the same memory allocator/deallocator that Tcl is
     * using. Before 8.1 it used malloc/free.
     */
#ifdef WIN32
    defMallocProc = (Blt_MallocProc *)Tcl_Alloc;
    defFreeProc = (Blt_FreeProc *)Tcl_Free; 
    defReallocProc = (Blt_ReallocProc *)Tcl_Realloc; 
#else
#if (TCL_VERSION_NUMBER >= _VERSION(8,1,0)) 
    /* 
     * We're pointing to the private TclpAlloc/TclpFree instead of public
     * Tcl_Alloc/Tcl_Free routines because they don't automatically
     * trigger a panic when not enough memory is available. There are
     * cases (such as allocating a very large vector) where an
     * out-of-memory error is recoverable.
     */
    defMallocProc = (Blt_MallocProc *)TclpAlloc;
    defFreeProc = (Blt_FreeProc *)TclpFree; 
    defReallocProc = (Blt_ReallocProc *)TclpRealloc; 
#else 
    defMallocProc = malloc;
    defFreeProc = free; 
    defReallocProc = realloc;
#endif /* >= 8.1.0 */
#endif /* WIN32 */
    if (bltMallocPtr == NULL) {
	bltMallocPtr = (mallocProc != NULL) ? mallocProc : defMallocProc;
    }
    if (bltFreePtr == NULL) {
	bltFreePtr = (freeProc != NULL) ? freeProc : defFreeProc;
    }
    if (bltReallocPtr == NULL) {
	bltReallocPtr = (reallocProc != NULL) ? reallocProc : defReallocProc;
    }
}

