/* 
 * tkCanvPgraph.c --
 *
 *	This file implements polar graph items for canvas
 *	widgets.
 *
 */

#include <stdio.h>
#include "tk.h"
#include "tkInt.h"
#include "tkPort.h"
#include "tkCanvas.h"
#include "tkCanvPgraph.h"

#ifdef USE_DMALLOC
#include<dmalloc.h>
#endif

/*
 * The structure below defines the record for each circle item.
 *
 * It's now located in tkCanvPhgraph.h
 */

/*
 * Prototypes for procedures defined in this file:
 */

static void		ComputeCircleBbox _ANSI_ARGS_((Tk_Canvas canvas,
			    PGraphItem *graphPtr));
static int		ConfigureCircle _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int argc,
			    char **argv, int flags));
static int		CreateCircle _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, struct Tk_Item *itemPtr,
			    int argc, char **argv));
static void		DeleteCircle _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, Display *display));
static void		DisplayCircle _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, Display *display, Drawable dst,
			    int x, int y, int width, int height));
static int		CircleToArea _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double *areaPtr));
static double		CircleToPoint _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double *pointPtr));
static int		CircleCoords _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int argc,
			    char **argv));
static int		CircleToPostscript _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int prepass));
static int		AxisToPostscript _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int prepass));
static void		ScaleCircle _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double originX, double originY,
			    double scaleX, double scaleY));
static void		TranslateCircle _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double deltaX, double deltaY));
/*
static int		CircleAccept _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Visitor *visitorPtr,
			    Tk_Item *itemPtr));
*/
static int		ParseArrowShape _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *interp, Tk_Window tkwin, char *value,
			    char *recordPtr, int offset));
static char *		PrintArrowShape _ANSI_ARGS_((ClientData clientData,
			    Tk_Window tkwin, char *recordPtr, int offset,
			    Tcl_FreeProc **freeProcPtr));
static int		ConfigureArrows _ANSI_ARGS_((Tk_Canvas canvas,
			    PGraphItem *linePtr));
static int		ArrowheadPostscript _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, PGraphItem *graphPtr,
			    double *arrowPtr));
static int		GraphUpdateGC _ANSI_ARGS_((Tk_Window tkwin,
			    PGraphItem *graphPtr));
static int		GraphUpdateDrawable _ANSI_ARGS_((Tk_Window tkwin,
			    PGraphItem *graphPtr));
static int		GraphCreateDrawable _ANSI_ARGS_((Tk_Window tkwin,
			    PGraphItem *graphPtr));

#define PTS_IN_ARROW 6


/*
 * Information used for parsing configuration specs:
 */

static Tk_CustomOption arrowShapeOption = {
    (Tk_OptionParseProc *) ParseArrowShape,
    PrintArrowShape, (ClientData) NULL
};
static Tk_CustomOption stateOption = {
    (Tk_OptionParseProc *) Tk_StateParseProc,
    Tk_StatePrintProc, (ClientData) 2
};
static Tk_CustomOption tagsOption = {
    (Tk_OptionParseProc *) Tk_CanvasTagsParseProc,
    Tk_CanvasTagsPrintProc, (ClientData) NULL
};
static Tk_CustomOption dashOption = {
    (Tk_OptionParseProc *) Tk_CanvasDashParseProc,
    Tk_CanvasDashPrintProc, (ClientData) NULL
};
static Tk_CustomOption tileOption = {
    (Tk_OptionParseProc *) Tk_TileParseProc,
    Tk_TilePrintProc, (ClientData) NULL
};
static Tk_CustomOption pixelOption = {
    (Tk_OptionParseProc *) Tk_PixelParseProc,
    Tk_PixelPrintProc, (ClientData) NULL
};

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_CUSTOM, "-activedash", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PGraphItem, outline.activeDash),
	TK_CONFIG_NULL_OK, &dashOption},
    {TK_CONFIG_COLOR, "-activefill", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PGraphItem, activeFillColor),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-activeoutline", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PGraphItem, outline.activeColor),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-axiscolor", (char *) NULL, (char *) NULL,
	(char *)NULL, Tk_Offset(PGraphItem, axis_outline.color),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-activeoutlinestipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PGraphItem, outline.activeStipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-activeoutlinetile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PGraphItem, outline.activeTile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_BITMAP, "-activestipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PGraphItem, activeFillStipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-activetile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PGraphItem, activeFillTile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_CUSTOM, "-activewidth", (char *) NULL, (char *) NULL,
	"0.0", Tk_Offset(PGraphItem, outline.activeWidth),
	TK_CONFIG_DONT_SET_DEFAULT, &pixelOption},
    {TK_CONFIG_BOOLEAN, "-arrow", (char *) NULL, (char *) NULL,
	"1", Tk_Offset(PGraphItem, arrowon),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-arrowshape", (char *) NULL, (char *) NULL,
	"5 5 2", Tk_Offset(PGraphItem, arrowShapeA),
	TK_CONFIG_DONT_SET_DEFAULT, &arrowShapeOption},
    {TK_CONFIG_PIXELS, "-axislength", (char *) NULL, (char *) NULL,
	"20", Tk_Offset(PGraphItem, axislength),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-axiswidth", (char *) NULL, (char *) NULL,
	"1.0", Tk_Offset(PGraphItem, axis_outline.width),
	TK_CONFIG_DONT_SET_DEFAULT, &pixelOption},
    {TK_CONFIG_CUSTOM, "-dash", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PGraphItem, outline.dash),
	TK_CONFIG_NULL_OK, &dashOption},
    {TK_CONFIG_PIXELS, "-dashoffset", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(PGraphItem, outline.offset),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-disableddash", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PGraphItem, outline.disabledDash),
	TK_CONFIG_NULL_OK, &dashOption},
    {TK_CONFIG_COLOR, "-disabledfill", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PGraphItem, disabledFillColor),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-disabledoutline", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PGraphItem, outline.disabledColor),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-disabledoutlinestipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PGraphItem, outline.disabledStipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-disabledoutlinetile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PGraphItem, outline.disabledTile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_BITMAP, "-disabledstipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PGraphItem, disabledFillStipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-disabledtile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PGraphItem, disabledFillTile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_CUSTOM, "-disabledwidth", (char *) NULL, (char *) NULL,
	"0.0", Tk_Offset(PGraphItem, outline.disabledWidth),
	TK_CONFIG_DONT_SET_DEFAULT, &pixelOption},
    {TK_CONFIG_COLOR, "-fill", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PGraphItem, fillColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-outline", (char *) NULL, (char *) NULL,
	"black", Tk_Offset(PGraphItem, outline.color), TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-outlinestipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PGraphItem, outline.stipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-outlinetile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PGraphItem, outline.tile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_DOUBLE, "-poledir", (char *) NULL, (char *) NULL,
	"0.0", Tk_Offset(PGraphItem, poledir),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_DOUBLE, "-rmax", (char *) NULL, (char *) NULL,
	"1.0", Tk_Offset(PGraphItem, rmax),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_INT, "-sense", (char *) NULL, (char *) NULL,
	"1", Tk_Offset(PGraphItem, sense),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-state", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(Tk_Item, state),TK_CONFIG_NULL_OK,
	&stateOption},
    {TK_CONFIG_BITMAP, "-stipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PGraphItem, fillStipple), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &tagsOption},
    {TK_CONFIG_CUSTOM, "-tile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PGraphItem, fillTile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_STRING, "-updatecommand", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(Tk_Item, updateCmd), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-width", (char *) NULL, (char *) NULL,
	"1.0", Tk_Offset(PGraphItem, outline.width),
	TK_CONFIG_DONT_SET_DEFAULT, &pixelOption},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * The structures below defines the circle item types
 * by means of procedures that can be invoked by generic item code.
 */

Tk_ItemType tkPGraphType = {
    "pgraph",				/* name */
    sizeof(PGraphItem),		/* itemSize */
    CreateCircle,			/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureCircle,			/* configureProc */
    CircleCoords,			/* coordProc */
    DeleteCircle,			/* deleteProc */
    DisplayCircle,			/* displayProc */
    TK_ITEM_VISITOR_SUPPORT,		/* flags */
    CircleToPoint,			/* pointProc */
    CircleToArea,			/* areaProc */
    CircleToPostscript,			/* postscriptProc */
    ScaleCircle,			/* scaleProc */
    TranslateCircle,			/* translateProc */
    (Tk_ItemIndexProc *) NULL,		/* indexProc */
    (Tk_ItemCursorProc *) NULL,		/* cursorProc */
    (Tk_ItemSelectionProc *) NULL,	/* selectionProc */
    (Tk_ItemInsertProc *) NULL,		/* insertProc */
    (Tk_ItemDCharsProc *) NULL,		/* dTextProc */
    (Tk_ItemType *) NULL,		/* nextPtr */
    (Tk_ItemBboxProc *) ComputeCircleBbox,/* bboxProc */
    Tk_Offset(Tk_VisitorType, visitLine), /* acceptProc */
    (Tk_ItemGetCoordProc *) NULL,	/* getCoordPtr */
    (Tk_ItemSetCoordProc *) NULL	/* setCoordPtr */
};

/*
 *--------------------------------------------------------------
 *
 * CreateCircle --
 *
 *	This procedure is invoked to create a new circle
 *	item in a canvas.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	interp->result;  in this case itemPtr is left uninitialized,
 *	so it can be safely freed by the caller.
 *
 * Side effects:
 *	A new circle item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateCircle(interp, canvas, itemPtr, argc, argv)
    Tcl_Interp *interp;			/* For error reporting. */
    Tk_Canvas canvas;			/* Canvas to hold new item. */
    Tk_Item *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing circle. */
{
    PGraphItem *graphPtr = (PGraphItem *) itemPtr;
    int i, j;

    if ((argc==1) || ((argc>1) && (argv[1][0] == '-')
	    && (argv[1][1] >= 'a') && (argv[1][1] <= 'z'))) {
	i = 1;
    } else {
	i = 3;
    }

    if (argc < i) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		Tk_PathName(Tk_CanvasTkwin(canvas)), " create ",
		itemPtr->typePtr->name, " x1 y1 r ?options?\"",
		(char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Carry out initialization that is needed in order to clean
     * up after errors during the the remainder of this procedure.
     */

    Tk_CreateOutline(&(graphPtr->outline));
    Tk_CreateOutline(&(graphPtr->axis_outline));
    graphPtr->fillTile = NULL;
    graphPtr->canvas = canvas;
    graphPtr->activeFillTile = NULL;
    graphPtr->disabledFillTile = NULL;
    graphPtr->fillColor = NULL;
    graphPtr->activeFillColor = NULL;
    graphPtr->disabledFillColor = NULL;
    graphPtr->fillStipple = None;
    graphPtr->activeFillStipple = None;
    graphPtr->disabledFillStipple = None;
    graphPtr->fillGC = None;
    graphPtr->drawableGC = None;
    graphPtr->drawable = None;
    graphPtr->clipMask = None;
    graphPtr->poledir = 0.0;
    graphPtr->rmax = 1.0;
    graphPtr->resize_id = 1;
    graphPtr->arrowPtr = NULL;
    graphPtr->arrowShapeA = 5.0;
    graphPtr->arrowShapeB = 5.0;
    graphPtr->arrowShapeC = 2.0;
    graphPtr->arrowGC = None;
    graphPtr->sense = 1;
    graphPtr->configCmd = NULL;
    graphPtr->nchildren = 0;
    graphPtr->child_arr_size = 4;
    graphPtr->children = (Tk_Item **)ckalloc(sizeof(Tk_Item *)*graphPtr->child_arr_size);
    for(j = 0; j < graphPtr->child_arr_size; j++)
	graphPtr->children[j] = (Tk_Item *)NULL;

    /*
     * Process the arguments to fill in the item record.
     */

    if ((CircleCoords(interp, canvas, itemPtr, i, argv) != TCL_OK)) {
	goto error;
    }

    if (ConfigureCircle(interp, canvas, itemPtr, argc-i, argv+i, 0)
	    == TCL_OK) {
	return TCL_OK;
    }

    error:
    DeleteCircle(canvas, itemPtr, Tk_Display(Tk_CanvasTkwin(canvas)));
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * CircleCoords --
 *
 *	This procedure is invoked to process the "coords" widget
 *	command on circles.  See the user documentation
 *	for details on what it does.
 *
 * Results:
 *	Returns TCL_OK or TCL_ERROR, and sets interp->result.
 *
 * Side effects:
 *	The coordinates for the given item may be changed.
 *
 *--------------------------------------------------------------
 */

static int
CircleCoords(interp, canvas, itemPtr, argc, argv)
    Tcl_Interp *interp;			/* Used for error reporting. */
    Tk_Canvas canvas;			/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item whose coordinates are to be
					 * read or modified. */
    int argc;				/* Number of coordinates supplied in
					 * argv. */
    char **argv;			/* Array of coordinates: x1, y1,
					 * x2, y2, ... */
{
    PGraphItem *graphPtr = (PGraphItem *) itemPtr;
    char c0[TCL_DOUBLE_SPACE], c1[TCL_DOUBLE_SPACE];
    char c2[TCL_DOUBLE_SPACE], c3[TCL_DOUBLE_SPACE];

    /*
    printf("Validating memory... CircleCoords(start)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    if (argc == 0) {
	Tcl_PrintDouble(interp, graphPtr->origin[0], c0);
	Tcl_PrintDouble(interp, graphPtr->origin[1], c1);
	Tcl_PrintDouble(interp, graphPtr->radius, c2);
	Tcl_AppendResult(interp, c0, " ", c1, " ", c2, (char *) NULL);
    } else if ((argc == 1)||(argc == 3)) {
	char **largv = argv; 	if (argc==1) {
	    if (Tcl_SplitList(interp, argv[0], &argc, &largv) != TCL_OK) {
		if (largv != NULL) {
		    ckfree((char *) largv);
		}
		return TCL_ERROR;
	    } else if (argc != 3) {
		sprintf(c0,"%d",argc);
		Tcl_AppendResult(interp, "wrong # coordinates: expected 3, got ",
		c0, (char *) NULL);
		return TCL_ERROR;
	    }
	}
	if ((Tk_CanvasGetCoord(interp, canvas, largv[0],
 		    &graphPtr->origin[0]) != TCL_OK)
		|| (Tk_CanvasGetCoord(interp, canvas, largv[1],
		    &graphPtr->origin[1]) != TCL_OK)
		|| (Tk_CanvasGetCoord(interp, canvas, largv[2],
			&graphPtr->radius) != TCL_OK)) {
	    if (largv!=argv) ckfree((char *) largv);
	    return TCL_ERROR;
	}
	if (largv!=argv) ckfree((char *) largv);
	ComputeCircleBbox(canvas, graphPtr);

	GraphCreateDrawable(Tk_CanvasTkwin(canvas), graphPtr);
	GraphUpdateDrawable(Tk_CanvasTkwin(canvas), graphPtr);
	GraphUpdateGC(Tk_CanvasTkwin(canvas), graphPtr);
    } else {
	sprintf(interp->result,
		"wrong # coordinates: expected 0 or 3, got %d",
		argc);
	return TCL_ERROR;
    }

    if (graphPtr->arrowon) {
	ConfigureArrows(canvas, graphPtr);
    }

    graphPtr->resize_id = (graphPtr->resize_id + 1) % 256;

    /*
    printf("Validating memory... CircleCoords(end)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureCircle --
 *
 *	This procedure is invoked to configure various aspects
 *	of a circle item, such as its border and
 *	background colors.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in interp->result.
 *
 * Side effects:
 *	Configuration information, such as colors and stipple
 *	patterns, may be set for itemPtr.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureCircle(interp, canvas, itemPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Canvas canvas;		/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Circle item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    PGraphItem *graphPtr = (PGraphItem *) itemPtr;
    unsigned long mask;
    Tk_Item	*setPtr;
    XGCValues	gcValues;
    GC		newGC,
    		arrowGC;
    Tk_Window	tkwin;
    Tk_Tile	tile;
    XColor	*color;
    Pixmap	stipple;
    Tk_State	state;
    Pixmap	pixmap;
    int		i;

    /*
    printf("Validating memory... ConfigureCircle(start)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    tkwin = Tk_CanvasTkwin(canvas);

    if (Tk_ConfigureWidget(interp, tkwin, configSpecs, argc, argv,
	    (char *) graphPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    state = itemPtr->state;

    /*
     * A few of the options require additional processing, such as
     * graphics contexts.
     */

    if (graphPtr->outline.activeWidth > graphPtr->outline.width ||
	    graphPtr->outline.activeDash.number > 0 ||
	    graphPtr->outline.activeTile != NULL ||
	    graphPtr->outline.activeColor != NULL ||
	    graphPtr->outline.activeStipple != None ||
	    graphPtr->activeFillTile != NULL ||
	    graphPtr->activeFillColor != NULL ||
	    graphPtr->activeFillStipple != None) {
	itemPtr->redraw_flags |= TK_ITEM_STATE_DEPENDANT;
    } else {
	itemPtr->redraw_flags &= ~TK_ITEM_STATE_DEPENDANT;
    }

    GraphCreateDrawable(Tk_CanvasTkwin(canvas), graphPtr);
    GraphUpdateDrawable(Tk_CanvasTkwin(canvas), graphPtr);
    GraphUpdateGC(Tk_CanvasTkwin(canvas), graphPtr);

    mask = Tk_ConfigOutlineGC(&gcValues, canvas, itemPtr,
	     &(graphPtr->outline));

    if (mask) {
	gcValues.cap_style = CapProjecting;
	gcValues.clip_mask = graphPtr->clipMask;
	gcValues.clip_x_origin = 0;
	gcValues.clip_y_origin = 0;
	mask |= GCCapStyle;
	newGC = Tk_GetGC(tkwin, mask, &gcValues);
    } else {
	newGC = None;
    }
    if (graphPtr->outline.gc != None) {
	Tk_FreeGC(Tk_Display(tkwin), graphPtr->outline.gc);
    }
    graphPtr->outline.gc = newGC;

    if (graphPtr->arrowon) {
	ConfigureArrows(canvas, graphPtr);
    }

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }
    tile = graphPtr->fillTile;
    color = graphPtr->fillColor;
    stipple = graphPtr->fillStipple;
    if (state==TK_STATE_HIDDEN) {
	ComputeCircleBbox(canvas, graphPtr);
	return TCL_OK;
    }
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (graphPtr->activeFillTile!=NULL) {
	    tile = graphPtr->activeFillTile;
	}
	if (graphPtr->activeFillColor!=NULL) {
	    color = graphPtr->activeFillColor;
	}
	if (graphPtr->activeFillStipple!=None) {
	    stipple = graphPtr->activeFillStipple;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (graphPtr->disabledFillTile!=NULL) {
	    tile = graphPtr->disabledFillTile;
	}
	if (graphPtr->disabledFillColor!=NULL) {
	    color = graphPtr->disabledFillColor;
	}
	if (graphPtr->disabledFillStipple!=None) {
	    stipple = graphPtr->disabledFillStipple;
	}
    }

    Tk_SetTileCanvasItem(graphPtr->fillTile, canvas, (Tk_Item *) NULL);
    Tk_SetTileCanvasItem(graphPtr->activeFillTile, canvas, (Tk_Item *) NULL);
    Tk_SetTileCanvasItem(graphPtr->disabledFillTile, canvas, (Tk_Item *) NULL);
    Tk_SetTileCanvasItem(tile, canvas, itemPtr);

    if ((pixmap = Tk_PixmapOfTile(tile)) != None) {
	gcValues.fill_style = FillTiled;
	gcValues.tile = pixmap;
	newGC = Tk_GetGC(tkwin, GCTile|GCFillStyle, &gcValues);
    } else if (color == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = color->pixel;
	if (stipple != None) {
	    gcValues.stipple = stipple;
	    gcValues.fill_style = FillStippled;
	    mask = GCForeground|GCStipple|GCFillStyle;
	} else {
	    mask = GCForeground;
	}
	newGC = Tk_GetGC(tkwin, mask, &gcValues);
    }
    if (graphPtr->fillGC != None) {
	Tk_FreeGC(Tk_Display(tkwin), graphPtr->fillGC);
    }
    graphPtr->fillGC = newGC;

    /*
     * Update the GC for drawing the axis lines
     */
    mask = Tk_ConfigOutlineGC(&gcValues, canvas, itemPtr,
	     &(graphPtr->axis_outline));
    
    if (mask) {
	gcValues.cap_style = CapButt;
	mask |= GCCapStyle;
	newGC = Tk_GetGC(tkwin, mask, &gcValues);
	gcValues.line_width = 0;
	arrowGC = Tk_GetGC(tkwin, mask|GCLineWidth, &gcValues);
    } else {
	newGC = arrowGC = None;
    }

    if (graphPtr->axis_outline.gc != None) {
	Tk_FreeGC(Tk_Display(tkwin), graphPtr->axis_outline.gc);
    }
    if (graphPtr->arrowGC != None) {
	Tk_FreeGC(Tk_Display(tkwin), graphPtr->arrowGC);
    }
    graphPtr->axis_outline.gc = newGC;
    graphPtr->arrowGC = arrowGC;


    ComputeCircleBbox(canvas, graphPtr);

    graphPtr->resize_id = (graphPtr->resize_id + 1) % 256;

    for(i = 0, setPtr = *graphPtr->children;
	i < graphPtr->nchildren;
	setPtr = graphPtr->children[i++]){

	/*
	 * Update the set's coordinates
	 */

	(setPtr->typePtr->coordProc)(interp, canvas, setPtr, -1, (char **)NULL);
	setPtr->redraw_flags |= FORCE_REDRAW;

	((TkCanvas *)canvas)->flags |= REPICK_NEEDED;
    }

    graphPtr->resize_id = (graphPtr->resize_id + 1) % 256;

    /*
    printf("Validating memory... ConfigureCircle(end)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DeleteCircle --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a circle item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteCircle(canvas, itemPtr, display)
    Tk_Canvas canvas;			/* Info about overall widget. */
    Tk_Item *itemPtr;			/* Item that is being deleted. */
    Display *display;			/* Display containing window for
					 * canvas. */
{
    PGraphItem *graphPtr = (PGraphItem *) itemPtr;
    Tk_Item *setPtr;
    TkCanvas *cPtr = (TkCanvas *)canvas;
    int i;

    Tk_DeleteOutline(display, &(graphPtr->outline));
    Tk_DeleteOutline(display, &(graphPtr->axis_outline));
    if (graphPtr->fillTile != NULL) {
	Tk_FreeTile(graphPtr->fillTile);
    }
    if (graphPtr->activeFillTile != NULL) {
	Tk_FreeTile(graphPtr->activeFillTile);
    }
    if (graphPtr->disabledFillTile != NULL) {
	Tk_FreeTile(graphPtr->disabledFillTile);
    }
    if (graphPtr->fillColor != NULL) {
	Tk_FreeColor(graphPtr->fillColor);
    }
    if (graphPtr->activeFillColor != NULL) {
	Tk_FreeColor(graphPtr->activeFillColor);
    }
    if (graphPtr->disabledFillColor != NULL) {
	Tk_FreeColor(graphPtr->disabledFillColor);
    }
    if (graphPtr->fillStipple != None) {
	Tk_FreeBitmap(display, graphPtr->fillStipple);
    }
    if (graphPtr->activeFillStipple != None) {
	Tk_FreeBitmap(display, graphPtr->activeFillStipple);
    }
    if (graphPtr->disabledFillStipple != None) {
	Tk_FreeBitmap(display, graphPtr->disabledFillStipple);
    }
    if (graphPtr->fillGC != None) {
	Tk_FreeGC(display, graphPtr->fillGC);
    }
    if (graphPtr->arrowGC != None) {
	Tk_FreeGC(display, graphPtr->arrowGC);
    }
    if (graphPtr->drawableGC != None) {
	Tk_FreeGC(display, graphPtr->drawableGC);
    }
    if (graphPtr->clipMask != None) {
	Tk_FreePixmap(display, graphPtr->clipMask);
    }
    if (graphPtr->drawable != None) {
	Tk_FreePixmap(display, graphPtr->drawable);
    }

    /*
    for(i = 0, setPtr = *graphPtr->children ;
	i < graphPtr->nchildren;
	setPtr = graphPtr->children[i++]){

	if (canvas) {
	    char *argv[3] = {"-master", ""};
	    (setPtr->typePtr->configProc)(cPtr->interp, canvas, setPtr, 3,
		    argv, TK_CONFIG_ARGV_ONLY);
	    Tk_CanvasEventuallyRedraw(canvas, setPtr->x1, setPtr->y1,
		    setPtr->x2, setPtr->y2);
	    ((TkCanvas *)canvas)->flags |= REPICK_NEEDED;
	}
    }
    */
}

/*
 *--------------------------------------------------------------
 *
 * ComputeCircleBbox --
 *
 *	This procedure is invoked to compute the bounding box of
 *	all the pixels that may be drawn as part of a circle.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The fields x1, y1, x2, and y2 are updated in the header
 *	for itemPtr.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static void
ComputeCircleBbox(canvas, graphPtr)
    Tk_Canvas canvas;			/* Canvas that contains item. */
    PGraphItem *graphPtr;		/* Item whose bbox is to be
					 * recomputed. */
{
    int bloat, width;
    Tk_State state = graphPtr->header.state;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    width = graphPtr->outline.width;
    if (state==TK_STATE_HIDDEN) {
	graphPtr->header.x1 = graphPtr->header.y1 =
	graphPtr->header.x2 = graphPtr->header.y2 = -1;
	return;
    }
    if (((TkCanvas *)canvas)->currentItemPtr == (Tk_Item *)graphPtr) {
	if (graphPtr->outline.activeWidth>width) {
	    width = graphPtr->outline.activeWidth;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (graphPtr->outline.disabledWidth>0) {
	    width = graphPtr->outline.disabledWidth;
	}
    }

    /*
     * Add in the width of the outline
     */

    if (graphPtr->outline.gc == None) {
	bloat = 0;
    } else {
	bloat = (width+1)/2;
    }

    /*
     * Special note:  the circle is always drawn at least 1x1 in
     * size, so round up the upper coordinates to be at least 1 unit
     * greater than the lower ones.
     */

    graphPtr->header.x1= graphPtr->origin[0] - graphPtr->radius - bloat -
	.5;
    graphPtr->header.y1= graphPtr->origin[1] - graphPtr->radius - bloat -
	.5;
    graphPtr->header.x2= graphPtr->origin[0] + graphPtr->radius + bloat +
	.5;
    graphPtr->header.y2= graphPtr->origin[1] + graphPtr->radius + bloat +
	.5;
}

/*
 *--------------------------------------------------------------
 *
 * DisplayCircle --
 *
 *	This procedure is invoked to draw a circle item in a given drawable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in drawable using the transformation
 *	information in canvas.
 *
 *--------------------------------------------------------------
 */

static void
DisplayCircle(canvas, itemPtr, display, drawable, x, y, width, height)
    Tk_Canvas canvas;			/* Canvas that contains item. */
    Tk_Item *itemPtr;			/* Item to be displayed. */
    Display *display;			/* Display on which to draw item. */
    Drawable drawable;			/* Pixmap or window in which to draw
					 * item. */
    int x, y, width, height;		/* Describes region of canvas that
					 * must be redisplayed (not used). */
{
    PGraphItem *graphPtr = (PGraphItem *) itemPtr;
    short x1, y1, x2, y2;
    Tk_Tile fillTile;
    Pixmap fillStipple;
    Tk_State state;
    XGCValues gcValues;
    unsigned long origcolor;

    /*
    printf("Validating memory... DisplayCircle(start)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    if (graphPtr->drawable == None) {
	GraphCreateDrawable(Tk_CanvasTkwin(canvas), graphPtr);
	GraphUpdateDrawable(Tk_CanvasTkwin(canvas), graphPtr);
	GraphUpdateGC(Tk_CanvasTkwin(canvas), graphPtr);
    }

    /*
     * Compute the screen coordinates of the bounding box for the item.
     * Make sure that the bbox is at least one pixel large, since some
     * X servers will die if it isn't.
     */

    Tk_CanvasDrawableCoords(canvas, graphPtr->origin[0] - graphPtr->radius, 
	    graphPtr->origin[1] - graphPtr->radius,
	    &x1, &y1);
    Tk_CanvasDrawableCoords(canvas, graphPtr->origin[0] + graphPtr->radius, 
	    graphPtr->origin[1] + graphPtr->radius,
	    &x2, &y2);

    /*
     * Display filled part first (if wanted), then outline.  If we're
     * stippling, then modify the stipple offset in the GC.  Be sure to
     * reset the offset when done, since the GC is supposed to be
     * read-only.
     */

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }
    fillTile = graphPtr->fillTile;
    fillStipple = graphPtr->fillStipple;
    if (((TkCanvas *)canvas)->currentItemPtr == (Tk_Item *)graphPtr) {
	if (graphPtr->activeFillTile!=NULL) {
	    fillTile = graphPtr->activeFillTile;
	}
	if (graphPtr->activeFillStipple!=None) {
	    fillStipple = graphPtr->activeFillStipple;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (graphPtr->disabledFillTile!=NULL) {
	    fillTile = graphPtr->disabledFillTile;
	}
	if (graphPtr->disabledFillStipple!=None) {
	    fillStipple = graphPtr->disabledFillStipple;
	}
    }

    if (graphPtr->fillGC != None) {
	if ((fillTile != None) || (fillStipple != None)) {
	    Tk_CanvasSetStippleOrigin(canvas, graphPtr->fillGC);
	}
	XFillArc(display, drawable, graphPtr->fillGC,
		x1, y1, (unsigned) (x2-x1), (unsigned) (y2-y1),
		0, 360*64);
	if ((fillTile != None) || (fillStipple != None)) {
	    XSetTSOrigin(display, graphPtr->fillGC, 0, 0);
	}
    }
    if (graphPtr->outline.gc != None) {
	Tk_ChangeOutlineGC(canvas, itemPtr, &(graphPtr->outline));
	XDrawArc(display, drawable, graphPtr->outline.gc,
		x1, y1, (unsigned) (x2-x1), (unsigned) (y2-y1), 0, 360*64);
	Tk_ResetOutlineGC(canvas, itemPtr, &(graphPtr->outline));
    }

    if (graphPtr->axis_outline.gc != None) {
	double xcen, ycen;
	double xa, ya, xb ,yb;
	int origwidth;
	Tk_ChangeOutlineGC(canvas, itemPtr, &(graphPtr->axis_outline));

	/*
	 * fprintf(stderr, "axison = %d, axislength = %d, axiswidth = %d\n",
	 *	graphPtr->axison, graphPtr->axislength, graphPtr->axiswidth);
	 */

	XSetClipMask(display, graphPtr->axis_outline.gc, graphPtr->clipMask);
	XSetClipOrigin(display, graphPtr->axis_outline.gc, x1, y1);

	xcen = (x1 + x2) / 2;
	ycen = (y1 + y2) / 2;

	if (graphPtr->poledir == 0.0 || graphPtr->poledir == 180.0) {
	    xa = 0.0;
	    ya = -1.0;
	} else if (graphPtr->poledir == 90.0 || graphPtr->poledir == 270.0) {
	    xa = 1.0;
	    ya = 0.0;
	} else {
	    xa = cos((graphPtr->poledir) * 3.1415926536 / 180.0);
	    ya = sin((graphPtr->poledir) * 3.1415926536 / 180.0);
	    ya = -ya;
	}
	xb = xcen + xa*graphPtr->axislength;
	yb = ycen + ya*graphPtr->axislength;
	xa = xcen - xa*graphPtr->axislength;
	ya = ycen - ya*graphPtr->axislength;
	/*
	 * fprintf(stderr, "poledir = %g  \n", graphPtr->poledir);
	 * fprintf(stderr, "(%d, %d)\t(%d, %d)\n",
	 *	(int) xa, (int) ya, (int) xb, (int) yb);
	 */
	XDrawLine(display, drawable, graphPtr->axis_outline.gc,
		(int) xa, (int) ya, (int) xb, (int) yb);

	if (graphPtr->poledir == 0.0 || graphPtr->poledir == 180.0) {
	    xa = 1.0;
	    ya = 0.0;
	} else if (graphPtr->poledir == 90.0 || graphPtr->poledir == 270.0) {
	    xa = 0.0;
	    ya = -1.0;
	} else {
	    xa = cos((graphPtr->poledir + 90) * 3.1415926536 / 180.0);
	    ya = sin((graphPtr->poledir + 90) * 3.1415926536 / 180.0);
	    ya = -ya;
	}
	xb = xcen + xa*graphPtr->axislength;
	yb = ycen + ya*graphPtr->axislength;
	xa = xcen - xa*graphPtr->axislength;
	ya = ycen - ya*graphPtr->axislength;
	/*
	 * fprintf(stderr, "(%d, %d)\t(%d, %d)\n",
	 *	(int) xa, (int) ya, (int) xb, (int) yb);
	 */
	XDrawLine(display, drawable, graphPtr->axis_outline.gc,
		(int) xa, (int) ya, (int) xb, (int) yb);

	if (graphPtr->arrowon) {
	    XSetClipMask(display, graphPtr->arrowGC, graphPtr->clipMask);
	    XSetClipOrigin(display, graphPtr->arrowGC, x1, y1);
	    TkFillPolygon(canvas, graphPtr->arrowPtr, PTS_IN_ARROW,
		    display, drawable, graphPtr->arrowGC, NULL);
	    XSetClipOrigin(display, graphPtr->arrowGC, 0, 0);
	    XSetClipMask(display, graphPtr->arrowGC, None);
	}

	XSetClipOrigin(display, graphPtr->axis_outline.gc, 0, 0);
	XSetClipMask(display, graphPtr->axis_outline.gc, None);
	Tk_ResetOutlineGC(canvas, itemPtr, &(graphPtr->axis_outline));
    }

    /*
    printf("Validating memory... DisplayCircle(end)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */
}

/*
 *--------------------------------------------------------------
 *
 * CircleToPoint --
 *
 *	Computes the distance from a given point to a given
 *	circle, in canvas units.
 *
 * Results:
 *	The return value is 0 if the point whose x and y coordinates
 *	are coordPtr[0] and coordPtr[1] is inside the circle.  If the
 *	point isn't inside the circle then the return value is the
 *	distance from the point to the circle.  If itemPtr is filled,
 *	then anywhere in the interior is considered "inside"; if
 *	itemPtr isn't filled, then "inside" means only the area
 *	occupied by the outline.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static double
CircleToPoint(canvas, itemPtr, pointPtr)
    Tk_Canvas canvas;		/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against point. */
    double *pointPtr;		/* Pointer to x and y coordinates. */
{
    PGraphItem *graphPtr = (PGraphItem *) itemPtr;
    double width;
    int filled;
    double bbox[4];
    Tk_State state = itemPtr->state;

    bbox[0] = graphPtr->origin[0] - graphPtr->radius;
    bbox[1] = graphPtr->origin[1] - graphPtr->radius;
    bbox[2] = graphPtr->origin[0] + graphPtr->radius;
    bbox[3] = graphPtr->origin[1] + graphPtr->radius;
    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    width = (double) graphPtr->outline.width;
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (graphPtr->outline.activeWidth>width) {
	    width = (double) graphPtr->outline.activeWidth;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (graphPtr->outline.disabledWidth>0) {
	    width = (double) graphPtr->outline.disabledWidth;
	}
    }


    filled = graphPtr->fillGC != None;
    if (graphPtr->outline.gc == None) {
	width = 0.0;
	filled = 1;
    }
    filled = 1;
    return TkOvalToPoint(bbox, width, filled, pointPtr);
}

/*
 *--------------------------------------------------------------
 *
 * CircleToArea --
 *
 *	This procedure is called to determine whether an item
 *	lies entirely inside, entirely outside, or overlapping
 *	a given rectangular area.
 *
 * Results:
 *	-1 is returned if the item is entirely outside the area
 *	given by rectPtr, 0 if it overlaps, and 1 if it is entirely
 *	inside the given area.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
CircleToArea(canvas, itemPtr, areaPtr)
    Tk_Canvas canvas;		/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against circle. */
    double *areaPtr;		/* Pointer to array of four coordinates
				 * (x1, y1, x2, y2) describing rectangular
				 * area.  */
{
    PGraphItem *graphPtr = (PGraphItem *) itemPtr;
    double circle[4], halfWidth;
    int result;
    int width;
    Tk_State state = itemPtr->state;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    width = graphPtr->outline.width;
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (graphPtr->outline.activeWidth>width) {
	    width = graphPtr->outline.activeWidth;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (graphPtr->outline.disabledWidth>0) {
	    width = graphPtr->outline.disabledWidth;
	}
    }

    /*
     * Expand the circle to include the width of the outline, if any.
     */

    halfWidth = width/2.0;
    if (graphPtr->outline.gc == None) {
	halfWidth = 0.0;
    }
    circle[0] = graphPtr->origin[0] - graphPtr->radius - halfWidth;
    circle[1] = graphPtr->origin[1] - graphPtr->radius - halfWidth;
    circle[2] = graphPtr->origin[0] + graphPtr->radius + halfWidth;
    circle[3] = graphPtr->origin[1] + graphPtr->radius + halfWidth;

    result = TkOvalToArea(circle, areaPtr);

    /*
     * If the rectangle appears to overlap the circle and the circle
     * isn't filled, do one more check to see if perhaps all four
     * of the rectangle's corners are totally inside the circle's
     * unfilled center, in which case we should return "outside".
     */

    if ((result == 0) && (graphPtr->outline.gc != None)
	    && (graphPtr->fillGC == None)) {
	double centerX, centerY, width, height;
	double xDelta1, yDelta1, xDelta2, yDelta2;

	centerX = graphPtr->origin[0];
	centerY = graphPtr->origin[1];
	width = graphPtr->radius;
	height = graphPtr->radius;
	xDelta1 = (areaPtr[0] - centerX)/width;
	xDelta1 *= xDelta1;
	yDelta1 = (areaPtr[1] - centerY)/height;
	yDelta1 *= yDelta1;
	xDelta2 = (areaPtr[2] - centerX)/width;
	xDelta2 *= xDelta2;
	yDelta2 = (areaPtr[3] - centerY)/height;
	yDelta2 *= yDelta2;
	if (((xDelta1 + yDelta1) < 1.0)
		&& ((xDelta1 + yDelta2) < 1.0)
		&& ((xDelta2 + yDelta1) < 1.0)
		&& ((xDelta2 + yDelta2) < 1.0)) {
	    return -1;
	}
    }
    return result;
}

/*
 *--------------------------------------------------------------
 *
 * ScaleCircle --
 *
 *	This procedure is invoked to rescale a circle item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The circle referred to by itemPtr is rescaled
 *	so that the following transformation is applied to all
 *	point coordinates:
 *		x' = originX + scaleX*(x-originX)
 *		y' = originY + scaleY*(y-originY)
 *
 *--------------------------------------------------------------
 */

void
ScaleCircle(canvas, itemPtr, originX, originY, scaleX, scaleY)
    Tk_Canvas canvas;			/* Canvas containing circle. */
    Tk_Item *itemPtr;			/* Circle to be scaled. */
    double originX, originY;		/* Origin about which to scale rect. */
    double scaleX;			/* Amount to scale in X direction. */
    double scaleY;			/* Amount to scale in Y direction. */
{
    PGraphItem *graphPtr = (PGraphItem *) itemPtr;
    Tk_Item *setPtr;
    int i;

    graphPtr->origin[0] = originX + scaleX*(graphPtr->origin[0] - originX);
    graphPtr->origin[1] = originY + scaleY*(graphPtr->origin[1] - originY);
    graphPtr->radius = graphPtr->radius * (scaleX + scaleY) / 2.0;
    graphPtr->axislength = graphPtr->axislength * (scaleX + scaleY) / 2.0;
    ComputeCircleBbox(canvas, graphPtr);

    GraphCreateDrawable(Tk_CanvasTkwin(canvas), graphPtr);
    GraphUpdateDrawable(Tk_CanvasTkwin(canvas), graphPtr);
    GraphUpdateGC(Tk_CanvasTkwin(canvas), graphPtr);

    if (graphPtr->arrowon) {
	ConfigureArrows(canvas, graphPtr);
    }

    for(i = 0, setPtr = *graphPtr->children ;
	i < graphPtr->nchildren; 
	setPtr = graphPtr->children[i++]) {
	Tk_CanvasEventuallyRedraw(canvas, setPtr->x1, setPtr->y1, setPtr->x2,
		setPtr->y2);
	((TkCanvas *)canvas)->flags |= REPICK_NEEDED;
    }
}

/*
 *--------------------------------------------------------------
 *
 * TranslateCircle --
 *
 *	This procedure is called to move a circle by a
 *	given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The position of the circle is offset by
 *	(xDelta, yDelta), and the bounding box is updated in the
 *	generic part of the item structure.
 *
 *--------------------------------------------------------------
 */

void
TranslateCircle(canvas, itemPtr, deltaX, deltaY)
    Tk_Canvas canvas;			/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item that is being moved. */
    double deltaX, deltaY;		/* Amount by which item is to be
					 * moved. */
{
    PGraphItem *graphPtr = (PGraphItem *) itemPtr;

    graphPtr->origin[0] += deltaX;
    graphPtr->origin[1] += deltaY;
    ComputeCircleBbox(canvas, graphPtr);

    graphPtr->resize_id = (graphPtr->resize_id + 1) % 256;

    if (graphPtr->arrowon) {
	ConfigureArrows(canvas, graphPtr);
    }
}

/*
 *--------------------------------------------------------------
 *
 * AxisToPostscript --
 *
 *	This procedure is called to generate Postscript for
 *	the axis of phgraph items.
 *
 * Results:
 *	The return value is a standard Tcl result.  If an error
 *	occurs in generating Postscript then an error message is
 *	left in interp->result, replacing whatever used to be there.
 *	If no error occurs, then Postscript for the circle is
 *	appended to the result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
AxisToPostscript(interp, canvas, itemPtr, prepass)
    Tcl_Interp *interp;			/* Interpreter for error reporting. */
    Tk_Canvas canvas;			/* Information about overall canvas. */
    Tk_Item *itemPtr;			/* Item for which Postscript is
					 * wanted. */
    int prepass;			/* 1 means this is a prepass to
					 * collect font information;  0 means
					 * final Postscript is being created. */
{
    char pathCmd[500], string[100];
    PGraphItem *graphPtr = (PGraphItem *) itemPtr;
    double y1, y2;
    int width;
    Tk_Tile tile;
    XColor *color;
    Pixmap stipple;
    Tk_Tile fillTile;
    XColor *fillColor;
    Pixmap fillStipple;
    Tk_State state = itemPtr->state;
    pathCmd[0] = (char)NULL;

    y1 = Tk_CanvasPsY(canvas, graphPtr->origin[1] - graphPtr->radius);
    y2 = Tk_CanvasPsY(canvas, graphPtr->origin[1] + graphPtr->radius);

    /*
     * Generate a string for drawing the axis lines.
     */

    Tcl_AppendResult(interp, pathCmd, (char *)NULL);

    sprintf(pathCmd, "%.15g %.15g moveto %.15g %.15g rlineto\n%.15g %.15g moveto %.15g %.15g rlineto\n%.15g %.15g moveto %.15g %.15g rlineto\n%.15g %.15g moveto %.15g %.15g rlineto\n",
	graphPtr->origin[0], (y2 + y1)/2.0,
	graphPtr->axislength*cos(graphPtr->poledir * 3.1415926 / 180),
	graphPtr->axislength*sin(graphPtr->poledir * 3.1415926 / 180),
	graphPtr->origin[0], (y2 + y1)/2.0,
	graphPtr->axislength*cos((90.0  + graphPtr->poledir) * 3.1415926 / 180),
	graphPtr->axislength*sin((90.0  + graphPtr->poledir) * 3.1415926 / 180),
	graphPtr->origin[0], (y2 + y1)/2.0,
	graphPtr->axislength*cos((180.0 + graphPtr->poledir) * 3.1415926 / 180),
	graphPtr->axislength*sin((180.0 + graphPtr->poledir) * 3.1415926 / 180),
	graphPtr->origin[0], (y2 + y1)/2.0,
	graphPtr->axislength*cos((270.0 + graphPtr->poledir) * 3.1415926 / 180),
	graphPtr->axislength*sin((270.0 + graphPtr->poledir) * 3.1415926 / 180));

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }
    tile = graphPtr->axis_outline.tile;
    color = graphPtr->axis_outline.color;
    fillTile = graphPtr->fillTile;
    fillColor = graphPtr->fillColor;
    fillStipple = graphPtr->fillStipple;
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (graphPtr->axis_outline.activeTile!=NULL) {
	    tile = graphPtr->axis_outline.activeTile;
	}
	if (graphPtr->axis_outline.activeColor!=NULL) {
	    color = graphPtr->axis_outline.activeColor;
	}
	if (graphPtr->activeFillTile!=NULL) {
	    fillTile = graphPtr->activeFillTile;
	}
	if (graphPtr->activeFillColor!=NULL) {
	    fillColor = graphPtr->activeFillColor;
	}
	if (graphPtr->activeFillStipple!=None) {
	    fillStipple = graphPtr->activeFillStipple;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (graphPtr->axis_outline.disabledTile!=NULL) {
	    tile = graphPtr->axis_outline.disabledTile;
	}
	if (graphPtr->axis_outline.disabledColor!=NULL) {
	    color = graphPtr->axis_outline.disabledColor;
	}
	if (graphPtr->disabledFillTile!=NULL) {
	    fillTile = graphPtr->disabledFillTile;
	}
	if (graphPtr->disabledFillColor!=NULL) {
	    fillColor = graphPtr->disabledFillColor;
	}
	if (graphPtr->disabledFillStipple!=None) {
	    fillStipple = graphPtr->disabledFillStipple;
	}
    }

    /*
     * Now draw the outline, if there is one.
     */

    if (color != NULL) {
	Tcl_AppendResult(interp, pathCmd, "0 setlinejoin 2 setlinecap\n",
		(char *) NULL);
	if (Tk_CanvasPsOutline(canvas, itemPtr,
		&(graphPtr->axis_outline))!= TCL_OK) {
	    return TCL_ERROR;
	}
    }

    if (graphPtr->arrowon && color != NULL) {
	if (ArrowheadPostscript(interp, canvas, graphPtr, graphPtr->arrowPtr)
		!= TCL_OK) {
	    return TCL_ERROR;
	}
    }

    return TCL_OK;
}
/*
 *--------------------------------------------------------------
 *
 * CircleToPostscript --
 *
 *	This procedure is called to generate Postscript for
 *	circle items.
 *
 * Results:
 *	The return value is a standard Tcl result.  If an error
 *	occurs in generating Postscript then an error message is
 *	left in interp->result, replacing whatever used to be there.
 *	If no error occurs, then Postscript for the circle is
 *	appended to the result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
CircleToPostscript(interp, canvas, itemPtr, prepass)
    Tcl_Interp *interp;			/* Interpreter for error reporting. */
    Tk_Canvas canvas;			/* Information about overall canvas. */
    Tk_Item *itemPtr;			/* Item for which Postscript is
					 * wanted. */
    int prepass;			/* 1 means this is a prepass to
					 * collect font information;  0 means
					 * final Postscript is being created. */
{
    char pathCmd[500], string[100];
    PGraphItem *graphPtr = (PGraphItem *) itemPtr;
    double y1, y2;
    int width;
    Tk_Tile tile;
    XColor *color;
    Pixmap stipple;
    Tk_Tile fillTile;
    XColor *fillColor;
    Pixmap fillStipple;
    Tk_State state = itemPtr->state;

    y1 = Tk_CanvasPsY(canvas, graphPtr->origin[1] - graphPtr->radius);
    y2 = Tk_CanvasPsY(canvas, graphPtr->origin[1] + graphPtr->radius);

    /*
     * Generate a string that creates a path for the circle.
     * This is the only part of the procedure's code that is type-
     * specific.
     */


    sprintf(pathCmd, "matrix currentmatrix\n%.15g %.15g translate %.15g %.15g scale 1 0 moveto 0 0 1 0 360 arc\nsetmatrix\n",
	    graphPtr->origin[0], (y1 + y2) / 2.,
	    graphPtr->radius, graphPtr->radius);


    /*
     * Clip everything at the graph boundary
     */

    sprintf(pathCmd, "newpath %.15g %.15g %.15g 0 360 arc clip\n",
	    graphPtr->origin[0],
	    Tk_CanvasPsY(canvas, graphPtr->origin[1]),
	    graphPtr->radius);
    Tcl_AppendResult(interp, pathCmd, "newpath\n", (char *)NULL);

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }
    tile = graphPtr->outline.tile;
    color = graphPtr->outline.color;
    fillTile = graphPtr->fillTile;
    fillColor = graphPtr->fillColor;
    fillStipple = graphPtr->fillStipple;
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (graphPtr->outline.activeTile!=NULL) {
	    tile = graphPtr->outline.activeTile;
	}
	if (graphPtr->outline.activeColor!=NULL) {
	    color = graphPtr->outline.activeColor;
	}
	if (graphPtr->activeFillTile!=NULL) {
	    fillTile = graphPtr->activeFillTile;
	}
	if (graphPtr->activeFillColor!=NULL) {
	    fillColor = graphPtr->activeFillColor;
	}
	if (graphPtr->activeFillStipple!=None) {
	    fillStipple = graphPtr->activeFillStipple;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (graphPtr->outline.disabledTile!=NULL) {
	    tile = graphPtr->outline.disabledTile;
	}
	if (graphPtr->outline.disabledColor!=NULL) {
	    color = graphPtr->outline.disabledColor;
	}
	if (graphPtr->disabledFillTile!=NULL) {
	    fillTile = graphPtr->disabledFillTile;
	}
	if (graphPtr->disabledFillColor!=NULL) {
	    fillColor = graphPtr->disabledFillColor;
	}
	if (graphPtr->disabledFillStipple!=None) {
	    fillStipple = graphPtr->disabledFillStipple;
	}
    }

    /*
     * First draw the filled area of the phase plot.
     */

    if (fillColor != NULL) {
	Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	if (Tk_CanvasPsColor(interp, canvas, fillColor)
		!= TCL_OK) {
	    return TCL_ERROR;
	}
	if (fillStipple != None) {
	    Tcl_AppendResult(interp, "clip ", (char *) NULL);
	    if (Tk_CanvasPsStipple(interp, canvas, fillStipple)
		    != TCL_OK) {
		return TCL_ERROR;
	    }
	    if (color != NULL) {
		Tcl_AppendResult(interp, "grestore gsave\n", (char *) NULL);
	    }
	} else {
	    Tcl_AppendResult(interp, "fill\n", (char *) NULL);
	}
    }

    /*
     * Now draw the outline, if there is one.
     *
     * 97.09.16  Note from Mike:
     * We don't like outlines.  Don't draw this.
     */

    Tcl_AppendResult(interp, pathCmd, " newpath\n", (char *)NULL);
    if (color != NULL) {
	Tcl_AppendResult(interp, pathCmd, "0 setlinejoin 2 setlinecap\n",
		(char *) NULL);
	if (Tk_CanvasPsOutline(canvas, itemPtr,
		&(graphPtr->outline))!= TCL_OK) {
	    return TCL_ERROR;
	}
    }

    AxisToPostscript(interp, canvas, itemPtr, prepass);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ParseArrowShape --
 *
 *	This procedure is called back during option parsing to
 *	parse arrow shape information.
 *
 * Results:
 *	The return value is a standard Tcl result:  TCL_OK means
 *	that the arrow shape information was parsed ok, and
 *	TCL_ERROR means it couldn't be parsed.
 *
 * Side effects:
 *	Arrow information in recordPtr is updated.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
ParseArrowShape(clientData, interp, tkwin, value, recordPtr, offset)
    ClientData clientData;	/* Not used. */
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Window tkwin;		/* Not used. */
    char *value;		/* Textual specification of arrow shape. */
    char *recordPtr;		/* Pointer to item record in which to
				 * store arrow information. */
    int offset;			/* Offset of shape information in widget
				 * record. */
{
    PGraphItem *linePtr = (PGraphItem *) recordPtr;
    double a, b, c;
    int argc;
    char **argv = NULL;

    if (offset != Tk_Offset(PGraphItem, arrowShapeA)) {
	panic("ParseArrowShape received bogus offset");
    }

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	syntaxError:
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "bad arrow shape \"", value,
		"\": must be list with three numbers", (char *) NULL);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return TCL_ERROR;
    }
    if (argc != 3) {
	goto syntaxError;
    }
    if ((Tk_CanvasGetCoord(interp, linePtr->canvas, argv[0], &a) != TCL_OK)
	    || (Tk_CanvasGetCoord(interp, linePtr->canvas, argv[1], &b)
		!= TCL_OK)
	    || (Tk_CanvasGetCoord(interp, linePtr->canvas, argv[2], &c)
		!= TCL_OK)) {
	goto syntaxError;
    }
    linePtr->arrowShapeA = a;
    linePtr->arrowShapeB = b;
    linePtr->arrowShapeC = c;
    ckfree((char *) argv);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * PrintArrowShape --
 *
 *	This procedure is a callback invoked by the configuration
 *	code to return a printable value describing an arrow shape.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

    /* ARGSUSED */
static char *
PrintArrowShape(clientData, tkwin, recordPtr, offset, freeProcPtr)
    ClientData clientData;	/* Not used. */
    Tk_Window tkwin;		/* Window associated with linePtr's widget. */
    char *recordPtr;		/* Pointer to item record containing current
				 * shape information. */
    int offset;			/* Offset of arrow information in record. */
    Tcl_FreeProc **freeProcPtr;	/* Store address of procedure to call to
				 * free string here. */
{
    PGraphItem *linePtr = (PGraphItem *) recordPtr;
    char *buffer;

    buffer = (char *) ckalloc(120);
    sprintf(buffer, "%.5g %.5g %.5g", linePtr->arrowShapeA,
	    linePtr->arrowShapeB, linePtr->arrowShapeC);
    *freeProcPtr = TCL_DYNAMIC;
    return buffer;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureArrows --
 *
 *	If arrowheads have been requested for a line, this
 *	procedure makes arrangements for the arrowheads.
 *
 * Results:
 *	Always returns TCL_OK.
 *
 * Side effects:
 *	Information in linePtr is set up for one or two arrowheads.
 *	the firstArrowPtr and lastArrowPtr polygons are allocated
 *	and initialized, if need be, and the end points of the line
 *	are adjusted so that a thick line doesn't stick out past
 *	the arrowheads.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
ConfigureArrows(canvas, linePtr)
    Tk_Canvas canvas;			/* Canvas in which arrows will be
					 * displayed (interp and tkwin
					 * fields are needed). */
    PGraphItem *linePtr;		/* Item to configure for arrows. */
{
    double *poly, *coordPtr;
    double dx, dy, length, sinTheta, cosTheta, temp;
    double fracHeight;			/* Line width as fraction of
					 * arrowhead width. */
    double backup;			/* Distance to backup end points
					 * so the line ends in the middle
					 * of the arrowhead. */
    double vertX, vertY;		/* Position of arrowhead vertex. */
    double shapeA, shapeB, shapeC;	/* Adjusted coordinates (see
					 * explanation below). */
    double sinpd, cospd;		/* sine and cosine of the pole
					 * direction. */
    int width;
    int i;
    Tk_State state = linePtr->header.state;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    width = linePtr->outline.width;
    if (((TkCanvas *)canvas)->currentItemPtr == (Tk_Item *)linePtr) {
	if (linePtr->outline.activeWidth>width) {
	    width = linePtr->outline.activeWidth;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (linePtr->outline.disabledWidth>0) {
	    width = linePtr->outline.disabledWidth;
	}
    }

    /*
     * The code below makes a tiny increase in the shape parameters
     * for the line.  This is a bit of a hack, but it seems to result
     * in displays that more closely approximate the specified parameters.
     * Without the adjustment, the arrows come out smaller than expected.
     */

    shapeA = linePtr->arrowShapeA + 0.001;
    shapeB = linePtr->arrowShapeB + 0.001;
    shapeC = linePtr->arrowShapeC + width/2.0 + 0.001;

    /*
     * If there's an arrowhead on the first point of the line, compute
     * its polygon and adjust the first point of the line so that the
     * line doesn't stick out past the leading edge of the arrowhead.
     */

    fracHeight = (width/2.0)/shapeC;
    backup = fracHeight*shapeB + shapeA*(1.0 - fracHeight)/2.0;
    if (linePtr->arrowon) {
	cospd = cos((linePtr->poledir) * 3.1415926535 / 180.0);
	sinpd = sin((linePtr->poledir) * 3.1415926535 / 180.0);
	sinpd = -sinpd;
	poly = linePtr->arrowPtr;
	if (poly == NULL) {
	    poly = (double *) ckalloc((unsigned)
		    (2*PTS_IN_ARROW*sizeof(double)));
	    linePtr->arrowPtr = poly;
	}
	poly[0] = poly[10] = linePtr->origin[0]+cospd*linePtr->axislength;
	poly[1] = poly[11] = linePtr->origin[1]+sinpd*linePtr->axislength;
	/*
	 * fprintf(stderr, "origin = (%d, %d)\n",
	 *	(int) linePtr->origin[0], (int) linePtr->origin[1]);
	 * fprintf(stderr, "cos, sin * length = (%d, %d)\n",
	 *	(int) cospd * linePtr->axislength,
	 *	(int) sinpd * linePtr->axislength);
	 */
	dx = cospd * linePtr->axislength;
	dy = sinpd * linePtr->axislength;
	length = hypot(dx, dy);
	if (length == 0) {
	    sinTheta = cosTheta = 0.0;
	} else {
	    sinTheta = dy/length;
	    cosTheta = dx/length;
	}
	vertX = poly[0] - shapeA*cosTheta;
	vertY = poly[1] - shapeA*sinTheta;
	temp = shapeC*sinTheta;
	poly[2] = poly[0] - shapeB*cosTheta + temp;
	poly[8] = poly[2] - 2*temp;
	temp = shapeC*cosTheta;
	poly[3] = poly[1] - shapeB*sinTheta - temp;
	poly[9] = poly[3] + 2*temp;
	poly[4] = poly[2]*fracHeight + vertX*(1.0-fracHeight);
	poly[5] = poly[3]*fracHeight + vertY*(1.0-fracHeight);
	poly[6] = poly[8]*fracHeight + vertX*(1.0-fracHeight);
	poly[7] = poly[9]*fracHeight + vertY*(1.0-fracHeight);

	/*
	 * for(i = 0; i <= 11; i++) {
	 *     fprintf(stderr, "arrow[%d] = %d\n", i, (int) poly[i]);
	 * }
	 */
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ArrowheadPostscript --
 *
 *	This procedure is called to generate Postscript for
 *	an arrowhead for a line item.
 *
 * Results:
 *	The return value is a standard Tcl result.  If an error
 *	occurs in generating Postscript then an error message is
 *	left in interp->result, replacing whatever used
 *	to be there.  If no error occurs, then Postscript for the
 *	arrowhead is appended to the result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
ArrowheadPostscript(interp, canvas, graphPtr, arrowPtr)
    Tcl_Interp *interp;			/* Leave Postscript or error message
					 * here. */
    Tk_Canvas canvas;			/* Information about overall canvas. */
    PGraphItem *graphPtr;			/* Line item for which Postscript is
					 * being generated. */
    double *arrowPtr;			/* Pointer to first of five points
					 * describing arrowhead polygon. */
{
    Pixmap stipple;
    Tk_State state = graphPtr->header.state;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    stipple = graphPtr->outline.stipple;
    if (((TkCanvas *)canvas)->currentItemPtr == (Tk_Item *)graphPtr) {
	if (graphPtr->outline.activeStipple!=None) {
	    stipple = graphPtr->outline.activeStipple;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (graphPtr->outline.activeStipple!=None) {
	    stipple = graphPtr->outline.disabledStipple;
	}
    }

    Tk_CanvasPsPath(interp, canvas, arrowPtr, PTS_IN_ARROW);
    if (stipple != None) {
	Tcl_AppendResult(interp, "clip ", (char *) NULL);
	if (Tk_CanvasPsStipple(interp, canvas, stipple)
		!= TCL_OK) {
	    return TCL_ERROR;
	}
    } else {
	Tcl_AppendResult(interp, "fill\n", (char *) NULL);
    }
    return TCL_OK;
}

static int
GraphUpdateGC(Tk_Window tkwin, PGraphItem *graphPtr)
{
    GC newGC;
    XGCValues gcValues;
    unsigned long mask;

    if (graphPtr->clipMask == None || graphPtr->drawable == None) {
	GraphCreateDrawable(tkwin, graphPtr);
	GraphUpdateDrawable(tkwin, graphPtr);
    }

    if (graphPtr->clipMask != None && graphPtr->drawable != None) {
	mask = ~0;
	gcValues.function = GXcopy;
	gcValues.foreground = 0;
	gcValues.background = 0;
	gcValues.fill_style = FillTiled;
	gcValues.plane_mask = mask;
	gcValues.clip_x_origin = 0;
	gcValues.clip_y_origin = 0;
	gcValues.clip_mask = graphPtr->clipMask;
	gcValues.tile = graphPtr->drawable;
	gcValues.arc_mode = ArcPieSlice;

	newGC = Tk_GetGC(tkwin,
		(unsigned long) GCFunction|GCPlaneMask|GCClipXOrigin|
		GCClipYOrigin|GCClipMask|GCFillStyle|GCForeground|
		GCBackground|GCTile|GCArcMode, &gcValues);

	if (graphPtr->drawableGC != None)
	    Tk_FreeGC(Tk_Display(tkwin), graphPtr->drawableGC);

	graphPtr->drawableGC = newGC;
    }
}

static int
GraphUpdateDrawable(Tk_Window tkwin, PGraphItem *graphPtr)
{
    Tk_Item *setPtr;
    Display *display;
    int width, height, depth;
    XGCValues gcValues;
    GC newGC;
    int flag;
    unsigned long mask;

    display = Tk_Display(tkwin);
    width = (int) (graphPtr->radius * 2);
    height = (int) (graphPtr->radius * 2);
    depth = Tk_Depth(tkwin);

    /* Clear the drawable and draw a circle in it
     */

    gcValues.function = GXclear;
    if (graphPtr->clipMask != None) {
	newGC = XCreateGC(display, graphPtr->clipMask, GCFunction, &gcValues);
	XFillRectangle(display, graphPtr->clipMask, newGC, 0, 0,
		(unsigned int)width, (unsigned int)height);
	gcValues.function = GXset;
	XChangeGC(display, newGC, GCFunction, &gcValues);
	XFillArc(display, graphPtr->clipMask, newGC, 0, 0,
		(unsigned int)width, (unsigned int)height,
		0, 360*64);
	XFreeGC(display, newGC);
    }

    gcValues.function = GXclear;
    if (graphPtr->drawable != None) {
	newGC = XCreateGC(display, graphPtr->drawable, GCFunction, &gcValues);
	XFillRectangle(display, graphPtr->drawable, newGC, 0, 0,
		(unsigned int)width, (unsigned int)height);
	gcValues.function = GXset;
	XChangeGC(display, newGC, GCFunction, &gcValues);
	XFillArc(display, graphPtr->drawable, newGC, 0, 0,
		(unsigned int)width, (unsigned int)height,
		0, 360*64);
	XFreeGC(display, newGC);
    }

    return TCL_OK;
}

static int
GraphCreateDrawable(Tk_Window tkwin, PGraphItem *graphPtr)
{
    Display *display;
    Drawable d;
    int width, height, depth;
    XGCValues gcValues;
    GC newGC;

    display = Tk_Display(tkwin);
    d = RootWindowOfScreen(Tk_Screen(tkwin));
    width = (int) (graphPtr->radius * 2);
    height = (int) (graphPtr->radius * 2);
    depth = Tk_Depth(tkwin);

    /* Create a new pixmap on which to draw the data sets
     */

    if (graphPtr->drawable != None) {
	Tk_FreePixmap(display, graphPtr->drawable);
    }
    if (graphPtr->clipMask != None) {
	Tk_FreePixmap(display, graphPtr->clipMask);
    }

    if (d != (Drawable) NULL) {
	if (width <= 0) width=1;
	if (height <= 0) height=1;

	graphPtr->drawable = Tk_GetPixmap(display, d, width, height, depth);
	graphPtr->clipMask = Tk_GetPixmap(display, d, width, height, 1);
    }

    return TCL_OK;
}
