/* 
 * tkCanvDataset.c --
 *
 *	This file implements Dataset items for canvas widgets.
 *
 * Copyright (c) 1991-1994 The Regents of the University of California.
 * Copyright (c) 1994-1995 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * SCCS: @(#) tkCanvDataset.c 1.43 96/02/15 18:52:30
 */

#include <stdio.h>
#include <tk.h>
#include <tkInt.h>
#include <tkPort.h>
#include <tkCanvas.h>
#include <tkCanvXYGraph.h>
#include <tkCanvDataset.h>
/*
#include <bindata.h>
*/

#ifdef USE_DMALLOC
#include<dmalloc.h>
#endif
/*
 * the following structure is copied directly from tkCanvas.c.  If
 * that one changes, this one should change as well.
 */

typedef struct TagSearch {
    TkCanvas *canvasPtr;        /* Canvas widget being searched. */
    Tk_Uid tag;                 /* Tag to search for.   0 means return
				 * all items. */
    Tk_Item *prevPtr;           /* Item just before last one found (or NULL
				 * if last one found was first in the item
				 * list of canvasPtr). */
    Tk_Item *currentPtr;        /* Pointer to last item returned. */
    int searchOver;             /* Non-zero means NextItem should always
				 * return NULL. */
} TagSearch;

/*
 * Number of points in an arrowHead:
 */

#define PTS_IN_ARROW 6

/*
 * Prototypes for procedures defined in this file:
 */

static int		ArrowheadPostscript _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, DatasetItem *linePtr,
			    double *arrowPtr));
static void		ComputeDatasetBbox _ANSI_ARGS_((Tk_Canvas canvas,
			    DatasetItem *linePtr));
static int		ConfigureDataset _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int argc,
			    char **argv, int flags));
static int		ConfigureArrows _ANSI_ARGS_((Tk_Canvas canvas,
			    DatasetItem *linePtr));
static int		CreateDataset _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, struct Tk_Item *itemPtr,
			    int argc, char **argv));
static void		DeleteDataset _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, Display *display));
static void		DisplayDataset _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, Display *display, Drawable dst,
			    int x, int y, int width, int height));
static int		GetDatasetIndex _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr,
			    char *indexString, int *indexPtr));
/*
static int		DatasetAccept _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Visitor *visitorPtr,
			    Tk_Item *itemPtr));
*/
static int		DatasetCoords _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr,
			    int argc, char **argv));
static void		DatasetDeleteCoords _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, int first, int last));
static void		DatasetInsert _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, int beforeThis, char *string));
static int		DatasetToArea _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double *rectPtr));
static double		DatasetToPoint _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double *coordPtr));
static int		DatasetToPostscript _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int prepass));
static int		ParseArrowShape _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *interp, Tk_Window tkwin, char *value,
			    char *recordPtr, int offset));
static char *		PrintArrowShape _ANSI_ARGS_((ClientData clientData,
			    Tk_Window tkwin, char *recordPtr, int offset,
			    Tcl_FreeProc **freeProcPtr));
#define ScaleDataset TkScaleDataset
void			ScaleDataset _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double originX, double originY,
			    double scaleX, double scaleY));
#define TranslateDataset TkTranslateDataset
void			TranslateDataset _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double deltaX, double deltaY));
static void		DatasetUpdateGraphCoords _ANSI_ARGS_((DatasetItem *));
static void		DatasetUpdateCanvasCoords _ANSI_ARGS_((DatasetItem *));
static int		Tk_GraphParseProc _ANSI_ARGS_((ClientData,
			    Tcl_Interp *, Tk_Window, char *, char *, int));
static char *		Tk_GraphPrintProc _ANSI_ARGS_((ClientData, Tk_Window,
			    char *, int, Tcl_FreeProc **));

static char *		Tk_DatatypePrintProc _ANSI_ARGS_((ClientData,
			    Tk_Window, char *, int, Tcl_FreeProc **));

static int		Tk_DatastyleParseProc _ANSI_ARGS_((ClientData,
			    Tcl_Interp *, Tk_Window, char *, char *, int));
static char *		Tk_DatastylePrintProc _ANSI_ARGS_((ClientData,
			    Tk_Window, char *, int, Tcl_FreeProc **));

static int		Tk_SymstyleParseProc _ANSI_ARGS_((ClientData,
			    Tcl_Interp *, Tk_Window, char *, char *, int));
static char *		Tk_SymstylePrintProc _ANSI_ARGS_((ClientData,
			    Tk_Window, char *, int, Tcl_FreeProc **));

static Tk_Item *	StartTagSearch _ANSI_ARGS_((TkCanvas *canvasPtr,
			    char *tag, TagSearch *searchPtr));
static int		Tk_DontSetOption _ANSI_ARGS_((ClientData, Tcl_Interp *,
			    Tk_Window, char *, char *, int));
static char * 		Tk_CustomPrintDouble _ANSI_ARGS_((ClientData, 
			    Tk_Window, char *, int, Tcl_FreeProc **));
static void		AddDatasetToGraph _ANSI_ARGS_((DatasetItem *,
			    GraphItem *));
static void		RemoveDatasetFromGraph _ANSI_ARGS_((DatasetItem *,
			    GraphItem *));
static int		DatasetSetInitialType _ANSI_ARGS_((Tcl_Interp *,
			    Tk_Item *, char *));

/*
 * Information used for parsing configuration specs.  If you change any
 * of the default strings, be sure to change the corresponding default
 * values in CreateLine.
 */

static Tk_CustomOption arrowShapeOption = {
    (Tk_OptionParseProc *) ParseArrowShape,
    PrintArrowShape, (ClientData) NULL
};
static Tk_CustomOption stateOption = {
    (Tk_OptionParseProc *) Tk_StateParseProc,
    Tk_StatePrintProc, (ClientData) 2
};
static Tk_CustomOption tagsOption = {
    (Tk_OptionParseProc *) Tk_CanvasTagsParseProc,
    Tk_CanvasTagsPrintProc, (ClientData) NULL
};
static Tk_CustomOption dashOption = {
    (Tk_OptionParseProc *) Tk_CanvasDashParseProc,
    Tk_CanvasDashPrintProc, (ClientData) NULL
};
static Tk_CustomOption tileOption = {
    (Tk_OptionParseProc *) Tk_TileParseProc,
    Tk_TilePrintProc, (ClientData) NULL
};
static Tk_CustomOption graphOption = {Tk_GraphParseProc,
    Tk_GraphPrintProc, (ClientData) NULL
};
static Tk_CustomOption minmaxOption = {Tk_DontSetOption,
    Tk_CustomPrintDouble, (ClientData) NULL
};
static Tk_CustomOption datastyleOption = {Tk_DatastyleParseProc,
    Tk_DatastylePrintProc, (ClientData) NULL
};
static Tk_CustomOption symstyleOption = {Tk_SymstyleParseProc,
    Tk_SymstylePrintProc, (ClientData) NULL
};
static Tk_CustomOption pixelOption = {
    (Tk_OptionParseProc *) Tk_PixelParseProc,
    Tk_PixelPrintProc, (ClientData) NULL
};

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_CUSTOM, "-activedash", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DatasetItem, outline.activeDash),
	TK_CONFIG_NULL_OK, &dashOption},
    {TK_CONFIG_COLOR, "-activefill", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DatasetItem, outline.activeColor),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-activestipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DatasetItem, outline.activeStipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-activetile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DatasetItem, outline.activeTile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_CUSTOM, "-activewidth", (char *) NULL, (char *) NULL,
	"0.0", Tk_Offset(DatasetItem, outline.activeWidth),
	TK_CONFIG_DONT_SET_DEFAULT, &pixelOption},
    {TK_CONFIG_UID, "-arrow", (char *) NULL, (char *) NULL,
	"none", Tk_Offset(DatasetItem, arrow), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-arrowshape", (char *) NULL, (char *) NULL,
	"8 10 3", Tk_Offset(DatasetItem, arrowShapeA),
	TK_CONFIG_DONT_SET_DEFAULT, &arrowShapeOption},
    {TK_CONFIG_CAP_STYLE, "-capstyle", (char *) NULL, (char *) NULL,
	"butt", Tk_Offset(DatasetItem, capStyle), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_COLOR, "-fill", (char *) NULL, (char *) NULL,
	"black", Tk_Offset(DatasetItem, outline.color), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-graph", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DatasetItem, graph),
	TK_CONFIG_NULL_OK, &graphOption},
    {TK_CONFIG_CUSTOM, "-dash", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DatasetItem, outline.dash),
	TK_CONFIG_NULL_OK, &dashOption},
    {TK_CONFIG_PIXELS, "-dashoffset", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(DatasetItem, outline.offset),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-disableddash", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DatasetItem, outline.disabledDash),
	TK_CONFIG_NULL_OK, &dashOption},
    {TK_CONFIG_COLOR, "-disabledfill", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DatasetItem, outline.disabledColor),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-disabledstipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DatasetItem, outline.disabledStipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-disabledtile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DatasetItem, outline.disabledTile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_CUSTOM, "-disabledwidth", (char *) NULL, (char *) NULL,
	"0.0", Tk_Offset(DatasetItem, outline.disabledWidth),
	TK_CONFIG_DONT_SET_DEFAULT, &pixelOption},
    {TK_CONFIG_JOIN_STYLE, "-joinstyle", (char *) NULL, (char *) NULL,
	"round", Tk_Offset(DatasetItem, joinStyle), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_BOOLEAN, "-smooth", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(DatasetItem, smooth), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_INT, "-splinesteps", (char *) NULL, (char *) NULL,
	"12", Tk_Offset(DatasetItem, splineSteps), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-state", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(Tk_Item, state), TK_CONFIG_NULL_OK,
	&stateOption},
    {TK_CONFIG_BITMAP, "-stipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DatasetItem, outline.stipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &tagsOption},
    {TK_CONFIG_CUSTOM, "-tile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DatasetItem, outline.tile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_CUSTOM, "-xmin", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DatasetItem, xmin),
	TK_CONFIG_NULL_OK, &minmaxOption},
    {TK_CONFIG_CUSTOM, "-ymin", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DatasetItem, ymin),
	TK_CONFIG_NULL_OK, &minmaxOption},
    {TK_CONFIG_CUSTOM, "-xmax", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DatasetItem, xmax),
	TK_CONFIG_NULL_OK, &minmaxOption},
    {TK_CONFIG_CUSTOM, "-ymax", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DatasetItem, ymax),
	TK_CONFIG_NULL_OK, &minmaxOption},
    {TK_CONFIG_CUSTOM, "-width", (char *) NULL, (char *) NULL,
	"1.0", Tk_Offset(DatasetItem, outline.width),
	TK_CONFIG_DONT_SET_DEFAULT, &pixelOption},
    {TK_CONFIG_CUSTOM, "-datastyle", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DatasetItem, datastyle),
	TK_CONFIG_NULL_OK, &datastyleOption},
    {TK_CONFIG_CUSTOM, "-symstyle", (char *) NULL, (char *) NULL,
	"circle", Tk_Offset(DatasetItem, symstyle),
	TK_CONFIG_NULL_OK, &symstyleOption},
    {TK_CONFIG_STRING, "-updatecommand", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(Tk_Item, updateCmd), TK_CONFIG_NULL_OK},
    {TK_CONFIG_PIXELS, "-symsize", (char *) NULL, (char *) NULL,
	"4", Tk_Offset(DatasetItem, symsize),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-symwidth", (char *) NULL, (char *) NULL,
	"1.0", Tk_Offset(DatasetItem, symoutline.width),
	TK_CONFIG_DONT_SET_DEFAULT, &pixelOption},
    {TK_CONFIG_COLOR, "-symcolor", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DatasetItem, symoutline.color),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-symdash", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DatasetItem, symoutline.dash),
	TK_CONFIG_NULL_OK, &dashOption},
    {TK_CONFIG_BOOLEAN, "-symopaque", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(DatasetItem, symopaque), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * The structures below defines the line item type by means
 * of procedures that can be invoked by generic item code.
 */

Tk_ItemType tkDatasetType = {
    "dataset",				/* name */
    sizeof(DatasetItem),		/* itemSize */
    CreateDataset,			/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureDataset,			/* configureProc */
    DatasetCoords,			/* coordProc */
    DeleteDataset,			/* deleteProc */
    DisplayDataset,			/* displayProc */
    TK_ITEM_VISITOR_SUPPORT,		/* flags */
    DatasetToPoint,			/* pointProc */
    DatasetToArea,			/* areaProc */
    DatasetToPostscript,		/* postscriptProc */
    ScaleDataset,			/* scaleProc */
    TranslateDataset,			/* translateProc */
    GetDatasetIndex,			/* indexProc */
    (Tk_ItemCursorProc *) NULL,		/* icursorProc */
    (Tk_ItemSelectionProc *) NULL,	/* selectionProc */
    DatasetInsert,			/* insertProc */
    DatasetDeleteCoords,		/* dTextProc */
    (Tk_ItemType *) NULL,		/* nextPtr */
    (Tk_ItemBboxProc *) ComputeDatasetBbox,/* bboxProc */
    Tk_Offset(Tk_VisitorType, visitLine), /* acceptProc */
    (Tk_ItemGetCoordProc *) NULL,	/* getCoordProc */
    (Tk_ItemSetCoordProc *) NULL	/* setCoordProc */
};

/*
 * The Tk_Uid's below refer to uids for the various arrow types:
 */

static Tk_Uid noneUid = NULL;
static Tk_Uid firstUid = NULL;
static Tk_Uid lastUid = NULL;
static Tk_Uid bothUid = NULL;

/*
 * The next two Uid's are used in tag searches
 */

static Tk_Uid allUid = NULL;
static Tk_Uid currentUid = NULL;

/*
 * The definition below determines how large are static arrays
 * used to hold spline points (splines larger than this have to
 * have their arrays malloc-ed).
 */

#define MAX_STATIC_POINTS 200

/*
 *--------------------------------------------------------------
 *
 * CreateLine --
 *
 *	This procedure is invoked to create a new line item in
 *	a canvas.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	interp->result;  in this case itemPtr is left uninitialized,
 *	so it can be safely freed by the caller.
 *
 * Side effects:
 *	A new line item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateDataset(interp, canvas, itemPtr, argc, argv)
    Tcl_Interp *interp;			/* Interpreter for error reporting. */
    Tk_Canvas canvas;			/* Canvas to hold new item. */
    Tk_Item *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing line. */
{
    DatasetItem *linePtr = (DatasetItem *) itemPtr;
    int i;

    /*
     * Carry out initialization that is needed to set defaults and to
     * allow proper cleanup after errors during the the remainder of
     * this procedure.
     */

    if (allUid == NULL)
	allUid = Tk_GetUid("all");
    if (currentUid == NULL)
	currentUid = Tk_GetUid("current");

    Tk_CreateOutline(&(linePtr->outline));
    Tk_CreateOutline(&(linePtr->symoutline));
    linePtr->canvas = canvas;
    linePtr->numPoints = 0;
    linePtr->datacoordPtr = NULL;
    linePtr->graphcoordPtr = NULL;
    linePtr->canvascoordPtr = NULL;
    linePtr->errorPtr = NULL;
    linePtr->capStyle = CapButt;
    linePtr->joinStyle = JoinRound;
    linePtr->arrowGC = None;
    if (noneUid == NULL) {
	noneUid = Tk_GetUid("none");
	firstUid = Tk_GetUid("first");
	lastUid = Tk_GetUid("last");
	bothUid = Tk_GetUid("both");
    }
    linePtr->arrow = noneUid;
    linePtr->arrowShapeA = 8.0;
    linePtr->arrowShapeB = 10.0;
    linePtr->arrowShapeC = 3.0;
    linePtr->firstArrowPtr = NULL;
    linePtr->lastArrowPtr = NULL;
    linePtr->smooth = 0;
    linePtr->splineSteps = 12;
    linePtr->bitmapGC = None;
    linePtr->graph = NULL;
    linePtr->resize_id = 0;
    linePtr->delete_flag = 0;
    linePtr->datastyle = LINEDATA;
    linePtr->symstyle = SYMCIRCLE;
    linePtr->symsize = 4;
    linePtr->symopaque = 0;
    linePtr->xmax = 0.0;
    linePtr->xmin = 0.0;
    linePtr->ymax = 0.0;
    linePtr->ymin = 0.0;
    linePtr->datatype = NULL;

    /*
     * The very first argument must be the dataset type.  We can't just
     * use a normal config spec for this because we need to know the
     * dataset type before we set the coordinates so that we know
     * how many numbers represent a single point.  We need to set the
     * coordinates before we configure the dataset because many config
     * specs depend on knowing the coordinates.  It's a vicious circle
     * that we break out of by not making the dataset type a normal
     * config spec.
     */

    if (DatasetSetInitialType(interp, itemPtr, argv[0]) != TCL_OK) {
	return TCL_ERROR;
    }

    argc--, argv++;

    /*
     * Count the number of points and then parse them into a point
     * array.  Leading arguments are assumed to be points if they
     * start with a digit or a minus sign followed by a digit.
     */

    for (i = 0; i < argc; i++) {
	if ((argv[i][0] == '-') && (argv[i][1] >= 'a')
		&& (argv[i][1] <= 'z')) {
	    break;
	}
    }
    if (i && (DatasetCoords(interp, canvas, itemPtr, i, argv) != TCL_OK)) {
	goto error;
    }
    if (ConfigureDataset(interp, canvas, itemPtr, argc-i, argv+i, 0) == TCL_OK) {
	return TCL_OK;
    }

    error:
    DeleteDataset(canvas, itemPtr, Tk_Display(Tk_CanvasTkwin(canvas)));
    return TCL_ERROR;
}

static int
DatasetSetInitialType(Tcl_Interp *interp, Tk_Item *itemPtr, char *type)
{
    DatasetItem *linePtr = (DatasetItem *) itemPtr;

    if (strcasecmp(type, "xy") == 0) {
	linePtr->datatype = &xyData;
    } else if (strcasecmp(type, "xysigy") == 0) {
	linePtr->datatype = &xysigyData;
    } else if (strcasecmp(type, "xysigy1sigy2") == 0) {
	linePtr->datatype = &xysigy1sigy2Data;
    } else if (strcasecmp(type, "xsigxy") == 0) {
	linePtr->datatype = &xsigxyData;
    } else if (strcasecmp(type, "xsigx1sigx2y") == 0) {
	linePtr->datatype = &xsigx1sigx2yData;
    } else if (strcasecmp(type, "xsigxysigy") == 0) {
	linePtr->datatype = &xsigxysigyData;
    } else if (strcasecmp(type, "xsigx1sigx2ysigy1sigy2") == 0) {
	linePtr->datatype = &xsigx1sigx2ysigy1sigy2Data;
    } else {
	Tcl_SetResult(interp, "wrong # args: should be \"canvas create dataset datatype coords ?config specs?\"", TCL_STATIC);
	return TCL_ERROR;
    }

    return TCL_OK;
}


/*
 *--------------------------------------------------------------
 *
 * DatasetCoords --
 *
 *	This procedure is invoked to process the "coords" widget
 *	command on lines.  See the user documentation for details
 *	on what it does.
 *
 * Results:
 *	Returns TCL_OK or TCL_ERROR, and sets interp->result.
 *
 * Side effects:
 *	The coordinates for the given item may be changed.
 *
 *--------------------------------------------------------------
 */

static int
DatasetCoords(interp, canvas, itemPtr, argc, argv)
    Tcl_Interp *interp;			/* Used for error reporting. */
    Tk_Canvas canvas;			/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item whose coordinates are to be
					 * read or modified. */
    int argc;				/* Number of coordinates supplied in
					 * argv. */
    char **argv;			/* Array of coordinates: x1, y1,
					 * x2, y2, ... */
{
    DatasetItem *linePtr = (DatasetItem *) itemPtr;
    char buffer[TCL_DOUBLE_SPACE], **largv = NULL;
    int i, numPoints=0;
    double *coordPtr;
    Datapoint *errorPtr;
    Tcl_Obj *objPtr = (Tcl_Obj *)NULL;
    int ds = linePtr->datatype->data_size;
    int xi = linePtr->datatype->xindex;
    int yi = linePtr->datatype->yindex;
    int parseAsBinaryString = 0;

    if (argc < 0) {
	/*
	 * This is our special signal to recompute the graph/canvas coordinates
	 */
	DatasetUpdateGraphCoords(linePtr);
	DatasetUpdateCanvasCoords(linePtr);
	linePtr->resize_id = linePtr->graph->resize_id;
	ComputeDatasetBbox(canvas, linePtr);

	return TCL_OK;
    } else if (argc == 0) {
	int numCoords = linePtr->numPoints;

	coordPtr = linePtr->datacoordPtr;
	for (i = 0; i < numCoords; i++) {
	    Tcl_PrintDouble(interp, coordPtr[i*2], buffer);
	    Tcl_AppendElement(interp, buffer);

	    if (linePtr->datatype->xsighindex > -1) {
		Tcl_PrintDouble(interp, linePtr->errorPtr[i].xsigh, buffer);
		Tcl_AppendElement(interp, buffer);
	    }
	    if (linePtr->datatype->xsiglindex > -1 &&
		linePtr->datatype->xsiglindex != linePtr->datatype->xsighindex){
		Tcl_PrintDouble(interp, linePtr->errorPtr[i].xsigl, buffer);
		Tcl_AppendElement(interp, buffer);
	    }

	    Tcl_PrintDouble(interp, coordPtr[i*2+1], buffer);
	    Tcl_AppendElement(interp, buffer);

	    if (linePtr->datatype->ysighindex > -1) {
		Tcl_PrintDouble(interp, linePtr->errorPtr[i].ysigh, buffer);
		Tcl_AppendElement(interp, buffer);
	    }
	    if (linePtr->datatype->ysiglindex > -1 &&
		linePtr->datatype->ysiglindex != linePtr->datatype->ysighindex){
		Tcl_PrintDouble(interp, linePtr->errorPtr[i].ysigl, buffer);
		Tcl_AppendElement(interp, buffer);
	    }

	}
	return TCL_OK;
    }

    if (argc == 1) {
	/* If we have one argument, then assume that it's a Bin_Object object.
	 * Create the coordinate array from this object.  The idea is that
	 * we want to be able to pass the graph a pre-existing binary
	 * structure containing the coordinates so that creating a new
	 * dataset does not involve billions of calls to Tcl_GetDouble().
	 */
	/*
	 * If we have one argument, then assume it's a binary string
	 * containing floating point values in network byte order.
	 * De-byte-order the numbers into machine byte order and store
	 * in the coords arrays.
	 */

	if (Tcl_SplitList(interp, argv[0], &argc, &largv) != TCL_OK) {
	    if (largv != NULL) {
		ckfree((char *) largv);
	    }
	    return TCL_ERROR;
	}

	if (argc == 1) {
	    parseAsBinaryString = 1;
	}
    } else {
	largv = argv;
    }
    if ((argc%(linePtr->datatype->data_size) != 0) && !parseAsBinaryString) {
	Tcl_AppendResult(interp,
		"wrong number of coordinates specified for line",
		(char *) NULL);
	if (largv != NULL && largv != argv) {
	    ckfree((char *) largv);
	}
	return TCL_ERROR;
    } else {
	if (parseAsBinaryString) {
	    Tcl_Obj *part1 = Tcl_NewStringObj(*largv, strlen(*largv));

	    objPtr = Tcl_ObjGetVar2(interp, part1, (Tcl_Obj *)NULL, TCL_LEAVE_ERR_MSG);
	    if (!objPtr) {
		return TCL_ERROR;
	    }

	    numPoints = objPtr->length;
	    
	    if (numPoints % 4 != 0) {
		Tcl_SetResult(interp, "Bad number of bytes for data.  Should be a multiple of 4", TCL_VOLATILE);
		return TCL_ERROR;
	    }
	    
	    numPoints /= 4;
	} else {
	    numPoints = argc;
	}

	if ((numPoints % linePtr->datatype->data_size) != 0) {
	    char digit[4];
	    sprintf(digit, "%d", linePtr->datatype->data_size);

	    Tcl_AppendResult(interp,
	        "Wrong # of coordinates.  There should be a multiple of ",
		digit,
		" numbers for a ",
		linePtr->datatype->datatypestr,
		" dataset",
		(char *)NULL);
	}
	numPoints /= ds;

	if (linePtr->numPoints != numPoints) {
	    coordPtr = (double *) ckalloc((unsigned)
		    (sizeof(double) * numPoints*2));
	    if (linePtr->datacoordPtr != NULL) {
		ckfree((char *) linePtr->datacoordPtr);
	    }
	    linePtr->datacoordPtr = coordPtr;

	    coordPtr = (double *) ckalloc((unsigned)
		    (sizeof(double) * numPoints*2));
	    if (linePtr->canvascoordPtr != NULL) {
		ckfree((char *) linePtr->canvascoordPtr);
	    }
	    linePtr->canvascoordPtr = coordPtr;

	    coordPtr = (double *) ckalloc((unsigned)
		    (sizeof(double) * numPoints*2));
	    if (linePtr->graphcoordPtr != NULL) {
		ckfree((char *) linePtr->graphcoordPtr);
	    }
	    linePtr->graphcoordPtr = coordPtr;

	    errorPtr = (Datapoint *)ckalloc(sizeof(Datapoint)*numPoints);
	    if (linePtr->errorPtr != NULL) {
		ckfree((char *)linePtr->errorPtr);
	    }
	    linePtr->errorPtr = errorPtr;

	    linePtr->numPoints = numPoints;
	}

	/*
	 * Make a separate loop for the case where the data is
	 * a Binary string.  We want to make dealing with binary
	 * objects as fast as possible.
	 */

	if (objPtr != (Tcl_Obj *)NULL) {
	    double d;
	    float *objData = (float *)Tcl_GetStringFromObj(objPtr, (int *)NULL);
	    float *objCoordPtr = (float *)ckalloc(objPtr->length+1);
	    
	    /*
	     * First convert the floats from network byte order to machine
	     * byte order.
	     */
	    
	    for(i=0; i < objPtr->length / 4; i++) {
		/*
		long nl = ((long *)(objData))[i];
		
		nl = ntohl(nl);
		((long *)(objCoordPtr))[i] = nl;
		*/

		objCoordPtr[i] = objData[i];
	    }

	    coordPtr = linePtr->datacoordPtr;

	    linePtr->xmin = objCoordPtr[linePtr->datatype->xindex];
	    linePtr->ymin = objCoordPtr[linePtr->datatype->yindex];
	    linePtr->xmax = linePtr->xmin;
	    linePtr->ymax = linePtr->ymin;

	    for (i = 0; i <numPoints; i++) {
		linePtr->errorPtr[i].xsigh = 0;
		linePtr->errorPtr[i].xsigl = 0;
		linePtr->errorPtr[i].ysigh = 0;
		linePtr->errorPtr[i].ysigl = 0;

		coordPtr[i*ds] = objCoordPtr[i*ds+xi];
		coordPtr[i*ds+1] = objCoordPtr[i*ds+yi];
		if (linePtr->datatype->xsighindex > -1) {
		    linePtr->errorPtr[i].xsigh = 
			objCoordPtr[i*ds+linePtr->datatype->xsighindex];
		}
		if (linePtr->datatype->xsiglindex > -1) {
		    linePtr->errorPtr[i].xsigl = 
			objCoordPtr[i*ds+linePtr->datatype->xsiglindex];
		}
		if (linePtr->datatype->ysighindex > -1) {
		    linePtr->errorPtr[i].ysigh = 
			objCoordPtr[i*ds+linePtr->datatype->ysighindex];
		}
		if (linePtr->datatype->ysiglindex > -1) {
		    linePtr->errorPtr[i].ysigl = 
			objCoordPtr[i*ds+linePtr->datatype->ysiglindex];
		}

		if (coordPtr[i*ds] < linePtr->xmin)
		    linePtr->xmin = coordPtr[i*ds];
		if (coordPtr[i*ds] > linePtr->xmax)
		    linePtr->xmax = coordPtr[i*ds];
		if (coordPtr[i*ds+1] < linePtr->ymin)
		    linePtr->ymin = coordPtr[i*ds+1];
		if (coordPtr[i*ds+1] > linePtr->ymax)
		    linePtr->ymax = coordPtr[i*ds+1];

		linePtr->graphcoordPtr[i*ds] = coordPtr[i*ds];
		linePtr->canvascoordPtr[i*ds] = coordPtr[i*ds];
		linePtr->graphcoordPtr[i*ds+1] = coordPtr[i*ds+1];
		linePtr->canvascoordPtr[i*ds+1] = coordPtr[i*ds+1];
	    }
	    if (largv != NULL && largv != argv) {
		ckfree((char *) largv);
	    }
	    if (objCoordPtr) {
		ckfree(objCoordPtr);
	    }
	} else {
	    coordPtr = linePtr->datacoordPtr;
	    Tk_CanvasGetCoord(interp, canvas, largv[0], &linePtr->xmin);
	    linePtr->xmax = linePtr->xmin;
	    Tk_CanvasGetCoord(interp, canvas, largv[1], &linePtr->ymin);
	    linePtr->ymax = linePtr->ymin;

	    /*
	     * Set default min/max values to the first coordinate
	     */

	    for (i = 0; i <numPoints; i++) {
		linePtr->errorPtr[i].x = 0.0;
		linePtr->errorPtr[i].y = 0.0;
		linePtr->errorPtr[i].xsigh = 0.0;
		linePtr->errorPtr[i].xsigl = 0.0;
		linePtr->errorPtr[i].ysigh = 0.0;
		linePtr->errorPtr[i].ysigl = 0.0;

		/*
		 * When dealing with linePtr->datacoordPtr,
		 * linePtr->canvascoordPtr, and linePtr->graphcoordPtr,
		 * use "2" instead of "ds" since those arrays only
		 * hold the x-y pairs.
		 */

		if (Tk_CanvasGetCoord(interp, canvas, largv[i*ds],
			coordPtr+i*2) != TCL_OK) {
		    if (largv != NULL && largv != argv) {
			ckfree((char *) largv);
		    }
		    return TCL_ERROR;
		}
		if (Tk_CanvasGetCoord(interp, canvas, largv[i*ds+linePtr->datatype->yindex],
			coordPtr+i*2+1) != TCL_OK) {
		    if (largv != NULL && largv != argv) {
			ckfree((char *) largv);
		    }
		    return TCL_ERROR;
		}
		if (linePtr->datatype->xsighindex > -1) {
		    if (Tk_CanvasGetCoord(interp, canvas,
			    largv[i*ds+linePtr->datatype->xsighindex],
			    &(linePtr->errorPtr[i].xsigh)) != TCL_OK) {
			if (largv != NULL && largv != argv)
			    ckfree((char *)largv);
			return TCL_ERROR;
		    }
		}
		if (linePtr->datatype->xsiglindex > -1) {
		    if (Tk_CanvasGetCoord(interp, canvas,
			    largv[i*ds+linePtr->datatype->xsiglindex],
			    &(linePtr->errorPtr[i].xsigl)) != TCL_OK) {
			if (largv != NULL && largv != argv)
			    ckfree((char *)largv);
			return TCL_ERROR;
		    }
		}
		if (linePtr->datatype->ysighindex > -1) {
		    if (Tk_CanvasGetCoord(interp, canvas,
			    largv[i*ds+linePtr->datatype->ysighindex],
			    &(linePtr->errorPtr[i].ysigh)) != TCL_OK) {
			if (largv != NULL && largv != argv)
			    ckfree((char *)largv);
			return TCL_ERROR;
		    }
		}
		if (linePtr->datatype->ysiglindex > -1) {
		    if (Tk_CanvasGetCoord(interp, canvas,
			    largv[i*ds+linePtr->datatype->ysiglindex],
			    &(linePtr->errorPtr[i].ysigl)) != TCL_OK) {
			if (largv != NULL && largv != argv)
			    ckfree((char *)largv);
			return TCL_ERROR;
		    }
		}

		if (coordPtr[i*2] < linePtr->xmin)
		    linePtr->xmin = coordPtr[i*2];
		if (coordPtr[i*2] > linePtr->xmax)
		    linePtr->xmax = coordPtr[i*2];
		if (coordPtr[i*2+1] < linePtr->ymin)
		    linePtr->ymin = coordPtr[i*2+1];
		if (coordPtr[i*2+1] > linePtr->ymax)
		    linePtr->ymax = coordPtr[i*2+1];

		linePtr->graphcoordPtr[i*2] = coordPtr[i*2];
		linePtr->canvascoordPtr[i*2] = coordPtr[i*2];
		linePtr->graphcoordPtr[i*2+1] = coordPtr[i*2+1];
		linePtr->canvascoordPtr[i*2+1] = coordPtr[i*2+1];
	    }
	    if (largv != NULL && largv != argv) {
		ckfree((char *) largv);
	    }
	}

	DatasetUpdateGraphCoords(linePtr);
	DatasetUpdateCanvasCoords(linePtr);

	/*
	 * for(i = 0; i < numPoints*2; i+=2) {
	 *     fprintf(stderr, "(%g,%g)\n", linePtr->datacoordPtr[i], 
	 *	    linePtr->datacoordPtr[i+1]);
	 * }
	 */

	/*
	 * Update arrowheads by throwing away any existing arrow-head
	 * information and calling ConfigureArrows to recompute it.
	 */

	if (linePtr->firstArrowPtr != NULL) {
	    ckfree((char *) linePtr->firstArrowPtr);
	    linePtr->firstArrowPtr = NULL;
	}
	if (linePtr->lastArrowPtr != NULL) {
	    ckfree((char *) linePtr->lastArrowPtr);
	    linePtr->lastArrowPtr = NULL;
	}
	if (linePtr->arrow != noneUid) {
	    ConfigureArrows(canvas, linePtr);
	}
    }

    ComputeDatasetBbox(canvas, linePtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureDataset --
 *
 *	This procedure is invoked to configure various aspects
 *	of a line item such as its background color.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in interp->result.
 *
 * Side effects:
 *	Configuration information, such as colors and stipple
 *	patterns, may be set for itemPtr.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureDataset(interp, canvas, itemPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Canvas canvas;		/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Line item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    DatasetItem *linePtr = (DatasetItem *) itemPtr;
    XGCValues gcValues;
    GC newGC, arrowGC;
    unsigned long mask;
    Tk_Window tkwin;
    Tk_State state;

    tkwin = Tk_CanvasTkwin(canvas);
    if (Tk_ConfigureWidget(interp, tkwin, configSpecs, argc, argv,
	    (char *) linePtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * A few of the options require additional processing, such as
     * graphics contexts.
     *
     * Before we do these, however, we need to update the graph coordinates
     * and the canvas coordinates.
     */

    state = itemPtr->state;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    DatasetUpdateGraphCoords(linePtr);
    DatasetUpdateCanvasCoords(linePtr);

    if (linePtr->outline.activeWidth > linePtr->outline.width ||
	    linePtr->outline.activeDash.number > 0 ||
	    linePtr->outline.activeColor != NULL ||
	    linePtr->outline.activeStipple != None) {
	itemPtr->redraw_flags |= TK_ITEM_STATE_DEPENDANT;
    } else {
	itemPtr->redraw_flags &= ~TK_ITEM_STATE_DEPENDANT;
    }
    mask = Tk_ConfigOutlineGC(&gcValues, canvas, itemPtr,
	    &(linePtr->outline));
    if (mask) {
	if (linePtr->arrow == noneUid) {
	    gcValues.cap_style = linePtr->capStyle;
	    mask |= GCCapStyle;
	}
	gcValues.join_style = linePtr->joinStyle;
	mask |= GCJoinStyle;
	newGC = Tk_GetGC(tkwin, mask, &gcValues);
	gcValues.line_width = 0;
	arrowGC = Tk_GetGC(tkwin, mask, &gcValues);
    } else {
	newGC = arrowGC = None;
    }
    if (linePtr->outline.gc != None) {
	Tk_FreeGC(Tk_Display(tkwin), linePtr->outline.gc);
    }
    if (linePtr->arrowGC != None) {
	Tk_FreeGC(Tk_Display(tkwin), linePtr->arrowGC);
    }
    linePtr->outline.gc = newGC;
    linePtr->arrowGC = arrowGC;

    /*
     * Set up the outlines for the symbols
     */

    mask = Tk_ConfigOutlineGC(&gcValues, canvas, itemPtr,
     		&(linePtr->symoutline));
    
    if (mask) {
	gcValues.cap_style = CapButt;
	mask |= GCCapStyle;
	newGC = Tk_GetGC(tkwin, mask, &gcValues);
    } else {
	newGC = None;
    }

    if (linePtr->symoutline.gc != None) {
	Tk_FreeGC(Tk_Display(tkwin), linePtr->symoutline.gc);
    }
    linePtr->symoutline.gc = newGC;

    /*
     * Keep spline parameters within reasonable limits.
     */

    if (linePtr->splineSteps < 1) {
	linePtr->splineSteps = 1;
    } else if (linePtr->splineSteps > 100) {
	linePtr->splineSteps = 100;
    }

    if ((!linePtr->numPoints) || (state==TK_STATE_HIDDEN)) {
	ComputeDatasetBbox(canvas, linePtr);
	return TCL_OK;
    }
    /*
     * Setup arrowheads, if needed.  If arrowheads are turned off,
     * restore the line's endpoints (they were shortened when the
     * arrowheads were added).
     */

    if ((linePtr->firstArrowPtr != NULL) && (linePtr->arrow != firstUid)
	    && (linePtr->arrow != bothUid)) {
	linePtr->canvascoordPtr[0] = linePtr->firstArrowPtr[0];
	linePtr->canvascoordPtr[1] = linePtr->firstArrowPtr[1];
	ckfree((char *) linePtr->firstArrowPtr);
	linePtr->firstArrowPtr = NULL;
    }
    if ((linePtr->lastArrowPtr != NULL) && (linePtr->arrow != lastUid)
	    && (linePtr->arrow != bothUid)) {
	int i;

	i = 2*(linePtr->numPoints-1);
	linePtr->canvascoordPtr[i] = linePtr->lastArrowPtr[0];
	linePtr->canvascoordPtr[i+1] = linePtr->lastArrowPtr[1];
	ckfree((char *) linePtr->lastArrowPtr);
	linePtr->lastArrowPtr = NULL;
    }
    if (linePtr->arrow != noneUid) {
	if ((linePtr->arrow != firstUid) && (linePtr->arrow != lastUid)
		&& (linePtr->arrow != bothUid)) {
	    Tcl_AppendResult(interp, "bad arrow spec \"",
		    linePtr->arrow, "\": must be none, first, last, or both",
		    (char *) NULL);
	    linePtr->arrow = noneUid;
	    return TCL_ERROR;
	}
	ConfigureArrows(canvas, linePtr);
    }

    if (linePtr->bitmapGC == None) {
	Display *display = Tk_Display(Tk_CanvasTkwin(canvas));
	Drawable d = Tk_WindowId(Tk_MainWindow(interp));
	Pixmap bitmap = None;
	gcValues.function = GXclear;
	gcValues.foreground = 1;
	gcValues.background = 0;
	gcValues.fill_style = FillSolid;
	gcValues.plane_mask = 1;
	gcValues.clip_x_origin = 0;
	gcValues.clip_y_origin = 0;

	/* Create a dummy bitmap of depth one to use as our drawable for
	 * the GC creation so that our GC will be able to draw on a bitmap.
	 */

	if (d) {
	    bitmap = Tk_GetPixmap(display, d, 2, 2, 1);

	    newGC = XCreateGC(display, d, GCFunction|GCForeground|GCBackground|GCFillStyle|GCPlaneMask|GCClipXOrigin|GCClipYOrigin, &gcValues);
	    if (linePtr->bitmapGC != None)
		XFreeGC(display, linePtr->bitmapGC);

	    linePtr->bitmapGC = newGC;

	    Tk_FreePixmap(display, bitmap);
	}
    }

    /*
     * Recompute bounding box for line.
     */

    Tk_CanvasEventuallyRedraw(canvas, linePtr->header.x1, linePtr->header.y2,
	    linePtr->header.x2, linePtr->header.y2);
    ComputeDatasetBbox(canvas, linePtr);

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DeleteDataset --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a line item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteDataset(canvas, itemPtr, display)
    Tk_Canvas canvas;			/* Info about overall canvas widget. */
    Tk_Item *itemPtr;			/* Item that is being deleted. */
    Display *display;			/* Display containing window for
					 * canvas. */
{
    DatasetItem *linePtr = (DatasetItem *) itemPtr;

    linePtr->delete_flag = 1;
    Tk_DeleteOutline(display, &(linePtr->outline));
    if (linePtr->datacoordPtr != NULL) {
	ckfree((char *) linePtr->datacoordPtr);
    }
    if (linePtr->graphcoordPtr != NULL) {
	ckfree((char *) linePtr->graphcoordPtr);
    }
    if (linePtr->canvascoordPtr != NULL) {
	ckfree((char *) linePtr->canvascoordPtr);
    }
    if (linePtr->arrowGC != None) {
	Tk_FreeGC(display, linePtr->arrowGC);
    }
    if (linePtr->firstArrowPtr != NULL) {
	ckfree((char *) linePtr->firstArrowPtr);
    }
    if (linePtr->lastArrowPtr != NULL) {
	ckfree((char *) linePtr->lastArrowPtr);
    }
    if (linePtr->bitmapGC != None) {
	XFreeGC(display, linePtr->bitmapGC);
    }
    if (linePtr->graph) {
	RemoveDatasetFromGraph(linePtr, linePtr->graph);
    }
}

/*
 *--------------------------------------------------------------
 *
 * ComputeDatasetBbox --
 *
 *	This procedure is invoked to compute the bounding box of
 *	all the pixels that may be drawn as part of a line.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The fields x1, y1, x2, and y2 are updated in the header
 *	for itemPtr.
 *
 *--------------------------------------------------------------
 */

static void
ComputeDatasetBbox(canvas, linePtr)
    Tk_Canvas canvas;			/* Canvas that contains item. */
    DatasetItem *linePtr;		/* Item whose bbox is to be
					 * recomputed. */
{
    double *coordPtr;
    int i,width;
    Tk_State state = linePtr->header.state;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    if (!(linePtr->numPoints) || (state==TK_STATE_HIDDEN)
	|| (linePtr->graph == NULL)) {
	linePtr->header.x1 = -1;
	linePtr->header.x2 = -1;
	linePtr->header.y1 = -1;
	linePtr->header.y2 = -1;
	return;
    }

    /*
     * The bounding box for the dataset is identical to the bounding
     * box for the parent graph
     */

    if (linePtr->graph) {
	linePtr->header.x1 = linePtr->graph->header.x1;
	linePtr->header.x2 = linePtr->graph->header.x2;
	linePtr->header.y1 = linePtr->graph->header.y1;
	linePtr->header.y2 = linePtr->graph->header.y2;

	return;
    }

    width = linePtr->outline.width;
    if (((TkCanvas *)canvas)->currentItemPtr == (Tk_Item *)linePtr) {
	if (linePtr->outline.activeWidth>width) {
	    width = linePtr->outline.activeWidth;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (linePtr->outline.disabledWidth>0) {
	    width = linePtr->outline.disabledWidth;
	}
    }
    coordPtr = linePtr->canvascoordPtr;
    linePtr->header.x1 = linePtr->header.x2 = *coordPtr;
    linePtr->header.y1 = linePtr->header.y2 = coordPtr[1];

    if (linePtr->numPoints==1) {
	linePtr->header.x1 -= (width+3)/2;
	linePtr->header.y1 -= (width+3)/2;
	linePtr->header.x2 += (width+3)/2;
	linePtr->header.y2 += (width+3)/2;
	return;
    }
    /*
     * Compute the bounding box of all the points in the line,
     * then expand in all directions by the line's width to take
     * care of butting or rounded corners and projecting or
     * rounded caps.  This expansion is an overestimate (worst-case
     * is square root of two over two) but it's simple.  Don't do
     * anything special for curves.  This causes an additional
     * overestimate in the bounding box, but is faster.
     */

    for (i = 1, coordPtr = linePtr->canvascoordPtr+2; i < linePtr->numPoints;
	    i++, coordPtr += 2) {
	TkIncludePoint((Tk_Item *) linePtr, coordPtr);
    }
    linePtr->header.x1 -= width;
    linePtr->header.x2 += width;
    linePtr->header.y1 -= width;
    linePtr->header.y2 += width;

    /*
     * For mitered lines, make a second pass through all the points.
     * Compute the locations of the two miter vertex points and add
     * those into the bounding box.
     */

    if (linePtr->joinStyle == JoinMiter) {
	for (i = linePtr->numPoints, coordPtr = linePtr->canvascoordPtr; i >= 3;
		i--, coordPtr += 2) {
	    double miter[4];
	    int j;
    
	    if (TkGetMiterPoints(coordPtr, coordPtr+2, coordPtr+4,
		    (double) width, miter, miter+2)) {
		for (j = 0; j < 4; j += 2) {
		    TkIncludePoint((Tk_Item *) linePtr, miter+j);
		}
	    }
	}
    }

    /*
     * Add in the sizes of arrowheads, if any.
     */

    if (linePtr->arrow != noneUid) {
	if (linePtr->arrow != lastUid) {
	    for (i = 0, coordPtr = linePtr->firstArrowPtr; i < PTS_IN_ARROW;
		    i++, coordPtr += 2) {
		TkIncludePoint((Tk_Item *) linePtr, coordPtr);
	    }
	}
	if (linePtr->arrow != firstUid) {
	    for (i = 0, coordPtr = linePtr->lastArrowPtr; i < PTS_IN_ARROW;
		    i++, coordPtr += 2) {
		TkIncludePoint((Tk_Item *) linePtr, coordPtr);
	    }
	}
    }

    /*
     * Add one more pixel of fudge factor just to be safe (e.g.
     * X may round differently than we do).
     */

    linePtr->header.x1 -= 1;
    linePtr->header.x2 += 1;
    linePtr->header.y1 -= 1;
    linePtr->header.y2 += 1;
}

/*
 *--------------------------------------------------------------
 *
 * DisplayDataset --
 *
 *	This procedure is invoked to draw a line item in a given
 *	drawable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in drawable using the transformation
 *	information in canvas.
 *
 *--------------------------------------------------------------
 */

static void
DisplayDataset(canvas, itemPtr, display, drawable, x, y, width, height)
    Tk_Canvas canvas;			/* Canvas that contains item. */
    Tk_Item *itemPtr;			/* Item to be displayed. */
    Display *display;			/* Display on which to draw item. */
    Drawable drawable;			/* Pixmap or window in which to draw
					 * item. */
    int x, y, width, height;		/* Describes region of canvas that
					 * must be redisplayed (not used). */
{
    DatasetItem *linePtr = (DatasetItem *) itemPtr;
    XPoint staticPoints[MAX_STATIC_POINTS];
    XPoint *pointPtr = (XPoint *)NULL;
    XPoint *pPtr = (XPoint *)NULL;
    double *coordPtr = (double *)NULL;
    double *temp_coordPtr = (double *)NULL;
    short graph_clip_xorigin, graph_clip_yorigin;
    XRectangle rect;
    int i, numPoints;
    Tk_State state = itemPtr->state;
    Pixmap graph_clip_mask;
    int segcount=0;
    XSegment *errsegment=NULL;

    /*
     * Get the scale from the graph's axis limits
     */

    double xsize = linePtr->graph->xaxismax - linePtr->graph->xaxismin;
    double ysize = linePtr->graph->yaxismax - linePtr->graph->yaxismin;

    double xscale = (linePtr->graph->bbox[2] - linePtr->graph->bbox[0]) / xsize;
    double yscale = (linePtr->graph->bbox[3] - linePtr->graph->bbox[1]) / ysize;

    /*
     * Draw the line into the parent graph's drawable pixmap.  Then copy the
     * pixmap onto the drawable that we were provided here.
     *
     * If there is no parent graph, then don't do anything.
     */

    if (linePtr->graph == NULL)
	return;

    if (linePtr->graph->resize_id != linePtr->resize_id) {
	DatasetUpdateGraphCoords(linePtr);
	DatasetUpdateCanvasCoords(linePtr);
	ComputeDatasetBbox(canvas, linePtr);
	linePtr->resize_id = linePtr->graph->resize_id;
    }

    /*
     * We want to clip the datasets at the graph boundaries so we don't
     * get lines that extend past the edges of the graph.
     *
     * linePtr->graph->clipMask contains a bitmap of 1's which is exactly
     * the same size as the graph.  All we have to do is set the
     * clip origin.
     */

    graph_clip_mask = linePtr->graph->clipMask;
    Tk_CanvasDrawableCoords(canvas, linePtr->graph->bbox[0], 
	    linePtr->graph->bbox[1],
	    &graph_clip_xorigin, &graph_clip_yorigin);

    if ((!linePtr->numPoints)||(linePtr->outline.gc==None)) {
	return;
    }

    /*
     * Build up an array of points in screen coordinates.  Use a
     * static array unless the line has an enormous number of points;
     * in this case, dynamically allocate an array.  For smoothed lines,
     * generate the curve points on each redisplay.
     */

    if ((linePtr->smooth) && (linePtr->numPoints > 2)) {
	numPoints = 1 + linePtr->numPoints*linePtr->splineSteps;
    } else {
	numPoints = linePtr->numPoints;
    }

    if (numPoints <= MAX_STATIC_POINTS) {
	pointPtr = staticPoints;
    } else {
	pointPtr = (XPoint *) ckalloc((unsigned) (numPoints * sizeof(XPoint)));
    }

    if ((linePtr->smooth) && (linePtr->numPoints > 2)) {
	temp_coordPtr = (double *)ckalloc((unsigned) (numPoints * sizeof(double)));
	numPoints = TkMakeBezierCurve(canvas, linePtr->canvascoordPtr,
		linePtr->numPoints, linePtr->splineSteps, (XPoint *)NULL,
		temp_coordPtr);

	for(i = 0, pPtr = pointPtr, coordPtr = temp_coordPtr;
		i < linePtr->numPoints; i+= 2, coordPtr+=2, pPtr++) {
	    pPtr->x = (short) coordPtr[0];
	    pPtr->y = (short) coordPtr[1];
	}

	ckfree((char *)temp_coordPtr);
    } else {
	/*
	fprintf(stderr, "Display coords:\n");
	*/
	for (i = 0, coordPtr = linePtr->canvascoordPtr, pPtr = pointPtr;
		i < linePtr->numPoints;  i += 1, coordPtr += 2, pPtr++) {
		Tk_CanvasDrawableCoords(canvas, 
			coordPtr[0], coordPtr[1],
			&pPtr->x, &pPtr->y);
		/*
		fprintf(stderr, "\t(%d,%d)\n", pPtr->x, pPtr->y);
		*/
	}
    }
    /*
     * Display line, then free up line storage if it was dynamically
     * allocated.  If we're stippling, then modify the stipple offset
     * in the GC.  Be sure to reset the offset when done, since the
     * GC is supposed to be read-only.
     */

    if (Tk_ChangeOutlineGC(canvas, itemPtr, &(linePtr->outline))) {
	Tk_CanvasSetStippleOrigin(canvas, linePtr->arrowGC);
    }

    XSetClipMask(display, linePtr->outline.gc, linePtr->graph->clipMask);
    XSetClipOrigin(display, linePtr->outline.gc, graph_clip_xorigin,
	    graph_clip_yorigin);

    if (linePtr->symoutline.gc) {
	XSetClipMask(display, linePtr->symoutline.gc, linePtr->graph->clipMask);
	XSetClipOrigin(display, linePtr->symoutline.gc, graph_clip_xorigin,
		graph_clip_yorigin);
    }

    if ((linePtr->datastyle & LINEDATA) && linePtr->outline.gc != None) {

	if (numPoints>1) {
	    XDrawLines(display, drawable, linePtr->outline.gc, pointPtr,
		    numPoints, CoordModeOrigin);
	} else {
	    int width = linePtr->outline.width;
	    if(state == TK_STATE_NULL) {
		state = ((TkCanvas *)canvas)->canvas_state;
	    }
	    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
		if (linePtr->outline.activeWidth>width) {
		    width = linePtr->outline.activeWidth;
		}
	    } else if (state==TK_STATE_DISABLED) {
		if (linePtr->outline.disabledWidth>0) {
		    width = linePtr->outline.disabledWidth;
		}
	    }
	    XFillArc(display, drawable, linePtr->outline.gc, pointPtr->x - width/2,
		    pointPtr->y - width/2, width+1, width+1, 0, 64*360);
	}

	/*
	 * Display arrowheads, if they are wanted.
	 *
	 * 98.03.20  Note from Mike:
	 * They are not wanted.
	 */

	/*
	    if (linePtr->firstArrowPtr != NULL) {
		TkFillPolygon(canvas, linePtr->firstArrowPtr, PTS_IN_ARROW,
			display, drawable, linePtr->arrowGC, NULL);
	    }
	    if (linePtr->lastArrowPtr != NULL) {
		TkFillPolygon(canvas, linePtr->lastArrowPtr, PTS_IN_ARROW,
			display, drawable, linePtr->arrowGC, NULL);
	    }
	*/
    }
    if ((linePtr->datastyle & SCATTERDATA) && linePtr->symoutline.gc != None) {

	switch (linePtr->symstyle) {
	    case NONE: {
		break;
	    }
	    case SYMCIRCLE: {
		XArc *arclist = (XArc *)ckalloc(sizeof(XArc)*numPoints);

		for(i=0; i < numPoints; i++) {
		    arclist[i].x = pointPtr[i].x-linePtr->symsize/2;
		    arclist[i].y = pointPtr[i].y-linePtr->symsize/2;
		    arclist[i].width = arclist[i].height = linePtr->symsize;
		    arclist[i].angle1 = 0;
		    arclist[i].angle2 = 23040; /* 360 * 64 */
		}

		if (linePtr->symopaque) {
		    XFillArcs(display, drawable, linePtr->symoutline.gc,
				arclist, numPoints);
		} else {
		    XDrawArcs(display, drawable, linePtr->symoutline.gc,
				arclist, numPoints);
		}

		ckfree((char *)arclist);
		break;
	    }
	    break;
	    case SYMX: {
		XSegment *segment = (XSegment *)ckalloc(sizeof(XSegment)*numPoints * 2);
		int symsize = linePtr->symsize / 2;

		/*
		 * We need to draw one pixel past the endpoint so that
		 * Xlib won't cut off the end of the X due to "rounding"
		 */

		for(i=0; i < numPoints; i++) {
		    segment[i*2].x1 = pointPtr[i].x - symsize;
		    segment[i*2].y1 = pointPtr[i].y - symsize;
		    segment[i*2].x2 = pointPtr[i].x + symsize+1;
		    segment[i*2].y2 = pointPtr[i].y + symsize+1;

		    segment[i*2+1].x1 = pointPtr[i].x - symsize;
		    segment[i*2+1].y1 = pointPtr[i].y + symsize;
		    segment[i*2+1].x2 = pointPtr[i].x + symsize + 1;
		    segment[i*2+1].y2 = pointPtr[i].y - symsize - 1;
		}

		XDrawSegments(display, drawable, linePtr->symoutline.gc,
				segment, numPoints * 2);

		ckfree((char *)segment);

		break;
	    }
	    break;
	    case SYMEBAR: {
		XSegment *segment = (XSegment *)ckalloc(sizeof(XSegment)*numPoints*4);
		int xsymsize=0;
		int ysymsize=0;

		if (linePtr->datatype->datatype & (XSIGXYDATA|XSIGXYSIGYDATA|XSIGX1SIGX2YDATA|XSIGX1SIGX2YSIGY1SIGY2DATA)) {
		    xsymsize = linePtr->symsize/2;
		}

		if (linePtr->datatype->datatype & (XYSIGYDATA|XSIGXYSIGYDATA|XYSIGY1SIGY2DATA|XSIGX1SIGX2YSIGY1SIGY2DATA)) {
		    ysymsize = linePtr->symsize/2;
		}

		/* Draw tips at the end of the error bars
		 */

		for(i=0; i < numPoints; i++) {
		    /*
		     * Error bar on the top.
		     */
		    segment[i*4].x1 =
			pointPtr[i].x - ysymsize;
		    segment[i*4].x2 =
			pointPtr[i].x + ysymsize+1;
		    segment[i*4].y1 = segment[i*4].y2 =
			pointPtr[i].y - linePtr->errorPtr[i].ysigh*yscale;

		    /*
		     * Error bar on the bottom.
		     */
		    segment[i*4+1].x1 =
			pointPtr[i].x - ysymsize;
		    segment[i*4+1].x2 =
			pointPtr[i].x + ysymsize+1;
		    segment[i*4+1].y1 = segment[i*4+1].y2 =
			pointPtr[i].y + linePtr->errorPtr[i].ysigl*yscale;

		    /*
		     * Error bar on the right
		     */
		    segment[i*4+2].y1 =
			pointPtr[i].y - xsymsize;
		    segment[i*4+2].y2 =
			pointPtr[i].y + xsymsize+1;
		    segment[i*4+2].x1 = segment[i*4+2].x2 =
			pointPtr[i].x + linePtr->errorPtr[i].xsigh*xscale;

		    /*
		     * Error bar on the left
		     */
		    segment[i*4+3].y1 =
			pointPtr[i].y - xsymsize;
		    segment[i*4+3].y2 =
			pointPtr[i].y + xsymsize+1;
		    segment[i*4+3].x1 = segment[i*4+3].x2 =
			pointPtr[i].x - linePtr->errorPtr[i].xsigl*xscale;
		}

		XDrawSegments(display, drawable, linePtr->symoutline.gc,
				segment, numPoints * 4);

		ckfree((char *)segment);
	    }
	    break;
	    case SYMSQUARE: {
		XRectangle *rect = (XRectangle *)ckalloc(sizeof(XRectangle)*numPoints);
		int symsize = linePtr->symsize/2;

		for(i=0; i < numPoints; i++) {
		    rect[i].x = pointPtr[i].x - symsize;
		    rect[i].y = pointPtr[i].y - symsize;
		    rect[i].width = symsize*2;
		    rect[i].height = symsize*2;
		}

		if (linePtr->symopaque) {
		    XFillRectangles(display, drawable, linePtr->symoutline.gc,
				rect, numPoints);
		} else {
		    XDrawRectangles(display, drawable, linePtr->symoutline.gc,
				rect, numPoints);
		}

		ckfree((char *)rect);
	    }
	    break;
	    case SYMDIAMOND: {
		XSegment *segment = (XSegment *)ckalloc(sizeof(XSegment)*numPoints*4);
		XPoint *point = (XPoint *)ckalloc(sizeof(XPoint)*4);

		int symsize = linePtr->symsize/2;

		for(i=0; i < numPoints; i++) {
		    segment[4*i].x1 = pointPtr[i].x - symsize;
		    segment[4*i].x2 = pointPtr[i].x;
		    segment[4*i].y1 = pointPtr[i].y;
		    segment[4*i].y2 = pointPtr[i].y - symsize;

		    segment[4*i+1].x1 = pointPtr[i].x - symsize;
		    segment[4*i+1].x2 = pointPtr[i].x;
		    segment[4*i+1].y1 = pointPtr[i].y;
		    segment[4*i+1].y2 = pointPtr[i].y + symsize;

		    segment[4*i+2].x1 = pointPtr[i].x;
		    segment[4*i+2].x2 = pointPtr[i].x + symsize+1;
		    segment[4*i+2].y1 = pointPtr[i].y - symsize;
		    segment[4*i+2].y2 = pointPtr[i].y+1;

		    segment[4*i+3].x1 = pointPtr[i].x;
		    segment[4*i+3].x2 = pointPtr[i].x + symsize+1;
		    segment[4*i+3].y1 = pointPtr[i].y + symsize;
		    segment[4*i+3].y2 = pointPtr[i].y-1;

		    point[0*4].x = pointPtr[i].x - symsize;
		    point[0*4].y = pointPtr[i].y;
		    point[0*4+1].x = pointPtr[i].x;
		    point[0*4+1].y = pointPtr[i].y - symsize;
		    point[0*4+2].x = pointPtr[i].x + symsize;
		    point[0*4+2].y = pointPtr[i].y;
		    point[0*4+3].x = pointPtr[i].x;
		    point[0*4+3].y = pointPtr[i].y + symsize;
		    if (linePtr->symopaque) {
			XFillPolygon(display, drawable, linePtr->symoutline.gc,
				point, 4, Convex, CoordModeOrigin);
		    }
		}

		if (!linePtr->symopaque) {
		    XDrawSegments(display, drawable, linePtr->symoutline.gc,
				segment, numPoints * 4);
		}

		ckfree((char *)segment);
		ckfree((char *)point);
	    }
	    break;
	    case SYMNTRI: {
		XSegment *segment = (XSegment *)ckalloc(sizeof(XSegment)*numPoints*3);
		XPoint *point = (XPoint *)ckalloc(sizeof(XPoint)*3);
		int xsymsize = linePtr->symsize;
		int ysymsize = linePtr->symsize/1.4142136;

		for(i=0; i < numPoints; i++) {
		    segment[3*i].x1 = pointPtr[i].x - xsymsize/2;
		    segment[3*i].x2 = pointPtr[i].x;
		    segment[3*i].y1 = pointPtr[i].y + ysymsize/2;
		    segment[3*i].y2 = pointPtr[i].y - ysymsize/2;

		    segment[3*i+1].x1 = pointPtr[i].x - xsymsize/2;
		    segment[3*i+1].x2 = pointPtr[i].x + xsymsize/2;
		    segment[3*i+1].y1 = pointPtr[i].y + ysymsize/2;
		    segment[3*i+1].y2 = pointPtr[i].y + ysymsize/2;

		    segment[3*i+2].x1 = pointPtr[i].x;
		    segment[3*i+2].x2 = pointPtr[i].x + xsymsize/2;
		    segment[3*i+2].y1 = pointPtr[i].y - ysymsize/2;
		    segment[3*i+2].y2 = pointPtr[i].y + ysymsize/2;

		    point[0].x = pointPtr[i].x - xsymsize/2;
		    point[0].y = pointPtr[i].y + ysymsize/2;
		    point[1].x = pointPtr[i].x;
		    point[1].y = pointPtr[i].y - ysymsize/2;
		    point[2].x = pointPtr[i].x + xsymsize/2;
		    point[2].y = pointPtr[i].y + ysymsize/2;

		    if (linePtr->symopaque) {
			XFillPolygon(display, drawable, linePtr->symoutline.gc,
				point, 3, Convex, CoordModeOrigin);
		    }
		}

		if (!linePtr->symopaque) {
		    XDrawSegments(display, drawable, linePtr->symoutline.gc,
				segment, numPoints * 3);
		}

		ckfree((char *)segment);
		ckfree((char *)point);
	    }
	    break;
	    case SYMSTRI: {
		XSegment *segment = (XSegment *)ckalloc(sizeof(XSegment)*numPoints*3);
		XPoint *point = (XPoint *)ckalloc(sizeof(XPoint)*3);
		int xsymsize = linePtr->symsize;
		int ysymsize = linePtr->symsize/1.4142136;

		for(i=0; i < numPoints; i++) {
		    segment[3*i].x1 = pointPtr[i].x - xsymsize/2;
		    segment[3*i].x2 = pointPtr[i].x;
		    segment[3*i].y1 = pointPtr[i].y - ysymsize/2;
		    segment[3*i].y2 = pointPtr[i].y + ysymsize/2;

		    segment[3*i+1].x1 = pointPtr[i].x - xsymsize/2;
		    segment[3*i+1].x2 = pointPtr[i].x + xsymsize/2;
		    segment[3*i+1].y1 = pointPtr[i].y - ysymsize/2;
		    segment[3*i+1].y2 = pointPtr[i].y - ysymsize/2;

		    segment[3*i+2].x1 = pointPtr[i].x;
		    segment[3*i+2].x2 = pointPtr[i].x + xsymsize/2;
		    segment[3*i+2].y1 = pointPtr[i].y + ysymsize/2;
		    segment[3*i+2].y2 = pointPtr[i].y - ysymsize/2;

		    point[0].x = pointPtr[i].x - xsymsize/2;
		    point[0].y = pointPtr[i].y - ysymsize/2;
		    point[1].x = pointPtr[i].x;
		    point[1].y = pointPtr[i].y + ysymsize/2;
		    point[2].x = pointPtr[i].x + xsymsize/2;
		    point[2].y = pointPtr[i].y - ysymsize/2;

		    if (linePtr->symopaque) {
			XFillPolygon(display, drawable, linePtr->symoutline.gc,
				point, 3, Convex, CoordModeOrigin);
		    }
		}

		if (!linePtr->symopaque) {
		    XDrawSegments(display, drawable, linePtr->symoutline.gc,
				segment, numPoints * 3);
		}

		ckfree((char *)segment);
		ckfree((char *)point);
	    }
	    break;
	    case SYMWTRI: {
		XSegment *segment = (XSegment *)ckalloc(sizeof(XSegment)*numPoints*3);
		XPoint *point = (XPoint *)ckalloc(sizeof(XPoint)*3);
		int xsymsize = linePtr->symsize/1.4142136;
		int ysymsize = linePtr->symsize;

		for(i=0; i < numPoints; i++) {
		    segment[3*i].x1 = pointPtr[i].x - xsymsize/2;
		    segment[3*i].x2 = pointPtr[i].x + xsymsize/2;
		    segment[3*i].y1 = pointPtr[i].y;
		    segment[3*i].y2 = pointPtr[i].y + ysymsize/2;

		    segment[3*i+1].x1 = pointPtr[i].x - xsymsize/2;
		    segment[3*i+1].x2 = pointPtr[i].x + xsymsize/2;
		    segment[3*i+1].y1 = pointPtr[i].y;
		    segment[3*i+1].y2 = pointPtr[i].y - ysymsize/2;

		    segment[3*i+2].x1 = pointPtr[i].x + xsymsize/2;
		    segment[3*i+2].x2 = pointPtr[i].x + xsymsize/2;
		    segment[3*i+2].y1 = pointPtr[i].y + ysymsize/2;
		    segment[3*i+2].y2 = pointPtr[i].y - ysymsize/2;

		    point[0].x = pointPtr[i].x - xsymsize/2;
		    point[0].y = pointPtr[i].y;
		    point[1].x = pointPtr[i].x + xsymsize/2;
		    point[1].y = pointPtr[i].y + ysymsize/2;
		    point[2].x = pointPtr[i].x + xsymsize/2;
		    point[2].y = pointPtr[i].y - ysymsize/2;

		    if (linePtr->symopaque) {
			XFillPolygon(display, drawable, linePtr->symoutline.gc,
				point, 3, Convex, CoordModeOrigin);
		    }
		}

		if (!linePtr->symopaque) {
		    XDrawSegments(display, drawable, linePtr->symoutline.gc,
				segment, numPoints * 3);
		}

		ckfree((char *)segment);
		ckfree((char *)point);
	    }
	    break;
	    case SYMETRI: {
		XSegment *segment = (XSegment *)ckalloc(sizeof(XSegment)*numPoints*3);
		XPoint *point = (XPoint *)ckalloc(sizeof(XPoint)*3);
		int xsymsize = linePtr->symsize/1.4142136;
		int ysymsize = linePtr->symsize;

		for(i=0; i < numPoints; i++) {
		    segment[3*i].x1 = pointPtr[i].x + xsymsize/2+1;
		    segment[3*i].x2 = pointPtr[i].x - xsymsize/2;
		    segment[3*i].y1 = pointPtr[i].y-1;
		    segment[3*i].y2 = pointPtr[i].y + ysymsize/2;

		    segment[3*i+1].x1 = pointPtr[i].x + xsymsize/2+1;
		    segment[3*i+1].x2 = pointPtr[i].x - xsymsize/2;
		    segment[3*i+1].y1 = pointPtr[i].y+1;
		    segment[3*i+1].y2 = pointPtr[i].y - ysymsize/2;

		    segment[3*i+2].x1 = pointPtr[i].x - xsymsize/2;
		    segment[3*i+2].x2 = pointPtr[i].x - xsymsize/2;
		    segment[3*i+2].y1 = pointPtr[i].y + ysymsize/2;
		    segment[3*i+2].y2 = pointPtr[i].y - ysymsize/2;

		    point[0].x = pointPtr[i].x + xsymsize/2;
		    point[0].y = pointPtr[i].y;
		    point[1].x = pointPtr[i].x - xsymsize/2;
		    point[1].y = pointPtr[i].y + ysymsize/2;
		    point[2].x = pointPtr[i].x - xsymsize/2;
		    point[2].y = pointPtr[i].y - ysymsize/2;

		    if (linePtr->symopaque) {
			XFillPolygon(display, drawable, linePtr->symoutline.gc,
				point, 3, Convex, CoordModeOrigin);
		    }
		}

		if (!linePtr->symopaque) {
		    XDrawSegments(display, drawable, linePtr->symoutline.gc,
				segment, numPoints * 3);
		}

		ckfree((char *)segment);
		ckfree((char *)point);
	    }
	    break;
	    case SYMPLUS: {
		XSegment *segment = (XSegment *)ckalloc(sizeof(XSegment)*numPoints * 2);
		int symsize = linePtr->symsize / 2;

		/*
		 * We need to draw one pixel past the endpoint so that
		 * Xlib won't cut off the end of the X due to "rounding"
		 */

		for(i=0; i < numPoints; i++) {
		    segment[i*2].x1 = (pointPtr[i].x - symsize);
		    segment[i*2].x2 = (pointPtr[i].x + symsize+1);
		    segment[i*2].y1 = (pointPtr[i].y);
		    segment[i*2].y2 = (pointPtr[i].y);

		    segment[i*2+1].x1 = (pointPtr[i].x);
		    segment[i*2+1].x2 = (pointPtr[i].x);
		    segment[i*2+1].y1 = (pointPtr[i].y + symsize);
		    segment[i*2+1].y2 = (pointPtr[i].y - symsize - 1);
		}

		XDrawSegments(display, drawable, linePtr->symoutline.gc,
				segment, numPoints * 2);

		ckfree((char *)segment);

		break;
	    }
	    break;
	}


	if (pointPtr != staticPoints) {
	    ckfree((char *) pointPtr);
	}
    }
    errsegment = (XSegment *)ckalloc(sizeof(XSegment)*numPoints * 2);
    for(segcount=0, i=0; i < numPoints; i++) {
	if (linePtr->datatype->datatype & (XSIGXYDATA|XSIGXYSIGYDATA|XSIGX1SIGX2YDATA|XSIGX1SIGX2YSIGY1SIGY2DATA)) {
	    errsegment[segcount].x1 = pointPtr[i].x + linePtr->errorPtr[i].xsigh*xscale+1;
	    errsegment[segcount].x2 = pointPtr[i].x - linePtr->errorPtr[i].xsigl*xscale+1;
	    errsegment[segcount].y1 = errsegment[segcount].y2 = pointPtr[i].y;
	    segcount++;	
	}
	if (linePtr->datatype->datatype & (XYSIGYDATA|XSIGXYSIGYDATA|XYSIGY1SIGY2DATA|XSIGX1SIGX2YSIGY1SIGY2DATA)) {
	    errsegment[segcount].y1 = pointPtr[i].y - linePtr->errorPtr[i].ysigh*yscale;
	    errsegment[segcount].y2 = pointPtr[i].y + linePtr->errorPtr[i].ysigl*yscale+1;
	    errsegment[segcount].x1 = errsegment[segcount].x2 = pointPtr[i].x;
	    segcount++;	
	}
    }
    if (segcount) {
	if (linePtr->symoutline.gc != None) {
	    XDrawSegments(display, drawable, linePtr->symoutline.gc, errsegment, segcount);
	} else if (linePtr->outline.gc != None) {
	    XDrawSegments(display, drawable, linePtr->outline.gc, errsegment, segcount);
	}
    }
    ckfree((char *)errsegment);

    if (linePtr->symoutline.gc) {
	XSetClipMask(display, linePtr->symoutline.gc, None);
    }
    XSetClipMask(display, linePtr->outline.gc, None);
    if (Tk_ResetOutlineGC(canvas, itemPtr, &(linePtr->outline))) {
	XSetTSOrigin(display, linePtr->arrowGC, 0, 0);
    }
}

/*
 *--------------------------------------------------------------
 *
 * DatasetInsert --
 *
 *	Insert coords into a line item at a given index.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The coords in the given item is modified.
 *
 *--------------------------------------------------------------
 */

static void
DatasetInsert(canvas, itemPtr, beforeThis, string)
    Tk_Canvas canvas;		/* Canvas containing text item. */
    Tk_Item *itemPtr;		/* Line item to be modified. */
    int beforeThis;		/* Index before which new coordinates
				 * are to be inserted. */
    char *string;		/* New coordinates to be inserted. */
{
    DatasetItem *linePtr = (DatasetItem *) itemPtr;
    int length, argc, i;
    char **argv = (char **) NULL;
    double *newdata, *newgraph, *newcanvas, *coordPtr;
    Tk_State state = itemPtr->state;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    if(!string || !*string) {
	return;
    }
    if ((Tcl_SplitList(((TkCanvas *)canvas)->interp, string, &argc, &argv) != TCL_OK)
	    || argv == NULL || !argc || argc&1) {
	Tcl_ResetResult(((TkCanvas *)canvas)->interp);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return;
    }
    length = 2*linePtr->numPoints;
    if (beforeThis < 0) {
	beforeThis = 0;
    }
    if (beforeThis > length) {
	beforeThis = length;
    }
    if (linePtr->firstArrowPtr != NULL) {
	linePtr->datacoordPtr[0] = linePtr->firstArrowPtr[0];
	linePtr->datacoordPtr[1] = linePtr->firstArrowPtr[1];
    }
    if (linePtr->lastArrowPtr != NULL) {
	linePtr->datacoordPtr[length-2] = linePtr->lastArrowPtr[0];
	linePtr->datacoordPtr[length-1] = linePtr->lastArrowPtr[1];
    }
    newdata  = (double *) ckalloc((unsigned)(sizeof(double) * (length + argc)));
    newgraph = (double *) ckalloc((unsigned)(sizeof(double) * (length + argc)));
    newcanvas= (double *) ckalloc((unsigned)(sizeof(double) * (length + argc)));
    for(i=0; i<beforeThis; i++) {
	newdata[i] = linePtr->datacoordPtr[i];
    }
    for(i=0; i<argc; i++) {
	if (Tcl_GetDouble(((TkCanvas *)canvas)->interp,argv[i],
		newdata+(i+beforeThis))!=TCL_OK) {
	    Tcl_ResetResult(((TkCanvas *)canvas)->interp);
	    ckfree((char *) newdata);
	    ckfree((char *) newgraph);
	    ckfree((char *) newcanvas);
	    ckfree((char *) argv);
	    return;
	}
    }

    for(i=beforeThis; i<length; i++) {
	newdata[i+argc] = linePtr->datacoordPtr[i];
    }
    if(linePtr->datacoordPtr)   ckfree((char *)linePtr->datacoordPtr);
    if(linePtr->graphcoordPtr)  ckfree((char *)linePtr->graphcoordPtr);
    if(linePtr->canvascoordPtr) ckfree((char *)linePtr->canvascoordPtr);
    ckfree((char *) argv);
    linePtr->datacoordPtr   = newdata;
    linePtr->graphcoordPtr  = newgraph;
    linePtr->canvascoordPtr = newcanvas;
    linePtr->numPoints = (length + argc)/2;

    if ((length>3) && (state != TK_STATE_HIDDEN)) {
	/*
	 * This is some optimizing code that will result that only the part
	 * of the polygon that changed (and the objects that are overlapping
	 * with that part) need to be redrawn. A special flag is set that
	 * instructs the general canvas code not to redraw the whole
	 * object. If this flag is not set, the canvas will do the redrawing,
	 * otherwise I have to do it here.
	 */
	itemPtr->redraw_flags |= TK_ITEM_DONT_REDRAW;

	if (beforeThis>0) {beforeThis -= 2; argc+=2; }
	if ((beforeThis+argc)<length) argc+=2;
	if (linePtr->smooth) {
	    if(beforeThis>0) {
		beforeThis-=2; argc+=2;
	    }
	    if((beforeThis+argc+2)<length) {
		argc+=2;
	    }
	}
	itemPtr->x1 = itemPtr->x2 = linePtr->datacoordPtr[beforeThis];
	itemPtr->y1 = itemPtr->y2 = linePtr->datacoordPtr[beforeThis+1];
	if ((linePtr->firstArrowPtr != NULL) && (beforeThis<1)) {
	    /* include old first arrow */
	    for (i = 0, coordPtr = linePtr->firstArrowPtr; i < PTS_IN_ARROW;
		    i++, coordPtr += 2) {
		TkIncludePoint(itemPtr, coordPtr);
	    }
	}
	if ((linePtr->lastArrowPtr != NULL) && ((beforeThis+argc)>=length)) {
		/* include old last arrow */
	    for (i = 0, coordPtr = linePtr->lastArrowPtr; i < PTS_IN_ARROW;
		    i++, coordPtr += 2) {
		TkIncludePoint(itemPtr, coordPtr);
	    }
	}
	coordPtr = linePtr->datacoordPtr+beforeThis+2;
	for(i=2; i<argc; i+=2) {
	    TkIncludePoint(itemPtr, coordPtr);
		coordPtr+=2;
	}
    }
    if (linePtr->firstArrowPtr != NULL) {
	ckfree((char *) linePtr->firstArrowPtr);
	linePtr->firstArrowPtr = NULL;
    }
    if (linePtr->lastArrowPtr != NULL) {
	ckfree((char *) linePtr->lastArrowPtr);
	linePtr->lastArrowPtr = NULL;
    }
    if (linePtr->arrow != noneUid) {
	    ConfigureArrows(canvas, linePtr);
    }

    if(itemPtr->redraw_flags & TK_ITEM_DONT_REDRAW) {
	int width;
	if ((linePtr->firstArrowPtr != NULL) && (beforeThis>2)) {
	    /* include new first arrow */
	    for (i = 0, coordPtr = linePtr->firstArrowPtr; i < PTS_IN_ARROW;
		    i++, coordPtr += 2) {
		TkIncludePoint(itemPtr, coordPtr);
	    }
	}
	if ((linePtr->lastArrowPtr != NULL) && ((beforeThis+argc)<(length-2))) {
	    /* include new right arrow */
	    for (i = 0, coordPtr = linePtr->lastArrowPtr; i < PTS_IN_ARROW;
		    i++, coordPtr += 2) {
		TkIncludePoint(itemPtr, coordPtr);
	    }
	}
	width = linePtr->outline.width;
	if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
		if (linePtr->outline.activeWidth>width) {
		    width = linePtr->outline.activeWidth;
		}
	} else if (state==TK_STATE_DISABLED) {
		if (linePtr->outline.disabledWidth>0) {
		    width = linePtr->outline.disabledWidth;
		}
	}
	itemPtr->x1 -= width; itemPtr->y1 -= width;
	itemPtr->x2 += width; itemPtr->y2 += width;
	Tk_CanvasEventuallyRedraw(canvas, itemPtr->x1, itemPtr->y1,
		itemPtr->x2, itemPtr->y2);
    }

    DatasetUpdateGraphCoords(linePtr);
    DatasetUpdateCanvasCoords(linePtr);

    ComputeDatasetBbox(canvas, linePtr);
}

/*
 *--------------------------------------------------------------
 *
 * DatasetDeleteCoords --
 *
 *	Delete one or more coordinates from a line item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Characters between "first" and "last", inclusive, get
 *	deleted from itemPtr.
 *
 *--------------------------------------------------------------
 */

static void
DatasetDeleteCoords(canvas, itemPtr, first, last)
    Tk_Canvas canvas;		/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Item in which to delete characters. */
    int first;			/* Index of first character to delete. */
    int last;			/* Index of last character to delete. */
{
    DatasetItem *linePtr = (DatasetItem *) itemPtr;
    int count, i, first1, last1;
    int length = 2*linePtr->numPoints;
    double *coordPtr;
    Tk_State state = itemPtr->state;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    first &= -2;
    last &= -2;

    if (first < 0) {
	first = 0;
    }
    if (last >= length) {
	last = length-2;
    }
    if (first > last) {
	return;
    }
    if (linePtr->firstArrowPtr != NULL) {
	linePtr->datacoordPtr[0] = linePtr->firstArrowPtr[0];
	linePtr->datacoordPtr[1] = linePtr->firstArrowPtr[1];
    }
    if (linePtr->lastArrowPtr != NULL) {
	linePtr->datacoordPtr[length-2] = linePtr->lastArrowPtr[0];
	linePtr->datacoordPtr[length-1] = linePtr->lastArrowPtr[1];
    }
    first1 = first; last1 = last;
    if(first1>0) first1 -= 2;
    if(last1<length-2) last1 += 2;
    if (linePtr->smooth) {
	if(first1>0) first1 -= 2;
	if(last1<length-2) last1 += 2;
    }

    if((first1<2) && (last1 >= length-2)) {
	/*
	 * This is some optimizing code that will result that only the part
	 * of the line that changed (and the objects that are overlapping
	 * with that part) need to be redrawn. A special flag is set that
	 * instructs the general canvas code not to redraw the whole
	 * object. If this flag is set, the redrawing has to be done here,
	 * otherwise the general Canvas code will take care of it.
	 */

	itemPtr->redraw_flags |= TK_ITEM_DONT_REDRAW;
	itemPtr->x1 = itemPtr->x2 = linePtr->datacoordPtr[first1];
	itemPtr->y1 = itemPtr->y2 = linePtr->datacoordPtr[first1+1];
	if ((linePtr->firstArrowPtr != NULL) && (first1<2)) {
	    /* include old first arrow */
	    for (i = 0, coordPtr = linePtr->firstArrowPtr; i < PTS_IN_ARROW;
		    i++, coordPtr += 2) {
		TkIncludePoint(itemPtr, coordPtr);
	    }
	}
	if ((linePtr->lastArrowPtr != NULL) && (last1>=length-2)) {
		/* include old last arrow */
	    for (i = 0, coordPtr = linePtr->lastArrowPtr; i < PTS_IN_ARROW;
		    i++, coordPtr += 2) {
		TkIncludePoint(itemPtr, coordPtr);
	    }
	}
	coordPtr = linePtr->datacoordPtr+first1+2;
	for(i=first1+2; i<=last1; i+=2) {
	    TkIncludePoint(itemPtr, coordPtr);
		coordPtr+=2;
	}
    }

    count = last + 2 - first;
    for(i=last+2; i<length; i++) {
	linePtr->datacoordPtr[i-count] = linePtr->datacoordPtr[i];
    }
    linePtr->numPoints -= count/2;
    if (linePtr->firstArrowPtr != NULL) {
	ckfree((char *) linePtr->firstArrowPtr);
	linePtr->firstArrowPtr = NULL;
    }
    if (linePtr->lastArrowPtr != NULL) {
	ckfree((char *) linePtr->lastArrowPtr);
	linePtr->lastArrowPtr = NULL;
    }
    if (linePtr->arrow != noneUid) {
	    ConfigureArrows(canvas, linePtr);
    }
    if(itemPtr->redraw_flags & TK_ITEM_DONT_REDRAW) {
	int width;
	if ((linePtr->firstArrowPtr != NULL) && (first1<4)) {
	    /* include new first arrow */
	    for (i = 0, coordPtr = linePtr->firstArrowPtr; i < PTS_IN_ARROW;
		    i++, coordPtr += 2) {
		TkIncludePoint(itemPtr, coordPtr);
	    }
	}
	if ((linePtr->lastArrowPtr != NULL) && (last1>(length-4))) {
	    /* include new right arrow */
	    for (i = 0, coordPtr = linePtr->lastArrowPtr; i < PTS_IN_ARROW;
		    i++, coordPtr += 2) {
		TkIncludePoint(itemPtr, coordPtr);
	    }
	}
	width = linePtr->outline.width;
	if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
		if (linePtr->outline.activeWidth>width) {
		    width = linePtr->outline.activeWidth;
		}
	} else if (state==TK_STATE_DISABLED) {
		if (linePtr->outline.disabledWidth>0) {
		    width = linePtr->outline.disabledWidth;
		}
	}
	itemPtr->x1 -= width; itemPtr->y1 -= width;
	itemPtr->x2 += width; itemPtr->y2 += width;
	Tk_CanvasEventuallyRedraw(canvas, itemPtr->x1, itemPtr->y1,
		itemPtr->x2, itemPtr->y2);
    }
    ComputeDatasetBbox(canvas, linePtr);
}

/*
 *--------------------------------------------------------------
 *
 * DatasetToPoint --
 *
 *	Computes the distance from a given point to a given
 *	line, in canvas units.
 *
 * Results:
 *	The return value is 0 if the point whose x and y coordinates
 *	are pointPtr[0] and pointPtr[1] is inside the line.  If the
 *	point isn't inside the line then the return value is the
 *	distance from the point to the line.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static double
DatasetToPoint(canvas, itemPtr, pointPtr)
    Tk_Canvas canvas;		/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against point. */
    double *pointPtr;		/* Pointer to x and y coordinates. */
{
    int width;
    Tk_State state = itemPtr->state;
    DatasetItem *linePtr = (DatasetItem *) itemPtr;
    double *coordPtr, *linePoints;
    double staticSpace[2*MAX_STATIC_POINTS];
    double poly[10];
    double bestDist, dist;
    int numPoints, count;
    int changedMiterToBevel;	/* Non-zero means that a mitered corner
				 * had to be treated as beveled after all
				 * because the angle was < 11 degrees. */

    bestDist = 1.0e36;

    /*
     * Handle smoothed lines by generating an expanded set of points
     * against which to do the check.
     */

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    width = linePtr->outline.width;
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (linePtr->outline.activeWidth>width) {
	    width = linePtr->outline.activeWidth;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (linePtr->outline.disabledWidth>0) {
	    width = linePtr->outline.disabledWidth;
	}
    }

    if ((linePtr->smooth) && (linePtr->numPoints > 2)) {
	numPoints = 1 + linePtr->numPoints*linePtr->splineSteps;
	if (numPoints <= MAX_STATIC_POINTS) {
	    linePoints = staticSpace;
	} else {
	    linePoints = (double *) ckalloc((unsigned)
		    (2*numPoints*sizeof(double)));
	}
	numPoints = TkMakeBezierCurve(canvas, linePtr->canvascoordPtr,
		linePtr->numPoints, linePtr->splineSteps, (XPoint *) NULL,
		linePoints);
    } else {
	numPoints = linePtr->numPoints;
	linePoints = linePtr->canvascoordPtr;
    }

    if (!numPoints || itemPtr->state==TK_STATE_HIDDEN) {
	return bestDist;
    } else if (numPoints == 1) {
	bestDist = hypot(linePoints[0] - pointPtr[0], linePoints[1] - pointPtr[1])
		    - width/2.0;
	if (bestDist < 0) bestDist = 0;
	return bestDist;
    }

    /*
     * The overall idea is to iterate through all of the edges of
     * the line, computing a polygon for each edge and testing the
     * point against that polygon.  In addition, there are additional
     * tests to deal with rounded joints and caps.
     */

    changedMiterToBevel = 0;
    for (count = numPoints, coordPtr = linePoints; count >= 2;
	    count--, coordPtr += 2) {

	/*
	 * If rounding is done around the first point then compute
	 * the distance between the point and the point.
	 */

	if (((linePtr->capStyle == CapRound) && (count == numPoints))
		|| ((linePtr->joinStyle == JoinRound)
			&& (count != numPoints))) {
	    dist = hypot(coordPtr[0] - pointPtr[0], coordPtr[1] - pointPtr[1])
		    - width/2.0;
	    if (dist <= 0.0) {
		bestDist = 0.0;
		goto done;
	    } else if (dist < bestDist) {
		bestDist = dist;
	    }
	}

	/*
	 * Compute the polygonal shape corresponding to this edge,
	 * consisting of two points for the first point of the edge
	 * and two points for the last point of the edge.
	 */

	if (count == numPoints) {
	    TkGetButtPoints(coordPtr+2, coordPtr, (double) width,
		    linePtr->capStyle == CapProjecting, poly, poly+2);
	} else if ((linePtr->joinStyle == JoinMiter) && !changedMiterToBevel) {
	    poly[0] = poly[6];
	    poly[1] = poly[7];
	    poly[2] = poly[4];
	    poly[3] = poly[5];
	} else {
	    TkGetButtPoints(coordPtr+2, coordPtr, (double) width, 0,
		    poly, poly+2);

	    /*
	     * If this line uses beveled joints, then check the distance
	     * to a polygon comprising the last two points of the previous
	     * polygon and the first two from this polygon;  this checks
	     * the wedges that fill the mitered joint.
	     */

	    if ((linePtr->joinStyle == JoinBevel) || changedMiterToBevel) {
		poly[8] = poly[0];
		poly[9] = poly[1];
		dist = TkPolygonToPoint(poly, 5, pointPtr);
		if (dist <= 0.0) {
		    bestDist = 0.0;
		    goto done;
		} else if (dist < bestDist) {
		    bestDist = dist;
		}
		changedMiterToBevel = 0;
	    }
	}
	if (count == 2) {
	    TkGetButtPoints(coordPtr, coordPtr+2, (double) width,
		    linePtr->capStyle == CapProjecting, poly+4, poly+6);
	} else if (linePtr->joinStyle == JoinMiter) {
	    if (TkGetMiterPoints(coordPtr, coordPtr+2, coordPtr+4,
		    (double) width, poly+4, poly+6) == 0) {
		changedMiterToBevel = 1;
		TkGetButtPoints(coordPtr, coordPtr+2, (double) width,
			0, poly+4, poly+6);
	    }
	} else {
	    TkGetButtPoints(coordPtr, coordPtr+2, (double) width, 0,
		    poly+4, poly+6);
	}
	poly[8] = poly[0];
	poly[9] = poly[1];
	dist = TkPolygonToPoint(poly, 5, pointPtr);
	if (dist <= 0.0) {
	    bestDist = 0.0;
	    goto done;
	} else if (dist < bestDist) {
	    bestDist = dist;
	}
    }

    /*
     * If caps are rounded, check the distance to the cap around the
     * final end point of the line.
     */

    if (linePtr->capStyle == CapRound) {
	dist = hypot(coordPtr[0] - pointPtr[0], coordPtr[1] - pointPtr[1])
		- width/2.0;
	if (dist <= 0.0) {
	    bestDist = 0.0;
	    goto done;
	} else if (dist < bestDist) {
	    bestDist = dist;
	}
    }

    /*
     * If there are arrowheads, check the distance to the arrowheads.
     */

    if (linePtr->arrow != noneUid) {
	if (linePtr->arrow != lastUid) {
	    dist = TkPolygonToPoint(linePtr->firstArrowPtr, PTS_IN_ARROW,
		    pointPtr);
	    if (dist <= 0.0) {
		bestDist = 0.0;
		goto done;
	    } else if (dist < bestDist) {
		bestDist = dist;
	    }
	}
	if (linePtr->arrow != firstUid) {
	    dist = TkPolygonToPoint(linePtr->lastArrowPtr, PTS_IN_ARROW,
		    pointPtr);
	    if (dist <= 0.0) {
		bestDist = 0.0;
		goto done;
	    } else if (dist < bestDist) {
		bestDist = dist;
	    }
	}
    }

    done:
    if ((linePoints != staticSpace) && (linePoints != linePtr->canvascoordPtr)) {
	ckfree((char *) linePoints);
    }
    return bestDist;
}

/*
 *--------------------------------------------------------------
 *
 * DatasetToArea --
 *
 *	This procedure is called to determine whether an item
 *	lies entirely inside, entirely outside, or overlapping
 *	a given rectangular area.
 *
 * Results:
 *	-1 is returned if the item is entirely outside the
 *	area, 0 if it overlaps, and 1 if it is entirely
 *	inside the given area.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
DatasetToArea(canvas, itemPtr, rectPtr)
    Tk_Canvas canvas;		/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against line. */
    double *rectPtr;
{
    DatasetItem *linePtr = (DatasetItem *) itemPtr;
    double staticSpace[2*MAX_STATIC_POINTS];
    double *linePoints;
    double radius;
    int numPoints, result;
    int width;
    Tk_State state = itemPtr->state;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }
    width = linePtr->outline.width;
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (linePtr->outline.activeWidth>width) {
	    width = linePtr->outline.activeWidth;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (linePtr->outline.disabledWidth>0) {
	    width = linePtr->outline.disabledWidth;
	}
    }

    radius = (width+1)/2.0;

    if ((state==TK_STATE_HIDDEN) || !linePtr->numPoints) {
	return -1;
    } else if (linePtr->numPoints == 1) {
	double oval[4];
	oval[0] = linePtr->canvascoordPtr[0]-radius;
	oval[1] = linePtr->canvascoordPtr[1]-radius;
	oval[2] = linePtr->canvascoordPtr[0]+radius;
	oval[3] = linePtr->canvascoordPtr[1]+radius;
	return TkOvalToArea(oval, rectPtr);
    }

    /*
     * Handle smoothed lines by generating an expanded set of points
     * against which to do the check.
     */

    if ((linePtr->smooth) && (linePtr->numPoints > 2)) {
	numPoints = 1 + linePtr->numPoints*linePtr->splineSteps;
	if (numPoints <= MAX_STATIC_POINTS) {
	    linePoints = staticSpace;
	} else {
	    linePoints = (double *) ckalloc((unsigned)
		    (2*numPoints*sizeof(double)));
	}
	numPoints = TkMakeBezierCurve(canvas, linePtr->canvascoordPtr,
		linePtr->numPoints, linePtr->splineSteps, (XPoint *) NULL,
		linePoints);
    } else {
	numPoints = linePtr->numPoints;
	linePoints = linePtr->canvascoordPtr;
    }

    /*
     * Check the segments of the line.
     */

    result = TkThickPolyLineToArea(linePoints, numPoints, 
	    (double) width, linePtr->capStyle, linePtr->joinStyle,
	    rectPtr);
    if (result == 0) {
	goto done;
    }

    /*
     * Check arrowheads, if any.
     */

    if (linePtr->arrow != noneUid) {
	if (linePtr->arrow != lastUid) {
	    if (TkPolygonToArea(linePtr->firstArrowPtr, PTS_IN_ARROW,
		    rectPtr) != result) {
		result = 0;
		goto done;
	    }
	}
	if (linePtr->arrow != firstUid) {
	    if (TkPolygonToArea(linePtr->lastArrowPtr, PTS_IN_ARROW,
		    rectPtr) != result) {
		result = 0;
		goto done;
	    }
	}
    }

    done:
    if ((linePoints != staticSpace) && (linePoints != linePtr->canvascoordPtr)) {
	ckfree((char *) linePoints);
    }
    return result;
}

/*
 *--------------------------------------------------------------
 *
 * ScaleDataset --
 *
 *	This procedure is invoked to rescale a line item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The line referred to by itemPtr is rescaled so that the
 *	following transformation is applied to all point
 *	coordinates:
 *		x' = originX + scaleX*(x-originX)
 *		y' = originY + scaleY*(y-originY)
 *
 *--------------------------------------------------------------
 *
 * We shouldn't really be calling this...
 */

void
ScaleDataset(canvas, itemPtr, originX, originY, scaleX, scaleY)
    Tk_Canvas canvas;			/* Canvas containing line. */
    Tk_Item *itemPtr;			/* Line to be scaled. */
    double originX, originY;		/* Origin about which to scale rect. */
    double scaleX;			/* Amount to scale in X direction. */
    double scaleY;			/* Amount to scale in Y direction. */
{
    DatasetItem *linePtr = (DatasetItem *) itemPtr;
    double *coordPtr;
    int i;

    return;

    fprintf(stderr, "Error:  ScaleDataset called!  Generating core file\n");
    abort();

    /*
     * Delete any arrowheads before scaling all the points (so that
     * the end-points of the line get restored).
     */

    if (linePtr->firstArrowPtr != NULL) {
	linePtr->canvascoordPtr[0] = linePtr->firstArrowPtr[0];
	linePtr->canvascoordPtr[1] = linePtr->firstArrowPtr[1];
	ckfree((char *) linePtr->firstArrowPtr);
	linePtr->firstArrowPtr = NULL;
    }
    if (linePtr->lastArrowPtr != NULL) {
	int i;

	i = 2*(linePtr->numPoints-1);
	linePtr->canvascoordPtr[i] = linePtr->lastArrowPtr[0];
	linePtr->canvascoordPtr[i+1] = linePtr->lastArrowPtr[1];
	ckfree((char *) linePtr->lastArrowPtr);
	linePtr->lastArrowPtr = NULL;
    }
    for (i = 0, coordPtr = linePtr->canvascoordPtr; i < linePtr->numPoints;
	    i++, coordPtr += 2) {
	coordPtr[0] = originX + scaleX*(*coordPtr - originX);
	coordPtr[1] = originY + scaleY*(coordPtr[1] - originY);
    }
    if (linePtr->arrow != noneUid) {
	ConfigureArrows(canvas, linePtr);
    }
    ComputeDatasetBbox(canvas, linePtr);
}

/*
 *--------------------------------------------------------------
 *
 * GetDatasetIndex --
 *
 *	Parse an index into a line item and return either its value
 *	or an error.
 *
 * Results:
 *	A standard Tcl result.  If all went well, then *indexPtr is
 *	filled in with the index (into itemPtr) corresponding to
 *	string.  Otherwise an error message is left in
 *	interp->result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
GetDatasetIndex(interp, canvas, itemPtr, string, indexPtr)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Canvas canvas;		/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item for which the index is being
				 * specified. */
    char *string;		/* Specification of a particular coord
				 * in itemPtr's line. */
    int *indexPtr;		/* Where to store converted index. */
{
    DatasetItem *linePtr = (DatasetItem *) itemPtr;
    size_t length;

    length = strlen(string);

    if (string[0] == 'e') {
	if (strncmp(string, "end", length) == 0) {
	    *indexPtr = (2*linePtr->numPoints);
	} else {
	    badIndex:

	    /*
	     * Some of the paths here leave messages in interp->result,
	     * so we have to clear it out before storing our own message.
	     */

	    Tcl_SetResult(interp, (char *) NULL, TCL_STATIC);
	    Tcl_AppendResult(interp, "bad index \"", string, "\"",
		    (char *) NULL);
	    return TCL_ERROR;
	}
    } else if (string[0] == '@') {
	int i;
	double x ,y, bestDist, dist, *coordPtr;
	char *end, *p;

	p = string+1;
	x = strtod(p, &end);
	if ((end == p) || (*end != ',')) {
	    goto badIndex;
	}
	p = end+1;
	y = strtod(p, &end);
	if ((end == p) || (*end != 0)) {
	    goto badIndex;
	}
	bestDist = 1.0e36;
	coordPtr = linePtr->canvascoordPtr;
	*indexPtr = 0;
	for(i=0; i<linePtr->numPoints; i++) {
	    dist = hypot(coordPtr[0] - x, coordPtr[1] - y);
	    if (dist<bestDist) {
		bestDist = dist;
		*indexPtr = 2*i;
	    }
	    coordPtr += 2;
	}
    } else {
	if (Tcl_GetInt(interp, string, indexPtr) != TCL_OK) {
	    goto badIndex;
	}
	*indexPtr &= -2; /* if index is odd, make it even */
	if (*indexPtr < 0){
	    *indexPtr = 0;
	} else if (*indexPtr > (2*linePtr->numPoints)) {
	    *indexPtr = (2*linePtr->numPoints);
	}
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * TranslateDataset --
 *
 *	This procedure is called to move a line by a given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The position of the line is offset by (xDelta, yDelta), and
 *	the bounding box is updated in the generic part of the item
 *	structure.
 *
 *--------------------------------------------------------------
 *
 * We shouldn't really be calling this...
 */

void
TranslateDataset(canvas, itemPtr, deltaX, deltaY)
    Tk_Canvas canvas;			/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item that is being moved. */
    double deltaX, deltaY;		/* Amount by which item is to be
					 * moved. */
{
    DatasetItem *linePtr = (DatasetItem *) itemPtr;
    double *coordPtr;
    int i;

    return;

    fprintf(stderr, "Error:  TranslateDataset called!  Generating core file\n");
    abort();

    for (i = 0, coordPtr = linePtr->canvascoordPtr; i < linePtr->numPoints;
	    i++, coordPtr += 2) {
	coordPtr[0] += deltaX;
	coordPtr[1] += deltaY;
    }
    if (linePtr->firstArrowPtr != NULL) {
	for (i = 0, coordPtr = linePtr->firstArrowPtr; i < PTS_IN_ARROW;
		i++, coordPtr += 2) {
	    coordPtr[0] += deltaX;
	    coordPtr[1] += deltaY;
	}
    }
    if (linePtr->lastArrowPtr != NULL) {
	for (i = 0, coordPtr = linePtr->lastArrowPtr; i < PTS_IN_ARROW;
		i++, coordPtr += 2) {
	    coordPtr[0] += deltaX;
	    coordPtr[1] += deltaY;
	}
    }
    ComputeDatasetBbox(canvas, linePtr);
}

/*
 *--------------------------------------------------------------
 *
 * ParseArrowShape --
 *
 *	This procedure is called back during option parsing to
 *	parse arrow shape information.
 *
 * Results:
 *	The return value is a standard Tcl result:  TCL_OK means
 *	that the arrow shape information was parsed ok, and
 *	TCL_ERROR means it couldn't be parsed.
 *
 * Side effects:
 *	Arrow information in recordPtr is updated.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
ParseArrowShape(clientData, interp, tkwin, value, recordPtr, offset)
    ClientData clientData;	/* Not used. */
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Window tkwin;		/* Not used. */
    char *value;		/* Textual specification of arrow shape. */
    char *recordPtr;		/* Pointer to item record in which to
				 * store arrow information. */
    int offset;			/* Offset of shape information in widget
				 * record. */
{
    DatasetItem *linePtr = (DatasetItem *) recordPtr;
    double a, b, c;
    int argc;
    char **argv = NULL;

    if (offset != Tk_Offset(DatasetItem, arrowShapeA)) {
	panic("ParseArrowShape received bogus offset");
    }

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	syntaxError:
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "bad arrow shape \"", value,
		"\": must be list with three numbers", (char *) NULL);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return TCL_ERROR;
    }
    if (argc != 3) {
	goto syntaxError;
    }
    if ((Tk_CanvasGetCoord(interp, linePtr->canvas, argv[0], &a) != TCL_OK)
	    || (Tk_CanvasGetCoord(interp, linePtr->canvas, argv[1], &b)
		!= TCL_OK)
	    || (Tk_CanvasGetCoord(interp, linePtr->canvas, argv[2], &c)
		!= TCL_OK)) {
	goto syntaxError;
    }
    linePtr->arrowShapeA = a;
    linePtr->arrowShapeB = b;
    linePtr->arrowShapeC = c;
    ckfree((char *) argv);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * PrintArrowShape --
 *
 *	This procedure is a callback invoked by the configuration
 *	code to return a printable value describing an arrow shape.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

    /* ARGSUSED */
static char *
PrintArrowShape(clientData, tkwin, recordPtr, offset, freeProcPtr)
    ClientData clientData;	/* Not used. */
    Tk_Window tkwin;		/* Window associated with linePtr's widget. */
    char *recordPtr;		/* Pointer to item record containing current
				 * shape information. */
    int offset;			/* Offset of arrow information in record. */
    Tcl_FreeProc **freeProcPtr;	/* Store address of procedure to call to
				 * free string here. */
{
    DatasetItem *linePtr = (DatasetItem *) recordPtr;
    char *buffer;

    buffer = (char *) ckalloc(120);
    sprintf(buffer, "%.5g %.5g %.5g", linePtr->arrowShapeA,
	    linePtr->arrowShapeB, linePtr->arrowShapeC);
    *freeProcPtr = TCL_DYNAMIC;
    return buffer;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureArrows --
 *
 *	If arrowheads have been requested for a line, this
 *	procedure makes arrangements for the arrowheads.
 *
 * Results:
 *	Always returns TCL_OK.
 *
 * Side effects:
 *	Information in linePtr is set up for one or two arrowheads.
 *	the firstArrowPtr and lastArrowPtr polygons are allocated
 *	and initialized, if need be, and the end points of the line
 *	are adjusted so that a thick line doesn't stick out past
 *	the arrowheads.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
ConfigureArrows(canvas, linePtr)
    Tk_Canvas canvas;			/* Canvas in which arrows will be
					 * displayed (interp and tkwin
					 * fields are needed). */
    DatasetItem *linePtr;			/* Item to configure for arrows. */
{
    double *poly, *coordPtr;
    double dx, dy, length, sinTheta, cosTheta, temp;
    double fracHeight;			/* Line width as fraction of
					 * arrowhead width. */
    double backup;			/* Distance to backup end points
					 * so the line ends in the middle
					 * of the arrowhead. */
    double vertX, vertY;		/* Position of arrowhead vertex. */
    double shapeA, shapeB, shapeC;	/* Adjusted coordinates (see
					 * explanation below). */
    int width;
    Tk_State state = linePtr->header.state;

    if (linePtr->numPoints <2) {
	return TCL_OK;
    }

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    width = linePtr->outline.width;
    if (((TkCanvas *)canvas)->currentItemPtr == (Tk_Item *)linePtr) {
	if (linePtr->outline.activeWidth>width) {
	    width = linePtr->outline.activeWidth;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (linePtr->outline.disabledWidth>0) {
	    width = linePtr->outline.disabledWidth;
	}
    }

    /*
     * The code below makes a tiny increase in the shape parameters
     * for the line.  This is a bit of a hack, but it seems to result
     * in displays that more closely approximate the specified parameters.
     * Without the adjustment, the arrows come out smaller than expected.
     */

    shapeA = linePtr->arrowShapeA + 0.001;
    shapeB = linePtr->arrowShapeB + 0.001;
    shapeC = linePtr->arrowShapeC + width/2.0 + 0.001;

    /*
     * If there's an arrowhead on the first point of the line, compute
     * its polygon and adjust the first point of the line so that the
     * line doesn't stick out past the leading edge of the arrowhead.
     */

    fracHeight = (width/2.0)/shapeC;
    backup = fracHeight*shapeB + shapeA*(1.0 - fracHeight)/2.0;
    if (linePtr->arrow != lastUid) {
	poly = linePtr->firstArrowPtr;
	if (poly == NULL) {
	    poly = (double *) ckalloc((unsigned)
		    (2*PTS_IN_ARROW*sizeof(double)));
	    poly[0] = poly[10] = linePtr->canvascoordPtr[0];
	    poly[1] = poly[11] = linePtr->canvascoordPtr[1];
	    linePtr->firstArrowPtr = poly;
	}
	dx = poly[0] - linePtr->canvascoordPtr[2];
	dy = poly[1] - linePtr->canvascoordPtr[3];
	length = hypot(dx, dy);
	if (length == 0) {
	    sinTheta = cosTheta = 0.0;
	} else {
	    sinTheta = dy/length;
	    cosTheta = dx/length;
	}
	vertX = poly[0] - shapeA*cosTheta;
	vertY = poly[1] - shapeA*sinTheta;
	temp = shapeC*sinTheta;
	poly[2] = poly[0] - shapeB*cosTheta + temp;
	poly[8] = poly[2] - 2*temp;
	temp = shapeC*cosTheta;
	poly[3] = poly[1] - shapeB*sinTheta - temp;
	poly[9] = poly[3] + 2*temp;
	poly[4] = poly[2]*fracHeight + vertX*(1.0-fracHeight);
	poly[5] = poly[3]*fracHeight + vertY*(1.0-fracHeight);
	poly[6] = poly[8]*fracHeight + vertX*(1.0-fracHeight);
	poly[7] = poly[9]*fracHeight + vertY*(1.0-fracHeight);

	/*
	 * Polygon done.  Now move the first point towards the second so
	 * that the corners at the end of the line are inside the
	 * arrowhead.
	 */

	linePtr->canvascoordPtr[0] = poly[0] - backup*cosTheta;
	linePtr->canvascoordPtr[1] = poly[1] - backup*sinTheta;
    }

    /*
     * Similar arrowhead calculation for the last point of the line.
     */

    if (linePtr->arrow != firstUid) {
	coordPtr = linePtr->canvascoordPtr + 2*(linePtr->numPoints-2);
	poly = linePtr->lastArrowPtr;
	if (poly == NULL) {
	    poly = (double *) ckalloc((unsigned)
		    (2*PTS_IN_ARROW*sizeof(double)));
	    poly[0] = poly[10] = coordPtr[2];
	    poly[1] = poly[11] = coordPtr[3];
	    linePtr->lastArrowPtr = poly;
	}
	dx = poly[0] - coordPtr[0];
	dy = poly[1] - coordPtr[1];
	length = hypot(dx, dy);
	if (length == 0) {
	    sinTheta = cosTheta = 0.0;
	} else {
	    sinTheta = dy/length;
	    cosTheta = dx/length;
	}
	vertX = poly[0] - shapeA*cosTheta;
	vertY = poly[1] - shapeA*sinTheta;
	temp = shapeC*sinTheta;
	poly[2] = poly[0] - shapeB*cosTheta + temp;
	poly[8] = poly[2] - 2*temp;
	temp = shapeC*cosTheta;
	poly[3] = poly[1] - shapeB*sinTheta - temp;
	poly[9] = poly[3] + 2*temp;
	poly[4] = poly[2]*fracHeight + vertX*(1.0-fracHeight);
	poly[5] = poly[3]*fracHeight + vertY*(1.0-fracHeight);
	poly[6] = poly[8]*fracHeight + vertX*(1.0-fracHeight);
	poly[7] = poly[9]*fracHeight + vertY*(1.0-fracHeight);
	coordPtr[2] = poly[0] - backup*cosTheta;
	coordPtr[3] = poly[1] - backup*sinTheta;
    }

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DatasetToPostscript --
 *
 *	This procedure is called to generate Postscript for
 *	line items.
 *
 * Results:
 *	The return value is a standard Tcl result.  If an error
 *	occurs in generating Postscript then an error message is
 *	left in interp->result, replacing whatever used
 *	to be there.  If no error occurs, then Postscript for the
 *	item is appended to the result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
DatasetToPostscript(interp, canvas, itemPtr, prepass)
    Tcl_Interp *interp;			/* Leave Postscript or error message
					 * here. */
    Tk_Canvas canvas;			/* Information about overall canvas. */
    Tk_Item *itemPtr;			/* Item for which Postscript is
					 * wanted. */
    int prepass;			/* 1 means this is a prepass to
					 * collect font information;  0 means
					 * final Postscript is being created. */
{
    DatasetItem *linePtr = (DatasetItem *) itemPtr;
    char buffer[1024];
    char *style;
    double y1, y2;

    int width, i;
    XColor *color;
    Pixmap stipple;
    Tk_State state = itemPtr->state;
    double xsize = linePtr->graph->xaxismax - linePtr->graph->xaxismin;
    double ysize = linePtr->graph->yaxismax - linePtr->graph->yaxismin;

    double xscale = (linePtr->graph->bbox[2] - linePtr->graph->bbox[0]) / xsize;
    double yscale = (linePtr->graph->bbox[3] - linePtr->graph->bbox[1]) / ysize;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    y1 = Tk_CanvasPsY(canvas, linePtr->graph->bbox[1]);
    y2 = Tk_CanvasPsY(canvas, linePtr->graph->bbox[3]);

    width = linePtr->outline.width;
    color = linePtr->outline.color;
    stipple = linePtr->outline.stipple;
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (linePtr->outline.activeWidth>width) {
	    width = linePtr->outline.activeWidth;
	}
	if (linePtr->outline.activeColor!=NULL) {
	    color = linePtr->outline.activeColor;
	}
	if (linePtr->outline.activeStipple!=None) {
	    stipple = linePtr->outline.activeStipple;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (linePtr->outline.disabledWidth>0) {
	    width = linePtr->outline.disabledWidth;
	}
	if (linePtr->outline.disabledColor!=NULL) {
	    color = linePtr->outline.disabledColor;
	}
	if (linePtr->outline.disabledStipple!=None) {
	    stipple = linePtr->outline.disabledStipple;
	}
    }

    if (color == NULL || linePtr->numPoints<1 || linePtr->canvascoordPtr==NULL) {
	return TCL_OK;
    }

    /*
     * Clip the dataset around the graph boundaries
     */

    sprintf(buffer, "newpath %.15g %.15g moveto %.15g 0 rlineto 0 %.15g rlineto %.15g 0 rlineto closepath clip\n",
	    linePtr->graph->bbox[0], y1,
	    linePtr->graph->bbox[2]-linePtr->graph->bbox[0], y2-y1,
	    linePtr->graph->bbox[0]-linePtr->graph->bbox[2]);

    Tcl_AppendResult(interp, buffer, "newpath\n", (char *)NULL);

    if (linePtr->numPoints==1 && linePtr->datastyle == LINEDATA) {
	sprintf(buffer, "%.15g %.15g translate %.15g %.15g",
		linePtr->canvascoordPtr[0], Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[1]),
		width/2.0, width/2.0);
	Tcl_AppendResult(interp, "matrix currentmatrix\n",buffer,
		" scale 1 0 moveto 0 0 1 0 360 arc\nsetmatrix\n", (char *) NULL);
	if (Tk_CanvasPsColor(interp, canvas, color)
		!= TCL_OK) {
	    return TCL_ERROR;
	}
	if (stipple != None) {
	    Tcl_AppendResult(interp, "clip ", (char *) NULL);
	    if (Tk_CanvasPsStipple(interp, canvas, stipple) != TCL_OK) {
		return TCL_ERROR;
	    }
	} else {
	    Tcl_AppendResult(interp, "fill\n", (char *) NULL);
	}
	return TCL_OK;
    }
    /*
     * Generate a path for the line's center-line (do this differently
     * for straight lines and smoothed lines).
     */

    if ((linePtr->datastyle & LINEDATA) && (linePtr->outline.gc != None)) {
      if ((!linePtr->smooth) || (linePtr->numPoints < 3)) {
	Tk_CanvasPsPath(interp, canvas, linePtr->canvascoordPtr, linePtr->numPoints);
	} else {
	    if (stipple == None) {
		TkMakeBezierPostscript(interp, canvas, linePtr->canvascoordPtr,
			linePtr->numPoints, 0 /* Not Used */);
	    } else {
		/*
		* Special hack: Postscript printers don't appear to be able
		* to turn a path drawn with "curveto"s into a clipping path
		* without exceeding resource limits, so TkMakeBezierPostscript
		* won't work for stippled curves.  Instead, generate all of
		* the intermediate points here and output them into the
		* Postscript file with "lineto"s instead.
		*/

		double staticPoints[2*MAX_STATIC_POINTS];
		double *pointPtr;
		int numPoints;

		numPoints = 1 + linePtr->numPoints*linePtr->splineSteps;
		pointPtr = staticPoints;
		if (numPoints > MAX_STATIC_POINTS) {
		    pointPtr = (double *) ckalloc((unsigned)
			    (numPoints * 2 * sizeof(double)));
		}
		numPoints = TkMakeBezierCurve(canvas, linePtr->canvascoordPtr,
			linePtr->numPoints, linePtr->splineSteps, (XPoint *) NULL,
			pointPtr);
		Tk_CanvasPsPath(interp, canvas, pointPtr, numPoints);
		if (pointPtr != staticPoints) {
		    ckfree((char *) pointPtr);
		}
	    }
	}

	/*
	* Set other line-drawing parameters and stroke out the line.
	*/

	style = "0 setlinecap\n";
	if (linePtr->capStyle == CapRound) {
	    style = "1 setlinecap\n";
	} else if (linePtr->capStyle == CapProjecting) {
	    style = "2 setlinecap\n";
	}
	Tcl_AppendResult(interp, style, (char *) NULL);
	style = "0 setlinejoin\n";
	if (linePtr->joinStyle == JoinRound) {
	    style = "1 setlinejoin\n";
	} else if (linePtr->joinStyle == JoinBevel) {
	    style = "2 setlinejoin\n";
	}
	Tcl_AppendResult(interp, style, (char *) NULL);

	if (Tk_CanvasPsOutline(canvas, itemPtr,
		&(linePtr->outline)) != TCL_OK) {
	    return TCL_ERROR;
	}
    }

    /*
     * Draw the error bars before the symbols so that if the
     * symbols are opaque, they will cover the portion of the
     * error bar below
     */

    for(i=0; i < linePtr->numPoints; i++) {
	if (linePtr->datatype->datatype & (XSIGXYDATA|XSIGXYSIGYDATA|XSIGX1SIGX2YDATA|XSIGX1SIGX2YSIGY1SIGY2DATA)) {
	    sprintf(buffer, "%.15g %.15g moveto %.15g %.15g lineto\n",
		    linePtr->canvascoordPtr[i*2] + linePtr->errorPtr[i].xsigh*xscale,
		    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1]),
		    linePtr->canvascoordPtr[i*2] - linePtr->errorPtr[i].xsigl*xscale,
		    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1]));
	    Tcl_AppendResult(interp, buffer, (char *)NULL);
	}
	if (linePtr->datatype->datatype & (XYSIGYDATA|XSIGXYSIGYDATA|XYSIGY1SIGY2DATA|XSIGX1SIGX2YSIGY1SIGY2DATA)) {
	    sprintf(buffer, "%.15g %.15g moveto %.15g %.15g lineto\n",
		    linePtr->canvascoordPtr[i*2],
		    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1] - linePtr->errorPtr[i].ysigh*yscale),
		    linePtr->canvascoordPtr[i*2],
		    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1] + linePtr->errorPtr[i].ysigl*yscale));
	    Tcl_AppendResult(interp, buffer, (char *)NULL);
	}
    }

    /*
    * Set other line-drawing parameters and stroke out the error bars.
    */

    style = "0 setlinecap\n";
    if (linePtr->capStyle == CapRound) {
	style = "1 setlinecap\n";
    } else if (linePtr->capStyle == CapProjecting) {
	style = "2 setlinecap\n";
    }
    Tcl_AppendResult(interp, style, (char *) NULL);
    style = "0 setlinejoin\n";
    if (linePtr->joinStyle == JoinRound) {
	style = "1 setlinejoin\n";
    } else if (linePtr->joinStyle == JoinBevel) {
	style = "2 setlinejoin\n";
    }
    Tcl_AppendResult(interp, style, (char *) NULL);

    if (Tk_CanvasPsOutline(canvas, itemPtr,
	    &(linePtr->symoutline)) != TCL_OK) {
	return TCL_ERROR;
    }

    if ((linePtr->datastyle & SCATTERDATA) && linePtr->symoutline.gc != None) {

	switch (linePtr->symstyle) {
	    case NONE: {
		break;
	    }
	    case SYMCIRCLE: {
		for(i=0; i < linePtr->numPoints; i++) {
		    sprintf(buffer, "matrix currentmatrix\n%.15g %.15g translate %d %d scale 1 0 moveto 0 0 1 0 360 arc\nsetmatrix\n",
			linePtr->canvascoordPtr[i*2],
			Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1]),
			linePtr->symsize/2,
			linePtr->symsize/2);
		    Tcl_AppendResult(interp, buffer, (char *)NULL);

		    if (linePtr->symopaque) {
			Tcl_AppendResult(interp, "fill\n", (char *)NULL);
		    }
		}
		break;
	    }
	    case SYMX: {
		for(i=0; i < linePtr->numPoints; i++) {
		    sprintf(buffer, "%.15g %.15g moveto %.15g %.15g lineto %.15g %.15g moveto %.15g %.15g lineto\n",
			linePtr->canvascoordPtr[i*2]-linePtr->symsize/2.0,
			Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1]-linePtr->symsize/2.0),
			linePtr->canvascoordPtr[i*2]+linePtr->symsize/2.0,
			Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1]+linePtr->symsize/2.0),
			linePtr->canvascoordPtr[i*2]+linePtr->symsize/2.0,
			Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1]-linePtr->symsize/2.0),
			linePtr->canvascoordPtr[i*2]-linePtr->symsize/2.0,
			Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1]+linePtr->symsize/2.0));
		    Tcl_AppendResult(interp, buffer, (char *)NULL);
		}
		break;
	    }
	    case SYMEBAR: {
		double xsymsize=0;
		double ysymsize=0;

		if (linePtr->datatype->datatype & (XSIGXYDATA|XSIGXYSIGYDATA|XSIGX1SIGX2YDATA|XSIGX1SIGX2YSIGY1SIGY2DATA)) {
		    xsymsize = linePtr->symsize/2.0;
		}

		if (linePtr->datatype->datatype & (XYSIGYDATA|XSIGXYSIGYDATA|XYSIGY1SIGY2DATA|XSIGX1SIGX2YSIGY1SIGY2DATA)) {
		    ysymsize = linePtr->symsize/2.0;
		}
		    
		for(i=0; i < linePtr->numPoints; i++) {
		    sprintf(buffer, "%.15g %.15g moveto %.15g %.15g lineto %.15g %.15g moveto %.15g %.15g lineto\n",
		    /*
		     * Error bar on top
		     */
			    linePtr->canvascoordPtr[i*2] - ysymsize,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1] - linePtr->errorPtr[i].ysigh*yscale),
			    linePtr->canvascoordPtr[i*2] + ysymsize,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1] - linePtr->errorPtr[i].ysigh*yscale),
		    /*
		     * Error bar on bottom
		     */
			    linePtr->canvascoordPtr[i*2] - ysymsize,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1] + linePtr->errorPtr[i].ysigl*yscale),
			    linePtr->canvascoordPtr[i*2] + ysymsize,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1] + linePtr->errorPtr[i].ysigl*yscale));
		    Tcl_AppendResult(interp, buffer, (char *)NULL);

		    sprintf(buffer, "%.15g %.15g moveto %.15g %.15g lineto %.15g %.15g moveto %.15g %.15g lineto\n",
		    /*
		     * Error bar on left
		     */
			    linePtr->canvascoordPtr[i*2] - linePtr->errorPtr[i].xsigl*xscale,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1] - xsymsize),
			    linePtr->canvascoordPtr[i*2] - linePtr->errorPtr[i].xsigl*xscale,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1] + xsymsize),
		    /*
		     * Error bar on right
		     */

			    linePtr->canvascoordPtr[i*2] + linePtr->errorPtr[i].xsigh*xscale,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1] - xsymsize),
			    linePtr->canvascoordPtr[i*2] + linePtr->errorPtr[i].xsigh*xscale,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1] + xsymsize));
		    Tcl_AppendResult(interp, buffer, (char *)NULL);
		}
		break;
	    }
	    case SYMDIAMOND: {
		double symsize = linePtr->symsize/2;

		for(i=0; i < linePtr->numPoints; i++) {
		    sprintf(buffer, "%.15g %.15g moveto %.15g %.15g lineto %.15g %.15g lineto %.15g %.15g lineto %.15g %.15g lineto\n",
			    linePtr->canvascoordPtr[i*2] - symsize,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1]),

			    linePtr->canvascoordPtr[i*2],
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1] + symsize),

			    linePtr->canvascoordPtr[i*2] + symsize,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1]),

			    linePtr->canvascoordPtr[i*2],
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1] - symsize),

			    linePtr->canvascoordPtr[i*2] - symsize,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1]));
		    Tcl_AppendResult(interp, buffer, (char *)NULL);

		    if (linePtr->symopaque) {
			Tcl_AppendResult(interp, "fill\n", (char *)NULL);
		    }
		}

		break;
	    }
	    case SYMSQUARE: {
		double symsize = linePtr->symsize/2;

		for(i=0; i < linePtr->numPoints; i++) {
		    sprintf(buffer, "%.15g %.15g moveto %.15g %.15g lineto %.15g %.15g lineto %.15g %.15g lineto %.15g %.15g lineto\n",
			    linePtr->canvascoordPtr[i*2] - symsize,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1] - symsize),

			    linePtr->canvascoordPtr[i*2] + symsize,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1] - symsize),

			    linePtr->canvascoordPtr[i*2] + symsize,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1] + symsize),

			    linePtr->canvascoordPtr[i*2] - symsize,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1] + symsize),

			    linePtr->canvascoordPtr[i*2] - symsize,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[i*2+1] - symsize));

		    Tcl_AppendResult(interp, buffer, (char *)NULL);

		    if (linePtr->symopaque) {
			Tcl_AppendResult(interp, "fill\n", (char *)NULL);
		    }
		}

		break;
	    }
	    case SYMNTRI: {
		double xsymsize = linePtr->symsize;
		double ysymsize = linePtr->symsize/1.4142136;

		for(i=0; i < linePtr->numPoints; i++) {
		    sprintf(buffer, "%.15g %.15g moveto %.15g %.15g lineto %.15g %.15g lineto %.15g %.15g lineto\n",
			    linePtr->canvascoordPtr[2*i] - xsymsize/2.0,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[2*i+1] + ysymsize/2),

			    linePtr->canvascoordPtr[2*i] + xsymsize/2.0,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[2*i+1] + ysymsize/2),

			    linePtr->canvascoordPtr[2*i],
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[2*i+1] - ysymsize/2),

			    linePtr->canvascoordPtr[2*i] - xsymsize/2.0,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[2*i+1] + ysymsize/2));

		    Tcl_AppendResult(interp, buffer, (char *)NULL);

		    if (linePtr->symopaque) {
			Tcl_AppendResult(interp, "fill\n", (char *)NULL);
		    }
		}

		break;
	    }
	    case SYMSTRI: {
		double xsymsize = linePtr->symsize;
		double ysymsize = linePtr->symsize/1.4142136;

		for(i=0; i < linePtr->numPoints; i++) {
		    sprintf(buffer, "%.15g %.15g moveto %.15g %.15g lineto %.15g %.15g lineto %.15g %.15g lineto\n",
			    linePtr->canvascoordPtr[2*i] - xsymsize/2.0,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[2*i+1] - ysymsize/2),

			    linePtr->canvascoordPtr[2*i] + xsymsize/2.0,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[2*i+1] - ysymsize/2),

			    linePtr->canvascoordPtr[2*i],
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[2*i+1] + ysymsize/2),

			    linePtr->canvascoordPtr[2*i] - xsymsize/2.0,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[2*i+1] - ysymsize/2));

		    Tcl_AppendResult(interp, buffer, (char *)NULL);

		    if (linePtr->symopaque) {
			Tcl_AppendResult(interp, "fill\n", (char *)NULL);
		    }
		}

		break;
	    }
	    case SYMETRI: {
		double xsymsize = linePtr->symsize/1.4142136;
		double ysymsize = linePtr->symsize;

		for(i=0; i < linePtr->numPoints; i++) {
		    sprintf(buffer, "%.15g %.15g moveto %.15g %.15g lineto %.15g %.15g lineto %.15g %.15g lineto\n",
			    linePtr->canvascoordPtr[2*i] - xsymsize/2.0,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[2*i+1] - ysymsize/2),

			    linePtr->canvascoordPtr[2*i] - xsymsize/2.0,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[2*i+1] + ysymsize/2),

			    linePtr->canvascoordPtr[2*i] + xsymsize/2.0,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[2*i+1]),

			    linePtr->canvascoordPtr[2*i] - xsymsize/2.0,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[2*i+1] - ysymsize/2));

		    Tcl_AppendResult(interp, buffer, (char *)NULL);

		    if (linePtr->symopaque) {
			Tcl_AppendResult(interp, "fill\n", (char *)NULL);
		    }
		}

		break;
	    }
	    case SYMWTRI: {
		double xsymsize = linePtr->symsize/1.4142136;
		double ysymsize = linePtr->symsize;

		for(i=0; i < linePtr->numPoints; i++) {
		    sprintf(buffer, "%.15g %.15g moveto %.15g %.15g lineto %.15g %.15g lineto %.15g %.15g lineto\n",
			    linePtr->canvascoordPtr[2*i] + xsymsize/2.0,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[2*i+1] - ysymsize/2),

			    linePtr->canvascoordPtr[2*i] + xsymsize/2.0,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[2*i+1] + ysymsize/2),

			    linePtr->canvascoordPtr[2*i] - xsymsize/2.0,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[2*i+1]),

			    linePtr->canvascoordPtr[2*i] + xsymsize/2.0,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[2*i+1] - ysymsize/2));

		    Tcl_AppendResult(interp, buffer, (char *)NULL);

		    if (linePtr->symopaque) {
			Tcl_AppendResult(interp, "fill\n", (char *)NULL);
		    }
		}

		break;
	    }
	    case SYMPLUS: {
		double symsize = linePtr->symsize/2.0;

		for(i=0; i < linePtr->numPoints; i++) {
		    sprintf(buffer, "%.15g %.15g moveto %.15g %.15g lineto %.15g %.15g moveto %.15g %.15g lineto\n",
			    linePtr->canvascoordPtr[2*i] - symsize,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[2*i+1]),
			    linePtr->canvascoordPtr[2*i] + symsize,
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[2*i+1]),

			    linePtr->canvascoordPtr[2*i],
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[2*i+1] - symsize),
			    linePtr->canvascoordPtr[2*i],
			    Tk_CanvasPsY(canvas, linePtr->canvascoordPtr[2*i+1] + symsize));

		    Tcl_AppendResult(interp, buffer, (char *)NULL);
		}

		break;
	    }
	}

	/*
	* Set other line-drawing parameters and stroke out the symbols.
	*/

	style = "0 setlinecap\n";
	if (linePtr->capStyle == CapRound) {
	    style = "1 setlinecap\n";
	} else if (linePtr->capStyle == CapProjecting) {
	    style = "2 setlinecap\n";
	}
	Tcl_AppendResult(interp, style, (char *) NULL);
	style = "0 setlinejoin\n";
	if (linePtr->joinStyle == JoinRound) {
	    style = "1 setlinejoin\n";
	} else if (linePtr->joinStyle == JoinBevel) {
	    style = "2 setlinejoin\n";
	}
	Tcl_AppendResult(interp, style, (char *) NULL);


	if (Tk_CanvasPsOutline(canvas, itemPtr,
		&(linePtr->symoutline)) != TCL_OK) {
	    return TCL_ERROR;
	}
    }

    /*
     * Output polygons for the arrowheads, if there are any.
     */

    if (linePtr->firstArrowPtr != NULL) {
	if (stipple != None) {
	    Tcl_AppendResult(interp, "grestore gsave\n",
		    (char *) NULL);
	}
	if (ArrowheadPostscript(interp, canvas, linePtr,
		linePtr->firstArrowPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    if (linePtr->lastArrowPtr != NULL) {
	if (stipple != None) {
	    Tcl_AppendResult(interp, "grestore gsave\n", (char *) NULL);
	}
	if (ArrowheadPostscript(interp, canvas, linePtr,
		linePtr->lastArrowPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ArrowheadPostscript --
 *
 *	This procedure is called to generate Postscript for
 *	an arrowhead for a line item.
 *
 * Results:
 *	The return value is a standard Tcl result.  If an error
 *	occurs in generating Postscript then an error message is
 *	left in interp->result, replacing whatever used
 *	to be there.  If no error occurs, then Postscript for the
 *	arrowhead is appended to the result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
ArrowheadPostscript(interp, canvas, linePtr, arrowPtr)
    Tcl_Interp *interp;			/* Leave Postscript or error message
					 * here. */
    Tk_Canvas canvas;			/* Information about overall canvas. */
    DatasetItem *linePtr;		/* Line item for which Postscript is
					 * being generated. */
    double *arrowPtr;			/* Pointer to first of five points
					 * describing arrowhead polygon. */
{
    Pixmap stipple;
    Tk_State state = linePtr->header.state;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    stipple = linePtr->outline.stipple;
    if (((TkCanvas *)canvas)->currentItemPtr == (Tk_Item *)linePtr) {
	if (linePtr->outline.activeStipple!=None) {
	    stipple = linePtr->outline.activeStipple;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (linePtr->outline.activeStipple!=None) {
	    stipple = linePtr->outline.disabledStipple;
	}
    }

    Tk_CanvasPsPath(interp, canvas, arrowPtr, PTS_IN_ARROW);
    if (stipple != None) {
	Tcl_AppendResult(interp, "clip ", (char *) NULL);
	if (Tk_CanvasPsStipple(interp, canvas, stipple)
		!= TCL_OK) {
	    return TCL_ERROR;
	}
    } else {
	Tcl_AppendResult(interp, "fill\n", (char *) NULL);
    }
    return TCL_OK;
}

static void
DatasetUpdateGraphCoords(DatasetItem *linePtr)
{
    register double xsize, ysize;
    register double xscale, yscale;
    int i;

    if (linePtr->graph == NULL)
	return;

    /*
     * Get the scale from the graph's axis limits
     */

    xsize = linePtr->graph->xaxismax - linePtr->graph->xaxismin;
    ysize = linePtr->graph->yaxismax - linePtr->graph->yaxismin;

    xscale = (linePtr->graph->bbox[2] - linePtr->graph->bbox[0]) / xsize;
    yscale = (linePtr->graph->bbox[3] - linePtr->graph->bbox[1]) / ysize;

    /*
     * fprintf(stderr, "xsize,ysize = (%g,%g)\n", xsize, ysize);
     * fprintf(stderr, "xscale,yscale = (%g,%g)\n", xscale, yscale);
     * fprintf(stderr, "Graph coords:\n");
     */
    for(i = 0; i < linePtr->numPoints*2; i+=2) {
	linePtr->graphcoordPtr[i] = 
	    (linePtr->datacoordPtr[i]  - linePtr->graph->xaxismin) * xscale;
	linePtr->graphcoordPtr[i+1]=
	    linePtr->graph->bbox[3] - linePtr->graph->bbox[1] - 
	    (linePtr->datacoordPtr[i+1]- linePtr->graph->yaxismin) * yscale;
	/*
	fprintf(stderr, "\t(%g,%g)\n", linePtr->graphcoordPtr[i],
		linePtr->graphcoordPtr[i+1]);
	*/
    }
}

/*
 * Update the canvascoordPtr array in the line item.  This function
 * assumes that the proper scaled graph coordinates are stored
 * in graphcoordPtr.  If the graph coordinates have not been scaled
 * then this function will not work correctly.
 */

static void
DatasetUpdateCanvasCoords(DatasetItem *linePtr)
{
    register double xoffset, yoffset;
    int i;

    /*
     * This shouldn't be necessary.  I'll take it out later when I know
     * it works.
     */

    DatasetUpdateGraphCoords(linePtr);

    if (linePtr->graph == NULL)
	return;

    xoffset = linePtr->graph->bbox[0];
    yoffset = linePtr->graph->bbox[1];

    /*
    fprintf(stderr, "Canvas coords:\n");
    */
    for(i=0; i < linePtr->numPoints*2; i+=2) {
	linePtr->canvascoordPtr[i] = linePtr->graphcoordPtr[i] + xoffset;
	linePtr->canvascoordPtr[i+1]=linePtr->graphcoordPtr[i+1]+yoffset;
	/*
	fprintf(stderr, "\t(%g,%g)\n", linePtr->canvascoordPtr[i],
		linePtr->canvascoordPtr[i+1]);
	*/
    }
}

static int
Tk_GraphParseProc(ClientData clientData, Tcl_Interp *interp,
	Tk_Window tkwin, char *value, char *widgRec, int offset)
{
    GraphItem **graph = ((GraphItem **)(widgRec + offset));
    TkCanvas *canvas = ((TkCanvas *) ((DatasetItem *)widgRec)->canvas);
    int id;
    Tk_Item *itemPtr;
    TagSearch search;

    if (value == (char *)NULL) {
	/*
	 * fprintf(stderr, "Setting graph to nothing\n");
	 */
	RemoveDatasetFromGraph((DatasetItem *)widgRec, *graph);
	*graph = NULL;
    } else {
	/*
	 * Find the canvas item associated with this id
	 */
	itemPtr = StartTagSearch(canvas, value, &search);
 
	/*
	 * Make sure this canvas item is a "lgraph" item
	 */

	if (itemPtr==NULL || (strcmp(itemPtr->typePtr->name, "lgraph")!=0)){
	    Tcl_AppendResult(interp, "Bad graph item:  ", value, (char *)NULL);
	    return TCL_ERROR;
	}

	/*
	 * We have a valid graph item.  Add this dataset to it's child list
	 */

	RemoveDatasetFromGraph((DatasetItem *)widgRec, *graph);
	*graph = (GraphItem *)itemPtr;
	AddDatasetToGraph((DatasetItem *)widgRec, *graph);
    }

    return TCL_OK;
}

static char *
Tk_GraphPrintProc(ClientData clientData,
	Tk_Window tkwin, char *widgRec, int offset,
	Tcl_FreeProc **freeProcPtr)
{
    GraphItem *graph = ((DatasetItem *)widgRec)->graph;
    char *buf = (char *)ckalloc(sizeof(char)*TCL_DOUBLE_SPACE);

    if (graph == NULL)
	*buf = (char) NULL;
    else
	sprintf(buf, "%d", ((Tk_Item *)graph)->id);

    return buf;
}

/*
 *--------------------------------------------------------------
 *
 * StartTagSearch --
 *
 *	This procedure is called to initiate an enumeration of
 *	all items in a given canvas that contain a given tag.
 *
 * Results:
 *	The return value is a pointer to the first item in
 *	canvasPtr that matches tag, or NULL if there is no
 *	such item.  The information at *searchPtr is initialized
 *	such that successive calls to NextItem will return
 *	successive items that match tag.
 *
 * Side effects:
 *	SearchPtr is linked into a list of searches in progress
 *	on canvasPtr, so that elements can safely be deleted
 *	while the search is in progress.  EndTagSearch must be
 *	called at the end of the search to unlink searchPtr from
 *	this list.
 *
 *--------------------------------------------------------------
 */

static Tk_Item *
StartTagSearch(canvasPtr, tag, searchPtr)
    TkCanvas *canvasPtr;		/* Canvas whose items are to be
					 * searched. */
    char *tag;				/* String giving tag value. */
    TagSearch *searchPtr;		/* Record describing tag search;
					 * will be initialized here. */
{
    int id;
    Tk_Item *itemPtr, *prevPtr;
    Tk_Uid *tagPtr;
    Tk_Uid uid;
    int count;

    /*
     * Initialize the search.
     */

    searchPtr->canvasPtr = canvasPtr;
    searchPtr->searchOver = 0;

    /*
     * Find the first matching item in one of several ways. If the tag
     * is a number then it selects the single item with the matching
     * identifier.  In this case see if the item being requested is the
     * hot item, in which case the search can be skipped.
     */

    if (isdigit(UCHAR(*tag))) {
	char *end;
	Tcl_HashEntry *entryPtr;

	id = strtoul(tag, &end, 0);
	if (*end == 0) {
	    itemPtr = canvasPtr->hotPtr;
	    prevPtr = canvasPtr->hotPrevPtr;
	    if ((itemPtr == NULL) || (itemPtr->id != id) || (prevPtr == NULL)
		    || (prevPtr->nextPtr != itemPtr)) {
	      entryPtr = Tcl_FindHashEntry(&canvasPtr->idTable, (char *)id);
	      if (entryPtr) {
		itemPtr = (Tk_Item *)Tcl_GetHashValue(entryPtr);
		prevPtr = itemPtr->prevPtr;
	      } else {
	        prevPtr = itemPtr = NULL;
	      }
	    }
	    searchPtr->prevPtr = prevPtr;
	    searchPtr->searchOver = 1;
	    canvasPtr->hotPtr = itemPtr;
	    canvasPtr->hotPrevPtr = prevPtr;
	    return itemPtr;
	}
    }

    searchPtr->tag = uid = Tk_GetUid(tag);
    if (uid == allUid) {

	/*
	 * All items match.
	 */

	searchPtr->tag = NULL;
	searchPtr->prevPtr = NULL;
	searchPtr->currentPtr = canvasPtr->firstItemPtr;
	return canvasPtr->firstItemPtr;
    }

    /*
     * None of the above.  Search for an item with a matching tag.
     */

    for (prevPtr = NULL, itemPtr = canvasPtr->firstItemPtr; itemPtr != NULL;
	    prevPtr = itemPtr, itemPtr = itemPtr->nextPtr) {
	for (tagPtr = itemPtr->tagPtr, count = itemPtr->numTags;
		count > 0; tagPtr++, count--) {
	    if (*tagPtr == uid) {
		searchPtr->prevPtr = prevPtr;
		searchPtr->currentPtr = itemPtr;
		return itemPtr;
	    }
	}
    }
    searchPtr->prevPtr = prevPtr;
    searchPtr->searchOver = 1;
    return NULL;
}

static int
Tk_DontSetOption(ClientData clientData, Tcl_Interp *interp, Tk_Window tkwin,
	char *value, char *widgRec, int offset)
{
    return TCL_OK;
}

static char *
Tk_CustomPrintDouble(ClientData clientData, Tk_Window tkwin, char *widgRec,
	int offset, Tcl_FreeProc **freeProcPtr)
{
    register double *doublePtr = (double *)(widgRec + offset);
    char *buf = (char *)NULL;

    buf = (char *)ckalloc(sizeof(char)*TCL_DOUBLE_SPACE);


    *freeProcPtr = TCL_DYNAMIC;

    sprintf(buf, "%g", *doublePtr);

    return buf;
}

static void
RemoveDatasetFromGraph(DatasetItem *dataPtr, GraphItem *graphPtr)
{
    int		i;
    Tk_Item	*itemPtr = (Tk_Item *)NULL;

    if (!graphPtr)
	return;

    /*
     * printf("RemoveDatasetFromGraph():  nchildren = %d\n", graphPtr->nchildren);
     * printf("RemoveDatasetFromGraph():  child_arr_size = %d\n", graphPtr->child_arr_size);
     */
    for(i = 0, itemPtr = *graphPtr->children ;
	i < graphPtr->nchildren && (itemPtr != (Tk_Item *)dataPtr); 
	itemPtr = graphPtr->children[i++]);
	/*
	 * printf("RemoveDatasetFromGraph(1):  i = %d\n", i);
	 */

    for(;
	i < graphPtr->nchildren && (i+1 < graphPtr->child_arr_size);
	graphPtr->children[i] = graphPtr->children[i+1], i++);
	/*
	 * printf("RemoveDatasetFromGraph(2):  i = %d\n", i);
	 */

    for(;
	i < graphPtr->child_arr_size;
	graphPtr->children[i++] = (Tk_Item *)NULL);
	/*
	 * printf("RemoveDatasetFromGraph(3):  i = %d\n", i);
	 */

    graphPtr->children[graphPtr->nchildren] = (Tk_Item *)NULL;
    graphPtr->nchildren--;

    if (!dataPtr->delete_flag) {
	Tk_CanvasEventuallyRedraw(dataPtr->canvas, 
		dataPtr->header.x1, dataPtr->header.y1,
		dataPtr->header.x2, dataPtr->header.y2);
    }
}

static void
AddDatasetToGraph(DatasetItem *dataPtr, GraphItem *graphPtr)
{
    Tk_Item	*itemPtr = (Tk_Item *)NULL;
    int		i;

    Tk_CanvasEventuallyRedraw(dataPtr->canvas, 
	    dataPtr->header.x1, dataPtr->header.y1,
	    dataPtr->header.x2, dataPtr->header.y2);

    if (!graphPtr)
	return;

    if (graphPtr->nchildren+1 >= graphPtr->child_arr_size) {
	Tk_Item **tPtr = (Tk_Item **)ckalloc(sizeof(Tk_Item *)*graphPtr->child_arr_size*2);
	graphPtr->child_arr_size *= 2;
	for(i = 0; i < graphPtr->nchildren; tPtr[i] = graphPtr->children[i++]);
	for(; i < graphPtr->child_arr_size; tPtr[i++] = (Tk_Item *)NULL);
	ckfree((char *)(graphPtr->children));
	graphPtr->children = tPtr;
    }

    for(i = 0, itemPtr = *graphPtr->children ;
	i < graphPtr->nchildren && (itemPtr != (Tk_Item *)dataPtr); 
	itemPtr = graphPtr->children[i++]);

    if (i >= graphPtr->nchildren) {
	graphPtr->children[graphPtr->nchildren++] = (Tk_Item *)dataPtr;
    }
}

static char *
Tk_DatatypePrintProc(ClientData clientData, Tk_Window tkwin, char *widgRec, int offset, Tcl_FreeProc **freeProcPtr)
{
    register int *datatypePtr = (int *)(widgRec + offset);
    char *buf = (char *)NULL;

    buf = (char *)ckalloc(sizeof(char)*32);


    *freeProcPtr = TCL_DYNAMIC;

    switch (*datatypePtr) {
	case 0x00:	strcpy(buf, "none");	break;
	case 0x01:	strcpy(buf, "xy");	break;
	case 0x02:	strcpy(buf, "xysigy");	break;
	case 0x04:	strcpy(buf, "xsigxy");	break;
	case 0x08:	strcpy(buf, "xsigxysigy");	break;
	default:	strcpy(buf, "?");	break;
    }

    return buf;
}

static int
Tk_DatastyleParseProc(ClientData clientData, Tcl_Interp *interp, Tk_Window tkwin, char *value, char *widgRec, int offset)
{
    int *datastyle = ((int *)(widgRec + offset));

    if (strcasecmp(value, "line") == 0) {
	*datastyle = LINEDATA;
    } else if (strcasecmp(value, "scatter") == 0) {
	*datastyle = SCATTERDATA;
    } else if (strcasecmp(value, "line+scatter") == 0) {
	*datastyle = SCATTERDATA|LINEDATA;
    } else if (strcasecmp(value, "scatter+line") == 0) {
	*datastyle = SCATTERDATA|LINEDATA;
    } else {
	Tcl_AppendResult(interp, "Bad data style '", value, "'.  Must be one of 'line', 'scatter', or 'scatter+line'", (char *)NULL);
	return TCL_ERROR;
    }

    return TCL_OK;
}

static char *
Tk_DatastylePrintProc(ClientData clientData, Tk_Window tkwin, char *widgRec, int offset, Tcl_FreeProc **freeProcPtr)
{
    register int *datastylePtr = (int *)(widgRec + offset);
    char *buf = (char *)NULL;

    buf = (char *)ckalloc(sizeof(char)*32);


    *freeProcPtr = TCL_DYNAMIC;

    switch (*datastylePtr) {
	case 0x00:		strcpy(buf, "none");	break;
	case NONE:		strcpy(buf, "none");	break;
	case LINEDATA:		strcpy(buf, "line");	break;
	case SCATTERDATA:	strcpy(buf, "scatter");	break;
	default:		strcpy(buf, "?");	break;
    }

    return buf;
}

static int
Tk_SymstyleParseProc(ClientData clientData, Tcl_Interp *interp, Tk_Window tkwin, char *value, char *widgRec, int offset)
{
    int *symstyle = ((int *)(widgRec + offset));

    if (strcasecmp(value, "none") == 0) {
	*symstyle = NONE;
    } else if (strcasecmp(value, "circle") == 0) {
	*symstyle = SYMCIRCLE;
    } else if (strcasecmp(value, "x") == 0) {
	*symstyle = SYMX;
    } else if (strcasecmp(value, "ebar") == 0) {
	*symstyle = SYMEBAR;
    } else if (strcasecmp(value, "square") == 0) {
	*symstyle = SYMSQUARE;
    } else if (strcasecmp(value, "diamond") == 0) {
	*symstyle = SYMDIAMOND;
    } else if (strcasecmp(value, "plus") == 0) {
	*symstyle = SYMPLUS;
    } else if (strcasecmp(value, "triangle-up") == 0) {
	*symstyle = SYMNTRI;
    } else if (strcasecmp(value, "triangle-down") == 0) {
	*symstyle = SYMSTRI;
    } else if (strcasecmp(value, "triangle-left") == 0) {
	*symstyle = SYMWTRI;
    } else if (strcasecmp(value, "triangle-right") == 0) {
	*symstyle = SYMETRI;
    } else {
	Tcl_AppendResult(interp, "Bad point style '", value, "'.  Must be one of 'circle', 'x', 'ebar', 'square', 'diamond', 'plus', 'triangle-up', 'triangle-down, 'triangle-left', 'triangle-right', or 'none'", (char *)NULL);
	return TCL_ERROR;
    }

    return TCL_OK;
}

static char *
Tk_SymstylePrintProc(ClientData clientData, Tk_Window tkwin, char *widgRec, int offset, Tcl_FreeProc **freeProcPtr)
{
    register int *symstylePtr = (int *)(widgRec + offset);
    char *buf = (char *)NULL;

    buf = (char *)ckalloc(sizeof(char)*32);


    *freeProcPtr = TCL_DYNAMIC;

    switch (*symstylePtr) {
	case 0x00:		strcpy(buf, "none");	break;
	case NONE:		strcpy(buf, "none");	break;
	case SYMCIRCLE:	strcpy(buf, "circle");	break;
	case SYMX:		strcpy(buf, "x");	break;
	default:		strcpy(buf, "?");	break;
    }

    return buf;
}
