/* 
 * tkCanvCircle.c --
 *
 *	This file implements circle items for canvas
 *	widgets.
 *
 */

#include <stdio.h>
#include <tk.h>
#include <tkInt.h>
#include <tkPort.h>
#include <tkCanvas.h>

#ifdef USE_DMALLOC
#include<dmalloc.h>
#endif

/*
 * The structure below defines the record for each circle item.
 */

typedef struct CircleItem  {
    Tk_Item header;		/* Generic stuff that's the same for all
				 * types.  MUST BE FIRST IN STRUCTURE. */
    Tk_Outline outline;		/* Outline structure */
    double origin[2];		/* Origin of circle */
    double radius;		/* Radius of circle */
    Tk_Tile fillTile;		/* Tile for filling circle. */
    Tk_Tile activeFillTile;	/* Tile for filling circle if state is active. */
    Tk_Tile disabledFillTile;	/* Tile for filling circle if state is disabled. */
    XColor *fillColor;		/* Color for filling circle. */
    XColor *activeFillColor;	/* Color for filling circle if state is active. */
    XColor *disabledFillColor;	/* Color for filling circle if state is disabled. */
    Pixmap fillStipple;		/* Stipple bitmap for filling item. */
    Pixmap activeFillStipple;	/* Stipple bitmap for filling item if state is active. */
    Pixmap disabledFillStipple;	/* Stipple bitmap for filling item if state is disabled. */
    GC fillGC;			/* Graphics context for filling item. */
} CircleItem;

/*
 * Information used for parsing configuration specs:
 */

static Tk_CustomOption stateOption = {
    (Tk_OptionParseProc *) Tk_StateParseProc,
    Tk_StatePrintProc, (ClientData) 2
};
static Tk_CustomOption tagsOption = {
    (Tk_OptionParseProc *) Tk_CanvasTagsParseProc,
    Tk_CanvasTagsPrintProc, (ClientData) NULL
};
static Tk_CustomOption dashOption = {
    (Tk_OptionParseProc *) Tk_CanvasDashParseProc,
    Tk_CanvasDashPrintProc, (ClientData) NULL
};
static Tk_CustomOption tileOption = {
    (Tk_OptionParseProc *) Tk_TileParseProc,
    Tk_TilePrintProc, (ClientData) NULL
};
static Tk_CustomOption pixelOption = {
    (Tk_OptionParseProc *) Tk_PixelParseProc,
    Tk_PixelPrintProc, (ClientData) NULL
};

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_CUSTOM, "-activedash", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(CircleItem, outline.activeDash),
	TK_CONFIG_NULL_OK, &dashOption},
    {TK_CONFIG_COLOR, "-activefill", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(CircleItem, activeFillColor),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-activeoutline", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(CircleItem, outline.activeColor),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-activeoutlinestipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(CircleItem, outline.activeStipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-activeoutlinetile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(CircleItem, outline.activeTile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_BITMAP, "-activestipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(CircleItem, activeFillStipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-activetile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(CircleItem, activeFillTile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_CUSTOM, "-activewidth", (char *) NULL, (char *) NULL,
	"0.0", Tk_Offset(CircleItem, outline.activeWidth),
	TK_CONFIG_DONT_SET_DEFAULT, &pixelOption},
    {TK_CONFIG_CUSTOM, "-dash", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(CircleItem, outline.dash),
	TK_CONFIG_NULL_OK, &dashOption},
    {TK_CONFIG_PIXELS, "-dashoffset", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(CircleItem, outline.offset),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-disableddash", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(CircleItem, outline.disabledDash),
	TK_CONFIG_NULL_OK, &dashOption},
    {TK_CONFIG_COLOR, "-disabledfill", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(CircleItem, disabledFillColor),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-disabledoutline", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(CircleItem, outline.disabledColor),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-disabledoutlinestipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(CircleItem, outline.disabledStipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-disabledoutlinetile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(CircleItem, outline.disabledTile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_BITMAP, "-disabledstipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(CircleItem, disabledFillStipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-disabledtile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(CircleItem, disabledFillTile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_CUSTOM, "-disabledwidth", (char *) NULL, (char *) NULL,
	"0.0", Tk_Offset(CircleItem, outline.disabledWidth),
	TK_CONFIG_DONT_SET_DEFAULT, &pixelOption},
    {TK_CONFIG_COLOR, "-fill", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(CircleItem, fillColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-outline", (char *) NULL, (char *) NULL,
	"black", Tk_Offset(CircleItem, outline.color), TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-outlinestipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(CircleItem, outline.stipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-outlinetile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(CircleItem, outline.tile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_CUSTOM, "-state", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(Tk_Item, state),TK_CONFIG_NULL_OK,
	&stateOption},
    {TK_CONFIG_BITMAP, "-stipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(CircleItem, fillStipple), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &tagsOption},
    {TK_CONFIG_CUSTOM, "-tile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(CircleItem, fillTile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_STRING, "-updatecommand", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(Tk_Item, updateCmd), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-width", (char *) NULL, (char *) NULL,
	"1.0", Tk_Offset(CircleItem, outline.width),
	TK_CONFIG_DONT_SET_DEFAULT, &pixelOption},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * Prototypes for procedures defined in this file:
 */

static void		ComputeCircleBbox _ANSI_ARGS_((Tk_Canvas canvas,
			    CircleItem *circlePtr));
static int		ConfigureCircle _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int argc,
			    char **argv, int flags));
static int		CreateCircle _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, struct Tk_Item *itemPtr,
			    int argc, char **argv));
static void		DeleteCircle _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, Display *display));
static void		DisplayCircle _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, Display *display, Drawable dst,
			    int x, int y, int width, int height));
static int		CircleToArea _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double *areaPtr));
static double		CircleToPoint _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double *pointPtr));
static int		CircleCoords _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int argc,
			    char **argv));
static int		CircleToPostscript _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int prepass));
#define ScaleCircle TkScaleCircle
void			ScaleCircle _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double originX, double originY,
			    double scaleX, double scaleY));
#define TranslateCircle TkTranslateCircle
void			TranslateCircle _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double deltaX, double deltaY));

/*
 * The structures below defines the circle item types
 * by means of procedures that can be invoked by generic item code.
 */

Tk_ItemType tkCircleType = {
    "circle",				/* name */
    sizeof(CircleItem),			/* itemSize */
    CreateCircle,			/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureCircle,			/* configureProc */
    CircleCoords,			/* coordProc */
    DeleteCircle,			/* deleteProc */
    DisplayCircle,			/* displayProc */
    TK_ITEM_VISITOR_SUPPORT,		/* flags */
    CircleToPoint,			/* pointProc */
    CircleToArea,			/* areaProc */
    CircleToPostscript,			/* postscriptProc */
    ScaleCircle,			/* scaleProc */
    TranslateCircle,			/* translateProc */
    (Tk_ItemIndexProc *) NULL,		/* indexProc */
    (Tk_ItemCursorProc *) NULL,		/* cursorProc */
    (Tk_ItemSelectionProc *) NULL,	/* selectionProc */
    (Tk_ItemInsertProc *) NULL,		/* insertProc */
    (Tk_ItemDCharsProc *) NULL,		/* dTextProc */
    (Tk_ItemType *) NULL,		/* nextPtr */
    (Tk_ItemBboxProc *) ComputeCircleBbox,/* bboxProc */
    Tk_Offset(Tk_VisitorType, visitLine), /* acceptProc */
    (Tk_ItemGetCoordProc *) NULL,	/* getCoordPtr */
    (Tk_ItemSetCoordProc *) NULL	/* setCoordPtr */
};

/*
 *--------------------------------------------------------------
 *
 * CreateCircle --
 *
 *	This procedure is invoked to create a new circle
 *	item in a canvas.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	interp->result;  in this case itemPtr is left uninitialized,
 *	so it can be safely freed by the caller.
 *
 * Side effects:
 *	A new circle item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateCircle(interp, canvas, itemPtr, argc, argv)
    Tcl_Interp *interp;			/* For error reporting. */
    Tk_Canvas canvas;			/* Canvas to hold new item. */
    Tk_Item *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing circle. */
{
    CircleItem *circlePtr = (CircleItem *) itemPtr;
    int i;

    if ((argc==1) || ((argc>1) && (argv[1][0] == '-')
	    && (argv[1][1] >= 'a') && (argv[1][1] <= 'z'))) {
	i = 1;
    } else {
	i = 3;
    }

    if (argc < i) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		Tk_PathName(Tk_CanvasTkwin(canvas)), " create ",
		itemPtr->typePtr->name, " x1 y1 radius ?options?\"",
		(char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Carry out initialization that is needed in order to clean
     * up after errors during the the remainder of this procedure.
     */

    Tk_CreateOutline(&(circlePtr->outline));
    circlePtr->fillTile = NULL;
    circlePtr->activeFillTile = NULL;
    circlePtr->disabledFillTile = NULL;
    circlePtr->fillColor = NULL;
    circlePtr->activeFillColor = NULL;
    circlePtr->disabledFillColor = NULL;
    circlePtr->fillStipple = None;
    circlePtr->activeFillStipple = None;
    circlePtr->disabledFillStipple = None;
    circlePtr->fillGC = None;

    /*
     * Process the arguments to fill in the item record.
     */

    if ((CircleCoords(interp, canvas, itemPtr, i, argv) != TCL_OK)) {
	goto error;
    }
    if (ConfigureCircle(interp, canvas, itemPtr, argc-i, argv+i, 0)
	    == TCL_OK) {
	return TCL_OK;
    }

    error:
    DeleteCircle(canvas, itemPtr, Tk_Display(Tk_CanvasTkwin(canvas)));
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * CircleCoords --
 *
 *	This procedure is invoked to process the "coords" widget
 *	command on circles.  See the user documentation
 *	for details on what it does.
 *
 * Results:
 *	Returns TCL_OK or TCL_ERROR, and sets interp->result.
 *
 * Side effects:
 *	The coordinates for the given item may be changed.
 *
 *--------------------------------------------------------------
 */

static int
CircleCoords(interp, canvas, itemPtr, argc, argv)
    Tcl_Interp *interp;			/* Used for error reporting. */
    Tk_Canvas canvas;			/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item whose coordinates are to be
					 * read or modified. */
    int argc;				/* Number of coordinates supplied in
					 * argv. */
    char **argv;			/* Array of coordinates: x1, y1,
					 * x2, y2, ... */
{
    CircleItem *circlePtr = (CircleItem *) itemPtr;
    char c0[TCL_DOUBLE_SPACE], c1[TCL_DOUBLE_SPACE];
    char c2[TCL_DOUBLE_SPACE], c3[TCL_DOUBLE_SPACE];

    if (argc == 0) {
	Tcl_PrintDouble(interp, circlePtr->origin[0], c0);
	Tcl_PrintDouble(interp, circlePtr->origin[1], c1);
	Tcl_PrintDouble(interp, circlePtr->radius, c2);
	Tcl_AppendResult(interp, c0, " ", c1, " ", c2, (char *) NULL);
    } else if ((argc == 1)||(argc == 3)) {
	char **largv = argv; 	if (argc==1) {
	    if (Tcl_SplitList(interp, argv[0], &argc, &largv) != TCL_OK) {
		if (largv != NULL) {
		    ckfree((char *) largv);
		}
		return TCL_ERROR;
	    } else if (argc != 3) {
		sprintf(c0,"%d",argc);
		Tcl_AppendResult(interp, "wrong # coordinates: expected 3, got ",
		c0, (char *) NULL);
		return TCL_ERROR;
	    }
	}
	if ((Tk_CanvasGetCoord(interp, canvas, largv[0],
 		    &circlePtr->origin[0]) != TCL_OK)
		|| (Tk_CanvasGetCoord(interp, canvas, largv[1],
		    &circlePtr->origin[1]) != TCL_OK)
		|| (Tk_CanvasGetCoord(interp, canvas, largv[2],
			&circlePtr->radius) != TCL_OK)) {
	    if (largv!=argv) ckfree((char *) largv);
	    return TCL_ERROR;
	}
	if (largv!=argv) ckfree((char *) largv);
	ComputeCircleBbox(canvas, circlePtr);
    } else {
	sprintf(interp->result,
		"wrong # coordinates: expected 0 or 3, got %d",
		argc);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureCircle --
 *
 *	This procedure is invoked to configure various aspects
 *	of a circle item, such as its border and
 *	background colors.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in interp->result.
 *
 * Side effects:
 *	Configuration information, such as colors and stipple
 *	patterns, may be set for itemPtr.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureCircle(interp, canvas, itemPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Canvas canvas;		/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Circle item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    CircleItem *circlePtr = (CircleItem *) itemPtr;
    XGCValues gcValues;
    GC newGC;
    unsigned long mask;
    Tk_Window tkwin;
    Tk_Tile tile;
    XColor *color;
    Pixmap stipple;
    Tk_State state;
    Pixmap pixmap;

    tkwin = Tk_CanvasTkwin(canvas);

    if (Tk_ConfigureWidget(interp, tkwin, configSpecs, argc, argv,
	    (char *) circlePtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    state = itemPtr->state;

    /*
     * A few of the options require additional processing, such as
     * graphics contexts.
     */

    if (circlePtr->outline.activeWidth > circlePtr->outline.width ||
	    circlePtr->outline.activeDash.number > 0 ||
	    circlePtr->outline.activeTile != NULL ||
	    circlePtr->outline.activeColor != NULL ||
	    circlePtr->outline.activeStipple != None ||
	    circlePtr->activeFillTile != NULL ||
	    circlePtr->activeFillColor != NULL ||
	    circlePtr->activeFillStipple != None) {
	itemPtr->redraw_flags |= TK_ITEM_STATE_DEPENDANT;
    } else {
	itemPtr->redraw_flags &= ~TK_ITEM_STATE_DEPENDANT;
    }

    mask = Tk_ConfigOutlineGC(&gcValues, canvas, itemPtr,
	     &(circlePtr->outline));
    if (mask) {
	gcValues.cap_style = CapProjecting;
	mask |= GCCapStyle;
	newGC = Tk_GetGC(tkwin, mask, &gcValues);
    } else {
	newGC = None;
    }
    if (circlePtr->outline.gc != None) {
	Tk_FreeGC(Tk_Display(tkwin), circlePtr->outline.gc);
    }
    circlePtr->outline.gc = newGC;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }
    tile = circlePtr->fillTile;
    color = circlePtr->fillColor;
    stipple = circlePtr->fillStipple;
    if (state==TK_STATE_HIDDEN) {
	ComputeCircleBbox(canvas, circlePtr);
	return TCL_OK;
    }
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (circlePtr->activeFillTile!=NULL) {
	    tile = circlePtr->activeFillTile;
	}
	if (circlePtr->activeFillColor!=NULL) {
	    color = circlePtr->activeFillColor;
	}
	if (circlePtr->activeFillStipple!=None) {
	    stipple = circlePtr->activeFillStipple;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (circlePtr->disabledFillTile!=NULL) {
	    tile = circlePtr->disabledFillTile;
	}
	if (circlePtr->disabledFillColor!=NULL) {
	    color = circlePtr->disabledFillColor;
	}
	if (circlePtr->disabledFillStipple!=None) {
	    stipple = circlePtr->disabledFillStipple;
	}
    }

    Tk_SetTileCanvasItem(circlePtr->fillTile, canvas, (Tk_Item *) NULL);
    Tk_SetTileCanvasItem(circlePtr->activeFillTile, canvas, (Tk_Item *) NULL);
    Tk_SetTileCanvasItem(circlePtr->disabledFillTile, canvas, (Tk_Item *) NULL);
    Tk_SetTileCanvasItem(tile, canvas, itemPtr);

    if ((pixmap = Tk_PixmapOfTile(tile)) != None) {
	gcValues.fill_style = FillTiled;
	gcValues.tile = pixmap;
	newGC = Tk_GetGC(tkwin, GCTile|GCFillStyle, &gcValues);
    } else if (color == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = color->pixel;
	if (stipple != None) {
	    gcValues.stipple = stipple;
	    gcValues.fill_style = FillStippled;
	    mask = GCForeground|GCStipple|GCFillStyle;
	} else {
	    mask = GCForeground;
	}
	newGC = Tk_GetGC(tkwin, mask, &gcValues);
    }
    if (circlePtr->fillGC != None) {
	Tk_FreeGC(Tk_Display(tkwin), circlePtr->fillGC);
    }
    circlePtr->fillGC = newGC;
    ComputeCircleBbox(canvas, circlePtr);

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DeleteCircle --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a circle item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteCircle(canvas, itemPtr, display)
    Tk_Canvas canvas;			/* Info about overall widget. */
    Tk_Item *itemPtr;			/* Item that is being deleted. */
    Display *display;			/* Display containing window for
					 * canvas. */
{
    CircleItem *circlePtr = (CircleItem *) itemPtr;

    Tk_DeleteOutline(display, &(circlePtr->outline));
    if (circlePtr->fillTile != NULL) {
	Tk_FreeTile(circlePtr->fillTile);
    }
    if (circlePtr->activeFillTile != NULL) {
	Tk_FreeTile(circlePtr->activeFillTile);
    }
    if (circlePtr->disabledFillTile != NULL) {
	Tk_FreeTile(circlePtr->disabledFillTile);
    }
    if (circlePtr->fillColor != NULL) {
	Tk_FreeColor(circlePtr->fillColor);
    }
    if (circlePtr->activeFillColor != NULL) {
	Tk_FreeColor(circlePtr->activeFillColor);
    }
    if (circlePtr->disabledFillColor != NULL) {
	Tk_FreeColor(circlePtr->disabledFillColor);
    }
    if (circlePtr->fillStipple != None) {
	Tk_FreeBitmap(display, circlePtr->fillStipple);
    }
    if (circlePtr->activeFillStipple != None) {
	Tk_FreeBitmap(display, circlePtr->activeFillStipple);
    }
    if (circlePtr->disabledFillStipple != None) {
	Tk_FreeBitmap(display, circlePtr->disabledFillStipple);
    }
    if (circlePtr->fillGC != None) {
	Tk_FreeGC(display, circlePtr->fillGC);
    }
}

/*
 *--------------------------------------------------------------
 *
 * ComputeCircleBbox --
 *
 *	This procedure is invoked to compute the bounding box of
 *	all the pixels that may be drawn as part of a circle.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The fields x1, y1, x2, and y2 are updated in the header
 *	for itemPtr.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static void
ComputeCircleBbox(canvas, circlePtr)
    Tk_Canvas canvas;			/* Canvas that contains item. */
    CircleItem *circlePtr;		/* Item whose bbox is to be
					 * recomputed. */
{
    int bloat, tmp, width;
    double dtmp;
    Tk_State state = circlePtr->header.state;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    width = circlePtr->outline.width;
    if (state==TK_STATE_HIDDEN) {
	circlePtr->header.x1 = circlePtr->header.y1 =
	circlePtr->header.x2 = circlePtr->header.y2 = -1;
	return;
    }
    if (((TkCanvas *)canvas)->currentItemPtr == (Tk_Item *)circlePtr) {
	if (circlePtr->outline.activeWidth>width) {
	    width = circlePtr->outline.activeWidth;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (circlePtr->outline.disabledWidth>0) {
	    width = circlePtr->outline.disabledWidth;
	}
    }

    /*
     * Add in the width of the outline
     */

    if (circlePtr->outline.gc == None) {
	bloat = 0;
    } else {
	bloat = (width+1)/2;
    }

    /*
     * Special note:  the circle is always drawn at least 1x1 in
     * size, so round up the upper coordinates to be at least 1 unit
     * greater than the lower ones.
     */

    circlePtr->header.x1= circlePtr->origin[0] - circlePtr->radius - bloat - .5;
    circlePtr->header.y1= circlePtr->origin[1] - circlePtr->radius - bloat - .5;
    circlePtr->header.x2= circlePtr->origin[0] + circlePtr->radius + bloat + .5;
    circlePtr->header.y2= circlePtr->origin[1] + circlePtr->radius + bloat + .5;
}

/*
 *--------------------------------------------------------------
 *
 * DisplayCircle --
 *
 *	This procedure is invoked to draw a circle item in a given drawable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in drawable using the transformation
 *	information in canvas.
 *
 *--------------------------------------------------------------
 */

static void
DisplayCircle(canvas, itemPtr, display, drawable, x, y, width, height)
    Tk_Canvas canvas;			/* Canvas that contains item. */
    Tk_Item *itemPtr;			/* Item to be displayed. */
    Display *display;			/* Display on which to draw item. */
    Drawable drawable;			/* Pixmap or window in which to draw
					 * item. */
    int x, y, width, height;		/* Describes region of canvas that
					 * must be redisplayed (not used). */
{
    CircleItem *circlePtr = (CircleItem *) itemPtr;
    short x1, y1, x2, y2;
    Tk_Tile fillTile;
    Pixmap fillStipple;
    Tk_State state;

    /*
     * Compute the screen coordinates of the bounding box for the item.
     * Make sure that the bbox is at least one pixel large, since some
     * X servers will die if it isn't.
     */

    Tk_CanvasDrawableCoords(canvas, circlePtr->origin[0] - circlePtr->radius, 
	    circlePtr->origin[1] - circlePtr->radius,
	    &x1, &y1);
    Tk_CanvasDrawableCoords(canvas, circlePtr->origin[0] + circlePtr->radius, 
	    circlePtr->origin[1] + circlePtr->radius,
	    &x2, &y2);
    /*
    if (x2 <= x1) {
	x2 = x1+1;
    }
    if (y2 <= y1) {
	y2 = y1+1;
    }
    */

    /*
     * Display filled part first (if wanted), then outline.  If we're
     * stippling, then modify the stipple offset in the GC.  Be sure to
     * reset the offset when done, since the GC is supposed to be
     * read-only.
     */

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }
    fillTile = circlePtr->fillTile;
    fillStipple = circlePtr->fillStipple;
    if (((TkCanvas *)canvas)->currentItemPtr == (Tk_Item *)circlePtr) {
	if (circlePtr->activeFillTile!=NULL) {
	    fillTile = circlePtr->activeFillTile;
	}
	if (circlePtr->activeFillStipple!=None) {
	    fillStipple = circlePtr->activeFillStipple;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (circlePtr->disabledFillTile!=NULL) {
	    fillTile = circlePtr->disabledFillTile;
	}
	if (circlePtr->disabledFillStipple!=None) {
	    fillStipple = circlePtr->disabledFillStipple;
	}
    }

    if (circlePtr->fillGC != None) {
	if ((fillTile != None) || (fillStipple != None)) {
	    Tk_CanvasSetStippleOrigin(canvas, circlePtr->fillGC);
	}
	XFillArc(display, drawable, circlePtr->fillGC,
		x1, y1, (unsigned) (x2-x1), (unsigned) (y2-y1),
		0, 360*64);
	if ((fillTile != None) || (fillStipple != None)) {
	    XSetTSOrigin(display, circlePtr->fillGC, 0, 0);
	}
    }
    if (circlePtr->outline.gc != None) {
	Tk_ChangeOutlineGC(canvas, itemPtr, &(circlePtr->outline));
	XDrawArc(display, drawable, circlePtr->outline.gc,
		x1, y1, (unsigned) (x2-x1), (unsigned) (y2-y1), 0, 360*64);
	Tk_ResetOutlineGC(canvas, itemPtr, &(circlePtr->outline));
    }
}

/*
 *--------------------------------------------------------------
 *
 * CircleToPoint --
 *
 *	Computes the distance from a given point to a given
 *	circle, in canvas units.
 *
 * Results:
 *	The return value is 0 if the point whose x and y coordinates
 *	are coordPtr[0] and coordPtr[1] is inside the circle.  If the
 *	point isn't inside the circle then the return value is the
 *	distance from the point to the circle.  If itemPtr is filled,
 *	then anywhere in the interior is considered "inside"; if
 *	itemPtr isn't filled, then "inside" means only the area
 *	occupied by the outline.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static double
CircleToPoint(canvas, itemPtr, pointPtr)
    Tk_Canvas canvas;		/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against point. */
    double *pointPtr;		/* Pointer to x and y coordinates. */
{
    CircleItem *circlePtr = (CircleItem *) itemPtr;
    double width;
    int filled;
    double bbox[4];
    Tk_State state = itemPtr->state;

    bbox[0] = circlePtr->origin[0] - circlePtr->radius;
    bbox[1] = circlePtr->origin[1] - circlePtr->radius;
    bbox[2] = circlePtr->origin[0] + circlePtr->radius;
    bbox[3] = circlePtr->origin[1] + circlePtr->radius;
    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    width = (double) circlePtr->outline.width;
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (circlePtr->outline.activeWidth>width) {
	    width = (double) circlePtr->outline.activeWidth;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (circlePtr->outline.disabledWidth>0) {
	    width = (double) circlePtr->outline.disabledWidth;
	}
    }


    filled = circlePtr->fillGC != None;
    if (circlePtr->outline.gc == None) {
	width = 0.0;
	filled = 1;
    }
    filled = 1;
    return TkOvalToPoint(bbox, width, filled, pointPtr);
}

/*
 *--------------------------------------------------------------
 *
 * CircleToArea --
 *
 *	This procedure is called to determine whether an item
 *	lies entirely inside, entirely outside, or overlapping
 *	a given rectangular area.
 *
 * Results:
 *	-1 is returned if the item is entirely outside the area
 *	given by rectPtr, 0 if it overlaps, and 1 if it is entirely
 *	inside the given area.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
CircleToArea(canvas, itemPtr, areaPtr)
    Tk_Canvas canvas;		/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against circle. */
    double *areaPtr;		/* Pointer to array of four coordinates
				 * (x1, y1, x2, y2) describing rectangular
				 * area.  */
{
    CircleItem *circlePtr = (CircleItem *) itemPtr;
    double circle[4], halfWidth;
    int result;
    int width;
    Tk_State state = itemPtr->state;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    width = circlePtr->outline.width;
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (circlePtr->outline.activeWidth>width) {
	    width = circlePtr->outline.activeWidth;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (circlePtr->outline.disabledWidth>0) {
	    width = circlePtr->outline.disabledWidth;
	}
    }

    /*
     * Expand the circle to include the width of the outline, if any.
     */

    halfWidth = width/2.0;
    if (circlePtr->outline.gc == None) {
	halfWidth = 0.0;
    }
    circle[0] = circlePtr->origin[0] - circlePtr->radius - halfWidth;
    circle[1] = circlePtr->origin[1] - circlePtr->radius - halfWidth;
    circle[2] = circlePtr->origin[0] + circlePtr->radius + halfWidth;
    circle[3] = circlePtr->origin[1] + circlePtr->radius + halfWidth;

    result = TkOvalToArea(circle, areaPtr);

    /*
     * If the rectangle appears to overlap the circle and the circle
     * isn't filled, do one more check to see if perhaps all four
     * of the rectangle's corners are totally inside the circle's
     * unfilled center, in which case we should return "outside".
     */

    if ((result == 0) && (circlePtr->outline.gc != None)
	    && (circlePtr->fillGC == None)) {
	double centerX, centerY, width, height;
	double xDelta1, yDelta1, xDelta2, yDelta2;

	centerX = circlePtr->origin[0];
	centerY = circlePtr->origin[1];
	width = circlePtr->radius;
	height = circlePtr->radius;
	xDelta1 = (areaPtr[0] - centerX)/width;
	xDelta1 *= xDelta1;
	yDelta1 = (areaPtr[1] - centerY)/height;
	yDelta1 *= yDelta1;
	xDelta2 = (areaPtr[2] - centerX)/width;
	xDelta2 *= xDelta2;
	yDelta2 = (areaPtr[3] - centerY)/height;
	yDelta2 *= yDelta2;
	if (((xDelta1 + yDelta1) < 1.0)
		&& ((xDelta1 + yDelta2) < 1.0)
		&& ((xDelta2 + yDelta1) < 1.0)
		&& ((xDelta2 + yDelta2) < 1.0)) {
	    return -1;
	}
    }
    return result;
}

/*
 *--------------------------------------------------------------
 *
 * ScaleCircle --
 *
 *	This procedure is invoked to rescale a circle item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The circle referred to by itemPtr is rescaled
 *	so that the following transformation is applied to all
 *	point coordinates:
 *		x' = originX + scaleX*(x-originX)
 *		y' = originY + scaleY*(y-originY)
 *
 *--------------------------------------------------------------
 */

void
ScaleCircle(canvas, itemPtr, originX, originY, scaleX, scaleY)
    Tk_Canvas canvas;			/* Canvas containing circle. */
    Tk_Item *itemPtr;			/* Circle to be scaled. */
    double originX, originY;		/* Origin about which to scale rect. */
    double scaleX;			/* Amount to scale in X direction. */
    double scaleY;			/* Amount to scale in Y direction. */
{
    CircleItem *circlePtr = (CircleItem *) itemPtr;


    circlePtr->origin[0] = originX + scaleX*(circlePtr->origin[0] - originX);
    circlePtr->origin[1] = originY + scaleY*(circlePtr->origin[1] - originY);
    circlePtr->radius = circlePtr->radius * (scaleX + scaleY) / 2.0;
    ComputeCircleBbox(canvas, circlePtr);
}

/*
 *--------------------------------------------------------------
 *
 * TranslateCircle --
 *
 *	This procedure is called to move a circle by a
 *	given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The position of the circle is offset by
 *	(xDelta, yDelta), and the bounding box is updated in the
 *	generic part of the item structure.
 *
 *--------------------------------------------------------------
 */

void
TranslateCircle(canvas, itemPtr, deltaX, deltaY)
    Tk_Canvas canvas;			/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item that is being moved. */
    double deltaX, deltaY;		/* Amount by which item is to be
					 * moved. */
{
    CircleItem *circlePtr = (CircleItem *) itemPtr;

    circlePtr->origin[0] += deltaX;
    circlePtr->origin[1] += deltaY;
    ComputeCircleBbox(canvas, circlePtr);
}

/*
 *--------------------------------------------------------------
 *
 * CircleToPostscript --
 *
 *	This procedure is called to generate Postscript for
 *	circle items.
 *
 * Results:
 *	The return value is a standard Tcl result.  If an error
 *	occurs in generating Postscript then an error message is
 *	left in interp->result, replacing whatever used to be there.
 *	If no error occurs, then Postscript for the circle is
 *	appended to the result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
CircleToPostscript(interp, canvas, itemPtr, prepass)
    Tcl_Interp *interp;			/* Interpreter for error reporting. */
    Tk_Canvas canvas;			/* Information about overall canvas. */
    Tk_Item *itemPtr;			/* Item for which Postscript is
					 * wanted. */
    int prepass;			/* 1 means this is a prepass to
					 * collect font information;  0 means
					 * final Postscript is being created. */
{
    char pathCmd[500], string[100];
    CircleItem *circlePtr = (CircleItem *) itemPtr;
    double y1, y2;
    int width;
    Tk_Tile tile;
    XColor *color;
    Pixmap stipple;
    Tk_Tile fillTile;
    XColor *fillColor;
    Pixmap fillStipple;
    Tk_State state = itemPtr->state;

    y1 = Tk_CanvasPsY(canvas, circlePtr->origin[1] - circlePtr->radius);
    y2 = Tk_CanvasPsY(canvas, circlePtr->origin[1] + circlePtr->radius);

    /*
     * Generate a string that creates a path for the circle.
     * This is the only part of the procedure's code that is type-
     * specific.
     */


    /*
    sprintf(pathCmd, "matrix currentmatrix\n%.15g %.15g translate %.15g %.15g scale 1 0 moveto 0 0 1 0 360 arc\nsetmatrix\n",
	    circlePtr->origin[0], circlePtr->origin[1],
	    circlePtr->radius, circlePtr->radius);
    */
    sprintf(pathCmd, "matrix currentmatrix\n%.15g %.15g translate %.15g %.15g scale 1 0 moveto 0 0 1 0 360 arc\nsetmatrix\n",
	    circlePtr->origin[0], (y1 + y2) / 2,
	    circlePtr->radius, circlePtr->radius);

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }
    tile = circlePtr->outline.tile;
    color = circlePtr->outline.color;
    fillTile = circlePtr->fillTile;
    fillColor = circlePtr->fillColor;
    fillStipple = circlePtr->fillStipple;
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (circlePtr->outline.activeTile!=NULL) {
	    tile = circlePtr->outline.activeTile;
	}
	if (circlePtr->outline.activeColor!=NULL) {
	    color = circlePtr->outline.activeColor;
	}
	if (circlePtr->activeFillTile!=NULL) {
	    fillTile = circlePtr->activeFillTile;
	}
	if (circlePtr->activeFillColor!=NULL) {
	    fillColor = circlePtr->activeFillColor;
	}
	if (circlePtr->activeFillStipple!=None) {
	    fillStipple = circlePtr->activeFillStipple;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (circlePtr->outline.disabledTile!=NULL) {
	    tile = circlePtr->outline.disabledTile;
	}
	if (circlePtr->outline.disabledColor!=NULL) {
	    color = circlePtr->outline.disabledColor;
	}
	if (circlePtr->disabledFillTile!=NULL) {
	    fillTile = circlePtr->disabledFillTile;
	}
	if (circlePtr->disabledFillColor!=NULL) {
	    fillColor = circlePtr->disabledFillColor;
	}
	if (circlePtr->disabledFillStipple!=None) {
	    fillStipple = circlePtr->disabledFillStipple;
	}
    }

    /*
     * First draw the filled area of the rectangle.
     */

    if (fillColor != NULL) {
	Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	if (Tk_CanvasPsColor(interp, canvas, fillColor)
		!= TCL_OK) {
	    return TCL_ERROR;
	}
	if (fillStipple != None) {
	    Tcl_AppendResult(interp, "clip ", (char *) NULL);
	    if (Tk_CanvasPsStipple(interp, canvas, fillStipple)
		    != TCL_OK) {
		return TCL_ERROR;
	    }
	    if (color != NULL) {
		Tcl_AppendResult(interp, "grestore gsave\n", (char *) NULL);
	    }
	} else {
	    Tcl_AppendResult(interp, "fill\n", (char *) NULL);
	}
    }

    /*
     * Now draw the outline, if there is one.
     */

    if (color != NULL) {
	Tcl_AppendResult(interp, pathCmd, "0 setlinejoin 2 setlinecap\n",
		(char *) NULL);
	if (Tk_CanvasPsOutline(canvas, itemPtr,
		&(circlePtr->outline))!= TCL_OK) {
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}
