/* 
 * tkDial.c --
 *
 *	This module implements a dial widget for the Tk toolkit.
 *	A dial displays a dial that can be turned to change a
 *	value;  it also displays numeric labels and a textual label,
 *	if desired. The programmer's interface is very similar to the
 *	scale widget of the Tk toolkit.
 *
 *	The code is written for Tk8
 *
 * Copyright (c) 1998-1999 Jeffrey Hobbs, jeff at hobbs org
 *
 * See the file "license.terms" for information on usage
 * and redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * RCS: @(#) $Id: tkDial.c,v 1.11 2005/07/12 23:17:07 hobbs Exp $
 */

#include "pkg.h"

/* Oops, no rint on Windows */
#if defined(_WIN32)
#define rint(num)	ROUND(num)
#endif

/* Defaults for dials: */

#define DEF_DIAL_ACTIVE_BG_COLOR	ACTIVE_BG
#define DEF_DIAL_ACTIVE_BG_MONO		WHITE
#define DEF_DIAL_BG_COLOR		NORMAL_BG
#define DEF_DIAL_BG_MONO		WHITE
#define DEF_DIAL_BIG_INCREMENT		"0"
#define DEF_DIAL_BORDER_WIDTH		"2"
#define DEF_DIAL_COMMAND		""
#define DEF_DIAL_CONSTRAIN_VALUE	"0"
#define DEF_DIAL_CURSOR			""
#define DEF_DIAL_DIGITS			"0"
#define DEF_DIAL_FONT			DEF_FONT
#define DEF_DIAL_FG_COLOR		BLACK
#define DEF_DIAL_FG_MONO		BLACK
#define DEF_DIAL_FROM			"0"
#define DEF_DIAL_LABEL			""
#define DEF_DIAL_RELIEF			"flat"
#define DEF_DIAL_REPEAT_DELAY		"400"
#define DEF_DIAL_REPEAT_INTERVAL	"100"
#define DEF_DIAL_RESOLUTION		"1.0"
#define DEF_DIAL_TICK_COLOR		BLACK
#define DEF_DIAL_TICK_MONO		BLACK
#define DEF_DIAL_MODE			"circle"
#define DEF_DIAL_NEEDLE_COLOR		BLACK
#define DEF_DIAL_NEEDLE_MONO		BLACK
#define DEF_DIAL_NEEDLE_TYPE		"line"
#define DEF_DIAL_SHOW_VALUE		"1"
#define DEF_DIAL_STATE			"normal"
#define DEF_DIAL_TAKE_FOCUS		(char *) NULL
#define DEF_DIAL_MIN_TICK_INTERVAL	"0.0"
#define DEF_DIAL_TICK_INTERVAL		"0.0"
#define DEF_DIAL_TICK_WIDTH		"0"
#define DEF_DIAL_TO			"100.0"
#define DEF_DIAL_VARIABLE		""
#define DEF_DIAL_DIAL_COLOR		NORMAL_BG
#define DEF_DIAL_DIAL_MONO		BLACK
#define DEF_DIAL_DIAL_BORDER_WIDTH	"3"
#define DEF_DIAL_DIAL_RELIEF		"raised"
#define DEF_DIAL_PAD			"2"
#define DEF_DIAL_BEGIN_ANGLE		"-150"
#define DEF_DIAL_END_ANGLE		"150"
#define DEF_DIAL_SHOW_TAGS		"value"
#define DEF_DIAL_WIDTH			"50"
#define DEF_DIAL_HEIGHT			"60"

#define MIN_TICK_LENGTH 0.15	/* Length of minor tick (fract. of radius) */
#define TICK_LENGTH	0.25	/* Length of major tick (fract. of radius) */
#define NEEDLE_WIDTH	0.10	/* Width of needle (fraction of radius) */
#define NEEDLE_POINTS	4	/* Nr of points for the needle triangle */
#define DIAL_POINTS	24	/* Nr of points for the dial circle */
#define MAX_TICKS	500	/* Max nr of ticks to draw. */
#define M_DEGREE	23040

#define	deg2rad(d)	(M_PI*(double)(d)/180.0)
#define	rad2deg(r)	((double)(r)*180.0/M_PI)

#ifdef FUTURE_DIAL_TAGS
typedef struct TkDialElt {
    struct TkDialElt *next, *prev;
} TkDialElt;

typedef struct {
    char *label;
    int orient;
    int constrain;
    int tick;
    XColor *fg;
} TkDialTag;
#endif

/*
 * A data structure of the following type is kept for each dial
 * widget managed by this file:
 */

typedef struct {
    Tk_Window tkwin;		/* Window that embodies the dial.  NULL
				 * means that the window has been destroyed
				 * but the data structures haven't yet been
				 * cleaned up.*/
    Display *display;		/* Display containing widget.  Used, among
				 * other things, so that resources can be
				 * freed even after tkwin has gone away. */
    Tcl_Interp *interp;		/* Interpreter associated with dial. */
    Tcl_Command widgetCmd;	/* Token for dial's widget command. */
    Tk_OptionTable optionTable;	/* Table that defines configuration options
				 * available for this widget. */

    /* Border stuff */
    int borderWidth;		/* Width of 3-D border around window. */
    Tk_3DBorder bgBorder;	/* Used for drawing background areas. */
    Tk_Cursor cursor;		/* Current cursor for window, or None. */
    int highlightWidth;		/* Width in pixels of highlight to draw
				 * around widget when it has the focus.
				 * <= 0 means don't draw a highlight. */
    XColor *highlightBgColorPtr;/* Color for drawing traversal highlight
				 * area when highlight is off. */
    XColor *highlightColorPtr;	/* Color for drawing traversal highlight. */
    int inset;			/* Total width of all borders, including
				 * traversal highlight and 3-D border.
				 * Indicates how much interior stuff must
				 * be offset from outside edges to leave
				 * room for borders. */
    int relief;			/* Indicates whether window as a whole is
				 * raised, sunken, or flat. */

    int mode;			/* circle or ellipse modes */
    int reqRadius;		/* Desired radius of central dial */
    int radX, radY;		/* actual radii based on geometry */
    int beginAngle;		/* Desired starting angle in degrees */
    int endAngle;		/* Desired end angle of scale in degrees */
    double fromValue;		/* Value corresponding to left/top of dial */
    double toValue;		/* Value corresponding to right/bottom
				 * of dial */
    double minTickInterval;	/* Distance between minor tick marks;
				 * 0 means don't display any minor ticks. */
    double tickInterval;	/* Distance between major ticks; 0 means
				 * don't display any major ticks. */
    double resolution;		/* If > 0, all values are rounded to an
				 * even multiple of this value. */
    int digits;			/* Number of significant digits to print
				 * in values.  0 means we get to choose the
				 * number based on resolution and/or the
				 * range of the dial. */
    char valueFormat[10];	/* Sprintf conversion specifier computed from
				 * digits and other information; used for
				 * the value. */
    char tagFormat[10];		/* Sprintf conversion specifier used for the
				 * values next to the major ticks. */
    double bigIncrement;	/* Amount to use for large increments to
				 * dial value (0 means we pick a value). */
    char *command;		/* Command prefix to use when invoking Tcl
				 * commands because the dial value
				 * changed. NULL means don't invoke commands.
				 * Malloc'ed. */
    double value;		/* Current value of dial. */
    char *varName;		/* Name of variable (malloc'ed) or NULL.
				 * If non-NULL, dial's value tracks
				 * the contents of this variable and
				 * vice versa. */
    int repeatDelay;		/* How long to wait before auto-repeating
				 * on scrolling actions (in ms). */
    int repeatInterval;		/* Interval between autorepeats (in ms). */
    char *label;		/* Label to display above the dial;
				 * NULL means don't display a label.
				 * Malloc'ed. */
    int state;			/* Normal or disabled.  Value cannot be
				 * changed when dial is disabled. */
    int showValue;		/* Non-zero means to display the dial value
				 * below or to the left of the slider;  zero
				 * means don't display the value. */
    int constrain;		/* Whether or not to constrain value to
				 * the given from-to values */
    int showTags;		/* Non-zero means to display scale values along
				 * the major tick marks. */
    Tcl_HashTable *tags;	/* String tags to associate to numeric tags */
    int longestTag;		/* width required by longest tag */
    char *orient;		/* reserved for future use */

    /* Information used when displaying widget: */
    int dialBorderWidth;	/* Width of 3-D border around the dial. */
    Tk_3DBorder dialBorder;	/* For drawing the dial */
    Tk_3DBorder activeBorder;	/* For drawing the dial when active. */
    XPoint dial[DIAL_POINTS];	/* Points that make up the dial polygon. */
    XPoint needle[NEEDLE_POINTS];/* Points that make up the needle polygon. */
    int needleType;		/* type of needle to display */

    XColor *tickColorPtr;	/* Color of ticks. */
    GC tickGC;			/* GC for drawing ticks. */
    XColor *needleColorPtr;	/* Color of the needle. */
    GC needleGC;		/* GC for drawing the needle on the dial. */
    Tk_Font tkfont;		/* Information about text font, or NULL. */
    XColor *textColorPtr;	/* Color for drawing text. */
    GC textGC;			/* GC for drawing text in normal mode. */
    int dialRelief;		/* Indicates whether dial is raised, sunken,
				 * or flat */

    int reqWidth;		/* */
    int reqHeight;		

    /*
     * Layout information for dials, assuming that window
     * gets the size it requested:
     */
    int centerX;		/* Location of the center of the dial */
    int centerY;		/* in pixels. */
    int tickWidth;		/* width of the ticks */
    int pad;			/* padding information */

    /* Miscellaneous information: */

    char *takeFocus;		/* Value of -takefocus option;  not used in
				 * the C code, but used by keyboard traversal
				 * scripts.  Malloc'ed, but may be NULL. */
    int flags;			/* Various flags;  see below for
				 * definitions. */
} TkDial;

/*
 * Flag bits for dials:
 *
 * REDRAW_DIAL -		1 means needle on dial (and numerical readout)
 *				needs to be redrawn.
 * REDRAW_OTHER -		1 means other stuff besides needle and value
 *				need to be redrawn.
 * REDRAW_ALL -			1 means the entire widget needs to be redrawn.
 * ACTIVE -			1 means the widget is active (the mouse is
 *				in its window).
 * BUTTON_PRESSED -		1 means a button press is in progress, so
 *				needle should be draggable.
 * INVOKE_COMMAND -		1 means the dial's command needs to be
 *				invoked during the next redisplay (the
 *				value of the dial has changed since the
 *				last time the command was invoked).
 * SETTING_VAR -		1 means that the associated variable is
 *				being set by us, so there's no need for
 *				DialVarProc to do anything.
 * NEVER_SET -			1 means that the dial's value has never
 *				been set before (so must invoke -command and
 *				set associated variable even if the value
 *				doesn't appear to have changed).
 * GOT_FOCUS -			1 means that the focus is currently in
 *				this widget.
 */
#define REDRAW_DIAL		1
#define REDRAW_OTHER		2
#define REDRAW_ALL		3
#define ACTIVE			4
#define BUTTON_PRESSED		8
#define INVOKE_COMMAND		0x10
#define SETTING_VAR		0x20
#define NEVER_SET		0x40
#define GOT_FOCUS		0x80

/*
 * Symbolic values for the active parts of a dial.  These are
 * the values that may be returned by the DialElement procedure.
 */
#define OTHER		0
#define DIAL		1
#define LEFT		2
#define RIGHT		3

/*
 * The following enum is used to define a type for the -state option
 * of the widget.  These values are used as indices into the 
 * string table below.
 */

enum state {
    STATE_ACTIVE, STATE_DISABLED, STATE_NORMAL
};
static char *stateStrings[] = {
    "active", "disabled", "normal", (char *) NULL
};

enum needle {
    NEEDLE_ARC, NEEDLE_LINE, NEEDLE_TRIANGLE
};
static char *needleStrings[]= {
    "arc", "line", "triangle", (char *) NULL
};

enum mode {
    MODE_CIRCLE, MODE_ELLIPSE
};
static char *modeStrings[]= {
    "circle", "ellipse", (char *) NULL
};

enum showtags {
    SHOWTAGS_BOTH, SHOWTAGS_LABEL, SHOWTAGS_NONE, SHOWTAGS_VALUE
};
static char *showtagsStrings[]= {
    "both", "label", "none", "value", (char*) NULL
};

/*
 * Information used for argv parsing.
 */

static Tk_OptionSpec optionSpecs[] = {
    {TK_OPTION_BORDER, "-activebackground", "activeBackground", "Foreground",
     DEF_DIAL_ACTIVE_BG_COLOR, -1, Tk_Offset(TkDial, activeBorder),
     0, (ClientData) DEF_DIAL_ACTIVE_BG_MONO, 0},
    {TK_OPTION_BORDER, "-background", "background", "Background",
     DEF_DIAL_BG_COLOR, -1, Tk_Offset(TkDial, bgBorder),
     0, (ClientData) DEF_DIAL_BG_MONO, 0},
    {TK_OPTION_INT, "-beginangle", "beginAngle", "BeginAngle",
     DEF_DIAL_BEGIN_ANGLE, -1, Tk_Offset(TkDial, beginAngle), 0, 0, 0},
    {TK_OPTION_DOUBLE, "-bigincrement", "bigIncrement", "BigIncrement",
     DEF_DIAL_BIG_INCREMENT, -1, Tk_Offset(TkDial, bigIncrement), 0, 0, 0},
    {TK_OPTION_SYNONYM, "-bd", (char *) NULL, (char *) NULL,
     (char *) NULL, 0, -1, 0, (ClientData) "-borderwidth", 0},
    {TK_OPTION_SYNONYM, "-bg", (char *) NULL, (char *) NULL,
     (char *) NULL, 0, -1, 0, (ClientData) "-background", 0},
    {TK_OPTION_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
     DEF_DIAL_BORDER_WIDTH, -1, Tk_Offset(TkDial, borderWidth), 0, 0, 0},
    {TK_OPTION_STRING, "-command", "command", "Command",
     DEF_DIAL_COMMAND, -1, Tk_Offset(TkDial, command),
     TK_CONFIG_NULL_OK, 0, 0},
    {TK_OPTION_BOOLEAN, "-constrainvalue", "constrainValue", "ConstrainValue",
     DEF_DIAL_CONSTRAIN_VALUE, -1, Tk_Offset(TkDial, constrain), 0, 0, 0},
    {TK_OPTION_CURSOR, "-cursor", "cursor", "Cursor",
     DEF_DIAL_CURSOR, -1, Tk_Offset(TkDial, cursor), TK_CONFIG_NULL_OK, 0, 0},
    {TK_OPTION_PIXELS, "-dialborderwidth", "dialBorderWidth",
     "DialBorderWidth", DEF_DIAL_DIAL_BORDER_WIDTH, -1,
     Tk_Offset(TkDial, dialBorderWidth), 0, 0, 0},
    {TK_OPTION_BORDER, "-dialcolor", "dialColor", "DialColor",
     DEF_DIAL_DIAL_COLOR, -1, Tk_Offset(TkDial, dialBorder),
     0, (ClientData) DEF_DIAL_DIAL_MONO, 0},
    {TK_OPTION_RELIEF, "-dialrelief", "dialRelief", "DialRelief",
     DEF_DIAL_DIAL_RELIEF, -1, Tk_Offset(TkDial, dialRelief), 0, 0, 0},
    {TK_OPTION_INT, "-digits", "digits", "Digits",
     DEF_DIAL_DIGITS, -1, Tk_Offset(TkDial, digits), 0, 0, 0},
    {TK_OPTION_INT, "-endangle", "endAngle", "EndAngle",
     DEF_DIAL_END_ANGLE, -1, Tk_Offset(TkDial, endAngle), 0, 0, 0},
    {TK_OPTION_SYNONYM, "-fg", (char *) NULL, (char *) NULL,
     (char *) NULL, 0, -1, 0, (ClientData) "-foreground", 0},
    {TK_OPTION_FONT, "-font", "font", "Font",
     DEF_DIAL_FONT, -1, Tk_Offset(TkDial, tkfont), 0, 0, 0},
    {TK_OPTION_COLOR, "-foreground", "foreground", "Foreground",
     DEF_DIAL_FG_COLOR, -1, Tk_Offset(TkDial, textColorPtr),
     0, (ClientData) DEF_DIAL_FG_MONO, 0},
    {TK_OPTION_DOUBLE, "-from", "from", "From",
     DEF_DIAL_FROM, -1, Tk_Offset(TkDial, fromValue), 0, 0, 0},
    {TK_OPTION_PIXELS, "-height", "height", "Height",
     DEF_DIAL_HEIGHT, -1, Tk_Offset(TkDial, reqHeight), 0, 0, 0},
    {TK_OPTION_COLOR, "-highlightbackground", "highlightBackground",
     "HighlightBackground", HIGHLIGHT_BG, -1,
     Tk_Offset(TkDial, highlightBgColorPtr), 0, 0, 0},
    {TK_OPTION_COLOR, "-highlightcolor", "highlightColor", "HighlightColor",
     HIGHLIGHT, -1, Tk_Offset(TkDial, highlightColorPtr), 0, 0, 0},
    {TK_OPTION_PIXELS, "-highlightthickness", "highlightThickness",
     "HighlightThickness",
     HIGHLIGHT_WIDTH, -1, Tk_Offset(TkDial, highlightWidth), 0, 0, 0},
    {TK_OPTION_STRING, "-label", "label", "Label",
     DEF_DIAL_LABEL, -1, Tk_Offset(TkDial, label), TK_CONFIG_NULL_OK, 0, 0},
    {TK_OPTION_DOUBLE, "-minortickinterval", "minorTickInterval", "TickInterval",
     DEF_DIAL_MIN_TICK_INTERVAL, -1, Tk_Offset(TkDial, minTickInterval),
     0, 0, 0},
    {TK_OPTION_STRING_TABLE, "-mode", "mode", "Mode",
     DEF_DIAL_MODE, -1, Tk_Offset(TkDial, mode),
     0, (ClientData) modeStrings, 0},
    {TK_OPTION_COLOR, "-needlecolor", "needleColor", "NeedleColor",
     DEF_DIAL_NEEDLE_COLOR, -1, Tk_Offset(TkDial, needleColorPtr),
     0, (ClientData) DEF_DIAL_NEEDLE_MONO, 0},
    {TK_OPTION_STRING_TABLE, "-needletype", "needleType", "NeedleType",
     DEF_DIAL_NEEDLE_TYPE, -1, Tk_Offset(TkDial, needleType),
     0, (ClientData) needleStrings, 0},
    {TK_OPTION_PIXELS, "-pad", "pad", "Pad",
     DEF_DIAL_PAD, -1, Tk_Offset(TkDial, pad), 0, 0, 0},
    {TK_OPTION_SYNONYM, "-radius", (char *) NULL, (char *) NULL,
     (char *) NULL, 0, -1, 0, (ClientData) "-width", 0},
    {TK_OPTION_RELIEF, "-relief", "relief", "Relief",
     DEF_DIAL_RELIEF, -1, Tk_Offset(TkDial, relief), 0, 0, 0},
    {TK_OPTION_INT, "-repeatdelay", "repeatDelay", "RepeatDelay",
     DEF_DIAL_REPEAT_DELAY, -1, Tk_Offset(TkDial, repeatDelay), 0, 0, 0},
    {TK_OPTION_INT, "-repeatinterval", "repeatInterval", "RepeatInterval",
     DEF_DIAL_REPEAT_INTERVAL, -1, Tk_Offset(TkDial, repeatInterval), 0, 0, 0},
    {TK_OPTION_DOUBLE, "-resolution", "resolution", "Resolution",
     DEF_DIAL_RESOLUTION, -1, Tk_Offset(TkDial, resolution), 0, 0, 0},
    {TK_OPTION_STRING_TABLE, "-showtags", "showTags", "ShowTags",
     DEF_DIAL_SHOW_TAGS, -1, Tk_Offset(TkDial, showTags),
     0, (ClientData) showtagsStrings, 0},
    {TK_OPTION_BOOLEAN, "-showvalue", "showValue", "ShowValue",
     DEF_DIAL_SHOW_VALUE, -1, Tk_Offset(TkDial, showValue), 0, 0, 0},
    {TK_OPTION_STRING_TABLE, "-state", "state", "State",
     DEF_DIAL_STATE, -1, Tk_Offset(TkDial, state),
     0, (ClientData) stateStrings, 0},
    {TK_OPTION_STRING, "-takefocus", "takeFocus", "TakeFocus",
     DEF_DIAL_TAKE_FOCUS, -1, Tk_Offset(TkDial, takeFocus),
     TK_CONFIG_NULL_OK, 0, 0},
    {TK_OPTION_COLOR, "-tickcolor", "tickColor", "TickColor",
     DEF_DIAL_TICK_COLOR, -1, Tk_Offset(TkDial, tickColorPtr),
     0, (ClientData) DEF_DIAL_TICK_MONO, 0},
    {TK_OPTION_DOUBLE, "-tickinterval", "tickInterval", "TickInterval",
     DEF_DIAL_TICK_INTERVAL, -1, Tk_Offset(TkDial, tickInterval), 0, 0, 0},
    {TK_OPTION_PIXELS, "-tickwidth", "tickWidth", "TickWidth",
     DEF_DIAL_TICK_WIDTH, -1, Tk_Offset(TkDial, tickWidth), 0, 0, 0},
    {TK_OPTION_DOUBLE, "-to", "to", "To",
     DEF_DIAL_TO, -1, Tk_Offset(TkDial, toValue), 0, 0, 0},
    {TK_OPTION_PIXELS, "-width", "width", "Width",
     DEF_DIAL_WIDTH, -1, Tk_Offset(TkDial, reqWidth), 0, 0, 0},
    {TK_OPTION_STRING, "-variable", "variable", "Variable",
     DEF_DIAL_VARIABLE, -1, Tk_Offset(TkDial, varName),
     TK_CONFIG_NULL_OK, 0, 0},

    /* These are maintained purely for 100% compatibility with scale
     * widget options, but are not guaranteed to translate 100% correctly */
    {TK_OPTION_SYNONYM, "-length", (char *) NULL, (char *) NULL,
     (char *) NULL, 0, -1, 0, (ClientData) "-height", 0},
    {TK_OPTION_STRING, "-orient", "orient", "Orient",
     "", -1, Tk_Offset(TkDial, orient), TK_CONFIG_NULL_OK, 0, 0},
    {TK_OPTION_SYNONYM, "-sliderlength", (char *) NULL, (char *) NULL,
     (char *) NULL, 0, -1, 0, (ClientData) "-width", 0},
    {TK_OPTION_SYNONYM, "-sliderrelief", (char *) NULL, (char *) NULL,
     (char *) NULL, 0, -1, 0, (ClientData) "-dialrelief", 0},
    {TK_OPTION_SYNONYM, "-troughcolor", (char *) NULL, (char *) NULL,
     (char *) NULL, 0, -1, 0, (ClientData) "-needlecolor", 0},

    {TK_OPTION_END, (char *) NULL, (char *) NULL, (char *) NULL,
     (char *) NULL, 0, 0}
};

#ifdef FUTURE_DIAL_TAGS
/*
 * Dial tags are for the future.  This needs to be objectified here first.
 */

enum orientTag {
    ORIENT_TAG_LEFT, ORIENT_TAG_RIGHT, ORIENT_TAG_TOP, ORIENT_TAG_BOTTOM,
    ORIENT_TAG_INSIDE, ORIENT_TAG_OUTSIDE
};
static char *orientTagStrings[] = {
    "left", "right", "top", "bottom", "inside", "outside", (char*) NULL
};

static Tk_OptionSpec dialTagSpec[] = {
    {TK_OPTION_SYNONYM, "-fg", (char *) NULL, (char *) NULL,
     (char *) NULL, 0, -1, 0, (ClientData) "-foreground", 0},
    {TK_OPTION_COLOR, "-foreground", "foreground", "Foreground",
     DEF_DIAL_FG_COLOR, -1, Tk_Offset(TkDialTag, fg),
     0, (ClientData) DEF_DIAL_FG_MONO, 0},
    {TK_OPTION_STRING, "-label", "label", "Label",
     DEF_DIAL_LABEL, -1, Tk_Offset(TkDialTag, label),
     TK_CONFIG_NULL_OK, 0, 0},
    {TK_OPTION_STRING_TABLE, "-orient", "orient", "Orient",
     "", -1, Tk_Offset(TkDialTag, orient),
     0, (ClientData) orientTagStrings, 0},

    {TK_OPTION_END, (char *) NULL, (char *) NULL, (char *) NULL,
     (char *) NULL, 0, 0}
};
#endif

/*
 * The following tables define the widget commands (and sub-
 * commands) and map the indexes into the string tables into 
 * enumerated types used to dispatch the widget command.
 */

static CONST84 char *commandNames[] = {
    "cget", "configure", "constrain", "coords",
    "get", "identify", "label", "set", "tag", (char *) NULL
};

enum command {
    CMD_CGET, CMD_CONFIGURE, CMD_CONSTRAIN, CMD_COORDS,
    CMD_GET, CMD_IDENTIFY, CMD_LABEL, CMD_SET, CMD_TAG
};

/*
 * Forward declaration of functions used in this file:
 */
static int		DialWidgetObjCmd _ANSI_ARGS_((ClientData clientData,
				Tcl_Interp *interp, int objc,
				Tcl_Obj *CONST objv[]));
static void		DestroyDial _ANSI_ARGS_((ClientData clientData));
static int		ConfigureDial _ANSI_ARGS_((Tcl_Interp *interp,
				TkDial *dPtr, int objc,
				Tcl_Obj *CONST objv[], int flags));
static void		DisplayDial _ANSI_ARGS_((ClientData clientData));
static void		ComputeDialGeometry _ANSI_ARGS_((TkDial *dPtr));
static void		DialEventProc _ANSI_ARGS_((ClientData clientData,
				XEvent *eventPtr));
static void		DialCmdDeletedProc _ANSI_ARGS_((ClientData clientData));
static void		EventuallyRedrawDial _ANSI_ARGS_((TkDial *dPtr,
				int what));

static void		ComputeFormat _ANSI_ARGS_((TkDial *dPtr));
static void		DisplayTicks _ANSI_ARGS_((TkDial *dPtr,
				Drawable drawable, int minor));
static double		PixelToValue _ANSI_ARGS_((TkDial *dPtr,
				int x, int y));
static double		RoundToResolution _ANSI_ARGS_((TkDial *dPtr,
				double value));
static double		ConstrainDialValue _ANSI_ARGS_((TkDial *dPtr,
				double value));
static int		DialElement _ANSI_ARGS_((TkDial *dPtr,
				int x, int y));
static char *		DialVarProc _ANSI_ARGS_((ClientData clientData,
				Tcl_Interp *interp, char *name1, char *name2,
				int flags));
static void		SetDialValue _ANSI_ARGS_((TkDial *dPtr,
				double value, int setVar, int invokeCommand));
static double		ValueToAngle _ANSI_ARGS_((TkDial *dPtr, double value));
static void		ValueToPixel _ANSI_ARGS_((TkDial *dPtr,
				double value, int *xPtr, int *yPtr));


/*
 *--------------------------------------------------------------
 *
 * Tk_DialCmd --
 *
 *	This procedure is invoked to process the "dial" Tcl
 *	command.  See the user documentation for details on what
 *	it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */
int
Tk_DialObjCmd(clientData, interp, objc, objv)
     ClientData clientData;	/* do not use */
     Tcl_Interp *interp;	/* Current interpreter. */
     int objc;			/* Number of arguments. */
     Tcl_Obj *CONST objv[];      /* Argument objects. */
{
    register TkDial *dPtr;
    Tk_OptionTable optionTable;
    Tk_Window tkwin;

    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "pathName ?options?");
	return TCL_ERROR;
    }

    tkwin = Tk_CreateWindowFromPath(interp, Tk_MainWindow(interp),
            Tcl_GetString(objv[1]), (char *) NULL);
    if (tkwin == NULL) {
	return TCL_ERROR;
    }

    /*
     * Create the option table for this widget class.  If it has already
     * been created, Tk will return the cached value.
     */

    optionTable = Tk_CreateOptionTable(interp, optionSpecs);

    dPtr = (TkDial *) ckalloc(sizeof(TkDial));
    memset((VOID *) dPtr, 0, sizeof(TkDial));

    /*
     * Set the structure elments that aren't 0/NULL by default,
     * and that won't be set by the initial configure call.
     */
    dPtr->tkwin			= tkwin;
    dPtr->display		= Tk_Display(tkwin);
    dPtr->interp		= interp;
    dPtr->widgetCmd = Tcl_CreateObjCommand(interp, Tk_PathName(dPtr->tkwin),
					   DialWidgetObjCmd, (ClientData) dPtr,
					   DialCmdDeletedProc);
    dPtr->optionTable = optionTable;

    dPtr->resolution		= 1;
    dPtr->reqRadius		= 20;
    dPtr->state			= STATE_NORMAL;

    dPtr->tags = (Tcl_HashTable *) ckalloc(sizeof(Tcl_HashTable));
    Tcl_InitHashTable(dPtr->tags, TCL_STRING_KEYS);

    dPtr->tickGC		= None;
    dPtr->needleGC		= None;
    dPtr->textGC		= None;
    dPtr->relief		= TK_RELIEF_FLAT;
    dPtr->dialRelief		= TK_RELIEF_RAISED;
    dPtr->cursor		= None;
    dPtr->flags			= NEVER_SET;

    Tk_SetClass(dPtr->tkwin, "Dial");
    Tk_CreateEventHandler(dPtr->tkwin,
			  ExposureMask|StructureNotifyMask|FocusChangeMask,
			  DialEventProc, (ClientData) dPtr);

    if (Tk_InitOptions(interp, (char *) dPtr, optionTable, tkwin) != TCL_OK) {
	goto error;
    }
    if (ConfigureDial(interp, dPtr, objc-2, objv+2, 0) != TCL_OK) {
	goto error;
    }

    Tcl_SetStringObj(Tcl_GetObjResult(interp), Tk_PathName(dPtr->tkwin), -1);
    return TCL_OK;

error:
    Tk_DestroyWindow(dPtr->tkwin);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * DialWidgetCmd --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */
static int
DialWidgetObjCmd(clientData, interp, objc, objv)
     ClientData clientData;		/* Information about dial widget */
     Tcl_Interp *interp;		/* Current interpreter. */
     int objc;			/* Number of arguments. */
     Tcl_Obj *CONST objv[];	/* Argument objects. */
{
    register TkDial *dPtr = (TkDial *) clientData;
    int cmdIndex, result = TCL_OK;
    Tcl_Obj *objPtr, *resultPtr;

    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "option ?arg arg ...?");
	return TCL_ERROR;
    }
    Tcl_Preserve((ClientData) dPtr);

    /* parse the first parameter */
    result = Tcl_GetIndexFromObj(interp, objv[1], commandNames,
				 "option", 0, &cmdIndex);
    if (result != TCL_OK) {
	return result;
    }
    resultPtr = Tcl_GetObjResult(interp);

    switch (cmdIndex) {
    case CMD_CGET:
	if (objc != 3) {
	    Tcl_WrongNumArgs(interp, 1, objv, "cget option");
	    result = TCL_ERROR;
	} else {
	    objPtr = Tk_GetOptionValue(interp, (char *) dPtr,
				       dPtr->optionTable, objv[2],
				       dPtr->tkwin);
	    if (objPtr == NULL) {
		result = TCL_ERROR;
	    } else {
		Tcl_SetObjResult(interp, objPtr);
	    }
	}
	break;

    case CMD_CONFIGURE:
	if (objc <= 3) {
	    objPtr = Tk_GetOptionInfo(interp, (char *) dPtr,
				      dPtr->optionTable,
				      (objc == 3) ? objv[2] : (Tcl_Obj *) NULL,
				      dPtr->tkwin);
	    if (objPtr == NULL) {
		result = TCL_ERROR;
	    } else {
		Tcl_SetObjResult(interp, objPtr);
	    }
	} else {
	    result = ConfigureDial(interp, dPtr, objc-2, objv+2, 0);
	}
	break;

    case CMD_CONSTRAIN: {
	double value;

	if ((objc != 2) && (objc != 3)) {
	    Tcl_WrongNumArgs(interp, 1, objv, "constrain ?value?");
	    result = TCL_ERROR;
	    break;
	} else if ((objc == 3) &&
		   (Tcl_GetDoubleFromObj(interp, objv[2], &value) != TCL_OK)) {
	    result = TCL_ERROR;
	    break;
	} else {
	    value = dPtr->value;
	}
	Tcl_SetDoubleObj(resultPtr, ConstrainDialValue(dPtr, value));
	break;
    }

    case CMD_COORDS: {
	int x, y;
	double value;

	if ((objc != 2) && (objc != 3)) {
	    Tcl_WrongNumArgs(interp, 1, objv, "coords ?value?");
	    result = TCL_ERROR;
	    break;
	}
	if ((objc == 3) &&
	    (Tcl_GetDoubleFromObj(interp, objv[2], &value) != TCL_OK)) {
	    result = TCL_ERROR;
	    break;
	} else {
	    value = dPtr->value;
	}
	ValueToPixel(dPtr, value, &x, &y);
	objPtr = Tcl_NewIntObj(x);
	if (Tcl_ListObjAppendElement(interp, resultPtr, objPtr)
	    != TCL_OK) {
	    Tcl_DecrRefCount(objPtr); /* free obj */
	    result = TCL_ERROR;
	    break;
	}
	objPtr = Tcl_NewIntObj(y);
	if (Tcl_ListObjAppendElement(interp, resultPtr, objPtr)
	    != TCL_OK) {
	    Tcl_DecrRefCount(objPtr); /* free obj */
	    result = TCL_ERROR;
	    break;
	}
	break;
    }

    case CMD_GET: {
	int x, y;

	if ((objc != 2) && (objc != 4)) {
	    Tcl_WrongNumArgs(interp, 1, objv, "get ?x y?");
	    result = TCL_ERROR;
	} else if (objc == 2) {
	    Tcl_SetDoubleObj(resultPtr, dPtr->value);
	} else {
	    if ((Tcl_GetIntFromObj(interp, objv[2], &x) != TCL_OK) ||
		(Tcl_GetIntFromObj(interp, objv[3], &y) != TCL_OK)) {
		result = TCL_ERROR;
		break;
	    }
	    Tcl_SetDoubleObj(resultPtr, PixelToValue(dPtr, x, y));
	}
	break;
    }

    case CMD_IDENTIFY: {
	int x, y;

	if (objc != 4) {
	    Tcl_WrongNumArgs(interp, 1, objv, "identify x y");
	    result = TCL_ERROR;
	} else if ((Tcl_GetIntFromObj(interp, objv[2], &x) != TCL_OK) ||
		   (Tcl_GetIntFromObj(interp, objv[3], &y) != TCL_OK)) {
	    result = TCL_ERROR;
	} else {
	    switch (DialElement(dPtr, x, y)) {
	    case DIAL:	Tcl_SetStringObj(resultPtr, "dial", -1); break;
	    case LEFT:	Tcl_SetStringObj(resultPtr, "left", -1); break;
	    case RIGHT:	Tcl_SetStringObj(resultPtr, "right", -1); break;
	    }
	}
	break;
    }

    case CMD_LABEL: {
	Tcl_HashEntry *entryPtr;
	double value;

	if (objc < 2) {
	    Tcl_WrongNumArgs(interp, 1, objv, "label ?-constrain? ?value? ?string? ?value string ...?");
	    result = TCL_ERROR;
	    break;
	}
	if (objc == 2) {
	    Tcl_HashSearch search;

	    for (entryPtr = Tcl_FirstHashEntry(dPtr->tags, &search);
		 entryPtr != NULL; entryPtr = Tcl_NextHashEntry(&search)) {
		objPtr = Tcl_NewStringObj(Tcl_GetHashKey(dPtr->tags,
							 entryPtr), -1);
		if (Tcl_ListObjAppendElement(interp, resultPtr, objPtr)
		    != TCL_OK) {
		    Tcl_DecrRefCount(objPtr); /* free obj */
		    result = TCL_ERROR;
		    break;
		}
		objPtr = Tcl_NewStringObj((char *)Tcl_GetHashValue(entryPtr),
			-1);
		if (Tcl_ListObjAppendElement(interp, resultPtr, objPtr)
		    != TCL_OK) {
		    Tcl_DecrRefCount(objPtr); /* free obj */
		    result = TCL_ERROR;
		    break;
		}
	    }
	    return TCL_OK;
	}
	if (objc == 3) {
	    /* Check that it is a valid double */
	    if (Tcl_GetDoubleFromObj(interp, objv[2], &value) != TCL_OK) {
		result = TCL_ERROR;
		break;
	    }
	    entryPtr = Tcl_FindHashEntry(dPtr->tags, Tcl_GetString(objv[2]));
	    if (entryPtr != NULL) {
		Tcl_SetStringObj(resultPtr,
				 (char *) Tcl_GetHashValue(entryPtr), -1);
	    }
	} else {
	    int i, new, len, constrain = 0, startArg = 2;
	    char *str, tagStr[TCL_DOUBLE_SPACE];

	    str = Tcl_GetStringFromObj(objv[2], &len);
	    if ((*str == '-') && (len >= 2) &&
		(strncmp(str, "-constrain", (unsigned) len)==0)) {
		constrain = 1;
		startArg = 3;
		objc--;
	    }

	    for (i = startArg; i < objc; i+= 2) {
		/* Check that it is a valid double */
		if (Tcl_GetDoubleFromObj(interp, objv[i], &value) != TCL_OK) {
		    result = TCL_ERROR;
		    break;
		}
		if (constrain) {
		    value = ConstrainDialValue(dPtr, value);
		    sprintf(tagStr, dPtr->tagFormat, value);
		    entryPtr = Tcl_CreateHashEntry(dPtr->tags, tagStr, &new);
		} else {
		    entryPtr = Tcl_CreateHashEntry(dPtr->tags,
						   Tcl_GetString(objv[i]),
						   &new);
		}
		str = Tcl_GetStringFromObj(objv[i+1], &len);
		if (len == 0) {
		    if (!new) {
			ckfree((char *) Tcl_GetHashValue(entryPtr));
		    }
		    Tcl_DeleteHashEntry(entryPtr);
		} else {
		    char *hash;
		    hash = (char *) ckalloc((unsigned) (len+1));
		    memcpy(hash, str, (unsigned) len);
		    Tcl_SetHashValue(entryPtr, hash);
		}
	    }
	    ComputeDialGeometry(dPtr);
	    EventuallyRedrawDial(dPtr, REDRAW_ALL);
	}
	break;
    }

    case CMD_SET: {
	double value;

	if (objc != 3 && objc != 2) {
	    Tcl_WrongNumArgs(interp, 1, objv, "set ?value?");
	    result = TCL_ERROR;
	    break;
	} else if (objc == 3) {
	    if (Tcl_GetDoubleFromObj(interp, objv[2], &value) != TCL_OK) {
		result = TCL_ERROR;
		break;
	    }
	    if (dPtr->state != STATE_DISABLED) {
		SetDialValue(dPtr, value, 1, 1);
	    }
	}
	Tcl_SetDoubleObj(resultPtr, dPtr->value);
	break;
    }

    }

    Tcl_Release((ClientData) dPtr);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * DestroyDial --
 *
 *	This procedure is invoked by Tk_EventuallyFree or Tcl_Release
 *	to clean up the internal structure of a button at a safe time
 *	(when no-one is using it anymore).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the dial is freed up.
 *
 *----------------------------------------------------------------------
 */
static void
DestroyDial(clientData)
     ClientData clientData;	/* Info about dial widget. */
{
    register TkDial *dPtr = (TkDial *) clientData;
    Tcl_HashSearch search;
    Tcl_HashEntry *entryPtr;

    Tcl_CancelIdleCall(DisplayDial, (ClientData) dPtr);
    dPtr->flags = 0;

    Tcl_DeleteCommandFromToken(dPtr->interp, dPtr->widgetCmd);

    /*
     * Free up all the stuff that requires special handling, then
     * let Tk_FreeOptions handle all the standard option-related
     * stuff.
     */

    if (dPtr->varName != NULL) {
	Tcl_UntraceVar(dPtr->interp, dPtr->varName,
		       TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		       (Tcl_VarTraceProc *)DialVarProc, (ClientData) dPtr);
    }
    if (dPtr->tickGC != None) {
	Tk_FreeGC(dPtr->display, dPtr->tickGC);
    }
    if (dPtr->needleGC != None) {
	Tk_FreeGC(dPtr->display, dPtr->needleGC);
    }
    if (dPtr->textGC != None) {
	Tk_FreeGC(dPtr->display, dPtr->textGC);
    }
    for (entryPtr = Tcl_FirstHashEntry(dPtr->tags, &search);
	 entryPtr != NULL; entryPtr = Tcl_NextHashEntry(&search)) {
	ckfree((char *) Tcl_GetHashValue(entryPtr));
    }
    Tcl_DeleteHashTable(dPtr->tags);
    ckfree((char *) dPtr->tags);

    Tk_FreeConfigOptions((char *) dPtr, dPtr->optionTable, dPtr->tkwin);
    dPtr->tkwin = NULL;
    Tcl_EventuallyFree((ClientData) dPtr, TCL_DYNAMIC);
}

/*
 *----------------------------------------------------------------------
 *
 * ConfigureDial --
 *
 *	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or
 *	reconfigure) a dial widget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as colors, border width,
 *	etc. get set for dPtr;  old resources get freed,
 *	if there were any.
 *
 *----------------------------------------------------------------------
 */
static int
ConfigureDial(interp, dPtr, objc, objv, flags)
     Tcl_Interp *interp;	/* Used for error reporting. */
     register TkDial *dPtr;	/* Information about widget; may or may not
				 * already have values for some fields */
     int objc;
     Tcl_Obj *CONST objv[];
     int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    Tk_SavedOptions savedOptions;
    Tcl_Obj *errorResult = NULL;
    int error;

    XGCValues gcValues;
    GC newGC;
    unsigned int mask;

    /* Eliminate any existing trace on a variable monitored by the dial.  */
    if (dPtr->varName != NULL) {
	Tcl_UntraceVar(interp, dPtr->varName, 
		       TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		       (Tcl_VarTraceProc *)DialVarProc, (ClientData) dPtr);
    }

    for (error = 0; error <= 1; error++) {
	if (!error) {
	    /*
	     * First pass: set options to new values.
	     */

	    if (Tk_SetOptions(interp, (char *) dPtr,
		    dPtr->optionTable, objc, objv,
		    dPtr->tkwin, &savedOptions, (int *) NULL) != TCL_OK) {
		continue;
	    }
	} else {
	    /*
	     * Second pass: restore options to old values.
	     */

	    errorResult = Tcl_GetObjResult(interp);
	    Tcl_IncrRefCount(errorResult);
	    Tk_RestoreSavedOptions(&savedOptions);
	}

	/*
	 * A few other options also need special processing, such as parsing
	 * the geometry and setting the background from a 3-D border.
	 */

	dPtr->tickInterval	= fabs(dPtr->tickInterval);
	dPtr->minTickInterval = fabs(dPtr->minTickInterval);
	dPtr->tickWidth	= MAX(dPtr->tickWidth, 0);

	break;
    }
    if (!error) {
	Tk_FreeSavedOptions(&savedOptions);
    }

    /*
     * If the dial is tied to the value of a variable, then set up
     * a trace on the variable's value and set the dial's value from
     * the value of the variable, if it exists.
     */
    if (dPtr->varName != NULL) {
	CONST84 char *stringValue;
	char *end;
	double value;

	stringValue = Tcl_GetVar(interp, dPtr->varName, TCL_GLOBAL_ONLY);
	if (stringValue != NULL) {
	    value = strtod(stringValue, &end);
	    if ((end != stringValue) && (*end == 0)) {
		dPtr->value = value;
	    }
	}
	Tcl_TraceVar(interp, dPtr->varName,
		     TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		     (Tcl_VarTraceProc *)DialVarProc, (ClientData) dPtr);
    }

    /*
     * Set the dial value to itself;  all this does is to make sure
     * that the dial's value is within the new acceptable range for
     * the dial and reflect the value in the associated variable,
     * if any.
     */
    ComputeFormat(dPtr);
    SetDialValue(dPtr, dPtr->value, 1, 1);

    /* Create GC's */
    gcValues.foreground = dPtr->tickColorPtr->pixel;
    gcValues.line_width = dPtr->tickWidth;
    newGC = Tk_GetGC(dPtr->tkwin, GCForeground|GCLineWidth, &gcValues);
    if (dPtr->tickGC != None) {
	Tk_FreeGC(dPtr->display, dPtr->tickGC);
    }
    dPtr->tickGC = newGC;

    gcValues.foreground = dPtr->needleColorPtr->pixel;
    mask = GCForeground;
    if (dPtr->needleType == NEEDLE_LINE) {
	/* we only want line needles to have a width */
	gcValues.line_width = (int)rint(NEEDLE_WIDTH*dPtr->reqRadius);
	mask |= GCLineWidth;
    }
    newGC = Tk_GetGC(dPtr->tkwin, mask, &gcValues);
    if (dPtr->needleGC != None) {
	Tk_FreeGC(dPtr->display, dPtr->needleGC);
    }
    dPtr->needleGC = newGC;

    if (dPtr->highlightWidth < 0) {
	dPtr->highlightWidth = 0;
    }
    gcValues.font = Tk_FontId(dPtr->tkfont);
    gcValues.foreground = dPtr->textColorPtr->pixel;
    gcValues.graphics_exposures = False;
    newGC = Tk_GetGC(dPtr->tkwin, GCForeground|GCFont|GCGraphicsExposures,
		     &gcValues);
    if (dPtr->textGC != None) {
	Tk_FreeGC(dPtr->display, dPtr->textGC);
    }
    dPtr->textGC = newGC;

    dPtr->inset = dPtr->highlightWidth + dPtr->borderWidth;

    /*
     * Recompute display-related information, and let the geometry
     * manager know how much space is needed now.
     */
    Tk_SetBackgroundFromBorder(dPtr->tkwin, dPtr->bgBorder);
    ComputeDialGeometry(dPtr);
    Tk_SetInternalBorder(dPtr->tkwin, dPtr->inset);
    EventuallyRedrawDial(dPtr, REDRAW_ALL);
    if (error) {
	Tcl_SetObjResult(interp, errorResult);
	Tcl_DecrRefCount(errorResult);
	return TCL_ERROR;
    } else {
	return TCL_OK;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ComputeDialGeometry --
 *
 *	This procedure is called to compute various geometrical
 *	information for a dial, such as where various things get
 *	displayed.  It's called when the window is reconfigured.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Display-related numbers get changed in *dPtr.  The
 *	geometry manager gets told about the window's preferred size.
 *
 *----------------------------------------------------------------------
 */
static void
ComputeDialGeometry(dPtr)
     register TkDial *dPtr;	/* Information about widget. */
{
    char str[TCL_DOUBLE_SPACE];
    int width, towidth, reqWidth, reqHeight;
    Tk_FontMetrics fm;

    Tk_GetFontMetrics(dPtr->tkfont, &fm);

    reqWidth  = 2*(dPtr->reqRadius+dPtr->inset);
    reqHeight = 2*(dPtr->reqRadius+dPtr->inset);
    /* Radii of the minor ticks */
    if (dPtr->minTickInterval != 0.0) {
	reqWidth  += 2*rint(MIN_TICK_LENGTH * dPtr->reqRadius);
	reqHeight += 2*rint(MIN_TICK_LENGTH * dPtr->reqRadius);
    }

    if (dPtr->tickInterval != 0.0) {
	/* Make room for major ticks */
	reqWidth  += 2*rint(TICK_LENGTH * dPtr->reqRadius);
	reqHeight += 2*rint(TICK_LENGTH * dPtr->reqRadius);

	if (dPtr->showTags != SHOWTAGS_NONE) {
	    Tcl_HashEntry *entryPtr;
	    char *label;
	    /* Tags should lean against this radius */
	    reqWidth  += dPtr->pad;
	    reqHeight += dPtr->pad;
	    width = towidth = 0;

	    /*
	     * Compute the width needed to display the dial tags by formatting
	     * strings for the lower and upper limit.
	     */
	    label = NULL;
	    sprintf(str, dPtr->tagFormat, dPtr->fromValue);
	    if ((dPtr->showTags & SHOWTAGS_LABEL) &&
		(entryPtr = Tcl_FindHashEntry(dPtr->tags, str)) != NULL) {
		label = (char *)Tcl_GetHashValue(entryPtr);
	    } else if (dPtr->showTags & SHOWTAGS_VALUE) {
		label = str;
	    }
	    if (label) {
		width = Tk_TextWidth(dPtr->tkfont, label, (int) strlen(label));
	    }
	    label = NULL;
	    sprintf(str, dPtr->tagFormat, dPtr->toValue);
	    if ((dPtr->showTags & SHOWTAGS_LABEL) &&
		(entryPtr = Tcl_FindHashEntry(dPtr->tags, str)) != NULL) {
		label = (char *)Tcl_GetHashValue(entryPtr);
	    } else if (dPtr->showTags & SHOWTAGS_VALUE) {
		label = str;
	    }
	    if (label) {
		towidth = Tk_TextWidth(dPtr->tkfont, label, (int) strlen(label));
	    }

	    reqWidth  += width + towidth;
	    reqHeight += fm.linespace;
	}
    }
    reqWidth  += dPtr->pad;
    reqHeight += dPtr->pad;
    dPtr->centerY = reqHeight/2;

    /* Also make sure the value fits.  */
    if (dPtr->showValue) {
	/*
	 * Determine the max width of the value from the upper and lower limit,
	 * using the value's format string this time.
	 */
	sprintf(str, dPtr->valueFormat, dPtr->fromValue);
	width = Tk_TextWidth(dPtr->tkfont, str, (int) strlen(str));
	sprintf(str, dPtr->valueFormat, dPtr->toValue);
	towidth = Tk_TextWidth(dPtr->tkfont, str, (int) strlen(str));
	if (towidth > width) {
	    width = towidth;
	}

	reqWidth   = MAX(reqWidth, width);
	reqHeight += fm.linespace + 2*dPtr->pad;
    }
    if (dPtr->label != NULL) {
	width = Tk_TextWidth(dPtr->tkfont, str, (int) strlen(str));
	reqWidth   = MAX(reqWidth, width);
	reqHeight += fm.linespace + 2*dPtr->pad;
	dPtr->centerY += fm.linespace + 2*dPtr->pad;
    }
    dPtr->centerX = reqWidth/2;
    if (dPtr->mode == MODE_ELLIPSE) {
	dPtr->radX = dPtr->centerX/2;
	dPtr->radY = dPtr->centerY/2;
    } else {
	/* Maintain the dial as a circle */
	dPtr->radX = dPtr->radY = MIN(dPtr->centerX,dPtr->centerY)/2;
    }

    Tk_GeometryRequest(dPtr->tkwin, reqWidth, reqHeight);
}

/*
 *--------------------------------------------------------------
 *
 * DisplayTicks --
 *
 *	This procedure draws the tickmarks corresponding to the
 *	specified interval. If the flag major is false, there is
 *	minor tickmarks are drawn; otherwise major tickmarks are
 *	drawn.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Either the minor or the major ticks are drawn in the
 *	drawable.
 *
 *--------------------------------------------------------------
 */
static void
DisplayTicks(dPtr, drawable, minorTicks)
     TkDial *dPtr;		/* Widget record for dial. */
     Drawable drawable;		/* Where to display dial (window or pixmap) */
     int minorTicks;		/* 0: minor, 1: major ticks */
{
    double from, to, interval, tickValue;
    int i, sign;
    Tk_FontMetrics fm;
    double angle, c, s;
    int x1, y1, x2, y2, len, width, radX, radY, oRadX, oRadY;
    Tcl_HashEntry *entryPtr;

    Tk_GetFontMetrics(dPtr->tkfont, &fm);

    radX = dPtr->radX;
    radY = dPtr->radY;
    if (minorTicks) {
	interval = dPtr->minTickInterval;
	/* we offset the minor ticks to be slightly away from the main dial */
	i = rint((TICK_LENGTH-MIN_TICK_LENGTH) * dPtr->reqRadius);
	radX += i;
	radY += i;
	i = rint(MIN_TICK_LENGTH * dPtr->reqRadius);
    } else {
	interval = dPtr->tickInterval;
	i = rint(TICK_LENGTH * dPtr->reqRadius);
    }
    oRadX = radX + i;
    oRadY = radY + i;

    if (interval == 0.0)
	return;

    from = dPtr->fromValue;
    to = dPtr->toValue;
    sign = (to > from ? 1 : -1);

    if ((to-from != 0.0) && (interval/(to-from) > MAX_TICKS))
	return;

    /* Round the lower tick towards the end value */
    i = (int) ((from < to) ? ceil(from/interval) : floor(from/interval));

    for (tickValue = i*interval; sign*tickValue <= sign*to;
	 i += sign, tickValue = i*interval) {
	/*
	 * The RoundToResolution call gets rid of accumulated
	 * round-off errors, if any.
	 */
	tickValue = RoundToResolution(dPtr, tickValue);

	/* Determine endpoints of the minortick.  */
	angle = ValueToAngle(dPtr, tickValue);
	s = sin(angle);
	c = cos(angle);

	if (minorTicks) {
	    /* Prevent drawing at the position of a major tick.  */
	    if (fmod(tickValue,dPtr->tickInterval) == 0.0) {
		continue;
	    }
	} else if (dPtr->showTags != SHOWTAGS_NONE) {
	    /*
	     * Display a label next to the tick. The location of the label is
	     * corrected assuming that the label has the form of an ellipse,
	     * which is of course only approximately true.
	     */
	    char *label, str[TCL_DOUBLE_SPACE];

	    label = NULL;
	    sprintf(str, dPtr->tagFormat, tickValue);
	    if ((dPtr->showTags & SHOWTAGS_LABEL) &&
		(entryPtr = Tcl_FindHashEntry(dPtr->tags, str)) != NULL) {
		label = (char *)Tcl_GetHashValue(entryPtr);
	    } else if (dPtr->showTags & SHOWTAGS_VALUE) {
		label = str;
	    }
	    if (label) {
		len = strlen(label);
		width = Tk_TextWidth(dPtr->tkfont, label, len);
		x1 = (int) (dPtr->centerX +
			(rint((oRadX+dPtr->pad)*s)+(0.5*width*(s-1))));
		y1 = (int) (dPtr->centerY -
			(rint((oRadY+dPtr->pad)*c)+(0.5*fm.linespace*(c-1))));
		Tk_DrawChars(dPtr->display, drawable, dPtr->textGC,
			     dPtr->tkfont, label, len, x1, y1);
	    }
	}
	x1 = dPtr->centerX + rint(radX*s);
	y1 = dPtr->centerY - rint(radY*c);
	x2 = dPtr->centerX + rint(oRadX*s);
	y2 = dPtr->centerY - rint(oRadY*c);
	/* Draw the tick */
	XDrawLine(dPtr->display, drawable, dPtr->tickGC, x1, y1, x2, y2);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * DisplayInnerDial --
 *
 *	This procedure is invoked by DisplayDial to redisplay
 *	the inner contents of a dial widget.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The inner dial gets redrawn.
 *
 *----------------------------------------------------------------------
 */
static void
DisplayInnerDial(register TkDial *dPtr, Drawable pixmap)
{
    double angle, c, s;
    int i, radX = dPtr->radX, radY = dPtr->radY;
    Tk_3DBorder border;

    border = (dPtr->state==STATE_ACTIVE)?dPtr->activeBorder:dPtr->dialBorder;
    if (dPtr->dialRelief != TK_RELIEF_FLAT) {
	/* Calculate dial polygon points.  */
	for (i = 0; i < DIAL_POINTS; ++i) {
	    angle = 2*M_PI * i / (DIAL_POINTS-1);
	    dPtr->dial[i].x = dPtr->centerX + rint(radX * sin(angle));
	    dPtr->dial[i].y = dPtr->centerY + rint(radY * cos(angle));
	}
	/* draw the dial, if any, first because of the fill */
	Tk_Fill3DPolygon(dPtr->tkwin, pixmap, border,
			 dPtr->dial, DIAL_POINTS, dPtr->dialBorderWidth,
			 dPtr->dialRelief);
	/* offset future use of radius to come inside border */
	radX -= dPtr->dialBorderWidth;
	radY -= dPtr->dialBorderWidth;
    } else {
	/* If a flat relief is chosen, we fill an arc instead, but only
	 * between the begin and end points */
	XFillArc(dPtr->display, pixmap,
		 Tk_3DBorderGC(dPtr->tkwin, border, TK_3D_FLAT_GC),
		 dPtr->centerX - radX, dPtr->centerY - radY,
		 (unsigned) radX*2, (unsigned) radY*2,
		 (int) ((M_DEGREE/4.0)-(dPtr->beginAngle*64.0) + 0.5),
		 ((dPtr->beginAngle-dPtr->endAngle)*64));
    }
    angle = ValueToAngle(dPtr, dPtr->value);
    c = cos(angle);
    s = sin(angle);
    if (dPtr->needleType == NEEDLE_TRIANGLE) {
	/* draw a triangular needle */
	double wc = (NEEDLE_WIDTH * MIN(radX,radY) * c);
	double ws = (NEEDLE_WIDTH * MIN(radX,radY) * s);

	/* 0 && 3 */
	dPtr->needle[3].x = dPtr->needle[0].x = dPtr->centerX + rint(s*radX);
	dPtr->needle[3].y = dPtr->needle[0].y = dPtr->centerY - rint(c*radY);
	/* 1 */
	dPtr->needle[1].x = dPtr->centerX - rint(wc+ws);
	dPtr->needle[1].y = dPtr->centerY + rint(wc-ws);
	/* 2 */
	dPtr->needle[2].x = dPtr->centerX - rint(ws-wc);
	dPtr->needle[2].y = dPtr->centerY + rint(ws+wc);
	XFillPolygon(dPtr->display, pixmap, dPtr->needleGC,
		     dPtr->needle, NEEDLE_POINTS, Convex, CoordModeOrigin);
	return;
    } else if (dPtr->needleType == NEEDLE_ARC) {
	int start, extent;
	start = (int) ((M_DEGREE/4.0)-(dPtr->beginAngle*64.0) + 0.5);
	extent = (int) ((dPtr->beginAngle-rad2deg(angle))*64.0 + 0.5);
	if (ABS(extent) > 64) {
	    /* require at least 1 degree worth of arc to draw */
	    XFillArc(dPtr->display, pixmap, dPtr->needleGC,
		     dPtr->centerX - radX, dPtr->centerY - radY,
		     (unsigned) radX*2, (unsigned) radY*2, start, extent);
	    return;
	}
	/* otherwise this will go through and just draw a line */
    }
    /* NEEDLE_LINE: */
    XDrawLine(dPtr->display, pixmap, dPtr->needleGC,
	      dPtr->centerX, dPtr->centerY,
	      (int)rint(dPtr->centerX + s*radX),
	      (int)rint(dPtr->centerY - c*radY));
}

/*
 *----------------------------------------------------------------------
 *
 * DisplayDial --
 *
 *	This procedure is invoked as an idle handler to redisplay
 *	the contents of a dial widget.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The dial gets redisplayed.
 *
 *----------------------------------------------------------------------
 */
static void
DisplayDial(clientData)
     ClientData clientData;	/* Widget record for dial. */
{
    TkDial *dPtr = (TkDial *) clientData;
    Tk_Window tkwin = dPtr->tkwin;
    Pixmap pixmap;
    int width, valPix, lblPix;
    Tk_FontMetrics fm;

    if ((dPtr->tkwin == NULL) || !Tk_IsMapped(dPtr->tkwin)) {
	goto done;
    }

    /* Invoke the dial's command if needed.  */
    Tcl_Preserve((ClientData) dPtr);
    if ((dPtr->flags & INVOKE_COMMAND) && (dPtr->command != NULL)) {
	char commandString[TCL_DOUBLE_SPACE];
	int result;

	sprintf(commandString, dPtr->valueFormat, dPtr->value);
	result = Tcl_VarEval(dPtr->interp, dPtr->command,
			     " ", commandString, (char *) NULL);
	if (result != TCL_OK) {
	    Tcl_AddErrorInfo(dPtr->interp, "\n    (command executed by dial)");
	    Tk_BackgroundError(dPtr->interp);
	}
    }
    dPtr->flags &= ~INVOKE_COMMAND;
    Tcl_Release((ClientData) dPtr);
    if (dPtr->tkwin == NULL || !Tk_IsMapped(dPtr->tkwin)) {
	goto done;
    }

    /*
     * In order to avoid screen flashes, this procedure redraws
     * the dial in a pixmap, then copies the pixmap to the
     * screen in a single operation.  This means that there's no
     * point in time where the on-sreen image has been cleared.
     */
    pixmap = Tk_GetPixmap(dPtr->display, Tk_WindowId(tkwin),
			  Tk_Width(tkwin), Tk_Height(tkwin), Tk_Depth(tkwin));

    Tk_GetFontMetrics(dPtr->tkfont, &fm);

    /* Fill drawnArea with the background */
    Tk_Fill3DRectangle(tkwin, pixmap, dPtr->bgBorder, 0, 0,
		       Tk_Width(tkwin), Tk_Height(tkwin), 0, TK_RELIEF_FLAT);

    dPtr->centerX = Tk_Width(tkwin)/2;
    valPix = lblPix = 0;
    /* Display the value centered, if required */
    if (dPtr->showValue) {
	char str[TCL_DOUBLE_SPACE];
	int length;

	sprintf(str, dPtr->valueFormat, dPtr->value);
	length = strlen(str);
	width = Tk_TextWidth(dPtr->tkfont, str, length);
	Tk_DrawChars(dPtr->display, pixmap, dPtr->textGC,
		     dPtr->tkfont, str, length,
		     dPtr->centerX - width/2,
		     Tk_Height(tkwin) - (dPtr->inset + dPtr->pad));
	valPix = fm.ascent + dPtr->pad;
    }

    /* Draw the label north of the dial.  */
    if ((dPtr->flags && REDRAW_OTHER) && (dPtr->label != NULL)) {
	int len = strlen(dPtr->label);
	width = Tk_TextWidth(dPtr->tkfont, dPtr->label, len);
	Tk_DrawChars(dPtr->display, pixmap, dPtr->textGC,
		     dPtr->tkfont, dPtr->label, len,
		     dPtr->centerX - width/2, dPtr->inset + fm.linespace);
	lblPix = fm.linespace + dPtr->pad;
    }

    dPtr->centerY = lblPix + (Tk_Height(tkwin)-lblPix-valPix)/2;
    if (dPtr->mode == MODE_ELLIPSE) {
	dPtr->radX = dPtr->centerX/2;
	dPtr->radY = dPtr->centerY/2;
    } else {
	/* Maintain the dial as a circle */
	dPtr->radX = dPtr->radY = MIN(dPtr->centerX,dPtr->centerY)/2;
    }

    /* Display minor and major ticks */
    DisplayTicks(dPtr, pixmap, 1); /* minor */
    DisplayTicks(dPtr, pixmap, 0); /* major */

    /* Draw the inner dial &| needle */
    DisplayInnerDial(dPtr, pixmap);

    /* Handle border and traversal highlight.  */
    if (dPtr->flags & REDRAW_OTHER) {
	int hl = dPtr->highlightWidth;
	if (dPtr->relief != TK_RELIEF_FLAT) {
	    Tk_Draw3DRectangle(tkwin, pixmap, dPtr->bgBorder, hl, hl,
			       Tk_Width(tkwin) - 2*hl, Tk_Height(tkwin) - 2*hl,
			       dPtr->borderWidth, dPtr->relief);

	}
	if (hl) {
	    GC gc;
	    if (dPtr->flags & GOT_FOCUS) {
		gc = Tk_GCForColor(dPtr->highlightColorPtr, pixmap);
	    } else {
		gc = Tk_GCForColor(dPtr->highlightBgColorPtr, pixmap);
	    }
	    Tk_DrawFocusHighlight(tkwin, gc, hl, pixmap);
	}
    }

    /*
     * Copy the information from the off-screen pixmap onto the screen,
     * then delete the pixmap.
     */
    XCopyArea(dPtr->display, pixmap, Tk_WindowId(tkwin), dPtr->textGC,
	      0, 0, (unsigned) Tk_Width(tkwin), (unsigned) Tk_Height(tkwin),
	      0, 0);
    Tk_FreePixmap(dPtr->display, pixmap);

done:
    dPtr->flags &= ~REDRAW_ALL;
}

/*
 *--------------------------------------------------------------
 *
 * DialEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher for various
 *	events on dials.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get
 *	cleaned up.  When it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */
static void
DialEventProc(clientData, eventPtr)
     ClientData clientData;	/* Information about window. */
     XEvent *eventPtr;		/* Information about event. */
{
    TkDial *dPtr = (TkDial *) clientData;

    switch (eventPtr->type) {
    case Expose:
	if (eventPtr->xexpose.count == 0)
	    EventuallyRedrawDial(dPtr, REDRAW_ALL);
	break;
    case DestroyNotify:
	DestroyDial(clientData);
	break;
    case ConfigureNotify:
	ComputeDialGeometry(dPtr);
	EventuallyRedrawDial(dPtr, REDRAW_ALL);
	break;
    case FocusIn:
	if (eventPtr->xfocus.detail != NotifyInferior) {
	    dPtr->flags |= GOT_FOCUS;
	    if (dPtr->highlightWidth > 0) {
		EventuallyRedrawDial(dPtr, REDRAW_ALL);
	    }
	}
	break;
    case FocusOut:
	if (eventPtr->xfocus.detail != NotifyInferior) {
	    dPtr->flags &= ~GOT_FOCUS;
	    if (dPtr->highlightWidth > 0) {
		EventuallyRedrawDial(dPtr, REDRAW_ALL);
	    }
	}
	break;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * DialCmdDeletedProc --
 *
 *	This procedure is invoked when a widget command is deleted.  If
 *	the widget isn't already in the process of being destroyed,
 *	this command destroys it.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The widget is destroyed.
 *
 *----------------------------------------------------------------------
 */
static void
DialCmdDeletedProc(clientData)
     ClientData clientData;	/* Pointer to widget record for widget. */
{
    TkDial *dPtr = (TkDial *) clientData;
    Tk_Window tkwin = dPtr->tkwin;

    /*
     * This procedure could be invoked either because the window was
     * destroyed and the command was then deleted (in which case tkwin
     * is NULL) or because the command was deleted, and then this procedure
     * destroys the widget.
     */

    if (tkwin != NULL) {
	dPtr->tkwin = NULL;
	Tk_DestroyWindow(tkwin);
    }
}

/*
 *--------------------------------------------------------------
 *
 * EventuallyRedrawDial --
 *
 *	Arrange for part or all of a dial widget to redrawn at
 *	the next convenient time in the future.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	If "what" is REDRAW_DIAL then just the dial and the
 *	value readout will be redrawn;  if "what" is REDRAW_ALL
 *	then the entire widget will be redrawn.
 *
 *--------------------------------------------------------------
 */
static void
EventuallyRedrawDial(dPtr, what)
     register TkDial *dPtr;	/* Information about widget. */
     int what;			/* What to redraw:  REDRAW_DIAL
				 * or REDRAW_ALL. */
{
    if ((what == 0) || (dPtr->tkwin == NULL) || !Tk_IsMapped(dPtr->tkwin)) {
	return;
    }
    if ((dPtr->flags & REDRAW_ALL) == 0) {
	Tcl_DoWhenIdle(DisplayDial, (ClientData) dPtr);
    }
    dPtr->flags |= what;
}

/*
 *----------------------------------------------------------------------
 *
 * ComputeFormat --
 *
 *	This procedure is invoked to recompute the "format" fields
 *	of a dial's widget record, which determines how the value
 *	of the dial is converted to a string.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The format fields of dPtr are modified.
 *
 *----------------------------------------------------------------------
 */
static void
ComputeFormat(dPtr)
     TkDial *dPtr;			/* Information about dial widget. */
{
    double maxValue, x;
    int mostSigDigit, numDigits, leastSigDigit, afterDecimal;
    int eDigits, fDigits, tagLeastSigDigit;

    /*
     * Compute the displacement from the decimal of the most significant
     * digit required for any number in the dial's range.
     */

    maxValue = fabs(dPtr->fromValue);
    x = fabs(dPtr->toValue);
    if (x > maxValue) {
	maxValue = x;
    }
    if (maxValue == 0) {
	maxValue = 1;
    }
    mostSigDigit = (int)floor(log10(maxValue));

    /*
     * If the number of significant digits wasn't specified explicitly,
     * compute it. It's the difference between the most significant
     * digit needed to represent any number on the dial and the
     * most significant digit of the smallest difference between
     * numbers on the dial.  In other words, display enough digits so
     * that at least one digit will be different between any two adjacent
     * positions of the dial.
     */

    numDigits = dPtr->digits;
    if (numDigits <= 0) {
	if  (dPtr->resolution > 0) {
	    /* A resolution was specified for the dial, so just use it.  */
	    leastSigDigit = (int)floor(log10(dPtr->resolution));
	} else {
	    /*
	     * No resolution was specified, so compute the difference
	     * in value between adjacent pixels at the requested dial
	     * circumference and use it for the least significant digit.
	     */

	    x = fabs(dPtr->fromValue - dPtr->toValue);
	    if (dPtr->reqRadius > 0) {
		x /= M_PI*2*dPtr->reqRadius;
	    }
	    if (x > 0) {
		leastSigDigit = (int)floor(log10(x));
	    } else {
		leastSigDigit = 0;
	    }
	}
	numDigits = mostSigDigit - leastSigDigit + 1;
	if (numDigits < 1) {
	    numDigits = 1;
	}
    }

    /*
     * Compute the number of characters required using "e" format and
     * "f" format, and then choose whichever one takes fewer characters.
     */

    eDigits = numDigits + 4;
    if (numDigits > 1) {
	eDigits++;			/* Decimal point. */
    }
    afterDecimal = numDigits - mostSigDigit - 1;
    if (afterDecimal < 0) {
	afterDecimal = 0;
    }
    fDigits = (mostSigDigit >= 0) ? mostSigDigit + afterDecimal : afterDecimal;
    if (afterDecimal > 0) {
	fDigits++;			/* Decimal point. */
    }
    if (mostSigDigit < 0) {
	fDigits++;			/* Zero to left of decimal point. */
    }
    if (fDigits <= eDigits) {
	sprintf(dPtr->valueFormat, "%%.%df", afterDecimal);
    } else {
	sprintf(dPtr->valueFormat, "%%.%de", numDigits-1);
    }

    /*
     * Determine the format of the tag field. Each digit after the decimal
     * point should be significant.
     */
    strcpy(dPtr->tagFormat, dPtr->valueFormat);
    if (dPtr->tickInterval != 0) {
	tagLeastSigDigit = (int)floor(log10(dPtr->tickInterval));
	if (tagLeastSigDigit < 0)
	    sprintf(dPtr->tagFormat, "%%.%df", -tagLeastSigDigit);
	else
	    sprintf(dPtr->tagFormat, "%%.0f");
    }
}

/*
 *----------------------------------------------------------------------
 *
 * DialElement --
 *
 *	Determine which part of a dial widget lies under a given
 *	point.
 *
 * Results:
 *	The return value is either DIAL, LEFT, or RIGHT,
 *	depending on which of the dial's active elements
 *	(if any) is under the point at (x,y).
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static int
DialElement(dPtr, x, y)
     TkDial *dPtr;		/* Widget record for dial. */
     int x, y;			/* Coordinates within dPtr's window. */
{
    int dx, dy;

    dx = x - dPtr->centerX;
    dy = y - dPtr->centerY;
    if (dx*dx+dy*dy <= dPtr->radX*dPtr->radY) {
	return DIAL;
    } else if (dx < 0) {
	return LEFT;
    } else {
	return RIGHT;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * PixelToValue --
 *
 *	Given a pixel within a dial window, return the dial
 *	reading corresponding to that pixel.
 *
 * Results:
 *	A double-precision dial reading.  If the value is outside
 *	the legal range for the dial then it's rounded to the nearest
 *	end of the dial.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static double
PixelToValue(dPtr, x, y)
     register TkDial *dPtr;		/* Information about widget. */
     int x, y;				/* Coordinates of point within
					 * window. */
{
    double value, beginRad, endRad, angleRange, valueRange, fraction, angle;
    double center;
    int dx, dy;

    beginRad = deg2rad(dPtr->beginAngle);
    endRad = deg2rad(dPtr->endAngle);
    angleRange = endRad-beginRad;
    valueRange = dPtr->toValue - dPtr->fromValue;

    if (valueRange == 0.0 || angleRange == 0.0) {
	value = dPtr->fromValue;
    } else {
	dx = x - dPtr->centerX;
	dy = y - dPtr->centerY;
	if (dx == 0 && dy == 0) {
	    angle = 0;
	} else {
	    /* Angle will be zero at 12 o'clock and positive clockwise.  */
	    angle = atan2((double)dx,(double)-dy);
	}
	/* Make angle flip sign just opposite the center.  */
	center = (beginRad+endRad)/2;
	angle = fmod(angle-center+5*M_PI,2*M_PI)-M_PI;

	/* Angle is zero at the center. Convert to a fraction.  */
	fraction = 0.5+angle/angleRange;
	if (fraction < 0.0)
	    fraction = 0.0;
	else if (fraction > 1.0)
	    fraction = 1.0;

	value = dPtr->fromValue + fraction * valueRange;
    }

    return RoundToResolution(dPtr, value);
}

/*
 *----------------------------------------------------------------------
 *
 * ValueToAngle --
 *
 *	Given a reading of the dial, return the corresponding
 *	angle of the dial.
 *
 * Results:
 *	The angle in radians of the dial that corresponds to the reading
 *	that was specified in the call. Zero is 12 o'clock, positive is clockwise.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static double
ValueToAngle(dPtr, value)
     register TkDial *dPtr;	/* Information about widget. */
     double value;		/* Reading of the widget. */
{
    double angleRange, valueRange, fraction;

    valueRange = dPtr->toValue - dPtr->fromValue;
    if (valueRange == 0.0) {
	fraction = 0.0;
    } else {
	fraction = (value-dPtr->fromValue)/valueRange;
	if (fraction < 0.0)
	    fraction = 0.0;
	else if (fraction > 1.0)
	    fraction = 1.0;
    }

    angleRange = deg2rad(dPtr->endAngle - dPtr->beginAngle);
    return deg2rad(dPtr->beginAngle) + fraction * angleRange;
}

/*
 *----------------------------------------------------------------------
 *
 * ValueToPixel --
 *
 *	Given a reading of the dial, return the x and
 *	y-coordinate corresponding to that reading.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The integers whose pointers are passed are set to a value giving
 *	the pixel location corresponding
 *	to reading.  The value is restricted to lie within the
 *	defined range for the dial.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static void
ValueToPixel(dPtr, value, xPtr, yPtr)
     register TkDial *dPtr;	/* Information about widget. */
     double value;		/* Reading of the widget. */
     int *xPtr, *yPtr;		/* Integers to store the result. */
{
    double angle;

    angle = ValueToAngle(dPtr, value);
    *xPtr = (int) (dPtr->centerX + dPtr->radX * sin(angle));
    *yPtr = (int) (dPtr->centerY + dPtr->radY * cos(angle));
}

/*
 *--------------------------------------------------------------
 *
 * SetDialValue --
 *
 *	This procedure changes the value of a dial and invokes
 *	a Tcl command to reflect the current position of a dial
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	A Tcl command is invoked, and an additional error-processing
 *	command may also be invoked.  The dial's slider is redrawn.
 *
 *--------------------------------------------------------------
 */
static void
SetDialValue(dPtr, value, setVar, invokeCommand)
     register TkDial *dPtr;	/* Info about widget. */
     double value;		/* New value for dial.  Gets adjusted
				 * if it's off the dial. */
     int setVar;		/* Non-zero means reflect new value through
				 * to associated variable, if any. */
     int invokeCommand;		/* Non-zero means invoked -command option
				 * to notify of new value, 0 means don't. */
{
    char string[TCL_DOUBLE_SPACE];

    /* (real value == displayed value) iff (dPtr->constrain == 1) */
    if (dPtr->constrain) {
	value = ConstrainDialValue(dPtr, value);
    }
    if (dPtr->flags & NEVER_SET) {
	dPtr->flags &= ~NEVER_SET;
    } else if (dPtr->value == value) {
	return;
    }
    dPtr->value = value;
    if (invokeCommand) {
	/*
	 * This is done in the DisplayDial routine to emulate the scale
	 * although that seems inappropriate...
	 */
	dPtr->flags |= INVOKE_COMMAND;
    }
    EventuallyRedrawDial(dPtr, REDRAW_DIAL);

    if (setVar && (dPtr->varName != NULL)) {
	sprintf(string, (dPtr->constrain) ? dPtr->valueFormat : "%g",
		dPtr->value);
	dPtr->flags |= SETTING_VAR;
	Tcl_SetVar(dPtr->interp, dPtr->varName, string, TCL_GLOBAL_ONLY);
	dPtr->flags &= ~SETTING_VAR;
    }
}

/*
 *--------------------------------------------------------------
 *
 * RoundToResolution --
 *
 *	Round a given floating-point value to the nearest multiple
 *	of the dial's resolution.
 *
 * Results:
 *	The return value is the rounded result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static double
RoundToResolution(dPtr, value)
     TkDial *dPtr;	/* Information about dial widget. */
     double value;	/* Value to round. */
{
    double rem;

    if (dPtr->resolution <= 0) {
	return value;
    }
    rem = fmod(value, dPtr->resolution);
    if (rem < 0) {
	rem = dPtr->resolution + rem;
    }
    value -= rem;
    if (rem >= dPtr->resolution/2) {
	value += dPtr->resolution;
    }
    return value;
}

static double
ConstrainDialValue(dPtr, value)
     TkDial *dPtr;	/* Information about dial widget. */
     double value;	/* Value to round. */
{
    value = RoundToResolution(dPtr, value);
    if ((value < dPtr->fromValue) ^ (dPtr->toValue < dPtr->fromValue)) {
	value = dPtr->fromValue;
    }
    if ((value > dPtr->toValue) ^ (dPtr->toValue < dPtr->fromValue)) {
	value = dPtr->toValue;
    }
    return value;
}

/*
 *----------------------------------------------------------------------
 *
 * DialVarProc --
 *
 *	This procedure is invoked by Tcl whenever someone modifies a
 *	variable associated with a dial widget.
 *
 * Results:
 *	NULL is always returned.
 *
 * Side effects:
 *	The value displayed in the dial will change to match the
 *	variable's new value.  If the variable has a bogus value then
 *	it is reset to the value of the dial.
 *
 *----------------------------------------------------------------------
 */
static char *
DialVarProc(clientData, interp, name1, name2, flags)
     ClientData clientData;	/* Information about button. */
     Tcl_Interp *interp;	/* Interpreter containing variable. */
     char *name1;		/* Name of variable. */
     char *name2;		/* Second part of variable name. */
     int flags;			/* Information about what happened. */
{
    register TkDial *dPtr = (TkDial *) clientData;
    CONST84 char *stringValue;
    char *end, *result;
    double value;

    /*
     * If the variable is unset, then immediately recreate it unless
     * the whole interpreter is going away.
     */

    if (flags & TCL_TRACE_UNSETS) {
	if ((flags & TCL_TRACE_DESTROYED) && !(flags & TCL_INTERP_DESTROYED)) {
	    Tcl_TraceVar(interp, dPtr->varName,
			 TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
			 (Tcl_VarTraceProc *)DialVarProc, clientData);
	    dPtr->flags |= NEVER_SET;
	    SetDialValue(dPtr, dPtr->value, 1, 0);
	}
	return (char *) NULL;
    }

    /*
     * If we came here because we updated the variable (in SetDialValue),
     * then ignore the trace.  Otherwise update the dial with the value
     * of the variable.
     */

    if (dPtr->flags & SETTING_VAR) {
	return (char *) NULL;
    }
    result = NULL;
    stringValue = Tcl_GetVar(interp, dPtr->varName, TCL_GLOBAL_ONLY);
    if (stringValue != NULL) {
	value = strtod(stringValue, &end);
	if ((end == stringValue) || (*end != 0)) {
	    result = "can't assign non-numeric value to dial variable";
	} else {
	    dPtr->value = value;
	}

	/*
	 * This code is a bit tricky because it sets the dial's value before
	 * calling SetDialValue.  This way, SetDialValue won't bother to
	 * set the variable again or to invoke the -command.  However, it
	 * also won't redisplay the dial, so we have to ask for that
	 * explicitly.
	 */

	SetDialValue(dPtr, dPtr->value, 1, 0);
	EventuallyRedrawDial(dPtr, REDRAW_DIAL);
    }

    return result;
}
