# man2html-pdffix.tcl
#
#       Updates HTML files generated by tix-man2html.tcl so that
#       they can be passed to HTMLDOC to generate better PDF files.
#
#       This program is dependent on the specific HTML format
#       generated by ./tix-man2html.tcl.
#
#       This program is usually invoked by ../man/GNUmakefile.
#
# Copyright (c) 2000-2001 Tix Project Group.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
# $Id: man2html-pdffix.tcl,v 1.1 2001/02/09 06:45:22 ioilam Exp $

if {[llength $argv] != 1} {
    puts stderr "Usage:\n    $argv0 <htmldir>"
    exit 1
}

set tcltkdirs {TclCmd TkCmd TclLib TkLib UserCmd}
set alldirs [concat $tcltkdirs TixCmd]

set appdx_title(TclCmd) {Tcl Commands}
set appdx_title(TkCmd)  {Tk Commands}

set appdx_title(TclLib) {Tcl Library References}
set appdx_title(TkLib)  {Tk Library References}
set appdx_title(UserCmd) {Tcl/Tk Applications}

set root [lindex $argv 0]
set filesfd [open [file join $root files.txt] w]

puts $filesfd [file join $root about.htm]
puts $filesfd [file join $root copyright.htm]

set copyexp "<HR><PRE>\n<A HREF=\"\[^\"\]*copyright.htm\">.*</PRE>"

#
# [1] Create the list of all Tix commands, separated by categories
#
source [file join [file dirname [info script]] cmdcategory.tcl]

set chapter 0
foreach cat $::cmdcategory::categories {
    set catfile [file join $root $cat.htm]
    set catfd [open $catfile w]
    incr chapter
    puts -nonewline $catfd {
        <HTML>
        <TITLE>Tix Reference Manual</TITLE>
        <BODY>
        <H1>Chapter}
    puts -nonewline $catfd " $chapter: "
    puts -nonewline $catfd [string trim $::cmdcategory::title($cat)]
    
    puts -nonewline $catfd {</H1>
        </BODY>
        </HTML>
    }
    close $catfd

    puts $filesfd $catfile
    foreach cmd [lsort -dictionary $::cmdcategory::cmds($cat)] {
        foreach dir {TixCmd UserCmd} {
            set file [file join $root $dir $cmd.htm]
            if {[file exists $file]} {
                puts $filesfd $file
                break
            }
        }
    }
}

#
# [2] List of Tcl/Tk man pages
#

set appendix 1
foreach dir $tcltkdirs {
    set list [glob -nocomplain [file join $root $dir *.htm]]
    if {[llength $list] < 2} {
        continue
    }
    if {![regexp {contents[.]htm} $list]} {
        continue
    }
    if {[regexp {tixwish[.]htm} $list] && [llength $list] < 3} {
        # Don't want to include tixwish.htm into the appendix. It's
        # already included above

        continue
    }

    set catfile [file join $root $dir.htm]
    set catfd [open $catfile w]
    puts -nonewline $catfd {
        <HTML>
        <TITLE>Tix Reference Manual</TITLE>
        <BODY>
        <H1>Appendix}
    puts -nonewline $catfd " $appendix: "
    puts -nonewline $catfd $appdx_title($dir)
    puts -nonewline $catfd {</H1>}

    incr appendix

    if {![info exists env(WITH_TCL_TK)]} {
        puts $catfd "This chapter includes a subset of the $appdx_title($dir)"
        puts $catfd {
            that are frequently mentioned in the Tix documentation.
            Please consult your Tcl/Tk documentation for other Tcl/Tk
            references.
        }
    }

    puts -nonewline $catfd {
        </BODY>
        </HTML>
    }
    close $catfd

    puts $filesfd $catfile

#   puts $filesfd [file join $root $dir contents.htm]

    foreach file [lsort -dict $list] {
        if {[string match *contents.htm $file]} {
            continue
        }
        if {[string match *tixwish.htm $file]} {
            continue
        }
        puts $filesfd $file
    }
}

#
# [3] List of keyword pages (as well as clean up)
#

set list [glob -nocomplain [file join $root Keywords {[A-Z].htm}]]
if {[llength $list] > 0} {
    set catfile [file join $root Keywords.htm]
    set fd [open $catfile w]
    puts $fd "
        <HTML><HEAD><TITLE>Keywords</TITLE></HEAD>
        <BODY>
        <H1>Appendix $appendix: Keywords</H1>
        &nbsp;<br>
        &nbsp;<br>
        &nbsp;<br>
        &nbsp;<br>
        </BODY></HTML>"
    incr appendix
    close $fd

    puts $filesfd $catfile

    foreach file [lsort -dict $list] {
        puts $filesfd $file
        set fd [open $file]
        set data [read $fd]
        close $fd

        regsub "<TITLE>.*</TITLE>" $data "" data
        regsub "<H2>.*</H2><HR><DL>" $data "<DL>" data
        regsub $copyexp $data "" data

        set titleexp {<HR><H3>Tcl/Tk Keywords - ([A-Z])</H3><HR>}

        if {[regexp <DT> $data]} {
            regsub $titleexp $data "<H2>\\1</H2>" data
        } else {
            regsub $titleexp $data "" data
            regsub <DL>.*</DL> $data "" data
        }

        set fd [open $file w]
        puts -nonewline $fd  $data
        close $fd
    }
}



#
# [4] Clean up the HTML pages for better PDF conversion -- remove
#     the summary section at the top and the copyright notices at the
#     bottom.
#

close $filesfd

foreach dir $alldirs {
    puts $dir
    if {![file exist [file join $root $dir]]} {
        continue
    }

    foreach file [glob -nocomplain [file join $root $dir *.htm]] {
        set fd [open $file]
        set data [read $fd]
        close $fd

        if {"[file tail $file]" == "contents.htm"} {
            set headexp "<HR><H3>(\[^<\]+)</H3><HR>"
            set replace ""
            if {![regsub $headexp $data $replace data]} {
                #puts "$file -- already fixed"
                #continue
            }
        } else {
            if {[regexp "<H2><A NAME=\"M2\">" $data]} {
                #puts "$file -- already fixed"
                #continue
            } else {
                set line1 "<H3><A NAME=\"M2\">NAME</A></H3>"
                set line2 "<BLOCKQUOTE>"
                set line3 "(\[^\n\]+)"
                set line4 "</BLOCKQUOTE>"
                set headexp0 "$line1\n$line2\n$line3\n$line4"
                set headexp  "<DL>.*$headexp0"
                set b "\\&nbsp;<br>"
                set replace  "$b$b$b$b<H2><A NAME=\"M2\">\\1</A></H2><hr>"

                if {![regsub $headexp $data $replace data]} {
                    if {![regsub $headexp0 $data $replace data]} {
                        puts "$file *** missing header"
                        continue
                    }
                }
            }
        }

        #
        # Trim title so that it's not too long. The title of
        # a mane page is displayed at the footer of the PDF file
        #

        ## TODO: don't know how to fix this for HTMLDOC yet
        ## instead, I just disabled the footer in ../man/GNUmakefile.

        # Remove copyright. Copyright notices are included as
        # a separate section of the PDF file.
        #

        regsub $copyexp $data "" data

        #puts "$file -- done --"
        set fd [open $file w]
        puts -nonewline $fd  $data
        close $fd
    }
}
