# -*- tcl -*-
# TclBench join - Knows the special column of csv files generated by
# tclbench: Removes the id column from the join. The key column is
# fixed to the 'test description', column 1 (after the id).

# ------------------------------------------------------

package require tools
package require struct
package require optcsv
package require csv

namespace eval app-tbjoin {}

# ------------------------------------------------------
# Acceptable syntax for cmdline functionality

set ::app-tbjoin::help(cmdline) {
[call [cmd {@appname@}] [method tbjoin] [opt "[option -sep] [arg sepChar]"] [arg file]...]

This is essentially the [cmd join] command, but with special knowledge
about the organization of csv files generated by [syscmd tclbench].

[nl]

Because of that

[list_begin bullet]
[bullet]
Column 0 of all input files is ignored as it only contains a running row id of no consequence.

[bullet]
The key column is hardwired to column 1, containing the description of the tests.

[bullet]

Column 0 in the output is not the key column, but an auto-generated
column of row ids. The key column is column 1.
[list_end]

}

proc ::app-tbjoin::help {topic} {
    variable help
    if {[info exists help($topic)]} {return $help($topic)}
    return {}
}

# ------------------------------------------------------
# Implementation of cmdline functionality.

proc ::app-tbjoin::run {argv} {
    set errstring "wrong#args: join ?-sep char? file..."

    # ==================================================
    # Process arguments

    if {[llength $argv] < 1} {tools::usage $errstring}

    set sepChar ,
    set alt     0

    while {[llength $argv]} {
        set err [cmdline::getopt argv {sep.arg} opt arg]
	if {$err == 1} {
	    switch -exact -- $opt {
		sep       {set sepChar $arg}
	    }
	} elseif {$err < 0} {
	    tools::usage $errstring
	} else {
	    # Non argument found, stop processing.
	    break
	}
    }

    if {[llength $argv] < 1} {tools::usage $errstring}

    ::optcsv::files [set files $argv] $errstring

    # ==================================================
    # Process csv input

    # ==================================================
    # Special case: Only one file, this is identity.

    if {[llength $files] == 1} {
	::optcsv::foreach_record $sepChar 0 data $files {
	    puts stdout [::csv::join $data $sepChar]
	}
	return
    }

    # ==================================================
    # Multiple files ... Read them all, then invoke the
    # appropriate list operation. We are using keyed tables.
    # We keep the key column for the first table. We do a full
    # join to get every key ever used in a benchmark.

    set reqlen 2
    set rkeys  {}
    set cmd    [list ::struct::list dbJoinKeyed -full -keys rkeys]

    foreach f $files {
	set bwidth 0
	set ktable [list]
	::optcsv::foreach_record $sepChar $alt data [list $f] {
	    if {[llength $data] < $reqlen} {
		tools::usage "\"$f\": \n\tNeed at least $reqlen columns in all records,\
			got a record with only [llength $data] columns"
	    }
	    # Trick: Get keys out of row, keep only data.

	    # Skip keys, and id column, in data.
	    lappend ktable [list [lindex $data 1] [lrange $data 2 end]]

	    if {$bwidth < [set bw [llength $data]]} {set bwidth $bw}
	}
	incr bwidth -2; # No key, no id columns.
	lappend cmd [::optcsv::fill $bwidth $ktable]
	set first 0
    }

    set joinedtable [eval $cmd]

    # ==================================================
    # Merge result and keys. This allow us to sort things when writing.

    set res [list]
    foreach row $joinedtable key $rkeys {
	lappend res [linsert $row 0 $key]
    }
    set joinedtable $res ; unset res rkeys

    # ==================================================
    # iii. Write result

    set id 0
    foreach row [lsort $joinedtable] {
	puts stdout [::csv::join [linsert $row 0 [incr id]] $sepChar]
    }
    return
}

# ------------------------------------------------------
package provide app-tbjoin 0.1
