
/*
 * bltUnixFont.c --
 *
 * This module implements Xft fonts for the BLT toolkit.
 *
 *	Copyright 2005 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "bltInt.h"

#include "tkDisplay.h"
#include "tkFont.h"

#ifdef HAVE_LIBXFT
#include <ft2build.h>
#include FT_FREETYPE_H
#include <X11/Xft/Xft.h>

#define DEBUG_FONT_SELECTION 0

#define FONT_UNKNOWN	0
#define FONT_TK		1
#define FONT_XFT	2

typedef struct {
    char *name;			/* Name of the font (malloc-ed). */
    int refCount;		/* Reference count for this structure.
				 * When refCount reaches zero, it
				 * means to free the resources
				 * associated with this structure. */

    Blt_HashEntry *hashPtr;	/* Pointer to this entry in global
				 * font hash table.  Used to remove
				 * the entry from the table.  */

    Font fid;			/* Font id used to fake out Tk_FontId. */

    XftPattern *pattern;	/* Pattern matching the current
				 * non-rotated font. Used to create
				 * rotated fonts by duplicating the
				 * pattern and adding a rotation
				 * matrix. */

    Blt_HashTable xftFontTable;	/* Hash table containing an Xft font
				 * for each angle it's used at. Will
				 * always contain a 0 degree entry. */

    /* Information specific to the display/drawable being used. The
     * drawables are changed as the drawable changes for each drawing
     * request.  Typically this will change for each pixmap. */

    Drawable drawable;		/* Drawable associated with draw. */
    XftDraw *draw;		/* Current Xft drawable. */
    int drawDepth;		/* Depth of current drawable. */

    XftColor color;		/* Color to be displayed.  We don't
				 * actually allocate this color, since
				 * we assume it's been already
				 * allocated by the standard Tk
				 * procedures. */

    /* Saved Information from the Tk_Window used to created the
     * initial font. */
    Display *display;		
    Visual *visual;
    int screenNum;
    Colormap colormap;

} FontInfo;

struct Blt_FontStruct {
    Tk_Font tkFont;
    FontInfo *infoPtr;
    int type;			/* Indicates the type of font used. */
};

typedef struct {
    char *name;
    int minChars;
    char *key;
    int value;
} FontSpec;
    
static FontSpec fontSpecs[] ={
    { "black",		2, XFT_WEIGHT,		XFT_WEIGHT_BLACK	},
    { "bold",		3, XFT_WEIGHT,		XFT_WEIGHT_BOLD,	},
    { "book",		3, XFT_WEIGHT,		XFT_WEIGHT_MEDIUM,	},
    { "demi",		4, XFT_WEIGHT,		XFT_WEIGHT_BOLD,	},
    { "demibold",	5, XFT_WEIGHT,		XFT_WEIGHT_DEMIBOLD,	},
    { "i",		1, XFT_SLANT,		XFT_SLANT_ITALIC,	},
    { "italic",		2, XFT_SLANT,		XFT_SLANT_ITALIC,	},
    { "light",		1, XFT_WEIGHT,		XFT_WEIGHT_LIGHT,	},
    { "medium",		2, XFT_WEIGHT,		XFT_WEIGHT_MEDIUM,	},
    { "mono",		2, XFT_SPACING,		XFT_MONO,		},
    { "normal",		3, XFT_WEIGHT,		XFT_WEIGHT_MEDIUM,	},
    { "o",		1, XFT_SLANT,		XFT_SLANT_OBLIQUE,	},
    { "obilque",	3, XFT_SLANT,		XFT_SLANT_OBLIQUE,	},
    { "overstrike",	3, NULL,		0			},
    { "proportional",	3, XFT_SPACING,		XFT_PROPORTIONAL,	},
    { "r",		1, XFT_SLANT,		XFT_SLANT_ROMAN,	},
    { "roman",		2, XFT_SLANT,		XFT_SLANT_ROMAN,	},
    { "underline",	1, NULL,		0,			},
};
static int nFontSpecs = sizeof(fontSpecs) / sizeof(FontSpec);

static FontSpec weightSpecs[] ={
    { "black",		2, XFT_WEIGHT,		XFT_WEIGHT_BLACK	},
    { "bold",		3, XFT_WEIGHT,		XFT_WEIGHT_BOLD,	},
    { "book",		3, XFT_WEIGHT,		XFT_WEIGHT_MEDIUM,	},
    { "demi",		4, XFT_WEIGHT,		XFT_WEIGHT_BOLD,	},
    { "demibold",	5, XFT_WEIGHT,		XFT_WEIGHT_DEMIBOLD,	},
    { "light",		1, XFT_WEIGHT,		XFT_WEIGHT_LIGHT,	},
    { "medium",		1, XFT_WEIGHT,		XFT_WEIGHT_MEDIUM,	},
    { "normal",		1, XFT_WEIGHT,		XFT_WEIGHT_MEDIUM,	},
};
static int nWeightSpecs = sizeof(weightSpecs) / sizeof(FontSpec);

static FontSpec slantSpecs[] ={
    { "i",		1, XFT_SLANT,		XFT_SLANT_ITALIC,	},
    { "italic",		2, XFT_SLANT,		XFT_SLANT_ITALIC,	},
    { "o",		1, XFT_SLANT,		XFT_SLANT_OBLIQUE,	},
    { "obilque",	3, XFT_SLANT,		XFT_SLANT_OBLIQUE,	},
    { "r",		1, XFT_SLANT,		XFT_SLANT_ROMAN,	},
    { "roman",		2, XFT_SLANT,		XFT_SLANT_ROMAN,	},
};
static int nSlantSpecs = sizeof(slantSpecs) / sizeof(FontSpec);

static FontSpec widthSpecs[] ={
    { "condensed",	1, FC_WIDTH,		FC_WIDTH_CONDENSED,	},
    { "expanded",	3, FC_WIDTH,		FC_WIDTH_EXPANDED,	},
    { "extracondensed", 6, FC_WIDTH,		FC_WIDTH_EXTRACONDENSED,},
    { "extraexpanded",	6, FC_WIDTH,		FC_WIDTH_EXTRAEXPANDED,	},
    { "normal",		1, FC_WIDTH,		FC_WIDTH_NORMAL,	},
    { "semicondensed",	5, FC_WIDTH,		FC_WIDTH_SEMICONDENSED,	},
    { "semiexpanded",	5, FC_WIDTH,		FC_WIDTH_SEMIEXPANDED,	},
    { "ultracondensed",	6, FC_WIDTH,		FC_WIDTH_ULTRACONDENSED,},
    { "ultraexpanded",	6, FC_WIDTH,		FC_WIDTH_ULTRAEXPANDED,	},
};

static int nWidthSpecs = sizeof(widthSpecs) / sizeof(FontSpec);

static int haveXRenderExtension;
static int initialized = 0;
static Blt_HashTable fontTable;

static void
CheckForXRenderExtension()
{
    Blt_InitHashTable(&fontTable, BLT_STRING_KEYS);
    haveXRenderExtension = 1;
}

static int
CanRotateXftFont(FontInfo *infoPtr, float angle)
{
    XftPattern *match, *pattern;
    XftResult result;
    int boolean;
    long angle10;

    angle10 = (int)(angle * 10.0);
    if (Blt_FindHashEntry(&infoPtr->xftFontTable, (char *)angle10) != NULL) {
	return TRUE;		/* Rotated font already exists. */
    }

    /* 
     * I don't know if this is correct.  Some PCF fonts don't rotate
     * properly.  The chararcter positions are rotated but the glyph
     * itself is drawn with no rotation.  The standard Adobe Helvetica
     * font is a good example of this.  So I need to bail on those
     * fonts.  I check if antialias=True in the Xft font pattern to
     * determine if the font will rotate properly.
     */
    if (XftPatternGetBool(infoPtr->pattern, XFT_ANTIALIAS, 0, &boolean) 
	== XftResultMatch) {
 	if (!boolean) {
  	    return FALSE;
 	 }
    }

    {
	XftMatrix matrix;
	double cosTheta, sinTheta, theta;

	theta = ((double)angle / 180.0) * M_PI;
	sinTheta = sin(theta);
	cosTheta = cos(theta);
	
	XftMatrixInit(&matrix);
	XftMatrixRotate(&matrix, cosTheta, sinTheta);
	pattern = XftPatternDuplicate(infoPtr->pattern);
	XftPatternAddMatrix(pattern, XFT_MATRIX, &matrix);
    }

    /* XftFontMatch only sets *result* on complete match failures.
     * So initialize it here for a successful match. We'll accept
     * partial matches. */
    result = XftResultMatch; 
    match = XftFontMatch(infoPtr->display, infoPtr->screenNum, pattern, 
		&result);
    if ((match != NULL) && (result == XftResultMatch)) {
	XftFont *fontPtr;
	
	fontPtr = XftFontOpenPattern(infoPtr->display, match);
	if (fontPtr != NULL) {
	    Blt_HashEntry *hPtr;
	    int isNew;
	    long angle10;

	    angle10 = (long)(angle * 10.0);
	    hPtr = Blt_CreateHashEntry(&infoPtr->xftFontTable, (char *)angle10, 
		&isNew);
	    assert((hPtr != NULL) && (isNew));
	    Blt_SetHashValue(hPtr, fontPtr);
	    XftPatternDestroy(pattern);
	    return TRUE;
	}
    }
    XftPatternDestroy(pattern);
    return FALSE;
}

/*
 *----------------------------------------------------------------------
 *
 * SearchForFontSpec --
 *
 *      Performs a binary search on the array of font specification to
 *      find a partial, anchored match for the given option string.
 *
 * Results:
 *	If the string matches unambiguously the index of the specification
 *	in the array is returned.  If the string does not match, even
 *	as an abbreviation, any operation, -1 is returned.  If the string
 *	matches, but ambiguously -2 is returned.
 *
 *----------------------------------------------------------------------
 */
static int
SearchForFontSpec(
    FontSpec *table,		/* Table of font options.  */
    int nSpecs,			/* # specifications in font spec table. */
    CONST char *string)		/* Name of font option to search for. */
{
    char c;
    int high, low;
    size_t length;

    low = 0;
    high = nSpecs - 1;
    c = string[0];
    length = strlen(string);
    while (low <= high) {
	FontSpec *sp;
	int compare;
	int median;
	
	median = (low + high) >> 1;
	sp = table + median;

	/* Test the first character */
	compare = c - sp->name[0];
	if (compare == 0) {
	    /* Now test the entire string */
	    compare = strncasecmp(string, sp->name, length);
	    if (compare == 0) {
		if ((int)length < sp->minChars) {
		    return -2;	/* Ambiguous spec name */
		}
	    }
	}
	if (compare < 0) {
	    high = median - 1;
	} else if (compare > 0) {
	    low = median + 1;
	} else {
	    return median;	/* Spec found. */
	}
    }
    return -1;			/* Can't find spec */
}

static FontSpec *
FindSpec(FontSpec *tablePtr, int nSpecs, CONST char *string)
{
    int n;
    
    n = SearchForFontSpec(tablePtr, nSpecs, string);
    if (n < 0) {
	if (n == -1) {
	    fprintf(stderr, "can't find font specification \"%s\"'\n", string);
	}
	if (n == -2) {
	    fprintf(stderr, "ambigous font specification \"%s\"'\n", string);
	}
	return NULL;
    }
    return tablePtr + n;
}


/*
 *---------------------------------------------------------------------------
 *
 * ParseTkDesc --
 *
 *	Try to open a Xft font from an Tk style font description.
 *
 * Results:
 *	Return value is TCL_ERROR if string was not a fully specified XLFD.
 *	Otherwise, fills font attribute buffer with the values parsed
 *	from the XLFD and returns TCL_OK.  
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */
static XftPattern *
ParseTkDesc(int objc, Tcl_Obj **objv)
{
    XftPattern *pattern;
    double size;
    int i;

    pattern = XftPatternCreate();

    /* Font family. */
    XftPatternAddString(pattern, XFT_FAMILY, Tcl_GetString(objv[0]));

    /* Size */
    size = 12.0;
    if ((objc > 1) && (Tcl_GetDoubleFromObj(NULL, objv[1], &size) != TCL_OK)) {
	goto error;
    }
    XftPatternAddDouble(pattern, XFT_SIZE, size);
    i = 2;
    if (objc == 3) {
	if (Tcl_ListObjGetElements(NULL, objv[2], &objc, &objv) != TCL_OK) {
	    goto error;
	}
	i = 0;
    }
    for (/*empty*/; i < objc; i++) {
	FontSpec *specPtr;
	
	specPtr = FindSpec(fontSpecs, nFontSpecs, Tcl_GetString(objv[i]));
	if (specPtr == NULL) {
	    goto error;
	}
	if (specPtr->key != NULL) {
	    XftPatternAddInteger(pattern, specPtr->key, specPtr->value);
	}
    }
#if DEBUG_FONT_SELECTION
    fprintf(stderr, "parsed TkDesc-XFT font \"%s\"\n", Tcl_GetString(objv[0]));
#endif
    return pattern;
 error:
    if (pattern != NULL) {
	XftPatternDestroy(pattern);
    }
    return NULL;
}	

static XftPattern *
ParseTkFontAttributeList(Tcl_Interp *interp, Tcl_Obj *objPtr) 
{
    XftPattern *pattern;
    Tcl_Obj **objv;
    int objc;
    int i;

    if ((Tcl_ListObjGetElements(NULL, objPtr, &objc, &objv) != TCL_OK) ||
	(objc < 1)) {
	return NULL;		/* Can't split list or list is empty. */
    }
    if (objc & 0x1) {
	return NULL;		/* Odd number of elements in list. */
    }
    pattern = XftPatternCreate();
    for (i = 0; i < objc; i += 2) {
	char *key, *value;

	key = Tcl_GetString(objv[i]);
	value = Tcl_GetString(objv[i+1]);
	if (strcmp(key, "-family") == 0) {
	    XftPatternAddString(pattern, XFT_FAMILY, value);
	} else if (strcmp(key, "-size") == 0) {
	    double size;

	    if (Tcl_GetDoubleFromObj(interp, objv[i+1], &size) != TCL_OK) {
		goto error;
	    }
	    XftPatternAddDouble(pattern, XFT_SIZE, size);
	} else if (strcmp(key, "-weight") == 0) {
	    FontSpec *specPtr;

	    specPtr = FindSpec(weightSpecs, nWeightSpecs, value);
	    if (specPtr == NULL) {
		goto error;
	    }
	    XftPatternAddInteger(pattern, FC_WEIGHT, specPtr->value);
	} else if (strcmp(key, "-slant") == 0) {
	    FontSpec *specPtr;

	    specPtr = FindSpec(slantSpecs, nSlantSpecs, value);
	    if (specPtr == NULL) {
		goto error;
	    }
	    XftPatternAddInteger(pattern, XFT_SLANT, specPtr->value);
	} else if (strcmp(key, "-underline") == 0) {
	    /* Ignore */
	} else if (strcmp(key, "-overstrike") == 0) {
	    /* Ignore */
	} else {
	    goto error;
	}
    }
#if DEBUG_FONT_SELECTION
    fprintf(stderr, "parsed TkAttrList-XFT font \"%s\"\n", 
	    Tcl_GetString(objPtr));
#endif
    return pattern;
 error:
    XftPatternDestroy(pattern);
    return NULL;
}

static XftPattern *
ParseTkNamedFont(Tcl_Interp *interp, Tcl_Obj *objPtr) 
{
    XftPattern *pattern;
    Tcl_Obj *cmdObjv[3];

    cmdObjv[0] = Tcl_NewStringObj("font", -1);
    cmdObjv[1] = Tcl_NewStringObj("configure", -1);
    cmdObjv[2] = objPtr;
    if (Tcl_EvalObjv(interp, 3, cmdObjv, 0) != TCL_OK) {
	return NULL;
    }
    pattern = ParseTkFontAttributeList(interp, Tcl_GetObjResult(interp));
    Tcl_ResetResult(interp);
#if DEBUG_FONT_SELECTION
    if (pattern) {
	fprintf(stderr, "parsed TkFontObject-XFT font \"%s\"\n", 
	    Tcl_GetString(objPtr));
    }
#endif
    return pattern;
}

/*
 *---------------------------------------------------------------------------
 *
 * ParseXlfdDesc --
 *
 *	Try to open a Xft font from an XLFD description.
 *
 * Results:
 *	Return value is TCL_ERROR if string was not a fully specified XLFD.
 *	Otherwise, fills font attribute buffer with the values parsed
 *	from the XLFD and returns TCL_OK.  
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */

static XftPattern *
ParseXlfdDesc(Tk_Window tkwin, char *fontName)
{
    char *string, *word;
    int count;
    XftPattern *pattern;
    FontSpec *specPtr;

    enum XlfdFields { 
	XLFD_FOUNDRY, XLFD_FAMILY, XLFD_WEIGHT, XLFD_SLANT,	
	XLFD_SETWIDTH, XLFD_ADD_STYLE, XLFD_PIXEL_SIZE,
	XLFD_POINT_SIZE, XLFD_RESOLUTION_X, XLFD_RESOLUTION_Y,
	XLFD_SPACING, XLFD_AVERAGE_WIDTH, XLFD_CHARSET
    };

    if (*fontName == '-') {
	fontName++;
    }
    string = Blt_Strdup(fontName);
    count = 0;
    pattern = XftPatternCreate();

    /* Add the default font size */
    XftPatternAddInteger(pattern, XFT_SIZE, 12);

    for (word = strtok(string, "-"); word != NULL; word = strtok(NULL, "-")) {
	char *p;

	for (p = word; *p != '\0'; p++) {
	    if (!(*p & 0x80) && Tcl_UniCharIsUpper(UCHAR(*p))) {
		*p = (char)Tcl_UniCharToLower(UCHAR(*p));
	    }
	}
	count++;
	if ((*word == '\0') || (*word == '*') || (*word == '?')) {
	    continue;		/* Field not specified. */
	}
	switch (count - 1) {
	case XLFD_FOUNDRY:
	    XftPatternAddString(pattern, XFT_FOUNDRY, word);
	    break;

	case XLFD_FAMILY:
	    XftPatternAddString(pattern, XFT_FAMILY, word);
	    break;

	case XLFD_WEIGHT:
	    specPtr = FindSpec(weightSpecs, nWeightSpecs, word);
	    if (specPtr == NULL) {
		goto error;
	    }
	    XftPatternAddInteger(pattern, FC_WEIGHT, specPtr->value);
	    break;

	case XLFD_SLANT:
	    specPtr = FindSpec(slantSpecs, nSlantSpecs, word);
	    if (specPtr == NULL) {
		goto error;
	    }
	    XftPatternAddInteger(pattern, XFT_SLANT, specPtr->value);
	    break;

	case XLFD_SETWIDTH:
	    specPtr = FindSpec(widthSpecs, nWidthSpecs, word);
	    if (specPtr == NULL) {
		goto error;
	    }
	    XftPatternAddInteger(pattern, FC_WIDTH, specPtr->value);

	case XLFD_ADD_STYLE:
	    /*
	     * An XLFD of the form -adobe-times-medium-r-*-12-*-* is
	     * pretty common, but it is (strictly) malformed, because
	     * the first * is eliding both the Setwidth and the
	     * Addstyle fields.  If the Addstyle field is a number,
	     * then assume the above incorrect form was used and shift
	     * all the rest of the fields right by one, so the number
	     * gets interpreted as a pixelsize.  This fix is so that
	     * we don't get a million reports that "it works under X
	     * (as a native font name), but gives a syntax error under
	     * Windows (as a parsed set of attributes)".
	     */
	    if (atoi(word) != 0) {
		count++;
	    }
	    XftPatternAddString(pattern, XFT_STYLE, word);
	    break;

	case XLFD_PIXEL_SIZE:
	case XLFD_POINT_SIZE:
	    { 
		int size;

		if (*word == '[') {
		    /*
		     * Some X fonts have the pixel size specified as follows:
		     *
		     *	    [ N1 N2 N3 N4 ]
		     *
		     * where N1 is the pixel size, and where N2, N3, and N4 
		     * are some additional numbers that I don't know
		     * the purpose of, so I ignore them.
		     */
		    word++;
		}
		if (Tcl_GetInt(NULL, word, &size) != TCL_OK) {
		    goto error;
		}
		if (count == XLFD_PIXEL_SIZE) {
		    XftPatternAddInteger(pattern, XFT_PIXEL_SIZE, size);
		} else {
		    XftPatternAddInteger(pattern, XFT_SIZE, size);
		}
	    }
	    break;

	case XLFD_RESOLUTION_X:
	case XLFD_RESOLUTION_Y:
	case XLFD_SPACING:
	case XLFD_AVERAGE_WIDTH:
	    /* Ignored, just like in Tk. */
	    break;

	case XLFD_CHARSET:
	    XftPatternAddString(pattern, FC_CHARSET, word);
	    break;
	}
    }
    Blt_Free(string);
#if DEBUG_FONT_SELECTION
    fprintf(stderr, "parsed Xlfd-XFT font \"%s\"\n", fontName);
#endif
    return pattern;
 error:
    fprintf(stderr, "can't open font \"%s\" as XLFD\n", fontName);
    Blt_Free(string);
    if (pattern != NULL) {
	XftPatternDestroy(pattern);
    }
    return NULL;
}


static void
DestroyXftFontInfo(FontInfo *infoPtr)
{
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;
    
    for (hPtr = Blt_FirstHashEntry(&infoPtr->xftFontTable, &cursor); 
	 hPtr != NULL; hPtr = Blt_NextHashEntry(&cursor)) {
	XftFont *fontPtr;
	
	fontPtr = Blt_GetHashValue(hPtr);
	XftFontClose(infoPtr->display, fontPtr);
    }
    Blt_DeleteHashTable(&infoPtr->xftFontTable);
    
    if (infoPtr->name != NULL) {
	Blt_Free(infoPtr->name);
    }
    if (infoPtr->draw != 0) {
	XftDrawDestroy(infoPtr->draw);
    }
    if (infoPtr->fid) {
	XUnloadFont(infoPtr->display, infoPtr->fid);
    }
    Blt_DeleteHashEntry(&fontTable, infoPtr->hashPtr);
    Blt_Free(infoPtr);
}

static FontInfo *
NewXftFontInfo(
    Tk_Window tkwin, 
    CONST char *fontName, 
    XftFont *fontPtr)
{
    Blt_HashEntry *hPtr;
    FontInfo *infoPtr;
    int isNew;

    infoPtr = Blt_Calloc(1, sizeof(FontInfo));
    assert(infoPtr);
    infoPtr->name = Blt_Strdup(fontName);
    infoPtr->visual = Tk_Visual(tkwin);
    infoPtr->colormap = Tk_Colormap(tkwin);
    infoPtr->display = Tk_Display(tkwin);
    infoPtr->fid = XLoadFont(Tk_Display(tkwin), "fixed");
    infoPtr->color.pixel = 0xFFFFFFFF;
    infoPtr->pattern = fontPtr->pattern;
    infoPtr->refCount = 1;
	    
    /* 
     * Initialize the Xft font table for this font.  Add the initial
     * Xft font for the case of 0 degrees rotation.
     */
    Blt_InitHashTable(&infoPtr->xftFontTable, BLT_ONE_WORD_KEYS);
    hPtr = Blt_CreateHashEntry(&infoPtr->xftFontTable, (char *)0L, &isNew);
    assert((hPtr != NULL) && (isNew));
    Blt_SetHashValue(hPtr, fontPtr);

    /* Add the font information to the font table. */
    hPtr = Blt_CreateHashEntry(&fontTable, fontName, &isNew);
    assert((hPtr != NULL) && (isNew));
    Blt_SetHashValue(hPtr, infoPtr);
    infoPtr->hashPtr = hPtr;
    return infoPtr;
}

static void
FreeXftFont(FontInfo *infoPtr)
{
    infoPtr->refCount--;
    if (infoPtr->refCount <= 0) {
	DestroyXftFontInfo(infoPtr);
    }
}

/*
 * GetXftFontInfoFromObj --
 *
 *	Generates an Xft font based upon the description provided.
 *	The description is parsed base upon Tk's font selection 
 *	rules (listed below).
 *
 *      Tk's Font Selection Rules:
 *
 *	When font description font is used, the system attempts to
 *	parse the description according to each of the above five
 *	rules, in the order specified.  Cases [1] and [2] must match
 *	the name of an existing named font or of a system font.  Cases
 *	[3], [4], and [5] are accepted on all platforms and the
 *	closest available font will be used.  In some situations it
 *	may not be possible to find any close font (e.g., the font
 *	family was a garbage value); in that case, some
 *	system-dependant default font is chosen.  If the font
 *	description does not match any of the above patterns, an error
 *	is generated.
 *
 * [1] fontname
 *	The name of a named font, created using the font create
 *	command.  When a widget uses a named font, it is guaranteed
 *	that this will never cause an error, as long as the named font
 *	exists, no mat- ter what potentially invalid or meaningless
 *	set of attributes the named font has.  If the named font
 *	cannot be displayed with exactly the specified attributes,
 *	some other close font will be substituted automatically.
 *	
 *	[Query the named font (using "font configure") and generate
 *	an Xft font with the same attributes.  It's assumed that
 *	these names don't start with a '*' or '-'.]
 *
 * [2] systemfont
 *	The platform-specific name of a font, interpreted by the
 *	graphics server.  This also includes, under X, an XLFD (see
 *	[4]) for which a single ``*'' character was used to elide more
 *	than one field in the middle of the name.  See
 *	PLATFORM-SPECIFIC issues for a list of the system fonts.
 *
 *	[Same as above. Query the named font (using "font configure")
 *	and generate an Xft font with the same attributes.]
 *
 * [3] family ?size? ?style? ?style ...? 
 *	A properly formed list whose first element is the desired font
 *	family and whose optional second element is the desired size.
 *	The interpretation of the size attribute follows the same
 *	rules described for -size in FONT OPTIONS below.  Any
 *	additional optional arguments following the size are font
 *	styles.  Possible values for the style arguments are as
 *	follows:
 *
 *	   normal, bold, roman, italic, underline, overstrike 
 *
 *	[Parse the list of attributes and generate a corresponding
 *	Xft font.]
 *
 * [4] X-font names (XLFD)
 *	A Unix-centric font name of the form -foundry-family-weight
 *	slant-setwidth-addstyle-pixel-point-resx-resy-spacing-width
 *	charset-encoding.  The ``*'' character may be used to skip
 *	indi vidual fields that the user does not care about.  There
 *	must be exactly one ``*'' for each field skipped, except that
 *	a ``*'' at the end of the XLFD skips any remaining fields; the
 *	shortest valid XLFD is simply ``*'', signifying all fields as
 *	defaults.  Any fields that were skipped are given default
 *	values.  For compatibility, an XLFD always chooses a font of
 *	the specified pixel size (not point size); although this
 *	interpretation is not strictly correct, all existing
 *	applications using XLFDs assumed that one ``point'' was in
 *	fact one pixel and would display incorrectly (generally
 *	larger) if the correct size font were actually used.
 *
 *	[Parse the font description and generate a corresponding
 *	Xft font.]
 *
 * [5] option value ?option value ...?
 *	A properly formed list of option-value pairs that specify the
 *	desired attributes of the font, in the same format used when
 *	defining a named .
 *
 *	[Parse the option-value list and generate a corresponding
 *	Xft font.]
 *
 *  Extra:
 * [6] Xft font description.
 *
 *	[Handle the newer Xft font descriptions.]
 */
static FontInfo *
GetXftFontInfoFromObj(
    Tcl_Interp *interp,
    Tk_Window tkwin, 
    Tcl_Obj *objPtr)
{
    XftPattern *pattern;
    char *desc;

    desc = Tcl_GetString(objPtr);
    while (isspace(UCHAR(*desc))) {
	desc++;			/* Skip leading blanks. */
    }
    {
	Blt_HashEntry *hPtr;

	/* Is the font already in the cache? */
	hPtr = Blt_FindHashEntry(&fontTable, desc);
	if (hPtr != NULL) {
	    FontInfo *infoPtr;
	    
	    infoPtr = Tcl_GetHashValue(hPtr);
	    infoPtr->refCount++;
	    return infoPtr;
	} 
    }
    if (*desc == '-') {
	/* 
	 * Case 1: XLFD font description or Tk attribute list.   
	 *
	 *   If the font description starts with a '-', it could
	 *   be either an old fashion XLFD font description or
	 *   a Tk font attribute option-value list.
	 */
	pattern = ParseTkFontAttributeList(interp, objPtr);
	if (pattern == NULL) {
	    /* Try parsing it as an XLFD font description. */
	    pattern = ParseXlfdDesc(tkwin, desc);
	}
    } else if (*desc == '*') {
	pattern = ParseXlfdDesc(tkwin, desc);
    } else if (strpbrk(desc, ":,=") != NULL) {
	/* 
	 * Case 2: XFT font description.   
	 *
	 *   If the font description contains a ':', '=', or ',' in
	 *   it, we assume it's a new XFT font description. We want to
	 *   allow these font descriptions too.
	 */
	pattern = NULL;
	if (strstr(desc, "::") != NULL) {
	    pattern = ParseTkNamedFont(interp, objPtr);
	} 
	if (pattern == NULL) {
	    pattern = XftNameParse(desc);
	}
#if DEBUG_FONT_SELECTION
	if (pattern != NULL) {
	    fprintf(stderr, "parsed XFT-XFT font \"%s\"\n", desc);
	}
#endif
    } else {
	int objc;
	Tcl_Obj **objv;
	/* 
	 * Case 3: Tk-style description.   
	 */
	if ((Tcl_ListObjGetElements(NULL, objPtr, &objc, &objv) != TCL_OK) || 
	    (objc < 1)) {
	    return NULL;		/* Can't split into a list or
					 * list is empty. */
	}
	if (objc == 1) {
	    /* 
	     * Case 3a: Tk font object name.
	     *
	     *   Assuming that Tk font object names won't contain
	     *   whitespace, see if its a font object.
	     */
	    pattern = ParseTkNamedFont(interp, objv[0]);
	    if (pattern == NULL) {
		pattern = XftNameParse(desc);
#if DEBUG_FONT_SELECTION
		if (pattern != NULL) {
		    fprintf(stderr, "parsed XFT-XFT font \"%s\"\n", desc);
		}
#endif
	    }
	} else {
	    /* 
	     * Case 3b: List of font attributes in the form 
	     *		"family size ?attrs?"
	     */
	    pattern = ParseTkDesc(objc, objv);
	}
    }	
    if (pattern != NULL) {
	XftPattern *match;
	XftResult result;
	/* 
	 * XftFontMatch only sets *result* on complete match failures.
	 * So initialize it here for a successful match. We'll accept
	 * partial matches. 
	 */
	result = XftResultMatch; 
	match = XftFontMatch(Tk_Display(tkwin), Tk_ScreenNumber(tkwin), 
			     pattern, &result);
#if DEBUG_FONT_SELECTION
	if (match != NULL) {
	    Tcl_DString dString;

	    Tcl_DStringInit(&dString);
	    Tcl_DStringSetLength(&dString, 1024);
	    XftNameUnparse(match, Tcl_DStringValue(&dString), 1024);
	    fprintf(stderr, "font %s(%d) is %s\n", desc, result,
		    Tcl_DStringValue(&dString));
	    Tcl_DStringFree(&dString);
	}
#endif
	XftPatternDestroy(pattern);
	if ((match != NULL) && (result == XftResultMatch)) {
	    XftFont *fontPtr;
	    
	    fontPtr = XftFontOpenPattern(Tk_Display(tkwin), match);
	    if (fontPtr != NULL) {
		return NewXftFontInfo(tkwin, desc, fontPtr);
	    }
	}
    }
#if DEBUG_FONT_SELECTION
    fprintf(stderr, "can't open font pattern \"%s\"\n", desc);
#endif
    return NULL;
}

static CONST char *
FamilyOfXftFont(FontInfo *infoPtr)
{
    CONST char *familyName;

    if (XftPatternGetString(infoPtr->pattern, XFT_FAMILY, 0, &familyName) 
	!= XftResultMatch) {
	return NULL;
    }
    return familyName;
}

static void
GetXftFontMetrics(FontInfo *infoPtr, Blt_FontMetrics *metricsPtr)
{
    Blt_HashEntry *hPtr;

    /* Always take font metrics from the non-rotated font. */
    hPtr = Blt_FindHashEntry(&infoPtr->xftFontTable, (char *)0L);
    if (hPtr != NULL) {
	FT_UInt glyph;
	XGlyphInfo metrics;
	XftFont *fontPtr;

	fontPtr = Blt_GetHashValue(hPtr);
	glyph = XftCharIndex(infoPtr->display, fontPtr, '0');
	XftGlyphExtents(infoPtr->display, fontPtr, &glyph, 1, &metrics);
	metricsPtr->ascent = fontPtr->ascent;
	metricsPtr->descent = fontPtr->descent;
	metricsPtr->linespace = metricsPtr->ascent + metricsPtr->descent;
    }
}

static int
MeasureXftChars(
    FontInfo *infoPtr,
    CONST char *source,	
    int nBytes,
    int maxLength,
    int flags,
    int *lengthPtr)
{
    FcChar32 c;
    XGlyphInfo extents;
    int clen;
    int curX, newX;
    int termByte = 0, termX = 0;
    int curByte, newByte, sawNonSpace;
#if 0
    char string[256];
    int len = 0;
#endif
    XftFont *fontPtr;
    Blt_HashEntry *hPtr;
    
    hPtr = Blt_FindHashEntry(&infoPtr->xftFontTable, (char *)0L);
    if (hPtr == NULL) {
	return 0;
    }
    fontPtr = Blt_GetHashValue(hPtr);
    curX = 0;
    curByte = 0;
    sawNonSpace = 0;
    while (nBytes > 0) {
	Tcl_UniChar unichar;

	clen = Tcl_UtfToUniChar(source, &unichar);
	c = (FcChar32)unichar;

	if (clen <= 0) {
	    /* This can't happen (but see #1185640) */
	    *lengthPtr = curX;
	    return curByte;
	}

	source += clen;
	nBytes -= clen;
	if (c < 256 && isspace(c)) {		/* I18N: ??? */
	    if (sawNonSpace) {
		termByte = curByte;
		termX = curX;
		sawNonSpace = 0;
	    }
	} else {
	    sawNonSpace = 1;
	}

#if 0
	string[len++] = (char) c;
#endif
	XftTextExtents32(infoPtr->display, fontPtr, &c, 1, &extents);

	newX = curX + extents.xOff;
	newByte = curByte + clen;
	if (maxLength >= 0 && newX > maxLength) {
	    if ((flags & TK_PARTIAL_OK) ||
		((flags & TK_AT_LEAST_ONE && curByte == 0))) {
		curX = newX;
		curByte = newByte;
	    } else if ((flags & TK_WHOLE_WORDS) && (termX != 0)) {
		curX = termX;
		curByte = termByte;
	    }
	    break;
	}

	curX = newX;
	curByte = newByte;
    }
#if 0
    string[len] = '\0';
    printf("MeasureChars %s length %d bytes %d\n", string, curX, curByte);
#endif
    *lengthPtr = curX;
    return curByte;
}


static void
DrawXftChars(
    Display *display,		/* Display on which to draw. */
    Drawable drawable,		/* Window or pixmap in which to draw. */
    int depth,			/* Depth of drawable. */
    GC gc,			/* Graphics context for drawing characters. */
    FontInfo *infoPtr,		/* Font in which characters will be drawn;
				 * must be the same as font used in GC. */
    float angle,
    CONST char *source,		/* UTF-8 string to be displayed.  Need not be
				 * '\0' terminated.  All Tk meta-characters
				 * (tabs, control characters, and newlines)
				 * should be stripped out of the string that
				 * is passed to this function.  If they are
				 * not stripped out, they will be displayed as
				 * regular printing characters. */
    int nBytes,			/* Number of bytes in string. */
    int x, int y)		/* Coordinates at which to place origin of
				 * string when drawing. */
{
    XftFont *fontPtr;
    Blt_HashEntry *hPtr;
    long angle10;

    angle10 = (long)(angle * 10.0);
    hPtr = Blt_FindHashEntry(&infoPtr->xftFontTable, (char *)angle10);
    if (hPtr == NULL) {
	fprintf(stderr, "can't find font %s at %g rotated\n", 
		infoPtr->name, angle);
	return;			/* Can't find instance at requested angle. */
    }
    fontPtr = Blt_GetHashValue(hPtr);
    if ((infoPtr->draw == 0) || (infoPtr->drawDepth != depth)) {
	XftDraw *draw;

	if (depth == 1) {
	    draw = XftDrawCreateBitmap(display, drawable);
	} else {
	    draw = XftDrawCreate(display, drawable, infoPtr->visual, 
			infoPtr->colormap);
	}
	if (infoPtr->draw != 0) {
	    XftDrawDestroy(infoPtr->draw);
	}
	infoPtr->drawDepth = depth;
	infoPtr->draw = draw;
	infoPtr->drawable = drawable;
    } else {
	Tk_ErrorHandler handler;
#if 0
	printf("Switch to drawable 0x%x\n", drawable);
#endif
        handler = Tk_CreateErrorHandler(display, -1, -1, -1,
		(Tk_ErrorProc *)NULL, (ClientData) NULL);
	XftDrawChange(infoPtr->draw, drawable);
	infoPtr->drawable = drawable;
	Tk_DeleteErrorHandler(handler);
    }
    {
	XGCValues values;

	XGetGCValues(display, gc, GCForeground, &values);
	if (values.foreground != infoPtr->color.pixel) {
	    XColor xc;
	    
	    xc.pixel = values.foreground;
	    XQueryColor(display, infoPtr->colormap, &xc);
	    infoPtr->color.color.red = xc.red;
	    infoPtr->color.color.green = xc.green;
	    infoPtr->color.color.blue = xc.blue;
	    infoPtr->color.color.alpha = 0xffff; /* Assume opaque. */
	    infoPtr->color.pixel = values.foreground;
	}
    }

    {
#define NUM_SPEC    1024
	XftGlyphFontSpec *specPtr;
	XftGlyphFontSpec specs[NUM_SPEC];
	int nSpecs;
	const int maxCoord = 0x7FFF; /* Xft coordinates are 16 bit values */

	nSpecs = 0;
	specPtr = specs;
	while ((nBytes > 0) && (x <= maxCoord) && (y <= maxCoord)) {
	    FcChar32 c;
	    int charLen;
	    XGlyphInfo metrics;
	    
	    charLen = FcUtf8ToUcs4((FcChar8 *)source, &c, nBytes);
	    if (charLen <= 0) {
		/* This should not happen, but it can. */
		return;
	    }
	    source += charLen;
	    nBytes -= charLen;
	    
	    specPtr = specs + nSpecs;
	    specPtr->font = fontPtr;
	    specPtr->glyph = XftCharIndex(infoPtr->display, fontPtr, c);
	    specPtr->x = x;
	    specPtr->y = y;
	    XftGlyphExtents(infoPtr->display, fontPtr, &specPtr->glyph, 1,
			    &metrics);
	    x += metrics.xOff;
	    y += metrics.yOff;
	    nSpecs++, specPtr++;
	    if (nSpecs == NUM_SPEC) {
		XftDrawGlyphFontSpec(infoPtr->draw, &infoPtr->color, specs, 
			nSpecs);
		nSpecs = 0;
		specPtr = specs;
	    }
	}
	if (nSpecs > 0) {
	    XftDrawGlyphFontSpec(infoPtr->draw, &infoPtr->color, specs, nSpecs);
	}
    }
}

/*
 *---------------------------------------------------------------------------
 *
 * PostscriptXftFontName --
 *
 *	Given a Xft font, return the name of the corresponding
 *	Postscript font.
 *
 * Results:
 *	The return value is the pointsize of the given Xft font.
 *	The name of the Postscript font is appended to dsPtr.
 *
 * Side effects:
 *	If the font does not exist on the printer, the print job will
 *	fail at print time.  Given a "reasonable" Postscript printer,
 *	the following Tk_Font font families should print correctly:
 *
 *	    Avant Garde, Arial, Bookman, Courier, Courier New, Geneva,
 *	    Helvetica, Monaco, New Century Schoolbook, New York,
 *	    Palatino, Symbol, Times, Times New Roman, Zapf Chancery,
 *	    and Zapf Dingbats.
 *
 *	Any other Xft font families may not print correctly
 *	because the computed Postscript font name may be incorrect.
 *
 *---------------------------------------------------------------------------
 */

static int
PostScriptXftFontName(
    FontInfo *infoPtr,		
    Tcl_DString *resultPtr)	
{
    CONST char *familyName, *weightName, *slantName;
    int weight, slant;
    double size;
    int len;

    len = Tcl_DStringLength(resultPtr);

    /*
     * Convert the case-insensitive Tk_Font family name to the
     * case-sensitive Postscript family name.  Take out any spaces and
     * capitalize the first letter of each word.
     */
    if (XftPatternGetString(infoPtr->pattern, XFT_FAMILY, 0, &familyName) 
	!= XftResultMatch) {
	familyName = "Unknown";
    }
    if (strncasecmp(familyName, "itc ", 4) == 0) {
	familyName += 4;
    }
    if ((strcasecmp(familyName, "Arial") == 0) || 
	(strcasecmp(familyName, "Geneva") == 0)) {
	familyName = "Helvetica";
    } else if ((strcasecmp(familyName, "Times New Roman") == 0) || 
	       (strcasecmp(familyName, "New York") == 0)) {
	familyName = "Times";
    } else if ((strcasecmp(familyName, "Courier New") == 0) || 
	       (strcasecmp(familyName, "Monaco") == 0)) {
	familyName = "Courier";
    } else if (strcasecmp(familyName, "AvantGarde") == 0) {
	familyName = "AvantGarde";
    } else if (strcasecmp(familyName, "ZapfChancery") == 0) {
	familyName = "ZapfChancery";
    } else if (strcasecmp(familyName, "ZapfDingbats") == 0) {
	familyName = "ZapfDingbats";
    } else {
	Tcl_UniChar ch;
	char *src, *dest;
	int upper;

	/*
	 * Inline, capitalize the first letter of each word, lowercase the
	 * rest of the letters in each word, and then take out the spaces
	 * between the words.  This may make the DString shorter, which is
	 * safe to do.
	 */

	Tcl_DStringAppend(resultPtr, familyName, -1);

	src = dest = Tcl_DStringValue(resultPtr) + len;
	upper = 1;
	for (; *src != '\0'; ) {
	    while (isspace(UCHAR(*src))) { /* INTL: ISO space */
		src++;
		upper = 1;
	    }
	    src += Tcl_UtfToUniChar(src, &ch);
	    if (upper) {
		ch = Tcl_UniCharToUpper(ch);
		upper = 0;
	    } else {
	        ch = Tcl_UniCharToLower(ch);
	    }
	    dest += Tcl_UniCharToUtf(ch, dest);
	}
	*dest = '\0';
	Tcl_DStringSetLength(resultPtr, dest - Tcl_DStringValue(resultPtr));
	familyName = Tcl_DStringValue(resultPtr) + len;
    }
    if (familyName != Tcl_DStringValue(resultPtr) + len) {
	Tcl_DStringAppend(resultPtr, familyName, -1);
	familyName = Tcl_DStringValue(resultPtr) + len;
    }

    if (strcasecmp(familyName, "NewCenturySchoolbook") == 0) {
	Tcl_DStringSetLength(resultPtr, len);
	Tcl_DStringAppend(resultPtr, "NewCenturySchlbk", -1);
	familyName = Tcl_DStringValue(resultPtr) + len;
    }

    /*
     * Get the string to use for the weight.
     */
    if (XftPatternGetInteger(infoPtr->pattern, XFT_WEIGHT, 0, &weight) 
	!= XftResultMatch) {
	weight = XFT_WEIGHT_MEDIUM;
    }
    weightName = NULL;
    if (weight <= XFT_WEIGHT_MEDIUM) {
	if (strcmp(familyName, "Bookman") == 0) {
	    weightName = "Light";
	} else if (strcmp(familyName, "AvantGarde") == 0) {
	    weightName = "Book";
	} else if (strcmp(familyName, "ZapfChancery") == 0) {
	    weightName = "Medium";
	}
    } else {
	if ((strcmp(familyName, "Bookman") == 0) || 
	    (strcmp(familyName, "AvantGarde") == 0)) {
	    weightName = "Demi";
	} else {
	    weightName = "Bold";
	}
    }

    /*
     * Get the string to use for the slant.
     */

    if (XftPatternGetInteger(infoPtr->pattern, XFT_SLANT, 0, &slant) 
	!= XftResultMatch) {
	slant = XFT_SLANT_ROMAN;
    }
    slantName = NULL;
    if (slant > XFT_SLANT_ROMAN) {
	if ((strcmp(familyName, "Helvetica") == 0) || 
	    (strcmp(familyName, "Courier") == 0) || 
	    (strcmp(familyName, "AvantGarde") == 0)) {
	    slantName = "Oblique";
	} else {
	    slantName = "Italic";
	}
    }

    /*
     * The string "Roman" needs to be added to some fonts that are not bold
     * and not italic.
     */

    if ((slantName == NULL) && (weightName == NULL)) {
	if ((strcmp(familyName, "Times") == 0) || 
	    (strcmp(familyName, "NewCenturySchlbk") == 0) || 
	    (strcmp(familyName, "Palatino") == 0)) {
	    Tcl_DStringAppend(resultPtr, "-Roman", -1);
	}
    } else {
	Tcl_DStringAppend(resultPtr, "-", -1);
	if (weightName != NULL) {
	    Tcl_DStringAppend(resultPtr, weightName, -1);
	}
	if (slantName != NULL) {
	    Tcl_DStringAppend(resultPtr, slantName, -1);
	}
    }
    if (XftPatternGetDouble(infoPtr->pattern, XFT_SIZE, 0, &size) 
	!= XftResultMatch) {
	size = 12.0;
    }
    return (int)size;
}

/*
 *---------------------------------------------------------------------------
 *
 * Blt_GetFontFromObj -- 
 *
 *	Given a string description of a font, map the description to a
 *	corresponding Tk_Font that represents the font.
 *
 * Results:
 *	The return value is token for the font, or NULL if an error
 *	prevented the font from being created.  If NULL is returned, an
 *	error message will be left in the interp's result.
 *
 * Side effects:
 *	The font is added to an internal database with a reference
 *	count.  For each call to this procedure, there should eventually
 *	be a call to Tk_FreeFont() or Tk_FreeFontFromObj() so that the
 *	database is cleaned up when fonts aren't in use anymore.
 *
 *---------------------------------------------------------------------------
 */
Blt_Font
Blt_GetFontFromObj(
    Tcl_Interp *interp,		/* Interp for database and error return. */
    Tk_Window tkwin,		/* For display on which font will be used. */
    Tcl_Obj *objPtr)		/* String describing font, as: named font,
				 * native format, or parseable string. */
{
    struct Blt_FontStruct *fsPtr; 
    
    fsPtr = Blt_Calloc(1, sizeof(struct Blt_FontStruct));
    if (fsPtr == NULL) {
	return NULL;		/* Out of memory. */
    }
    if (!initialized) {
	CheckForXRenderExtension(Tk_Display(tkwin));
	initialized = 1;
    }
    if (haveXRenderExtension) {
	FontInfo *infoPtr;

	/* Check first if we open the specified font as an XFT font. */
	infoPtr = GetXftFontInfoFromObj(interp, tkwin, objPtr);
	if (infoPtr != NULL) {
	    fsPtr->type = FONT_XFT;
	    fsPtr->infoPtr = infoPtr;
#if DEBUG_FONT_SELECTION2
	    fprintf(stderr, "SUCCESS: Found XFT font \"%s\"\n", Tcl_GetString(objPtr));
#endif
	    return fsPtr;	/* Found Xft font.  */
	}
	/* Otherwise fall thru and try to open font as a normal Tk
	 * font. */
    }
    fsPtr->tkFont = Tk_GetFont(interp, tkwin, Tcl_GetString(objPtr));
    if (fsPtr->tkFont == NULL) {
	Blt_Free(fsPtr);
#if DEBUG_FONT_SELECTION
    fprintf(stderr, "FAILED to find either Xft or Tk font \"%s\"\n", Tcl_GetString(objPtr));
#endif
	return NULL;		/* Failed to find either Xft or Tk fonts. */
    }
#if DEBUG_FONT_SELECTION
    fprintf(stderr, "SUCCESS: Found Tk font \"%s\"\n", Tcl_GetString(objPtr));
#endif
    fsPtr->type = FONT_TK;
    return fsPtr;		/* Found Tk font. */
}

/*
 *---------------------------------------------------------------------------
 *
 * Blt_AllocFontFromObj -- 
 *
 *	Given a string description of a font, map the description to a
 *	corresponding Tk_Font that represents the font.
 *
 * Results:
 *	The return value is token for the font, or NULL if an error
 *	prevented the font from being created.  If NULL is returned, an
 *	error message will be left in interp's result object.
 *
 * Side effects:
 * 	The font is added to an internal database with a reference
 * 	count.  For each call to this procedure, there should
 * 	eventually be a call to Blt_FreeFont so that the database is
 * 	cleaned up when fonts aren't in use anymore.
 *
 *---------------------------------------------------------------------------
 */

Blt_Font
Blt_GetFont(
    Tcl_Interp *interp,		/* Interp for database and error return. */
    Tk_Window tkwin,		/* For screen on which font will be used. */
    CONST char *string)		/* Object describing font, as: named font,
				 * native format, or parseable string. */
{
    Blt_Font font;
    Tcl_Obj *objPtr;

    objPtr = Tcl_NewStringObj(string, strlen(string));
    font = Blt_GetFontFromObj(interp, tkwin, objPtr);
    Tcl_DecrRefCount(objPtr);
    return font;
}

Blt_Font
Blt_AllocFontFromObj(
    Tcl_Interp *interp,		/* Interp for database and error return. */
    Tk_Window tkwin,		/* For screen on which font will be used. */
    Tcl_Obj *objPtr)		/* Object describing font, as: named font,
				 * native format, or parseable string. */
{
    return Blt_GetFontFromObj(interp, tkwin, objPtr);
}

char *
Blt_NameOfFont(Blt_Font font) 
{
    struct Blt_FontStruct *fsPtr = (struct Blt_FontStruct *)font;

    if (fsPtr->type == FONT_XFT) {
	return fsPtr->infoPtr->name;
    } else if (fsPtr->type == FONT_TK) {
	return Tk_NameOfFont(fsPtr->tkFont);
    } 
    panic("Blt_NameOfFont: unknown font type");
    return NULL;
}

CONST char *
Blt_FontFamily(Blt_Font font)
{
    struct Blt_FontStruct *fsPtr = (struct Blt_FontStruct *)font;

    if (fsPtr->type == FONT_XFT) {
	return FamilyOfXftFont(fsPtr->infoPtr);
    } else if (fsPtr->type == FONT_TK) {
	return ((TkFont *)fsPtr->tkFont)->fa.family;
    } 
    panic("Blt_FamilyOfFont: unknown font type");
    return NULL;
    
}

int
Blt_PostscriptFontName(Blt_Font font, Tcl_DString *resultPtr) 
{
    struct Blt_FontStruct *fsPtr = (struct Blt_FontStruct *)font;

    if (fsPtr->type == FONT_XFT) {
	return PostScriptXftFontName(fsPtr->infoPtr, resultPtr);
    } else if (fsPtr->type == FONT_TK) {
	return Tk_PostscriptFontName(fsPtr->tkFont, resultPtr);
    } 
    panic("Blt_PostscriptFontName: unknown font type");
    return 0;
}

/*
 *---------------------------------------------------------------------------
 *
 * Blt_GetFontMetrics --
 *
 *	Returns overall ascent and descent metrics for the given font.
 *	These values can be used to space multiple lines of text and
 *	to align the baselines of text in different fonts.
 *
 * Results:
 *	If *heightPtr is non-NULL, it is filled with the overall height
 *	of the font, which is the sum of the ascent and descent.
 *	If *ascentPtr or *descentPtr is non-NULL, they are filled with
 *	the ascent and/or descent information for the font.
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */
void
Blt_GetFontMetrics(Blt_Font font, Blt_FontMetrics *metricsPtr)
{
    struct Blt_FontStruct *fsPtr = (struct Blt_FontStruct *)font;

    if (fsPtr->type == FONT_XFT) {
	GetXftFontMetrics(fsPtr->infoPtr, metricsPtr);
    } else if (fsPtr->type == FONT_TK) {
	Tk_GetFontMetrics(fsPtr->tkFont, metricsPtr);
    } else {
	panic("Blt_GetFontMetrics: unknown font type");
    }
}

#ifdef notdef
/*
 *---------------------------------------------------------------------------
 *
 * Tk_UnderlineChars --
 *
 *	This procedure draws an underline for a given range of characters
 *	in a given string.  It doesn't draw the characters (which are
 *	assumed to have been displayed previously); it just draws the
 *	underline.  This procedure would mainly be used to quickly
 *	underline a few characters without having to construct an
 *	underlined font.  To produce properly underlined text, the
 *	appropriate underlined font should be constructed and used. 
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information gets displayed in "drawable".
 *
 *----------------------------------------------------------------------
 */

static void
UnderlineCharsXft(
    Display *display,		/* Display on which to draw. */
    Drawable drawable,		/* Window or pixmap in which to draw. */
    GC gc,			/* Graphics context for actually drawing
				 * line. */
    Blt_Font font,		/* Font used in GC;  must have been allocated
				 * by Tk_GetFont().  Used for character
				 * dimensions, etc. */
    CONST char *string,		/* String containing characters to be
				 * underlined or overstruck. */
    int x, int y,		/* Coordinates at which first character of
				 * string is drawn. */
    int firstByte,		/* Index of first byte of first character. */
    int lastByte)		/* Index of first byte after the last
				 * character. */
{
    TkFont *fontPtr;
    int startX, endX;

    
    Blt_MeasureChars(font, string, firstByte, -1, 0, &startX);
    Blt_MeasureChars(font, string, lastByte, -1, 0, &endX);

    underlinePos = descent / 2;
    underlineHeight = TkFontGetPixels(tkwin, fontPtr->fa.size) / 10;

    fontPtr = (TkFont *) tkfont;
    XFillRectangle(display, drawable, gc, x + startX,
	    y + fontPtr->underlinePos, (unsigned int) (endX - startX),
	    (unsigned int) fontPtr->underlineHeight);
}
#endif

Font
Blt_FontId(Blt_Font font) 
{
    struct Blt_FontStruct *fsPtr = (struct Blt_FontStruct *)font;

    if (fsPtr->type == FONT_XFT) {
	return fsPtr->infoPtr->fid;
    } else if (fsPtr->type == FONT_TK) {
	return Tk_FontId(fsPtr->tkFont);
    }
    panic("unknown font type");
    return 0;
}

int
Blt_CanRotateFont(Blt_Font font, double angle) 
{
    struct Blt_FontStruct *fsPtr = (struct Blt_FontStruct *)font;

    if (fsPtr->type == FONT_XFT) {
	return CanRotateXftFont(fsPtr->infoPtr, (float)angle);
    } else if (fsPtr->type == FONT_TK) {
	return FALSE;
    }
    panic("unknown font type");
    return 0;
}

int
Blt_TextWidth(Blt_Font font, CONST char *string, int nBytes)
{
    struct Blt_FontStruct *fsPtr = (struct Blt_FontStruct *)font;

    if (fsPtr->type == FONT_XFT) {
	int width;

	MeasureXftChars(fsPtr->infoPtr, string, nBytes, -1, 0, &width);
	return width;
    } else if (fsPtr->type == FONT_TK) {
	return Tk_TextWidth(fsPtr->tkFont, string, nBytes);
    } else {
	panic("unknown font type");
    }
    return 0;
}    

void
Blt_FreeFont(Blt_Font font) 
{
    struct Blt_FontStruct *fsPtr = (struct Blt_FontStruct *)font;

    if (fsPtr->type == FONT_XFT) {
	FreeXftFont(fsPtr->infoPtr);
    } else if (fsPtr->type == FONT_TK) {
	Tk_FreeFont(fsPtr->tkFont);
    } else {
	panic("unknown font type");
    }
    Blt_Free(fsPtr);
}

int
Blt_MeasureChars(
    Blt_Font font,
    CONST char *text,
    int nBytes,
    int maxLength,
    int flags,
    int *lengthPtr)
{
    struct Blt_FontStruct *fsPtr = (struct Blt_FontStruct *)font;

    if (fsPtr->type == FONT_XFT) {
	return MeasureXftChars(fsPtr->infoPtr, text, nBytes, maxLength, flags, 
		lengthPtr);
    } else if (fsPtr->type == FONT_TK) {
	return Tk_MeasureChars(fsPtr->tkFont, text, nBytes, maxLength, 
		flags, lengthPtr);
    }
    panic("unknown font type");
    return -1;
}

void
Blt_DrawChars(
    Display *display,		/* Display on which to draw. */
    Drawable drawable,		/* Window or pixmap in which to draw. */
    int depth,			/* Depth of drawable. */
    GC gc,			/* Graphics context for drawing characters. */
    Blt_Font font,		/* Font in which characters will be drawn;
				 * must be the same as font used in GC. */
    CONST char *text,		/* UTF-8 string to be displayed.  Need not be
				 * '\0' terminated.  All Tk meta-characters
				 * (tabs, control characters, and newlines)
				 * should be stripped out of the string that
				 * is passed to this function.  If they are
				 * not stripped out, they will be displayed as
				 * regular printing characters. */
    int length,			/* Number of bytes in string. */
    int x, int y)		/* Coordinates at which to place origin of
				 * string when drawing. */
{
    struct Blt_FontStruct *fsPtr = (struct Blt_FontStruct *)font;

    if (fsPtr->type == FONT_XFT) {
	return DrawXftChars(display, drawable, depth, gc, fsPtr->infoPtr, 0.0f,
		text, length, x, y);
    } else if (fsPtr->type == FONT_TK) {
	return Tk_DrawChars(display, drawable, gc, fsPtr->tkFont, text, 
		length, x, y);
    }
    panic("unknown font type");
}

void
Blt_DrawRotatedChars(
    Display *display,		/* Display on which to draw. */
    Drawable drawable,		/* Window or pixmap in which to draw. */
    int depth,			/* Depth of drawable. */
    GC gc,			/* Graphics context for drawing characters. */
    Blt_Font font,		/* Font in which characters will be drawn;
				 * must be the same as font used in GC. */
    double angle,		/* Angle at which to display font. */
    CONST char *text,		/* UTF-8 string to be displayed.  Need not be
				 * '\0' terminated.  All Tk meta-characters
				 * (tabs, control characters, and newlines)
				 * should be stripped out of the string that
				 * is passed to this function.  If they are
				 * not stripped out, they will be displayed as
				 * regular printing characters. */
    int length,			/* Number of bytes in string. */
    int x, int y)		/* Coordinates at which to place origin of
				 * string when drawing. */
{
    struct Blt_FontStruct *fsPtr = (struct Blt_FontStruct *)font;

    if (fsPtr->type != FONT_XFT) {
	fprintf (stderr, "can't rotate font: %s is not an xft font.\n", 
		 Blt_NameOfFont(font));
	return;
    }
    if (!CanRotateXftFont(fsPtr->infoPtr, (float)angle)) {
	fprintf (stderr, "can't rotate %s font\n", Blt_NameOfFont(font));
	return;
    }
    DrawXftChars(display, drawable, depth, gc, fsPtr->infoPtr, (float)angle, 
	text, length, x, y);
}
#endif /* HAVE_LIBXFT */

