
/*
 * bltText.c --
 *
 * This module implements multi-line, rotate-able text for the BLT toolkit.
 *
 *	Copyright 1993-2004 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "bltInt.h"
#include <X11/Xutil.h>
#include "tkIntBorder.h"
#include "tkDisplay.h"
#include "bltImage.h"
#include "bltBitmap.h"

#define WINDEBUG	0

static Blt_HashTable bitmapGCTable;
static int initialized;

/*
 * -----------------------------------------------------------------
 *
 * Blt_TextCreateLayout --
 *
 *	Get the extents of a possibly multiple-lined text string.
 *
 * Results:
 *	Returns via *widthPtr* and *heightPtr* the dimensions of
 *	the text string.
 *
 * -----------------------------------------------------------------
 */
TextLayout *
Blt_TextCreateLayout(char *string, TextStyle *tsPtr)
{
    TextFragment *fp;
    TextLayout *layoutPtr;
    Blt_FontMetrics fontMetrics;
    int count;			/* Count # of characters on each line */
    int lineHeight;
    int maxHeight, maxWidth;
    int nFrags;
    int width;			/* Running dimensions of the text */
    char *p;
    int i;
    size_t size;

    nFrags = 0;
    for (p = string; *p != '\0'; p++) {
	if (*p == '\n') {
	    nFrags++;
	}
    }
    if ((p != string) && (*(p - 1) != '\n')) {
	nFrags++;
    }
    size = sizeof(TextLayout) + (sizeof(TextFragment) * (nFrags - 1));

    layoutPtr = Blt_Calloc(1, size);
    layoutPtr->nFrags = nFrags;

    nFrags = count = 0;
    width = maxWidth = 0;
    maxHeight = tsPtr->padTop;
    Blt_GetFontMetrics(tsPtr->font, &fontMetrics);
    lineHeight = fontMetrics.linespace + tsPtr->leader;

    fp = layoutPtr->fragments;
    for (p = string; *p != '\0'; p++) {
	if (*p == '\n') {
	    if (count > 0) {
		width = Blt_TextWidth(tsPtr->font, string, count);
		if (width > maxWidth) {
		    maxWidth = width;
		}
	    }
	    fp->width = width;
	    fp->count = count;
	    fp->y = maxHeight + fontMetrics.ascent;
	    fp->text = string;
	    fp++;
	    nFrags++;
	    maxHeight += lineHeight;
	    string = p + 1;	/* Start the string on the next line */
	    count = 0;		/* Reset to indicate the start of a new line */
	    continue;
	}
	count++;
    }
    if (nFrags < layoutPtr->nFrags) {
	width = Blt_TextWidth(tsPtr->font, string, count);
	if (width > maxWidth) {
	    maxWidth = width;
	}
	fp->width = width;
	fp->count = count;
	fp->y = maxHeight + fontMetrics.ascent;
	fp->text = string;
	maxHeight += lineHeight;
	nFrags++;
    }
    maxHeight += tsPtr->padBottom;
    maxWidth += PADDING(tsPtr->padX);
    fp = layoutPtr->fragments;
    for (i = 0; i < nFrags; i++, fp++) {
	switch (tsPtr->justify) {
	default:
	case TK_JUSTIFY_LEFT:
	    /* No offset for left justified text strings */
	    fp->x = tsPtr->padLeft;
	    break;
	case TK_JUSTIFY_RIGHT:
	    fp->x = (maxWidth - fp->width) - tsPtr->padRight;
	    break;
	case TK_JUSTIFY_CENTER:
	    fp->x = (maxWidth - fp->width) / 2;
	    break;
	}
    }
    layoutPtr->width = maxWidth;
    layoutPtr->height = maxHeight - tsPtr->leader;
    return layoutPtr;
}


/*
 * -----------------------------------------------------------------
 *
 * Blt_TextGetExtents --
 *
 *	Get the extents of a possibly multiple-lined text string.
 *
 * Results:
 *	Returns via *widthPtr* and *heightPtr* the dimensions of
 *	the text string.
 *
 * -----------------------------------------------------------------
 */
void
Blt_GetTextExtents(
    Blt_Font font, 
    int leader,
    CONST char *string, 
    int *widthPtr, int *heightPtr)
{
    Blt_FontMetrics fontMetrics;
    int lineHeight;
    int maxWidth, maxHeight;
    CONST char *line;

    if (string == NULL) {
	return;			/* NULL string? */
    }
    Blt_GetFontMetrics(font, &fontMetrics);
    lineHeight = fontMetrics.linespace;
    maxWidth = maxHeight = 0;

    { 
	int count;		/* Count # of characters on each line */
	CONST char *p;

	count = 0;
	for (p = line = string; *p != '\0'; p++) {
	    if (*p == '\n') {
		if (count > 0) {
		    int w;
		    
		    w = Blt_TextWidth(font, line, count);
		    if (w > maxWidth) {
			maxWidth = w;
		    }
		}
		maxHeight += lineHeight;
		line = p + 1;	/* Point to the start of the next line. */
		count = 0;		/* Reset counter to indicate the start
					 * of a new line. */
		continue;
	    }
	    count++;
	}
	if ((count > 0) && (*(p - 1) != '\n')) {
	    int w;
	    
	    maxHeight += lineHeight;
	    w = Blt_TextWidth(font, line, count);
	    if (w > maxWidth) {
		maxWidth = w;
	    }
	}
    }
    *widthPtr = maxWidth;
    *heightPtr = maxHeight;
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_TextGetExtents --
 *
 *	Get the extents of a possibly multiple-lined text string.
 *
 * Results:
 *	Returns via *widthPtr* and *heightPtr* the dimensions of
 *	the text string.
 *
 * -----------------------------------------------------------------
 */
void
Blt_TextGetExtents(
    TextStyle *tsPtr, 
    char *string, 
    int *widthPtr, int *heightPtr)
{
    int width, height;

    if (string == NULL) {
	return;			/* NULL string? */
    }
    Blt_GetTextExtents(tsPtr->font, tsPtr->leader, string, &width, &height);
    *widthPtr = width + PADDING(tsPtr->padX);
    *heightPtr = height + PADDING(tsPtr->padY);
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_GetBoundingBox
 *
 *	Computes the dimensions of the bounding box surrounding a
 *	rectangle rotated about its center.  If pointArr isn't NULL,
 *	the coordinates of the rotated rectangle are also returned.
 *
 *	The dimensions are determined by rotating the rectangle, and
 *	doubling the maximum x-coordinate and y-coordinate.
 *
 *		w = 2 * maxX,  h = 2 * maxY
 *
 *	Since the rectangle is centered at 0,0, the coordinates of the
 *	bounding box are (-w/2,-h/2 w/2,-h/2, w/2,h/2 -w/2,h/2).
 *
 *  		0 ------- 1
 *  		|         |
 *  		|    x    |
 *  		|         |
 *  		3 ------- 2
 *
 * Results:
 *	The width and height of the bounding box containing the
 *	rotated rectangle are returned.
 *
 * -----------------------------------------------------------------
 */
void
Blt_GetBoundingBox(
    int width, int height,	/* Unrotated region */
    double angle,		/* Rotation of box */
    double *rotWidthPtr, 
    double *rotHeightPtr,	/* (out) Bounding box region */
    Point2D *bbox)		/* (out) Points of the rotated box */
{
    int i;
    double sinTheta, cosTheta;
    double radians;
    double xMax, yMax;
    double x, y;
    Point2D corner[4];

    angle = FMOD(angle, 360.0);
    if (FMOD(angle, (double)90.0) == 0.0) {
	int ll, ur, ul, lr;
	double rotWidth, rotHeight;
	int quadrant;

	/* Handle right-angle rotations specially. */

	quadrant = (int)(angle / 90.0);
	switch (quadrant) {
	case ROTATE_270:	/* 270 degrees */
	    ul = 3, ur = 0, lr = 1, ll = 2;
	    rotWidth = (double)height;
	    rotHeight = (double)width;
	    break;
	case ROTATE_90:		/* 90 degrees */
	    ul = 1, ur = 2, lr = 3, ll = 0;
	    rotWidth = (double)height;
	    rotHeight = (double)width;
	    break;
	case ROTATE_180:	/* 180 degrees */
	    ul = 2, ur = 3, lr = 0, ll = 1;
	    rotWidth = (double)width;
	    rotHeight = (double)height;
	    break;
	default:
	case ROTATE_0:		/* 0 degrees */
	    ul = 0, ur = 1, lr = 2, ll = 3;
	    rotWidth = (double)width;
	    rotHeight = (double)height;
	    break;
	}
	if (bbox != NULL) {
	    x = rotWidth * 0.5;
	    y = rotHeight * 0.5;
	    bbox[ll].x = bbox[ul].x = -x;
	    bbox[ur].y = bbox[ul].y = -y;
	    bbox[lr].x = bbox[ur].x = x;
	    bbox[ll].y = bbox[lr].y = y;
	}
	*rotWidthPtr = rotWidth;
	*rotHeightPtr = rotHeight;
	return;
    }
    /* Set the four corners of the rectangle whose center is the
     * origin. */

    corner[1].x = corner[2].x = (double)width * 0.5;
    corner[0].x = corner[3].x = -corner[1].x;
    corner[2].y = corner[3].y = (double)height * 0.5;
    corner[0].y = corner[1].y = -corner[2].y;

    radians = (-angle / 180.0) * M_PI;
    sinTheta = sin(radians), cosTheta = cos(radians);
    xMax = yMax = 0.0;

    /* Rotate the four corners and find the maximum X and Y coordinates */

    for (i = 0; i < 4; i++) {
	x = (corner[i].x * cosTheta) - (corner[i].y * sinTheta);
	y = (corner[i].x * sinTheta) + (corner[i].y * cosTheta);
	if (x > xMax) {
	    xMax = x;
	}
	if (y > yMax) {
	    yMax = y;
	}
	if (bbox != NULL) {
	    bbox[i].x = x;
	    bbox[i].y = y;
	}
    }

    /*
     * By symmetry, the width and height of the bounding box are twice
     * the maximum x and y coordinates.
     */
    *rotWidthPtr = xMax + xMax;
    *rotHeightPtr = yMax + yMax;
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_TranslateAnchor --
 *
 * 	Translate the coordinates of a given bounding box based
 *	upon the anchor specified.  The anchor indicates where
 *	the given xy position is in relation to the bounding box.
 *
 *  		nw --- n --- ne
 *  		|            |
 *  		w   center   e
 *  		|            |
 *  		sw --- s --- se
 *
 * 	The coordinates returned are translated to the origin of the
 * 	bounding box (suitable for giving to XCopyArea, XCopyPlane,
 * 	etc.)
 *
 * Results:
 *	The translated coordinates of the bounding box are returned.
 *
 * -----------------------------------------------------------------
 */
void
Blt_TranslateAnchor(
    int x, int y,		/* Window coordinates of anchor */
    int width, int height,	/* Extents of the bounding box */
    Tk_Anchor anchor,		/* Direction of the anchor */
    int *xPtr, int *yPtr)
{
    switch (anchor) {
    case TK_ANCHOR_NW:		/* Upper left corner */
	break;
    case TK_ANCHOR_W:		/* Left center */
	y -= (height / 2);
	break;
    case TK_ANCHOR_SW:		/* Lower left corner */
	y -= height;
	break;
    case TK_ANCHOR_N:		/* Top center */
	x -= (width / 2);
	break;
    case TK_ANCHOR_CENTER:	/* Center */
	x -= (width / 2);
	y -= (height / 2);
	break;
    case TK_ANCHOR_S:		/* Bottom center */
	x -= (width / 2);
	y -= height;
	break;
    case TK_ANCHOR_NE:		/* Upper right corner */
	x -= width;
	break;
    case TK_ANCHOR_E:		/* Right center */
	x -= width;
	y -= (height / 2);
	break;
    case TK_ANCHOR_SE:		/* Lower right corner */
	x -= width;
	y -= height;
	break;
    }
    *xPtr = x;
    *yPtr = y;
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_AnchorPoint --
 *
 * 	Translates a position, using both the dimensions of the
 * 	bounding box, and the anchor direction, returning the
 * 	coordinates of the upper-left corner of the box. The anchor
 * 	indicates where the given xy position is in relation to the
 * 	bounding box.
 *
 *  		nw --- n --- ne
 *  		|            |
 *  		w   center   e
 *  		|            |
 *  		sw --- s --- se
 *
 * 	The coordinates returned are translated to the origin of the
 * 	bounding box (suitable for giving to XCopyArea, XCopyPlane,
 * 	etc.)
 *
 * Results:
 *	The translated coordinates of the bounding box are returned.
 *
 * -----------------------------------------------------------------
 */
Point2D
Blt_AnchorPoint(
    double x, double y,		/* Coordinates of anchor. */
    double width, double height, /* Extents of the bounding box */
    Tk_Anchor anchor)		/* Direction of the anchor */
{
    Point2D t;

    switch (anchor) {
    case TK_ANCHOR_NW:		/* Upper left corner */
	break;
    case TK_ANCHOR_W:		/* Left center */
	y -= (height * 0.5);
	break;
    case TK_ANCHOR_SW:		/* Lower left corner */
	y -= height;
	break;
    case TK_ANCHOR_N:		/* Top center */
	x -= (width * 0.5);
	break;
    case TK_ANCHOR_CENTER:	/* Center */
	x -= (width * 0.5);
	y -= (height * 0.5);
	break;
    case TK_ANCHOR_S:		/* Bottom center */
	x -= (width * 0.5);
	y -= height;
	break;
    case TK_ANCHOR_NE:		/* Upper right corner */
	x -= width;
	break;
    case TK_ANCHOR_E:		/* Right center */
	x -= width;
	y -= (height * 0.5);
	break;
    case TK_ANCHOR_SE:		/* Lower right corner */
	x -= width;
	y -= height;
	break;
    }
    t.x = x;
    t.y = y;
    return t;
}

static INLINE int
SizeOfUtfChar(CONST char *s)	/* Buffer in which the UTF-8
				 * representation of the Tcl_UniChar
				 * is stored.  Buffer must be large
				 * enough to hold the UTF-8 character
				 * (at most TCL_UTF_MAX bytes). */
{
    int byte;
    
    byte = *((unsigned char *)s);
    if (byte < 0xC0) {
	return 1;
    } else if ((byte < 0xE0) && ((s[1] & 0xC0) == 0x80)) {
	return 2;
    } else if ((byte < 0xF0) && 
	       ((s[1] & 0xC0) == 0x80) && ((s[2] & 0xC0) == 0x80)) {
	return 3;
    }
    return 1;
}

/*
 *---------------------------------------------------------------------------
 *
 * DrawCharsWithEllipsis --
 *
 *	Draw a string of characters on the screen.  Blt_DrawChars()
 *	expands control characters that occur in the string to 
 *	\xNN sequences.  
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information gets drawn on the screen.
 *
 *---------------------------------------------------------------------------
 */
static void
DrawCharsWithEllipsis(
    Display *display,		/* Display on which to draw. */
    Drawable drawable,		/* Window or pixmap in which to draw. */
    int depth,			/* Depth of drawable. */
    GC gc,			/* Graphics context for drawing characters. */
    Blt_Font font,		/* Font in which characters will be drawn;
				 * must be the same as font used in GC. */
    CONST char *string,		/* UTF-8 string to be displayed.  Need not be
				 * '\0' terminated.  All Tk meta-characters
				 * (tabs, control characters, and newlines)
				 * should be stripped out of the string that
				 * is passed to this function.  If they are
				 * not stripped out, they will be displayed as
				 * regular printing characters. */
    int length,		        /* # of bytes in string. */
    int x, int y,		/* Coordinates at which to place origin of
				 * string when drawing. */
    int xMax)
{
    CONST char *s, *send;
    Tcl_DString dString;
    int nBytes;
    int accum, threshold;
#if HAVE_UTF
    Tcl_UniChar ch;
#endif /* HAVE_UTF */
    accum = x;
    threshold = xMax - Blt_TextWidth(font, "...", 3);
    Tcl_DStringInit(&dString);
#if !HAVE_UTF
    nBytes = 1;
#endif /* !HAVE_UTF */
    for (s = string, send = s + length; s < send; s += nBytes) {
#if HAVE_UTF
	nBytes =  Tcl_UtfToUniChar (s, &ch);
#endif /* HAVE_UTF */
	accum += Blt_TextWidth(font, s, nBytes);
	if (accum > threshold) {
	    break;
	}
	Tcl_DStringAppend(&dString, s, nBytes);
    }
    Tcl_DStringAppend(&dString, "...", 3);
    nBytes = Tcl_DStringLength(&dString);
    string = Tcl_DStringValue(&dString);
    Blt_DrawChars(display, drawable, depth, gc, font, string, nBytes, x, y);
    Tcl_DStringFree(&dString);
}

/*
 *---------------------------------------------------------------------------
 *
 * DrawRotatedCharsWithEllipsis --
 *
 *	Draw a string of characters on the screen.  Blt_DrawChars()
 *	expands control characters that occur in the string to 
 *	\xNN sequences.  
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information gets drawn on the screen.
 *
 *---------------------------------------------------------------------------
 */
static void
DrawRotatedCharsWithEllipsis(
    Display *display,		/* Display on which to draw. */
    Drawable drawable,		/* Window or pixmap in which to draw. */
    int depth,
    GC gc,			/* Graphics context for drawing characters. */
    Blt_Font font,		/* Font in which characters will be drawn;
				 * must be the same as font used in GC. */
    double angle,
    CONST char *string,		/* UTF-8 string to be displayed.  Need not be
				 * '\0' terminated.  All Tk meta-characters
				 * (tabs, control characters, and newlines)
				 * should be stripped out of the string that
				 * is passed to this function.  If they are
				 * not stripped out, they will be displayed as
				 * regular printing characters. */
    int length,		        /* # of bytes in string. */
    int x, int y,		/* Coordinates at which to place origin of
				 * string when drawing. */
    int xMax)
{
    CONST char *s, *send;
    Tcl_DString dString;
    int nBytes;
    int accum, threshold;
#if HAVE_UTF
    Tcl_UniChar ch;
#endif /* HAVE_UTF */
    accum = x;
    threshold = xMax - Blt_TextWidth(font, "...", 3);
    Tcl_DStringInit(&dString);
#if !HAVE_UTF
    nBytes = 1;
#endif /* !HAVE_UTF */
    for (s = string, send = s + length; s < send; s += nBytes) {
#if HAVE_UTF
	nBytes =  Tcl_UtfToUniChar (s, &ch);
#endif /* HAVE_UTF */
	accum += Blt_TextWidth(font, s, nBytes);
	if (accum > threshold) {
	    break;
	}
	Tcl_DStringAppend(&dString, s, nBytes);
    }
    Tcl_DStringAppend(&dString, "...", 3);
    Blt_DrawRotatedChars(display, drawable, depth, gc, font, angle, 
	Tcl_DStringValue(&dString), Tcl_DStringLength(&dString), x, y);
    Tcl_DStringFree(&dString);
}


static void
DrawLayout(
    Display *display,
    Drawable drawable,
    int depth,
    GC gc,
    Blt_Font font,
    int x, int y,          /* Origin of text */
    TextLayout *layoutPtr,
    int xMax)
{
    TextFragment *fp, *fend;

    for (fp = layoutPtr->fragments, fend = fp + layoutPtr->nFrags; 
	 fp < fend; fp++) {
	int sx, sy;

	sx = x + fp->x, sy = y + fp->y;
	if ((xMax > 0) && ((sx + fp->width) > xMax)) {
	    DrawCharsWithEllipsis(display, drawable, depth, gc, font, fp->text, 
		fp->count, sx, sy, xMax);
	} else {
	    Blt_DrawChars(display, drawable, depth, gc, font, fp->text, 
		fp->count, sx, sy);
	}
    }
}

static void
DrawRotatedLayout(
    Display *display,
    Drawable drawable,
    int depth,
    GC gc,
    Blt_Font font,
    double angle,
    TextLayout *layoutPtr,
    int xMax)
{
    TextFragment *fp, *fend;

    for (fp = layoutPtr->fragments, fend = fp + layoutPtr->nFrags; fp < fend; 
	 fp++) {
	if ((xMax > 0) && ((fp->sx + fp->width) > xMax)) {
	    DrawRotatedCharsWithEllipsis(display, drawable, depth, gc, font, 
		angle, fp->text, fp->count, fp->sx, fp->sy, xMax);
	} else {
	    Blt_DrawRotatedChars(display, drawable, depth, gc, font, angle, 
		fp->text, fp->count, fp->sx, fp->sy);
	}
    }
}

#ifdef WIN32
/*
 * -----------------------------------------------------------------
 *
 * Blt_TextToBitmap --
 *
 *	Draw a bitmap, using the the given window coordinates
 *	as an anchor for the text bounding box.
 *
 * Results:
 *	Returns the bitmap representing the text string.
 *
 * Side Effects:
 *	Bitmap is drawn using the given font and GC in the
 *	drawable at the given coordinates, anchor, and rotation.
 *
 * -----------------------------------------------------------------
 */
Pixmap
Blt_TextToBitmap(
    Tk_Window tkwin,
    TextLayout *layoutPtr,	/* Text string to draw */
    TextStyle *tsPtr,		/* Text attributes: rotation, color, font,
				 * linespacing, justification, etc. */
    int *bmWidthPtr,
    int *bmHeightPtr,		/* Extents of rotated text string */
    int xMax)
{
    Pixmap bitmap;
    Window root;
    GC gc;
    HDC hDC;
    TkWinDCState state;

    /* Create a temporary bitmap to contain the text string */
    root = Tk_RootWindow(tkwin);
    bitmap = Tk_GetPixmap(Tk_Display(tkwin), root, layoutPtr->width, 
	layoutPtr->height, 1);
    assert(bitmap != None);
    if (bitmap == None) {
	return None;		/* Can't allocate pixmap. */
    }
    gc = Blt_GetBitmapGC(tkwin);

    /* Clear the pixmap and draw the text string into it */
    hDC = TkWinGetDrawableDC(Tk_Display(tkwin), bitmap, &state);
    PatBlt(hDC, 0, 0, layoutPtr->width, layoutPtr->height, WHITENESS);
    TkWinReleaseDrawableDC(bitmap, hDC, &state);

    XSetFont(Tk_Display(tkwin), gc, Blt_FontId(tsPtr->font));
    XSetForeground(Tk_Display(tkwin), gc, 1);
    DrawLayout(Tk_Display(tkwin), bitmap, 1, gc, tsPtr->font, 0, 0, layoutPtr, 
	       xMax);

    /*
     * Under Win32 when drawing into a bitmap, the bits are
     * reversed. Which is why we are inverting the bitmap here.
     */
    hDC = TkWinGetDrawableDC(Tk_Display(tkwin), bitmap, &state);
    PatBlt(hDC, 0, 0, layoutPtr->width, layoutPtr->height, DSTINVERT);
    TkWinReleaseDrawableDC(bitmap, hDC, &state);

    *bmWidthPtr = layoutPtr->width, *bmHeightPtr = layoutPtr->height;
    return bitmap;
}
#else 
/*
 * -----------------------------------------------------------------
 *
 * Blt_TextToBitmap --
 *
 *	Draw a bitmap, using the the given window coordinates
 *	as an anchor for the text bounding box.
 *
 * Results:
 *	Returns the bitmap representing the text string.
 *
 * Side Effects:
 *	Bitmap is drawn using the given font and GC in the
 *	drawable at the given coordinates, anchor, and rotation.
 *
 * -----------------------------------------------------------------
 */
Pixmap
Blt_TextToBitmap(
    Tk_Window tkwin,
    TextLayout *layoutPtr,	/* Text string to draw */
    TextStyle *tsPtr,		/* Text attributes: rotation, color, font,
				 * linespacing, justification, etc. */
    int *bmWidthPtr,
    int *bmHeightPtr,		/* Extents of rotated text string */
    int xMax)
{
    Pixmap bitmap;
    GC gc;

    /* Create a bitmap big enough to contain the text. */
    bitmap = Tk_GetPixmap(Tk_Display(tkwin), Tk_RootWindow(tkwin), 
	layoutPtr->width, layoutPtr->height, 1);
    assert(bitmap != None);
    if (bitmap == None) {
	return None;		/* Can't allocate pixmap. */
    }
    gc = Blt_GetBitmapGC(tkwin);

    /* Clear the bitmap.  Background is 0. */
    XSetForeground(Tk_Display(tkwin), gc, 0);
    XFillRectangle(Tk_Display(tkwin), bitmap, gc, 0, 0, 
		   layoutPtr->width, layoutPtr->height);

    /* Draw the text into the bitmap. Foreground is 1. */
    XSetFont(Tk_Display(tkwin), gc, Blt_FontId(tsPtr->font));
    XSetForeground(Tk_Display(tkwin), gc, 1);
    DrawLayout(Tk_Display(tkwin), bitmap, 1, gc, tsPtr->font, 0, 0, layoutPtr, 
	       xMax);
    *bmWidthPtr = layoutPtr->width, *bmHeightPtr = layoutPtr->height;
    return bitmap;
}
#endif /* WIN32 */

void
Blt_TextSetDrawStyle(
    TextStyle *tsPtr,
    Blt_Font font,
    GC gc,
    XColor *normalColor, 
    double angle,
    Tk_Anchor anchor,
    Tk_Justify justify,
    int leader)
{
    tsPtr->padX.side1 = tsPtr->padX.side2 = 0;
    tsPtr->padY.side1 = tsPtr->padY.side2 = 0;
    tsPtr->state = 0;
    tsPtr->anchor = anchor;
    tsPtr->color = normalColor;
    tsPtr->font = font;
    tsPtr->gc = gc;
    tsPtr->justify = justify;
    tsPtr->leader = leader;
    tsPtr->angle = (float)angle;
}

static void
DrawStandardLayout(
    Display *display, 
    Drawable drawable, 
    int depth,
    TextStyle *tsPtr, 
    TextLayout *layoutPtr, 
    int x, int y, 
    int xMax)			/* If text crossing this threshold,
				 * draw ellipsis.  */
{
    int width, height;
    /*
     * This is the easy case of no rotation. Simply draw the text
     * using the standard drawing routines.  Handle offset printing
     * for engraved (disabled) text.
     */
    width = layoutPtr->width, height = layoutPtr->height;
    Blt_TranslateAnchor(x, y, width, height, tsPtr->anchor, &x, &y);
    if (tsPtr->state & (STATE_DISABLED | STATE_EMPHASIS)) {
	TkBorder *borderPtr = (TkBorder *) tsPtr->border;
	XColor *color1, *color2;

	color1 = borderPtr->lightColor, color2 = borderPtr->darkColor;
	if (tsPtr->state & STATE_EMPHASIS) {
	    XColor *hold;
	    
	    hold = color1, color1 = color2, color2 = hold;
	}
	if (color1 != NULL) {
	    XSetForeground(display, tsPtr->gc, color1->pixel);
	}
	DrawLayout(display, drawable, depth, tsPtr->gc, tsPtr->font, 
		x + 1, y + 1, layoutPtr, xMax);
	if (color2 != NULL) {
	    XSetForeground(display, tsPtr->gc, color2->pixel);
	}
	DrawLayout(display, drawable, depth, tsPtr->gc, tsPtr->font, x, y, 
		layoutPtr, xMax);
	
	/* Reset the foreground color back to its original setting,
	 * so not to invalidate the GC cache. */
	XSetForeground(display, tsPtr->gc, tsPtr->color->pixel);
	
	return;		/* Done */
    }
    DrawLayout(display, drawable, depth, tsPtr->gc, tsPtr->font, x, y, 
	layoutPtr, xMax);
}


void
Blt_RotateTextStartingPositions(
    TextLayout *textPtr, 
    int x, int y, 
    double angle, 
    int anchor)
{
    Point2D center;
    TextFragment *fp, *fend;
    double radians;
    double rotWidth, rotHeight;
    double sinTheta, cosTheta;
    int bbWidth, bbHeight;
    
    Blt_GetBoundingBox(textPtr->width, textPtr->height, angle, &rotWidth, 
		       &rotHeight, (Point2D *)NULL);
    bbWidth = ROUND(rotWidth);
    bbHeight = ROUND(rotHeight);
    Blt_TranslateAnchor(x, y, bbWidth, bbHeight, anchor, &x, &y);
    
    center.x = (double)textPtr->width * -0.5;
    center.y = (double)textPtr->height * -0.5;
    radians = (-angle / 180.0) * M_PI;
    
    sinTheta = sin(radians), cosTheta = cos(radians);
    for (fp = textPtr->fragments, fend = fp + textPtr->nFrags; fp < fend; 
	 fp++) {
	Point2D p, q;
	
	p.x = center.x + (double)fp->x;
	p.y = center.y + (double)fp->y;
	q.x = x + (p.x * cosTheta) - (p.y * sinTheta) + (bbWidth * 0.5);
	q.y = y + (p.x * sinTheta) + (p.y * cosTheta) + (bbHeight * 0.5);
	fp->sx = ROUND(q.x);
	fp->sy = ROUND(q.y);
    }
}

int
Blt_DrawTextWithRotatedFont(
    Tk_Window tkwin,
    Drawable drawable,
    double angle,
    TextStyle *tsPtr,
    TextLayout *layoutPtr,
    int x, int y,
    int xMax)
{
    Blt_RotateTextStartingPositions(layoutPtr, x, y, angle, tsPtr->anchor);

    if (tsPtr->state & (STATE_DISABLED | STATE_EMPHASIS)) {
	TkBorder *borderPtr = (TkBorder *)tsPtr->border;
	XColor *color1, *color2;
	
	color1 = borderPtr->lightColor, color2 = borderPtr->darkColor;
	if (tsPtr->state & STATE_EMPHASIS) {
	    XColor *hold;
	    
	    hold = color1, color1 = color2, color2 = hold;
	}
	if (color1 != NULL) {
	    XSetForeground(Tk_Display(tkwin), tsPtr->gc, color1->pixel);
	    DrawRotatedLayout(Tk_Display(tkwin), drawable, Tk_Depth(tkwin), 
		tsPtr->gc, tsPtr->font, angle, layoutPtr, xMax); 
	}
	if (color2 != NULL) {
	    XSetForeground(Tk_Display(tkwin), tsPtr->gc, color2->pixel);
	    DrawRotatedLayout(Tk_Display(tkwin), drawable, Tk_Depth(tkwin), 
		tsPtr->gc, tsPtr->font, angle, layoutPtr, xMax); 
	}
	XSetForeground(Tk_Display(tkwin), tsPtr->gc, tsPtr->color->pixel);
	return TRUE;
    }
    {
	XSetForeground(Tk_Display(tkwin), tsPtr->gc, tsPtr->color->pixel);
	DrawRotatedLayout(Tk_Display(tkwin), drawable, Tk_Depth(tkwin), 
		tsPtr->gc, tsPtr->font, angle, layoutPtr, xMax); 
    }
    return TRUE;
}

#ifndef WIN32
void
Blt_DrawTextWithRotatedBitmap(
    Tk_Window tkwin,
    Drawable drawable,
    double angle,
    TextStyle *tsPtr,		/* Text attribute information */
    TextLayout *layoutPtr,
    int x, int y,		/* Window coordinates to draw text */
    int xMax)
{
    int width, height;
    Display *display;
    Pixmap bitmap;

    display = Tk_Display(tkwin);
    /*
     * Rotate the text by writing the text into a bitmap and rotating
     * the bitmap.  Set the clip mask and origin in the GC first.  And
     * make sure we restore the GC because it may be shared.
     */
    tsPtr->angle = angle;

    bitmap = Blt_TextToBitmap(tkwin, layoutPtr, tsPtr, &width, &height, xMax);
    if ((bitmap != None) && (tsPtr->angle != 0.0)) {
	Pixmap rotated;

	rotated = Blt_RotateBitmap(tkwin, bitmap, width, height, tsPtr->angle, 
		&width, &height);
	Tk_FreePixmap(display, bitmap);
	bitmap = rotated;
    }
    if (bitmap == None) {
	return;
    }
    Blt_TranslateAnchor(x, y, width, height, tsPtr->anchor, &x, &y);
    XSetClipMask(display, tsPtr->gc, bitmap);

    if (tsPtr->state & (STATE_DISABLED | STATE_EMPHASIS)) {
	TkBorder *borderPtr = (TkBorder *) tsPtr->border;
	XColor *color1, *color2;

	color1 = borderPtr->lightColor, color2 = borderPtr->darkColor;
	if (tsPtr->state & STATE_EMPHASIS) {
	    XColor *hold;

	    hold = color1, color1 = color2, color2 = hold;
	}
	if (color1 != NULL) {
	    XSetForeground(display, tsPtr->gc, color1->pixel);
	}
	XSetClipOrigin(display, tsPtr->gc, x + 1, y + 1);
	XCopyPlane(display, bitmap, drawable, tsPtr->gc, 0, 0, width, 
		height, x + 1, y + 1, 1);
	if (color2 != NULL) {
	    XSetForeground(display, tsPtr->gc, color2->pixel);
	}
	XSetClipOrigin(display, tsPtr->gc, x, y);
	XCopyPlane(display, bitmap, drawable, tsPtr->gc, 0, 0, width, 
		height, x, y, 1);
	XSetForeground(display, tsPtr->gc, tsPtr->color->pixel);
    } else {
	XSetForeground(display, tsPtr->gc, tsPtr->color->pixel);
	XSetClipOrigin(display, tsPtr->gc, x, y);
	XCopyPlane(display, bitmap, drawable, tsPtr->gc, 0, 0, width, height, 
		x, y, 1);
    }
    XSetClipMask(display, tsPtr->gc, None);
    Tk_FreePixmap(display, bitmap);
}
#endif /* !WIN32*/

/*
 * -----------------------------------------------------------------
 *
 * Blt_TextDrawLayout --
 *
 *	Draw a text string, possibly rotated, using the the given
 *	window coordinates as an anchor for the text bounding box.
 *	If the text is not rotated, simply use the X text drawing
 *	routines. Otherwise, generate a bitmap of the rotated text.
 *
 * Results:
 *	Returns the x-coordinate to the right of the text.
 *
 * Side Effects:
 *	Text string is drawn using the given font and GC at the
 *	the given window coordinates.
 *
 *      The Stipple, FillStyle, and TSOrigin fields of the GC are
 *      modified for rotated text.  This assumes the GC is private,
 *      *not* shared (via Tk_GetGC)
 *
 * -----------------------------------------------------------------
 */
void
Blt_TextDrawLayout(
    Tk_Window tkwin,
    Drawable drawable,
    TextLayout *layoutPtr,
    TextStyle *tsPtr,		/* Text attribute information */
    int x, int y,		/* Window coordinates to draw text */
    int xMax)
{
    double angle;

    if ((tsPtr->gc == NULL) || (tsPtr->flags & UPDATE_GC)) {
	Blt_TextResetStyle(tkwin, tsPtr);
    }
    angle = FMOD(tsPtr->angle, (double)360.0);
    if (angle < 0.0) {
	angle += 360.0;
    }
    if (angle == 0.0) {
	/*
	 * This is the easy case of no rotation. Simply draw the text
	 * using the standard drawing routines.  Handle offset printing
	 * for engraved (disabled) text.
	 */
	DrawStandardLayout(Tk_Display(tkwin), drawable, Tk_Depth(tkwin),
		tsPtr, layoutPtr, x, y, xMax);
	return;
    }
    if (Blt_CanRotateFont(tsPtr->font, angle)) {
	if (Blt_DrawTextWithRotatedFont(tkwin, drawable, angle,
		tsPtr, layoutPtr, x, y, xMax)) {
	    return;		/* Success. */
	}
    }
    /*Fallthru*/
    tsPtr->angle = (float)angle;
    Blt_DrawTextWithRotatedBitmap(tkwin, drawable, angle, tsPtr, layoutPtr,
		x, y, xMax);
}


void
Blt_DrawText2(
    Tk_Window tkwin,
    Drawable drawable,
    char *string,
    TextStyle *tsPtr,		/* Text attribute information */
    int x, int y,		/* Window coordinates to draw text */
    Dim2D *areaPtr)
{
    TextLayout *layoutPtr;
    int width, height;
    double angle;

    if ((string == NULL) || (*string == '\0')) {
	return;			/* Empty string, do nothing */
    }
    layoutPtr = Blt_TextCreateLayout(string, tsPtr);
    Blt_TextDrawLayout(tkwin, drawable, layoutPtr, tsPtr, x, y, -1);
    angle = FMOD(tsPtr->angle, (double)360.0);
    if (angle < 0.0) {
	angle += 360.0;
    }
    width = layoutPtr->width;
    height = layoutPtr->height;
    if (angle != 0.0) {
	double rotWidth, rotHeight;

	Blt_GetBoundingBox(width, height, angle, &rotWidth, &rotHeight, 
	   (Point2D *)NULL);
	width = ROUND(rotWidth);
	height = ROUND(rotHeight);
    }
    areaPtr->width = width;
    areaPtr->height = height;
    Blt_Free(layoutPtr);
}

void
Blt_DrawText(
    Tk_Window tkwin,
    Drawable drawable,
    char *string,
    TextStyle *tsPtr,		/* Text attribute information */
    int x, int y)		/* Window coordinates to draw text */
{
    TextLayout *layoutPtr;

    if ((string == NULL) || (*string == '\0')) {
	return;			/* Empty string, do nothing */
    }
    layoutPtr = Blt_TextCreateLayout(string, tsPtr);
    Blt_TextDrawLayout(tkwin, drawable, layoutPtr, tsPtr, x, y, -1);
    Blt_Free(layoutPtr);
}

GC
Blt_GetBitmapGC(Tk_Window tkwin)
{
    int isNew;
    GC gc;
    Display *display;
    Blt_HashEntry *hPtr;

    if (!initialized) {
	Blt_InitHashTable(&bitmapGCTable, BLT_ONE_WORD_KEYS);
	initialized = TRUE;
    }
    display = Tk_Display(tkwin);
    hPtr = Blt_CreateHashEntry(&bitmapGCTable, (char *)display, &isNew);
    if (isNew) {
	Pixmap bitmap;
	XGCValues gcValues;
	unsigned long gcMask;
	Window root;

	root = Tk_RootWindow(tkwin);
	bitmap = Tk_GetPixmap(display, root, 1, 1, 1);
	gcValues.foreground = gcValues.background = 0;
	gcMask = (GCForeground | GCBackground);
	gc = Blt_GetPrivateGCFromDrawable(display, bitmap, gcMask, &gcValues);
	Tk_FreePixmap(display, bitmap);
	Blt_SetHashValue(hPtr, gc);
    } else {
	gc = (GC)Blt_GetHashValue(hPtr);
    }
    return gc;
}

void
Blt_TextResetStyle(Tk_Window tkwin, TextStyle *tsPtr)
{
    GC newGC;
    XGCValues gcValues;
    unsigned long gcMask;

    gcMask = GCFont;
    gcValues.font = Blt_FontId(tsPtr->font);
    if (tsPtr->color != NULL) {
	gcMask |= GCForeground;
	gcValues.foreground = tsPtr->color->pixel;
    }
    newGC = Tk_GetGC(tkwin, gcMask, &gcValues);
    if (tsPtr->gc != NULL) {
	Tk_FreeGC(Tk_Display(tkwin), tsPtr->gc);
    }
    tsPtr->gc = newGC;
    tsPtr->flags &= ~UPDATE_GC;
}

void
Blt_TextFreeStyle(display, tsPtr)
    Display *display;
    TextStyle *tsPtr;
{
    if (tsPtr->gc != NULL) {
	Tk_FreeGC(display, tsPtr->gc);
    }
}


#ifndef HAVE_LIBXFT

#include "tkFont.h"
CONST char *
Blt_FontFamily(Blt_Font font)
{
    return ((TkFont *)font)->fa.family;
    
}
#endif

#ifdef notdef
int
Blt_TextDrawLayoutPlus(
    Tk_Window tkwin,
    Drawable drawable,
    TextLayout *layoutPtr,
    TextStyle *tsPtr,		/* Text attribute information */
    int x, int y,		/* Window coordinates to draw text */
    int xMax)
{
    Blt_Painter painter;
    Blt_Picture picture, rotated;
    Pix32 fg, bg;
    Pixmap bitmap;
    int width, height;

    bitmap = Blt_TextToBitmap(tkwin, layoutPtr, tsPtr, &width, &height, xMax);
    if (bitmap == None) {
	return FALSE;
    }
    if (tsPtr->color == NULL) {
	fg.u32 = 0x00000000;
    } else {
	fg.Red = tsPtr->color->red >> 8;
	fg.Green = tsPtr->color->green >> 8;
	fg.Blue = tsPtr->color->blue >> 8;
	fg.Alpha = 0xFF;
    }
    if (tsPtr->border == NULL) {
	bg.u32 = fg.u32;	/* Transparent pixel. */
	bg.Alpha = 0x00;
    } else {
	XColor *colorPtr;

	colorPtr = Tk_3DBorderColor(tsPtr->border);
	bg.Red = colorPtr->red >> 8;
	bg.Green = colorPtr->green >> 8;
	bg.Blue = colorPtr->blue >> 8;
	bg.Alpha = 0xFF;
    }
    picture = Blt_BitmapToPicture(Tk_Display(tkwin), bitmap, width, height, 
	&fg, &bg);
    if (picture == NULL) {
	return FALSE;
    }
    rotated = Blt_RotatePicture(picture, tsPtr->angle);
    Blt_FreePicture(picture);
    if (rotated == NULL) {
	return FALSE;
    }
    picture = rotated;

    Blt_TranslateRotatedAnchor(x, y, width, height, tsPtr->anchor, 
	tsPtr->angle, &x, &y);
    painter = Blt_GetPainter(tkwin, 1.0);
    width = Blt_PictureWidth(picture);
    height = Blt_PictureHeight(picture);
    if ((x + width) > Tk_Width(tkwin)) {
	width = Tk_Width(tkwin) - x;
    }
    if ((y + height) > Tk_Height(tkwin)) {
	height = Tk_Height(tkwin) - y;
    }
    Blt_PaintPicture(painter, drawable, picture, 0, 0, width, height, x, y, 0);
    Blt_FreePainter(painter);
    Blt_FreePicture(picture);
    return TRUE;
}

void
Blt_DrawText3(
    Tk_Window tkwin,
    Drawable drawable,
    char *string,
    TextStyle *tsPtr,		/* Text attribute information */
    int x, int y)		/* Window coordinates to draw text */
{
    TextLayout *layoutPtr;

    if ((string == NULL) || (*string == '\0')) {
	return;			/* Empty string, do nothing */
    }
    layoutPtr = Blt_TextCreateLayout(string, tsPtr);
    Blt_TextDrawLayoutPlus(tkwin, drawable, layoutPtr, tsPtr, x, y, -1);
    Blt_Free(layoutPtr);
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_TranslateAnchor --
 *
 * 	Translate the coordinates of a given bounding box based
 *	upon the anchor specified.  The anchor indicates where
 *	the given xy position is in relation to the bounding box.
 *
 *  		nw --- n --- ne
 *  		|            |
 *  		w   center   e
 *  		|            |
 *  		sw --- s --- se
 *
 * 	The coordinates returned are translated to the origin of the
 * 	bounding box (suitable for giving to XCopyArea, XCopyPlane, etc.)
 *
 * Results:
 *	The translated coordinates of the bounding box are returned.
 *
 * -----------------------------------------------------------------
 */
void
Blt_TranslateRotatedAnchor(
    int x, int y,		/* Window coordinates of anchor */
    int width, int height,	/* Extents of the bounding box */
    Tk_Anchor anchor,		/* Direction of the anchor */
    double angle,
    int *xPtr, int *yPtr)
{
    angle = FMOD(angle, 360.0);
    if (FMOD(angle, (double)90.0) == 0.0) {
	int quadrant;

	quadrant = (int)(angle / 90.0);
	if ((quadrant == ROTATE_270) || (quadrant == ROTATE_90)) {
	    Blt_TranslateAnchor(x, y, height, width, anchor, xPtr, yPtr);
	} else {
	    Blt_TranslateAnchor(x, y, width, height, anchor, xPtr, yPtr);
	}
    } else {
	double rWidth, rHeight;
	Point2D points[4];

	Blt_GetBoundingBox(width, height, angle, &rWidth, &rHeight, points);
	if (anchor <= TK_ANCHOR_CENTER) {
	    Blt_TranslateAnchor(x, y, ROUND(rWidth), ROUND(rHeight), anchor, 
		xPtr, yPtr);
	} else {
	    Point2D top, bottom, left, right;
	    int i;
	    
	    top = bottom = left = right = points[0];
	    for (i = 1; i < 4; i++) {
		if (points[i].y < top.y) {
		    top = points[i];
		} 
		if (points[i].y > bottom.y) {
		    bottom = points[i];
		}
		if (points[i].x < left.x) {
		    left = points[i];
		}
		if (points[i].x > right.x) {
		    right = points[i];
		}
	    }	
	    switch ((int)anchor) {
	    case BLT_ANCHOR_LEFT: /* Left */
		y -= (rHeight / 2) - left.y;
		break;
		
	    case BLT_ANCHOR_RIGHT: /* Right */
		x -= rWidth;
		y -= (rHeight / 2) - right.y;
		break;
		
	    case BLT_ANCHOR_TOP: /* Top */
		x -= (rWidth / 2) - top.x;
		break;
		
	    case BLT_ANCHOR_BOTTOM: /* Bottom */
		x -= (rWidth / 2) - bottom.x;
		y -= height;
		break;
	    }
	    *xPtr = x, *yPtr = y;
	}
    }
}

#endif
