
/*
 * bltPictureXpm.c --
 *
 * This module implements XPM file format conversion routines for
 * the picture image type in the BLT toolkit.
 *
 *	Copyright 2003-2005 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

#include "bltInt.h"

#ifdef HAVE_LIBXPM

#include <bltSink.h>
#include <bltHash.h>
#include "bltPicture.h"
#include "bltPictureFormats.h"

#include <X11/Xutil.h>

typedef struct Blt_PictureStruct Picture;

#include <X11/xpm.h>

static int
XpmHeader(Blt_DataSink data)
{
    char *line, *next;

    Blt_SinkResetCursor(data);
    for (line = Blt_SinkPointer(data); *line != '\0'; line = next) {
#define XPM_MAX_LINE 4097
	char substring[XPM_MAX_LINE+1];
	int value;

	/* Find the start of the next line */
	if ((*line == '\n') || (*line == '\r')) {
	    line++;
	}
	next = line;
	while ((*next != '\r') && (*next != '\n') && (*next != '\0')) {
	    if (!isascii(*next)) {
		return FALSE;
	    }
	    next++;
	}
	/* Verify that we won't overrun the buffer with "sscanf". */
	if ((next - line) > XPM_MAX_LINE) {
	    return FALSE;
	}
	if (sscanf(line, "#define %s %d", substring,  &value) == 2) {
	    char *p;
	    char c;
	    char *name;

	    p = strrchr(substring, '_');

	    if (p == NULL) {
		name = substring;
	    } else {
		name = p + 1;
	    }
	    c = name[0];
	    if ((c == 'f') && (strcmp("format", name) == 0)) {
		return TRUE;
	    } else {
		return FALSE;
	    }
	} else if (sscanf(line, "/* %s */", substring) == 1) {
	    if ((strcmp("XPM", substring) == 0) || 
		(strcmp("XPM2", substring) == 0)) {
		return TRUE;
	    }
	}
    }
    return FALSE;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_IsXpm --
 *
 *      Attempts to parse an XBM file header.
 *
 * Results:
 *      Returns 1 is the header is XBM and 0 otherwise.  Note that
 *      the validity of the header values is not checked here.  That's
 *	done in Blt_XbmToPicture.
 *
 *----------------------------------------------------------------------
 */
int
Blt_IsXpm(Blt_DataSink data)
{
    int bool;

    bool = XpmHeader(data);
    return bool;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_XbmToPicture --
 *
 *      Reads an XBM file and converts it into a picture.
 *
 * Results:
 *      The picture is returned.  If an error occured, such
 *	as the designated file could not be opened, NULL is returned.
 *
 *----------------------------------------------------------------------
 */
Blt_Picture
Blt_XpmToPicture(
    Tcl_Interp *interp, 
    char *fileName,
    Blt_DataSink data)
{
    Picture *destPtr;
    Pix32 *palette;
    XpmImage xpm;
    int i, result;
    int maskColorIndex;

    Blt_SinkResetCursor(data);
    result = XpmCreateXpmImageFromBuffer((char *)Blt_SinkBuffer(data), &xpm,
	(XpmInfo *)NULL);
    if (result != XpmSuccess) {
	Tcl_AppendResult(interp, "error reading \"", fileName, 
		"\" can't read XPM image. ", (char *)NULL);
	return NULL;
    }
    destPtr = NULL;
    palette = NULL;
    if ((xpm.height < 1) || (xpm.width < 1)) {
	Tcl_AppendResult(interp, "error reading \"", fileName, 
		"\" invalid XPM dimensions \"", (char *)NULL);
	Tcl_AppendResult(interp, Blt_Itoa(xpm.width), " x ", (char *)NULL);
	Tcl_AppendResult(interp, Blt_Itoa(xpm.height), "\"", (char *)NULL);
	goto bad;
    }
    if (xpm.colorTable == NULL) {
	Tcl_AppendResult(interp, "error reading \"", fileName, 
		"\" no XPM color table available. ", (char *)NULL);
	goto bad;
    }
    destPtr = Blt_CreatePicture(xpm.width, xpm.height);
    if (destPtr == NULL) {
	Tcl_AppendResult(interp, "error reading \"", fileName, 
		"\" can't allocate memory for picture.", (char *)NULL);
	goto bad;
    }
    palette = Blt_Malloc(xpm.ncolors * sizeof(Pix32));
    if (palette == NULL) {
	Tcl_AppendResult(interp, "error reading \"", fileName, 
		"\" can't allocate a ", Blt_Itoa(xpm.ncolors), 
		" color XPM palette.", (char *)NULL);
	goto bad;
    }
    maskColorIndex = -1;
    for (i = 0; i < xpm.ncolors; i++) {
        char *colorName;

        if (xpm.colorTable[i].c_color) {
             colorName = xpm.colorTable[i].c_color;
	} else if (xpm.colorTable[i].g_color) {
             colorName = xpm.colorTable[i].g_color;
	} else if (xpm.colorTable[i].g4_color) {
             colorName = xpm.colorTable[i].g4_color;
	} else if (xpm.colorTable[i].m_color) {
             colorName = xpm.colorTable[i].m_color;
	} else if (xpm.colorTable[i].symbolic) {
	    colorName = xpm.colorTable[i].symbolic;
	} else {
	    palette[i].color = 0xFFBEBEBE;
            continue;
	}
	if (strncmp(colorName, "None", 4) == 0) {
	    maskColorIndex = i;
	    palette[i].color = 0x00000000;
	    continue;
	}
	if (Blt_GetPix32(interp, colorName, palette + i) != TCL_OK) {
	    palette[i].color = 0xFFBEBEBE;
	}
    }
    {
	int y;
	unsigned int *pixelPtr;		/* Pointer */
	Pix32 *destRowPtr;

	destRowPtr = destPtr->bits;
	pixelPtr = (int*)xpm.data;
	for (y = 0; y < xpm.height; y++) {
	    Pix32 *dp, *dend;
	    
	    for (dp = destRowPtr, dend = dp + xpm.width; dp < dend; dp++) {
		if (*pixelPtr >= xpm.ncolors) {
		    Tcl_AppendResult(interp, "error reading \"", fileName, 
			"\" bad color index ", Blt_Itoa(*pixelPtr), 
			" in XPM image.", (char *)NULL);
		    goto bad;
		}
		if (*pixelPtr == maskColorIndex) {
		    destPtr->flags |= BLT_PICTURE_MASK;
		}
		*dp = palette[*pixelPtr];
		pixelPtr++;
	    }
	    destRowPtr += destPtr->pixelsPerRow;
	}
    }
    Blt_Free(palette);
    XpmFreeXpmImage(&xpm);
    return destPtr;

 bad:
    if (destPtr != NULL) {
	Blt_FreePicture(destPtr);
    }
    if (palette != NULL) {
	Blt_Free(palette);
    }
    XpmFreeXpmImage(&xpm);
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_PictureToXpm --
 *
 *      Reads an XBM file and converts it into a picture.
 *
 * Results:
 *      The picture is returned.  If an error occured, such
 *	as the designated file could not be opened, NULL is returned.
 *
 *----------------------------------------------------------------------
 */
int
Blt_PictureToXpm(
    Tcl_Interp *interp, 
    Blt_Picture picture,
    Blt_DataSink data,
    int quantize,
    Pix32 *bgColorPtr)
{
    Picture *srcPtr;
    int nColors;
    Blt_HashTable colorTable;
    char fmt[20];

    srcPtr = picture;
    nColors = Blt_QueryColors(srcPtr, (Blt_HashTable *)NULL);
    if (Blt_PictureIsBlended(srcPtr)) {
	Blt_Picture background, mask;
	Pix32 black, white;

	background = Blt_CreatePicture(srcPtr->width, srcPtr->height);
	Blt_BlankPicture(background, bgColorPtr);

	mask = Blt_CreatePicture(srcPtr->width, srcPtr->height);
	/* Don't select 100% transparent pixels */
	white.color = 0xFFFFFFFF;
	black.color = 0x01000000;
	Blt_SelectPixels(mask, srcPtr, &black, &white);
	Blt_BlendPictures(background, srcPtr);

	/* Put back the mask by and-ing the pictures together */
	Blt_AndPictures(background, mask);

	nColors = Blt_QueryColors(background, (Blt_HashTable *)NULL);
	srcPtr = background;
    }
    if ((quantize) && (nColors > 256)) {
	Blt_Picture quant;

	quant = Blt_QuantizePicture(srcPtr, 256);
	if (srcPtr != picture) {
	    Blt_FreePicture(srcPtr);
	}
	srcPtr = quant;
    }
    nColors = Blt_QueryColors(srcPtr, &colorTable);
    if (Blt_PictureIsMasked(srcPtr)) {
	nColors++;
    }

    /* Header. */
    {
	unsigned int size;
	int cpp;

	cpp = 1;
	size = 16;
	while (size < nColors) {
	    size *= 16;
	    cpp++;
	}
	sprintf(fmt, "%%0%dx", cpp);
	/* Write the header line */
	Blt_SinkPrint(data, "/* XPM */\n");
	Blt_SinkPrint(data, "static char * image_name[] = {\n");
	Blt_SinkPrint(data, "/* Creator: BLT %s */\n", BLT_VERSION);
	Blt_SinkPrint(data, "\"%d %d %d %d\",\n", srcPtr->width, 
		      srcPtr->height, nColors, cpp);
	Blt_SinkPrint(data, "/* Colors used: %d */\n", nColors);
    }

    /* Color table. */
    {
	unsigned long i;
	char *colorkey;
	Blt_HashEntry *hPtr;
	Blt_HashSearch cursor;

	colorkey = (Blt_PictureIsColor(srcPtr)) ? "c" : "m";
	i = 0;
	Blt_SinkPrint(data, "    ");
	for (hPtr = Blt_FirstHashEntry(&colorTable, &cursor); hPtr != NULL;
	     hPtr = Blt_NextHashEntry(&cursor)) {
	    Pix32 pixel;
	    unsigned long key;

	    Blt_SetHashValue(hPtr, i);
	    key = (unsigned long)Blt_GetHashKey(&colorTable, hPtr);
	    pixel.color = (unsigned int)key;
	    Blt_SinkPrint(data, "\"");
	    Blt_SinkPrint(data, fmt, i);
	    Blt_SinkPrint(data, " %s #%02x%02x%02x\", ", colorkey, 
		pixel.Red, pixel.Green, pixel.Blue);
	    i++;
	    if ((i % 4) == 0) {
		Blt_SinkPrint(data, "\n    ");
	    }
	}
	if (Blt_PictureIsMasked(srcPtr)) {
	    i++;
	    Blt_SinkPrint(data, "\"");
	    Blt_SinkPrint(data, fmt, i);
	    Blt_SinkPrint(data, " %s None\",\n", colorkey, i);
	}	    
	if ((i % 4) != 0) {
	    Blt_SinkPrint(data, "\n");
	}
    }
    /* Image data. */
    {
	Pix32 *srcRowPtr;
	int y;
	int count;

	srcRowPtr = srcPtr->bits;
	count = 0;
	for (y = 0; y < srcPtr->height; y++) {
	    Pix32 *sp, *send;
	    Blt_SinkPrint(data, "\"");
	    for (sp = srcRowPtr, send = sp + srcPtr->width; sp < send; sp++) {
		Blt_HashEntry *hPtr;
		unsigned long i;
		unsigned long key;
		Pix32 pixel;

		pixel.color = sp->color;
		pixel.Alpha = 0xFF;
		key = (unsigned long)pixel.color;
		hPtr = Blt_FindHashEntry(&colorTable, (char *)key);
		if (hPtr == NULL) {
		    fprintf(stderr, "can't find %x\n", sp->color);
		    Blt_SinkPrint(data, fmt, nColors);
		    continue;
		}
		if (sp->Alpha == 0x00) {
		    i = (unsigned long)nColors;
		} else {
		    i = (unsigned long)Blt_GetHashValue(hPtr);
		}
		Blt_SinkPrint(data, fmt, i);
	    }
	    Blt_SinkPrint(data, "\",\n");
	    srcRowPtr += srcPtr->pixelsPerRow;
	}
    } 
    Blt_SinkPrint(data, "};\n");
    if (srcPtr != picture) {
	Blt_FreePicture(srcPtr);
    }
    Blt_DeleteHashTable(&colorTable);
    return TCL_OK;
}

#endif /* HAVE_LIBXPM */
 
