
/*
 * bltPictureXbm.c --
 *
 * This module implements XBM file format conversion routines for
 * the picture image type in the BLT toolkit.
 *
 *	Copyright 2003-2005 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

#include "bltInt.h"
#include <bltSink.h>
#include "bltPicture.h"
#include "bltPictureFormats.h"

#include <X11/Xutil.h>

typedef struct Blt_PictureStruct Picture;

#include <ctype.h>

typedef struct {
    int hotX, hotY;
    int width, height;
    int version;
}  Xbm;

/*
 * Parse the lines that define the dimensions of the bitmap,
 * plus the first line that defines the bitmap data (it declares
 * the name of a data variable but doesn't include any actual
 * data).  These lines look something like the following:
 *
 *		#define foo_width 16
 *		#define foo_height 16
 *		#define foo_x_hot 3
 *		#define foo_y_hot 3
 *		static char foo_bits[] = {
 *
 * The x_hot and y_hot lines may or may not be present.  It's
 * important to check for "char" in the last line, in order to
 * reject old X10-style bitmaps that used shorts.
 */

static int
XbmHeader(Blt_DataSink data, Xbm *xbmPtr)
{
    unsigned char *line, *next;

    xbmPtr->width = xbmPtr->height = 0;
    xbmPtr->hotX = xbmPtr->hotY = -1;
    xbmPtr->version = 11;

    Blt_SinkResetCursor(data);
    for (line = Blt_SinkPointer(data); *line != '\0'; line = next) {
#define XBM_MAX_LINE 1023
	char substring[XBM_MAX_LINE+1];
	int value;

	/* Find the start of the next line */
	if ((*line == '\n') || (*line == '\r')) {
	    line++;
	}
	next = line;
	while ((*next != '\r') && (*next != '\n') && (*next != '\0')) {
	    if (!isascii(*next)) {
		return FALSE;
	    }
	    next++;
	}
	/* Verify that we won't overrun the buffer with "sscanf". */
	if ((next - line) > XBM_MAX_LINE) {
	    return FALSE;
	}
	if (sscanf(line, "#define %s %d", substring,  &value) == 2) {
	    char *name;
	    char c;
	    char *p;

	    p = strrchr(substring, '_');
	    if (p == NULL) {
		name = substring;
	    } else {
		name = p + 1;
	    }
	    c = name[0];
	    if ((c == 'w') && (strcmp("width", name) == 0)) {
		xbmPtr->width = value;
	    } else if ((c == 'h') && (strcmp("height", name) == 0)){
		xbmPtr->height = value;
	    } else if ((c == 'h') && (strcmp("hot", name) == 0)) {
		name -= 2;
		if (name > substring) {
		    if (name[1] == '_') {
			if (name[0] == 'x') {
			    xbmPtr->hotX = value;
			} else if (name[0] == 'y') {
			    xbmPtr->hotY = value;
			}
		    }
		}
	    }
	    continue;
	} else if (sscanf(line, "static short %s = {", substring) == 1) {
	    xbmPtr->version = 10;
	} else if (sscanf(line,"static unsigned char %s = {", substring) == 1) {
	    xbmPtr->version = 11;
	} else if (sscanf(line, "static char %s = {", substring) == 1) {
	    xbmPtr->version = 11;
	} else {
	    continue;
	}
	if (*next == '\r') {
	    next++;
	}
	Blt_SinkSetPointer(data, next);
	return TRUE;
    }
    return FALSE;
}

/*
 * -----------------------------------------------------------------------
 *
 * XbmGetHexValue --
 *
 *	Converts the hexadecimal string into an unsigned integer
 *	value.  The hexadecimal string need not have a leading "0x".
 *
 * Results:
 *	Returns a standard TCL result. If the conversion was
 *	successful, TCL_OK is returned, otherwise TCL_ERROR.
 *
 * Side Effects:
 * 	If the conversion fails, interp->result is filled with an
 *	error message.
 *
 * -----------------------------------------------------------------------
 */
static int
XbmGetHexValue(const char *string, int *valuePtr)
{
    static unsigned char hexTable[256];
    static int initialized = 0;
    const char *s;
    int accum;

    if (!initialized) {
	Blt_InitHexTable(hexTable);
	initialized++;
    }

    s = string;
    if ((s[0] == '0') && ((s[1] == 'x') || (s[1] == 'X'))) {
	s += 2;
    }
    if (s[0] == '\0') {
	return FALSE;	/* Error: empty string or only found "0x".  */
    }
    accum = 0;
    for ( /*empty*/ ; *s != '\0'; s++) {
	/* Check if character is a hex digit and accumulate. */
	if (!isxdigit(*s)) {
	    return FALSE;	/* Not a hexadecimal number */
	}
	accum = (accum << 4) + hexTable[(int)*s];
    }
    *valuePtr = accum;
    return TRUE;
}

/*
 * -----------------------------------------------------------------------
 *
 * XbmBitmapData --
 *
 *	Converts a list of ASCII values into a picture.
 *
 * Results:
 *	A standard TCL result.
 *
 * Side Effects:
 * 	If an error occurs while processing the data, interp->result
 * 	is filled with a corresponding error message.  
 *
 *	The data sink is damaged by "strtok".  Unlike XbmReadHeader,
 *	here it's okay, since if we're already this far, we know it's
 *	an XBM image and we assume that the data sink won't be reused.
 *
 * -----------------------------------------------------------------------
 */
static int
XbmBitmapData(
    Blt_DataSink data,		/* Data sink to be read from. */
    int version,		/* X10 or X11. */
    Pix32 *fgColorPtr,		/* Foreground bitmap color (bit is 1). */
    Pix32 *bgColorPtr,		/* Background bitmap color (bit is 0). */
    Picture *destPtr)		/* Picture to be populated from XBM
				 * image. */
{
    Pix32 *destRowPtr;
    char *string;		/* Used to tell strtok that have
				 * continued processing the same
				 * string. */
    int y;

    destRowPtr = destPtr->bits;
    string = Blt_SinkPointer(data);
    for (y = 0; y < destPtr->height; y++) {
	int x;
	Pix32 *dp;

	dp = destRowPtr;
	if (version == 10) {
	    for (x = 0; x < destPtr->width; /* empty */) {
		char *p;
		int i, u16;
		
		p = strtok(string, ",;}\n\r\t "), string = NULL;
		if ((p == NULL) || (!XbmGetHexValue(p, &u16))) {
		    return FALSE;
		}
		for (i = 0; (x < destPtr->width) && (i < 16); i++, x++) {
		    dp->color = (u16 & (1<<i)) ? 
			fgColorPtr->color : bgColorPtr->color;
		    dp++;
		}
	    }
	} else {
	    for (x = 0; x < destPtr->width; /* empty */) {
		char *p;
		int i, u8;

		p = strtok(string, ",;}\n\r\t "), string = NULL;
		if ((p == NULL) || (!XbmGetHexValue(p, &u8))) {
		    return FALSE;
		}
		for (i = 0; (x < destPtr->width) && (i < 8); i++, x++) {
		    dp->color = (u8 & (1<<i)) ? 
			fgColorPtr->color : bgColorPtr->color;
		    dp++;
		}
	    }
	}
	destRowPtr += destPtr->pixelsPerRow;
    }
    if (bgColorPtr->Alpha == 0x00) {	/* Background is 100% transparent. */
	if (fgColorPtr->Alpha == 0xFF) {
	    destPtr->flags |= BLT_PICTURE_MASK;
	} else {
	    destPtr->flags |= BLT_PICTURE_BLEND;
	}
    } else if (bgColorPtr->Alpha != 0xFF) { /* Partial transparency. */
	destPtr->flags |= BLT_PICTURE_BLEND;
    } else if (fgColorPtr->Alpha == 0x00) { 
	/* Background is 100% opaque and foreground is 100%
	 * transparent. */
	destPtr->flags |= BLT_PICTURE_MASK;
    }
    return TRUE;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_IsXbm --
 *
 *      Attempts to parse an XBM file header.
 *
 * Results:
 *      Returns 1 is the header is XBM and 0 otherwise.  Note that
 *      the validity of the header values is not checked here.  That's
 *	done in Blt_XbmToPicture.
 *
 *----------------------------------------------------------------------
 */
int
Blt_IsXbm(Blt_DataSink data)
{
    Xbm xbm;
    int bool;

    bool = XbmHeader(data, &xbm);
    return bool;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_XbmToPicture --
 *
 *      Reads an XBM file and converts it into a picture.
 *
 * Results:
 *      The picture is returned.  If an error occured, such
 *	as the designated file could not be opened, NULL is returned.
 *
 *----------------------------------------------------------------------
 */
Blt_Picture
Blt_XbmToPicture(
    Tcl_Interp *interp, 
    char *fileName,
    Blt_DataSink data,
    Pix32 *fgColorPtr, 
    Pix32 *bgColorPtr)
{
    Blt_Picture picture;
    Xbm xbm;

    if (!XbmHeader(data, &xbm)) {
	    Tcl_AppendResult(interp, "error reading \"", fileName, 
		"\" invalid XBM header", (char *)NULL);
	return NULL;
    }
    if ((xbm.width > 0) && (xbm.height > 0)) {
	picture = Blt_CreatePicture(xbm.width, xbm.height);
	if (!XbmBitmapData(data, xbm.version, fgColorPtr, bgColorPtr, picture)) {
	    Tcl_AppendResult(interp, "error reading \"", fileName, 
		"\" invalid XBM data", (char *)NULL);
	    Blt_FreePicture(picture);
	    return NULL;
	}
    } else {
	Tcl_AppendResult(interp, "error reading \"", fileName, 
		"\" invalid XBM dimensions \"", (char *)NULL);
	Tcl_AppendResult(interp, Blt_Itoa(xbm.width), " x ", (char *)NULL);
	Tcl_AppendResult(interp, Blt_Itoa(xbm.height), "\"", (char *)NULL);
	return NULL;
    }
    return picture;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_PictureToXbm --
 *
 *      Writes a XBM format image to the provided data buffer.
 *
 * Results:
 *      A standard Tcl result.  If an error occured, TCL_ERROR is
 *	returned and an error message will be place in the interpreter
 *	result. Otherwise, the data sink will contain the binary
 *	output of the image.
 *
 * Side Effects:
 *	Memory is allocated for the data sink.
 *
 *----------------------------------------------------------------------
 */
int
Blt_PictureToXbm(
    Tcl_Interp *interp, 
    Blt_Picture picture,
    Blt_DataSink data,
    Pix32 *bgColorPtr)
{
    Picture *srcPtr;

    srcPtr = picture;
    Blt_SinkPrint(data, "#define picture_width %d\n", srcPtr->width);
    Blt_SinkPrint(data, "#define picture_height %d\n", srcPtr->height);
    Blt_SinkPrint(data, "#define picture_x_hot %d\n", srcPtr->width / 2);
    Blt_SinkPrint(data, "#define picture_y_hot %d\n", srcPtr->height / 2);
    Blt_SinkPrint(data, "static char picture_bits[] = {\n    ");
    {
	Pix32 *cp, *srcRowPtr;
	Pix32 palette[256];
	int count, i;
	int y;
	
	/* Create a B&W palette for dithering the image. */
	for (cp = palette, i = 0; i < 256; i++, cp++) {
	    int c;
	    
	    c = (i + 127) / 255;
	    cp->Blue = cp->Green = cp->Red = (unsigned char) (c * 255 + 0.5);
	    cp->Alpha = ALPHA_OPAQUE;
	} 

	/* Remove any transparent pixels by blending into a solid
	 * white background. */
	if (!Blt_PictureIsOpaque(srcPtr)) {
	    Blt_Picture background;

	    /* Blend picture with solid color background. */
	    background = Blt_CreatePicture(srcPtr->width, srcPtr->height);
	    Blt_BlankPicture(background, bgColorPtr); 
	    Blt_BlendPictures(background, srcPtr);
	    srcPtr = background;
	}
	/* Now dither the picture to 2 colors. */
	{
	    Blt_Picture dither;

	    dither = Blt_DitherPicture(srcPtr, palette);
	    if (srcPtr != picture) {
		Blt_FreePicture(srcPtr);
	    }
	    srcPtr = dither;
	}
	/* Write the dithered image data. */
	count = 0;
	srcRowPtr = srcPtr->bits;
	for (y = 0; y < srcPtr->height; y++) {
	    Pix32 *sp;
	    int x;
	    
	    sp = srcRowPtr;
	    for (x = 0; x < srcPtr->width; /* empty */) {
		unsigned char bits;
		int i;

		bits = 0;
		for (i = 0; (x < srcPtr->width) && (i < 8); i++, x++) {
		    if (sp->Red == 0x00) {
			bits |= (1<<i);
		    }
		    sp++;
		}
		count++;
		Blt_SinkPrint(data, "0x%02x, ", bits);
		if (count > 11) {
		    Blt_SinkPrint(data, "\n    ");
		    count = 0;
		}
	    }
	    srcRowPtr += srcPtr->pixelsPerRow;
	}
	Blt_SinkPrint(data, "\n};\n");

	/* Free the dithered image. */
	Blt_FreePicture(srcPtr);
    }
    return TCL_OK;
}

