/* 
 * tkCanvLine.c --
 *
 *	This file implements line items for canvas widgets.
 *
 * Copyright (c) 1991-1994 The Regents of the University of California.
 * Copyright (c) 1994-1995 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * SCCS: @(#) tkCanvLine.c 1.43 96/02/15 18:52:30
 */

#include <stdio.h>
#include <tcl.h>
#include <tkInt.h>
#include <tkPort.h>
#include <tkCanvas.h>
#include <tkCanvPgraph.h>
/*
#include <bindata.h>
*/

#ifdef USE_DMALLOC
#include<dmalloc.h>
#endif

#define POLAR_SEGMENTED  0
#define POLAR_CONTINUOUS 1
#define POLAR_COMPLETE   2

/*
 * The structure below defines the record for each line item.
 */

typedef struct PDatasetItem  {
    Tk_Item header;		/* Generic stuff that's the same for all
				 * types.  MUST BE FIRST IN STRUCTURE. */
    Tk_Outline outline;		/* Outline structure */
    Tk_Canvas canvas;		/* Canvas containing item.  Needed for
				 * parsing arrow shapes. */
    int numPoints;		/* Number of points in line (always >= 0). */
    double *datacoordPtr;	/* Pointers to malloc-ed array containing
				 * x- and y-coords of all points in line.
				 * X-coords are even-valued indices, y-coords
				 * are corresponding odd-valued indices. */
    double *graphcoordPtr;
    double *canvascoordPtr;	/* Pointers to the list of segments to draw */
    int capStyle;		/* Cap style for line. */
    int joinStyle;		/* Join style for line. */
    int smooth;			/* Non-zero means draw line smoothed (i.e.
				 * with Bezier splines). */
    int splineSteps;		/* Number of steps in each spline segment. */
    GC bitmapGC;		/* Graphics context for drawing onto a bitmap
				 * mask. */
    double rmin, rmax;		/* Limits of the dataset */
    double tmin, tmax;		/* Limits of the dataset */
    double roffset;		/* When drawing, apply this offset to every
    				 * radial coordinate in the dataset. */
    double rscale;		/* Scaling factor to use for drawing the
    				 * dataset (should not be used) */
    int style;			/* One of POLAR_SEGMENTED, POLAR_CONTINUOUS,
     				 * or POLAR_COMPLETE */
    PGraphItem *graph;		/* Pointer to the graph in which the
				 * dataset is to be drawn.  If this is
				 * NULL, then don't draw the dataset. */
    int resize_id;		/* Id of the last graph resize.  If this
				 * does not match the value of resize_id
				 * in the graph, then we need to resize
				 * this dataset. */
    int delete_flag;		/* Set to 1 when the dataset item is being
				 * deleted or else we get a core dump when
				 * the application exits.
				 */
} PDatasetItem;

/*
 * the following structure is copied directly from tkCanvas.c.  If
 * that one changes, this one should change as well.
 */

typedef struct TagSearch {
    TkCanvas *canvasPtr;        /* Canvas widget being searched. */
    Tk_Uid tag;                 /* Tag to search for.   0 means return
				 * all items. */
    Tk_Item *prevPtr;           /* Item just before last one found (or NULL
				 * if last one found was first in the item
				 * list of canvasPtr). */
    Tk_Item *currentPtr;        /* Pointer to last item returned. */
    int searchOver;             /* Non-zero means NextItem should always
				 * return NULL. */
} TagSearch;

/*
 * Prototypes for procedures defined in this file:
 */

static void		ComputeLineBbox _ANSI_ARGS_((Tk_Canvas canvas,
			    PDatasetItem *linePtr));
static int		ConfigureLine _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int argc,
			    char **argv, int flags));
static int		CreateLine _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, struct Tk_Item *itemPtr,
			    int argc, char **argv));
static void		DeleteLine _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, Display *display));
static void		DisplayLine _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, Display *display, Drawable dst,
			    int x, int y, int width, int height));
static int		GetLineIndex _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr,
			    char *indexString, int *indexPtr));
static int		LineCoords _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr,
			    int argc, char **argv));
static void		LineDeleteCoords _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, int first, int last));
static void		LineInsert _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, int beforeThis, char *string));
static int		LineToArea _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double *rectPtr));
static double		LineToPoint _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double *coordPtr));
static int		LineToPostscript _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int prepass));
static void		ScaleDataset _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double originX, double originY,
			    double scaleX, double scaleY));
static void		TranslateDataset _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double deltaX, double deltaY));
static void		LineUpdateGraphCoords _ANSI_ARGS_((PDatasetItem *));
static void		LineUpdateCanvasCoords _ANSI_ARGS_((PDatasetItem *));
static int Tk_GraphParseProc(ClientData, Tcl_Interp *, Tk_Window, char *, char *, int);
static char * Tk_GraphPrintProc(ClientData, Tk_Window, char *, int, Tcl_FreeProc **);
static Tk_Item *	StartTagSearch _ANSI_ARGS_((TkCanvas *canvasPtr,
			    char *tag, TagSearch *searchPtr));
static int		Tk_DontSetOption _ANSI_ARGS_((ClientData, Tcl_Interp *,
			    Tk_Window, char *, char *, int));
static int		Tk_StyleParseProc _ANSI_ARGS_((ClientData, Tcl_Interp *,
			    Tk_Window, char *, char *, int));
static char * 		Tk_StylePrintProc _ANSI_ARGS_((ClientData, 
			    Tk_Window, char *, int, Tcl_FreeProc **));
static char * 		Tk_CustomPrintDouble _ANSI_ARGS_((ClientData, 
			    Tk_Window, char *, int, Tcl_FreeProc **));
static void		AddDatasetToGraph _ANSI_ARGS_((PDatasetItem *,
			    PGraphItem *));
static void		RemoveDatasetFromGraph _ANSI_ARGS_((PDatasetItem *,
			    PGraphItem *));

/*
 * Information used for parsing configuration specs.  If you change any
 * of the default strings, be sure to change the corresponding default
 * values in CreateLine.
 */

static Tk_CustomOption stateOption = {
    (Tk_OptionParseProc *) Tk_StateParseProc,
    Tk_StatePrintProc, (ClientData) 2
};
static Tk_CustomOption tagsOption = {
    (Tk_OptionParseProc *) Tk_CanvasTagsParseProc,
    Tk_CanvasTagsPrintProc, (ClientData) NULL
};
static Tk_CustomOption dashOption = {
    (Tk_OptionParseProc *) Tk_CanvasDashParseProc,
    Tk_CanvasDashPrintProc, (ClientData) NULL
};
static Tk_CustomOption tileOption = {
    (Tk_OptionParseProc *) Tk_TileParseProc,
    Tk_TilePrintProc, (ClientData) NULL
};
static Tk_CustomOption graphOption = {Tk_GraphParseProc,
    Tk_GraphPrintProc, (ClientData) NULL
};
static Tk_CustomOption minmaxOption = {Tk_DontSetOption,
    Tk_CustomPrintDouble, (ClientData) NULL
};
static Tk_CustomOption styleOption = {Tk_StyleParseProc,
    Tk_StylePrintProc, (ClientData) NULL
};

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_CUSTOM, "-activedash", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PDatasetItem, outline.activeDash),
	TK_CONFIG_NULL_OK, &dashOption},
    {TK_CONFIG_COLOR, "-activefill", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PDatasetItem, outline.activeColor),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-activestipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PDatasetItem, outline.activeStipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-activetile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PDatasetItem, outline.activeTile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_PIXELS, "-activewidth", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(PDatasetItem, outline.activeWidth),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CAP_STYLE, "-capstyle", (char *) NULL, (char *) NULL,
	"butt", Tk_Offset(PDatasetItem, capStyle), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_COLOR, "-fill", (char *) NULL, (char *) NULL,
	"black", Tk_Offset(PDatasetItem, outline.color), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-graph", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PDatasetItem, graph),
	TK_CONFIG_NULL_OK, &graphOption},
    {TK_CONFIG_CUSTOM, "-dash", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PDatasetItem, outline.dash),
	TK_CONFIG_NULL_OK, &dashOption},
    {TK_CONFIG_PIXELS, "-dashoffset", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(PDatasetItem, outline.offset),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-disableddash", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PDatasetItem, outline.disabledDash),
	TK_CONFIG_NULL_OK, &dashOption},
    {TK_CONFIG_COLOR, "-disabledfill", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PDatasetItem, outline.disabledColor),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-disabledstipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PDatasetItem, outline.disabledStipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-disabledtile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PDatasetItem, outline.disabledTile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_PIXELS, "-disabledwidth", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(PDatasetItem, outline.disabledWidth),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_JOIN_STYLE, "-joinstyle", (char *) NULL, (char *) NULL,
	"round", Tk_Offset(PDatasetItem, joinStyle), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_BOOLEAN, "-smooth", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(PDatasetItem, smooth), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_INT, "-splinesteps", (char *) NULL, (char *) NULL,
	"12", Tk_Offset(PDatasetItem, splineSteps), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-state", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(Tk_Item, state), TK_CONFIG_NULL_OK,
	&stateOption},
    {TK_CONFIG_BITMAP, "-stipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PDatasetItem, outline.stipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &tagsOption},
    {TK_CONFIG_CUSTOM, "-tile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PDatasetItem, outline.tile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_CUSTOM, "-rmin", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PDatasetItem, rmin),
	TK_CONFIG_NULL_OK, &minmaxOption},
    {TK_CONFIG_CUSTOM, "-rmax", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PDatasetItem, rmax),
	TK_CONFIG_NULL_OK, &minmaxOption},
    {TK_CONFIG_DOUBLE, "-roffset", (char *) NULL, (char *) NULL,
	"0.0", Tk_Offset(PDatasetItem, roffset), TK_CONFIG_NULL_OK},
    {TK_CONFIG_DOUBLE, "-rscale", (char *) NULL, (char *) NULL,
	"1.0", Tk_Offset(PDatasetItem, rscale), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-tmin", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PDatasetItem, tmin),
	TK_CONFIG_NULL_OK, &minmaxOption},
    {TK_CONFIG_CUSTOM, "-tmax", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PDatasetItem, tmax),
	TK_CONFIG_NULL_OK, &minmaxOption},
    {TK_CONFIG_CUSTOM, "-style", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(PDatasetItem, style),
	TK_CONFIG_NULL_OK, &styleOption},
    {TK_CONFIG_PIXELS, "-width", (char *) NULL, (char *) NULL,
	"1", Tk_Offset(PDatasetItem, outline.width),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_STRING, "-updatecommand", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(Tk_Item, updateCmd), TK_CONFIG_NULL_OK},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * The structures below defines the line item type by means
 * of procedures that can be invoked by generic item code.
 */

Tk_ItemType tkPDatasetType = {
    "pdataset",				/* name */
    sizeof(PDatasetItem),		/* itemSize */
    CreateLine,				/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureLine,			/* configureProc */
    LineCoords,				/* coordProc */
    DeleteLine,				/* deleteProc */
    DisplayLine,			/* displayProc */
    TK_ITEM_VISITOR_SUPPORT,		/* flags */
    LineToPoint,			/* pointProc */
    LineToArea,				/* areaProc */
    LineToPostscript,			/* postscriptProc */
    ScaleDataset,			/* scaleProc */
    TranslateDataset,			/* translateProc */
    GetLineIndex,			/* indexProc */
    (Tk_ItemCursorProc *) NULL,		/* icursorProc */
    (Tk_ItemSelectionProc *) NULL,	/* selectionProc */
    LineInsert,				/* insertProc */
    LineDeleteCoords,			/* dTextProc */
    (Tk_ItemType *) NULL,		/* nextPtr */
    (Tk_ItemBboxProc *) ComputeLineBbox,/* bboxProc */
    Tk_Offset(Tk_VisitorType, visitLine), /* acceptProc */
    (Tk_ItemGetCoordProc *) NULL,	/* getCoordProc */
    (Tk_ItemSetCoordProc *) NULL	/* setCoordProc */
};

/*
 * The next two Uid's are used in tag searches
 */

static Tk_Uid allUid = NULL;
static Tk_Uid currentUid = NULL;

/*
 * The definition below determines how large are static arrays
 * used to hold spline points (splines larger than this have to
 * have their arrays malloc-ed).
 */

#define MAX_STATIC_POINTS 200

/*
 *--------------------------------------------------------------
 *
 * CreateLine --
 *
 *	This procedure is invoked to create a new line item in
 *	a canvas.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	interp->result;  in this case itemPtr is left uninitialized,
 *	so it can be safely freed by the caller.
 *
 * Side effects:
 *	A new line item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateLine(interp, canvas, itemPtr, argc, argv)
    Tcl_Interp *interp;			/* Interpreter for error reporting. */
    Tk_Canvas canvas;			/* Canvas to hold new item. */
    Tk_Item *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing line. */
{
    PDatasetItem *linePtr = (PDatasetItem *) itemPtr;
    int i;

    /*
    printf("Validating memory... CreateLine(start)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    /*
     * Carry out initialization that is needed to set defaults and to
     * allow proper cleanup after errors during the the remainder of
     * this procedure.
     */

    if (allUid == NULL)
	allUid = Tk_GetUid("all");
    if (currentUid == NULL)
	currentUid = Tk_GetUid("current");

    Tk_CreateOutline(&(linePtr->outline));
    linePtr->canvas = canvas;
    linePtr->numPoints = 0;
    linePtr->datacoordPtr = NULL;
    linePtr->graphcoordPtr = NULL;
    linePtr->canvascoordPtr = NULL;
    linePtr->capStyle = CapButt;
    linePtr->joinStyle = JoinRound;
    linePtr->smooth = 0;
    linePtr->splineSteps = 12;
    linePtr->bitmapGC = None;
    linePtr->graph = NULL;
    linePtr->style = POLAR_CONTINUOUS;
    linePtr->roffset = 0.0;
    linePtr->rscale = 1.0;
    linePtr->resize_id = 0;
    linePtr->delete_flag = 0;

    /*
     * Count the number of points and then parse them into a point
     * array.  Leading arguments are assumed to be points if they
     * start with a digit or a minus sign followed by a digit.
     */

    for (i = 0; i < argc; i++) {
	if ((argv[i][0] == '-') && (argv[i][1] >= 'a')
		&& (argv[i][1] <= 'z')) {
	    break;
	}
    }
    if (i && (LineCoords(interp, canvas, itemPtr, i, argv) != TCL_OK)) {
	goto error;
    }
    if (ConfigureLine(interp, canvas, itemPtr, argc-i, argv+i, 0) == TCL_OK) {
	return TCL_OK;
    }

    error:
    DeleteLine(canvas, itemPtr, Tk_Display(Tk_CanvasTkwin(canvas)));

    /*
    printf("Validating memory... CreateLine(end)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * LineCoords --
 *
 *	This procedure is invoked to process the "coords" widget
 *	command on lines.  See the user documentation for details
 *	on what it does.
 *
 * Results:
 *	Returns TCL_OK or TCL_ERROR, and sets interp->result.
 *
 * Side effects:
 *	The coordinates for the given item may be changed.
 *
 *--------------------------------------------------------------
 */

static int
LineCoords(interp, canvas, itemPtr, argc, argv)
    Tcl_Interp *interp;			/* Used for error reporting. */
    Tk_Canvas canvas;			/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item whose coordinates are to be
					 * read or modified. */
    int argc;				/* Number of coordinates supplied in
					 * argv. */
    char **argv;			/* Array of coordinates: x1, y1,
					 * x2, y2, ... */
{
    PDatasetItem *linePtr = (PDatasetItem *) itemPtr;
    char buffer[TCL_DOUBLE_SPACE], **largv = NULL;
    int i, numPoints;
    double *coordPtr;
    int radius;
    int maxphsize;
    Tcl_Obj *objPtr = (Tcl_Obj *)NULL;
    int parseAsBinaryString = 0;

    /*
    printf("Validating memory... LineCoords(start)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    if (argc < 0) {
	/*
	 * This is our special signal to recompute the graph/canvas coordinates
	 */
	LineUpdateGraphCoords(linePtr);
	LineUpdateCanvasCoords(linePtr);
	linePtr->resize_id = linePtr->graph->resize_id;
	ComputeLineBbox(canvas, linePtr);

	return TCL_OK;
    } else if (argc == 0) {
	int numCoords;

	numCoords = 2*linePtr->numPoints;
	coordPtr = linePtr->datacoordPtr;
	for (i = 0; i < numCoords; i++, coordPtr++) {
	    /*
	    if (i == 2) {
		coordPtr = linePtr->datacoordPtr+2;
	    }
	    */
	    Tcl_PrintDouble(interp, *coordPtr, buffer);
	    Tcl_AppendElement(interp, buffer);
	}
	return TCL_OK;
    }
    if (argc == 1) {
	if (Tcl_SplitList(interp, argv[0], &argc, &largv) != TCL_OK) {
	    if (largv != NULL) {
		ckfree((char *) largv);
	    }
	    return TCL_ERROR;
	}

	/* If we have one argument, then assume that it's a Tcl_Obj object.
	 * Create the coordinate array from this object.  The idea is that
	 * we want to be able to pass the graph a pre-existing binary
	 * structure containing the coordinates so that creating a new
	 * dataset does not involve billions of calls to Tcl_GetDouble().
	 */

	if (argc == 1) {
	    Tcl_Obj *part1 = Tcl_NewStringObj(*largv, -1);
	    parseAsBinaryString = 1;

	    objPtr = Tcl_ObjGetVar2(interp, part1, (Tcl_Obj *)NULL, TCL_LEAVE_ERR_MSG);
	    if (!objPtr) {
		return TCL_ERROR;
	    }
	}
    } else {
	largv = argv;
    }
    if (argc & 1 && (objPtr == (Tcl_Obj *)NULL)) {
	Tcl_AppendResult(interp,
		"odd number of coordinates specified for line",
		(char *) NULL);
	if (largv != NULL && largv != argv) {
	    ckfree((char *) largv);
	}
	return TCL_ERROR;
    } else {
	if (parseAsBinaryString) {
	    /*
	     * 4 == size of network float
	     */

	    numPoints = objPtr->length / 4 / 2;
	} else {
	    numPoints = argc/2;
	}

	if (linePtr->numPoints != numPoints) {
	    coordPtr = (double *) ckalloc((unsigned)
		    (sizeof(double) * numPoints*2));
	    if (linePtr->datacoordPtr != NULL) {
		ckfree((char *) linePtr->datacoordPtr);
	    }
	    linePtr->datacoordPtr = coordPtr;

	    coordPtr = (double *) ckalloc((unsigned)
		    (sizeof(double) * numPoints * 2));
	    if (linePtr->canvascoordPtr != NULL) {
		ckfree((char *) linePtr->canvascoordPtr);
	    }
	    linePtr->canvascoordPtr = coordPtr;

	    coordPtr = (double *) ckalloc((unsigned)
		    (sizeof(double) * numPoints * 2));
	    if (linePtr->graphcoordPtr != NULL) {
		ckfree((char *) linePtr->graphcoordPtr);
	    }
	    linePtr->graphcoordPtr = coordPtr;

	    linePtr->numPoints = numPoints;
	}

	/*
	 * Make a separate loop for the case where the data is
	 * a Tcl_Obj.  We want to make dealing with binary
	 * objects as fast as possible.
	 */

	if (objPtr) {
	    double d;
	    float *objData = (float *)Tcl_GetStringFromObj(objPtr, (int *)NULL);
	    float *objCoordPtr = (float *)ckalloc(objPtr->length);
	    
	    /*
	     * First convert the floats from network byte order to machine
	     * byte order.
	     */
	    
	    /*
	    for(i=0; i < objPtr->length / 4; i++) {
		long nl = ((long *)(objData))[i];
		
		nl = ntohl(nl);
		((long *)(objCoordPtr))[i] = nl;
	    }
	    */

	    coordPtr = linePtr->datacoordPtr;
	    linePtr->rmin = objCoordPtr[0];
	    linePtr->tmin = objCoordPtr[1];

	    /*
	    linePtr->rmin /= width_of_screen * widthmm_of_screen;
	    linePtr->tmin /= width_of_screen * widthmm_of_screen;
	    */
	    linePtr->rmax = linePtr->rmin;
	    linePtr->tmax = linePtr->tmin;

	    for (i = 0; i <numPoints*2; i++) {
		*coordPtr = objCoordPtr[i];
		if (i%2) {
		    if (*coordPtr < linePtr->tmin) linePtr->tmin = *coordPtr;
		    if (*coordPtr > linePtr->tmax) linePtr->tmax = *coordPtr;
		} else {
		    if (*coordPtr < linePtr->rmin) linePtr->rmin = *coordPtr;
		    if (*coordPtr > linePtr->rmax) linePtr->rmax = *coordPtr;
		}
		/*
		 * No default need be supplied to the other coord arrays.
	         * These will be set shortly.
		 *
		 * linePtr->graphcoordPtr[i] = *coordPtr;
		 * linePtr->canvascoordPtr[i] = *coordPtr;
		 */
		coordPtr++;
	    }
	    if (largv != NULL && largv != argv) {
		ckfree((char *) largv);
	    }
	    if (objCoordPtr) {
		ckfree(objCoordPtr);
	    }
	} else {
	    coordPtr = linePtr->datacoordPtr;
	    Tk_CanvasGetCoord(interp, canvas, largv[0], &linePtr->rmin);
	    linePtr->rmax = linePtr->rmin;
	    Tk_CanvasGetCoord(interp, canvas, largv[1], &linePtr->tmin);
	    linePtr->tmax = linePtr->tmin;
	    for (i = 0; i <numPoints*2; i++) {
		if (Tk_CanvasGetCoord(interp, canvas, largv[i],
			coordPtr) != TCL_OK) {
		    if (largv != NULL && largv != argv) {
			ckfree((char *) largv);
		    }
		    return TCL_ERROR;
		}
		if (i%2) {
		    if (*coordPtr < linePtr->tmin) linePtr->tmin = *coordPtr;
		    if (*coordPtr > linePtr->tmax) linePtr->tmax = *coordPtr;
		} else {
		    if (*coordPtr < linePtr->rmin) linePtr->rmin = *coordPtr;
		    if (*coordPtr > linePtr->rmax) linePtr->rmax = *coordPtr;
		}
		coordPtr++;
	    }
	    if (largv != NULL && largv != argv) {
		ckfree((char *) largv);
	    }
	}

	LineUpdateGraphCoords(linePtr);
	LineUpdateCanvasCoords(linePtr);

	/*
	 * for(i = 0; i < numPoints*2; i+=2) {
	 *     fprintf(stderr, "(%g,%g)\n", linePtr->datacoordPtr[i], 
	 *	    linePtr->datacoordPtr[i+1]);
	 * }
	 */
    }
    ComputeLineBbox(canvas, linePtr);

    /*
    printf("Validating memory... LineCoords(end)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureLine --
 *
 *	This procedure is invoked to configure various aspects
 *	of a line item such as its background color.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in interp->result.
 *
 * Side effects:
 *	Configuration information, such as colors and stipple
 *	patterns, may be set for itemPtr.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureLine(interp, canvas, itemPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Canvas canvas;		/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Line item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    PDatasetItem *linePtr = (PDatasetItem *) itemPtr;
    XGCValues gcValues;
    GC newGC;
    unsigned long mask;
    Tk_Window tkwin;
    Tk_State state;

    /*
    printf("Validating memory... ConfigureLine(pre)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    if (canvas == NULL) return;
    tkwin = Tk_CanvasTkwin(canvas);
    if (tkwin == NULL) return;

    Tk_CanvasEventuallyRedraw(canvas, linePtr->header.x1, linePtr->header.y2,
	    linePtr->header.x2, linePtr->header.y2);
    if (Tk_ConfigureWidget(interp, tkwin, configSpecs, argc, argv,
	    (char *) linePtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
    printf("Validating memory... ConfigureLine(pos)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    /*
     * A few of the options require additional processing, such as
     * graphics contexts.
     *
     * Before we do these, however, we need to update the graph coordinates
     * and the canvas coordinates.
     */

    state = itemPtr->state;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    LineUpdateGraphCoords(linePtr);
    LineUpdateCanvasCoords(linePtr);

    if (linePtr->outline.activeWidth > linePtr->outline.width ||
	    linePtr->outline.activeDash.number > 0 ||
	    linePtr->outline.activeColor != NULL ||
	    linePtr->outline.activeStipple != None) {
	itemPtr->redraw_flags |= TK_ITEM_STATE_DEPENDANT;
    } else {
	itemPtr->redraw_flags &= ~TK_ITEM_STATE_DEPENDANT;
    }
    mask = Tk_ConfigOutlineGC(&gcValues, canvas, itemPtr,
	    &(linePtr->outline));
    if (mask) {
	gcValues.join_style = linePtr->joinStyle;
	mask |= GCJoinStyle;
	newGC = Tk_GetGC(tkwin, mask, &gcValues);
    } else {
	newGC = None;
    }
    if (linePtr->outline.gc != None) {
	Tk_FreeGC(Tk_Display(tkwin), linePtr->outline.gc);
    }
    linePtr->outline.gc = newGC;

    /*
     * Keep spline parameters within reasonable limits.
     */

    if (linePtr->splineSteps < 1) {
	linePtr->splineSteps = 1;
    } else if (linePtr->splineSteps > 100) {
	linePtr->splineSteps = 100;
    }

    if ((!linePtr->numPoints) || (state==TK_STATE_HIDDEN)) {
	ComputeLineBbox(canvas, linePtr);
	return TCL_OK;
    }

    if (linePtr->bitmapGC == None) {
	Display *display = Tk_Display(Tk_CanvasTkwin(canvas));
	Drawable d = Tk_WindowId(Tk_MainWindow(interp));
	Pixmap bitmap = None;
	gcValues.function = GXclear;
	gcValues.foreground = 1;
	gcValues.background = 0;
	gcValues.fill_style = FillSolid;
	gcValues.plane_mask = 1;
	gcValues.clip_x_origin = 0;
	gcValues.clip_y_origin = 0;

	/* Create a dummy bitmap of depth one to use as our drawable for
	 * the GC creation so that our GC will be able to draw on a bitmap.
	 */

	if (d) {
	    bitmap = Tk_GetPixmap(display, d, 2, 2, 1);

	    newGC = XCreateGC(display, d, GCFunction|GCForeground|GCBackground|GCFillStyle|GCPlaneMask|GCClipXOrigin|GCClipYOrigin, &gcValues);
	    if (linePtr->bitmapGC != None)
		XFreeGC(display, linePtr->bitmapGC);

	    linePtr->bitmapGC = newGC;

	    Tk_FreePixmap(display, bitmap);
	}
    }

    /*
     * Recompute bounding box for line.
     */

    Tk_CanvasEventuallyRedraw(canvas, linePtr->header.x1, linePtr->header.y2,
	    linePtr->header.x2, linePtr->header.y2);
    ComputeLineBbox(canvas, linePtr);

    /*
    printf("Validating memory... ConfigureLine(end)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DeleteLine --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a line item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteLine(canvas, itemPtr, display)
    Tk_Canvas canvas;			/* Info about overall canvas widget. */
    Tk_Item *itemPtr;			/* Item that is being deleted. */
    Display *display;			/* Display containing window for
					 * canvas. */
{
    PDatasetItem *linePtr = (PDatasetItem *) itemPtr;

    /*
    printf("Validating memory... DeleteLine(start)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    linePtr->delete_flag = 1;

    Tk_DeleteOutline(display, &(linePtr->outline));
    if (linePtr->datacoordPtr != NULL) {
	ckfree((char *) linePtr->datacoordPtr);
    }
    if (linePtr->graphcoordPtr != NULL) {
	ckfree((char *) linePtr->graphcoordPtr);
    }
    if (linePtr->canvascoordPtr != NULL) {
	ckfree((char *) linePtr->canvascoordPtr);
    }
    if (linePtr->bitmapGC != None) {
	XFreeGC(display, linePtr->bitmapGC);
    }
    if (linePtr->graph) {
	RemoveDatasetFromGraph(linePtr, linePtr->graph);
    }

    /*
    printf("Validating memory... DeleteLine(end)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */
}

/*
 *--------------------------------------------------------------
 *
 * ComputeLineBbox --
 *
 *	This procedure is invoked to compute the bounding box of
 *	all the pixels that may be drawn as part of a line.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The fields x1, y1, x2, and y2 are updated in the header
 *	for itemPtr.
 *
 *--------------------------------------------------------------
 */

static void
ComputeLineBbox(canvas, linePtr)
    Tk_Canvas canvas;			/* Canvas that contains item. */
    PDatasetItem *linePtr;		/* Item whose bbox is to be
					 * recomputed. */
{
    double *coordPtr;
    int i,width;
    Tk_State state = linePtr->header.state;
    double *segPtr;

    /*
    printf("Validating memory... ComputeLineBbox(start)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    if (!(linePtr->numPoints) || (state==TK_STATE_HIDDEN)
	|| (linePtr->graph == NULL)) {
	linePtr->header.x1 = -1;
	linePtr->header.x2 = -1;
	linePtr->header.y1 = -1;
	linePtr->header.y2 = -1;
	return;
    }

    width = linePtr->outline.width;
    if (((TkCanvas *)canvas)->currentItemPtr == (Tk_Item *)linePtr) {
	if (linePtr->outline.activeWidth>width) {
	    width = linePtr->outline.activeWidth;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (linePtr->outline.disabledWidth>0) {
	    width = linePtr->outline.disabledWidth;
	}
    }
    segPtr = linePtr->canvascoordPtr;
    linePtr->header.x1 = segPtr[0];
    linePtr->header.y1 = segPtr[1];
    linePtr->header.x2 = segPtr[0];
    linePtr->header.y2 = segPtr[1];

    /*
     * Compute the bounding box of all the points in the dataset,
     * then expand in all directions by the line's width to take
     * care of butting or rounded corners and projecting or
     * rounded caps.  This expansion is an overestimate (worst-case
     * is square root of two over two) but it's simple.  Don't do
     * anything special for curves.  This causes an additional
     * overestimate in the bounding box, but is faster.
     */

    for (i = 0, segPtr = linePtr->canvascoordPtr;
	 i < linePtr->numPoints;
	 i++, segPtr+=2) {
	if (linePtr->header.x1 > segPtr[0])
	    linePtr->header.x1 = segPtr[0];
	if (linePtr->header.y1 > segPtr[1])
	    linePtr->header.y1 = segPtr[1];
	if (linePtr->header.x2 < segPtr[0])
	    linePtr->header.x2 = segPtr[0];
	if (linePtr->header.y2 < segPtr[1])
	    linePtr->header.y2 = segPtr[1];
    }
    linePtr->header.x1 -= width;
    linePtr->header.x2 += width;
    linePtr->header.y1 -= width;
    linePtr->header.y2 += width;

    /*
     * Add one more pixel of fudge factor just to be safe (e.g.
     * X may round differently than we do).
     */

    linePtr->header.x1 -= 1;
    linePtr->header.x2 += 1;
    linePtr->header.y1 -= 1;
    linePtr->header.y2 += 1;

    /*
    printf("Validating memory... ComputeLineBbox(end)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */
}

/*
 *--------------------------------------------------------------
 *
 * DisplayLine --
 *
 *	This procedure is invoked to draw a line item in a given
 *	drawable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in drawable using the transformation
 *	information in canvas.
 *
 *--------------------------------------------------------------
 */

static void
DisplayLine(canvas, itemPtr, display, drawable, x, y, width, height)
    Tk_Canvas canvas;			/* Canvas that contains item. */
    Tk_Item *itemPtr;			/* Item to be displayed. */
    Display *display;			/* Display on which to draw item. */
    Drawable drawable;			/* Pixmap or window in which to draw
					 * item. */
    int x, y, width, height;		/* Describes region of canvas that
					 * must be redisplayed (not used). */
{
    PDatasetItem *linePtr = (PDatasetItem *) itemPtr;
    XSegment	staticSegs[MAX_STATIC_POINTS];
    XPoint	staticPoints[MAX_STATIC_POINTS];
    XSegment	*segPtr;
    XPoint	*pointPtr;
    short	origin[2];
    double	*coordPtr;
    double	*temp_coordPtr;
    double	length,
    		rad;
    double	xtmp,
    		ytmp,
		r0,
		r1;
    short	graph_clip_xorigin,
    		graph_clip_yorigin;
    double	xorigin,
    		yorigin;
    XRectangle	rect;
    int		i,
    		p,
    		numPoints;
    Tk_State state = itemPtr->state;
    Pixmap graph_clip_mask;

    /*
    printf("Validating memory... DisplayLine(start)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    /*
     * Draw the line into the parent graph's drawable pixmap.  Then copy the
     * pixmap onto the drawable that we were provided here.
     *
     * If there is no parent graph, then don't do anything.
     */

    if (linePtr->graph == NULL)
	return;

    if (linePtr->graph->resize_id != linePtr->resize_id) {
	LineUpdateGraphCoords(linePtr);
	LineUpdateCanvasCoords(linePtr);
	ComputeLineBbox(canvas, linePtr);
	linePtr->resize_id = linePtr->graph->resize_id;
    }

    /*
     * We want to clip the datasets at the graph boundaries so we don't
     * get lines that extend past the edges of the graph.
     *
     * linePtr->graph->clipMask contains a bitmap of 1's which is exactly
     * the same size as the graph.  All we have to do is set the
     * clip origin.
     */

    graph_clip_mask = linePtr->graph->clipMask;
    Tk_CanvasDrawableCoords(canvas,
	    linePtr->graph->origin[0] - linePtr->graph->radius,
	    linePtr->graph->origin[1] - linePtr->graph->radius,
	    &graph_clip_xorigin, &graph_clip_yorigin);

    if ((!linePtr->numPoints)||(linePtr->outline.gc==None)) {
	return;
    }

    XSetClipMask(display, linePtr->outline.gc, linePtr->graph->clipMask);
    XSetClipOrigin(display, linePtr->outline.gc, graph_clip_xorigin,
	    graph_clip_yorigin);

    /*
     * Build up an array of points in screen coordinates.  Use a
     * static array unless the line has an enormous number of points;
     * in this case, dynamically allocate an array.  For smoothed lines,
     * generate the curve points on each redisplay.
     */

    Tk_CanvasDrawableCoords(canvas, 
	    linePtr->graph->origin[0], linePtr->graph->origin[1],
	    origin, origin+1);

    if ((linePtr->smooth) && (linePtr->numPoints > 2)) {
	numPoints = 1 + linePtr->numPoints*linePtr->splineSteps;
    } else {
	numPoints = linePtr->numPoints;
    }

    switch (linePtr->style) {
	case POLAR_SEGMENTED:
	    if (numPoints <= MAX_STATIC_POINTS) {
		segPtr = staticSegs;
	    } else {
		segPtr = (XSegment *) ckalloc((unsigned) (numPoints * sizeof(XSegment)));
	    }

	    for(i = 0, p=0; i+3 < (linePtr->numPoints*2); i+=4, p++) {
		segPtr[p].x1 = linePtr->canvascoordPtr[i] - 
		    linePtr->graph->origin[0] + origin[0];
		segPtr[p].y1 = linePtr->canvascoordPtr[i+1] - 
		    linePtr->graph->origin[1] + origin[1];
		segPtr[p].x2 = linePtr->canvascoordPtr[i+2] - 
		    linePtr->graph->origin[0] + origin[0];
		segPtr[p].y2 = linePtr->canvascoordPtr[i+3] - 
		    linePtr->graph->origin[1] + origin[1];
	    }
	    numPoints = p;

	    /*
	    for(i=0; i < numPoints; i++) {
		printf("Displaying line:  (%d, %d) --> (%d, %d)\n", segPtr[i].x1, segPtr[i].y1, segPtr[i].x2, segPtr[i].y2);
	    }
	    */

	    XDrawSegments(display, drawable, linePtr->outline.gc, segPtr,
		    numPoints);

	    if (segPtr != staticSegs) {
		ckfree((char *) segPtr);
	    }

	    break;
	case POLAR_COMPLETE:
	    numPoints++;
	case POLAR_CONTINUOUS:
	    if (numPoints <= MAX_STATIC_POINTS) {
		pointPtr = staticPoints;
	    } else {
		pointPtr = (XPoint *) ckalloc((unsigned) (numPoints * sizeof(XPoint)));
	    }

	    for(i = 0, p=0; i < linePtr->numPoints*2; i+=2, p++) {
		pointPtr[p].x = linePtr->canvascoordPtr[i] - 
		    linePtr->graph->origin[0] + origin[0];
		pointPtr[p].y = linePtr->canvascoordPtr[i+1] - 
		    linePtr->graph->origin[1] + origin[1];
	    }
	    if (linePtr->style == POLAR_COMPLETE) {
		pointPtr[p].x = linePtr->canvascoordPtr[0] - 
		    linePtr->graph->origin[0] + origin[0];
		pointPtr[p].y = linePtr->canvascoordPtr[1] - 
		    linePtr->graph->origin[1] + origin[1];
	    }

	    /*
	    for(i=0; i < numPoints; i++) {
		printf("Displaying line:  %d, %d\n", pointPtr[i].x, pointPtr[i].y);
	    }
	    */

	    XDrawLines(display, drawable, linePtr->outline.gc, pointPtr,
		    numPoints, CoordModeOrigin);

	    if (pointPtr != staticPoints) {
		ckfree((char *) pointPtr);
	    }
	    break;
    }
    XSetClipMask(display, linePtr->outline.gc, None);
    XSetClipOrigin(display, linePtr->outline.gc, 0, 0);

    /*
    printf("Validating memory... DisplayLine(end)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */
}

/*
 *--------------------------------------------------------------
 *
 * LineInsert --
 *
 *	Insert coords into a line item at a given index.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The coords in the given item is modified.
 *
 *--------------------------------------------------------------
 */

static void
LineInsert(canvas, itemPtr, beforeThis, string)
    Tk_Canvas canvas;		/* Canvas containing text item. */
    Tk_Item *itemPtr;		/* Line item to be modified. */
    int beforeThis;		/* Index before which new coordinates
				 * are to be inserted. */
    char *string;		/* New coordinates to be inserted. */
{
    PDatasetItem *linePtr = (PDatasetItem *) itemPtr;
    int length, argc, i;
    char **argv = (char **) NULL;
    double *newdata, *coordPtr;
    double *newgraph, *newcanvas;
    Tk_State state = itemPtr->state;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    if(!string || !*string) {
	return;
    }
    if ((Tcl_SplitList(((TkCanvas *)canvas)->interp, string, &argc, &argv) != TCL_OK)
	    || argv == NULL || !argc || argc&1) {
	Tcl_ResetResult(((TkCanvas *)canvas)->interp);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return;
    }
    length = 2*linePtr->numPoints;
    if (beforeThis < 0) {
	beforeThis = 0;
    }
    if (beforeThis > length) {
	beforeThis = length;
    }
    newdata  = (double *)ckalloc((unsigned)(sizeof(double)*(length + argc)));
    newgraph = (double *)ckalloc((unsigned)(sizeof(double)*(length + argc)));
    newcanvas= (double *)ckalloc((unsigned)(sizeof(double)*(length + argc)));
    for(i=0; i<beforeThis; i++) {
	newdata[i] = linePtr->datacoordPtr[i];
    }
    for(i=0; i<argc; i++) {
	if (Tcl_GetDouble(((TkCanvas *)canvas)->interp,argv[i],
		newdata+(i+beforeThis))!=TCL_OK) {
	    Tcl_ResetResult(((TkCanvas *)canvas)->interp);
	    ckfree((char *) newdata);
	    ckfree((char *) newgraph);
	    ckfree((char *) newcanvas);
	    ckfree((char *) argv);
	    return;
	}
    }

    for(i=beforeThis; i<length; i++) {
	newdata[i+argc] = linePtr->datacoordPtr[i];
    }
    if(linePtr->datacoordPtr)   ckfree((char *)linePtr->datacoordPtr);
    if(linePtr->graphcoordPtr)  ckfree((char *)linePtr->graphcoordPtr);
    if(linePtr->canvascoordPtr) ckfree((char *)linePtr->canvascoordPtr);
    ckfree((char *) argv);
    linePtr->datacoordPtr   = newdata;
    linePtr->graphcoordPtr  = newgraph;
    linePtr->canvascoordPtr = newcanvas;
    linePtr->numPoints = (length + argc)/2;

    if ((length>3) && (state != TK_STATE_HIDDEN)) {
	/*
	 * This is some optimizing code that will result that only the part
	 * of the polygon that changed (and the objects that are overlapping
	 * with that part) need to be redrawn. A special flag is set that
	 * instructs the general canvas code not to redraw the whole
	 * object. If this flag is not set, the canvas will do the redrawing,
	 * otherwise I have to do it here.
	 */
	itemPtr->redraw_flags |= TK_ITEM_DONT_REDRAW;

	if (beforeThis>0) {beforeThis -= 2; argc+=2; }
	if ((beforeThis+argc)<length) argc+=2;
	if (linePtr->smooth) {
	    if(beforeThis>0) {
		beforeThis-=2; argc+=2;
	    }
	    if((beforeThis+argc+2)<length) {
		argc+=2;
	    }
	}
	itemPtr->x1 = itemPtr->x2 = linePtr->datacoordPtr[beforeThis];
	itemPtr->y1 = itemPtr->y2 = linePtr->datacoordPtr[beforeThis+1];
	coordPtr = linePtr->datacoordPtr+beforeThis+2;
	for(i=2; i<argc; i+=2) {
	    TkIncludePoint(itemPtr, coordPtr);
		coordPtr+=2;
	}
    }

    if(itemPtr->redraw_flags & TK_ITEM_DONT_REDRAW) {
	int width;
	width = linePtr->outline.width;
	if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
		if (linePtr->outline.activeWidth>width) {
		    width = linePtr->outline.activeWidth;
		}
	} else if (state==TK_STATE_DISABLED) {
		if (linePtr->outline.disabledWidth>0) {
		    width = linePtr->outline.disabledWidth;
		}
	}
	itemPtr->x1 -= width; itemPtr->y1 -= width;
	itemPtr->x2 += width; itemPtr->y2 += width;
	Tk_CanvasEventuallyRedraw(canvas, itemPtr->x1, itemPtr->y1,
		itemPtr->x2, itemPtr->y2);
    }

    LineUpdateGraphCoords(linePtr);
    LineUpdateCanvasCoords(linePtr);

    ComputeLineBbox(canvas, linePtr);
}

/*
 *--------------------------------------------------------------
 *
 * LineDeleteCoords --
 *
 *	Delete one or more coordinates from a line item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Characters between "first" and "last", inclusive, get
 *	deleted from itemPtr.
 *
 *--------------------------------------------------------------
 */

static void
LineDeleteCoords(canvas, itemPtr, first, last)
    Tk_Canvas canvas;		/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Item in which to delete characters. */
    int first;			/* Index of first character to delete. */
    int last;			/* Index of last character to delete. */
{
    PDatasetItem *linePtr = (PDatasetItem *) itemPtr;
    int count, i, first1, last1;
    int length = 2*linePtr->numPoints;
    double *coordPtr;
    Tk_State state = itemPtr->state;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    first &= -2;
    last &= -2;

    if (first < 0) {
	first = 0;
    }
    if (last >= length) {
	last = length-2;
    }
    if (first > last) {
	return;
    }
    first1 = first; last1 = last;
    if(first1>0) first1 -= 2;
    if(last1<length-2) last1 += 2;
    if (linePtr->smooth) {
	if(first1>0) first1 -= 2;
	if(last1<length-2) last1 += 2;
    }

    if((first1<2) && (last1 >= length-2)) {
	/*
	 * This is some optimizing code that will result that only the part
	 * of the line that changed (and the objects that are overlapping
	 * with that part) need to be redrawn. A special flag is set that
	 * instructs the general canvas code not to redraw the whole
	 * object. If this flag is set, the redrawing has to be done here,
	 * otherwise the general Canvas code will take care of it.
	 */

	itemPtr->redraw_flags |= TK_ITEM_DONT_REDRAW;
	itemPtr->x1 = itemPtr->x2 = linePtr->datacoordPtr[first1];
	itemPtr->y1 = itemPtr->y2 = linePtr->datacoordPtr[first1+1];
	coordPtr = linePtr->datacoordPtr+first1+2;
	for(i=first1+2; i<=last1; i+=2) {
	    TkIncludePoint(itemPtr, coordPtr);
		coordPtr+=2;
	}
    }

    count = last + 2 - first;
    for(i=last+2; i<length; i++) {
	linePtr->datacoordPtr[i-count] = linePtr->datacoordPtr[i];
    }
    linePtr->numPoints -= count/2;
    if(itemPtr->redraw_flags & TK_ITEM_DONT_REDRAW) {
	int width;
	width = linePtr->outline.width;
	if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
		if (linePtr->outline.activeWidth>width) {
		    width = linePtr->outline.activeWidth;
		}
	} else if (state==TK_STATE_DISABLED) {
		if (linePtr->outline.disabledWidth>0) {
		    width = linePtr->outline.disabledWidth;
		}
	}
	itemPtr->x1 -= width; itemPtr->y1 -= width;
	itemPtr->x2 += width; itemPtr->y2 += width;
	Tk_CanvasEventuallyRedraw(canvas, itemPtr->x1, itemPtr->y1,
		itemPtr->x2, itemPtr->y2);
    }
    ComputeLineBbox(canvas, linePtr);
}

/*
 *--------------------------------------------------------------
 *
 * LineToPoint --
 *
 *	Computes the distance from a given point to a given
 *	line, in canvas units.
 *
 * Results:
 *	The return value is 0 if the point whose x and y coordinates
 *	are pointPtr[0] and pointPtr[1] is inside the line.  If the
 *	point isn't inside the line then the return value is the
 *	distance from the point to the line.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static double
LineToPoint(canvas, itemPtr, pointPtr)
    Tk_Canvas canvas;		/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against point. */
    double *pointPtr;		/* Pointer to x and y coordinates. */
{
    int width;
    Tk_State state = itemPtr->state;
    PDatasetItem *linePtr = (PDatasetItem *) itemPtr;
    double *coordPtr;
    double *linePoints;
    double cPtr[4];
    double poly[10];
    double bestDist, dist;
    int numPoints, count;
    int changedMiterToBevel;	/* Non-zero means that a mitered corner
				 * had to be treated as beveled after all
				 * because the angle was < 11 degrees. */

    bestDist = 1.0e36;

    /*
     * Handle smoothed lines by generating an expanded set of points
     * against which to do the check.
     */

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    width = linePtr->outline.width;
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (linePtr->outline.activeWidth>width) {
	    width = linePtr->outline.activeWidth;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (linePtr->outline.disabledWidth>0) {
	    width = linePtr->outline.disabledWidth;
	}
    }

    numPoints = linePtr->numPoints;
    linePoints = linePtr->canvascoordPtr;

    if (!numPoints || itemPtr->state==TK_STATE_HIDDEN) {
	return bestDist;
    }

    /*
     * The overall idea is to iterate through all of the edges of
     * the line, computing a polygon for each edge and testing the
     * point against that polygon.  In addition, there are additional
     * tests to deal with rounded joints and caps.
     */

    changedMiterToBevel = 0;
    for (count = numPoints, coordPtr = linePoints; count > 1;) {
	cPtr[0] = (double) coordPtr[0];
	cPtr[1] = (double) coordPtr[1];
	cPtr[2] = (double) coordPtr[2];
	cPtr[3] = (double) coordPtr[3];

	/*
	* If rounding is done around the first point then compute
	* the distance between the point and the point.
	*/

	if (((linePtr->capStyle == CapRound) && (count == numPoints))
		|| ((linePtr->joinStyle == JoinRound)
			&& (count != numPoints))) {
	    dist = hypot(cPtr[0] - pointPtr[0], cPtr[1] - pointPtr[1])
		    - width/2.0;
	    if (dist <= 0.0) {
		bestDist = 0.0;
		goto done;
	    } else if (dist < bestDist) {
		bestDist = dist;
	    }

	    dist = hypot(cPtr[2] - pointPtr[0], cPtr[2] - pointPtr[1])
		    - width/2.0;
	    if (dist <= 0.0) {
		bestDist = 0.0;
		goto done;
	    } else if (dist < bestDist) {
		bestDist = dist;
	    }
	}

	/*
	* Compute the polygonal shape corresponding to this edge,
	* consisting of two points for the first point of the edge
	* and two points for the last point of the edge.
	*/

	if ((linePtr->joinStyle == JoinMiter) && !changedMiterToBevel) {
	    poly[0] = poly[6];
	    poly[1] = poly[7];
	    poly[2] = poly[4];
	    poly[3] = poly[5];
	} else {
	    cPtr[0] = (double) coordPtr[0];
	    cPtr[1] = (double) coordPtr[1];
	    cPtr[2] = (double) coordPtr[2];
	    cPtr[3] = (double) coordPtr[3];
	    TkGetButtPoints(cPtr+2, cPtr, (double) width, 0,
		    poly, poly+2);

	    /*
	    * If this line uses beveled joints, then check the distance
	    * to a polygon comprising the last two points of the previous
	    * polygon and the first two from this polygon;  this checks
	    * the wedges that fill the mitered joint.
	    */

	    if ((linePtr->joinStyle == JoinBevel) || changedMiterToBevel) {
		poly[8] = poly[0];
		poly[9] = poly[1];
		dist = TkPolygonToPoint(poly, 5, pointPtr);
		if (dist <= 0.0) {
		    bestDist = 0.0;
		    goto done;
		} else if (dist < bestDist) {
		    bestDist = dist;
		}
		changedMiterToBevel = 0;
	    }
	}

	TkGetButtPoints(cPtr, cPtr+2, (double) width, 0,
		poly+4, poly+6);

	poly[8] = poly[0];
	poly[9] = poly[1];
	dist = TkPolygonToPoint(poly, 5, pointPtr);
	if (dist <= 0.0) {
	    bestDist = 0.0;
	    goto done;
	} else if (dist < bestDist) {
	    bestDist = dist;
	}
	if (linePtr->style == POLAR_SEGMENTED)
	    count-=2, coordPtr+=2;
	else
	    count--, coordPtr++;
    }

    done:
    return bestDist;
}

/*
 *--------------------------------------------------------------
 *
 * LineToArea --
 *
 *	This procedure is called to determine whether an item
 *	lies entirely inside, entirely outside, or overlapping
 *	a given rectangular area.
 *
 * Results:
 *	-1 is returned if the item is entirely outside the
 *	area, 0 if it overlaps, and 1 if it is entirely
 *	inside the given area.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
LineToArea(canvas, itemPtr, rectPtr)
    Tk_Canvas canvas;		/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against line. */
    double *rectPtr;
{
    PDatasetItem *linePtr = (PDatasetItem *) itemPtr;
    XSegment	*linePoints;
    double	radius;
    double	cPtr[4];
    double	bestDist = 1.0e36;
    int		numPoints, result;
    int		width, 
    		i;
    Tk_State	state = itemPtr->state;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }
    width = linePtr->outline.width;
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (linePtr->outline.activeWidth>width) {
	    width = linePtr->outline.activeWidth;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (linePtr->outline.disabledWidth>0) {
	    width = linePtr->outline.disabledWidth;
	}
    }

    radius = (width+1)/2.0;

    if ((state==TK_STATE_HIDDEN) || !linePtr->numPoints) {
	return -1;
    }

    /*
     * Check the segments of the line.
     */

    for(i = 0; i+3 < (numPoints*2); i+=4) {
	cPtr[0] = linePtr->canvascoordPtr[i];
	cPtr[1] = linePtr->canvascoordPtr[i+1];
	cPtr[2] = linePtr->canvascoordPtr[i+2];
	cPtr[3] = linePtr->canvascoordPtr[i+3];
	result = TkThickPolyLineToArea(cPtr, 2,
		(double) width, linePtr->capStyle, linePtr->joinStyle,
		rectPtr);
	if (result == 0) {
	    goto done;
	}
	if (result < bestDist)
	    bestDist = result;
    }

    done:
    return result;
}

/*
 *--------------------------------------------------------------
 *
 * ScaleDataset --
 *
 *	This procedure is invoked to rescale a line item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The line referred to by itemPtr is rescaled so that the
 *	following transformation is applied to all point
 *	coordinates:
 *		x' = originX + scaleX*(x-originX)
 *		y' = originY + scaleY*(y-originY)
 *
 *--------------------------------------------------------------
 *
 * We shouldn't really be calling this...
 */

void
ScaleDataset(canvas, itemPtr, originX, originY, scaleX, scaleY)
    Tk_Canvas canvas;			/* Canvas containing line. */
    Tk_Item *itemPtr;			/* Line to be scaled. */
    double originX, originY;		/* Origin about which to scale rect. */
    double scaleX;			/* Amount to scale in X direction. */
    double scaleY;			/* Amount to scale in Y direction. */
{
    PDatasetItem *linePtr = (PDatasetItem *) itemPtr;
    double *coordPtr;
    int i;

    return;

    printf("called!  Generating core file\n");
    abort();

    ComputeLineBbox(canvas, linePtr);
}

/*
 *--------------------------------------------------------------
 *
 * GetLineIndex --
 *
 *	Parse an index into a line item and return either its value
 *	or an error.
 *
 * Results:
 *	A standard Tcl result.  If all went well, then *indexPtr is
 *	filled in with the index (into itemPtr) corresponding to
 *	string.  Otherwise an error message is left in
 *	interp->result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
GetLineIndex(interp, canvas, itemPtr, string, indexPtr)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Canvas canvas;		/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item for which the index is being
				 * specified. */
    char *string;		/* Specification of a particular coord
				 * in itemPtr's line. */
    int *indexPtr;		/* Where to store converted index. */
{
    PDatasetItem *linePtr = (PDatasetItem *) itemPtr;
    size_t length;

    length = strlen(string);

    if (string[0] == 'e') {
	if (strncmp(string, "end", length) == 0) {
	    *indexPtr = (2*linePtr->numPoints);
	} else {
	    badIndex:

	    /*
	     * Some of the paths here leave messages in interp->result,
	     * so we have to clear it out before storing our own message.
	     */

	    Tcl_SetResult(interp, (char *) NULL, TCL_STATIC);
	    Tcl_AppendResult(interp, "bad index \"", string, "\"",
		    (char *) NULL);
	    return TCL_ERROR;
	}
    } else if (string[0] == '@') {
	int i;
	double x ,y, bestDist, dist, *coordPtr;
	double cPtr[4];
	double pPtr[2];
	char *end, *p;
	double *segPtr;

	pPtr[0] = (double) x;
	pPtr[1] = (double) y;
	p = string+1;
	x = strtod(p, &end);
	if ((end == p) || (*end != ',')) {
	    goto badIndex;
	}
	p = end+1;
	y = strtod(p, &end);
	if ((end == p) || (*end != (char)0)) {
	    goto badIndex;
	}
	bestDist = 1.0e36;
	segPtr = linePtr->canvascoordPtr;
	*indexPtr = 0;
	for(i=0; i+3<(linePtr->numPoints*2); i+=4) {
	    cPtr[0] = (double) segPtr[i];
	    cPtr[1] = (double) segPtr[i+1];
	    cPtr[2] = (double) segPtr[i+2];
	    cPtr[3] = (double) segPtr[i+3];

	    dist = hypot(cPtr[0] - x, cPtr[1] - y);
	    if (dist<bestDist) {
		bestDist = dist;
		*indexPtr = i;
	    }
	    dist = hypot(cPtr[2] - x, cPtr[3] - y);
	    if (dist<bestDist) {
		bestDist = dist;
		*indexPtr = i;
	    }

	    dist = TkLineToPoint(cPtr, cPtr+2, pPtr);
	    if (dist<bestDist) {
		bestDist = dist;
		*indexPtr = i;
	    }
	}
    } else {
	if (Tcl_GetInt(interp, string, indexPtr) != TCL_OK) {
	    goto badIndex;
	}
	*indexPtr &= -2; /* if index is odd, make it even */
	if (*indexPtr < 0){
	    *indexPtr = 0;
	} else if (*indexPtr > (2*linePtr->numPoints)) {
	    *indexPtr = (2*linePtr->numPoints);
	}
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * TranslateDataset --
 *
 *	This procedure is called to move a line by a given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The position of the line is offset by (xDelta, yDelta), and
 *	the bounding box is updated in the generic part of the item
 *	structure.
 *
 *--------------------------------------------------------------
 *
 * We shouldn't really be calling this...
 */

void
TranslateDataset(canvas, itemPtr, deltaX, deltaY)
    Tk_Canvas canvas;			/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item that is being moved. */
    double deltaX, deltaY;		/* Amount by which item is to be
					 * moved. */
{
    PDatasetItem *linePtr = (PDatasetItem *) itemPtr;
    double *coordPtr;
    int i;

    return;

    fprintf(stderr, "Error:  TranslateDataset called!  Generating core file\n");
    abort();
}

/*
 *--------------------------------------------------------------
 *
 * LineToPostscript --
 *
 *	This procedure is called to generate Postscript for
 *	line items.
 *
 * Results:
 *	The return value is a standard Tcl result.  If an error
 *	occurs in generating Postscript then an error message is
 *	left in interp->result, replacing whatever used
 *	to be there.  If no error occurs, then Postscript for the
 *	item is appended to the result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
LineToPostscript(interp, canvas, itemPtr, prepass)
    Tcl_Interp *interp;			/* Leave Postscript or error message
					 * here. */
    Tk_Canvas canvas;			/* Information about overall canvas. */
    Tk_Item *itemPtr;			/* Item for which Postscript is
					 * wanted. */
    int prepass;			/* 1 means this is a prepass to
					 * collect font information;  0 means
					 * final Postscript is being created. */
{
    PDatasetItem *linePtr = (PDatasetItem *) itemPtr;
    char	buffer[1024];
    char	*style;
    int		width,
    		i;
    XColor	*color;
    Pixmap	stipple;
    Tk_State	state = itemPtr->state;
    double y1 = Tk_CanvasPsY(canvas, linePtr->graph->origin[1]);

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    width = linePtr->outline.width;
    color = linePtr->outline.color;
    stipple = linePtr->outline.stipple;
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (linePtr->outline.activeWidth>width) {
	    width = linePtr->outline.activeWidth;
	}
	if (linePtr->outline.activeColor!=NULL) {
	    color = linePtr->outline.activeColor;
	}
	if (linePtr->outline.activeStipple!=None) {
	    stipple = linePtr->outline.activeStipple;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (linePtr->outline.disabledWidth>0) {
	    width = linePtr->outline.disabledWidth;
	}
	if (linePtr->outline.disabledColor!=NULL) {
	    color = linePtr->outline.disabledColor;
	}
	if (linePtr->outline.disabledStipple!=None) {
	    stipple = linePtr->outline.disabledStipple;
	}
    }

    if (color == NULL || linePtr->numPoints<1 || linePtr->canvascoordPtr==NULL) {
	return TCL_OK;
    }

    /*
     * Clip the dataset at the graph boundary
     */


    sprintf(buffer, "newpath %.15g %.15g %.15g 0 360 arc clip\n",
	    linePtr->graph->origin[0],
	    Tk_CanvasPsY(canvas, linePtr->graph->origin[1]),
	    linePtr->graph->radius);
    Tcl_AppendResult(interp, buffer, "newpath\n", (char *)NULL);


    /*
     * Generate a path for the line's center-line (do this differently
     * for straight lines and smoothed lines).
     */

    if (linePtr->style == POLAR_SEGMENTED) {
	for(i = 0; i+3 < (linePtr->numPoints*2); i+=4) {
	    double cPtr[4];

	    cPtr[0] = linePtr->canvascoordPtr[i];
	    cPtr[1] = linePtr->canvascoordPtr[i+1];
	    cPtr[2] = linePtr->canvascoordPtr[i+2];
	    cPtr[3] = linePtr->canvascoordPtr[i+3];

	    if (stipple == None) {
		/*
		printf("Generating postscript for (%g, %g) -> (%g, %g)\n",
			cPtr[0],
			cPtr[1],
			cPtr[2],
			cPtr[3]);
		*/
		Tk_CanvasPsPath(interp, canvas, cPtr, 2);
		/*
		TkMakeBezierPostscript(interp, canvas, cPtr, 2);
		*/
	    } else {
		/*
		* Special hack: Postscript printers don't appear to be able
		* to turn a path drawn with "curveto"s into a clipping path
		* without exceeding resource limits, so TkMakeBezierPostscript
		* won't work for stippled curves.  Instead, generate all of
		* the intermediate points here and output them into the
		* Postscript file with "lineto"s instead.
		*/

		double staticPoints[2*MAX_STATIC_POINTS];
		double *pointPtr = staticPoints;
		int numPoints = 2;

		numPoints = TkMakeBezierCurve(canvas, cPtr,
			2, linePtr->splineSteps, (XPoint *) NULL,
			pointPtr);
		Tk_CanvasPsPath(interp, canvas, pointPtr, numPoints);
	    }
	}
    } else {
	Tk_CanvasPsPath(interp, canvas, linePtr->canvascoordPtr, linePtr->numPoints);
    }

    /*
     * Set other line-drawing parameters and stroke out the line.
     */

    style = "0 setlinecap\n";
    if (linePtr->capStyle == CapRound) {
	style = "1 setlinecap\n";
    } else if (linePtr->capStyle == CapProjecting) {
	style = "2 setlinecap\n";
    }
    Tcl_AppendResult(interp, style, (char *) NULL);
    style = "0 setlinejoin\n";
    if (linePtr->joinStyle == JoinRound) {
	style = "1 setlinejoin\n";
    } else if (linePtr->joinStyle == JoinBevel) {
	style = "2 setlinejoin\n";
    }
    Tcl_AppendResult(interp, style, (char *) NULL);

    if (Tk_CanvasPsOutline(canvas, itemPtr,
	    &(linePtr->outline)) != TCL_OK) {
	return TCL_ERROR;
    }

    return TCL_OK;
}

static void
LineUpdateGraphCoords(PDatasetItem *linePtr)
{
    register double xsize, ysize;
    register double xscale, yscale;
    register double r0, r1;
    double	xtmp, ytmp;
    double	*datacoordPtr;
    double	*graphcoordPtr;
    int		i;
    double	rad;
    double	radius;

    /*
    printf("Validating memory... LineUpdateGraphCoords(start)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    if (linePtr->graph == NULL)
	return;

    /*
     * Get the radius at which we are drawing this data
     */

    /*
    printf("num_points = %d\n", linePtr->numPoints);
    */
    for (i = 0,
	    datacoordPtr=linePtr->datacoordPtr,
	    graphcoordPtr=linePtr->graphcoordPtr;
	 i < linePtr->numPoints;
	 i += 1,
	    datacoordPtr+=2,
	    graphcoordPtr+=2) {
	
	r0 = (datacoordPtr[0] + linePtr->roffset) * linePtr->rscale;
	/*
	 * printf("LineUpdateGraphCoords():  r0 = %g (%g)\n", r0, coordPtr[0]);
	 */

	graphcoordPtr[0] = r0;
	if (linePtr->graph->sense >= 0) {
	    graphcoordPtr[1] = (datacoordPtr[1] - linePtr->graph->poledir) * 3.1415926535 / 180.0;
	} else {
	    graphcoordPtr[1] = -(datacoordPtr[1] + linePtr->graph->poledir) * 3.1415926535 / 180.0;
	}
	/*
	 * printf("LineUpdateGraphCoords():  %g, %g\n", sPtr[0], sPtr[1]);
	 */
    }

    /*
    printf("Validating memory... LineUpdateGraphCoords(end)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */
}

/*
 * Update the canvascoordPtr array in the line item.  This function
 * assumes that the proper scaled graph coordinates are stored
 * in graphcoordPtr.  If the graph coordinates have not been scaled
 * then this function will not work correctly.
 */

static void
LineUpdateCanvasCoords(PDatasetItem *linePtr)
{
    register double xoffset, yoffset;
    register double r0, r1;
    double	*sPtr;
    double	*coordPtr;
    double	xtmp, ytmp;
    double	rad;
    double	scale;
    int		i;

    /*
    printf("Validating memory... LineUpdateCanvasCoords(start)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    /*
     * This shouldn't be necessary.  I'll take it out later when I know
     * it works.
     */

    LineUpdateGraphCoords(linePtr);

    if (linePtr->graph == NULL)
	return;

    scale = linePtr->graph->radius / linePtr->graph->rmax;

    for (i = 0, coordPtr=linePtr->datacoordPtr, sPtr=linePtr->canvascoordPtr;
	 i < linePtr->numPoints;  i += 1, coordPtr += 2, sPtr+=2) {
	
	r0 = (coordPtr[0] +linePtr->roffset) * scale * linePtr->rscale;
	if (linePtr->graph->sense >= 0) {
	    rad = (coordPtr[1] - linePtr->graph->poledir) * 3.1415926535 / 180.0;
	} else {
	    rad = -(coordPtr[1] + linePtr->graph->poledir) * 3.1415926535 / 180.0;
	}

	xtmp =cos(rad);
	ytmp =sin(rad);
	sPtr[0] = linePtr->graph->origin[0] + r0 * xtmp;
	sPtr[1] = linePtr->graph->origin[1] + r0 * ytmp;
	/*
	 * printf("LineUpdateCanvasCoords():  %g, %g\n", sPtr[0], sPtr[1]);
	 */
    }

    /*
    printf("Validating memory... LineUpdateCanvasCoords(end)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */
}

static int
Tk_GraphParseProc(ClientData clientData, Tcl_Interp *interp,
	Tk_Window tkwin, char *value, char *widgRec, int offset)
{
    PGraphItem **graph = ((PGraphItem **)(widgRec + offset));
    TkCanvas *canvas = ((TkCanvas *) ((PDatasetItem *)widgRec)->canvas);
    int id;
    Tk_Item *itemPtr;
    TagSearch search;

    if (value == (char *)NULL) {
	fprintf(stderr, "Setting graph to nothing\n");
	RemoveDatasetFromGraph((PDatasetItem *)widgRec, *graph);
	*graph = NULL;
    } else {
	/*
	 * Find the canvas item associated with this id
	 */
	itemPtr = StartTagSearch(canvas, value, &search);
 
	/*
	 * Make sure this canvas item is a "pgraph" item
	 */

	if (itemPtr==NULL || (strcmp(itemPtr->typePtr->name, "pgraph")!=0)){
	    Tcl_AppendResult(interp, "Bad graph item:  ", value, (char *)NULL);
	    return TCL_ERROR;
	}

	/*
	 * We have a valid graph item.  Add this dataset to it's child list
	 */

	RemoveDatasetFromGraph((PDatasetItem *)widgRec, *graph);
	*graph = (PGraphItem *)itemPtr;
	AddDatasetToGraph((PDatasetItem *)widgRec, *graph);
    }

    return TCL_OK;
}

static char *
Tk_GraphPrintProc(ClientData clientData,
	Tk_Window tkwin, char *widgRec, int offset,
	Tcl_FreeProc **freeProcPtr)
{
    PGraphItem *graph = ((PDatasetItem *)widgRec)->graph;
    char *buf = (char *)ckalloc(sizeof(char)*TCL_DOUBLE_SPACE);

    if (graph == NULL)
	*buf = (char) NULL;
    else
	sprintf(buf, "%d", ((Tk_Item *)graph)->id);

    return buf;
}

static int
Tk_StyleParseProc (ClientData clientData, Tcl_Interp *interp,
	Tk_Window tkwin, char *value, char *widgRec, int offset)
{
    PDatasetItem *linePtr = (PDatasetItem *)widgRec;

    /*
    printf("Validating memory... Tk_StyleParseProc(start)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    if (strncasecmp(value, "segmented", 7) == 0) {
	linePtr->style = POLAR_SEGMENTED;
	return TCL_OK;
    }
    if (strcasecmp(value, "continuous") == 0) {
	linePtr->style = POLAR_CONTINUOUS;
	return TCL_OK;
    }
    if (strcasecmp(value, "complete") == 0) {
	linePtr->style = POLAR_CONTINUOUS;
	return TCL_OK;
    }

    Tcl_AppendResult(interp, "Invalid Polar Dataset Style:  ", value, (char *)NULL);

    /*
    printf("Validating memory... Tk_StyleParseProc(end)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */
	
    return TCL_ERROR;
}

static char *
Tk_StylePrintProc (ClientData clientData,
	Tk_Window tkwin, char *widgRec, int offset,
	Tcl_FreeProc **freeProcPtr)
{
    PDatasetItem *linePtr = (PDatasetItem *)widgRec;
    char *buf = (char *)ckalloc(sizeof(char) * 20);

    /*
    printf("Validating memory... Tk_StylePrintProc(start)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    *freeProcPtr = TCL_DYNAMIC;

    switch (linePtr->style) {
	case POLAR_SEGMENTED:
		strcpy(buf, "segmented");
		return buf;
	case POLAR_CONTINUOUS:
		strcpy(buf, "continuous");
		return buf;
	case POLAR_COMPLETE:
		strcpy(buf, "complete");
		return buf;
    }

    strcpy(buf, "<null>");

    /*
    printf("Validating memory... Tk_StylePrintProc(end)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    return buf;
}

/*
 *--------------------------------------------------------------
 *
 * StartTagSearch --
 *
 *	This procedure is called to initiate an enumeration of
 *	all items in a given canvas that contain a given tag.
 *
 * Results:
 *	The return value is a pointer to the first item in
 *	canvasPtr that matches tag, or NULL if there is no
 *	such item.  The information at *searchPtr is initialized
 *	such that successive calls to NextItem will return
 *	successive items that match tag.
 *
 * Side effects:
 *	SearchPtr is linked into a list of searches in progress
 *	on canvasPtr, so that elements can safely be deleted
 *	while the search is in progress.  EndTagSearch must be
 *	called at the end of the search to unlink searchPtr from
 *	this list.
 *
 *--------------------------------------------------------------
 */

static Tk_Item *
StartTagSearch(canvasPtr, tag, searchPtr)
    TkCanvas *canvasPtr;		/* Canvas whose items are to be
					 * searched. */
    char *tag;				/* String giving tag value. */
    TagSearch *searchPtr;		/* Record describing tag search;
					 * will be initialized here. */
{
    int id;
    Tk_Item *itemPtr, *prevPtr;
    Tk_Uid *tagPtr;
    Tk_Uid uid;
    int count;

    /*
     * Initialize the search.
     */

    searchPtr->canvasPtr = canvasPtr;
    searchPtr->searchOver = 0;

    /*
     * Find the first matching item in one of several ways. If the tag
     * is a number then it selects the single item with the matching
     * identifier.  In this case see if the item being requested is the
     * hot item, in which case the search can be skipped.
     */

    if (isdigit(UCHAR(*tag))) {
	char *end;
	Tcl_HashEntry *entryPtr;

	id = strtoul(tag, &end, 0);
	if (*end == 0) {
	    itemPtr = canvasPtr->hotPtr;
	    prevPtr = canvasPtr->hotPrevPtr;
	    if ((itemPtr == NULL) || (itemPtr->id != id) || (prevPtr == NULL)
		    || (prevPtr->nextPtr != itemPtr)) {
	      entryPtr = Tcl_FindHashEntry(&canvasPtr->idTable, (char *)id);
	      if (entryPtr) {
		itemPtr = (Tk_Item *)Tcl_GetHashValue(entryPtr);
		prevPtr = itemPtr->prevPtr;
	      } else {
	        prevPtr = itemPtr = NULL;
	      }
	    }
	    searchPtr->prevPtr = prevPtr;
	    searchPtr->searchOver = 1;
	    canvasPtr->hotPtr = itemPtr;
	    canvasPtr->hotPrevPtr = prevPtr;
	    return itemPtr;
	}
    }

    searchPtr->tag = uid = Tk_GetUid(tag);
    if (uid == allUid) {

	/*
	 * All items match.
	 */

	searchPtr->tag = NULL;
	searchPtr->prevPtr = NULL;
	searchPtr->currentPtr = canvasPtr->firstItemPtr;
	return canvasPtr->firstItemPtr;
    }

    /*
     * None of the above.  Search for an item with a matching tag.
     */

    for (prevPtr = NULL, itemPtr = canvasPtr->firstItemPtr; itemPtr != NULL;
	    prevPtr = itemPtr, itemPtr = itemPtr->nextPtr) {
	for (tagPtr = itemPtr->tagPtr, count = itemPtr->numTags;
		count > 0; tagPtr++, count--) {
	    if (*tagPtr == uid) {
		searchPtr->prevPtr = prevPtr;
		searchPtr->currentPtr = itemPtr;
		return itemPtr;
	    }
	}
    }
    searchPtr->prevPtr = prevPtr;
    searchPtr->searchOver = 1;
    return NULL;
}

static int
Tk_DontSetOption(ClientData clientData, Tcl_Interp *interp, Tk_Window tkwin,
	char *value, char *widgRec, int offset)
{
    return TCL_OK;
}

static char *
Tk_CustomPrintDouble(ClientData clientData, Tk_Window tkwin, char *widgRec,
	int offset, Tcl_FreeProc **freeProcPtr)
{
    register double *doublePtr = (double *)(widgRec + offset);
    char *buf=(char *)NULL;
    *freeProcPtr = TCL_DYNAMIC;

    /*
    printf("Validating memory... Tk_CustomPrintDouble(pre)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    printf("mallocing %d bytes of memory for buf\n", sizeof(char)*TCL_DOUBLE_SPACE);
    */
    buf = (char *)ckalloc(sizeof(char)*TCL_DOUBLE_SPACE);
    /*
    printf("malloc'd memory\n");
    printf("Validating memory... Tk_CustomPringDouble(pos)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    sprintf(buf, "%g", *doublePtr);

    return buf;
}

static void
RemoveDatasetFromGraph(PDatasetItem *dataPtr, PGraphItem *graphPtr)
{
    int		i;
    Tk_Item	*itemPtr = (Tk_Item *)NULL;

    if (!graphPtr)
	return;

    for(i = 0, itemPtr = *graphPtr->children ;
	i < graphPtr->nchildren && (itemPtr != (Tk_Item *)dataPtr); 
	itemPtr = graphPtr->children[i++]);

    for(;
	i < graphPtr->nchildren && (i+1 < graphPtr->child_arr_size);
	graphPtr->children[i] = graphPtr->children[i+1], i++);
    for(;
	i < graphPtr->child_arr_size;
	graphPtr->children[i++] = (Tk_Item *)NULL);
	/*
	 * printf("RemoveDatasetFromGraph(3):  i = %d\n", i);
	 */

    graphPtr->children[graphPtr->nchildren] = (Tk_Item *)NULL;
    graphPtr->nchildren--;

    if (!dataPtr->delete_flag) {
	Tk_CanvasEventuallyRedraw(dataPtr->canvas, 
		dataPtr->header.x1, dataPtr->header.y1,
		dataPtr->header.x2, dataPtr->header.y2);
    }
}

static void
AddDatasetToGraph(PDatasetItem *dataPtr, PGraphItem *graphPtr)
{
    Tk_Item	*itemPtr = (Tk_Item *)NULL;
    int		i;

    /*
    printf("Validating memory... AddDatasetToGraph(start)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */

    Tk_CanvasEventuallyRedraw(dataPtr->canvas, 
	    dataPtr->header.x1, dataPtr->header.y1,
	    dataPtr->header.x2, dataPtr->header.y2);

    if (!graphPtr)
	return;

    if (graphPtr->nchildren+1 >= graphPtr->child_arr_size) {
	Tk_Item **tPtr = (Tk_Item **)ckalloc(sizeof(Tk_Item *)*graphPtr->child_arr_size*2);
	graphPtr->child_arr_size *= 2;
	for(i = 0; i < graphPtr->nchildren; tPtr[i] = graphPtr->children[i++]);
	for(; i < graphPtr->child_arr_size; tPtr[i++] = (Tk_Item *)NULL);
	ckfree((char *)(graphPtr->children));
	graphPtr->children = tPtr;
    }

    for(i = 0, itemPtr = *graphPtr->children ;
	i < graphPtr->nchildren && (itemPtr != (Tk_Item *)dataPtr); 
	itemPtr = graphPtr->children[i++]);

    if (i >= graphPtr->nchildren) {
	graphPtr->children[graphPtr->nchildren++] = (Tk_Item *)dataPtr;
    }

    /*
    printf("Validating memory... AddDatasetToGraph(end)\n");
    Tcl_ValidateAllMemory(__FILE__, __LINE__);
    printf("Done\n");
    */
}
