/* studButton.c --
 *
 *      This module implements a new and improved button widget for
 *	the Tk toolkit.  The "studButton" allows for multi-line button
 *	text, right justified tags which may be * either text (in its
 *	own font) or a bitmap, and buttons which include both text and
 *	a bitmap.  A studButton should * also act just like a button
 *	unless given certain extra * arguments
 *
 * moreButtons : button widgets for Tk
 * Copyright (C) 1993 Daniel Simmons, Michael Hewitt & Idaho State University
 * For distribution terms see the file COPYING (GNU GPL).
 *
 * Based on code from Tk-3.2:
 *
 * Copyright 1990-1992 Regents of the University of California.
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies.  The University of California makes no representations
 * about the suitability of this software for any purpose.  It is
 * provided "as is" without express or implied warranty. 
 */

#include <tkInt.h>


/* local macros for this module */

#define max(a,b) (a > b ? a : b)
#define min(a,b) (a < b ? a : b)


/*
 * A data structure of the following type is kept for each
 * widget managed by this file:
 */

typedef struct {
    Tk_Window tkwin;		/* Window that embodies the button.  NULL
				 * means that the window has been destroyed. */
    Display *display;		/* Display containing widget.  Needed to
				 * free up resources after tkwin is gone. */
    Tcl_Interp *interp;		/* Interpreter associated with button. */


    /* 
     * zone width information
     */

    int leftZone;               /* Width of bitmap + spacing */
    int rightZone;              /* Width of tag + spacing */
    int centerZone;             /* Width of zone containing the text */


    /*
     * bitmap 
     */

    Pixmap bitmap;		/* Bitmap to display or None. */
    int bitmapWidth;            /* If > 0, this is the width of the bitmap. */
    int bitmapHeight;           /* If > 0, this is the height of the bitmap. */
    int bitmapX, bitmapY;       /* X & Y offset from top left of window */


    /* 
     * text 
     */

    char *text;			/* Text to display in button (malloc'ed)
				 * or NULL. */
    int textLength;		/* # of characters in text. */
    char *textVarName;		/* Name of variable (malloc'ed) or NULL.
				 * If non-NULL, button displays the contents
				 * of this variable. */
    XFontStruct *fontPtr;	/* Information about text font, or NULL. */
    GC normalTextGC;		/* GC for drawing text in normal mode.  Also
				 * used to copy from off-screen pixmap onto
				 * screen. */
    GC activeTextGC;		/* GC for drawing text in active mode (NULL
				 * means use normalTextGC). */
    GC disabledGC;		/* Used to produce disabled effect.  If
				 * disabledFg isn't NULL, this GC is used to
				 * draw button text or icon.  Otherwise
				 * text or icon is drawn with normalGC and
				 * this GC is used to stipple background
				 * across it. */
    int textWidth;	        /* Width of text in pixels. */
    int textHeight;		/* Height of text. */
    int textX, textY;           /* X & Y offset from top left of window */


    /* 
     * tag 
     */

    char *tag;                  /* Right justified text to display in button 
				 * (malloc'ed) or NULL. */
    int tagLength;              /* # of characters in tag. */
    Pixmap tagBitmap;           /* Bitmap to display for tag or None. 
				 * If not none, then it over-rides any
				 * definition of tag. */
    XFontStruct *tagFontPtr;    /* Information about tag font, or NULL. */
    GC normalTagGC;             /* GC for tags in normal mode. */
    GC activeTagGC;             /* GC for tags in active mode. */
    GC disabledTagGC;           /* GC for tags in disabled mode. */
    int tagWidth, tagHeight;    /* width and height of tag (text or bitmap) */
    int tagX, tagY;             /* X & Y offset of tag from upper right */


    /*
     * Miscellaneous information:
     */

    int borderWidth;		/* Width of border. */
    int relief;			/* 3-d effect: TK_RELIEF_RAISED, etc. */
    int width, height;		/* If > 0, these specify dimensions to request
				 * for window, in characters for text and in
				 * pixels for bitmaps.  In this case the actual
				 * size of the button contents is ignored in
				 * computing desired window size. */
    int minWidth, maxWidth;     /* If > 0, these specify a minimum and/or
				 * maximum width to request for the window. */
    int padX, padY;		/* Extra space around text or bitmap (pixels
				 * on each side). */
    int spacing;                /* Blank space between bitmap and text, and/or 
				 * text and tag -- specified in pixels. */
    Tk_Uid state;		/* State of button for display purposes:
				 * normal, active, or disabled. */
    Tk_3DBorder normalBorder;	/* Structure used to draw 3-D
				 * border and background when window
				 * isn't active.  NULL means no such
				 * border exists. */
    Tk_3DBorder activeBorder;	/* Structure used to draw 3-D
				 * border and background when window
				 * is active.  NULL means no such
				 * border exists. */
    XColor *normalFg;		/* Foreground color in normal mode. */
    XColor *activeFg;		/* Foreground color in active mode.  NULL
				 * means use normalFg instead. */
    XColor *disabledFg;		/* Foreground color when disabled.  NULL
				 * means use normalFg with a 50% stipple
				 * instead. */
    Pixmap gray;		/* Pixmap for displaying disabled text if
				 * disabledFg is NULL. */
    Cursor cursor;		/* Current cursor for window, or None. */
    char *command;		/* Command to execute when button is
				 * invoked; valid for buttons only.
				 * If not NULL, it's malloc-ed. */
    int flags;			/* Various flags;  see below for
				 * definitions. */
} Button;


/*
 * Flag bits for buttons:
 *
 * REDRAW_PENDING:		Non-zero means a DoWhenIdle handler
 *				has already been queued to redraw
 *				this window.
 * SELECTED:			Non-zero means this button is selected,
 *				so special highlight should be drawn.
 */

#define REDRAW_PENDING		1
#define SELECTED		2


/*
 * Information used for parsing configuration specs:
 */

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_BORDER, "-activebackground", "activeBackground", "Foreground",
	"#eed5b7", Tk_Offset(Button, activeBorder), TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_BORDER, "-activebackground", "activeBackground", "Foreground",
	"black", Tk_Offset(Button, activeBorder), TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_COLOR, "-activeforeground", "activeForeground", "Background",
	"black", Tk_Offset(Button, activeFg), TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_COLOR, "-activeforeground", "activeForeground", "Background",
	"white", Tk_Offset(Button, activeFg), TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_BORDER, "-background", "background", "Background",
	"#ffe4c4", Tk_Offset(Button, normalBorder), TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_BORDER, "-background", "background", "Background",
	"white", Tk_Offset(Button, normalBorder), TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_SYNONYM, "-bd", "borderWidth", (char *) NULL,
	(char *) NULL, 0, 0},
    {TK_CONFIG_SYNONYM, "-bg", "background", (char *) NULL,
	(char *) NULL, 0, 0},
    {TK_CONFIG_BITMAP, "-bitmap", "bitmap", "Bitmap",
	((char *) NULL), Tk_Offset(Button, bitmap),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
	"2", Tk_Offset(Button, borderWidth), 0},
    {TK_CONFIG_STRING, "-command", "command", "Command",
	((char *) NULL), Tk_Offset(Button, command), 0},
    {TK_CONFIG_ACTIVE_CURSOR, "-cursor", "cursor", "Cursor",
	((char *) NULL), Tk_Offset(Button, cursor),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-disabledforeground", "disabledForeground",
	"DisabledForeground", "#b0b0b0", Tk_Offset(Button, disabledFg), 0},
    {TK_CONFIG_COLOR, "-disabledforeground", "disabledForeground",
	"DisabledForeground", ((char *) NULL), Tk_Offset(Button, disabledFg),
 	TK_CONFIG_MONO_ONLY|TK_CONFIG_NULL_OK},
    {TK_CONFIG_SYNONYM, "-fg", "foreground", (char *) NULL,
	(char *) NULL, 0, 0},
    {TK_CONFIG_FONT, "-font", "font", "Font",
	"-Adobe-Helvetica-Bold-R-Normal--*-120-*", 
        Tk_Offset(Button, fontPtr), 0},
    {TK_CONFIG_COLOR, "-foreground", "foreground", "Foreground",
	"black", Tk_Offset(Button, normalFg), 0},
    {TK_CONFIG_INT, "-height", "height", "Height",
	"0", Tk_Offset(Button, height), 0},
    {TK_CONFIG_PIXELS, "-padx", "padX", "Pad",
	"1", Tk_Offset(Button, padX), 0},
    {TK_CONFIG_PIXELS, "-pady", "padY", "Pad",
	"1", Tk_Offset(Button, padY), 0},
    {TK_CONFIG_RELIEF, "-relief", "relief", "Relief",
       "raised", Tk_Offset(Button, relief), 0},
    {TK_CONFIG_UID, "-state", "state", "State",
	"normal", Tk_Offset(Button, state), 0},
    {TK_CONFIG_STRING, "-tag", "tag", "Text", ((char *) NULL), 
       Tk_Offset(Button, tag), TK_CONFIG_NULL_OK}, 
    {TK_CONFIG_BITMAP, "-tagbitmap", "tagbitmap", "Bitmap",
	((char *) NULL), Tk_Offset(Button, tagBitmap),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_FONT, "-tagfont", "tagfont", "Font",
	"-Adobe-Helvetica-Bold-R-Normal--*-120-*", 
        Tk_Offset(Button, tagFontPtr), 0},
    {TK_CONFIG_STRING, "-text", "text", "Text", ((char *) NULL),
	Tk_Offset(Button, text), TK_CONFIG_NULL_OK},
    {TK_CONFIG_STRING, "-textvariable", "textVariable", "Variable",
	((char *) NULL), Tk_Offset(Button, textVarName),
        TK_CONFIG_NULL_OK},
    {TK_CONFIG_INT, "-width", "width", "Width",
	"0", Tk_Offset(Button, width), 0},
    {TK_CONFIG_INT, "-height", "height", "Height",
       "0", Tk_Offset(Button, height), 0},
    {TK_CONFIG_INT, "-minwidth", "minwidth", "MinWidth",
       "0", Tk_Offset(Button, minWidth), 0},
    {TK_CONFIG_INT, "-maxwidth", "maxwidth", "MaxWidth",
       "0", Tk_Offset(Button, maxWidth), 0},
    {TK_CONFIG_INT, "-spacing", "spacing", "Spacing",
       "0", Tk_Offset(Button, spacing), 0},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}

};

/*
 * String to print out in error messages, identifying options for
 * the widget command:
 */

static char *optionString = 
        "activate, configure, deactivate, flash, or invoke";


/*
 * Forward declarations for procedures defined later in this file:
 */

static void		studButtonEventProc _ANSI_ARGS_((ClientData clientData,
			    XEvent *eventPtr));
static char *		studButtonTextVarProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp, 
			    char *name1, char *name2, int flags));
static int		studButtonWidgetCmd _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *interp, int argc, char **argv));
static void		ComputeStudButtonGeometry _ANSI_ARGS_((
                            Button *butPtr));
static int		ConfigureStudButton _ANSI_ARGS_((Tcl_Interp *interp,
			    Button *butPtr, int argc, char **argv,
			    int flags));
static void		DestroyStudButton _ANSI_ARGS_((ClientData clientData));
static void		DisplayStudButton _ANSI_ARGS_((ClientData clientData));
static int		InvokeStudButton  _ANSI_ARGS_((Button *butPtr));

/*
 *--------------------------------------------------------------
 *
 * studButtonCmd --
 *
 *	This procedure is invoked to process the "studButton", 
 *	Tcl command.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

int
studButtonCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Main window associated with
				 * interpreter. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    register Button *butPtr;
    int type;
    Tk_Window tkwin = (Tk_Window) clientData;
    Tk_Window new;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args:  should be \"",
		argv[0], " pathName ?options?\"", (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Create the new window.
     */

    new = Tk_CreateWindowFromPath(interp, tkwin, argv[1], (char *) NULL);
    if (new == NULL) {
	return TCL_ERROR;
    }

    /*
     * Initialize the data structure for the button.
     *
     *     Note: Some of these initializations are NOT absolutely necessary,
     *           but completeness in this area makes me feel better.
     */

    butPtr = (Button *) ckalloc(sizeof(Button));
    butPtr->tkwin = new;
    butPtr->display = Tk_Display(new);
    butPtr->interp = interp;
    butPtr->leftZone = 0;
    butPtr->rightZone = 0;
    butPtr->centerZone = 0;
    butPtr->bitmap = None;
    butPtr->bitmapWidth = 0;
    butPtr->bitmapHeight = 0;
    butPtr->bitmapX = 0;
    butPtr->bitmapY = 0;
    butPtr->text = NULL;
    butPtr->textLength = 0;
    butPtr->textVarName = NULL;
    butPtr->fontPtr = NULL;
    butPtr->normalTextGC = None;
    butPtr->activeTextGC = None;
    butPtr->disabledGC = None;
    butPtr->textWidth = 0;
    butPtr->textHeight = 0;
    butPtr->textX = 0;
    butPtr->textY = 0;
    butPtr->tag = NULL;
    butPtr->tagLength = 0;
    butPtr->tagBitmap = None;
    butPtr->tagFontPtr = NULL;
    butPtr->normalTagGC = None;
    butPtr->activeTagGC = None;
    butPtr->disabledTagGC = None;
    butPtr->tagWidth = 0;
    butPtr->tagHeight = 0;
    butPtr->tagX = 0;
    butPtr->tagY = 0;
    butPtr->borderWidth = 0;
    butPtr->relief = TK_RELIEF_FLAT;
    butPtr->width = 0;
    butPtr->height = 0;
    butPtr->minWidth = 0;
    butPtr->maxWidth = 0;
    butPtr->padX = 0;
    butPtr->padY = 0;
    butPtr->spacing = 0;
    butPtr->state = tkNormalUid;
    butPtr->normalBorder = NULL;
    butPtr->activeBorder = NULL;
    butPtr->normalFg = NULL;
    butPtr->activeFg = NULL;
    butPtr->disabledFg = NULL;
    butPtr->gray = None;
    butPtr->cursor = None;
    butPtr->command = NULL;
    butPtr->flags = 0;

    Tk_SetClass(new, "StudButton");
    Tk_CreateEventHandler(butPtr->tkwin, ExposureMask|StructureNotifyMask,
	    studButtonEventProc, (ClientData) butPtr);
    Tcl_CreateCommand(interp, Tk_PathName(butPtr->tkwin), studButtonWidgetCmd,
	    (ClientData) butPtr, (void (*)()) NULL);
    if (ConfigureStudButton(interp, butPtr, argc-2, argv+2, 0) != TCL_OK) {
	Tk_DestroyWindow(butPtr->tkwin);
	return TCL_ERROR;
    }

    interp->result = Tk_PathName(butPtr->tkwin);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * studButtonWidgetCmd --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

static int
studButtonWidgetCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Information about button widget. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    register Button *butPtr = (Button *) clientData;
    int result = TCL_OK;
    int length;
    char c;

    if (argc < 2) {
	sprintf(interp->result,
		"wrong # args: should be \"%.50s option [arg arg ...]\"",
		argv[0]);
	return TCL_ERROR;
    }
    Tk_Preserve((ClientData) butPtr);
    c = argv[1][0];
    length = strlen(argv[1]);
    if ((c == 'a') && (strncmp(argv[1], "activate", length) == 0)) 
    {
	if (argc > 2) 
	{
	    sprintf(interp->result,
		    "wrong # args: should be \"%.50s activate\"",
		    argv[0]);
	    goto error;
	}
	if (butPtr->state != tkDisabledUid) 
	{
	    butPtr->state = tkActiveUid;
	    Tk_SetBackgroundFromBorder(butPtr->tkwin, butPtr->activeBorder);
	    goto redisplay;
	}
    } 
    else if ((c == 'c') && (strncmp(argv[1], "configure", length) == 0)) 
    {
	if (argc == 2) 
	{
	    result = Tk_ConfigureInfo(interp, butPtr->tkwin, configSpecs,
		    (char *) butPtr, (char *) NULL, 0);
	} 
	else if (argc == 3) 
	{
	    result = Tk_ConfigureInfo(interp, butPtr->tkwin, configSpecs,
		    (char *) butPtr, argv[2], 0);
	} 
	else 
	{
	    result = ConfigureStudButton(interp, butPtr, argc-2, argv+2,
		    TK_CONFIG_ARGV_ONLY);
	}
    } 
    else if ((c == 'd') && (strncmp(argv[1], "deactivate", length) == 0)
	    && (length > 2)) 
    {
	if (argc > 2) 
	{
	    sprintf(interp->result,
		    "wrong # args: should be \"%.50s deactivate\"",
		    argv[0]);
	    goto error;
	}
	if (butPtr->state != tkDisabledUid) 
	{
	    butPtr->state = tkNormalUid;
	    Tk_SetBackgroundFromBorder(butPtr->tkwin, butPtr->normalBorder);
	    goto redisplay;
	}
    } 
    else if ((c == 'f') && (strncmp(argv[1], "flash", length) == 0))
    {
	int i;

	if (argc > 2) 
	{
	    sprintf(interp->result,
		    "wrong # args: should be \"%.50s flash\"",
		    argv[0]);
	    goto error;
	}
	if (butPtr->state != tkDisabledUid) 
	{
	    for (i = 0; i < 4; i++) 
	    {
		butPtr->state = (butPtr->state == tkNormalUid)
			? tkActiveUid : tkNormalUid;
		Tk_SetBackgroundFromBorder(butPtr->tkwin,
			(butPtr->state == tkActiveUid) ? butPtr->activeBorder
			: butPtr->normalBorder);
		DisplayStudButton((ClientData) butPtr);
		XFlush(butPtr->display);
		Tk_Sleep(50);
	    }
	}
    } 
    else if ((c == 'i') && (strncmp(argv[1], "invoke", length) == 0))
    {
	if (argc > 2) 
	{
	    sprintf(interp->result,
		    "wrong # args: should be \"%.50s invoke\"",
		    argv[0]);
	    goto error;
	}
	if (butPtr->state != tkDisabledUid) 
	{
	    result = InvokeStudButton(butPtr);
	}
    } 
    else 
    {
	sprintf(interp->result,
		"bad option \"%.50s\":  must be %s", argv[1],
		optionString);
	goto error;
    }
    Tk_Release((ClientData) butPtr);
    return result;

    redisplay:
    if (Tk_IsMapped(butPtr->tkwin) && !(butPtr->flags & REDRAW_PENDING)) {
	Tk_DoWhenIdle(DisplayStudButton, (ClientData) butPtr);
	butPtr->flags |= REDRAW_PENDING;
    }
    Tk_Release((ClientData) butPtr);
    return TCL_OK;

    error:
    Tk_Release((ClientData) butPtr);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * DestroyStudButton --
 *
 *	This procedure is invoked by Tk_EventuallyFree or Tk_Release
 *	to clean up the internal structure of a studButton at a safe time
 *	(when no-one is using it anymore).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the widget is freed up.
 *
 *----------------------------------------------------------------------
 */

static void
DestroyStudButton(clientData)
    ClientData clientData;		/* Info about entry widget. */
{
    register Button *butPtr = (Button *) clientData;

    if (butPtr->bitmap != None) 
    {
      Tk_FreeBitmap(butPtr->display, butPtr->bitmap);
    }
    if (butPtr->text != NULL) 
      ckfree(butPtr->text);
    if (butPtr->textVarName != NULL) 
    {
      Tcl_UntraceVar(butPtr->interp, butPtr->textVarName,
		     TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		     studButtonTextVarProc, (ClientData) butPtr);
      ckfree(butPtr->textVarName);
    }
    if (butPtr->fontPtr != NULL) 
    {
      Tk_FreeFontStruct(butPtr->fontPtr);
    }
    if (butPtr->normalTextGC != None) 
    {
      Tk_FreeGC(butPtr->display, butPtr->normalTextGC);
    }
    if (butPtr->activeTextGC != None) 
    {
      Tk_FreeGC(butPtr->display, butPtr->activeTextGC);
    }
    if (butPtr->disabledGC != None) 
    {
      Tk_FreeGC(butPtr->display, butPtr->disabledGC);
    }
    if (butPtr->tag != NULL) 
    {
      ckfree(butPtr->tag);
    }
    if (butPtr->tagBitmap != None) 
    {
      Tk_FreeBitmap(butPtr->display, butPtr->tagBitmap);
    }
    if (butPtr->tagFontPtr != NULL)
    {
      Tk_FreeFontStruct(butPtr->tagFontPtr);
    }
    if (butPtr->normalTagGC != None) 
    {
      Tk_FreeGC(butPtr->display, butPtr->normalTagGC);
    }
    if (butPtr->activeTagGC != None) 
    {
      Tk_FreeGC(butPtr->display, butPtr->activeTagGC);
    }
    if (butPtr->disabledTagGC != None) 
    {
      Tk_FreeGC(butPtr->display, butPtr->disabledTagGC);
    }
    if (butPtr->normalBorder != NULL) 
    {
      Tk_Free3DBorder(butPtr->normalBorder);
    }
    if (butPtr->activeBorder != NULL) 
    {
      Tk_Free3DBorder(butPtr->activeBorder);
    }
    if (butPtr->normalFg != NULL) 
    {
      Tk_FreeColor(butPtr->normalFg);
    }
    if (butPtr->activeFg != NULL) 
    {
      Tk_FreeColor(butPtr->activeFg);
    }
    if (butPtr->disabledFg != NULL) 
    {
      Tk_FreeColor(butPtr->disabledFg);
    }
    if (butPtr->gray != None) 
    {
      Tk_FreeBitmap(butPtr->display, butPtr->gray);
    }
    if (butPtr->cursor != None) 
    {
      Tk_FreeCursor(butPtr->display, butPtr->cursor);
    }
    if (butPtr->command != NULL) 
    {
      ckfree(butPtr->command);
    }
    ckfree((char *) butPtr);
  }

/*
 *----------------------------------------------------------------------
 *
 * ConfigureStudButton --
 *
 *	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or
 *	reconfigure) a studButton widget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as text string, colors, font,
 *	etc. get set for butPtr;  old resources get freed, if there
 *	were any.  The button is redisplayed.
 *
 *----------------------------------------------------------------------
 */

static int
ConfigureStudButton(interp, butPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    register Button *butPtr;	/* Information about widget;  may or may
				 * not already have values for some fields. */
    int argc;			/* Number of valid entries in argv. */
    char **argv;		/* Arguments. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    XGCValues gcValues;
    GC newGC;
    unsigned long mask;

    /*
     * Eliminate any existing trace on variables monitored by the button.
     */

    if (butPtr->textVarName != NULL) {
	Tcl_UntraceVar(interp, butPtr->textVarName, 
		TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		studButtonTextVarProc, (ClientData) butPtr);
    }

    if (Tk_ConfigureWidget(interp, butPtr->tkwin, configSpecs,
	    argc, argv, (char *) butPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * A few options need special processing, such as setting the
     * background from a 3-D border, or filling in complicated
     * defaults that couldn't be specified to Tk_ConfigureWidget.
     */

    if (butPtr->state == tkActiveUid) {
	Tk_SetBackgroundFromBorder(butPtr->tkwin, butPtr->activeBorder);
    } else {
	Tk_SetBackgroundFromBorder(butPtr->tkwin, butPtr->normalBorder);
	if ((butPtr->state != tkNormalUid)
		&& (butPtr->state != tkDisabledUid)) {
	    Tcl_AppendResult(interp, "bad state value \"", butPtr->state,
		    "\":  must be normal, active, or disabled", (char *) NULL);
	    butPtr->state = tkNormalUid;
	    return TCL_ERROR;
	}
    }

    gcValues.font = butPtr->fontPtr->fid;
    gcValues.foreground = butPtr->normalFg->pixel;
    gcValues.background = Tk_3DBorderColor(butPtr->normalBorder)->pixel;

    /*
     * Note: GraphicsExpose events are disabled in normalTextGC because it's
     * used to copy stuff from an off-screen pixmap onto the screen (we know
     * that there's no problem with obscured areas).
     */

    gcValues.graphics_exposures = False;
    newGC = Tk_GetGC(butPtr->tkwin,
	    GCForeground|GCBackground|GCFont|GCGraphicsExposures, &gcValues);
    if (butPtr->normalTextGC != None) {
	Tk_FreeGC(butPtr->display, butPtr->normalTextGC);
    }
    butPtr->normalTextGC = newGC;

    gcValues.font = butPtr->tagFontPtr->fid;
    newGC = Tk_GetGC(butPtr->tkwin,
	    GCForeground|GCBackground|GCFont|GCGraphicsExposures, &gcValues);
    if (butPtr->normalTagGC != None) {
	Tk_FreeGC(butPtr->display, butPtr->normalTagGC);
    }
    butPtr->normalTagGC = newGC;

    if (butPtr->activeFg != NULL) {
	gcValues.font = butPtr->fontPtr->fid;
	gcValues.foreground = butPtr->activeFg->pixel;
	gcValues.background = Tk_3DBorderColor(butPtr->activeBorder)->pixel;
	newGC = Tk_GetGC(butPtr->tkwin, GCForeground|GCBackground|GCFont,
		&gcValues);
	if (butPtr->activeTextGC != None) {
	    Tk_FreeGC(butPtr->display, butPtr->activeTextGC);
	}
	butPtr->activeTextGC = newGC;

	gcValues.font = butPtr->tagFontPtr->fid;
	gcValues.foreground = butPtr->activeFg->pixel;
	gcValues.background = Tk_3DBorderColor(butPtr->activeBorder)->pixel;
	newGC = Tk_GetGC(butPtr->tkwin, GCForeground|GCBackground|GCFont,
		&gcValues);
	if (butPtr->activeTagGC != None) {
	    Tk_FreeGC(butPtr->display, butPtr->activeTagGC);
	}
	butPtr->activeTagGC = newGC;
    }

    gcValues.font = butPtr->fontPtr->fid;
    gcValues.background = Tk_3DBorderColor(butPtr->normalBorder)->pixel;
    if (butPtr->disabledFg != NULL) {
	gcValues.foreground = butPtr->disabledFg->pixel;
	mask = GCForeground|GCBackground|GCFont;
    } else {
	gcValues.foreground = gcValues.background;
	if (butPtr->gray == None) {
	    butPtr->gray = Tk_GetBitmap(interp, butPtr->tkwin,
		    Tk_GetUid("gray50"));
	    if (butPtr->gray == None) {
		return TCL_ERROR;
	    }
	}
	gcValues.fill_style = FillStippled;
	gcValues.stipple = butPtr->gray;
	mask = GCForeground|GCFillStyle|GCStipple;
    }
    newGC = Tk_GetGC(butPtr->tkwin, mask, &gcValues);
    if (butPtr->disabledGC != None) {
	Tk_FreeGC(butPtr->display, butPtr->disabledGC);
    }
    butPtr->disabledGC = newGC;

    gcValues.font = butPtr->tagFontPtr->fid;
    newGC = Tk_GetGC(butPtr->tkwin, mask, &gcValues);
    if (butPtr->disabledTagGC != None) {
	Tk_FreeGC(butPtr->display, butPtr->disabledTagGC);
    }
    butPtr->disabledTagGC = newGC;

    if (butPtr->padX < 0) {
	butPtr->padX = 0;
    }
    if (butPtr->padY < 0) {
	butPtr->padY = 0;
    }

    /*
     * If the button is to display the value of a variable, then set up
     * a trace on the variable's value, create the variable if it doesn't
     * exist, and fetch its current value.
     */

    if (butPtr->textVarName != NULL) {
	char *value;

	value = Tcl_GetVar(interp, butPtr->textVarName, TCL_GLOBAL_ONLY);
	if (value == NULL) {
	    Tcl_SetVar(interp, butPtr->textVarName, butPtr->text,
		    TCL_GLOBAL_ONLY);
	} else {
	    if (butPtr->text != NULL) {
		ckfree(butPtr->text);
	    }
	    butPtr->text = (char *) ckalloc((unsigned) (strlen(value) + 1));
	    strcpy(butPtr->text, value);
	}
	Tcl_TraceVar(interp, butPtr->textVarName,
		TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		studButtonTextVarProc, (ClientData) butPtr);
    }

    ComputeStudButtonGeometry(butPtr);

    /*
     * Lastly, arrange for the button to be redisplayed.
     */

    if (Tk_IsMapped(butPtr->tkwin) && !(butPtr->flags & REDRAW_PENDING)) {
	Tk_DoWhenIdle(DisplayStudButton, (ClientData) butPtr);
	butPtr->flags |= REDRAW_PENDING;
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * DisplayStudButton --
 *
 *	This procedure is invoked to display a studButton widget.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Commands are output to X to display the button in its
 *	current mode.
 *
 *----------------------------------------------------------------------
 */

static void
DisplayStudButton(clientData)
    ClientData clientData;	/* Information about widget. */
{
  register Button *butPtr = (Button *) clientData;
  register Tk_Window tkwin = butPtr->tkwin;
  GC gc, tagGC;
  Tk_3DBorder border;
  Pixmap pixmap;
  int offset = 0;


/* compute the geometry, then return unless the window is mapped */

  ComputeStudButtonGeometry(butPtr); 

  butPtr->flags &= ~REDRAW_PENDING;
  if ((butPtr->tkwin == NULL) || !Tk_IsMapped(tkwin)) {
    return;
  }


/* setup the GCs, the border, and the offsets (for the relief) */

  if ((butPtr->state == tkDisabledUid) && (butPtr->disabledFg != NULL)) 
  {
    gc = butPtr->disabledGC;
    tagGC = butPtr->disabledTagGC;
    border = butPtr->normalBorder;
  } 
  else if (butPtr->state == tkActiveUid) 
  {
    gc = butPtr->activeTextGC;
    tagGC = butPtr->activeTagGC;
    border = butPtr->activeBorder;
  } 
  else 
  {
    gc = butPtr->normalTextGC;
    tagGC = butPtr->normalTagGC;
    border = butPtr->normalBorder;
  }
  
  if (butPtr->relief == TK_RELIEF_RAISED) 
  {
    offset = -1;
  } 
  else if (butPtr->relief == TK_RELIEF_SUNKEN) 
  {
    offset = 1;
  }

  
/* prepare pixmap */

  /*
   * In order to avoid screen flashes, this procedure redraws
   * the button in a pixmap, then copies the pixmap to the
   * screen in a single operation.  This means that there's no
   * point in time where the on-sreen image has been cleared.
   */

  pixmap = XCreatePixmap(butPtr->display, Tk_WindowId(tkwin),
			 Tk_Width(tkwin), Tk_Height(tkwin), 
			 Tk_Depth(tkwin));
  Tk_Fill3DRectangle(butPtr->display, pixmap, border,
		     0, 0, Tk_Width(tkwin), Tk_Height(tkwin), 
		     0, TK_RELIEF_FLAT);


/* display the bitmap */

  if (butPtr->bitmap != None)
  {
    XCopyPlane(butPtr->display, butPtr->bitmap, pixmap, gc, 0, 0,
	       butPtr->bitmapWidth, butPtr->bitmapHeight,
	       butPtr->bitmapX + offset, butPtr->bitmapY + offset, 1);
  }


/* display the tag -- we'll leave the hard part for last */

  /* don't forget that tagX is in pixels from the RIGHT */

  if (butPtr->tagBitmap != None)
  {
    XCopyPlane(butPtr->display, butPtr->tagBitmap, pixmap, gc, 0, 0,
	       butPtr->tagWidth, butPtr->tagHeight,
	       Tk_Width(butPtr->tkwin) - butPtr->tagX + offset,
	       butPtr->tagY + offset, 1);
  }
  else if (butPtr->tag)
  {
    XDrawString(butPtr->display, pixmap, tagGC, 
		Tk_Width(butPtr->tkwin) - butPtr->tagX + offset,
		butPtr->tagY + offset, butPtr->tag, butPtr->tagLength);
  }
  

/* display the text */

  if (butPtr->text)
  {
    int lineWidth, numChars, y, charsLeft = butPtr->textLength;
    int fontHeight = butPtr->fontPtr->ascent + butPtr->fontPtr->descent;
    char *p;
    
    for (p=butPtr->text, y=butPtr->textY + offset; *p != 0; y += fontHeight)
    {
      /* if the user requests a newline, go to the next line */
      if (*p == '\n')
      {
	p++;
	charsLeft--;
	continue;
      }

      /* wrap at whole words, display the line */
      numChars = TkMeasureChars(butPtr->fontPtr, p, charsLeft, 0,
				butPtr->centerZone,
				TK_WHOLE_WORDS|TK_AT_LEAST_ONE,
				&lineWidth);
      TkDisplayChars(butPtr->display, pixmap, gc, butPtr->fontPtr, p,
		     numChars, butPtr->textX + offset, y, 0);
      
      /* increase p, decrease charsLeft */
      p += numChars;
      charsLeft -= numChars;
      
      /* skip spaces and tabs at the beginning of any line which
	 is the result of a word-wrap (not user-requested) */
      while (isspace(*p))
      {
	charsLeft--;
	if (*p == '\n')
	{
	  p++;
	  break;
	}
	p++;
      } /* while isspace */

    } /* word-wrap loop */

  } /* if (butPtr->text) */
    

/* handle stippling if necessary for disabled button */

  if ((butPtr->state == tkDisabledUid) && (butPtr->disabledFg == NULL)) 
  {
    XFillRectangle(butPtr->display, pixmap, butPtr->disabledGC,
		   butPtr->borderWidth, butPtr->borderWidth,
		   (unsigned) (Tk_Width(tkwin) - 2*butPtr->borderWidth),
		   (unsigned) (Tk_Height(tkwin) - 2*butPtr->borderWidth));
  }


/* draw the border (last to cover-up any overflow) */

  if (butPtr->relief != TK_RELIEF_FLAT) 
  {
    Tk_Draw3DRectangle(butPtr->display, pixmap, border,0, 0,
		       Tk_Width(tkwin), Tk_Height(tkwin), butPtr->borderWidth,
		       butPtr->relief);
  }


/* copy the pixmap onto the screen, then delete it */

  XCopyArea(butPtr->display, pixmap, Tk_WindowId(tkwin),
	    butPtr->normalTextGC, 0, 0, 
	    Tk_Width(tkwin), Tk_Height(tkwin), 0, 0);
  XFreePixmap(butPtr->display, pixmap);
}

/*
 *--------------------------------------------------------------
 *
 * studButtonEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher for various
 *	events on buttons.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get
 *	cleaned up.  When it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */

static void
studButtonEventProc(clientData, eventPtr)
    ClientData clientData;	/* Information about window. */
    XEvent *eventPtr;		/* Information about event. */
{
    Button *butPtr = (Button *) clientData;
    if ((eventPtr->type == Expose) && (eventPtr->xexpose.count == 0)) {
	if ((butPtr->tkwin != NULL) && !(butPtr->flags & REDRAW_PENDING)) {
	    Tk_DoWhenIdle(DisplayStudButton, (ClientData) butPtr);
	    butPtr->flags |= REDRAW_PENDING;
	}
    } else if (eventPtr->type == DestroyNotify) {
	Tcl_DeleteCommand(butPtr->interp, Tk_PathName(butPtr->tkwin));
	butPtr->tkwin = NULL;
	if (butPtr->flags & REDRAW_PENDING) {
	    Tk_CancelIdleCall(DisplayStudButton, (ClientData) butPtr);
	}
	Tk_EventuallyFree((ClientData) butPtr, DestroyStudButton);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ComputeStudButtonGeometry --
 *
 * 	After changes in a button's text, tag or bitmap, this p *
 * recomputes the button's geometry and passes this information *
 * along to the geometry manager for the window.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The button's window may change size.  The X and Y positions of
 *      the text, tag and bitmap are set in the widget record.
 *
 * BEWARE, ALL YE WHO ENTER: 
 *      THERE BE CRUFTY DRAGONS HERE 
 *      (distant cousins of the dragons found in xterm)
 *
 *----------------------------------------------------------------------
 */

static void
ComputeStudButtonGeometry(butPtr)
    register Button *butPtr;	/* Button whose geometry may have changed. */
{
  unsigned int width, height;  /* NOTE: These variables serve double duty.
                                        1. Final calculations at the end.
					2. Interim unsigned ints for 
					   working with bitmap sizes. */

/* initialize left and right zones to include border and padding */

  butPtr->leftZone = butPtr->padX + butPtr->borderWidth;
  butPtr->rightZone = butPtr->padX + butPtr->borderWidth;


/* bitmap */

  if (butPtr->bitmap != None)
  {
    /* size */
    Tk_SizeOfBitmap(butPtr->display, butPtr->bitmap, &width, &height);
    butPtr->bitmapWidth = width;
    butPtr->bitmapHeight = height;
    

    /* X position -- modified slightly in display depending on the relief */
    /* Note: We have to calculate Y positions after the sizes are complete 
             since the Y positions are dependent on the relative heights 
	     of the text, tag and bitmap. */
    butPtr->bitmapX = butPtr->borderWidth + butPtr->padX;
    
    /* leftZone */
    butPtr->leftZone += butPtr->bitmapWidth;
    if (butPtr->text)
    {
      butPtr->leftZone += butPtr->spacing;
    }
  }
  else
  {
    butPtr->bitmapWidth = 0;
    butPtr->bitmapHeight = 0;
    butPtr->bitmapX = 0;
  }


/* tag */  

  if (butPtr->tagBitmap != None)
  {
    /* size */
    Tk_SizeOfBitmap(butPtr->display, butPtr->tagBitmap, &width, &height);
    butPtr->tagWidth = width;
    butPtr->tagHeight = height;

    /* X position -- specified as an offset from right of window */
    butPtr->tagX = butPtr->tagWidth + butPtr->borderWidth + butPtr->padX;

    /* rightZone */
    butPtr->rightZone += butPtr->tagWidth;
    if (butPtr->text || (butPtr->bitmap != None))
    {
      butPtr->rightZone += butPtr->spacing;
    }
  }
  else if (butPtr->tag) 
  {
    XCharStruct bbox;
    int direction, ascent, descent;

    /* size */
    butPtr->tagLength = strlen(butPtr->tag);
    XTextExtents(butPtr->tagFontPtr, butPtr->tag, butPtr->tagLength,
		 &direction, &ascent, &descent, &bbox);
    butPtr->tagWidth = bbox.lbearing + bbox.rbearing;
    butPtr->tagHeight = ascent + descent;
    
    /* position -- specified as an offset from top right of window */
    butPtr->tagX = bbox.rbearing + butPtr->borderWidth + butPtr->padX;
    
    /* rightZone */
    butPtr->rightZone += butPtr->tagWidth;
    if (butPtr->text || (butPtr->bitmap != None))
    {
      butPtr->rightZone += butPtr->spacing;
    }
  }
  else
  {
    butPtr->tagWidth = 0;
    butPtr->tagHeight = 0;
    butPtr->tagX = 0;
  }
  

/* text */

  if (butPtr->text)
  {
    butPtr->textLength = strlen(butPtr->text);

    if ((butPtr->width > 0) || (butPtr->maxWidth > 0))
    {
      int windowWidth, widthAvail, numLines, lineWidth, maxLineWidth = 0;
      char *p;

    /* calculate the width available:
          if the window is mapped, base calculations on fact
	  if not, base calculations on what the user specified */

      if (butPtr->tkwin && (Tk_IsMapped(butPtr->tkwin)))
      {
	windowWidth = Tk_Width(butPtr->tkwin);
      }
      else
      {
	windowWidth = (butPtr->width > 0 ? butPtr->width : butPtr->maxWidth);
      }
      widthAvail = windowWidth - butPtr->leftZone - butPtr->rightZone;

    /* do the word wrapping */

      for (numLines = 0, p = butPtr->text; *p != 0; numLines++)
      {
	/* if the user requests a newline, just increase the numLines */
	if (*p == '\n')
	{
	  p++;
	  continue;
	}

	/* wrap at whole words, keep track of maximum width */
	p += TkMeasureChars(butPtr->fontPtr, p,
			    butPtr->textLength - (p - butPtr->text), 0,
			    widthAvail, TK_WHOLE_WORDS|TK_AT_LEAST_ONE,
			    &lineWidth);
	maxLineWidth = max(maxLineWidth, lineWidth);
	
	/* skip spaces and tabs at the beginning of any line which
	   is the result of a word-wrap (not user-requested) */
	while (isspace(*p))
	{
	  if (*p == '\n') 
	  {
	    p++;
	    break;
	  }
	  p++;
	}
      }

      /* size */
      butPtr->textWidth = maxLineWidth;
      butPtr->textHeight = numLines * (butPtr->fontPtr->ascent 
				       + butPtr->fontPtr->descent);

      /* X-position -- offset from left of window */
      butPtr->textX = butPtr->leftZone;

      /* centerZone */
      butPtr->centerZone = widthAvail;
    }
    else
    {
      /* size */
      butPtr->textWidth = XTextWidth(butPtr->fontPtr, butPtr->text, 
				     butPtr->textLength);
      butPtr->textHeight = butPtr->fontPtr->ascent + butPtr->fontPtr->descent;

      /* X-position -- offset from left of window */
      butPtr->textX = butPtr->leftZone;

      /* centerZone */
      butPtr->centerZone = butPtr->textWidth;
    }
  }
  else
  {
    butPtr->textWidth = 0;
    butPtr->textHeight = 0;
    butPtr->textX = 0;
  }


/* now calculate the Y positions: vertically center smaller objects, but
   make the tag and the text line up (if both exist) */

  if (butPtr->bitmap)
  {

  /* we have a bitmap so do the centering */

    int yOffset = butPtr->borderWidth + butPtr->padY + 1;
    int textTagMaxHeight = max(butPtr->textHeight, butPtr->tagHeight);
    int centeringOffset = abs(textTagMaxHeight - butPtr->bitmapHeight) / 2;

    /* everything is at least at the offset (+ font ascent if text) */
    butPtr->bitmapY = yOffset;
    if (butPtr->text) 
      butPtr->textY = yOffset + butPtr->fontPtr->ascent;
    if (butPtr->tagBitmap != None) 
      butPtr->tagY = yOffset;
    else if (butPtr->tag) 
      butPtr->tagY = yOffset + butPtr->tagFontPtr->ascent;

    /* center either the text and tag or the bitmap */
    if (butPtr->bitmapHeight > textTagMaxHeight)
    {
      if (butPtr->text) 
	butPtr->textY += centeringOffset;
      if (butPtr->tag || (butPtr->tagBitmap != None))
	butPtr->tagY += centeringOffset;
    } 
    else
    {
      butPtr->bitmapY += centeringOffset;
    }
  }
  else
  {

  /* no bitmap, just make the text and tag appear at the yOffset */

    int yOffset = butPtr->borderWidth + butPtr->padY + 1;

    if (butPtr->text)
    {
      butPtr->textY = yOffset + butPtr->fontPtr->ascent;
    }
    
    if (butPtr->tagBitmap != None)
    {
      butPtr->tagY = yOffset;
    }
    else if (butPtr->tag)
    {
      butPtr->tagY = yOffset + butPtr->tagFontPtr->ascent;
    }
  }
  

/* force width and height within user specs, then make our request */

  if (butPtr->width > 0)
  {
    /* request specified width */
    width = butPtr->width;
  }
  else
  {
    /* request sum of leftZone, rightZone and textWidth, as long as
       that sum falls between minWidth and maxWidth */
    width = butPtr->leftZone + butPtr->textWidth + butPtr->rightZone;
    width = max(width, butPtr->minWidth);
    width = min(width, butPtr->maxWidth);
  }

  if (butPtr->height > 0)
  {
    /* request specified height */
    height = butPtr->height;
  }
  else
  {
    /* request the greatest height among: bitmap, text and tag */
    height = max(butPtr->bitmapHeight, butPtr->textHeight);
    height = max(height, butPtr->tagHeight);

    /* add room for the border and padding */
    height += 2*butPtr->borderWidth + 2*butPtr->padY + 2; 
  }
  
  Tk_GeometryRequest(butPtr->tkwin, (int) width, (int) height);
  Tk_SetInternalBorder(butPtr->tkwin, butPtr->borderWidth);
}

/*
 *----------------------------------------------------------------------
 *
 * InvokeStudButton --
 *
 *	This procedure is called to carry out the actions associated
 *	with a button, such as invoking a Tcl command or setting a
 *	variable.  This procedure is invoked, for example, when the
 *	button is invoked via the mouse.
 *
 * Results:
 *	A standard Tcl return value.  Information is also left in
 *	interp->result.
 *
 * Side effects:
 *	Depends on the button and its associated command.
 *
 *----------------------------------------------------------------------
 */

static int
InvokeStudButton(butPtr)
    register Button *butPtr;		/* Information about button. */
{
    if (butPtr->command != NULL) 
    {
	return Tcl_GlobalEval(butPtr->interp, butPtr->command);
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * studButtonTextVarProc --
 *
 *	This procedure is invoked when someone changes the variable
 *	whose contents are to be displayed in a button.
 *
 * Results:
 *	NULL is always returned.
 *
 * Side effects:
 *	The text displayed in the button will change to match the
 *	variable.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static char *
studButtonTextVarProc(clientData, interp, name1, name2, flags)
    ClientData clientData;	/* Information about button. */
    Tcl_Interp *interp;		/* Interpreter containing variable. */
    char *name1;		/* Name of variable. */
    char *name2;		/* Second part of variable name. */
    int flags;			/* Information about what happened. */
{
    register Button *butPtr = (Button *) clientData;
    char *value;

    /*
     * If the variable is unset, then immediately recreate it unless
     * the whole interpreter is going away.
     */

    if (flags & TCL_TRACE_UNSETS) {
	if ((flags & TCL_TRACE_DESTROYED) && !(flags & TCL_INTERP_DESTROYED)) {
	    Tcl_SetVar2(interp, name1, name2, butPtr->text,
		    flags & TCL_GLOBAL_ONLY);
	    Tcl_TraceVar2(interp, name1, name2,
		    TCL_GLOBAL_ONLY|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		    studButtonTextVarProc, clientData);
	}
	return (char *) NULL;
    }

    value = Tcl_GetVar2(interp, name1, name2, flags & TCL_GLOBAL_ONLY);
    if (value == NULL) {
	value = "";
    }
    if (butPtr->text != NULL) {
	ckfree(butPtr->text);
    }
    butPtr->text = (char *) ckalloc((unsigned) (strlen(value) + 1));
    strcpy(butPtr->text, value);
    ComputeStudButtonGeometry(butPtr); 

    if ((butPtr->tkwin != NULL) && Tk_IsMapped(butPtr->tkwin)
	    && !(butPtr->flags & REDRAW_PENDING)) {
	Tk_DoWhenIdle(DisplayStudButton, (ClientData) butPtr);
	butPtr->flags |= REDRAW_PENDING;
    }
    return (char *) NULL;
}

