# Jots Client Tcl code for tkgoodstuff
# Mark Crimmins (markcrim@umich.edu)

setifunset JotsDir "$env(HOME)/.tkgjots"
setifunset JotsHotFile "$JotsDir/.hotjots"
setifunset Jots_startwithnew 1
setifunset Jots_lines 20
setifunset Jots_columns 60
setifunset Jots_showfindtool 0
setifunset Jots_hotcolor \#bcdfff
setifunset Jots_makebutton 0
setifunset Jots_image $images/jots.$TKG_defformat

setifunset JotsHotsCBG $Jots_hotcolor
setifunset JotsHotsCFG $TKGforeground
setifunset JotsHotsCABG $JotsHotsCBG
setifunset JotsHotsCAFG $JotsHotsCFG

if { [lsearch $Clients Clock ] == -1 } {
    error {'Client Clock' must be before 'Client Jots' in your ~/.tkgoodstuffrc}
}

proc create-Jots-window {} {
    global Clock_window Jots_makebutton Jots_image
    if $Jots_makebutton {
	SetImage Jots_image $Jots_image
	TKGButton JotsButton -image Jots_image -text Jots -tclcmd Jotspopup
    }
    RecursiveBind [set Clock_window] <3> {Jotspopup}
}

proc Jotspopup {} {
    global env JotsList jots_edited jots_index
    global textwindow JotsSB Jots_columns Jots_lines
    global Jots_startwithnew Jots_showfindtool JotsHotsWindow
    global Jots_showhotlist JotsFolder Jots_selectfolders
    
    set JotsList {}
    set jots_index 0
    set Jots_selectfolders 1
    set jots_edited -1
    
    set w .jots
    if [winfo exists $w] {
	focus $w
	raise $w
	return
    }
    toplevel $w
    wm withdraw $w
    wm title $w "Jots"
    wm iconname $w "Jots"
    
    frame $w.menu -relief raised -bd 2
    pack $w.menu -side top -fill x
    
    set m $w.menu.file.m
    menubutton $w.menu.file -text "File" -menu $m -underline 0
    menu $m
    $m add command -label "Open folder" -command JotsOpen
    $m add command -label "New folder" -command JotsNew
    $m add command -label "Save current folder" -command JotsSave -underline 0
    $m add command -label "Save current folder as ..." -command JotsSaveAs
    $m add command -label "Save to ascii file ..." -command JotsSaveAscii
    $m add command -label "Delete current folder ..." -command JotsDeleteFolder
    $m add separator
    $m add command -label "Exit without saving ..." -command "JotsExit"

    set m $w.menu.hotlist.m
    menubutton $w.menu.hotlist -text "Hotlist" -menu $m
    menu $m
    $m add command -label "Add folder to hot list" -command JotsAddHot
    $m add command -label "Remove folder from hot list" -command JotsRemoveHot
    $m add separator
    
    set m $w.menu.edit.m
    menubutton $w.menu.edit -text "Edit" -menu $m -underline 0
    menu $m
    $m add command -label "Delete Entry" -command JotsDelete -underline 0
    
    set m $w.menu.find.m
    menubutton $w.menu.find -text "Find" -menu $m
    menu $m
    $m add checkbutton -label "Show Find Tool" -variable Jots_showfindtool
    trace variable Jots_showfindtool w JotsPackFind
    $m add separator
    
    pack $w.menu.file $w.menu.edit $w.menu.find $w.menu.hotlist -side left
    
    set m $w.menu.help.m
    menubutton $w.menu.help -text "Help" -menu $m -underline 0
    menu $m
    $m add command -label "About Jots" -command JotsAbout -underline 0
    $m add separator
    $m add command -label "Help" -command JotsHelp -underline 0
    
    pack $w.menu.help -side right
    
    frame $w.hots
    pack $w.hots -side top -fill x -padx .1c -pady .1c
    set JotsHotsWindow $w.hots
    
    frame $w.middle 
    
    set ww $w.middle.jotsentry 
    frame $ww -relief sunken -bd 3
    frame $ww.view
    text $ww.view.text -width $Jots_columns -height $Jots_lines\
	-takefocus 0 -yscrollcommand "$ww.view.scrollbar set" \
	-relief sunken -borderwidth 2 -state disabled -wrap word
    set textwindow $ww.view.text
    pack $ww.view.text -side left -fill both -expand 1
    scrollbar $ww.view.scrollbar -command "$ww.view.text yview"
    pack $ww.view.scrollbar -side left -fill y -padx 2
    pack $ww.view -side top -fill both -expand 1 -padx .1c
    $ww.view.text configure -state normal
    bind $textwindow <Key> {set jots_edited $jots_index}
    bind $textwindow <Button> {set jots_edited $jots_index}
    pack $ww.view -fill x -pady .3c -padx .3c
    
    frame $ww.indices
    frame $ww.indices.jots_folder
    label $ww.indices.jots_folder.jots_folderlabel -text Folder:
    pack $ww.indices.jots_folder.jots_folderlabel -side left
    entry $ww.indices.jots_folder.jots_folderentry -textvariable JotsFolder
    pack $ww.indices.jots_folder.jots_folderentry -side left -expand y
    frame $ww.indices.jots_date
    label $ww.indices.jots_date.jots_datelabel -text Date:
    pack $ww.indices.jots_date.jots_datelabel -side left
    entry $ww.indices.jots_date.jots_dateentry -textvariable jots_date -width 18
    pack $ww.indices.jots_date.jots_dateentry -side left -fill x -expand y
    frame $ww.indices.jots_time
    label $ww.indices.jots_time.jots_timelabel -text Time:
    pack $ww.indices.jots_time.jots_timelabel -side left
    entry $ww.indices.jots_time.jots_timeentry -textvariable jots_time -width 5
    pack $ww.indices.jots_time.jots_timeentry -side left -fill x -expand y
    $ww.indices.jots_folder.jots_folderentry configure -background [$ww.view cget -background]
    $ww.indices.jots_date.jots_dateentry configure -background [$ww.view cget -background]
    $ww.indices.jots_time.jots_timeentry configure -background [$ww.view cget -background]
    
    pack $ww.indices.jots_folder $ww.indices.jots_date $ww.indices.jots_time\
	-fill x -padx .2c -pady .1c -side left
    pack $ww.indices
    pack $ww -side top
    
    pack $w.middle -padx .3c -pady .2c
    
    scrollbar $w.scroll -orient horizontal -command "JotsScroll" \
	-width .7c
    pack $w.scroll -expand y -fill x -pady .3c -padx .3c
    set JotsSB $w.scroll
    trace variable jots_index w JotsSUpdate

    frame $w.buttons
    button $w.buttons.new -command JotsNewEntry  -text "New\nEntry"
    pack $w.buttons.new -side left -expand y -fill y -padx .4c -pady .4c
    button $w.buttons.del -command JotsDelete  -text "Delete\nEntry"
    pack $w.buttons.del -side left -expand y -fill y -padx .4c -pady .4c
    button $w.buttons.prev -command JotsPrev  -text "Previous\nEntry"
    pack $w.buttons.prev -side left -expand y -fill y -padx .4c -pady .4c
    button $w.buttons.next -command JotsNext  -text "Next\nEntry"
    pack $w.buttons.next -side left -expand y -fill y -padx .4c -pady .4c
    button $w.buttons.dismiss -text "Save and\nExit"\
	-command JotsDismiss
    pack $w.buttons.dismiss -side left -expand y -fill y -padx .4c -pady .4c
    pack $w.buttons -fill x -expand y

    bind $w <Alt-q> {JotsDismiss;break}
    bind $w <Alt-x> {JotsDismiss;break}
    bind $w <Shift-Alt-Right> {JotsNext;break}
    bind $w <Shift-Alt-Left> {JotsPrev;break}
    bind $w <Shift-Alt-D> {JotsDelete;break}
    bind $w <Shift-Alt-N> {JotsNewEntry;break}

    JotsDirInit
}

######################################
#  File handling routines
######################################

proc JotsDirInit {} {
    global JotsDir JotsOldCWD
    set JotsOldCWD [pwd]
    if [file exists $JotsDir] {
	if ![file isdirectory $JotsDir] {
	    TKGError "$JotsDir (selected as your Jots directory) is not a directory"
	    JotsExit
	    return
	}
	if { ![file readable $JotsDir] || 
	     ![file writable $JotsDir] || 
	     ![file executable $JotsDir] } {
	    if [file owned $JotsDir] {
		if [catch "exec chmod u+rwx $JotsDir" err ] {
		    TKGError "Can't set permissions on \"$JotsDir\" (selected\
as your Jots directory): $err"
		    JotsExit
		    return
		}
		JotsDirOpen
	    } else {
		TKGError "You have inadequate permissions for\
 directory \"$JotsDir\" (selected as your Jots directory), and you don't own it."
	        JotsExit
		return
	    }
	}	
	JotsDirOpen
	return
    }
    TKGDialog jotspop\
	-title "Jots:"\
	-message "Directory \"$JotsDir\", needed for Jots, does not exist." \
	-nodismiss \
	-buttons { 
	    { create "Create it" {
		if [catch "exec mkdir $JotsDir" text] {
		    error $text
		}
		if [catch "exec chmod 750 $JotsDir" err] {
		    error $err
		}
		destroy .jotspop
		JotsDirOpen
	    }			
	    }		    
	    { exit "Exit Jots" {
		destroy .jotspop
		JotsExit
	    }
	    }
	}
}

proc JotsDirOpen {} {    
    global JotsDir JotsFolder JotsFolderList
    cd $JotsDir
    if [catch "set filelist \"[exec /bin/ls $JotsDir]\"" err] {
	TKGError $err
	return
    }
    set JotsFolderList ""
    if {"$filelist" != ""} {
	foreach f $filelist {
	    if [Jotsisfolder $f] {
		lappend JotsFolderList $f
	    }
	}
    }
    if {$JotsFolderList == ""} {
	JotsCreateFolder Notes
	set JotsFolder Notes
	set JotsFolderList Notes
    }
    JotsGetHots
}

proc JotsGetHots {} {
    global JotsHotFile JotsHotList JotsFolderList JotsFolder
    global Jots_showhotlist
    if ![file exists $JotsHotFile] {
	set JotsHotList ""
	if {[lsearch $JotsFolderList $JotsFolder] == -1 } {
	    if {$JotsFolder == "" } {
		set JotsFolder [lindex $JotsFolderList 0]
	    }
	}
    } else {
	source $JotsHotFile
	set sanelist ""
	foreach folder $JotsHotList {
	    if {[lsearch $JotsFolderList $folder] != -1} {
		lappend sanelist $folder
	    }
	}
	set JotsHotList $sanelist
	if {[lsearch $JotsFolderList $JotsFolder] == -1 } {
	    if {[set JotsFolder [lindex $JotsHotList 0]] == ""} {
		set JotsFolder [lindex $JotsFolderList 0]
	    }
	}
    }
    if {[llength $JotsHotList] > 0} { setifunset Jots_showhotlist 1 }
    .jots.menu.hotlist.m add checkbutton -label "Show hot list" -variable Jots_showhotlist
    trace variable Jots_showhotlist w JotsUpdateHots
    JotsFinishInit
}

proc JotsFinishInit {} {
    global Jots_startwithnew JotsFolder
    JotsUpdateHots
    JotsPackFind
    JotsGet
    JotsDisplay
    if $Jots_startwithnew JotsNewEntry
    trace variable JotsFolder w JotsUpdateHots
    TKGCenter .jots
}

proc JotsSaveHots {} {
    global JotsHotFile JotsHotList JotsFolder
    if [catch "set id [ open $JotsHotFile w ]" err] {
	error $err
	return 0
    }
    puts $id "set JotsHotList \{$JotsHotList\}"
    puts $id "setifunset JotsFolder $JotsFolder"
    close $id
    return 1
}    

proc Jotsisfolder {f} {
    if ![file readable $f] {
	TKGError "Error:  $f isn't readable."
	return
    }
    if ![file writable $f] {
	TKGError "Error:  $f isn't writable."
	return
    }
    set id [open $f]
    gets $id s
    close $id
    if {$s != "\#Jots Folder"} {
	return 0
    } else {
	return 1 
    }
}

proc JotsCreateFolder {f} {
    global JotsFolder
    if [file exists $f] {
	TKGError "can't create $f: folder exists"
	return
    }
    if [catch "exec touch $f"] {
	TKGError "can't create $f."
	return
    }
    set JotsFolder $f
    set id [ open $f w ]
    puts $id "\#Jots Folder"
    puts $id "set JotsList \{\}"
    close $id
    JotsGet
}

proc JotsOpen {} {
    global JotsFolder jots_index JotsList images
    if ![select f "Open folder:"] {
	return
    }
    if ![file exists $f] {
	TKGDialog jotsopen \
	    -image $images/question\
	    -title Jots: \
	    -message "Folder \"$f\" doesn't exist."\
	    -buttons [subst {
		{ create "Create it" {JotsCreateFolder $f;destroy .jotsopen} }
	    }]
	return
    }
    if ![file readable $f] {
	TKGError "$f isn't readable."
	return
    }
    set JotsFolder $f
    JotsGet
}

proc JotsNew {} {
    global JotsFolder jots_index 
    if ![select f "New folder name:"] {
	return
    }
    JotsCreateFolder $f
}

proc JotsDeleteFolder {} {
    global JotsFolder images
    TKGDialog jotsdf \
	-title Jots:\
	-image $images/question\
	-message "Delete folder \"$JotsFolder\"?"\
	-nodismiss \
	-buttons {
	    { yes Yes {
		exec /bin/rm $JotsFolder
		JotsDirInit
		destroy .jotsdf
	    } }
	    { no Cancel {
		destroy .jotsdf
	    } }
	}
}

proc JotsSave {{f ""}} {
    global JotsFolder JotsList
    if {$f != ""} {set JotsFolder $f}
    if {$JotsFolder == ""} {
	if [JotsSaveAs] { return 1 } else { return 0 }
    }
    set id [ open $JotsFolder w ]
    puts $id "\#Jots Folder"
    Jots_update_list
    puts $id "set JotsList \{$JotsList\}"
    close $id
    return 1
}

proc JotsSaveAs {} {
    global JotsFolder
    if ![select f "Folder name:"] {
	return 0
    }
    if [file exists $f] {
	TKGError "can't create $f: folder exists"
	return 0
    }
    if [catch "exec touch $f"] {
	TKGError "can't create $f."
	return 0
    }
    set JotsFolder $f
    JotsSave
    return 1
}

proc JotsSaveAscii {} {
    global JotsFolder JotsList JotsOldCWD JotsDir Jots_selectfolders
    Jots_update_list
    cd $JotsOldCWD
    set Jots_selectfolders 0
    if ![select f "Name for Ascii File:"] {
	return 0
    }
    if [file exists $f] {
	TKGError "can't create $f: file exists"
	return 0
    }
    if [catch "exec touch $f"] {
	TKGError "can't create $f."
	return 0
    }
    set id [ open $f w ]
    puts $id "Jots Folder \"$JotsFolder\" as of [exec date]\n"
    foreach entry $JotsList {
	puts $id "\n-------\n\nDATE: [lindex $entry 0]  TIME: [lindex $entry 1]"
	puts $id [JotsWrap [lindex $entry 2]]
    }
    close $id
    cd $JotsDir
    return 1  
}

proc JotsWrap {text} {
    global Jots_columns
    set linelist [split $text "\n"]
    set outtext ""
    foreach line $linelist {
	set s ""
	set wordlist [split $line]
	foreach word $wordlist {
	    if {$s == ""} {
		set s $word 
	    } elseif {[string length "$s $word"] <= $Jots_columns} {
		set s "$s $word"
	    } else {
		set outtext "$outtext\n$s"
		set s $word
	    }
	}
	if {$s != ""} { set outtext "$outtext\n$s" }
    }
    return $outtext
}

proc JotsClear {} {
    global JotsList
    set JotsList {}
}

proc JotsGet {} {
    global JotsFolder JotsList jots_index
    source $JotsFolder
    set jots_index [expr [llength $JotsList] - 1]
    JotsDisplay
}

##########################
# General display, maneuvering, and deleting routines
#

proc JotsDisplayFolder {} {
    global Jots_startwithnew JotsList jots_index
    JotsClear
    JotsGet
    set jots_index [expr [llength $JotsList] - 1]
    if $Jots_startwithnew JotsNewEntry else JotsDisplay
}

proc JotsScroll {cmd {num ""} {unit ""}} {
    global JotsList jots_index JotsSB
    switch -- $cmd {
	moveto {
	    Jots_update_list
	    if {$num > 1} {set num .999}
	    if {$num < 0} {set num 0}
	    set di [expr int( ([llength $JotsList] * $num)) ]
	    if {$di != $jots_index} {
		set jots_index $di
		JotsDisplay
	    }
	} scroll {
	    switch -- $num {
		-1 { 
		    JotsPrev
		} 1 {
		    JotsNext
		}
	    }
	} default {
	    return
	}
    }
}

proc JotsSUpdate args {
    global JotsSB jots_index JotsList
    set w [expr 1/([llength $JotsList] + 0.00001)]
    $JotsSB set [expr $w * $jots_index]  [expr $w * ($jots_index + 1)] 
}

proc Jots_update_list {} {
    global JotsList jots_index jots_time jots_date jots_edited
    global jots_findmessage textwindow
    set jots_findmessage ""
    if {($JotsList != {}) && ($jots_edited != -1)} {
	set text [$textwindow get 1.0 end]
	set JotsList [lreplace $JotsList $jots_edited $jots_edited \
			   [list $jots_date $jots_time \
				$text ]]
    }
}

proc JotsDisplay args {
    global JotsList jots_date jots_time jots_index jots_edited
    global textwindow
    set jots_date ""
    set jots_time ""
    set text ""
    $textwindow configure -state normal
    $textwindow delete 1.0 end
    if { $JotsList != "" } {
	set entry [lindex $JotsList $jots_index]
	set jots_date [lindex $entry 0]
	set jots_time [lindex $entry 1]
	set text [lindex $entry 2]
	set state normal
    } else { 
	set text "\n\n\n\n\n                           (No entries in folder.)"
	set state disabled
    }
    $textwindow insert 1.0 $text
    $textwindow configure -state $state
    set jots_edited -1
}

proc JotsNewEntry {} {
    global jots_time jots_date jots_index JotsList jots_edited
    global textwindow
    Jots_update_list
    set timedate [ exec date "+%H %M %a %h %d %Y"] 
    scan $timedate {%s %s %s %s %s %s} hour minute weekday month day year
    lappend JotsList [list \
			    "$weekday, $month [Clock_zerotrim $day], $year"\
			    "[Clock_zerotrim [set hour]]:[set minute]"\
			    ""\
		       ]
    set jots_index [expr [llength $JotsList] - 1]
    JotsDisplay
    focus $textwindow
}

proc JotsDelete {} {
    global jots_index JotsList    
    Jots_update_list
    if {$JotsList == ""} { return 1 }
    set JotsList [lreplace $JotsList $jots_index $jots_index]
    if {[llength $JotsList] == 0} {
	set jots_index -1
    } elseif {$jots_index == [llength $JotsList]} {
	incr jots_index -1
    }
    JotsDisplay
    JotsSUpdate
}

proc JotsPackFind args {
    global Jots_showfindtool
    set w .jots
    switch $Jots_showfindtool {
	0 {
	    catch "destroy $w.find"
	} 1 {
	    if [winfo exists $w.find] return
	    frame $w.find 
	    button $w.find.find -command JotsFind  -text "Find"
	    pack $w.find.find -side left -expand y -fill y -padx .4c
	    entry $w.find.entry -textvariable jots_find -width 30
	    pack $w.find.entry -side left -fill x -padx .3c 
	    bind $w.find.entry <Key-Return> JotsFind
	    label $w.find.label -textvariable jots_findmessage -width 15
	    pack $w.find.label -side left -fill x -padx .3c 
	    pack $w.find -after .jots.scroll
	    update
	}
    }
}

proc JotsFind {} {
    global JotsList jots_index jots_find jots_findmessage Jots_findmode
    Jots_update_list
    if {$jots_find == ""} return
    set l ""
    for {set i 0} {$i < [llength $JotsList]} {incr i} {
	lappend l [lindex [lindex $JotsList $i] 2]
    }
    if {[set i [expr ($jots_index + 1 + [lsearch \
					     [lrange $l [expr $jots_index + 1] end]\
					     *$jots_find*])]] == $jots_index} {
	if {[set i [lsearch $l *$jots_find* ]] == -1} {
	    set jots_findmessage "Not found."
	    return
	}
    }
    set jots_index $i
    JotsDisplay
    set jots_findmessage "Found."
}

proc JotsPrev {} {
    global jots_index textwindow
    Jots_update_list
    if { $jots_index == 0 } {return 1}
    incr jots_index -1
    JotsDisplay
    focus $textwindow
}

proc JotsNext {} {
    global jots_index JotsList textwindow
    Jots_update_list
    if { $jots_index == ([llength $JotsList] - 1 )} {return 1}
    incr jots_index
    JotsDisplay
    focus $textwindow
}

proc JotsDismiss {} {
    Jots_update_list
    if {[JotsSave] && [JotsSaveHots]} {
	JotsExit
    } else {
	error "Can't save; use command in \"File\" menu to exit anyway."
    }
}

proc JotsExit {} {
    global JotsOldCWD
    cd $JotsOldCWD
    trace vdelete Jots_showfindtool w JotsPackFind
    trace vdelete jots_index w JotsSUpdate
    trace vdelete Jots_showhotlist w JotsUpdateHots
    trace vdelete JotsFolder w JotsUpdateHots

    foreach v { JotsList JotsFolder textwindow JotsOldCWD 
	JotsHotsWindow JotsHotList jots_time jots_date jots_index 
	jots_edited JotsSB Jots_selectfolders
	fileselect selection} {
	global $v
	catch "unset $v"
    }
    destroy .jots
}

#########################
# Folder Management Stuff
#

proc JotsUpdateHots args {
    global JotsFolder JotsHotList JotsHotsWindow
    global JotsHotsCBG JotsHotsCFG JotsHotsCABG JotsHotsCAFG
    global jots_findmessage Jots_showhotlist
    set jots_findmessage ""
    pack forget $JotsHotsWindow
    if !$Jots_showhotlist {return}
    foreach w [winfo children $JotsHotsWindow] {destroy $w}
    frame $JotsHotsWindow.pad
    pack $JotsHotsWindow.pad -side left -expand y -fill both
    foreach folder $JotsHotList {
	set w $JotsHotsWindow.[string tolower $folder]
	button $w -text $folder -command "JotsGotoFolder $folder"
	if {$folder == $JotsFolder} {
	    $w configure -background $JotsHotsCBG -foreground $JotsHotsCFG\
		-activebackground $JotsHotsCABG -activeforeground $JotsHotsCAFG\
		-command {}
	}
	pack $w -side left
    }
    pack $JotsHotsWindow -after .jots.menu\
	-side top -padx .1c -pady .1c -padx .3c -fill x -expand y
}

proc JotsGotoFolder {folder} {
    global JotsFolder
    JotsSave
    set JotsFolder $folder
    JotsGet
    JotsDisplay
    JotsUpdateHots
}

proc JotsAddHot {} {
    global JotsFolder JotsHotList
    if {[lsearch $JotsHotList $JotsFolder] != -1} return
    lappend JotsHotList $JotsFolder
    set JotsHotList [lsort $JotsHotList]
    JotsUpdateHots
}

proc JotsRemoveHot {} {
    global JotsFolder JotsHotList
    if {[set i [lsearch $JotsHotList $JotsFolder]] == -1} return
    set JotsHotList [lreplace $JotsHotList $i $i]
    JotsUpdateHots
}
    

#########################
# Help menu stuff
#

proc JotsAbout {} {

    set text "
Jots, by Mark Crimmins (markcrim@umich.edu), copyright 1995.\n\
Look for the latest version in the tkgoodstuff distribution at:\n\
\ \ \ ftp://merv.philosophy.lsa.umich.edu/pub
"
    set w .about
    catch {destroy $w}
    toplevel $w
    wm title $w "About Jots"
    wm iconname $w "About Jots"

    label $w.title -text "About Jots" 
    pack $w.title -pady .3c -fill x -expand y

    frame $w.view -relief groove -borderwidth 5
    message $w.view.text -width 18c -text $text

    pack $w.view.text -side left -fill both -expand 1
    pack $w.view -side top -fill both -expand 1 -padx .3c

    frame $w.buttons
    button $w.buttons.dismiss -text Dismiss -command "destroy $w"
    pack $w.buttons.dismiss -side left -expand y -fill y -padx .4c -pady .4c
    pack $w.buttons -side bottom -fill x -expand y

}    

proc JotsHelp {} {

    set text "

Jots looks for jots folders in the directory named in the variable\
JotsDir (or, by default, the directory \".tkgjots\" in the user's home\
directory).  It also looks there for the file \".hotjots\" which\
contains the \"hotlist\" and the last-visited folder name.  The first\
time the user runs Jots, this directory and file (and an intial\
folder called \"Notes\") are created.  

The first folder visited will\
be the folder named in the variable JotsFolder, or if that variable is\
not set the folder last visited (according to the .hotjots file), or\
if there is no .hotjots file the alphabetically first folder in the\
directory.

At a given time there is one entry of the current folder on the\
screen.  Scroll through it with the vertical scrollbar, or scroll\
through the entries in the folder with the large, horizontal\
scrollbar.

The hotlist consists of buttons in the upper right of the Jots window\
for visiting folders.  Clicking a hotlist button saves the current\
folder and opens the folder selected.  This list is displayed by\
default when Jots is started if the hotlist contains any folders.\
Otherwise, one can display it manually with a command under the\
\"Hotlist\" menu.  When visiting a folder you want in the hotlist,\
other command under the \"Hotlist\" menu will add and delete that\
folder from the hotlist.

Creating, opening, and saving folders is done with commands under the\
\"File\" menu. 

To search for an expression in a folder, use the command under the\
\"Find\" menu to display the Find tool, type in the phrase and press\
the Enter key or the \"Find:\" button.  Searching is case-sensitive,\
and the search\
string may use \"glob\" devices (see the tcl documentation for\
\"string match\" (under \"man -n string\") for more.  There is no\
provision for searching multiple folders at once.

The following keyboard shortcuts are defined:
     <Alt-q> or <Alt-x>	Save and Exit
     <Shift-Alt-LeftArrow>	Previous Entry
     <Shift-Alt-RightArrow>	Next Entry
     <Shift-Alt-d>		Delete Entry
     <Shift-Alt-n>		New Entry

The rest should be self-explanatory.  See the tkgoodstuff\
documentation for information about customization.


"

    set w .loginscripthelp
    catch {destroy $w}
    toplevel $w
    wm title $w "Jots Help"
    wm iconname $w "Jots Help"

    label $w.title -text "JOTS HELP"
    pack $w.title -pady .3c -fill x -expand y

    frame $w.view
    text $w.view.text -width 65 -height 20 \
	-takefocus 0 -yscrollcommand "$w.view.scrollbar set" \
	-relief sunken -borderwidth 2 -state disabled \
        -wrap word

    pack $w.view.text -side left -fill both -expand 1
    scrollbar $w.view.scrollbar -command "$w.view.text yview"
    pack $w.view.scrollbar -side left -fill y -padx 3
    pack $w.view -side top -fill both -expand 1 -padx .3c
    $w.view.text configure -state normal
    $w.view.text insert end "$text"
    $w.view.text configure -state disabled

    frame $w.buttons
    button $w.buttons.dismiss -text Dismiss -command "destroy $w"
    pack $w.buttons.dismiss -side left -expand y -fill y -padx .4c -pady .4c
    pack $w.buttons -side bottom -fill x -expand y

}    


# File selection stuff

proc select {var {message "Open File"}} {
    global selection
    set selection ""
    fileselect selectdone $message
    tkwait window .fileSelectWindow
    uplevel "set $var \"$selection\"" 
    if { $selection == "" } {
	return 0
    } else {
	return 1
    }
}

proc selectdone {f} {
    global selection 
    set selection $f
}

#
# fileselect.tcl --
# simple file selector.
#
# Mario Jorge Silva			          msilva@cs.Berkeley.EDU
# University of California Berkeley                 Ph:    +1(510)642-8248
# Computer Science Division, 571 Evans Hall         Fax:   +1(510)642-5775
# Berkeley CA 94720                                 
# 
# Layout:
#
#  file:                  +----+
#  ____________________   | OK |
#                         +----+
#
#  +------------------+    Cancel
#  | ..               |S
#  | file1            |c
#  | file2            |r
#  |                  |b
#  | filen            |a
#  |                  |r
#  +------------------+
#  currrent-directory
#
# Copyright 1993 Regents of the University of California
# Permission to use, copy, modify, and distribute this
# software and its documentation for any purpose and without
# fee is hereby granted, provided that this copyright
# notice appears in all copies.  The University of California
# makes no representations about the suitability of this
# software for any purpose.  It is provided "as is" without
# express or implied warranty.
#


# names starting with "fileselect" are reserved by this module
# no other names used.

# use the "option" command for further configuration



# this is the default proc  called when "OK" is pressed
# to indicate yours, give it as the first arg to "fileselect"

proc fileselect.default.cmd {f} {
  puts stderr "selected file $f"
}


# this is the default proc called when error is detected
# indicate your own pro as an argument to fileselect

proc fileselect.default.errorHandler {errorMessage} {
    puts stdout "error: $errorMessage"
    catch { cd ~ }
}

# this is the proc that creates the file selector box

proc fileselect {
    {cmd fileselect.default.cmd} 
    {purpose "Open file:"} 
    {w .fileSelectWindow} 
    {errorHandler fileselect.default.errorHandler}} {

    catch {destroy $w}

    toplevel $w
    wm withdraw $w
    grab $w
    wm title $w "Select File"


    # path independent names for the widgets
    global fileselect

    set fileselect(entry) $w.file.eframe.entry
    set fileselect(list) $w.file.sframe.list
    set fileselect(scroll) $w.file.sframe.scroll
    set fileselect(ok) $w.bframe.okframe.ok
    set fileselect(cancel) $w.bframe.cancel
    set fileselect(dirlabel) $w.file.dir.dirlabel

    # widgets
    frame $w.file -bd 10 
    frame $w.bframe -bd 10
    pack append $w \
        $w.file {left filly} \
        $w.bframe {left expand frame n}

    frame $w.file.eframe
    frame $w.file.sframe
    frame $w.file.dir
    label $w.file.dir.dir  -anchor e -width 6 -text "Dir:"
    label $w.file.dir.dirlabel -anchor e -width 24 -text [pwd] 
    pack $w.file.dir.dir $w.file.dir.dirlabel -side left

    pack append $w.file \
        $w.file.eframe {top frame w} \
	$w.file.sframe {top fillx} \
	$w.file.dir {top frame w}


    label $w.file.eframe.label -anchor w -width 24 -text $purpose
    entry $w.file.eframe.entry -relief sunken 

    pack append $w.file.eframe \
		$w.file.eframe.label {top expand frame w} \
                $w.file.eframe.entry {top fillx frame w} 


    scrollbar $w.file.sframe.yscroll -relief sunken \
	 -command "$w.file.sframe.list yview"
    listbox $w.file.sframe.list -relief sunken \
	-yscroll "$w.file.sframe.yscroll set" 

    pack append $w.file.sframe \
        $w.file.sframe.yscroll {right filly} \
 	$w.file.sframe.list {left expand fill} 

    # buttons
    frame $w.bframe.okframe -borderwidth 2 -relief sunken
 
    button $w.bframe.okframe.ok -text OK -relief raised -padx .3c \
        -command "fileselect.ok.cmd $w $cmd $errorHandler"

    button $w.bframe.cancel -text cancel -relief raised -padx .3c \
        -command "fileselect.cancel.cmd $w"
    pack append $w.bframe.okframe $w.bframe.okframe.ok {padx .3c pady .3c}

    pack append $w.bframe $w.bframe.okframe {expand padx 20 pady 20}\
                          $w.bframe.cancel {top}

    # Fill the listbox with a list of the files in the directory (run
    # the "/bin/ls" command to get that information).
    # to not display the "." files, remove the -a option and fileselect
    # will still work
 
    global Jots_selectfolders
    if $Jots_selectfolders {
	foreach i [exec /bin/ls [pwd]] {
	    if {[string compare $i "."] != 0 && \
		    [string compare $i ".."] != 0 } {
		$fileselect(list) insert end $i
	    }
	}
    } else {
	$fileselect(list) insert end ".."
	foreach i [exec /bin/ls -a [pwd]] {
	    if {[string compare $i "."] != 0 && \
		    [string compare $i ".."] != 0 } {
		$fileselect(list) insert end $i
	    }
	}
    }

   # Set up bindings for the browser.
    bind $fileselect(entry) <Return> {eval $fileselect(ok) invoke; break}
    bind $fileselect(entry) <Control-c> {eval $fileselect(cancel) invoke; break}

    bind $w <Control-c> {eval $fileselect(cancel) invoke;break}
    bind $w <Return> {eval $fileselect(ok) invoke;break}


#    tk_listboxSingleSelect $fileselect(list)


    bind $fileselect(list) <Button-1> {
        # puts stderr "button 1 release"
        %W selection set [%W nearest %y]
	$fileselect(entry) delete 0 end
	$fileselect(entry) insert 0 [%W get [%W nearest %y]]
	break
    }

    bind $fileselect(list) <Key> {
        %W selection set [%W nearest %y]
        $fileselect(entry) delete 0 end
	$fileselect(entry) insert 0 [%W get [%W nearest %y]]
	break
    }

    bind $fileselect(list) <Double-ButtonPress-1> {
        # puts stderr "double button 1"
        %W selection set [%W nearest %y]
	$fileselect(entry) delete 0 end
	$fileselect(entry) insert 0 [%W get [%W nearest %y]]
	$fileselect(ok) invoke
	break
    }

    bind $fileselect(list) <Return> {
        %W selection set [%W nearest %y]
	$fileselect(entry) delete 0 end
	$fileselect(entry) insert 0 [%W get [%W nearest %y]]
	$fileselect(ok) invoke
	break
    }

    # set kbd focus to entry widget

    focus $fileselect(entry)

    #center window
    TKGCenter $w
}


# auxiliary button procedures

proc fileselect.cancel.cmd {w} {
    # puts stderr "Cancel"
    destroy $w
}

proc fileselect.ok.cmd {w cmd errorHandler} {
    global fileselect Jots_selectfolders
    set selected [$fileselect(entry) get]

    # some nasty file names may cause "file isdirectory" to return an error
    set sts [catch { 
	file isdirectory $selected
    }  errorMessage ]

    if { $sts != 0 } then {
	$errorHandler $errorMessage
	destroy $w
	return

    }

    # clean the text entry and prepare the list
    $fileselect(entry) delete 0 end
    $fileselect(list) delete 0 end
    if !$Jots_selectfolders {$fileselect(list) insert end ".."}

    # selection may be a directory. Expand it.

    if {[file isdirectory $selected] != 0} {
	if !$Jots_selectfolders {
	    cd $selected
	    set dir [pwd]
	    $fileselect(dirlabel) configure -text $dir
	    foreach i [exec /bin/ls -a $dir] {
		if {[string compare $i "."] != 0 && \
			[string compare $i ".."] != 0} {
		    $fileselect(list) insert end $i
		}
	    }
	}
	return
    }

    destroy $w
    $cmd $selected
}
##### end of fileselect code


DEBUG "Loaded Jots.tcl"
