#
# $Id: util.tcl,v 1.11 1995/03/21 03:50:27 sls Exp $
#
# Various shorthand utility stuff
#
# This software is copyright (C) 1995 by the Lawrence Berkeley Laboratory.
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that: (1) source code distributions
# retain the above copyright notice and this paragraph in its entirety, (2)
# distributions including binary code include the above copyright notice and
# this paragraph in its entirety in the documentation or other materials
# provided with the distribution, and (3) all advertising materials mentioning
# features or use of this software display the following acknowledgement:
# ``This product includes software developed by the University of California,
# Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
# the University nor the names of its contributors may be used to endorse
# or promote products derived from this software without specific prior
# written permission.
# 
# THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#

document_title util "various utility procs" {
    These procedures fill some gaps in the core Tcl command set.
}

set util(verbose) 0

document_proc read_file {
    returns the contents of `file'.
}
proc read_file {file} {
    set fp [open $file r]
    set txt [read $fp]
    close $fp
    return $txt
}

document_proc write_file {
    writes `txt' to `file'.
}
proc write_file {file txt} {
    set fp [open $file w]
    puts $fp $txt
    close $fp
}

document_proc foreach_line {
    sets `var' to each line of `file', and executes `action'.
}
proc foreach_line {var file action} {
    uplevel [list foreach $var [split [read_file $file] "\n"] $action]
}

document_proc ifexists {
    executes `action' if the variable `var' exists, or `else_clause'
    if it doesn't.  `else_keyword' must be the word #else#.  `else_clause'
    is optional.
}
proc ifexists {var action {else_keyword ""} {else_clause ""}} {
    upvar $var v
    if [info exists v] {
	uplevel $action
	return
    }
    if {$else_keyword == "else"} {
	uplevel $else_clause
    }
}

document_proc ifnexists {
    executes `action' if the variable `var' does not exist, or `else_clause'
    if it does.  `else_keyword' must be the word #else#.  `else_clause'
    is optional.
}
proc ifnexists {var action {else_keyword ""} {else_clause ""}} {
    upvar $var v
    if ![info exists v] {
	uplevel $action
    }
    if {$else_keyword == "else"} {
	uplevel $else_clause
    }
}

document_proc run {
    executes the last element of `args'.  If any errors occur during
    execution, then #run# prints #errorInfo# on #stderr# and exits with
    status code 1.  Otherwise it exists with status code 0.
    If the #-terse# flag appears in `args', then only print the
    result of the command that caused the error.
}
proc run {args} {
    set end [expr [llength $args] - 1]
    set body [lindex $args $end]
    args [lrange $args 0 [expr $end-1]] {
	{-terse 0}
    }
    if {[catch {uplevel $body} error] == 1} {
	if [info exists terse] {
	    puts stderr $error
	} else {
	    global errorInfo
	    puts stderr $errorInfo
	}
	exit 1
    }
    exit 0
}

document_proc args {
    parses `argv', using entries from the list `tbl'.  Each entry
    is a list of the form {-`flag' ?`arity'? ?#list#?}.  For each
    option that #args# finds, it sets the variable `prefix'`flag' to
    the `arity' next arguments in `argv'.  `arity' defaults
    to 1 and `prefix' defaults to "".
    If `arity' is 0, then #args# sets the variable to 1.
    If the keyword #list# is present, #args# uses #lappend#
    to accumulate multipe instances of the flag.
}
proc args {argv tbl {prefix ""}} {
    foreach spec $tbl {
        set flag [lindex $spec 0]
        upvar $prefix[string range $flag 1 end] $flag
        set arity($flag) [lindex $spec 1]
	if ![string length $arity($flag)] {
	    set arity($flag) 1
	}
	set islist($flag) [lindex $spec 2]
    }
    while {[llength $argv]} {
        set arg [lindex $argv 0]
        if [info exists arity($arg)] {
            set n $arity($arg)
            if {$n == 0} {
                set val 1
            } elseif {$n == 1} {
                set val [lindex $argv 1]
            } else {
                set val [lrange $argv 1 $n]
            }
	    if {$islist($arg) != ""} {
		lappend $arg $val
	    } else {
		set $arg $val
	    }
        } else {
            error "unknown argument $arg, should be one of: $tbl"
        }
        set argv [lrange $argv [expr 1 + $n] end]
    }
}

document_proc iswhite {
    returns 1 if `s' is composed entirely of whitespace, or is empty.
}
proc iswhite {s} {
    return [regexp "^( \t\n)*$" $s]
}

document_proc lassign {
    assigns the n'th variable in the list `vars' to the n'th element
    of `list'.
}
proc lassign {vars list} {
    set i 0
    foreach elt $list {
        upvar [lindex $vars $i] var
        set var $elt
        incr i
    }
}

document_proc vtime {
    executes `expression', printing (with #msg#) the time taken in seconds.
    Returns the result of `expression'.
}
proc vtime expression {
    set t [lindex [time {set result [uplevel $expression]}] 0]
    msg "[expr $t/1.e6] sec for \"$expression\""
    return $result
}

document_proc capitalize {
    capitalizes the first character of `s'.
}
proc capitalize s {
    return "[string toupper [string index $s 0]][string range $s 1 end]"
}

document_proc word_wrap {
    returns `string' wrapped on word boundaries.  Uses `columns' as
    the maximum number of columns (defaults to 70).
}
proc word_wrap {string {columns 70}} {
    regsub -all "\[ \n\t\]+" $string " " string
    set c 0
    set result ""
    foreach word [split $string " "] {
	set len [expr [string length $word] + 1]
	if {$c + $len > $columns} {
	    append result "\n"
	    set c 0
	}
	append result $word
	append result " "
	set c [expr $c + $len]
    }
    return $result
}

document_proc lappend_unique {
    #lappend#'s `val' to `var' if `val' is not an element of `var'.
    (This proc has runtime O(length of the list), whereas #lappend#
    runs in constant time.)
}
proc lappend_unique {var val} {
    upvar $var l
    if [info exists l] {
        if [lsearch -exact $l $val]<0 {
            lappend l $val
        }
    } else {
        lappend l $val
    }
}
