#
# $Id: object.tcl,v 1.11 1995/07/26 21:41:25 sls Exp $
#
# This software is copyright (C) 1995 by the Lawrence Berkeley Laboratory.
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that: (1) source code distributions
# retain the above copyright notice and this paragraph in its entirety, (2)
# distributions including binary code include the above copyright notice and
# this paragraph in its entirety in the documentation or other materials
# provided with the distribution, and (3) all advertising materials mentioning
# features or use of this software display the following acknowledgement:
# ``This product includes software developed by the University of California,
# Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
# the University nor the names of its contributors may be used to endorse
# or promote products derived from this software without specific prior
# written permission.
# 
# THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#

document_title object "a simple object/widget system" {

    This is a simple object and Tk widget system.  An (instantiation
    of an) object consists of a set of procedures (the objects
    methods) and an array (the objects slots).  An object has two
    kinds of slots, #param# and #member#.  #param#'s have a resource
    class and can be accessed through the object's #config# method.  A
    widget is an object and a Tk window.

    The object system defines a couple of object slots for private
    use, all starting with #__# (two underscores).  The only
    documented one is #__class#, which is the class of the object.
}

set object_priv(currentClass) {}
set object_priv(objectCounter) 0

document_proc object_class {
    defines a new class of objects.  #object_class# works by setting
    up an empty class named `name', evaluating `spec', and then
    creating a #config# method.  `spec' is typically a sequence of
    of #member#, #param#, #method#, #object_class_var#, and
    #object_class_method# commands.
}
proc object_class {name spec} {
    global object_priv
    set object_priv(currentClass) $name
    lappend object_priv(objects) $name
    upvar #0 ${name}_priv class
    set class(__members) {}
    set class(__methods) {}
    set class(__params) {}
    set class(__class_vars) {}
    set class(__class_methods) {}
    uplevel $spec
    proc $name:config args "uplevel \[concat object_config \$args]"
    proc $name:configure args "uplevel \[concat object_config \$args]"
    proc $name:cget {self option} "uplevel \[list object_cget \$self \$option]"
}

document_proc method {
    defines an object method `name'.  The slots of the object can
    be accessed through the array #slot# (for example, the member
    `member' is $#slot#(`member')).  The name of the object is
    in the variable #self#.  The class variables of the object can
    be accessed through the array #class_var# (the class variable
    `foo' is $#class_var#(`foo')).
}
proc method {name args body} {
    global object_priv
    set className $object_priv(currentClass)
    upvar #0 ${className}_priv class
    if {[lsearch $class(__methods) $name] < 0} {
	lappend class(__methods) $name
    }
    set methodArgs self
    append methodArgs " " $args
    proc $className:$name $methodArgs "upvar #0 \$self slot ${className}_priv class_var\n$body"
}
document_proc object_method {
    is the same as #method#.
}
proc object_method {name {defaultValue {}}} [info body method]


document_proc member {
    defines an object slot `name'.  When the object is created,
    the slot will be set to `defaultValue'.  `defaultValue' defaults
    to {}.
}
proc member {name {defaultValue {}}} {
    global object_priv
    set className $object_priv(currentClass)
    upvar #0 ${className}_priv class
    lappend class(__members) [list $name $defaultValue]
}
document_proc object_member {
    is the same as #member#.
}
proc object_member {name {defaultValue {}}} [info body member]

document_proc param {
    defines an object slot `name'.  When the object is created, the
    slot will be set to the result of #option get $self# `name'
    `resourceClass' (i.e. through the Tk resource mechanism.)
    `resourceClass' defaults to `name' capitalized.  If this fails or
    the result is {}, the slot will be set to `defaultValue'.
    `defaultValue' defaults to {}.  Whenever the slot is changed with
    #config#, `configCode' is run.  `configCode' defaults to {}.
}
proc param {name {defaultValue {}} {resourceClass {}} {configCode {}}} {
    global object_priv
    set className $object_priv(currentClass)
    upvar #0 ${className}_priv class
    if {$resourceClass == ""} {
	set resourceClass \
	    [string toupper [string index $name 0]][string range $name 1 end]
    }
    if ![info exists class(__param_info/$name)] {
	lappend class(__params) $name
    }
    set class(__param_info/$name) [list $defaultValue $resourceClass]
    if {$configCode != {}} {
	proc $className:config:$name self $configCode
    }
}
document_proc object_param {
    is the same as #param#.
}
proc object_param {name {defaultValue {}} {resourceClass {}} {configCode {}}} \
    [info body param]

document_proc object_class_var {
    defines a class variable.  Class variables exist per class and
    accessed through the #class_var# array.  The object system creates
    several class variables, all starting with #__# (two underscores).
    (None are documented.)
}
proc object_class_var {name {initialValue ""}} {
    global object_priv
    set className $object_priv(currentClass)
    upvar #0 ${className}_priv class
    set class($name) $initialValue
    set class(__initial_value.$name) $initialValue
    lappend class(__class_vars) $name
}

document_proc object_class_method {
    defines a proc `class':`name' in which the class variables are
    available in the array `class_var'.
}
proc object_class_method {name args body} {
    global object_priv
    set className $object_priv(currentClass)
    upvar #0 ${className}_priv class
    if {[lsearch $class(__class_methods) $name] < 0} {
	lappend class(__class_methods) $name
    }
    proc $className:$name $args "upvar #0 ${className}_priv class_var\n$body"
}

document_proc object_include {
    includes the definition of `super_class_name' in the current object.
    This has the same effect as repeating all the #param#, #member#,
    and #method# definitions of `super_class_name' in the current object.
    These definitions can be overridden by subsequent #param#, #member#,
    and #method# statements.
}
proc object_include {super_class_name} {
    global object_priv
    set className $object_priv(currentClass)
    upvar #0 ${className}_priv class
    upvar #0 ${super_class_name}_priv super_class
    foreach p $super_class(__params) {
	lappend class(__params) $p
	set class(__param_info/$p) $super_class(__param_info/$p)
    }
    set class(__members) [concat $super_class(__members) $class(__members)]
    set class(__class_vars) \
	[concat $super_class(__class_vars) $class(__class_vars)]
    foreach v $super_class(__class_vars) {
	set class($v) \
	    [set class(__initial_value.$v) $super_class(__initial_value.$v)]
    }
    set class(__class_methods) \
	[concat $super_class(__class_methods) $class(__class_methods)]
    set class(__methods) \
	[concat $super_class(__methods) $class(__methods)]
    foreach m $super_class(__methods) {
	set proc $super_class_name:$m
	proc $className:$m [object_get_formals $proc] [info body $proc]
    }
    foreach m $super_class(__class_methods) {
	set proc $super_class_name:$m
	regexp "^\[^\n\]+\n(.*)" [info body $proc] dummy body
	proc $className:$m [object_get_formals $proc] \
	    "upvar #0 ${className}_priv class_var\n$body"
    }
}

document_proc object_new {
    instantiates an object of class `className'.  The name of the
    object will be `name'.  If `name' is not given, then #object_new#
    generates a name of the form #O_#`N', where `N' is an integer.
    #object_new# starts by creating a global array `name' and filling
    it in with the default values of #member# and #param#'s.  It then
    creates a new proc `name' which can be used to run the methods
    of the object.
}
proc object_new {className {name {}}} {
    if {$name == {}} {
	global object_priv
	set name O_[incr object_priv(objectCounter)]
    }
    upvar #0 $name object
    upvar #0 ${className}_priv class
    set object(__class) $className
    foreach var $class(__params) {
	set info $class(__param_info/$var)
	set resourceClass [lindex $info 1]
	if ![catch {set val [option get $name $var $resourceClass]}] {
	    if {$val == ""} {
		set val [lindex $info 0]
	    }
	} else {
	    set val [lindex $info 0]
	}
	set object($var) $val
    }
    foreach var $class(__members) {
	set object([lindex $var 0]) [lindex $var 1]
    }
    proc $name {method args} [format {
	upvar #0 %s object
	uplevel [concat $object(__class):$method %s $args]
    } $name $name]
    return $name
}

proc object_define_creator {windowType name spec} {
    object_class $name $spec
    if {[info procs $name:create] == {}} {
	error "widget \"$name\" must define a create method"
    }
    if {[info procs $name:reconfig] == {}} {
	error "widget \"$name\" must define a reconfig method"
    }
    proc $name {window args} [format {
	%s $window -class %s
	rename $window object_window_of$window
	upvar #0 $window object
	set object(__window) $window
	object_new %s $window
	proc %s:frame {self args} \
	    "uplevel \[concat object_window_of$window \$args]"
	uplevel [concat $window config $args]
	$window create
	set object(__created) 1
	bind $window <Destroy> \
	    "if !\[string compare %%W $window\] { object_delete $window }"
	$window reconfig
	return $window
    } $windowType \
	  [string toupper [string index $name 0]][string range $name 1 end] \
	  $name $name]
}

document_proc widget {

    defines a new class of widgets that are built with a Tk frame.  It
    works the same as #object_class#, except that it requires `spec'
    to contains definitions for a #create# and #reconfig# method, and
    it creates a proc `name' that creates new widgets.

    This creator proc works like the built-in Tk widget creators.
    It is invoked as `name .window' ?-`parameter' `value'? `...'.  The
    creator creates a new Tk frame `.window' with a class of `name'
    capitalized, creates a new object `.window' with #object_new#,
    creates a method #frame# that passes its args to the Tk frame,
    calls the objects #config# command with the -`parameter' `value'
    arguments, calls the widgets #create# method, sets up a Tk binding
    on <Destroy> that calls #object_delete#, calls the widgets
    #reconfig# method, and (finally!) returns `.window'.

    The upshot of that long story is that the widgets #create# method
    should create (and usually pack) any subwidgets.  The #reconfig#
    method should reconfigure (via a series of `$self.widget'
    #config# commands) the subwidgets.  
}

proc widget {name spec} {
    object_define_creator frame $name $spec
}

document_proc dialog {

    defines a new class of widgets that are built with a Tk toplevel.
    #dialog# is identical to #widget#, except that the creator proc
    uses a Tk toplevel instead of a Tk frame.

}
proc dialog {name spec} {
    object_define_creator toplevel $name $spec
}

proc object_config {self args} {
    upvar #0 $self object
    set len [llength $args]
    if {$len == 0} {
	upvar #0 $object(__class)_priv class
	set result {}
	foreach param $class(__params) {
	    set info $class(__param_info/$param)
	    lappend result \
		[list -$param $param [lindex $info 1] [lindex $info 0] \
		 $object($param)]
	}
	if [info exists object(__window)] {
	    set result [concat $result [object_window_of$object(__window) config]]
	}
	return $result
    }
    if {$len == 1} {
	upvar #0 $object(__class)_priv class
	if {[string index $args 0] != "-"} {
	    error "param '$args' didn't start with dash"
	}
	set param [string range $args 1 end]
	if {[set ndx [lsearch -exact $class(__params) $param]] == -1} {
	    if [info exists object(__window)] {
		return [object_window_of$object(__window) config -$param]
	    }
	    error "no param '$args'"
	}
	set info $class(__param_info/$param)
	return [list -$param $param [lindex $info 1] [lindex $info 0] \
		$object($param)]
    }
    # accumulate commands and eval them later so that no changes will take
    # place if we find an error
    set cmds ""
    while {$args != ""} {
	set fieldId [lindex $args 0]
        if {[string index $fieldId 0] != "-"} {
            error "param '$fieldId' didn't start with dash"
        }
        set fieldId [string range $fieldId 1 end]
        if ![info exists object($fieldId)] {
	    if {[info exists object(__window)]} {
		if [catch [list object_window_of$object(__window) config -$fieldId]] {
		    error "tried to set param '$fieldId' which did not exist."
		} else {
		    lappend cmds \
			[list object_window_of$object(__window) config -$fieldId [lindex $args 1]]
		    set args [lrange $args 2 end]
		    continue
		}
	    }

        }
	if {[llength $args] == 1} {
	    return $object($fieldId)
	} else {
	    lappend cmds [list set object($fieldId) [lindex $args 1]]
	    if {[info procs $object(__class):config:$fieldId] != {}} {
		lappend cmds [list $self config:$fieldId]
	    }
	    set args [lrange $args 2 end]
	}
    }
    foreach cmd $cmds {
	eval $cmd
    }
    if {[info exists object(__created)] && [info procs $object(__class):reconfig] != {}} {
	$self reconfig
    }
}

proc object_cget {self var} {
    upvar #0 $self object
    return [lindex [object_config $self $var] 4]
}

document_proc object_delete {
    deletes the object `self' by running `self' #destroy# (if such a
    method exists), deleting the command `self', unsetting
    the array `self', and destroying the window `self'.
}
proc object_delete self {
    upvar #0 $self object
    if {[info exists object(__class)] && [info commands $object(__class):destroy] != ""} {
	$object(__class):destroy $self
    }
    if [info exists object(__window)] {
	if [string length [info commands object_window_of$self]] {
	    catch {rename $self {}}
	    rename object_window_of$self $self
	}
	destroy $self
    }
    catch {unset object}
}

document_proc object_slotname {
    returns the global name of slot `slot'.
}
proc object_slotname slot {
    upvar self self
    return [set self]($slot)
}

proc object_get_formals {proc} {
    set formals {}
    foreach arg [info args $proc] {
	if [info default $proc $arg def] {
	    lappend formals [list $arg $def]
	} else {
	    lappend formals $arg
	}
    }
    return $formals
}

document_section NOTES {
    This is about my 4th object system written in Tcl, and it really
    is the simplest.  Each (instantiation of an) object is an array,
    and the procs here are just syntactic sugar.  The only complicated
    stuff is the code to handle widgets.
}
