#
# $Id: html.tcl,v 1.9 1995/03/21 03:48:38 sls Exp $
#
# This software is copyright (C) 1995 by the Lawrence Berkeley Laboratory.
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that: (1) source code distributions
# retain the above copyright notice and this paragraph in its entirety, (2)
# distributions including binary code include the above copyright notice and
# this paragraph in its entirety in the documentation or other materials
# provided with the distribution, and (3) all advertising materials mentioning
# features or use of this software display the following acknowledgement:
# ``This product includes software developed by the University of California,
# Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
# the University nor the names of its contributors may be used to endorse
# or promote products derived from this software without specific prior
# written permission.
# 
# THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#
# html support routines
#
# html_priv(fp) -- current output fp
# html_priv(sink) -- proc that takes html_priv(fp) and some text to print
# html_priv(style) -- current style
# html_priv(style_table) -- table of style names and begin/end directives
#

document_title html "print HTML to strings and files" {

    These procedures print HTML to strings and files.

}

set html_priv(fp) stdout
set html_priv(sink) html_file_sink
set html_priv(close) close
set html_priv(flush) flush
set html_priv(style) normal
set html_priv(dnl) 0
set html_priv(style_table) {
    {normal "" ""}
    {bold <B> </B>}
    {strong <STRONG> </STRONG>}
    {em <EM> </EM>}
    {italic <I> </I>}
    {site <SITE> </SITE>}
    {var <VAR> </VAR>}
    {tt <TT> </TT>}
    {code <CODE> </CODE>}
    {samp <SAMP> </SAMP>}
    {kbd <KBD> </KBD>}
}

document_proc html_set_file_sink {
    sets the output of the html commands to print to the file `file'.
}
proc html_set_file_sink {file} {
    global html_priv
    set html_priv(fp) [open $file w]
    set html_priv(sink) html_file_sink
    set html_priv(close) html_file_close
    set html_priv(flush) flush
}

proc html_file_sink {fp txt} {
    global html_priv
    if $html_priv(dnl) {
	puts -nonewline $fp $txt
    } else {
	puts $fp $txt
    }
}

proc html_file_close {fp} {
    if {[string first file $fp] == 0} {
	close $fp
    }
}

document_proc html_set_string_sink {
    set the output of the html commands to append onto the variable
    `string_var'.  `string_var' must be a global variable.
}
proc html_set_string_sink {string_var} {
    global html_priv
    set html_priv(fp) $string_var
    set html_priv(sink) html_string_sink
    set html_priv(close) html_string_close
    set html_priv(flush) html_string_flush
}

proc html_string_sink {fp txt} {
    global html_priv
    upvar #0 $fp string
    append string $txt
    if !$html_priv(dnl) {
	append string \n
    }
}

proc html_string_close {fp} {
}

proc html_string_flush {fp} {
}

document_proc html_dnl {
    disables the newline at after each html command.  Usually newlines
    have no effect on the rendering of HTML, but sometimes they do
    (such as during <PRE> or <LISTING> blocks.)
}
proc html_dnl {} {
    global html_priv
    set html_priv(dnl) 1
}

document_proc html_nl {
    re-enables the newline after each html command.
}
proc html_nl {} {
    global html_priv
    set html_priv(dnl) 0
}

document_proc html_squote {
    quotes `text'.  Currently it quotes "&", "<", and ">".  Returns
    the quoted text.
}
proc html_squote {text} {
    regsub -all "&" $text "\\&amp;" text
    regsub -all "<" $text "\\&lt;" text
    regsub -all ">" $text "\\&gt;" text
    return $text
}

document_proc html_quote {
    prints `text' with the special characters quoted via #html_squote#.
}
proc html_quote {text} {
    global html_priv
    $html_priv(sink) $html_priv(fp) [html_squote $text]
}

document_proc html {
    print `txt' to the current sink.  No quoting is done on `txt'.
}
proc html {txt} {
    global html_priv
    $html_priv(sink) $html_priv(fp) $txt
}

document_proc html_comment {
    embed `txt' in an HTML comment.  `txt' will be be quoted with #html_quote#.
}
proc html_comment {txt} {
    global html_priv
    $html_priv(sink) $html_priv(fp) "<!-- [html_quote $txt]>"
}

proc html_style_lookup {s} {
    global html_priv
    foreach e $html_priv(style_table) {
	if {[lindex $e 0] == $s} {
	    return $e
	}
    }
    error "unknown html style \"$s\""
}

document_proc html_set_style {
    sets the current type style to `s'.  `s' must be one of #normal#,
    #bold#, #strong#, #em#, #italic#, #site#, #var#, #tt#, #code#,
    #samp#, or #kbd#.  #html_set_style# works by printing the ending
    anchor for the current style and then the beginning anchor for
    the new style.
}
proc html_set_style {s} {
    global html_priv
    if {$s != $html_priv(style)} {
	html [lindex [html_style_lookup $html_priv(style)] 2]
	html [lindex [html_style_lookup $s] 1]
	set html_priv(style) $s
    }
}

document_proc html_style {
    prints `txt' in the html style `s', restoring the old style afterwards.
}
proc html_style {s txt} {
    global html_priv
    set old_style $html_priv(style)
    html_set_style $s
    html $txt
    html_set_style $old_style
}

document_proc html_begin {
    begins a new document by printing the HTML headers.  The document
    is given the title `title'.
}
proc html_begin {title} {
    html "<HTML><HEAD>"
    html "<TITLE>$title</TITLE>"
    html "</HEAD><BODY>"
}

document_proc html_end {
    ends the current document, by checking if the command #html_sign#
    exists and calling it if so, then printing the HTML trailers and
    closing the sink.
}
proc html_end {{include_address 1}} {
    global html_priv
    if $include_address {
	if {[info commands html_sign] != ""} {
	    html_sign
	}
    }
    html "</BODY></HTML>"
    $html_priv(close) $html_priv(fp)
}

document_proc html_flush {
    flushes the current sink.
}
proc html_flush {} {
    global html_priv
    $html_priv(flush) $html_priv(fp)
}

document_proc html_heading {
    prints a heading `hdr' at heading level `level'.
}
proc html_heading {hdr {level 1}} {
    html "<H$level>$hdr</H$level>"
}

document_proc html_link {
    prints `string' surrounded by a HTML href link to `link'.
}
proc html_link {link string} {
    html "<a href=$link>$string</a>"
}

document_proc html_run {
    executes `body'.  If there are any errors, then print a HTML
    <LISTING> of the #errorInfo#.  #html_run# is useful inside
    CGI scripts to catch and display errors.
}
proc html_run {body} {
    if {[catch {uplevel $body}] == 1} {
	global errorInfo
	html "<HR><H1>Ooops!</H1>An error occurred in a tcl script:"
	html "<LISTING>"
	html "$errorInfo"
	html "</LISTING>"
	html "</HR>"
    }
}
