#!/bin/sh
#\
exec @tclsh@ "$0" ${1+"$@"}
#
# $Id: document.tcl,v 1.11 1995/08/05 06:52:05 sls Exp $
#
# This software is copyright (C) 1995 by the Lawrence Berkeley Laboratory.
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that: (1) source code distributions
# retain the above copyright notice and this paragraph in its entirety, (2)
# distributions including binary code include the above copyright notice and
# this paragraph in its entirety in the documentation or other materials
# provided with the distribution, and (3) all advertising materials mentioning
# features or use of this software display the following acknowledgement:
# ``This product includes software developed by the University of California,
# Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
# the University nor the names of its contributors may be used to endorse
# or promote products derived from this software without specific prior
# written permission.
# 
# THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#
# Produce man pages.
#

if [file isdir @stl_library@] {
    lappend auto_path @stl_library@
    set stl_library @stl_library@
} else {
    lappend auto_path .
    set stl_library .
}

proc massage text {
    set text [string trim $text]
    regsub -all ` $text "\\fI" text
    regsub -all ' $text "\\fR" text
    regsub -all "#(\[^#\]*)#" $text "\\fB\\1\\fR" text
    regsub -all "\n\n" $text "\n.LP\n" text
    regsub -all "\[ \t\]+" $text " " text
    regsub -all "\n *" $text "\n" text
    return $text
}

proc document_proc {proc text} {
    global document
    if {![info exists document(current_file)]} return
    if {$document(current_file) != [info script]} return
    if [info exists text_of.$proc] return
    lappend document(procs) $proc
    set document(text_of.$proc) [massage $text]
}

proc document_title {name summary description} {
    global document
    if {![info exists document(current_file)]} return
    if {$document(current_file) != [info script]} return
    set document(name) $name
    set document(summary) $summary
    set document(description) [massage $description]
    set document(is_program) 0
}

proc document_program {name summary description} {
    global document
    if {![info exists document(current_file)]} return
    if {$document(current_file) != [info script]} return
    document_title $name $summary $description
    set document(is_program) 1
    # look for args_parse
    set text [read_file $document(current_file)]
    if [regexp -indices "\[ \n\t\r\]+args_parse\[ \n\t\r\]+\\\$argv\[ \n\t\r\]+" $text match] {
	set text [string range $text [lindex $match 0] end]
	auto_load args_parse
	proc args_parse {argv body} {
	    uplevel $body
	}
	proc args_arg {flag formals description body} {
	    global document
	    lappend document(flags) $flag
	    set document(formals_of.$flag) $formals
	    set document(description_of.$flag) $description
	}
	proc args_remaining {description body} {
	    global document
	    set document(remaining_description) $description
	}
	set argv ""
	eval [lrange $text 0 2]
    }
}

proc document_section {name text} {
    global document
    if {![info exists document(current_file)]} return
    if {$document(current_file) != [info script]} return
    lappend document(sections) $name
    set document(section.$name) [massage $text]
}

proc document_example {name description code} {
    global document
    if {![info exists document(current_file)]} return
    if {$document(current_file) != [info script]} return
    lappend document(examples) $name
    set document(example_description.$name) [massage $description]
    regsub -all "\t" $code "        " code
    set document(example_code.$name) $code
}

proc document_widget {name text} {
    global document
    if {![info exists document(current_file)]} return
    if {$document(current_file) != [info script]} return
    lappend document(widgets) $name
    set document(current_widget) $name
    set document(widget_text.$name) [massage $text]
    set document(methods_of.$name) ""
    set document(params) ""
}

proc document_param {name text} {
    global document
    if {![info exists document(current_file)]} return
    if {$document(current_file) != [info script]} return
    lappend document(params) $name
    set document(param_text.$name) [massage $text]
}

proc document_method {name text} {
    global document
    if {![info exists document(current_file)]} return
    if {$document(current_file) != [info script]} return
    set document(method_text.$document(current_widget).$name) [massage $text]
    lappend document(methods_of.$document(current_widget)) $name
}

proc describe_self {} {
    document_title document "document Tcl code in-line" {
	These procedures allow a programmer to embed manual page
	descriptions in Tcl code.  During normal operation these
	descriptions are ignored. 

	A Tcl script #document.tcl# is supplied to generate man pages
	from annotated source code.  When invoked as #tclsh
	document.tcl# `script'#.tcl#, in reads in `script'#.tcl# and
	produces a man page `script'#.n#.
	
    }

    document_proc document_title {
	sets the title, one-line summary, and description of a Tcl
	source file containing library code.  In `description',
	strings enclosed in single quotes will be `italicized', string
	brackets by hash marks will be typeset in #bold#, and white
	space will be removed.
    }

    document_proc document_program {
	sets the title, one-line summary, and description of a Tcl
	source file containing the code for a program.
	#document_program# will generate a man page that contains the
	descriptions of arguments in #args_parse# constructs.

	Programs using #document_program# should execute a #return#
	after using the document routines but before doing anything
	else, if the array #document# exists.  See the #STL# programs
	for example usage.
    }

    document_proc document_proc {
	documents the procedure `proc'.  `text' is treated the
	same way `description' is treated in #document_title#.
    }

    document_proc document_section {
	produces a seperate section `name' with text `text'.
	`text' is treated the same way `description' is treated in
	#document_title#.
    }

    document_proc document_example {
	produces a section titled `name', with text `description'
	preceding `code'.  `code' is typeset without any formatting.
    }

    document_proc document_widget {
	documents the widget `name'.  `text' is treated the
	same way `description' is treated in #document_title#.
	Widgets are documented by listing the creating proc in
	the #SYNOPSIS# section, then documenting options in the
	#WIDGET-SPECIFIC OPTIONS# section, then documenting the
	methods in the #DESCRIPTION# section.
    }

    document_proc document_param {
	documents the widget parameter `name'.  `text' is treated the
	same way `description' is treated in #document_title#.
    }

    document_proc document_method {
	documents the widget method `name'.  `text' is treated the
	same way `description' is treated in #document_title#.
    }

}

proc get_rcsid {file} {
    # have to be careful with the Id bit, lest RCS subsitute for us
    if [regexp "\\\$\[I\]d: \[^$\]+\\\$" [read_file $file] id] {
	return $id
    }
    return ""
}

run {
    set ext n
    set files ""
    args_parse $argv {
	args_arg -extension ext "extension to use for man page.  Defaults to \"n\"." {
	    args_gset ext $ext
	}
	args_remaining "Any remaining args are the tcl source files to generate man pages from." {
	    args_gset files $argv
	}
	args_error { args_usage; exit 1 }
    }
    foreach tcl_file $files {
	catch {unset document}
	set document(procs) ""
	set document(sections) ""
	set document(examples) ""
	set document(widgets) ""
	set document(flags) ""
	set document(current_file) $tcl_file
	if {$tcl_file == [info script]} {
	    # we're doing ourself
	    describe_self
	} else {
	    source $tcl_file
	}
	set man_file [file rootname $tcl_file].$ext
	msg_verbose
	msg "Creating man page $man_file"
	set fp [open $man_file w]
	puts $fp [read_file $stl_library/man_macros]
	puts $fp ".HS $document(name) cmds $document(name)"
	puts $fp ".BS"
	puts $fp ".SH NAME"
	puts $fp "$document(name) \\- $document(summary)"
	if !$document(is_program) {
	    puts $fp ".SH SYNOPSIS"
	    foreach proc $document(procs) {
		puts -nonewline $fp "\\fB$proc \\fI"
		foreach arg [info args $proc] {
		    puts -nonewline $fp "$arg "
		}
		puts $fp "\\fR\n"
	    }
	    set has_widgets 0
	    foreach widget $document(widgets) {
		puts $fp "\\fB$widget \\fIpathName ?options?\\fR"
		set has_widgets 1
	    }
	    if $has_widgets {
		puts $fp ".SH WIDGET-SPECIFIC OPTIONS"
		foreach param $document(params) {
		    puts $fp ".LP\n.nf"
		    puts $fp "Name:    \\fB$param\\fR"
		    upvar #0 ${widget}_priv class
		    puts $fp "Class:   \\fB[lindex $class(__param_info/$param) 1]\\fR"
		    puts $fp "Command-Line Switch:     \\fB-$param\\fR"
		    puts $fp ".fi\n.IP"
		    puts $fp $document(param_text.$param)
		    puts $fp ""
		}
	    }
	} else {
	    if {[llength $document(flags)] || [info exists document(remaining_description)]} {
		puts $fp ".SH SYNOPSIS"
	    }
	    foreach flag $document(flags) {
		puts $fp ".IP \"\\fB\\$flag \\fI$document(formals_of.$flag)\\fR\" 20"
		puts $fp $document(description_of.$flag)
	    }
	    if [info exists document(remaining_description)] {
		puts $fp ".IP \"\\fI...\\fR\" 20"
		puts $fp $document(remaining_description)
	    }
	}
	puts $fp ".BE"
	puts $fp ".SH DESCRIPTION"
	puts $fp $document(description)
	foreach proc $document(procs) {
	    puts $fp ".LP"
	    puts $fp "\\fB$proc\\fR $document(text_of.$proc)"
	}
	foreach widget $document(widgets) {
	    puts $fp ".LP"
	    puts $fp "\\fB$widget\\fR $document(widget_text.$widget)"
	    puts $fp ".LP"
	    puts $fp "The command that \\fB$widget\\fR creates understands"
	    puts $fp "the following options:"
	    foreach method $document(methods_of.$widget) {
		puts $fp ".TP"
		puts -nonewline $fp "\\fIpathName \\fB$method \\fI"
		foreach arg [lrange [info args $widget:$method] 1 end] {
		    puts -nonewline $fp "$arg "
		}
		puts $fp "\\fR"
		puts $fp $document(method_text.$widget.$method)
	    }
	}
	foreach section $document(sections) {
	    puts $fp ".SH $section"
	    puts $fp $document(section.$section)
	}
	foreach example $document(examples) {
	    puts $fp ".SH $example"
	    puts $fp $document(example_description.$example)
	    puts $fp ".nf"
	    puts $fp $document(example_code.$example)
	}
	set rcsid [get_rcsid $tcl_file]
	if {$rcsid != ""} {
	    puts $fp ".SH VERSION"
	    puts $fp "Version [lindex $rcsid 2] ([lindex $rcsid 3])"
	}
	close $fp
    }
}
