/* messaging.c - Enabled Mail -- message handling */

#include <ctype.h>
#include <errno.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <varargs.h>
#include <unistd.h>

#include <tcl.h>
#include "interfaceStyle.h"

#include "mh-6.8/h/mh.h"
#include "mh-6.8/h/addrsbr.h"
#include "mh-6.8/h/dropsbr.h"
#include "mh-6.8/h/mhn.h"
#include "mh-6.8/zotnet/mts.h"
#include "mh-6.8/zotnet/tws.h"

#if	defined(BANG) || defined(BERK) || defined(DUMB)
do not define BANG or BERK or DUMB...
#endif

/*    DATA */

/* Additional Messaging Functionality */

static char *addrs[] = { "To", "cc", "bcc", "Reply-To", "Resent-To",
			 "Resent-cc", "Resent-bcc", "Resent-Reply-To",
			 NULL };

static char *lmonth[] = { "January",  "February","March",   "April",
			  "May",      "June",    "July",    "August",
			  "September","October", "November","December" };

static	int	endian = 0;

static char ebcdicsafe[0x100] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static char hex2nib[0x80] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x08, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static unsigned char b642nib[0x80] = {
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0x3e, 0xff, 0xff, 0xff, 0x3f,
    0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
    0x3c, 0x3d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 
    0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e,
    0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16,
    0x17, 0x18, 0x19, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 
    0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
    0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30,
    0x31, 0x32, 0x33, 0xff, 0xff, 0xff, 0xff, 0xff
};
static char nib2b64[0x40+1] =
	"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

struct bodypart {
    char   *bp_part;

    char   *bp_type;
#define	NPARMS	10
    char   *bp_attrs[NPARMS + 2];
    char   *bp_values[NPARMS];

    char   *bp_id;

    char   *bp_descr;

    long    bp_start;

    long    bp_begin;
    long    bp_end;

    char   *bp_encoding;

    struct bodypart *bp_children;
    struct bodypart *bp_sibling;

    int	    ep_access;
#define	EP_MISS	(-2)
#define	EP_UNKN	(-1)
#define	EP_NONE	0
#define	EP_FILE	1
#define	EP_ANON	2
#define	EP_MAIL	3
    char   *ep_body;
    int	    ep_wantbody;

    struct bodypart *bp_parent;
    char   *bp_file;
    int	    bp_flags;
#define	FP_NONE	0x00
#define	FP_UNLK	0x01
#define	FP_FREE	0x02
};

static	int	bodyno;
static	char   *msgfmt;


/* Additional Delivery-Time Functionality */

	char   *SafeTcl_message = NULL;
	FILE   *SafeTcl_fp = NULL;
static struct bodypart
	       *SafeTcl_bp = NULL;


/* Recommended Extensions to the Trusted Tcl Interpreter */

static	int	uid = NOTOK;
static	int	gid = NOTOK;

static	char	ddate[BUFSIZ];
static	char	udate[BUFSIZ];
static	char   *folder = NULL;
static	char	mailbox[BUFSIZ];
static	char   *ftp = NULL;
static	char   *mailer = "/usr/lib/sendmail";
static	char   *printer = "/usr/ucb/lpr";
static	char   *render = "metamail";


#define	CACHE_NEVER	0
#define	CACHE_PRIVATE	1
#define	CACHE_PUBLIC	2

static	int	cache_rpolicy = CACHE_PRIVATE;
static	int	cache_wpolicy = CACHE_PRIVATE;
static	char   *cache_private = NULL;
static	char   *cache_public = NULL;


static	char   *sm_servers = NULL;

#ifdef	SMTP
int	sm_send ();
#endif


void	free_part (), record_status (), scan_parts ();
int	get_comment (), get_x400_comp (), find_cache (), find_cache_aux (),
	find_cache_aux2 (), compute_yday ();
struct bodypart *find_part (), *get_parts ();
FILE   *get_message ();


/* PUBLIC */

static	int	mhuser = 0;
static	int	mhnuser = 0;
extern	Tcl_Interp *unrestricted_interp;
extern int SafeTcl_GenidCmd();

/* EXTERN */

extern	int	generic;
extern	int	runsafely;

extern	int	SafeTcl_ExitCmd ();
#ifndef	_UNISTD_H
extern	free ();
extern  char *malloc ();
#endif

/*    Additional Messaging Functionality */

/* ARGSUSED */

static int  SafeTcl_getaddrs (window, interp, argc, argv)
Tk_Window	window;
Tcl_Interp     *interp;
int		argc;
char	      **argv;
{
    char   *cp,
	    buffer[BUFSIZ];

    if (argc != 2) {
	(void) sprintf (buffer,
			"wrong # args: should be \"%s\" addressesString",
			argv[0]);
	goto tcl_error;
    }
    argv++;

    Tcl_ResetResult (interp);
    while (cp = getname (*argv)) {
	char   *bp;
	register struct mailname *mp;

	if (!(mp = getm (cp, NULLCP, 0, AD_NHST, buffer))) {
	    while (getname (*argv))
		continue;
	    goto tcl_error;
	}

	bp = buffer;
	if (mp -> m_pers) {
	    (void) sprintf (bp, "%s <", legal_person (mp -> m_pers));
	    bp += strlen (bp);
	}
	(void) sprintf (bp, "%s", mp -> m_mbox);
	bp += strlen (bp);
	if (mp -> m_host && !mp -> m_nohost) {
	    (void) sprintf (bp, "@%s", mp -> m_host);
	    bp += strlen (bp);
	}
	if (mp -> m_pers) {
	    (void) sprintf (bp, ">");
	    bp += strlen (bp);
	}
	if (mp -> m_note) {
	    (void) sprintf (bp, mp -> m_note);
	    bp += strlen (bp);
	}

	mnfree (mp);

	Tcl_AppendElement (interp, buffer);
    }

    return TCL_OK;

tcl_error: ;
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    return TCL_ERROR;
}

/*  */

/* ARGSUSED */

static int  SafeTcl_getaddrprop (window, interp, argc, argv)
Tk_Window	window;
Tcl_Interp     *interp;
int		argc;
char	      **argv;
{
    char   *ap,
	   *bp,
	   *cp,
	    buffer[BUFSIZ],
	    from[BUFSIZ];
    register struct mailname *mp;

    if (argc != 3) {
	(void) sprintf (buffer,
			"wrong # args: should be \"%s\" addressString propertyName",
			argv[0]);
	goto tcl_error;
    }
    argv++;

    if (*(ap = *argv) == '\0') {
	if ((cp = getfullname ()) && *cp) {
	    char    sigbuf[BUFSIZ];

	    (void) strcpy (sigbuf, cp);
	    (void) sprintf (from, "%s <%s>", sigbuf,
			    adrsprintf (NULLCP, NULLCP));
	}
	else
	    (void) strcpy (from, adrsprintf (NULLCP, NULLCP));
	ap = from;
    }
    if (!(cp = getname (ap))) {
	(void) strcpy (buffer, "null input");
	goto tcl_error;
    }
    if (!(mp = getm (cp, NULLCP, 0, AD_NHST, buffer))) {
	while (getname (ap))
	    continue;
	goto tcl_error;
    }
    if (cp = getname (ap)) {
	mnfree (mp);
	(void) strcpy (buffer, "too much input");
	goto tcl_error;
    }
    argv++;

    *(bp = buffer) = '\0';
    switch (*(cp = *argv++)) {
	case 'a':
	    if (strcmp (cp, "address") == 0) {
address: ;
		(void) sprintf (bp, "%s", mp -> m_mbox);
		bp += strlen (bp);
		if (mp -> m_host) {
		    (void) sprintf (bp, "@%s", mp -> m_host);
		    bp += strlen (bp);
		}
		if (!cp) {
		    (void) sprintf (bp, ">");
		    bp += strlen (bp);
		}
		break;
	    }
	    goto losing;

	case 'd':
	    if (strcmp (cp, "domain") == 0) {
		if (mp -> m_host)
		    (void) strcpy (bp, mp -> m_host);
		break;
	    }
	    goto losing;

	case 'f':
	    if (strcmp (cp, "friendly") == 0) {
		register char *mbox;
		char    given[BUFSIZ],
			surname[BUFSIZ];

		if (mp -> m_pers) {
		    (void) strcpy (bp, mp -> m_pers);
		    break;
		}
		if (mp -> m_note) {
		    if (*(mbox = mp -> m_note) == '(') {
			mbox++;
			while (isspace (*mbox))
			    mbox++;
		    }
		    (void) strcpy (bp, mbox);
		    if ((mbox = bp + strlen (bp) - 1) >= bp && *mbox == ')') {
			*mbox-- = '\0';
			while (mbox >= bp && isspace (*mbox))
			    *mbox-- = '\0';
		    }
		    if (*bp)
			break;
		}
		if (!(mbox = mp -> m_mbox))
		    break;
		if (*mbox == '"')
		    mbox++;
		if (*mbox != '/')
		    goto local;
		if (get_x400_comp (mbox, "/PN=", bp)) {
		    for (mbox = bp; mbox = index (mbox, '.'); )
			*mbox++ = ' ';

		    break;
		}
		if (!get_x400_comp (mbox, "/S=", surname))
		    goto local;
		if (get_x400_comp (mbox, "/G=", given))
		    (void) sprintf (bp, "%s %s", given, surname);
		else
		    (void) strcpy (bp, surname);
		break;
	    }
	    goto losing;

	case 'l':
	    if (strcmp (cp, "local") == 0) {
local: ;
		if (mp -> m_mbox)
		    (void) strcpy (bp, mp -> m_mbox);
		break;
	    }
	    goto losing;

	case 'm':
	    if (strcmp (cp, "mymbox") == 0) {
		(void) sprintf (bp, "%d", ismymbox (mp) ? 1 : 0);
		break;
	    }
	    goto losing;

	case 'p':
	    if (strcmp (cp, "phrase") == 0) {
		if (mp -> m_pers)
		    (void) strcpy (bp, mp -> m_pers);
		break;
	    }
	    else
	    if (strcmp (cp, "proper") == 0) {
		if (mp -> m_pers) {
		    (void) sprintf (bp, "%s <", legal_person (mp -> m_pers));
		    bp += strlen (bp);
		    cp = NULL;
		}
		goto address;
	    }
	    goto losing;

	default:
losing: ;
	    mnfree (mp);
	    (void) sprintf (buffer, "unknown property %s", cp);
	    goto tcl_error;
    }

    mnfree (mp);

    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    return TCL_OK;

tcl_error: ;
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    return TCL_ERROR;
}

/*  */

/* ARGSUSED */

static int  SafeTcl_getdateprop (window, interp, argc, argv)
Tk_Window	window;
Tcl_Interp     *interp;
int		argc;
char	      **argv;
{
    long    l;
    char   *bp,
	   *cp,
	    buffer[BUFSIZ];
    register struct tws *tw;

    if (argc != 3) {
	(void) sprintf (buffer,
			"wrong # args: should be \"%s\" dateString propertyName",
			argv[0]);
        goto tcl_error;
    }
    argv++;

    if (**argv == '\0') {
	if (!(tw = dtwstime ()))
	    (void) strcpy (buffer, "unable to determine current time");
    }
    else {
	if (!(tw = dparsetime (*argv)))
	    (void) strcpy (buffer, "invalid date-time specification");
	tw -> tw_yday = compute_yday (tw);
    }
    if (!tw)
	goto tcl_error;
    argv++;

    *(bp = buffer) = '\0';
    switch (*(cp = *argv++)) {
	case 'd':
	    if (strcmp (cp, "date2gmt") == 0) {
		if (!(l = tw -> tw_clock))
		    l = twclock (tw);
		(void) sprintf (bp, dasctime (dgmtime (&l), TW_ZONE));
		break;
	    }
	    else
	    if (strcmp (cp, "date2local") == 0) {
		if (!(l = tw -> tw_clock))
		    l = twclock (tw);
		(void) sprintf (bp, dasctime (dlocaltime (&l), TW_ZONE));
		break;
	    }
	    else
	    if (strcmp (cp, "day") == 0) {
		if (!(tw -> tw_flags & (TW_SEXP | TW_SIMP)))
		    set_dotw (tw);
		(void) sprintf (bp, tw_dotw[tw -> tw_wday]);
		break;
	    }
	    else
	    if (strcmp (cp, "dst") == 0) {
		(void) sprintf (bp, "%d", tw -> tw_flags & TW_DST ? 1 : 0);
		break;
	    }
	    goto losing;

	case 'h':
	    if (strcmp (cp, "hour") == 0) {
		(void) sprintf (bp, "%d", tw -> tw_hour);
		break;
	    }
	    goto losing;

	case 'l':
	    if (strcmp (cp, "lmonth") == 0) {
		(void) strcpy (bp, lmonth[tw -> tw_mon]);
		break;
	    }
	    goto losing;

	case 'm':
	    if (strcmp (cp, "mday") == 0) {
		(void) sprintf (bp, "%d", tw -> tw_mday);
		break;
	    }
	    else
	    if (strcmp (cp, "min") == 0) {
		(void) sprintf (bp, "%d", tw -> tw_min);
		break;
	    }
	    else
	    if (strcmp (cp, "mon") == 0) {
		(void) sprintf (bp, "%d", tw -> tw_mon + 1);
		break;
	    }
	    else
	    if (strcmp (cp, "month") == 0) {
		(void) sprintf (bp, "%s", tw_moty[tw -> tw_mon]);
		break;
	    }
	    goto losing;

	case 'p':
	    if (strcmp (cp, "proper") == 0) {
		(void) strcpy (bp, dasctime (tw, TW_ZONE));
		break;
	    }
	    goto losing;

	case 'r':
	    if (strcmp (cp, "rclock") == 0) {
		if (!(l = tw -> tw_clock))
		    l = twclock (tw);
		(void) sprintf (bp, "%ld", time ((long *) 0) - l);
		break;
	    }
	    goto losing;

	case 's':
	    if (strcmp (cp, "sday") == 0) {
		int	flags;

		if (!(tw -> tw_flags & (TW_SEXP | TW_SIMP)))
		    set_dotw (tw);
		flags = tw -> tw_flags & TW_SDAY;
		(void) sprintf (bp, "%d", flags == TW_SEXP ? 1
					: flags == TW_SIMP ? 0 : -1);
		break;
	    }
	    else
	    if (strcmp (cp, "sec") == 0) {
		(void) sprintf (bp, "%d", tw -> tw_sec);
		break;
	    }
	    else
	    if (strcmp (cp, "szone") == 0) {
		(void) sprintf (bp, "%d",
				tw -> tw_flags & TW_SZONE == TW_SZEXP ? 1
								      : -1);
		break;
	    }
	    goto losing;

	case 't':
	    if (strcmp (cp, "tzone") == 0) {
		(void) strcpy (bp, dtwszone (tw));
		break;
	    }
	    goto losing;

	case 'w':
	    if (strcmp (cp, "wday") == 0) {
		if (!(tw -> tw_flags & (TW_SEXP | TW_SIMP)))
		    set_dotw (tw);
		(void) sprintf (bp, "%d", tw -> tw_wday);
		break;
	    }
	    else
	    if (strcmp (cp, "weekday") == 0) {
		if (!(tw -> tw_flags & (TW_SEXP | TW_SIMP)))
		    set_dotw (tw);
		(void) sprintf (bp, tw_ldotw[tw -> tw_wday]);
		break;

		break;
	    }
	    goto losing;

	case 'y':
	    if (strcmp (cp, "yday") == 0) {
		(void) sprintf (bp, "%d", tw -> tw_yday);
		break;
	    }
	    else
	    if (strcmp (cp, "year") == 0) {
		(void) sprintf (bp, "%d", tw -> tw_year);
		break;
	    }
	    goto losing;

	case 'z':
	    if (strcmp (cp, "zone") == 0) {
		(void) sprintf (bp, "%d", tw -> tw_zone);
		break;
	    }
	    goto losing;

	default:
losing: ;
	    (void) sprintf (buffer, "unknown property %s", cp);
	    goto tcl_error;
    }

    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    return TCL_OK;

tcl_error: ;
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    return TCL_ERROR;
}

/*  */

/* ARGSUSED */

static int  SafeTcl_getheader (window, interp, argc, argv)
Tk_Window	window;
Tcl_Interp     *interp;
int		argc;
char	      **argv;
{
    int	    addon,
	    compnum,
	    state;
    char   *cp,
	   *header,
	    buffer[BUFSIZ],
	    name[NAMESZ],
	  **ap;
    FILE   *fp;

    if (argc < 2 || argc > 3) {
	(void) sprintf (buffer,
			"wrong # args: should be \"%s\" headerName ?bodyPart?",
			argv[0]);
	goto tcl_error;
    }
    argv++;

    header = *argv++;
    for (ap = addrs; *ap; ap++)
	if (uleq (*ap, header))
	    break;
    addon = *ap ? 1 : 0;

    if (argc == 2) {
	if (!(fp = SafeTcl_fp)) {
	    (void) strcpy (buffer, "no default message");
	    goto tcl_error;
	}

	rewind (fp);
    }
    else
	if (!(fp = get_message (interp, *argv, NULLCP)))
	    return TCL_ERROR;

    cp = NULL;
    Tcl_ResetResult (interp);
    for (compnum = 0, state = FLD;;) {
	switch (state = m_getfld (state, name, buffer, sizeof buffer, fp)) {
	    case FLD:
	    case FLDEOF:
	    case FLDPLUS:
	        compnum++;

		if (uleq (name, header)) {
		    if (addon && cp)
			cp = add (", ", cp);
		    cp = add (buffer, cp);
		    while (state == FLDPLUS) {
			state = m_getfld (state, name, buffer, sizeof buffer,
					  fp);
			cp = add (buffer, cp);
		    }
		    if (addon)
			continue;
		    break;
		}
		while (state == FLDPLUS)
		    state = m_getfld (state, name, buffer, sizeof buffer, fp);
		continue;

	    case BODY:
	    case BODYEOF:
	    case FILEEOF:
		break;

	    case LENERR:
	    case FMTERR:
		(void) sprintf (buffer,
				"message format error in component #%d",
				compnum);
		goto losing;

	    default:
		(void) sprintf (buffer, "m_getfld returned %d", state);
losing: ;
		if (cp)
		    free (cp);
		if (fp != SafeTcl_fp)
		    (void) fclose (fp);
		goto tcl_error;
	}
	break;
    }

    if (cp) {
	Tcl_SetResult (interp, trimcpy (cp), TCL_DYNAMIC);
	free (cp);
    }
	
    if (fp != SafeTcl_fp)
	(void) fclose (fp);
    return TCL_OK;

tcl_error: ;
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    return TCL_ERROR;
}

/*  */

/* ARGSUSED */

static int  SafeTcl_getheaders (window, interp, argc, argv)
Tk_Window	window;
Tcl_Interp     *interp;
int		argc;
char	      **argv;
{
    int	    compnum,
	    state;
    char   *cp,
    	    buffer[BUFSIZ],
	    name[NAMESZ],
	   *vec[3];
    FILE   *fp;

    if (argc < 1 || argc > 2) {
	(void) sprintf (buffer,
			"wrong # args: should be \"%s\" ?bodyPart?",
			argv[0]);
	goto tcl_error;
    }
    argv++;

    if (argc == 1) {
	if (!(fp = SafeTcl_fp)) {
	    (void) strcpy (buffer, "no default message");
	    goto tcl_error;
	}

	rewind (fp);
    }
    else
	if (!(fp = get_message (interp, *argv, NULLCP)))
	    return TCL_ERROR;

    Tcl_ResetResult (interp);
    for (compnum = 0, state = FLD;;) {
	switch (state = m_getfld (state, name, buffer, sizeof buffer, fp)) {
	    case FLD:
	    case FLDEOF:
	    case FLDPLUS:
	        compnum++;
		cp = add (buffer, NULLCP);
		while (state == FLDPLUS) {
		    state = m_getfld (state, name, buffer, sizeof buffer, fp);
		    cp = add (buffer, cp);
		}
		vec[0] = name;
		vec[1] = trimcpy (cp);
		vec[2] = NULL;
		free (cp);
		Tcl_AppendElement (interp, cp = Tcl_Merge (2, vec));
		free (cp);
		continue;

	    case BODY:
	    case BODYEOF:
	    case FILEEOF:
		break;

	    case LENERR:
	    case FMTERR:
		(void) sprintf (buffer,
				"message format error in component #%d",
				compnum);
		goto losing;

	    default:
		(void) sprintf (buffer, "m_getfld returned %d", state);
losing: ;
		if (fp != SafeTcl_fp)
		    (void) fclose (fp);
		goto tcl_error;
	}
	break;
    }

    if (fp != SafeTcl_fp)
	(void) fclose (fp);
    return TCL_OK;

tcl_error: ;
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    return TCL_ERROR;
}

/*  */

#define	BMAX	70


/* ARGSUSED */

static int  SafeTcl_makebody (window, interp, argc, argv)
Tk_Window	window;
Tcl_Interp     *interp;
int		argc;
char	      **argv;
{
    int	    multipart;
    char   *cp,
	   *dp,
	   *ip,
	    boundary[BMAX + 1],
	    buffer[BUFSIZ];

    cp = dp = ip = NULL;
    if (argc < 3) {
	(void) sprintf (buffer,
			"wrong # args: should be \"%s\" typeString ?-parameter string ...? ?-description string? ?-id string? [cookedData ?encodingName? | bodyPart ...]",
			argv[0]);
	goto tcl_error;
    }
    argc--, argv++;

    cp = concat ("Content-Type: ", **argv ? *argv : "text/plain", NULLCP);
    multipart = uprf (*argv, "multipart/");
    argc--, argv++;

    for (; argc > 0; argc--, argv++) {
	if (strcmp (*argv, "-parameter") == 0) {
	    argc--, argv++;
	    if (argc <= 0) {
		(void) strcpy (buffer, "missing argument to -parameter");
		goto tcl_error;
	    }
	    (void) sprintf (buffer, "; %s", *argv);
	    cp = add (buffer, cp);
	    continue;
	}

	if (strcmp (*argv, "-description") == 0) {
	    if (dp) {
		(void) strcpy (buffer, "too many descriptions");
		goto tcl_error;
	    }
	    argc--, argv++;
	    if (argc <= 0) {
		(void) strcpy (buffer, "missing argument to -description");
		goto tcl_error;
	    }
	    dp = *argv;
	    continue;
	}

	if (strcmp (*argv, "-id") == 0) {
	    if (ip) {
		(void) strcpy (buffer, "too many IDs");
		goto tcl_error;
	    }
	    argc--, argv++;
	    if (argc <= 0) {
		(void) strcpy (buffer, "missing argument to -id");
		goto tcl_error;
	    }
	    ip = *argv;
	    continue;
	}

	break;
    }
    if (multipart) {
	int	ac,
		blen;
	char   *bp,
	      **av;

	if (argc < 1) {
	    (void) strcpy (buffer, "need at least one body-part");
	    goto tcl_error;
	}

	(void) sprintf (boundary, "----- =_%ld", time ((long *) 0));
	bp = boundary + (blen = strlen (boundary));
	for (ac = argc, av = argv; ac > 0; ac--, av++) {
	    register char *ep;

	    for (ep = *av; ep; (ep = index (ep, '\n')) ? ep++ : 0) {
		if (strncmp (ep, "--", sizeof "--" - 1))
		    continue;
		ep += sizeof "--" - 1;
		while (strncmp (ep, boundary, blen) == 0) {
		    if (blen >= BMAX - 1) {
			(void) strcpy (buffer, "unable to generate boundary");
			goto tcl_error;
		    }
		    (void) sprintf (bp + 1, *bp == '_' ? "=" : "_");
		    bp++, blen++;
		}
	    }
	}

	bp = concat ("; boundary=\"", boundary, "\"", NULLCP);
	cp = add (bp, cp);
	free (bp);
    }
    cp = add ("\n", cp);
    if (dp) {
	dp = concat ("Content-Description: ", dp, "\n", NULLCP);
	cp = add (dp, cp);
	free (dp);
    }
    if (ip) {
	ip = concat ("Content-ID: ", ip, "\n", NULLCP);
	cp = add (ip, cp);
	free (ip);
    }
    else {
	(void) sprintf (buffer, msgfmt, ++bodyno);
	cp = add (buffer, cp);
    }
    if (multipart) {
	for (; argc > 0; argc--, argv++) {
	    dp = concat ("\n--", boundary, "\n", *argv, NULLCP);
	    cp = add (dp, cp);
	    free (dp);
	}
	dp = concat ("\n--", boundary, "--\n", NULLCP);
	cp = add (dp, cp);
	free (dp);
    }
    else {
	if (argc > 2) {
	    (void) strcpy (buffer, "need exactly one data value");
	    goto tcl_error;
	}

	if (argc == 2 && *(dp = argv[1])) {
	    dp = concat ("Content-Transfer-Encoding: ", dp, "\n", NULLCP);
	    cp = add (dp, cp);
	    free (dp);
	}

	dp = concat ("\n", argv[0], NULLCP);
	cp = add (dp, cp);
	free (dp);
    }

    Tcl_SetResult (interp, cp, TCL_DYNAMIC);
    return TCL_OK;

tcl_error: ;
    if (cp)
	free (cp);
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    return TCL_ERROR;
}

/*  */

/* ARGSUSED */

static int  SafeTcl_getparts (window, interp, argc, argv)
Tk_Window	window;
Tcl_Interp     *interp;
int		argc;
char	      **argv;
{
    char    buffer[BUFSIZ];
    struct bodypart *c1;
    FILE   *fp;

    if (argc < 1 || argc > 2) {
	(void) sprintf (buffer,
			"wrong # args: should be \"%s\" ?bodyPart?",
			argv[0]);
	goto tcl_error;
    }
    argv++;

    if (argc == 1) {
	if (!(fp = SafeTcl_fp)) {
	    (void) strcpy (buffer, "no default message");
	    goto tcl_error;
	}

	rewind (fp);
    }
    else
	if (!(fp = get_message (interp, *argv, NULLCP)))
	    return TCL_ERROR;

    if (!(c1 = get_parts (interp, fp, 1, "1"))) {
	if (fp != SafeTcl_fp)
	    (void) fclose (fp);
	return TCL_ERROR;
    }

    Tcl_ResetResult (interp);
    scan_parts (interp, c1);

    free_part (c1);
    if (fp != SafeTcl_fp)
	(void) fclose (fp);
    return TCL_OK;

tcl_error: ;
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    return TCL_ERROR;
}

/*  */

/* ARGSUSED */

static int  SafeTcl_getbodyprop (window, interp, argc, argv)
Tk_Window	window;
Tcl_Interp     *interp;
int		argc;
char	      **argv;
{
    int	    cc;
    long    len,
	    start;
    char   *cp,
	   *part,
	   *prop,
	    buffer[BUFSIZ];
    struct bodypart *c1,
		    *c2;
    struct stat	     st;
    FILE   *fp,
	   *gp;

    if (argc < 3 || argc > 4) {
	(void) sprintf (buffer,
			"wrong # args: should be \"%s\" partNumber propertyName ?bodyPart?",
			argv[0]);
	goto tcl_error;
    }
    argv++;

    part = *argv++;

    switch (*(prop = *argv++)) {
	case 'a':
	    if (strcmp (prop, "all") == 0)
		break;
	    goto losing;

	case 'd':
	    if (strcmp (prop, "descr") == 0)
		break;
	    goto losing;

	case 'e':
	    if (strcmp (prop, "encoding") == 0)
		break;
	    goto losing;

	case 'h':
	    if (strcmp (prop, "headers") == 0)
		break;
	    goto losing;

	case 'i':
	    if (strcmp (prop, "id") == 0)
		break;
	    goto losing;

	case 'p':
	    if (strcmp (prop, "parms") == 0)
		break;
	    goto losing;

	case 's':
	    if (strcmp (prop, "size") == 0)
		break;
	    goto losing;

	case 't':
	    if (strcmp (prop, "type") == 0)
		break;
	    goto losing;

	case 'v':
	    if (strcmp (prop, "value") == 0)
		break;
	    goto losing;

	default:
losing: ;
	    (void) sprintf (buffer, "unknown property %s", prop);
	    goto tcl_error;
    }

    if (argc == 3) {
	if (!(fp = SafeTcl_fp)) {
	    (void) strcpy (buffer, "no default message");
	    goto tcl_error;
	}

	rewind (fp);
    }
    else
	if (!(fp = get_message (interp, *argv, NULLCP)))
	    return TCL_ERROR;

    gp = NULL;

    if (!(c1 = get_parts (interp, fp, 1, "1"))
	    || !(c2 = find_part (interp, c1, part))) {
	if (c1)
	    free_part (c1);
	if (fp != SafeTcl_fp)
	    (void) fclose (fp);
	(void) sprintf (buffer, "no such part %s", part);
	goto tcl_error;
    }

    if (c2 -> ep_access != EP_NONE) {
	char  cachefile[BUFSIZ];

	switch (*prop) {
	    case 'a':	/* all */
	    case 's':	/* size */
	    case 'v':	/* value */
	        if (find_cache (cache_rpolicy, (int *) 0, c2 -> bp_id,
				cachefile) != NOTOK) {
		    c2 -> bp_file = getcpy (cachefile);
		    c2 -> bp_flags = FP_FREE;
		}

		if (!c2 -> bp_file)
		    switch (c2 -> ep_access) {
		        case EP_MISS:
		            (void) strcpy (buffer, "missing access-type");
			    goto tcl_losing;

			case EP_UNKN:
ep_unkn: ;
			    (void) strcpy (buffer, "unsupported access-type");
			    goto tcl_losing;

			case EP_FILE:
			    if (interp != unrestricted_interp)
				goto ep_unkn;
			    if (OpenFile (interp, c2) == NOTOK)
				goto tcl_extern_losing;
			    break;

			case EP_ANON:
			    if (OpenFtp (interp, c2) == NOTOK) {
tcl_extern_losing: ;
				free_part (c1);
				if (fp != SafeTcl_fp)
				    (void) fclose (fp);
				return TCL_ERROR;
			    }
			    break;

			case EP_MAIL:
			    (void) strcpy (buffer, "mail-server access-type");
			    goto tcl_losing;
		    }

		if (!(gp = fopen (c2 -> bp_file, "r"))) {
		    (void) sprintf (buffer, "unable to read %s",
				    c2 -> bp_file);
		    goto tcl_losing;
		}
		if (fstat (fileno (gp), &st) == NOTOK) {
		    (void) sprintf (buffer, "error determining file size");
		    goto tcl_losing;
		}
		break;

	    default:
	        break;
	}
    }

    Tcl_ResetResult (interp);
    switch (*prop) {
	case 'a':	/* all */
	    if (c2 -> ep_access > EP_NONE) {
		len = (c2 -> bp_begin - 1) - (start = c2 -> bp_start);
		goto extract;
	    }
	    else
		if ((len = c2 -> bp_end - (start = c2 -> bp_start)) > 0)
		    goto extract;
	    break;

	case 'd':	/* descr */
	    if (c2 -> bp_descr)
		Tcl_SetResult (interp, c2 -> bp_descr, TCL_VOLATILE);
	    break;

	case 'e':	/* encoding */
	    if (c2 -> ep_access > EP_NONE)
		Tcl_SetResult (interp, "base64", TCL_STATIC);
	    else
		if (c2 -> bp_encoding)
		    Tcl_SetResult (interp, c2 -> bp_encoding, TCL_VOLATILE);
	        else
		    Tcl_SetResult (interp, "7bit", TCL_STATIC);
	    break;

	case 'h':	/* headers */
	    if ((len = (c2 -> bp_begin - 1) - (start = c2 -> bp_start)) > 0)
		goto extract;
	    break;

	case 'i':	/* id */
	    if (c2 -> bp_id)
		Tcl_SetResult (interp, c2 -> bp_id, TCL_VOLATILE);
	    break;

	case 'p':	/* parms */
	    if (c2 -> bp_attrs[0]) {
		int	vecp;
		char  **ap,
		      **ep,
		       *vec[3];

		for (ap = c2 -> bp_attrs, ep = c2 -> bp_values;
		         *ap;
		         ap++, ep++) {
		    vecp = 0;
		    vec[vecp++] = *ap;
		    vec[vecp++] = *ep;
		    Tcl_AppendElement (interp, cp = Tcl_Merge (vecp, vec));
		    free (cp);
		}
		if (c2 -> ep_wantbody) {
		    struct bodypart *child = c2 -> bp_children;

		    if ((cc = c2 -> bp_end - child -> bp_begin) > 0) {
			char   *bp;

			if (!(c2 -> ep_body = bp =
						malloc ((unsigned int) cc))) {
			    (void) strcpy (buffer, "out of memory");
			    goto tcl_losing;
			}
			(void) fseek (fp, child -> bp_begin, 0);
			if (fread (bp, sizeof *bp, cc, fp) != cc) {
			    (void) strcpy (buffer, "error reading message");
			    goto tcl_losing;
			}
			*(bp + cc) = '\0';
		    }

		    c2 -> ep_wantbody = 0;
		}
		if (c2 -> ep_body) {
		    vecp = 0;
		    vec[vecp++] = "body";
		    vec[vecp++] = c2 -> ep_body;
		    Tcl_AppendElement (interp, cp = Tcl_Merge (vecp, vec));
		    free (cp);
		}
	    }
	    break;

	case 's':	/* size */
	    if (c2 -> ep_access > EP_NONE)
		(void) sprintf (buffer, "%ld", (long) st.st_size);
	    else
		(void) sprintf (buffer, "%ld", c2 -> bp_end - c2 -> bp_begin);
	    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
	    break;

	case 't':	/* type */
	    if (c2 -> bp_type)
		Tcl_SetResult (interp, c2 -> bp_type, TCL_VOLATILE);
	    break;

	case 'v':	/* value */
	    if (c2 -> ep_access > EP_NONE)
		goto get_value;
	    if ((len = c2 -> bp_end - (start = c2 -> bp_begin)) > 0) {
extract: ;
		cc = len;
		if (!(cp = malloc ((unsigned int) (cc + 1)))) {
no_mem: ;
		    (void) strcpy (buffer, "out of memory");
oops: ;
		    if (cp)
			free (cp);
		    goto tcl_losing;
		}
		if (fseek (fp, start, 0) == NOTOK) {
		    (void) strcpy (buffer, "error positioning body-part");
		    goto oops;
		}
		if (fread (cp, sizeof *cp, cc, fp) != cc) {
bad_read: ;
		    (void) strcpy (buffer, "error reading message");
		    goto oops;
		}
		*(cp + cc) = '\0';
		Tcl_SetResult (interp, cp, TCL_DYNAMIC);

		if (c2 -> ep_access > EP_NONE && *prop != 'h') {
		    int	    result;

		    if (*prop == 'a')
			Tcl_AppendResult (interp,
					  "Content-Transfer-Encoding: base64\n\n",
					  NULLCP);
get_value: ;
		    if (!(cp = malloc ((unsigned int) (cc = st.st_size))))
			goto no_mem;
		    if (fread (cp, sizeof *cp, cc, gp) != cc)
			goto bad_read;
		    result = SafeTcl_encode_aux (interp, "base64", cp, cc);
		    free (cp);
		    if (result != TCL_OK)
			return result;
		}
	    }
	    break;
    }

    free_part (c1);
    if (fp != SafeTcl_fp)
	(void) fclose (fp);
    if (gp)
	(void) fclose (gp);
    return TCL_OK;

tcl_losing: ;
    free_part (c1);
    if (fp != SafeTcl_fp)
	(void) fclose (fp);
    if (gp)
	(void) fclose (gp);

tcl_error: ;
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    return TCL_ERROR;
}

/*  */

static int  OpenFile (interp, c)
Tcl_Interp	*interp;
struct bodypart *c;
{
    char  **ap,
	  **ep;
    struct bodypart *parent = c -> bp_parent;

    for (ap = parent -> bp_attrs, ep = parent -> bp_values; *ap; ap++, ep++)
	if (uleq (*ap, "name")) {
	    c -> bp_file = *ep;
	    c -> bp_flags = FP_NONE;
	    return OK;
	}

    Tcl_SetResult (interp, "missing name", TCL_STATIC);
    return NOTOK;
}

/*  */

static int  OpenFtp (interp, c)
Tcl_Interp	*interp;
struct bodypart *c;
{
    int	    cachetype;
    char   *dir,
	   *mode,
	   *name,
	   *perm,
	   *site,
	    cachefile[BUFSIZ],
	  **ap,
	  **ep;
    struct bodypart *parent = c -> bp_parent;

    dir = mode = name = perm = site = NULL;
    for (ap = parent -> bp_attrs, ep = parent -> bp_values; *ap; ap++, ep++) {
	if (uleq (*ap, "directory")) {
	    dir = *ep;
	    continue;
	}
	if (uleq (*ap, "mode")) {
	    mode = *ep;
	    continue;
	}
	if (uleq (*ap, "name")) {
	    name = *ep;
	    continue;
	}
	if (uleq (*ap, "permission")) {
	    perm = *ep;
	    continue;
	}
	if (uleq (*ap, "site")) {
	    site = *ep;
	    continue;
	}
    }
    if (!name || !site) {
	Tcl_SetResult (interp, name ? "missing site" : "missing name",
		       TCL_STATIC);
	return NOTOK;
    }

    if ((!perm || !uleq (perm, "read-write"))
	    && find_cache (cache_wpolicy, &cachetype, c -> bp_id, cachefile)
		    != NOTOK) {
	c -> bp_file = getcpy (cachefile);
	c -> bp_flags = FP_FREE;
    }
    else {
	c -> bp_file = getcpy (m_tmpfil (invo_name));
	c -> bp_flags = FP_UNLK | FP_FREE;
    }

    if (OpenFtpAux (interp, site, "anonymous", NULLCP, dir, name, c -> bp_file,
		    mode) != TCL_OK) {
	c -> bp_flags |= FP_UNLK;
	return NOTOK;
    }    

    if (!(c -> bp_flags & FP_UNLK))
	(void) chmod (c -> bp_file, cachetype ? m_gmprot () : 0444);

    return OK;
}

/*  */

int	OpenFtpAux (interp, site, user, pass, dir, name, file, mode)
Tcl_Interp     *interp;
char	       *site,
	       *user,
	       *pass,
	       *dir,
	       *name,
	       *file,
	       *mode;
{
    char	buffer[BUFSIZ];

#ifndef	FTP
    if (!ftp) {
	if (interp) {
	    (void) sprintf (buffer,
			    mhuser ? "%s-access-ftp profile entry undefined"
				   : "ftp configuration setting undefined",
			    "swish");
	    Tcl_SetResult (interp, buffer, TCL_STATIC);
	}
	return TCL_ERROR;
    }
#endif

    if (!pass)
	(void) sprintf (pass = buffer, "%s@%s", getusr (), LocalName ());

#ifdef	FTP
    if (ftp)
#endif
    {
	int	i,
		pid,
		status;

	for (i = 0; (pid = fork ()) == NOTOK && i < 5; i++)
	    sleep (5);
	switch (pid) {
	    case NOTOK:
	        (void) strcpy (buffer, "unable to fork");
		return NOTOK;
		/* NOTREACHED */

	    case OK:
		execlp (ftp, r1bindex (ftp, '/'), site, user, pass, dir, name,
			file, mode && uleq (mode, "ascii") ? "ascii"
							   : "binary", NULLCP);
		_exit (-1);
		/* NOTREACHED */

	    default:
		if (status = pidwait (pid, OK)) {
#ifdef	FTP
losing_ftp: ;
#endif
		    if (interp)
			Tcl_SetResult (interp, "file transfer failed",
				       TCL_STATIC);
		    return TCL_ERROR;
		}
		break;
	}
    }
#ifdef	FTP
    else
	if (ftp_get (site, "anonymous", pass, dir, name, file,
		     mode && uleq (mode, "ascii"), 0) == NOTOK)
	    goto losing_ftp;
#endif

    return TCL_OK;
}

#if	defined(FTP) || defined(SMTP)
#undef	NULLVP
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>


#if	defined(BIND) && !defined(h_addr)
#define	h_addr	h_addr_list[0]
#endif

#define	inaddr_copy(hp,sin) \
    bcopy ((hp) -> h_addr, (char *) &((sin) -> sin_addr), (hp) -> h_length)


int	debugsw = 0;
int	verbosw = 0;

u_long	inet_addr ();


static char *empty = NULL;
#ifdef	h_addr
static char *addrlist[2] = { NULL };
#endif

struct hostent *gethostbystring (s)
char   *s;
{
    register struct hostent *h;
    static u_long iaddr;
    static struct hostent   hs;

    iaddr = inet_addr (s);
    if (iaddr == NOTOK && strcmp (s, "255.255.255.255"))
	return gethostbyname (s);

    h = &hs;
    h -> h_name = s;
    h -> h_aliases = &empty;
    h -> h_addrtype = AF_INET;
    h -> h_length = sizeof (iaddr);
#ifdef	h_addr
    h -> h_addr_list = addrlist;
    bzero ((char *) addrlist, sizeof addrlist);
#endif
    h -> h_addr = (char *) &iaddr;

    return h;
}
#endif

/*  */

/* ARGSUSED */

static int  SafeTcl_encode (window, interp, argc, argv)
Tk_Window	window;
Tcl_Interp     *interp;
int		argc;
char	      **argv;
{
    char   *cp,
	    buffer[BUFSIZ];
    
    if (argc != 3) {
	(void) sprintf (buffer,
			"wrong # args: should be \"%s\" encodingName rawData",
			argv[0]);
	goto tcl_error;
    }
    argv++;
    
    switch (*(cp = *argv++)) {
	case 'b':
	    if (strcmp (cp, "base64") == 0)
		break;
	    goto losing;

	case 'q':
	    if (strcmp (cp, "quoted-printable") == 0)
		break;
	    goto losing;

	default:
losing: ;
	    (void) sprintf (buffer, "unknown encoding %s", cp);
	    goto tcl_error;
    }

    Tcl_ResetResult (interp);

    return SafeTcl_encode_aux (interp, cp, *argv, strlen (*argv));

tcl_error: ;
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    return TCL_ERROR;
}

/*  */

int	SafeTcl_encode_aux (interp, mode, cp, len)
Tcl_Interp     *interp;
char	       *mode,
	       *cp;
int	len;
{
    int	    n;
    char    buffer[BUFSIZ];

    if (*mode == 'b') {
	for (n = BPERLIN; len > 0;) {
	    int	    cc;
	    unsigned long bits;
	    register char *bp;
	    char    outbuf[4 + 1];

	    if ((cc = len) > 3)
		cc = 3;
	    bits = (*cp++ & 0xff) << 16, len--;
	    if (len > 0) {
		bits |= (*cp++ & 0xff) << 8, len--;
		if (len > 0)
		    bits |= (*cp++ & 0xff), len--;
	    }

	    for (bp = outbuf + sizeof outbuf - 1; bp > outbuf; bits >>= 6)
		*--bp = nib2b64[bits & 0x3f];
	    outbuf[sizeof outbuf - 1] = '\0';
	    if (cc < 3) {
		outbuf[3] = '=';
		if (cc < 2)
		    outbuf[2] = '=';
	    }

	    Tcl_AppendResult (interp, outbuf, NULLCP);

	    if (--n <= 0) {
		n = BPERLIN;
		Tcl_AppendResult (interp, "\n", NULLCP);
	    }
	}
	if (n != BPERLIN)
	    Tcl_AppendResult (interp, "\n", NULLCP);	
    }
    else
	for (;;) {	/* cp must be null-terminated, len ignored... */
	    char   *dp;

	    if (dp = index (cp, '\n'))
		*dp++ = '\0';

	    if (strncmp (cp, "From ", sizeof "From " - 1) == 0) {
		(void) sprintf (buffer, "=%02X", *cp++ & 0xff);
		Tcl_AppendResult (interp, buffer, NULLCP);
		n = 3;
	    }
	    else
		n = 0;

	    for (; *cp; cp++) {
		if (n > CPERLIN - 3) {
		    Tcl_AppendResult (interp, "=\n", NULLCP);
		    n = 0;
		}

		switch (*cp) {
		    default:
			if (!ebcdicsafe[*cp & 0xff])
			    goto three_print;
			/* else fall... */
		    case ' ':
		    case '\t':
			(void) sprintf (buffer, "%c", *cp);
		        Tcl_AppendResult (interp, buffer, NULLCP);
			n++;
			break;

		    case '=':
three_print: ;
			(void) sprintf (buffer, "=%02X", *cp & 0xff);
			Tcl_AppendResult (interp, buffer, NULLCP);
			n += 3;
			break;
		}
	    }

	    if (dp) {
		if (n > 0 && (*--cp == ' ' || *cp == '\t'))
		    Tcl_AppendResult (interp, "=\n", NULLCP);

		Tcl_AppendResult (interp, "\n", NULLCP);
		if (!*(cp = dp))
		    break;
	    }
	    else {
		Tcl_AppendResult (interp, "=\n", NULLCP);
		break;
	    }
	}

    return TCL_OK;
}

/*  */

char	*SafeTcl_decode_aux ();

/* ARGSUSED */

static int  SafeTcl_decode (window, interp, argc, argv)
Tk_Window	window;
Tcl_Interp     *interp;
int		argc;
char	      **argv;
{
    int	    cc;
    char   *cp,
	    buffer[BUFSIZ];
    
    if (argc != 3) {
	(void) sprintf (buffer,
			"wrong # args: should be \"%s\" encodingName cookedData",
			argv[0]);
	goto tcl_error;
    }
    argv++;

    if (uleq (cp = argv[0], "7bit") || uleq (cp, "8bit") ||uleq (cp, "binary"))
	cp = argv[1];
    else
	if (!(cp = SafeTcl_decode_aux (interp, argv[0], argv[1], &cc)))
	    return TCL_ERROR;

    Tcl_SetResult (interp, cp, TCL_VOLATILE);
    return TCL_OK;

tcl_error: ;
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    return NOTOK;
}

/*  */

static char *SafeTcl_decode_aux (interp, mode, cp, cc)
Tcl_Interp     *interp;
char	       *mode,
	       *cp;
int	       *cc;
{
    int	    n;
    register char *bp;
    char   *ap,
	    buffer[BUFSIZ];

    ap = NULL;
    switch (*mode) {
	case 'b':
	    if (strcmp (mode, "base64") == 0)
		break;
	    goto losing;

	case 'q':
	    if (strcmp (mode, "quoted-printable") == 0)
		break;
	    goto losing;

	default:
losing: ;
	    (void) sprintf (buffer, "unknown encoding %s", mode);
	    goto tcl_error;
    }

    if (*mode == 'b') {
	int	bitno,
		skip;
	unsigned long	bits;
	char   *ep;
	unsigned char	value,
		       *b = (unsigned char *) &bits,
		       *b1 = &b[endian > 0 ? 1 : 2],
		       *b2 = &b[endian > 0 ? 2 : 1],
		       *b3 = &b[endian > 0 ? 3 : 0];

	n = (strlen (cp) * 3 + 3) >> 2;
	if (!(ap = malloc ((unsigned int) n + 1))) {
no_mem: ;
	    (void) sprintf (buffer, "out of memory");
	    goto tcl_error;
	}
	bp = ap;

	bitno = 18, bits = 0L, skip = 0;
	for (ep = cp + strlen (cp); cp < ep; cp++)
	    switch (*cp) {
		default:
		    if (isspace (*cp))
			continue;
		    if (skip
			    || (*cp & 0x80)
			    || (value = b642nib[*cp & 0x7f]) > 0x3f)
			continue;
		    bits |= value << bitno;
test_end: ;
		    if ((bitno -= 6) < 0) {
			*bp++ = *b1 & 0xff;

			if (skip < 2) {
			    *bp++ = *b2 & 0xff;

			    if (skip < 1)
				*bp++ = *b3 & 0xff;
			}

			bitno = 18, bits = 0L, skip = 0;
		    }
		    continue;

		case '=':
		    if (++skip > 3)
			goto self_delimiting;
		    goto test_end;
	    }
	if (bitno != 18) {
	    (void) strcpy (buffer, "invalid base64 encoding");
	    goto tcl_error;
	}
self_delimiting: ;

    }
    else {
	int	quoted = 0;
	char   *dp,
	       *ep;
	unsigned char mask = 0;

	if (!(ap = malloc ((unsigned int) ((n = strlen (cp)) + 1))))
	    goto no_mem;
	bp = ap;

	for (;;) {
	    if (dp = index (cp, '\n'))
		*dp++ = '\0';

	    for (ep = cp + strlen (cp) - 1; cp <= ep; ep--)
		if (!isspace (*ep))
		    break;
	    *++ep = '\n', ep++;

	    for (; cp < ep; cp++) {
		if (quoted) {
		    if (quoted > 1) {
			if (!isxdigit (*cp)) {
invalid_hex: ;
			    (void) strcpy (buffer,
					   "expecting hexidecimal-digit");
			    goto tcl_error;
			}
			mask <<= 4;
			mask |= hex2nib[*cp & 0x7f];
			*bp++ = mask & 0xff;
		    }
		    else
			switch (*cp) {
			    case ':':
				*bp++ = ':';
				break;

			    default:
				if (!isxdigit (*cp))
				    goto invalid_hex;
				mask = hex2nib[*cp & 0x7f];
				quoted = 2;
				continue;
			}

		    quoted = 0;
		    continue;
		}

		switch (*cp) {
		    default:
		        if (*cp < '!' || *cp > '~') {
invalid_encoding: ;
			    (void) strcpy (buffer,
					   "invalid quoted-printable encoding");
			    goto tcl_error;
			}
			/* and fall...*/
		    case ' ':
		    case '\t':
		    case '\n':
			*bp++ = *cp;
			break;

		    case '=':
			if (*++cp != '\n') {
			    quoted = 1;
			    cp--;
			}
			break;
		}
	    }

	    if (!(cp = dp) || !*cp)
		break;
	}
	if (quoted)
	    goto invalid_encoding;
    }

    *cc = bp - ap;
    *bp = '\0';
    return ap;

tcl_error: ;
    if (ap)
	free (ap);
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    return NULLCP;
}

/*  */

/* ARGSUSED */

int	SafeTclP_configdata (window, interp, argc, argv)
Tk_Window	window;
Tcl_Interp     *interp;
int		argc;
char	      **argv;
{
    int	   *icache;
    char   *cp,
	    buffer[BUFSIZ];

    if (argc != 3) {
	(void) sprintf (buffer,
			"wrong # args: should be \"%s\" key value",
			argv[0]);
	goto tcl_error;
    }
    argv++;

    switch (*(cp = *argv++)) {
	case 'f':
	    if (strcmp (cp, "filer") == 0) {
		static int didfiler = 0;

		if (mhuser)
		    break;

		if (didfiler)
		    free (fileproc);
		else
		    didfiler = 1;
		fileproc = getcpy (*argv);
		break;
	    }
	    if (strcmp (cp, "ftp") == 0) {
		static int didftp = 0;

		if (mhuser)
		    break;

		if (didftp)
		    free (ftp);
		else
		    didftp = 1;
		ftp = getcpy (*argv);
		break;
	    }
	    if (strcmp (cp, "folder") == 0) {
		static int didfolder = 0;

		if (didfolder)
		    free (folder);
		else
		    didfolder = 1;
		folder = getcpy (*argv);
		break;
	    }
	    break;

	case 'm':
	    if (strcmp (cp, "mailbox") == 0) {
		(void) strcpy (mailbox, *argv);
		break;
	    }
	    if (strcmp (cp, "mailer") == 0) {
		static int didmailer = 0;

		if (mhuser)
		    break;

		if (didmailer)
		    free (mailer);
		else
		    didmailer = 1;
		mailer = getcpy (*argv);
		break;
	    }
	    break;

	case 'p':
	    if (strcmp (cp, "printer") == 0) {
		static int didprinter = 0;

		if (didprinter)
		    free (printer);
		else
		    didprinter = 1;
		printer = getcpy (*argv);
		break;
	    }
	    if (strcmp (cp, "private-cache") == 0) {
		static int didcache = 0;

		if (mhuser)
		    break;

		if (didcache)
		    free (cache_private);
		else
		    didcache = 1;
		cache_private = getcpy (*argv);
		break;
	    }
	    if (strcmp (cp, "public-cache") == 0) {
		static int didcache = 0;

		if (mhuser)
		    break;

		if (didcache)
		    free (cache_public);
		else
		    didcache = 1;
		cache_public = getcpy (*argv);
		break;
	    }
	    break;

	case 'r':
	    if (strcmp (cp, "rcache-policy") == 0) {
		icache = &cache_rpolicy;
set_cache_policy: ;

		if (uleq (*argv, "never"))
		    *icache = CACHE_NEVER;
		else
		    if (uleq (*argv, "private"))
			*icache = CACHE_PRIVATE;
		    else
			if (uleq (*argv, "public"))
			    *icache = CACHE_PUBLIC;
		break;
	    }
	    if (strcmp (cp, "render") == 0) {
		static int didrender = 0;

		if (didrender)
		    free (render);
		else
		    didrender = 1;
		render = getcpy (*argv);
		break;
	    }
	    break;

	case 's':
	    if (strcmp (cp, "smtp-servers") == 0) {
		if (sm_servers)
		    free (sm_servers);
		sm_servers = getcpy (*argv);
		break;
	    }
	    break;

	case 'w':
	    if (strcmp (cp, "wcache-policy") == 0) {
		icache = &cache_wpolicy;
		goto set_cache_policy;
	    }
	    break;

	default:
	    break;
    }

    Tcl_ResetResult (interp);
    return TCL_OK;

tcl_error: ;
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    return TCL_ERROR;
}

/*  */

static int  SafeTcl_displaybody (window, interp, argc, argv)
Tk_Window	window;
Tcl_Interp     *interp;
int		argc;
char	      **argv;
{
    MIME_displayORprint (window, interp, argc, argv, render, "bodyPart");
}


/*    Recommended Extensions to the Trusted Tcl Interpreter */

#ifndef	SMTP

/* ARGSUSED */

static int  MIME_sendmessage (window, interp, argc, argv)
Tk_Window	window;
Tcl_Interp     *interp;
int		argc;
char	      **argv;
{
    int	    i,
	    pid,
	    queue,
	    resent,
	    status,
	    vecp;
    char   *body,
	   *cc,
	   *subject,
	   *to,
	    buffer[BUFSIZ],
	    tmpfil[BUFSIZ],
	   *vec[7];
    FILE   *fp;

    (void) strcpy (tmpfil, m_tmpfil (invo_name));
    if (!(fp = fopen (tmpfil, "w+"))) {
	(void) sprintf (buffer, "unable to write %s", tmpfil);
	goto tcl_error;
    }

    if (argc < 7) {
	(void) sprintf (buffer,
			"wrong # args: should be \"%s\" -to recipients ?-cc recipients? -subject string  ?-auxheader headerName headerValue...? ?-queue? ?-resent? -body bodyPart",
			argv[0]);
	goto tcl_error;
    }
    argc--, argv++;

    body = cc = subject = to = NULL;
    queue = resent = 0;
    for (; argc > 0; argc--, argv++) {
	if (strcmp (*argv, "-body") == 0) {
	    argc--, argv++;
	    if (argc <= 0)
		goto missing;
	    if (body) {
		(void) strcpy (buffer, "too many bodies");
		goto tcl_error;
	    }
	    body = *argv;
	    continue;
	}

	if (strcmp (*argv, "-cc") == 0) {
	    argc--, argv++;
	    if (argc <= 0)
		goto missing;
	    if (cc) {
		(void) strcpy (buffer, "too many secondary recipients");
		goto tcl_error;
	    }
	    cc = *argv;
	    continue;
	}

	if (strcmp (*argv, "-subject") == 0) {
	    argc--, argv++;
	    if (argc <= 0)
		goto missing;
	    if (subject) {
		(void) strcpy (buffer, "too many subjects");
		goto tcl_error;
	    }
	    subject = *argv;
	    continue;
	}

	if (strcmp (*argv, "-to") == 0) {
	    argc--, argv++;
	    if (argc <= 0)
		goto missing;
	    if (to) {
		(void) strcpy (buffer, "too many primary recipients");
		goto tcl_error;
	    }
	    to = *argv;
	    continue;
	}

	if (strcmp (*argv, "-auxheader") == 0) {
	    argc--, argv++;
	    if (argc <= 1)
		goto missing;
	    argc--, argv++;

	    fprintf (fp, "%s: %s\n", argv[-1], *argv);
	    continue;
	}

	if (strcmp (*argv, "-queue") == 0) {
	    queue = 1;
	    continue;
	}

	if (strcmp (*argv, "-resent") == 0) {
	    if (!mhuser) {
		(void) sprintf (buffer,
				"don't know how to do -resent, sorry...");
		goto tcl_error;
	    }

	    resent = 1;
	    continue;
	}

	(void) sprintf (buffer, "unknown option %s", *argv);
	goto tcl_error;
    }
    if (to)
	fprintf (fp, "To: %s\n", to);
    else
	if (!resent) {
	    (void) strcpy (buffer, "missing -to option");
	    goto tcl_error;
	}
    if (cc)
	fprintf (fp, "cc: %s\n", cc);
    if (subject)
	fprintf (fp, "Subject: %s\n", subject);
    else
	if (!resent) {
	    (void) strcpy (buffer, "missing -subject option");
	    goto tcl_error;
	}
    fprintf (fp, "Comments: generated by %s - %s\n", invo_name,
	     "Enabled Mail (EM) environment for UNIX");
    if (body) {
	fprintf (fp, "MIME-Version: 1.0\n");
	(void) fputs (body, fp);
	if ((i = strlen (body) - 1) >= 0 && body[i] != '\n')
	    (void) fputs ("\n", fp);
    }
    else {
	(void) strcpy (buffer, "missing -body option");
	goto tcl_error;
    }
    if (fflush (fp)) {
	(void) strcpy (buffer, "error writing to temporary file");
	goto tcl_error;
    }
    rewind (fp);

    for (i = 0; (pid = fork ()) == NOTOK && i < 5; i++)
	sleep (5);
    switch (pid) {
	case NOTOK:
	    (void) strcpy (buffer, "unable to fork");
	    goto tcl_error;

	case OK:
	    if (mhuser) {
		(void) fclose (fp);

		vecp = 0;
		vec[vecp++] = r1bindex (postproc, '/');
		vec[vecp++] = tmpfil;
		if (queue)
		    vec[vecp++] = "-queued";
		if (resent)
		    vec[vecp++] = "-dist";
		vec[vecp++] = "-mime";
		vec[vecp++] = "-msgid";
		vec[vecp] = NULL;

		execvp (postproc, vec);
	    }
	    else {
		int	sendmailP;
		char   *cp;

		sendmailP = strcmp (cp = r1bindex (mailer, '/'),
				    "sendmail") == 0;
		vecp = 0;
		vec[vecp++] = cp;
		if (sendmailP) {
		    vec[vecp++] = "-t";
		    if (queue)
			vec[vecp++] = "-odq";
		}
		vec[vecp] = NULL;

		(void) close (fileno (stdin));
		(void) dup (fileno (fp));
		(void) fclose (fp);

		execvp (mailer, vec);
	    }
	    _exit (-1);
	    /* NOTREACHED */

	default:
	    (void) fclose (fp), fp = NULL;
	    status = pidwait (pid, OK);
	    (void) unlink (tmpfil);
	    if (status) {
		record_status (interp, status, mhuser ? postproc : mailer);
		return TCL_ERROR;
	    }
	    break;
    }

    Tcl_ResetResult (interp);
    return TCL_OK;

missing: ;
    (void) sprintf (buffer, "missing argument to %s", argv[-1]);
    /* and fall... */
tcl_error: ;
    if (fp)
	(void) fclose (fp);
    (void) unlink (tmpfil);
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    return TCL_ERROR;
}

/*  */
#else	/* SMTP */

#define	addaddr(s,a)\
    if (Tcl_DStringLength (&s) > 0) \
    	Tcl_DStringAppend (&s, ",", 1); \
    Tcl_DStringAppend (&s, a, -1)
#define	addhead(s,h,v) \
    Tcl_DStringAppend (&s, h, -1), \
    Tcl_DStringAppend (&s, ": ", 2), \
    Tcl_DStringAppend (&s, v, -1), \
    Tcl_DStringAppend (&s, "\n", 1)


/* ARGSUSED */

static int  MIME_sendmessage (window, interp, argc, argv)
Tk_Window	window;
Tcl_Interp     *interp;
int		argc;
char	      **argv;
{
    int	    dateP,
	    fromP,
	    msgP,
	    queueP,
	    resentP,
	    result,
	    toP;
    long    clock;
    char   *body,
	   *cp,
	   *subject,
	    buffer[BUFSIZ],
	    from[BUFSIZ],
	    sender[BUFSIZ];
    struct mailname *mp;
    struct tws *tw;
    Tcl_DString *addrs,
		 addrs1,
		 addrs2,
		*bccs,
		 bccs1,
		 bccs2,
		 blind,
		 headers;
    static int msgno = 0;

    Tcl_DStringInit (&addrs1);
    Tcl_DStringInit (&addrs2);
    Tcl_DStringInit (&bccs1);
    Tcl_DStringInit (&bccs2);
    Tcl_DStringInit (&blind);
    Tcl_DStringInit (&headers);
    (void) sprintf (buffer,
		    "Comments: generated by %s - %s\nMIME-Version: 1.0\n",
		    invo_name, "Enabled Mail (EM) environment for UNIX");
    Tcl_DStringAppend (&headers, buffer, -1);

    if (argc < 7) {
	(void) sprintf (buffer,
			"wrong # args: should be \"%s\" -to recipients ?-cc recipients? -subject string  ?-auxheader headerName headerValue...? ?-queue? ?-resent? -body bodyPart",
			argv[0]);
	goto tcl_error;
    }
    argc--, argv++;

    dateP = fromP = msgP = queueP = resentP = toP = 0;
    body = subject = NULL;
    (void) strcpy (sender, adrsprintf (NULLCP, NULLCP));
    for (; argc > 0; argc--, argv++) {
	if (strcmp (*argv, "-body") == 0) {
	    argc--, argv++;
	    if (argc <= 0)
		goto missing;
	    if (body) {
		(void) strcpy (buffer, "too many bodies");
		goto tcl_error;
	    }
	    body = *argv;
	    continue;
	}

	if ((result = strcmp (*argv, "-cc")) == 0
	        || strcmp (*argv, "-to") == 0) {
	    argc--, argv++;
	    if (argc <= 0)
		goto missing;
	    toP++;
	    addhead (headers, result ? "To" : "cc", *argv);
	    addaddr (addrs1, *argv);
	    continue;
	}

	if (strcmp (*argv, "-subject") == 0) {
	    argc--, argv++;
	    if (argc <= 0)
		goto missing;
	    if (subject) {
		(void) strcpy (buffer, "too many subjects");
		goto tcl_error;
	    }
	    addhead (headers, "Subject", subject = *argv);
	    addhead (blind, "Subject", subject);
	    continue;
	}

	if (strcmp (*argv, "-auxheader") == 0) {
	    argc--, argv++;
	    if (argc <= 1)
		goto missing;
	    argc--, argv++;

		if (uleq (cp = argv[-1], "Dcc")) {
		    addaddr (addrs1, *argv);
		    continue;
		}
	    else
		if (uleq (cp, "Bcc")) {
		    addaddr (bccs1, *argv);
		    continue;
		}
	    else
		if (uleq (cp, "Resent-To")) {
		    addaddr (addrs2, *argv);
		}
	    else
		if (uleq (cp, "Resent-cc")) {
		    addaddr (addrs2, *argv);
		}
	    else
		if (uleq (cp, "Resent-Bcc")) {
		    addaddr (bccs2, *argv);
		    continue;
		}
	    else
		if (uleq (cp, "Resent-Dcc")) {
		    addaddr (addrs2, *argv);
		    continue;
		}
	    else
		if (uleq (cp, "From") || uleq (cp, "Resent-From")) {
		    char   *dp;

		    fromP++;
		    if (!(dp = getname (*argv))) {
			(void) sprintf (buffer, "null %s", cp);
			goto tcl_error;
		    }
		    if (!(mp = getm (dp, NULLCP, 0, AD_NHST, buffer))) {
			while (getname (*argv))
			    continue;
			goto tcl_error;
		    }
		    if (!(dp = getname (*argv)))
			(void) sprintf (sender, "%s@%s", mp -> m_mbox,
					mp -> m_host ? mp -> m_host
						     : LocalName ());
		    mnfree (mp);

		    addhead (blind, "From", *argv);
		}
	    else
		if (uleq (cp, "Date") || uleq (cp, "Resent-Date"))
		    dateP++;
	    else
		if (uleq (cp, "Message-ID") || uleq (cp, "Resent-Message-ID"))
		    msgP++;
	    addhead (headers, cp, *argv);
	    continue;
	}

	if (strcmp (*argv, "-queue") == 0) {
	    queueP = 1;
	    continue;
	}

	if (strcmp (*argv, "-resent") == 0) {
	    resentP = 1;
	    continue;
	}

	(void) sprintf (buffer, "unknown option %s", *argv);
	goto tcl_error;
    }

    if (!body) {
	(void) strcpy (buffer, "missing -body option");
	goto tcl_error;
    }

    if (resentP) {
	if (!Tcl_DStringLength (addrs = &addrs2)) {
	    (void) sprintf (buffer, "no resent addresses");
	    goto tcl_error;
	}
	bccs = &bccs2;
    }
    else {
	if (!Tcl_DStringLength (addrs = &addrs1)) {
	    (void) strcpy (buffer, "missing -to option");
	    goto tcl_error;
	}
	bccs = &bccs1;
	if (!toP)
	    addhead (headers, "Bcc", "");
	if (!subject) {
	    (void) strcpy (buffer, "missing -subject option");
	    goto tcl_error;
	}
    }
    while (cp = getname (Tcl_DStringValue (addrs)))
	if (!(mp = getm (cp, NULLCP, 0, AD_NHST, buffer))) {
	    while (getname (Tcl_DStringValue (addrs)))
		continue;
	    goto tcl_error;
	}
	else
	    mnfree (mp);

    if (!dateP || !msgP)
	(void) time (&clock);
    if (!dateP) {
	tw = dlocaltime (&clock);
	addhead (headers, resentP ? "Resent-Date" : "Date",
		 dasctime (tw, TW_ZONE));
    }
    if (!msgP) {
	(void) sprintf (buffer, "<%d.%ld.%d@%s>", getpid (), clock, ++msgno,
			LocalName ());
	addhead (headers, resentP ? "Resent-Message-ID" : "Message-ID",
		 buffer);
    }
    if (!fromP) {
	if ((cp = getfullname ()) && *cp) {
	    char    sigbuf[BUFSIZ];

	    (void) strcpy (sigbuf, cp);
	    (void) sprintf (from, "%s <%s>", sigbuf,
			    adrsprintf (NULLCP, NULLCP));
	}
	else
	    (void) strcpy (from, adrsprintf (NULLCP, NULLCP));
	addhead (headers, resentP ? "Resent-From" : "From", from);
    }
    Tcl_DStringAppend (&headers, body, -1);

    if (Tcl_DStringLength (bccs) > 0) {
	int	vecp;
	char   *vec[6];
	Tcl_DString digest;

	while (cp = getname (Tcl_DStringValue (bccs)))
	    if (!(mp = getm (cp, NULLCP, 0, AD_NHST, buffer))) {
		while (getname (Tcl_DStringValue (bccs)))
		    continue;
		goto tcl_error;
	    }
	    else
		mnfree (mp);

	(void) sprintf (buffer,
			"Comments: generated by %s - %s\nMIME-Version: 1.0\nBcc:\n",
			invo_name, "Enabled Mail (EM) environment for UNIX");
	Tcl_DStringAppend (&blind, buffer, -1);
	if (dateP && msgP)
	    (void) time (&clock);
	tw = dlocaltime (&clock);
	addhead (blind, "Date", dasctime (tw, TW_ZONE));
	(void) sprintf (buffer, "<%d.%ld.%d@%s>", getpid (), clock, ++msgno,
			LocalName ());
	addhead (blind, "Message-ID", buffer);
	if (!fromP)
	    addhead (blind, "From", from);

	vecp = 0;
	vec[vecp++] = "SafeTcl_makebody";
	vec[vecp++] = "message/rfc822";
	vec[vecp++] = "-description";
	vec[vecp++] = "Original Message";
	vec[vecp++] = Tcl_DStringValue (&headers);
	vec[vecp] = NULL;
	if ((result = SafeTcl_makebody (window, interp, vecp, vec)) != TCL_OK)
	    goto tcl_done;
	Tcl_DStringInit (&digest);
	Tcl_DStringAppend (&digest, interp -> result, -1);

	vecp = 0;
	vec[vecp++] = "SafeTcl_makebody";
	vec[vecp++] = "multipart/digest";
	vec[vecp++] = "-description";
	vec[vecp++] = "Blind Carbon Copy";
	vec[vecp++] = Tcl_DStringValue (&digest);
	vec[vecp] = NULL;
	result = SafeTcl_makebody (window, interp, vecp, vec);
	Tcl_DStringFree (&digest);
	if (result != TCL_OK)
	    goto tcl_done;
	Tcl_DStringAppend (&blind, interp -> result, -1);
    }
    
    result = TCL_ERROR;
    if (sm_send (interp, sender, addrs, headers, queueP,
		 Tcl_DStringLength (bccs) > 0) != TCL_OK)
	goto tcl_done;
    if (Tcl_DStringLength (bccs) > 0
	    && sm_send (interp, sender, bccs, blind, queueP, 0) != TCL_OK)
	goto tcl_done;

    result = TCL_OK;
    Tcl_ResetResult (interp);
    goto tcl_done;

missing: ;
    (void) sprintf (buffer, "missing argument to %s", argv[-1]);
    /* and fall... */
tcl_error: ;
    result = TCL_ERROR;
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    /* and fall... */
tcl_done: ;
    Tcl_DStringFree (&addrs1);
    Tcl_DStringFree (&addrs2);
    Tcl_DStringFree (&bccs1);
    Tcl_DStringFree (&bccs2);
    Tcl_DStringFree (&blind);
    Tcl_DStringFree (&headers);

    return result;
}

/*  */
static int sm_debug = 0;

#define	SM_OPEN	 90
#define	SM_HELO	 20
#define	SM_RSET	 15
#define	SM_MAIL	 40
#define	SM_RCPT	120
#define	SM_DATA	 20
#define	SM_TEXT	150
#define	SM_DOT	180
#define	SM_QUIT	 30
#define	SM_CLOS	 10


static int sm_addrs;
static int sm_alarmed = 0;
static int sm_ehlo;
static int sm_nl;
static int sm_reply;

static char sm_error[BUFSIZ];

#define	MAXEHLO	10
static char   *sm_keys[MAXEHLO + 1];

static FILE *sm_rfp = NULL;
static FILE *sm_wfp = NULL;


int	sm_init (), sm_end ();
int	sm_winit (), sm_wadr (), sm_waend (), sm_wtxt (), sm_wtend ();
int	smtalk (), sm_wrecord (), sm_wstream (), sm_werror ();
int	smhear (), sm_rrecord (), sm_rerror ();
char   *EHLOset ();
TYPESIG	alrmser ();

/*  */

static	sm_send (interp, sender, addrs, content, queued, more)
Tcl_Interp *interp;
char   *sender;
Tcl_DString *addrs,
	    *content;
int	queued,
	more;
{
    register int    cc,
		    n;
    register char  *bp;
    char   *ep;
    struct mailname *mp;

    if (sm_init (interp, !more, queued) != TCL_OK)
	return TCL_ERROR;

    if (sm_winit ("MAIL", sender) == NOTOK)
	goto tcl_error;
    
    while (ep = getname (Tcl_DStringValue (addrs))) {
	int	result;
	Tcl_DString a;

	if (!(mp = getm (ep, NULLCP, 0, AD_NHST, sm_error))) {
	    while (getname (Tcl_DStringValue (addrs)))
		continue;
	    goto tcl_error;
	}
	
	Tcl_DStringInit (&a);
	Tcl_DStringAppend (&a, mp -> m_mbox, -1);
	Tcl_DStringAppend (&a, "@", 1);
	Tcl_DStringAppend (&a, mp -> m_host ? mp -> m_host : LocalName (), -1);
	result = sm_wadr (Tcl_DStringValue (&a));
	Tcl_DStringFree (&a);

	mnfree (mp);

	if (result == NOTOK)
	    goto tcl_error;
    }

    if (sm_waend () == NOTOK)
	goto tcl_error;

    for (ep = (bp = Tcl_DStringValue (content))
	 	    + (n = Tcl_DStringLength (content));
	     bp < ep;
	     bp += cc, n -= cc) {
	if ((cc = BUFSIZ) > n)
	    cc = n;
	if (sm_wtxt (bp, cc) == NOTOK)
	    goto tcl_error;
    }
    
    if (sm_wtend () == NOTOK)
	goto tcl_error;

    if (!more)
	if (sm_end (OK) == NOTOK)
	    goto tcl_error;

    return TCL_OK;    

tcl_error: ;
    Tcl_SetResult (interp, sm_error, TCL_VOLATILE);
    (void) sm_end (NOTOK);
    return TCL_ERROR;
}

/*  */

static int  sm_init (interp, onex, queued)
Tcl_Interp *interp;
int	onex,
	queued;
{
    int	    result,
	    sd1,
	    sd2,
	    vecp;
    char   *client,
	   *servers,
	    buffer[BUFSIZ],
	    helo[BUFSIZ],
	  **ap,
	  **vec;
    struct servent *sp;

    if (sm_rfp != NULL && sm_wfp != NULL)
	return TCL_OK;

    if (!(sp = getservbyname ("smtp", "tcp"))) {
	Tcl_SetResult (interp, "tcp/smtp: unknown service", TCL_STATIC);
	return TCL_ERROR;
    }

    sd1 = NOTOK;
    Tcl_SetResult (interp, "smtp-servers list empty", TCL_STATIC);

    if (Tcl_SplitList (interp, sm_servers ? sm_servers : "localhost",
		       &vecp, &vec) != TCL_OK)
	return TCL_ERROR;
    for (ap = vec; *ap; ap++) {
	int	len;
	struct sockaddr_in in_socket;
	struct hostent *hp;

	if (!(hp = gethostbystring (*ap))) {
	    (void) sprintf (buffer, "%s: unknown host", *ap);
	    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
	    continue;
	}

	in_socket.sin_family = hp -> h_addrtype;
	inaddr_copy (hp, &in_socket);
	in_socket.sin_port = sp -> s_port;

	if ((sd1 = socket (AF_INET, SOCK_STREAM, 0)) == NOTOK
	        || connect (sd1, (struct sockaddr *) &in_socket,
			    sizeof in_socket) == NOTOK) {
	    interp -> result = Tcl_PosixError (interp);
	    if (sd1 == NOTOK)
		break;
	    (void) close (sd1);
	    sd1 = NOTOK;
	    continue;
	}

	client = "localhost";
	len = sizeof in_socket;
	if (getsockname (sd1, (struct sockaddr *) &in_socket, &len) != NOTOK
	        && in_socket.sin_addr.s_addr != INADDR_LOOPBACK
	        && gethostname (helo, sizeof helo) != NOTOK)
	    client = helo;
	break;
    }
    free ((char *) vec);

    if (sd1 == NOTOK)
	return TCL_ERROR;

    if ((sd2 = dup (sd1)) == NOTOK) {
	interp -> result = Tcl_PosixError (interp);
	(void) close (sd1);
	return TCL_ERROR;
    }

    (void) signal (SIGALRM, alrmser);
    (void) signal (SIGPIPE, SIG_IGN);

    if ((sm_rfp = fdopen (sd1, "r")) == NULL
	    || (sm_wfp = fdopen (sd2, "w")) == NULL) {
	interp -> result = Tcl_PosixError (interp);
	(void) close (sd1);
	(void) close (sd2);
	sm_rfp = sm_wfp = NULL;
	return TCL_ERROR;
    }
    sm_alarmed = 0;
    (void) alarm (SM_OPEN);
    result = smhear ();
    (void) alarm (0);
    switch (result) {
	case 220: 
	    break;

	default: 
	    (void) sm_end (NOTOK);
	    Tcl_SetResult (interp, sm_error, TCL_STATIC);
	    return TCL_ERROR;
    }

    sm_ehlo = 1;
    (void) sprintf (buffer, "EHLO %s", client);
    result = smtalk (SM_HELO, buffer);
    sm_ehlo = 0;

    if (500 <= result && result <= 599) {
	(void) sprintf (buffer, "HELO %s", client);
	result = smtalk (SM_HELO, buffer);
    }

    switch (result) {
        case 250:
	    break;

	default:
	    (void) sm_end (NOTOK);
	    Tcl_SetResult (interp, sm_error, TCL_STATIC);
	    return TCL_ERROR;
    }

    if (onex && EHLOset ("XONE"))
	(void) smtalk (SM_HELO, "ONEX");
    if (queued && EHLOset ("XQUE"))
	(void) smtalk (SM_HELO, "QUED");

    Tcl_ResetResult (interp);
    return TCL_OK;
}

/*  */

static int  sm_winit (mode, from)
register int	mode;
register char   *from;
{
    char buffer[BUFSIZ];

    (void) sprintf (buffer, "%s FROM:<%s>", mode, from);
    switch (smtalk (SM_MAIL, buffer)) {
	case 250: 
	    sm_addrs = 0;
	    return OK;

	default: 
	    return NOTOK;
    }
}


static int  sm_wadr (addr)
{
    char buffer[BUFSIZ];

    (void) sprintf (buffer, "RCPT TO:<%s>", addr);
    switch (smtalk (SM_RCPT, buffer)) {
	case 250: 
	case 251: 
	    sm_addrs++;
	    return OK;

	default: 
	    return NOTOK;
    }
}


static int  sm_waend () {
    switch (smtalk (SM_DATA, "DATA")) {
	case 354: 
	    sm_nl = 1;
	    return OK;

	default: 
	    return NOTOK;
    }
}

/*  */

static int  sm_wtxt (buffer, len)
register char   *buffer;
register int     len;
{
    register int    result;

    sm_alarmed = 0;
    (void) alarm (SM_TEXT);
    result = sm_wstream (buffer, len);
    (void) alarm (0);

    return result;
}


static int  sm_wtend () {
    if (sm_wstream ((char *) NULL, 0) == NOTOK)
	return NOTOK;

    switch (smtalk (SM_DOT + 3 * sm_addrs, ".")) {
	case 250: 
	case 251: 
	    return OK;

	default:
	    return NOTOK;
    }
}

/*  */

static int  sm_end (type)
register int     type;
{
    if (sm_rfp == NULL && sm_wfp == NULL)
	return OK;

    switch (type) {
	case OK: 
	    (void) smtalk (SM_QUIT, "QUIT");
	    break;

	case NOTOK: 
	case DONE: 
	    (void) smtalk (SM_RSET, "RSET");
	    (void) smtalk (SM_QUIT, "QUIT");
	    break;
    }
    if (sm_rfp != NULL) {
	(void) alarm (SM_CLOS);
	(void) fclose (sm_rfp);
	(void) alarm (0);
    }
    if (sm_wfp != NULL) {
	(void) alarm (SM_CLOS);
	(void) fclose (sm_wfp);
	(void) alarm (0);
    }

    sm_rfp = sm_wfp = NULL;

    return OK;
}

/*  */

/* VARARGS2 */

static int  smtalk (time, buffer)
register int     time;
char   *buffer;
{
    register int    result;

    if (sm_debug) {
	printf ("=> %s\n", buffer);
	(void) fflush (stdout);
    }

    sm_alarmed = 0;
    (void) alarm ((unsigned) time);
    if ((result = sm_wrecord (buffer, strlen (buffer))) != NOTOK)
	result = smhear ();
    (void) alarm (0);

    return result;
}

/*  */

static int  sm_wrecord (buffer, len)
register char   *buffer;
register int     len;
{
    if (sm_wfp == NULL)
	return sm_werror ();

    (void) fwrite (buffer, sizeof *buffer, len, sm_wfp);
    (void) fputs ("\r\n", sm_wfp);
    (void) fflush (sm_wfp);

    return (ferror (sm_wfp) ? sm_werror () : OK);
}


static int  sm_wstream (buffer, len)
register char   *buffer;
register int     len;
{
    register char  *bp;
    static char lc = 0;

    if (sm_wfp == NULL)
	return sm_werror ();

    if (buffer == NULL && len == 0) {
	if (lc != '\n')
	    (void) fputs ("\r\n", sm_wfp);
	lc = 0;
	return (ferror (sm_wfp) ? sm_werror () : OK);
    }

    for (bp = buffer; len > 0; bp++, len--) {
	switch (*bp) {
	    case '\n': 
		sm_nl = 1;
		(void) fputc ('\r', sm_wfp);
		break;

	    case '.': 
		if (sm_nl)
		    (void) fputc ('.', sm_wfp);
		/* and fall... */
	    default: 
		sm_nl = 0;
	}
	(void) fputc (*bp, sm_wfp);
	if (ferror (sm_wfp))
	    return sm_werror ();
    }

    if (bp > buffer)
	lc = *--bp;
    return (ferror (sm_wfp) ? sm_werror () : OK);
}

static int  sm_werror () {
    (void) strcpy (sm_error, sm_wfp == NULL ? "no socket opened"
			       : sm_alarmed ? "write to socket timed out"
			       : "error writing to socket");

    return NOTOK;
}

/*  */

static char *sm_noreply = "No reply text given";
static char *sm_moreply = "; ";

static int  smhear () {
    register int    i,
                    code,
                    cont,
		    more,
		    rc;
    int     bc;
    register char  *bp,
                   *rp;
    char  **ehlo,
	    buffer[BUFSIZ];

    if (sm_ehlo) {
	static	int	at_least_once = 0;

	if (at_least_once) {
	    char   *ep;

	    for (ehlo = sm_keys; ep = *ehlo; ehlo++)
		free (ep);
	}
	else
	    at_least_once = 1;

	*(ehlo = sm_keys) = NULL;
    }

again: ;
    *(rp = sm_error) = '\0', rc = sizeof sm_error - 1;
    for (more = 0; sm_rrecord (bp = buffer, &bc) != NOTOK;) {
	if (sm_debug) {
	    printf ("<= %s\n", buffer);
	    (void) fflush (stdout);
	}

	if (sm_ehlo
	        && strncmp (buffer, "250", sizeof "250" - 1) == 0
	        && (buffer[3] == '-' || sm_ehlo == 2)
	        && buffer[4]) {
	    if (sm_ehlo == 2) {
		int	len = strlen (buffer + 4);

		if (*ehlo = malloc ((unsigned) (strlen (buffer + 4) + 1))) {
		    (void) strcpy (*ehlo++, buffer + 4);
		    *ehlo = NULL;
		    if (ehlo >= sm_keys + MAXEHLO)
			sm_ehlo = 0;
		}
		else
		    sm_ehlo = 0;
	    }
	    else
		sm_ehlo = 2;
	}

	for (; bc > 0 && (!isascii (*bp) || !isdigit (*bp)); bp++, bc--)
	    continue;

	cont = 0;
	code = atoi (bp);
	bp += 3, bc -= 3;
	for (; bc > 0 && isspace (*bp); bp++, bc--)
	    continue;
	if (bc > 0 && *bp == '-') {
	    cont = 1;
	    bp++, bc--;
	    for (; bc > 0 && isspace (*bp); bp++, bc--)
		continue;
	}

	if (more) {
	    if (code != sm_reply || cont)
		continue;
	    more = 0;
	}
	else {
	    sm_reply = code;
	    more = cont;
	    if (bc <= 0) {
		(void) strcpy (bp = buffer, sm_noreply);
		bc = strlen (sm_noreply);
	    }
	}
#define	min(a,b)	((a) < (b) ? (a) : (b))
	if ((i = min (bc, rc)) > 0) {
	    (void) strncpy (rp, bp, i);
	    rp += i, rc -= i;
	    if (more && rc > strlen (sm_moreply) + 1) {
		(void) strcpy (sm_error + rc, sm_moreply);
		rc += strlen (sm_moreply);
	    }
	}
	if (more)
	    continue;
	if (sm_reply < 100)
	    goto again;

	*rp = '\0';

	return sm_reply;
    }

    return NOTOK;
}

/*  */

static int  sm_rrecord (buffer, len)
register char   *buffer;
register int    *len;
{
    if (sm_rfp == NULL)
	return sm_rerror ();

    buffer[*len = 0] = 0;

    (void) fgets (buffer, BUFSIZ, sm_rfp);
    *len = strlen (buffer);
    if (ferror (sm_rfp) || feof (sm_rfp))
	return sm_rerror ();
    if (buffer[*len - 1] != '\n')
	while (getc (sm_rfp) != '\n' && !ferror (sm_rfp) && !feof (sm_rfp))
	    continue;
    else
	if (buffer[*len - 2] == '\r')
	    *len -= 1;
    buffer[*len - 1] = 0;

    return OK;
}


static int  sm_rerror () {
    (void) strcpy (sm_error,
		   sm_wfp == NULL ? "no socket opened"
		     : sm_alarmed ? "read from socket timed out"
		     : feof (sm_rfp) ? "premature end-of-file on socket"
		     : "error reading from socket");

    return NOTOK;
}

/*  */

static char *EHLOset (s)
char *s;
{
    int	    len = strlen (s);
    register char  *ep,
		 **ehlo;

    for (ehlo = sm_keys; ep = *ehlo; ehlo++)
	if (strncmp (ep, s, len) == 0) {
	    for (ep += len; *ep == ' '; ep++)
		continue;
	    return ep;
	}

    return 0;
}

/*  */

static TYPESIG  alrmser (i)
int	i;
{
    sm_alarmed++;
}
#endif	/* SMTP */

/*  */

static int  MIME_printtext (window, interp, argc, argv)
Tk_Window	window;
Tcl_Interp     *interp;
int		argc;
char	      **argv;
{
    return MIME_displayORprint (window, interp, argc, argv, printer, "textString");
}


/* ARGSUSED */

static int  MIME_displayORprint (window, interp, argc, argv, command, optarg)
Tk_Window	window;
Tcl_Interp     *interp;
int		argc;
char	      **argv,
	       *command,
	       *optarg;
{
    int	    bg,
	    i,
	    pid,
	    renderP = command == render,
	    status;
    char   *cp,
	    buffer[BUFSIZ],
	    tmpfil[BUFSIZ];
    FILE   *fp = NULL;

    if (argc < 1 || argc > 3) {
usage: ;
	(void) sprintf (buffer,
			"wrong # args: should be \"%s\"%s ?%s?", argv[0],
			renderP ? " ?-background?" : "", optarg);
	goto tcl_error;
    }
    argc--, argv++;

    if (renderP && argc >= 1 && strcmp (*argv, "-background") == 0)
	bg = 1, argc--, argv++;
    else
	bg = 0;

    switch (argc) {
	case 0:
	    if (!(SafeTcl_message) || !(fp = SafeTcl_fp)) {
		(void) strcpy (buffer, "no default message");
		goto tcl_error;
	    }
	    (void) strcpy (tmpfil, SafeTcl_message);

	    rewind (fp);
	    break;

	case 1:
	    if (!(fp = get_message (interp, *argv, tmpfil)))
		return TCL_ERROR;
	    break;

	default:
	    goto usage;
    }

    Tcl_ReapDetachedProcs ();

    for (i = 0; (pid = fork ()) == NOTOK && i < 5; i++)
	sleep (5);
    switch (pid) {
	case NOTOK:
	    (void) strcpy (buffer, "unable to fork");
	    goto tcl_error;

	case OK:
	    if (bg || !renderP) {
		(void) close (fileno (stdin));
		(void) dup (fileno (fp));
		(void) fclose (fp);
	    }
	    cp = r1bindex (command, '/');
	    if (renderP) {
		if (bg)
		    (void) m_putenv ("MM_NOASK", "1");
		else
		    (void) fclose (fp);

		if (mhnuser)
		    execlp (command, cp, "-show", "-file", bg ? "-" : tmpfil,
			    "-form", "mhl.null", NULLCP);
		else
		    if (!generic && strcmp (cp, "metamail") == 0)
			execlp (command, cp, "-x", bg ? NULLCP : tmpfil,
				NULLCP);
		    else
			execlp (command, cp, bg ? "-" : tmpfil, NULLCP);
	    }
	    else {
		(void) close (fileno (stdin));
		(void) dup (fileno (fp));
		(void) fclose (fp);

		execlp (command, cp, NULLCP);
	    }
	    _exit (-1);
	    /* NOTREACHED */

	default:
	    if (fp != SafeTcl_fp)
		(void) fclose (fp);
	    if (bg) {
		status = OK;
		Tcl_DetachPids (1, &pid);
	    }
	    else
		status = pidwait (pid, OK);
	    if (fp != SafeTcl_fp)
		(void) unlink (tmpfil);
	    if (status) {
		record_status (interp, status, command);
		return TCL_ERROR;
	    }
	    break;
    }

    Tcl_ResetResult (interp);
    return TCL_OK;

tcl_error: ;
    if (fp && fp != SafeTcl_fp)
	(void) fclose (fp);
    if (fp != SafeTcl_fp)
	(void) unlink (tmpfil);
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    return TCL_ERROR;
}

/*  */

/* ARGSUSED */

static int  MIME_savemessage (window, interp, argc, argv)
Tk_Window	window;
Tcl_Interp     *interp;
int		argc;
char	      **argv;
{
    int	    i,
	    mapping,
	    md,
	    pid,
	    status;
    char   *cp,
	   *dest,
	   *from,
	   *type,
	    buffer[BUFSIZ],
	   *vec[20];
    
    if (argc < 2 || argc > 14) {
	(void) sprintf (buffer,
			"wrong # args: should be \"%s\" destinationType ?destination? ?folder options...?",
			argv[0]);
	goto tcl_error;
    }
    argv++;

    switch (*(type = *argv++)) {
	case 'c':
	    if (strcmp (type, "cache") == 0)
		break;
	    goto losing;

	case 'f':
	    if (strcmp (type, "folder") == 0)
		break;
	    goto losing;

	case 'm':
	    if (strcmp (type, "mailbox") == 0 || strcmp (type, "mbox") == 0)
		break;
	    goto losing;

	default:
losing: ;
	    (void) sprintf (buffer, "unknown destination type %s", type);
	    goto tcl_error;
    }

    if (!SafeTcl_message) {
	(void) strcpy (buffer, "no default message");
	goto tcl_error;
    }

    if (argc == 2) {
	switch (*type) {
	    case 'c':
		break;

	    case 'f':
	        if (!folder) {
		    if (!mhuser) {
			if (defalt && *defalt)
			    folder = defalt;
			else {
			    (void) strcpy (buffer, "no default folder");
			    goto tcl_error;
			}
		    }
		    else
			if (!(folder = m_find (inbox)))
			    folder = defalt;
		}
		dest = folder;
		break;

	    case 'm':
		if (!mailbox[0] && (cp = getenv ("MAILDROP")) && *cp)
		    (void) strcpy (mailbox, cp);
		if (!mailbox[0] && mhuser && (cp = m_find ("maildrop")) && *cp)
		    (void) strcpy (mailbox, cp);
		if (!mailbox[0] && (cp = getenv ("HOME")) && *cp)
		    (void) sprintf (mailbox, "%s/%s", MAILDIR, MAILFIL);
		else {
		    struct passwd *pw;

		    if (pw = getpwuid (getuid ()))
			(void) sprintf (mailbox, "%s/%s",
					mmdfldir[0] ? mmdfldir : pw -> pw_dir,
					mmdflfil[0] ? mmdflfil :pw -> pw_name);
		    else {
			(void) strcpy (buffer, "no default mailbox");
			goto tcl_error;
		    }
		}
		dest = mailbox;
		break;
	}
    }
    else
	switch (*type) {
	    case 'c':
		(void) sprintf (buffer,
				"wrong # args: should be \"%s\" cache",
				"MIME_savemessage");
		goto tcl_error;

	    case 'm':
		if (argc != 3) {
		    (void) sprintf (buffer,
				    "wrong # args: should be \"%s\" mailbox ?name?",
				    "MIME_savemessage");
		    goto tcl_error;
		}
		/* else fall... */
	    case 'f':
	        dest = *argv++;
		break;
	}
    

    switch (*type) {
	case 'c':
	    {
		int	cachetype,
			cc;
		long	len;
		char   *cp,
		       *dp,
			cachefile[BUFSIZ];
		struct bodypart *c;
		FILE   *gp;

		if (!(c = SafeTcl_bp)
		        && !(c = get_parts (interp, SafeTcl_fp, 1, "1")))
		    return TCL_ERROR;
		if (!c -> bp_id) {
		    (void) strcpy (buffer, "no Content-ID: field");
		    goto tcl_error;
		}
		cp = NULL;
		if ((len = c -> bp_end - c -> bp_begin) <= 0)
		    goto make_cache;

		cc = len;
		if (!(cp = malloc ((unsigned int) (cc + 1)))) {
		    (void) strcpy (buffer, "out of memory");
oops: ;
		    free (cp);
		    goto tcl_error;
		}
		if (fseek (SafeTcl_fp, c -> bp_begin, 0) == NOTOK) {
		    (void) strcpy (buffer, "error positioning message");
		    goto oops;
		}
		if (fread (cp, sizeof *cp, cc, SafeTcl_fp) != cc) {
		    (void) strcpy (buffer, "error reading message");
		    goto oops;
		}
		*(cp + cc) = '\0';

		if (c -> bp_encoding
		        && !uleq (c -> bp_encoding, "7bit")
		        && !uleq (c -> bp_encoding, "8bit")
		        && !uleq (c -> bp_encoding, "binary")) {
		    if (!(dp = SafeTcl_decode_aux (interp, c -> bp_encoding,
						  cp, &cc))) {
			free (cp);
			return TCL_ERROR;
		    }
		    free (cp);
		    cp = dp;
		}

make_cache: ;
		if (find_cache (cache_wpolicy, &cachetype, c -> bp_id,
				cachefile) == NOTOK) {
		    (void) strcpy (buffer, cachefile);
		    goto oops;
		}
		if (!(gp = fopen (cachefile, "w"))) {
		    (void) sprintf (buffer, "unable to write %s", cachefile);
		    goto oops;
		}
		(void) chmod (cachefile, cachetype ? m_gmprot () : 0444);
		
		if (cp && fwrite (cp, sizeof *cp, cc, gp) != cc) {
		    (void) fclose (gp);
		    (void) unlink (cachefile);
		    (void) strcpy (buffer, "error writing cache");
		    goto oops;
		}

		if (cp)
		    free (cp);
		(void) fclose (gp);
	    }
	    break;
	    
	case 'f':
	    if (!mhuser && (!fileproc || !*fileproc)) {
		(void) strcpy (buffer, "filer configure setting undefined");
		goto tcl_error;
	    }

	    (void) sprintf (buffer, "%s%s",
			    *dest == '+' || *dest == '@' ? "" : "+", dest);

	    i = 0;
	    vec[i++] = r1bindex (fileproc, '/');
	    if (mhuser) {
		vec[i++] = "-link";
		vec[i++] = "-file";
	    }
	    vec[i++] = SafeTcl_message;
	    vec[i++] = buffer;
	    for (; vec[i] = *argv; argv++, i++)
		continue;

	    for (i = 0; (pid = fork ()) == NOTOK && i < 5; i++)
		sleep (5);
	    switch (pid) {
		case NOTOK:
		    (void) strcpy (buffer, "unable to fork");
		    goto tcl_error;

		case OK:
		    execvp (fileproc, vec);
		    _exit (-1);
		    /* NOTREACHED */

		default:
		    if (status = pidwait (pid, OK)) {
			record_status (interp, status, fileproc);
			return TCL_ERROR;
		    }
		    break;
	    }
	    break;

	case 'm':
	    if (strcmp (type, "mailbox") == 0
		    && (from = Tcl_GetVar (interp, "SafeTcl_Originator",
					   TCL_GLOBAL_ONLY))) {
		(void) mbx_uucp ();
		(void) sprintf (cp = buffer, "From %s %s", from, udate);
		mapping = 0;
	    }
	    else
		cp = ddate, mapping = 1;

	    if (uid == NOTOK)
		uid = getuid (), gid = getgid ();
	    if ((md = mbx_open (dest, uid, gid, m_gmprot ())) == NOTOK) {
		(void) sprintf (buffer, "unable to open %s for writing", dest);
		goto tcl_error;
	    }
	    
	    (void) lseek (fileno (SafeTcl_fp), 0L, 0);
	    if (mbx_copy (dest, md, fileno (SafeTcl_fp), mapping, cp, 0)
		    == NOTOK) {
		(void) mbx_close (dest, md);
		(void) sprintf (buffer, "error writing to %s", dest);
		goto tcl_error;
	    }

	    (void) mbx_close (dest, md);
	    break;
    }

    Tcl_ResetResult (interp);
    return TCL_OK;

tcl_error: ;
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    return TCL_ERROR;
}

/*    swish extensions for receipt-time */

/*  */

/* ARGSUSED */

static int  swish_filebody (window, interp, argc, argv)
Tk_Window	window;
Tcl_Interp     *interp;
int		argc;
char	      **argv;
{
    int	    asciiP,
	    cc,
	    cperbuf;
    char   *cp,
	   *dp,
	   *ep,
	   *ip,
	    buffer[BUFSIZ],
	    tmpfil[BUFSIZ];
    long    pos;
    FILE   *fp,
	   *gp;

    gp = NULL;
    if (argc < 3) {
	(void) sprintf (buffer,
			"wrong # args: should be \"%s\" typeString fileString ?-description string? ?-id string?",
			argv[0]);
	goto tcl_error;
    }
    argc--, argv++;

    (void) strcpy (tmpfil, m_tmpfil (invo_name));
    if (!(gp = fopen (tmpfil, "w"))) {
	(void) sprintf (buffer, "unable to write %s", tmpfil);
	goto tcl_error;
    }

    fprintf (gp, "Content-Type: %s\n", **argv ? *argv : "text/plain");
    argc--, argv++;

    ep = *argv;
    argc--, argv++;

    for (dp = ip = NULL; argc > 0; argc--, argv++) {
	if (strcmp (*argv, "-description") == 0) {
	    if (dp) {
		(void) strcpy (buffer, "too many descriptions");
		goto tcl_error;
	    }
	    argc--, argv++;
	    if (argc <= 0) {
		(void) strcpy (buffer, "missing argument to -description");
		goto tcl_error;
	    }
	    dp = *argv;
	    continue;
	}

	if (strcmp (*argv, "-id") == 0) {
	    if (ip) {
		(void) strcpy (buffer, "too many IDs");
		goto tcl_error;
	    }
	    argc--, argv++;
	    if (argc <= 0) {
		(void) strcpy (buffer, "missing argument to -id");
		goto tcl_error;
	    }
	    ip = *argv;
	    continue;
	}

	break;
    }
    if (dp)
	fprintf (gp, "Content-Description: %s\n", dp);
    if (!ip)
	(void) sprintf (ip = buffer, msgfmt, ++bodyno);
    fprintf (gp, "Content-ID: %s\n", ip);
    pos = ftell (gp);
    (void) fputs ("\n", gp);

    if (!(fp = fopen (ep, "r"))) {
	(void) sprintf (buffer, "unable to read %s", ep);
	goto tcl_error;
    }
    asciiP = 1;
    while ((cc = fread (buffer, sizeof buffer[0], sizeof buffer, fp)) > 0) {
	for (dp = (cp = buffer) + cc; cp < dp; cp++)
	    if (!isascii (*cp) || (iscntrl (*cp) && !isspace (*cp))) {
		asciiP = 0;
		break;
	    }
	if (!asciiP)
	    break;

	if (fwrite (buffer, sizeof buffer[0], cc, gp) != cc) {
error_writing: ;
	    (void) sprintf (buffer, "error writing %s", tmpfil);
	    goto tcl_losing;
	}
    }
    if (cc < 0) {
error_reading: ;
	(void) sprintf (buffer, "error reading %s", ep);
	goto tcl_losing;
    }
    if (asciiP)
	goto all_done;
	
    if (fseek (gp, pos, 0) == NOTOK) {
	(void) sprintf (buffer, "error positioning %s", tmpfil);
	goto tcl_losing;
    }
    fprintf (gp, "Content-Transfer-Encoding: base64\n\n");

    rewind (fp);
    cperbuf = BUFSIZ - (BUFSIZ % (BPERLIN * 3));
    while ((cc = fread (buffer, sizeof buffer[0], cperbuf, fp)) > 0) {
	Tcl_ResetResult (interp);
	(void) SafeTcl_encode_aux (interp, "base64", buffer, cc);
	(void) fputs (interp -> result, gp);
    }
    if (cc < 0)
	goto error_reading;
    if (fflush (gp))
	goto error_writing;

all_done: ;
    (void) fclose (fp);
    (void) fclose (gp);

    Tcl_SetResult (interp, tmpfil, TCL_VOLATILE);
    return TCL_OK;

tcl_losing: ;
    (void) fclose (fp);

tcl_error: ;
    if (gp) {
	(void) fclose (gp);
	(void) unlink (tmpfil);
    }
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    return TCL_ERROR;
}

/*  */

/* ARGSUSED */

int	swish_ftruncate (window, interp, argc, argv)
Tk_Window	window;
Tcl_Interp     *interp;
int		argc;
char	      **argv;
{
    int	    length;
    char    buffer[BUFSIZ];
    FILE   *filePtr;

    if (argc > 3) {
	(void) sprintf (buffer,
			"wrong # args: should be \"%s\" handle ?length?",
			argv[0]);
	goto tcl_error;
    }

    if (Tcl_GetOpenFile (interp, argv[1], 1, 1, &filePtr) != TCL_OK)
	return TCL_ERROR;
    if (argc < 3)
	length = 0;
    else
	if (Tcl_GetInt (interp, argv[2], &length) != TCL_OK)
	    return TCL_ERROR;
        else
	    if (length < 0) {
		(void) strcpy (buffer, "length should be non-negative");
		goto tcl_error;
	    }

    (void) fflush (filePtr);
    if (ftruncate (fileno (filePtr), length) == NOTOK)
        goto unix_error;

    Tcl_ResetResult (interp);
    return TCL_OK;

unix_error: ;
    interp -> result = Tcl_PosixError (interp);
    return TCL_ERROR;

tcl_error: ;
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    return TCL_ERROR;
}

/*    MISC */

static void  free_part (c)
struct bodypart *c;
{
    register char  **ap;
    struct bodypart *bp;

    if (c == SafeTcl_bp)
	return;

    if (c -> bp_part)
	free (c -> bp_part);

    if (c -> bp_type)
	free (c -> bp_type);
    for (ap = c -> bp_attrs; *ap; ap++)
	free (*ap);
    if (c -> bp_id)
	free (c -> bp_id);
    if (c -> bp_descr)
	free (c -> bp_descr);
    if (c -> bp_encoding)
	free (c -> bp_encoding);
    if (c -> ep_body)
	free (c -> ep_body);

    for (bp = c -> bp_children; bp; bp = bp -> bp_sibling)
	free_part (bp);

    if (c -> bp_file) {
	if (c -> bp_flags & FP_UNLK)
	    (void) unlink (c -> bp_file);
	if (c -> bp_flags & FP_FREE)
	    (void) free (c -> bp_file);
    }

    free ((char *) c);
}

/*  */

static void  record_status (interp, status, program)
Tcl_Interp     *interp;
int	status;
char   *program;
{
    int	    signum;
    char    buffer[BUFSIZ];

    switch (signum = status & 0x007f) {
	case OK:
	    if ((signum = (status & 0xff00) >> 8) != 255)
		(void) sprintf (buffer, "Exit %d", signum);
	    else
		(void) sprintf (buffer, "unable to exec %s", program);
	    break;

	default:
	    (void) sprintf (buffer, "Signal %d", signum);
	    break;
    }

    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
}

/*  */

static void  scan_parts (interp, c)
Tcl_Interp     *interp;
struct bodypart *c;
{
    int    vecp;
    long   size = c -> bp_end - c -> bp_begin;
    char  *cp,
	   buffer[40],
	  *vec[5];
    struct bodypart *bp;

    vecp = 0;
    vec[vecp++] = c -> bp_part;
    vec[vecp++] = c -> bp_type;
    vec[vecp++] = c -> bp_descr ? c -> bp_descr : "";
    if (c -> bp_encoding && (strcmp (c -> bp_encoding, "base64") == 0))
	size = (size * 3) >> 2;
    (void) sprintf (buffer, "%ld", (size + 999) / 1000);
    vec[vecp++] = buffer;
    vec[vecp] = NULL;
    Tcl_AppendElement (interp, cp = Tcl_Merge (vecp, vec));
    free (cp);
    
    for (bp = c -> bp_children; bp; bp = bp -> bp_sibling)
	scan_parts (interp, bp);
}

/*  */

static int  get_comment (interp, ap)
Tcl_Interp     *interp;
char  **ap;
{
    register int    i;
    register char  *bp,
		   *cp;
    char    c,
	    buffer[BUFSIZ];

    cp = *ap;

    bp = buffer;
    cp++;
    for (i = 0;;) {
	switch (c = *cp++) {
	    case '\0':
invalid: ;
		Tcl_SetResult (interp, "invalid comment", TCL_STATIC);
		return NOTOK;

	    case '\\':
		*bp++ = c;
		if ((c = *cp++) == '\0')
		    goto invalid;
		*bp++ = c;
		continue;

	    case '(':
		i++;
		/* and fall... */
    	    default:
		*bp++ = c;
		continue;

	    case ')':
		if (--i < 0)
		    break;
		*bp++ = c;
		continue;
	}
	break;
    }
    *bp = '\0';

    while (isspace (*cp))
	cp++;

    *ap = cp;

    return OK;
}

/*  */

static int  get_x400_comp (mbox, key, buffer)
char   *mbox,
       *key,
       *buffer;
{
    int	    idx;
    char   *cp;

    if ((idx = stringdex (key, mbox)) < 0
	    || !(cp = index (mbox += idx + strlen (key), '/')))
	return 0;

    (void) sprintf (buffer, "%*.*s", cp - mbox, cp - mbox, mbox);
    return 1;
}

/*  */

static int  find_cache (policy, writing, id, buffer)
int	policy,
       *writing;
char   *id,
       *buffer;
{
    int	    status = NOTOK;

    if (writing)
	buffer[0] = '\0';
    if (!id)
	return NOTOK;
    id = trimcpy (id);

    switch (policy) {
	case CACHE_NEVER:
	default:
	    if (writing)
		(void) strcpy (buffer, "cache writing not allowed");
	    break;

	case CACHE_PUBLIC:
	    if (writing)
		(void) strcpy (buffer, "public cache not defined");
	    if (cache_private
		    && !writing
		    && find_cache_aux (writing ? 2 : 0, cache_private, id,
				       buffer) == OK) {
		if (access (buffer, 04) != NOTOK) {
got_private: ;
		    if (writing)
			*writing = 1;
got_it: ; 
		    status = OK;
		    break;
		}
	    }
	    if (cache_public
		    && find_cache_aux (writing ? 1 : 0, cache_public, id,
				       buffer) == OK) {
		if (writing || access (buffer, 04) != NOTOK) {
		    if (writing)
			*writing = 0;
		    goto got_it;
		}
	    }
	    break;

	case CACHE_PRIVATE:
	    if (writing)
		(void) strcpy (buffer, "private cache not defined");
	    if (cache_private
		    && find_cache_aux (writing ? 2 : 0, cache_private, id,
				       buffer) == OK) {
		if (writing || access (buffer, 04) != NOTOK)
		    goto got_private;
	    }
	    break;

    }

    if (status == OK && writing) {
	if (*writing && index (buffer, '/'))
	    (void) make_intermediates (buffer);
	(void) unlink (buffer);
    }

    free (id);
    return status;
}

/*  */

static int  find_cache_aux (writing, directory, id, buffer)
int	writing;
char   *directory,
       *id,
       *buffer;
{
    int	    mask;
#if	defined(BSD42) || defined(FTP)
    int	    usemap = index (id, '/') ? 1 : 0;
#else
    int	    usemap = 1;
#endif
    char    mapfile[BUFSIZ],
	    mapname[BUFSIZ];
    FILE   *fp;
    static int	partno,
		pid;
    static long clock = 0L;

    (void) sprintf (mapfile, "%s/cache.map", directory);
    if (find_cache_aux2 (mapfile, id, mapname) == OK)
	goto done_map;

    if (!writing) {
	if (usemap)
	    return NOTOK;

use_raw: ;
	(void) sprintf (buffer, "%s/%s", directory, id);
	return OK;
    }

    if (!usemap && access (mapfile, 02) == NOTOK)
	goto use_raw;

    if (clock != 0L) {
	long	now;
	
	(void) time (&now);
	if (now > clock)
	    clock = 0L;
    }
    else
	pid = getpid ();
    if (clock == 0L) {
	(void) time (&clock);
	partno = 0;
    }
    else
	if (partno > 0xff)
	    clock++, partno = 0;

    (void) sprintf (mapname, "%08x%04x%02x", clock & 0xffffffff,
		    pid & 0xffff, partno++ & 0xff);

    (void) make_intermediates (mapfile);
    mask = umask (writing == 2 ? 0077 : 0);
    if (!(fp = lkfopen (mapfile, "a")) && errno == ENOENT) {
	int	fd = creat (mapfile, 0666);

	if (fd != NOTOK) {
	    (void) close (fd);
	    fp = lkfopen (mapfile, "a");
	}
    }
    (void) umask (mask);
    if (!fp) {
	(void) sprintf (buffer, "unable to append to %s", mapfile);
	return NOTOK;
    }
    fprintf (fp, "%s: %s\n", mapname, id);
    (void) lkfclose (fp, mapfile);

done_map: ;
    if (*mapname == '/')
	(void) strcpy (buffer, mapname);
    else
	(void) sprintf (buffer, "%s/%s", directory, mapname);

    return OK;
}

/*  */

static int  find_cache_aux2 (mapfile, id, mapname)
char   *mapfile,
       *id,
       *mapname;
{
    int	    state;
    char    buf[BUFSIZ],
	    name[NAMESZ];
    FILE   *fp;

    if (!(fp = lkfopen (mapfile, "r")))
	return NOTOK;

    for (state = FLD;;) {
	int	result;
	register char  *cp,
		       *dp;

	switch (state = m_getfld (state, name, buf, sizeof buf, fp)) {
	    case FLD:
	    case FLDPLUS:
	    case FLDEOF:
	        (void) strcpy (mapname, name);
		if (state != FLDPLUS)
		    cp = buf;
		else {
		    cp = add (buf, NULLCP);
		    while (state == FLDPLUS) {
			state = m_getfld (state, name, buf, sizeof buf, fp);
			cp = add (buf, cp);
		    }
		}
		dp = trimcpy (cp);
		if (cp != buf)
		    free (cp);
		result = strcmp (id, dp);
		free (dp);
		if (result == 0) {
		    (void) lkfclose (fp, mapfile);
		    return OK;
		}
		if (state != FLDEOF)
		    continue;
		/* else fall... */

	    case BODY:
	    case BODYEOF:
	    case FILEEOF:
	    default:
		break;
	}
	break;
    }

    (void) lkfclose (fp, mapfile);
    return NOTOK;
}

static int  make_intermediates (file)
char   *file;
{
    register char *cp;

    for (cp = file + 1; cp = index (cp, '/'); cp++) {
	struct stat st;

	*cp = '\0';

	if (stat (file, &st) == NOTOK
	        && (errno != ENOENT || !makedir (file))) {
	    *cp = '/';
	    return NOTOK;
	}

	*cp = '/';
    }

    return OK;
}

/*  */

static int monlens[] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};

static int  compute_yday (tw)
register struct tws *tw;
{
    int i, tot = -1;

    for (i = 0; i< tw -> tw_mon; ++i)
	tot += monlens[i];
    tot += tw->tw_mday;

    /* Leap year adjustment */
    if (tw -> tw_mon > 1 /* after February */
	    && tw -> tw_year % 4 == 0
	    && ((tw -> tw_year % 100 != 0) || (tw -> tw_year % 400 == 0)))
	tot++;

    return tot;
}

/*  */

static struct bodypart *find_part (interp, c, part)
Tcl_Interp     *interp;
struct bodypart *c;
char   *part;
{
    char    buffer[BUFSIZ];
    struct bodypart *bp;

    if (*part == '<') {
	if (c -> bp_id && strcmp (c -> bp_id, part) == 0)
	    return c;
    }
    else {
	if (strcmp (c -> bp_part, part) == 0)
	    return c;

	(void) sprintf (buffer, "%s.", c -> bp_part);
	if (strncmp (part, buffer, strlen (buffer)))
	    return NULL;
    }

    for (bp = c -> bp_children; bp; bp = bp -> bp_sibling)
	if (c = find_part (interp, bp, part))
	    return c;

    return NULL;
}

/*  */

static struct pair {
    char   *p_field;
    char  **p_value;
}	pairs[] = {
    "Content-Type",		 (char **) 0,
    "Content-ID",		 (char **) 0,
    "Content-Description",	 (char **) 0,
    "Content-Transfer-Encoding", (char **) 0,
    NULL
};

static struct kv {
    char   *k_key;
    int	    k_value;
}	kvs[] = {
    "afs",		EP_FILE,
    "anon-ftp",		EP_ANON,
    "local-file", 	EP_FILE,
    "mail-server",	EP_MAIL,
    NULLCP,		EP_UNKN
};




static struct bodypart *get_parts (interp, fp, toplevel, partno)
Tcl_Interp     *interp;
FILE   *fp;
int	toplevel;
char   *partno;
{
    int     compnum,
	    state;
    char    ch,
	   *cp,
	   *dp,
	    buffer[BUFSIZ],
	    name[NAMESZ],
	  **ap,
	  **ep;
    struct bodypart *c;

    if (toplevel > 0 && fp == SafeTcl_fp && SafeTcl_bp)
	return SafeTcl_bp;

    if (!(c = (struct bodypart *) calloc (1, sizeof *c))) {
	Tcl_SetResult (interp, "out of memory", TCL_STATIC);
	return NULL;
    }
    pairs[0].p_value = &c -> bp_type;
    pairs[1].p_value = &c -> bp_id;
    pairs[2].p_value = &c -> bp_descr;
    pairs[3].p_value = &c -> bp_encoding;

    c -> bp_begin = (c -> bp_start = ftell (fp)) + 1;
    for (compnum = 1, state = FLD;;) {
	register struct pair *p;

	switch (state = m_getfld (state, name, buffer, sizeof buffer, fp)) {
	    case FLD:
	    case FLDPLUS:
	    case FLDEOF:
	        compnum++;

		for (p = pairs; p -> p_field; p++)
		    if (uleq  (name, p -> p_field)) {
			cp = add (buffer, NULLCP);
			while (state == FLDPLUS) {
			    state = m_getfld (state, name, buffer,
					      sizeof buffer, fp);
			    cp = add (buffer, cp);
			}

			if (*p -> p_value) {
			    free (cp);
			    (void) sprintf (buffer, "multiple %s fields",
					    p -> p_field);
			    goto out;
			}

			if (uleq (p -> p_field, "Content-Description"))
			    *p -> p_value = trimcpy (cp);
			else {
			    for (dp = cp; isspace (*dp); dp++)
				continue;
			    if (*dp == '('
				    && get_comment (interp, &dp) == NOTOK) {
				free (cp);
				goto out;
			    }
			    *p -> p_value = trimcpy (dp);
			}
			free (cp);
			break;
		    }
		if (!p -> p_field)
		    while (state == FLDPLUS)
			state = m_getfld (state, name, buffer, sizeof buffer,
					  fp);
		if (state != FLDEOF) {
		    c -> bp_begin = ftell (fp) + 1;
		    continue;
		}
		/* else fall... */

	    case BODY:
	    case BODYEOF:
		c -> bp_begin = ftell (fp) - strlen (buffer);
		break;

	    case FILEEOF:
		c -> bp_begin = ftell (fp);
		break;

	    case LENERR:
	    case FMTERR:
		(void) sprintf (buffer,
				"message format error in component #%d",
				compnum);
		goto out;

	    default:
		(void) sprintf (buffer, "m_getfld returned %d", state);
		goto out;
	}
	break;
    }

    if (!c -> bp_type)
	c -> bp_type = getcpy (toplevel < 0 ? "message/rfc822"
			       		    : "text/plain; charset=us-ascii");

    cp = c -> bp_type;

    for (dp = cp; istoken (*dp); dp++)
	continue;
    ch = *dp, *dp = '\0';
    (void) strcpy (name, cp);
    *dp = ch, cp = dp;
    if (!name[0]) {
	(void) strcpy (buffer, "invalid Content-Type field (empty type)");
	goto out;
    }
    while (isspace (*cp))
	cp++;

    if (*cp == '(' && get_comment (interp, &cp) == NOTOK)
	goto out;
    if (*cp != '/') {
	if (uleq (name, "text")) {		/* XXX: attmail bogosity! */
	    (void) strcpy (buffer, "plain");
	    goto carry_on;
	}
	else
	    if (uleq (name, "message")) {	/*   .. */
		(void) strcpy (buffer, "rfc822");
		goto carry_on;
	    }

	(void) strcpy (buffer, "invalid Content-Type field (missing subtype)");
	goto out;
    }
    cp++;
    while (isspace (*cp))
	cp++;

    if (*cp == '(' && get_comment (interp, &cp) == NOTOK)
	goto out;
    for (dp = cp; istoken (*dp); dp++)
	continue;
    ch = *dp, *dp = '\0';
    (void) strcpy (buffer, cp);
    *dp = ch, cp = dp;
    if (!buffer[0]) {
	(void) strcpy (buffer, "invalid Content-Type field (empty subtype)");
	goto out;
    }
carry_on: ;
    while (isspace (*cp))
	cp++;

    c -> bp_type = concat (name, "/", buffer, NULLCP);
    for (dp = c -> bp_type; *dp; dp++)
	if (isalpha (*dp) && isupper (*dp))
	    *dp = tolower (*dp);

    if (*cp == '(' && get_comment (interp, &cp) == NOTOK)
	goto out;

    ep = (ap = c -> bp_attrs) + NPARMS;
    while (*cp == ';') {
	char   *vp,
	       *up;

	if (ap >= ep) {
	    (void) sprintf (buffer, "too many parameters (%d max)", NPARMS);
	    goto out;
	}

	cp++;
	while (isspace (*cp))
	    cp++;

	if (*cp == '(' && get_comment (interp, &cp) == NOTOK)
	    goto out;
	if (*cp == 0)
	    goto done_params;

	for (dp = cp; istoken (*dp); dp++)
	    if (isalpha(*dp) && isupper (*dp))
		*dp = tolower (*dp);
	for (up = dp; isspace (*dp); )
	    dp++;
	if (dp == cp || *dp != '=') {
	    (void) strcpy (buffer, "invalid parameter");
	    goto out;
	}

	vp = (*ap = add (cp, NULLCP)) + (up - cp);
	*vp = '\0';
	for (dp++; isspace (*dp); )
	    dp++;
	c -> bp_values[ap - c -> bp_attrs] = vp = *ap + (dp - cp);
	if (*dp == '"') {
	    for (cp = ++dp, dp = vp;;) {
		switch (ch = *cp++) {
		    case '\0':
bad_quote: ;
			(void) strcpy (buffer,
				       "invalid quoted-string in parameter");
			goto out;

		    case '\\':
			*dp++ = ch;
			if ((ch = *cp++) == '\0')
			    goto bad_quote;
			/* else fall... */

		    default:
    			*dp++ = ch;
    			continue;

		    case '"':
			*dp = '\0';
			break;
		}
		break;
	    }
	}
	else {
	    for (cp = dp, dp = vp; istoken (*cp); cp++, dp++)
		continue;
	    *dp = '\0';
	}
	if (!*vp) {
	    (void) strcpy (buffer, "invalid parameter");
	    goto out;
	}
	ap++;

	while (isspace (*cp))
	    cp++;

	if (*cp == '(' && get_comment (interp, &cp) == NOTOK)
	    goto out;
    }
done_params: ;

    if ((cp = c -> bp_id) && *cp != '<') {
	(void) strcpy (buffer, "invalid Content-ID");
	goto out;
    }

    if (c -> bp_encoding) {
	for (cp = c -> bp_encoding; istoken (*cp); cp++)
	    if (isalpha (*cp) && isupper (*cp))
		*cp = tolower (*cp);
	*cp = '\0';
    }

    if (c -> bp_end == 0L) {
	(void) fseek (fp, 0L, 2);
	c -> bp_end = ftell (fp);
    }

    c -> bp_part = getcpy (partno);
    if (uprf (c -> bp_type, "multipart/")) {
	int	digest,
		inout,
		partnum;
	long	last,
		pos;
	char    partnam[BUFSIZ],
		start[BUFSIZ],
		stop[BUFSIZ];
	struct bodypart  *part,
			**next;

	if (c -> bp_encoding && !uleq (c -> bp_encoding, "7bit")) {
	    (void) strcpy (buffer, "multipart not encoded in 7bit");
	    goto out;
	}
	digest = uleq (c -> bp_type, "multipart/digest") ? -1 : 0;

	for (ap = c -> bp_attrs, ep = c -> bp_values; *ap; ap++, ep++)
	    if (uleq (*ap, "boundary"))
		break;
	if (!*ap) {
	    (void) strcpy (buffer, "multipart missing boundary parameter");
	    goto out;
	}
	for (cp = *ep; isspace (*cp); cp++)
	    continue;
	if (!*cp) {
	    (void) strcpy (buffer, "invalid boundary parameter");
	    goto out;
	}
	for (cp = *ep, dp = cp + strlen (cp) - 1; dp > cp; dp--)
	    if (!isspace (*dp))
		break;
	*++dp = '\0';
	(void) sprintf (start, "%s\n", *ep);
	(void) sprintf (stop, "%s--\n", *ep);

	(void) fseek (fp, pos = c -> bp_begin, 0);
	last = c -> bp_end;

	next = &c -> bp_children, part = NULL, inout = 1;
	(void) sprintf (cp = partnam, "%s.", partno);
	cp += strlen (cp);
	partnum = 1;
	while (fgets (buffer, sizeof buffer - 1, fp)) {
	    if (pos > last)
		break;

	    pos += strlen (buffer);

	    if (buffer[0] != '-' || buffer[1] != '-')
		continue;

	    if (inout) {
		if (strcmp (buffer + 2, start))
		    continue;

next_part: ;
		(void) sprintf (cp, "%d", partnum++);
		if (!(part = get_parts (interp, fp, digest, partnam)))
		    goto really_out;
		*next = part, next = &part -> bp_sibling;

		(void) fseek (fp, pos = part -> bp_begin, 0);
		inout = 0;
	    }
	    else
		if (strcmp (buffer + 2, start) == 0) {
		    inout = 1;

end_part: ;
		    part -> bp_end = ftell (fp) - (strlen (buffer) + 1);
		    if (part -> bp_end < part -> bp_begin)
			part -> bp_begin = part -> bp_end;
		    if (inout)
			goto next_part;
		    goto last_part;
		}
	        else
		    if (strcmp (buffer + 2, stop) == 0)
			goto end_part;
	}
	if (!inout && part) {
	    part -> bp_end = c -> bp_end;

	    if (part -> bp_begin >= part -> bp_end) {
		for (next = &c -> bp_children;
		         *next != part;
		         next = &((*next) -> bp_sibling))
		    continue;
		*next = NULL;
		free_part (part);
	    }
	}

last_part: ;
    }
    else
	if (uleq (c -> bp_type, "message/external-body")) {
	    char    partnam[BUFSIZ];
	    struct bodypart *e;

	    if (c -> bp_encoding && !uleq (c -> bp_encoding, "7bit")) {
		(void) strcpy (buffer, "external-body not encoded in 7bit");
		goto out;
	    }

	    (void) fseek (fp, c -> bp_begin, 0);

	    (void) sprintf (partnam, "%s.1", partno);
	    if (!(c -> bp_children = e = get_parts (interp, fp, 0, partnam)))
		goto really_out;
	    e -> bp_end = (e -> bp_parent = c) -> bp_end;

	    e -> ep_access = EP_MISS;
	    for (ap = c -> bp_attrs, ep = c -> bp_values; *ap; ap++, ep++)
		if (uleq (*ap, "access-type")) {
		    int	    cc;
		    register struct kv *k;

		    for (k = kvs; k -> k_key; k++)
			if (uleq (*ep, k -> k_key))
			    break;
		    if ((e -> ep_access = k -> k_value) == EP_MAIL)
			c -> ep_wantbody = 1;
		    break;
		}
	}

    if (toplevel > 0 && fp == SafeTcl_fp)
	SafeTcl_bp = c;

    return c;

out: ;
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
really_out: ;
    free_part (c);
    return NULL;
}

/*  */

static FILE  *get_message (interp, body, filename)
Tcl_Interp     *interp;
char   *body,
       *filename;
{
    char    buffer[BUFSIZ];
    FILE   *fp;

    if (!*body) {
	if ((filename && !SafeTcl_message) || !(fp = SafeTcl_fp)) {
	    (void) strcpy (buffer, "no default message");
	    goto tcl_error;
	}
	if (filename)
	    (void) strcpy (filename, SafeTcl_message);
    }
    else {
	char   *file,
		tmpfil[BUFSIZ];

	(void) strcpy (file = filename ? filename : tmpfil,
		       m_tmpfil (invo_name));

	if (!(fp = fopen (file, "w+"))) {
	    (void) sprintf (buffer, "unable to write %s", file);
	    goto tcl_error;
	}
	if (!filename)
	    (void) unlink (file);

	if (fputs (body, fp) == EOF || fflush (fp)) {
	    (void) fclose (fp);
	    (void) strcpy (buffer, "error writing to temporary file");
	    goto tcl_error;
	}
    }
    rewind (fp);

    return fp;

tcl_error: ;
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    return NULL;
}

/*    PUBLIC */

int	InitReceipt (interp)
Tcl_Interp *interp;
{
    int	    uid;
    char    buffer[BUFSIZ];
    struct passwd *pw;

    if ((pw = getpwuid (uid = getuid ())) && chdir (pw -> pw_dir) == NOTOK)
	(void) chdir ("/");

    if (geteuid () == 0) {
	(void) setgid (pw ? pw -> pw_gid : getgid ());
	if (pw)
	    (void) initgroups (pw -> pw_name, pw -> pw_gid);
	(void) setuid (pw ? pw -> pw_uid : uid);
    }
    
    if (pw) {
	(void) TclSetEnv ("USER", pw -> pw_name);
	(void) TclSetEnv ("HOME", pw -> pw_dir);
	(void) TclSetEnv ("SHELL", pw -> pw_shell);
    }
    (void) TclSetEnv ("MHCONTEXT", "/dev/null");
    (void) TclUnsetEnv ("MH");
    (void) umask (0077);

    (void) sprintf (buffer, "%s@%s", getusr (), LocalName ());
    (void) Tcl_SetVar (interp, "SafeTcl_Recipient", buffer, TCL_GLOBAL_ONLY);

    Tcl_CreateCommand (interp, "swish_filebody", swish_filebody,
		       (ClientData) 0, (void (*) ()) 0);
    Tcl_CreateCommand (interp, "exit", SafeTcl_ExitCmd,
                       (ClientData) 0, (void (*) ()) 0);
}

/*  */

int	InitMessaging (interp, myname, mbox, unrestricted)
Tcl_Interp *interp;
char   *myname,
       *mbox;
int	unrestricted;
{
    union {
	long	l;
	char	c[sizeof (long)];
    }	un;
    static int once_only = 1;

    if (once_only) {
	long	clock;
	register char *cp;
	char	buffer[BUFSIZ];
	struct tws *tw;

	once_only = 0;

	un.l = 1;
	endian = un.c[0] ? -1 : 1;

	(void) strcpy (mailbox, mbox ? mbox : "");

	(void) time (&clock);

	(void) sprintf (buffer, "Content-ID: <%d.%ld.%%d@%s>\n", getpid (),
			clock, LocalName ());
	bodyno = 0;
	msgfmt = getcpy (buffer);

	tw = dlocaltime (&clock);
	(void) sprintf (ddate, "Delivery-Date: %s\n", dasctime (tw, TW_ZONE));
	(void) strcpy (udate, dctime (tw));

#ifdef SAFETCL_INTERFACESTYLE_TK3_6
	mts_init (invo_name = uprf (myname, "swish") ? "swish" : myname);
#else
	mts_init (invo_name = uprf (myname, "stclsh") ? "stclsh" : myname);
#endif

	mhuser = 1;
	if (!getenv ("MH")) {
	    struct passwd *pw;

	    if (cp = getenv ("HOME"))
		(void) strcpy (buffer, cp);
	    else
		if ((pw = getpwuid (getuid ()))
		        && pw -> pw_dir
		        && *pw -> pw_dir)
		    (void) strcpy (buffer, pw -> pw_dir);
		else
		    goto foil;
	    if ((cp = buffer + strlen (buffer) - 1) > buffer && *cp == '/')
		*cp = '\0';
	    cp = buffer + strlen (buffer);
	    (void) sprintf (cp, "/%s", mh_profile);
	    if (access (buffer, W_OK) == NOTOK) {
foil: ;
		m_foil (NULLCP);
		mhuser = 0;
		defalt = fileproc = NULLCP;
	    }
	}

	if (mhuser) {
	    (void) sprintf (buffer, "%s-cache", "swish");
	    if ((cache_public = m_find (buffer)) && *cache_public != '/')
		cache_public = NULL;
	    (void) sprintf (buffer, "%s-private-cache", "swish");
	    if (!(cache_private = m_find (buffer)))
		cache_private = ".cache";
	    cache_private = getcpy (m_maildir (cache_private));

	    (void) sprintf (buffer, "%s-access-ftp", "swish");
	    if ((ftp = m_find (buffer)) && !*ftp)
		ftp = NULLCP;

	    if (!(render = m_find ("mhnproc"))) {
		render = "mhn";
		if (cp = getenv ("PATH"))
		    for (;;) {
			char   *pp;

			if (pp = index (cp, ':'))
			    *pp = '\0';
			(void) sprintf (buffer, "%s/%s", *cp ? cp : ".",
					render);
			if (access (buffer, 01) == OK)
			    mhnuser = 1;
			if (pp)
			    *pp++ = ':', cp = pp;
			if (mhnuser || !pp)
			    break;
		    }
	    }
	    else
		mhnuser = 1;
	    if (!mhnuser)
		render = "metamail";
	}
    }

/* Additional Messaging Functionality */
    Tcl_CreateCommand(interp, "SafeTcl_genid", SafeTcl_GenidCmd,
                       (ClientData) 0, (void (*) ()) 0);
    Tcl_CreateCommand (interp, "SafeTcl_getaddrs", SafeTcl_getaddrs,
		       (ClientData) 0, (void (*) ()) 0);
    Tcl_CreateCommand (interp, "SafeTcl_getaddrprop", SafeTcl_getaddrprop,
		       (ClientData) 0, (void (*) ()) 0);
    Tcl_CreateCommand (interp, "SafeTcl_getdateprop", SafeTcl_getdateprop,
		       (ClientData) 0, (void (*) ()) 0);
    Tcl_CreateCommand (interp, "SafeTcl_getheader", SafeTcl_getheader,
		       (ClientData) 0, (void (*) ()) 0);
    Tcl_CreateCommand (interp, "SafeTcl_getheaders", SafeTcl_getheaders,
		       (ClientData) 0, (void (*) ()) 0);
    Tcl_CreateCommand (interp, "SafeTcl_makebody", SafeTcl_makebody,
		       (ClientData) 0, (void (*) ()) 0);
    Tcl_CreateCommand (interp, "SafeTcl_getparts", SafeTcl_getparts,
		       (ClientData) 0, (void (*) ()) 0);
    Tcl_CreateCommand (interp, "SafeTcl_getbodyprop", SafeTcl_getbodyprop,
		       (ClientData) 0, (void (*) ()) 0);
    Tcl_CreateCommand (interp, "SafeTcl_encode", SafeTcl_encode,
		       (ClientData) 0, (void (*) ()) 0);
    Tcl_CreateCommand (interp, "SafeTcl_decode", SafeTcl_decode,
		       (ClientData) 0, (void (*) ()) 0);
    if (!Tcl_NoInterface ())
	Tcl_CreateCommand (interp, "SafeTcl_displaybody", SafeTcl_displaybody,
			   (ClientData) 0, (void (*) ()) 0);

/* Recommended Extensions to the Trusted Tcl Interpreter */
    if (unrestricted) {
	Tcl_CreateCommand (interp, "MIME_sendmessage", MIME_sendmessage,
			   (ClientData) 0, (void (*) ()) 0);
	Tcl_CreateCommand (interp, "MIME_printtext", MIME_printtext,
			   (ClientData) 0, (void (*) ()) 0);
	Tcl_CreateCommand (interp, "MIME_savemessage", MIME_savemessage,
			   (ClientData) 0, (void (*) ()) 0);
    }

    if (!runsafely)
	Tcl_CreateCommand (interp, "SafeTcl_setconfigdata",
			   SafeTclP_configdata,
  			   (ClientData) 0, (void (*) ()) 0);
}
