/*
 * tmConverters.c -
 *	This module implements all the Xt Converters to and
 *	from Strings needed by Tm
 *
 * Copyright 1993 Jan Newmarch, University of Canberra.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The author
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#ifdef MOTIF
#include <Xm/RepType.h>
#include <Xm/DragDrop.h>
#endif
#include <X11/StringDefs.h>
#include <X11/IntrinsicP.h>
#include <X11/CoreP.h>		/* for CoreRec */
#include <string.h>
#include <ctype.h>
#include <tcl.h>

#define done(address,type) \
	{	toVal->addr = (XtPointer) address; \
		toVal->size = sizeof(type); \
	}


int RXt_XmStringTableSize;

/*
 *--------------------------------------------------------------
 *
 * InstallReverseRepTypes --
 *
 *	Install the converters from internal representation
 *	types to their string values.
 *
 * Results:
 *
 * Side effects:
 *	Modifies the Motif RepType tables
 *
 *--------------------------------------------------------------
 */

static void
InstallReverseRepTypes()
{
#ifdef MOTIF
    XmRepTypeId id;
    XmRepTypeList start, list;
    int n;

    start = list = XmRepTypeGetRegistered();

    while (list->rep_type_name != NULL) {
        if ((id = XmRepTypeGetId(list->rep_type_name)) != XmREP_TYPE_INVALID) {
	    XmRepTypeAddReverse(id);
        } else {
	    fprintf(stderr, "Failed to install %s converter\n", 
			list->rep_type_name);
        }
	list++;
    }
    XtFree((char *)start);
#endif
}

/*
 *--------------------------------------------------------------
 *
 * RXt_CvtWidgetToString --
 *
 *	Converter from Widget to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
RXt_CvtWidgetToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    Widget w = *(Widget *) fromVal->addr;
    static char *buf;

    /* it is valid to ask for a NULL widget e.g. topWidget in a Form
       when the thing is attached to the Form (default is NULL)
     */
    if (w == NULL) {
        buf = "";
        done(&buf, String);
	return True;
    }

    buf = XtName(w);
    done(&buf, String);

    return True;
}


/*
 *--------------------------------------------------------------
 *
 * RXt_CvtWidgetListToString --
 *
 *	Converter from WidgetList to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
RXt_CvtWidgetListToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    Widget w = *(Widget *) args[0].addr;
    WidgetList children = *(WidgetList *) fromVal->addr;
    Cardinal num_children, final_num_children;
    String *widgetNames;
    int n, len;
    char *end, *p;
    static char *buf;

    if ( !XtIsComposite(w)) {
	buf = XtMalloc(1);
        *buf = '\0';
        done(&buf, char *);
        return True;
    }

    XtVaGetValues(w, 
	XtNnumChildren, &num_children,
	XtNchildren, &children, NULL);
    if (num_children == 0) {
	buf = XtMalloc(1);
        *buf = '\0';
        done(&buf, char *);
        return True;
    }

    widgetNames = (String *) XtMalloc(num_children * sizeof(String));
    len = 0;
    final_num_children = 0;
    for (n = 0; n < num_children; n++) {
	widgetNames[final_num_children] = XtName(children[n]);
	len += strlen(widgetNames[final_num_children]) + 2; /* for ", " */
	final_num_children++;
    }

    buf = XtMalloc(len + 1); /* we overshoot by 1 */
    if (len == 0) {
        *buf = '\0';
        done(&buf, char *);
        return True;
    }
    end = buf;
    for (n = 0; n < final_num_children; n++) {
        p = widgetNames[n];
        while (*end++ = *p++)
        ;
        *(end - 1) = ',';
        *end++ = ' ';
    }
    *(end - 2) = '\0';

    done(&buf, char *);

    return True;
}


#ifdef MOTIF
/*
 *--------------------------------------------------------------
 *
 * RXt_CvtXmStringToString --
 *
 *	Converter from XmString to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
RXt_CvtXmStringToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    XmStringContext context;
    static char *buf;
    char *p, *text;
    XmStringCharSet charset;
    XmStringDirection direction;
    Boolean separator;
    int size = 128;
    int current_len, segment_len;

    XmString xmstr = *(XmString *) fromVal->addr;
    p = buf = XtMalloc(size);
    if (!XmStringInitContext(&context, xmstr)) {
	XtFree(p);
	return False;
    }
    while (XmStringGetNextSegment(context, &text, &charset,
		&direction, &separator)) {
	segment_len = strlen(text);
	if (p + segment_len >= buf + size) {
	    /* string too large, double space */
	    current_len = p - buf;
	    size *= 2;
	    buf = XtRealloc(buf, size);
	    p = buf + current_len;
	}
	strcpy(p, text);
	p += segment_len;
	if (separator == True) {
	    *p++ = '\n';
	    *p = 0;
	}
	XtFree(text);
    }
    XmStringFreeContext(context);
 
    done(&buf, char *);

    return True;
}


/*
 *--------------------------------------------------------------
 *
 * RXt_CvtXmStringTableToString --
 *
 *	Converter from XmStringTable to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
RXt_CvtXmStringTableToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    String *items;
    static String buf;
    int n, len, item_count;
    char *p, *end;
    char *dummy;
    XmString *xmstrs = *(XmString **) fromVal->addr;

    extern Boolean _XmStringIsString _ANSI_ARGS_((XmString));

    len= 0;
    item_count = RXt_XmStringTableSize;
    items = NULL;

    if (item_count != 0) {
        items = (String *) XtMalloc(item_count * sizeof(String));
        for (n = 0; n < item_count; n++) {
            XmStringGetLtoR(xmstrs[n],
    		XmFONTLIST_DEFAULT_TAG, items + n);
            len += strlen(items[n]) + 2; /* for , */
        }
    }
    buf = XtMalloc(len + 1); /* we overshoot by 1 */
    if (len == 0) {
        *buf = '\0';
        done(&buf, char *);
        return True;
    }
    end = buf;
    for (n = 0; n < item_count; n++) {
        p = items[n];
        while (*end++ = *p++)
    	;
        *(end - 1) = ',';
        *end++ = ' ';
        XtFree(items[n]);
    }
    *(end - 2) = '\0';

    done(&buf, char *);

    if (items != NULL) {
	XtFree((char *) items);
    }

    return True;
}
#endif /* MOTIF */

/*
 *--------------------------------------------------------------
 *
 * RXt_CvtStringToString --
 *
 *	Converter from String to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
RXt_CvtStringToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;

    buf = *(char **) (fromVal->addr);
#   ifdef DEBUG
    fprintf(stderr, "converted string to %s\n", buf);
#   endif

    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * RXt_CvtBooleanToString --
 *
 *	Converter from Boolean to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
RXt_CvtBooleanToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;
    Boolean boolVal;
 
    boolVal = *(Boolean *) (fromVal->addr);

    if (boolVal) {
	buf = "true";
    } else {
	buf = "false";
    } 
 
    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * RXt_CvtIntToString --
 *
 *	Converter from Int to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
RXt_CvtIntToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;
    int intVal;
 
    intVal = *(int *) (fromVal->addr);

    /* memory leak here */
    buf = XtMalloc(16);
    sprintf(buf, "%d", intVal);
 
    done(&buf, char *);

    return True;
}


/*
 *--------------------------------------------------------------
 *
 * RXt_CvtUnsignedIntToString --
 *
 *	Converter from UnsignedInt to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
RXt_CvtUnsignedIntToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;
    unsigned int intVal;
 
    intVal = *(int *) (fromVal->addr);

    /* memory leak here */
    buf = XtMalloc(16);
    sprintf(buf, "%u", intVal);
 
    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * RXt_CvtDimensionToString --
 *
 *	Converter from Dimension to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
RXt_CvtDimensionToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;
    Dimension dimVal;
 
    dimVal = *(Dimension *) (fromVal->addr);

    /* memory leak here */
    buf = XtMalloc(16);
    sprintf(buf, "%hu", dimVal);
 
    done(&buf, char *);

    return True;
}

/*
 *--------------------------------------------------------------
 *
 * RXt_CvtShortToString --
 *
 *	Converter from Short to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
RXt_CvtShortToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;
    Dimension shortVal;
 
    shortVal = *(short *) (fromVal->addr);

    /* memory leak here */
    buf = XtMalloc(16);
    sprintf(buf, "%hd", shortVal);
 
    done(&buf, char *);

    return True;
}


/*
 *--------------------------------------------------------------
 *
 * RXt_CvtLongToString --
 *
 *	Converter from Long to String type
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
RXt_CvtLongToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;
    Dimension shortVal;
 
    shortVal = *(short *) (fromVal->addr);

    /* memory leak here */
    buf = XtMalloc(16);
    sprintf(buf, "%ld", shortVal);
 
    done(&buf, char *);

    return True;
}


/*
 *--------------------------------------------------------------
 *
 * RXt_CvtPixelToString --
 *
 *	Converter from Pixel to String type
 *	gives result in #RGB form
 *
 * Results:
 *
 *	changes the "toVal" arg to contain the new value.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Boolean
RXt_CvtPixelToString(display, args, num_args, fromVal, toVal, destructor_data)
    Display *display;
    XrmValuePtr args;
    Cardinal *num_args;
    XrmValuePtr fromVal;
    XrmValuePtr toVal;
    XtPointer *destructor_data;
{
    static char *buf;
    Pixel pixelVal;
    Screen *screen;
    Colormap colormap;
    XColor color;
 
    color.pixel = *(unsigned long *) fromVal->addr;
    screen = *((Screen **) args[0].addr);
    colormap = *((Colormap *) args[1].addr);

    XQueryColor(DisplayOfScreen(screen), colormap, &color);

    /* memory leak here */
    buf = XtMalloc(32);
    sprintf(buf, "#%04x%04x%04x", color.red, color.green, color.blue);

    done(&buf, char *);

    return True;
}


/*
 *--------------------------------------------------------------
 *
 * RXt_RegisterConverters --
 *
 *	Register all extra converters with Xt.
 *
 * Results:
 *
 * Side effects:
 *
 *	Modifies the Xt converter lists.
 *--------------------------------------------------------------
 */

void
RXt_RegisterConverters(interp, appContext)
    Tcl_Interp *interp;
    XtAppContext appContext;
{
    XtConvertArgRec convert_arg;
    XtConvertArgRec convert_args[128];
 
    /*
     * Register the type converters we need
     * This implementation installs everything, 
     * even if it is never used. A smarter method
     * would be to install on demand. Maybe someday...
     */

    InstallReverseRepTypes();


    /* 
     * Widget to String
     */

    XtSetTypeConverter(
		XtRWidget, XtRString,
		RXt_CvtWidgetToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * WidgetList to String
     */
    convert_args[0].address_mode = XtWidgetBaseOffset;
    convert_args[0].address_id = (XtPointer) XtOffsetOf(CoreRec, core.self);
    convert_args[0].size = sizeof(CoreWidget);

    XtSetTypeConverter(
		XtRWidgetList, XtRString,
		RXt_CvtWidgetListToString,
		convert_args,
		1,
		XtCacheNone,
		NULL);
#ifdef MOTIF
    /* 
     * XmString to String
     */
    XtSetTypeConverter(
		XmRXmString, XtRString,
		RXt_CvtXmStringToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * XmStringTable to String
     */
    XtSetTypeConverter(
		XmRXmStringTable, XtRString,
		RXt_CvtXmStringTableToString,
		NULL,
		0,
		XtCacheNone,
		NULL);
#endif /* MOTIF */

    /* 
     * String to String
     * I don't know why there isn't already one of these,
     * but getting value out of Text seems to need it.
     *
     * it doesn't seem to work anyway - maybe something
     * special case away X to X convertions
     */
/*
    convert_arg.address_mode = XtAddress;
    convert_arg.address_id = (XtPointer) interp;
    convert_arg.size = sizeof(XtPointer);

    XtSetTypeConverter(
		XtRString, XtRString,
		RXt_CvtStringToString,
		&convert_arg,
		1,
		XtCacheNone,
		NULL);
*/

    /* 
     * Int to String
     */
    XtSetTypeConverter(
		XtRInt, XtRString,
		RXt_CvtIntToString,
		NULL,
		0,
		XtCacheNone,
		NULL);


    /* 
     * Cardinal to String
     */
    XtSetTypeConverter(
		XtRCardinal, XtRString,
		RXt_CvtUnsignedIntToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

#ifdef MOTIF
    /* 
     * HorizontalInt to String
     */
    XtSetTypeConverter(
		XmRHorizontalInt, XtRString,
		RXt_CvtIntToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * VerticalInt to String
     */
    XtSetTypeConverter(
		XmRVerticalInt, XtRString,
		RXt_CvtIntToString,
		NULL,
		0,
		XtCacheNone,
		NULL);
#endif /* MOTIF */


    /* 
     * Boolean to String
     */
    XtSetTypeConverter(
		XtRBoolean, XtRString,
		RXt_CvtBooleanToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * Dimension to String
     */
    XtSetTypeConverter(
		XtRDimension, XtRString,
		RXt_CvtDimensionToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

#ifdef MOTIF
    /* 
     * BooleanDimension to String
     */
    XtSetTypeConverter(
		XmRBooleanDimension, XtRString,
		RXt_CvtDimensionToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * HorizontalDimension to String
     * Buggy? should use UnitType resource
     */
    XtSetTypeConverter(
		XmRHorizontalDimension, XtRString,
		RXt_CvtDimensionToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * VerticalDimension to String
     * Buggy? should use UnitType resource
     */

    XtSetTypeConverter(
		XmRVerticalDimension, XtRString,
		RXt_CvtDimensionToString,
		NULL,
		0,
		XtCacheNone,
		NULL);
#endif /* MOTIF */

    /* 
     * short to String
     */
    XtSetTypeConverter(
		XtRShort, XtRString,
		RXt_CvtShortToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * Position to String
     */
    XtSetTypeConverter(
		XtRPosition, XtRString,
		RXt_CvtShortToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

#ifdef MOTIF
    /* 
     * HorizontalPosition to String
     */
    XtSetTypeConverter(
		XmRHorizontalPosition, XtRString,
		RXt_CvtShortToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * VerticalPosition to String
     */
    XtSetTypeConverter(
		XmRVerticalPosition, XtRString,
		RXt_CvtShortToString,
		NULL,
		0,
		XtCacheNone,
		NULL);

    /* 
     * TextPosition to String
     */
    XtSetTypeConverter(
		XmRTextPosition, XtRString,
		RXt_CvtLongToString,
		NULL,
		0,
		XtCacheNone,
		NULL);
#endif /* MOTIF */


    /* 
     * Pixel to String
     */
    convert_args[0].address_mode = XtWidgetBaseOffset;
    convert_args[0].address_id = (XtPointer) XtOffsetOf(WidgetRec, core.screen);
    convert_args[0].size = sizeof(Screen *);

    convert_args[1].address_mode = XtWidgetBaseOffset;
    convert_args[1].address_id = (XtPointer) XtOffsetOf(CoreRec, core.colormap);
    convert_args[1].size = sizeof(Colormap);

    XtSetTypeConverter(
		XtRPixel, XtRString,
		RXt_CvtPixelToString,
		convert_args,
		2,
		XtCacheNone,
		NULL);
}
