#include <tcl.h>
#include <tclInt.h>
#include <tclPort.h>
#include <assert.h>
#include <memory.h>
#include "tclcomp.h"
#include "tclcompint.h"
#include "tclcompproc.h"

/*---------------------------------------------------------------------------
 *
 * The core Tcl commands are assumed immutable by the compiler. It
 * uses the table below to generate a private hash table by the
 * initialization procedure (Tcl_CompInit) in the same
 * manner as done by the Tcl_CreateInterp procedure in module
 * tclBasic.c. The difference is that the hash table is never modified
 * after initialization. Of course, this is incompatible with the
 * way purely interpreted Tcl works, where any command can be redefined.
 * I think, however, that this is a small price to pay.
 *
 * Note: If the core Tcl interpreter ever removes the 'static' from
 * variable builtInCmds, we may save some bytes.
 */

/*
 * The following structure defines all of the commands in the Tcl core,
 * and the C procedures that execute them.
 */

typedef struct {
    char *name;			/* Name of command. */
    Tcl_CmdProc *proc;		/* Procedure that executes command. */
} CmdInfo;

/*
 * Built-in commands, and the procedures associated with them:
 */

static CmdInfo builtInCmds[] = {
    /*
     * Commands in the generic core:
     */

    {"append",		Tcl_AppendCmd},
    {"array",		Tcl_ArrayCmd},
    {"break",		Tcl_BreakCmd},
    {"case",		Tcl_CaseCmd},
    {"catch",		Tcl_CatchCmd},
    {"concat",		Tcl_ConcatCmd},
    {"continue",	Tcl_ContinueCmd},
    {"error",		Tcl_ErrorCmd},
    {"eval",		Tcl_EvalCmd},
    {"expr",		Tcl_ExprCmd},
    {"for",		Tcl_ForCmd},
    {"foreach",		Tcl_ForeachCmd},
    {"format",		Tcl_FormatCmd},
    {"global",		Tcl_GlobalCmd},
    {"history",		Tcl_HistoryCmd},
    {"if",		Tcl_IfCmd},
    {"incr",		Tcl_IncrCmd},
    {"info",		Tcl_InfoCmd},
    {"join",		Tcl_JoinCmd},
    {"lappend",		Tcl_LappendCmd},
    {"lindex",		Tcl_LindexCmd},
    {"linsert",		Tcl_LinsertCmd},
    {"list",		Tcl_ListCmd},
    {"llength",		Tcl_LlengthCmd},
    {"lrange",		Tcl_LrangeCmd},
    {"lreplace",	Tcl_LreplaceCmd},
    {"lsearch",		Tcl_LsearchCmd},
    {"lsort",		Tcl_LsortCmd},
    {"proc",		Tcl_CompProcCmd}, /* Use replacement for proc here */
    {"regexp",		Tcl_RegexpCmd},
    {"regsub",		Tcl_RegsubCmd},
    {"rename",		Tcl_RenameCmd},
    {"return",		Tcl_ReturnCmd},
    {"scan",		Tcl_ScanCmd},
    {"set",		Tcl_SetCmd},
    {"split",		Tcl_SplitCmd},
    {"string",		Tcl_StringCmd},
    {"subst",		Tcl_SubstCmd},
    {"switch",		Tcl_SwitchCmd},
    {"trace",		Tcl_TraceCmd},
    {"unset",		Tcl_UnsetCmd},
    {"uplevel",		Tcl_UplevelCmd},
    {"upvar",		Tcl_UpvarCmd},
    {"while",		Tcl_WhileCmd},

    /*
     * Commands in the UNIX core:
     */

#ifndef TCL_GENERIC_ONLY
    {"cd",		Tcl_CdCmd},
    {"close",		Tcl_CloseCmd},
    {"eof",		Tcl_EofCmd},
    {"exec",		Tcl_ExecCmd},
    {"exit",		Tcl_ExitCmd},
    {"file",		Tcl_FileCmd},
    {"flush",		Tcl_FlushCmd},
    {"gets",		Tcl_GetsCmd},
    {"glob",		Tcl_GlobCmd},
    {"open",		Tcl_OpenCmd},
    {"pid",		Tcl_PidCmd},
    {"puts",		Tcl_PutsCmd},
    {"pwd",		Tcl_PwdCmd},
    {"read",		Tcl_ReadCmd},
    {"seek",		Tcl_SeekCmd},
    {"source",		Tcl_CompSourceCmd}, /* Use replacement for source */
    {"tell",		Tcl_TellCmd},
    {"time",		Tcl_TimeCmd},
#endif /* TCL_GENERIC_ONLY */
    {NULL,		(Tcl_CmdProc *) NULL}
};

/*
 * Fixed (global) commandTable used by the compiler similar to the one used in
 * Interp structures
 */

Tcl_HashTable commandTable;  

/*---------------------------------------------------------------------------*/

static void
FreeValue (ValStruct *value)		/* Contains a word (a value) */
/*
 * Deallocates dynamic memory associated with value.
 */
{
   ValStruct *tmp;
   while (value != NULL) {
      switch (value->type) {
	 case NullValue: {
	    break;
	 }
	 case LiteralValue: {
	    ckfree (value->detail.literal);
	    break;
	 }
	 case CommandValue: {
	    FreeCmd (value->detail.cmd);
	    break;
	 }
	 case VarValue: {
	    ckfree (value->detail.varvalue.name);
	    if (value->detail.varvalue.index != NULL) {
	       FreeValue (value->detail.varvalue.index);
	    }
	    break;
	 }
	 default: {
	    printf ("Unknown value");
	    assert (0);
	    break;
	 }
      }
      tmp = value->next;
      ckfree (value);
      value = tmp;
   } 
}

void
FreeCmd (CmdStruct *cmd)		/* Points to a list of commands */
/*
 * Deallocates dynamic memory associated with cmd
 */
{
   CmdStruct *tmp;
   while (cmd != NULL) {
      switch (cmd->type) {
	 case NullCmd: {
	    break;
	 }	 
	 case OtherCmd: {
	    int i;
	    for (i = 0; i < cmd->detail.othercmd.argc; i++) {
	       FreeValue (cmd->detail.othercmd.words [i]);
	    }
	    ckfree (cmd->detail.othercmd.words);
	    break;
	 }
	 case SetCmd: {
	    FreeValue (cmd->detail.setcmd.name);
	    FreeValue (cmd->detail.setcmd.index);
	    FreeValue (cmd->detail.setcmd.value);
	    break;
	 }
	 case IncrCmd: {
	    FreeValue (cmd->detail.incrcmd.name);
	    FreeValue (cmd->detail.incrcmd.increment);
	    FreeValue (cmd->detail.setcmd.index);
	    break;
	 }
	 case IfCmd: {
	    int i;
	    for (i = 0; i < cmd->detail.ifcmd.ncond; i++) {
	       FreeValue (cmd->detail.ifcmd.cond [i]);
	       FreeCmd (cmd->detail.ifcmd.body [i]);
	    }
	    ckfree (cmd->detail.ifcmd.cond);
	    ckfree (cmd->detail.ifcmd.body);
	    FreeCmd (cmd->detail.ifcmd.elsebody);
	    break;
	 }
	 case WhileCmd: {
	    FreeValue (cmd->detail.whilecmd.cond);
	    FreeCmd (cmd->detail.whilecmd.body);
	    break;
	 }
	 case RuntimeCmd: {
	    FreeValue (cmd->detail.runtimecmd.body);
	    break;
	 }
	 default: {
	    printf ("unknown command\n");
	    assert (0);
	 }
      }
      tmp = cmd;
      cmd = cmd->next;
      ckfree (tmp);
   }
}

static void
PrintCommand (Tcl_DString* s, CmdStruct * cmd);

static void 
PrintValue (Tcl_DString* s, ValStruct *value)
/*
 * Human readable output of value
 */
{
   do {
      switch (value->type) {
	 case NullValue : {
	    Tcl_DStringAppend (s, "NULL", -1); 
	    break;
	 }
	 case LiteralValue : {
	    Tcl_DStringAppend (s, "'", -1);
	    Tcl_DStringAppend (s, value->detail.literal, -1);
	    Tcl_DStringAppend (s, "'", -1);
	    break;
	 }
	 case CommandValue : {
	    Tcl_DStringAppend (s, "[", -1);	 
	    PrintCommand (s, value->detail.cmd);
	    Tcl_DStringAppend (s, "]", -1);	
	    break;
	 }
	 case VarValue : {
	    if (value->detail.varvalue.index == NULL) {
	       Tcl_DStringAppend (s, "${", -1);
	       Tcl_DStringAppend (s, value->detail.varvalue.name, -1);
	       Tcl_DStringAppend (s, "}", -1);
	    }
	    else {
	       Tcl_DStringAppend (s, "$", -1);
	       Tcl_DStringAppend (s, value->detail.varvalue.name, -1);
	       Tcl_DStringAppend (s, "(", -1);
	       PrintValue (s, value->detail.varvalue.index);
	       Tcl_DStringAppend (s, ")", -1);
	    }
	    break;
	 }
	 default: {
	    Tcl_DStringAppend (s, "unknown value", -1);
	    break;
	 }
      }
      value = value->next;
   } while (value != NULL);
   Tcl_DStringAppend (s, " ", -1);
}

static void 
PrintCommand (Tcl_DString* s, CmdStruct * cmd) 
/* 
 * Human readable output of cmd
 */
{
   CmdInfo *cmdInfoPtr;
   int i;
   
   do {
      switch (cmd->type) {
	 case NullCmd: {
	    Tcl_DStringAppend (s, "NullCmd", -1);
	    break;
	 }
	 case SetCmd: {
	    Tcl_DStringAppend (s, "set ", -1);
	    PrintValue (s, cmd->detail.setcmd.name);
	    if (cmd->detail.setcmd.index != NULL) {
	       Tcl_DStringAppend (s, "(", -1);
	       PrintValue (s, cmd->detail.setcmd.index);
	       Tcl_DStringAppend (s, ")", -1);
	    }
	    if (cmd->detail.setcmd.value != NULL) {
	       Tcl_DStringAppend (s, " ", -1);
	       PrintValue (s, cmd->detail.setcmd.value);
	    }
	    break;
	 }
	 case IncrCmd: {
	    Tcl_DStringAppend (s, "incr ", -1);
	    PrintValue (s, cmd->detail.incrcmd.name);
	    if (cmd->detail.incrcmd.index != NULL) {
	       Tcl_DStringAppend (s, "(", -1);
	       PrintValue (s, cmd->detail.incrcmd.index);
	       Tcl_DStringAppend (s, ")", -1);
	    }
	    Tcl_DStringAppend (s, " ", -1);
	    if (cmd->detail.incrcmd.increment != NULL) {
	       PrintValue (s, cmd->detail.incrcmd.increment);
	    }
	    break;
	 }
	 case IfCmd: {
	    for (i = 0; i < cmd->detail.ifcmd.ncond; ++i) {
	       Tcl_DStringAppend (s, i == 0 ? "if " : "elseif ", -1);
	       PrintValue (s, cmd->detail.ifcmd.cond [i]);
	       Tcl_DStringAppend (s, " {\n", -1);
	       PrintCommand (s, cmd->detail.ifcmd.body [i]);
	       Tcl_DStringAppend (s, "} ", -1);
	    }
	    if (cmd->detail.ifcmd.elsebody != NULL) {
	       Tcl_DStringAppend (s, "else {\n", -1);
	       PrintCommand (s, cmd->detail.ifcmd.elsebody);
	       Tcl_DStringAppend (s, "}", -1);
	    }
	    break;
	 }
	 case WhileCmd: {
	    Tcl_DStringAppend (s, "while ", -1);
	    PrintValue (s, cmd->detail.whilecmd.cond);
	    Tcl_DStringAppend (s, " {\n", -1);
	    PrintCommand (s, cmd->detail.whilecmd.body);
	    Tcl_DStringAppend (s, "} ", -1);
	    break;
	 }
	 case RuntimeCmd: {
	    Tcl_DStringAppend (s, "(RuntimeCmd) ", -1);
	    PrintValue (s, cmd->detail.runtimecmd.body);
	    break;
	 }
	 case OtherCmd: {
	    Tcl_DStringAppend (s, "(OtherCmd) ", -1);
	    if (cmd->detail.othercmd.tclcmdptr == NULL) {
	       Tcl_DStringAppend (s, "(runtime command)", -1);
	    }
	    for (i = 0; i < cmd->detail.othercmd.argc; i++) {
	       PrintValue (s, cmd->detail.othercmd.words [i]);
	       Tcl_DStringAppend (s, " ", -1);
	    }
	    break;
	 }
	 default: {
	    Tcl_DStringAppend (s, "unknown command", -1);
	    break;
	 }
      }
      if (cmd->next != NULL) Tcl_DStringAppend (s, ";", -1);
      cmd = cmd->next;
   } while (cmd != NULL);
}

static char*
MyGetVar (register Interp* iptr,/* Interp */
	  char* name,		/* Name of scalar or array var */
	  char* index)		/* Index if array or NULL of scalar */
{
   Tcl_HashTable *tablePtr = (iptr->varFramePtr == NULL ? 
			      (&iptr->globalTable) : 
			      (&iptr->varFramePtr->varTable));
   Tcl_HashEntry *hptr = Tcl_FindHashEntry (tablePtr, name);
   Var* varptr;
   
   if (hptr==NULL) {
      sprintf (iptr->result, "No such var: %.50s", name);
      return NULL;
   }

   varptr = (Var*) Tcl_GetHashValue (hptr);
   if (varptr->flags & VAR_UPVAR) {
      varptr = varptr->value.upvarPtr;
   }

   if (varptr->tracePtr != NULL) {
      /* Won't handle traces here. Use Tcl's slower getvar */
      return Tcl_GetVar2 ((Tcl_Interp*) iptr, name, index, 0);
   }

   if (index != NULL) {
      if (!(varptr->flags & VAR_ARRAY)) {
	 sprintf (iptr->result, "Not an array: %.50s", name);
	 return NULL;
      }
      hptr = Tcl_FindHashEntry (varptr->value.tablePtr, index);
      if (hptr == NULL) {
	 sprintf (iptr->result, "Array %.50s has no element %.50s", 
		  name, index);
	 return NULL;
      }
      varptr = (Var*) Tcl_GetHashValue (hptr);
      if (varptr->tracePtr != NULL) {
	 /* Won't handle traces here. Use Tcl's slower getvar */
	 return Tcl_GetVar2 ((Tcl_Interp*) iptr, name, index, 0);
      }
   }

   return varptr->value.string;
}

static char*
MySetVar (register Interp* iptr,/* Interp */
	  char* name,		/* Name of scalar or array var */
	  char* index,		/* Index if array or NULL of scalar */
	  char* value)		/* value to store */
{
   Tcl_HashTable *tablePtr = (iptr->varFramePtr == NULL ? 
			      (&iptr->globalTable) : 
			      (&iptr->varFramePtr->varTable));
   int new;
   Tcl_HashEntry *hptr = Tcl_CreateHashEntry (tablePtr, name, &new);
   int len = strlen (value);
   register Var* varptr;
   
   if (new) {
      varptr = (Var*) ckalloc (sizeof (Var));
      varptr->valueLength = 0;
      varptr->valueSpace = 0;
      varptr->value.string = NULL;
      varptr->hPtr = hptr;
      varptr->refCount = 0;
      varptr->tracePtr = NULL;
      varptr->searchPtr = NULL;
      varptr->flags = VAR_UNDEFINED;
      Tcl_SetHashValue (hptr, varptr);
   }
   else {
      varptr = (Var*) Tcl_GetHashValue (hptr);
      if (varptr->flags & VAR_UPVAR) {
	 varptr = varptr->value.upvarPtr;
      }

      if (varptr->tracePtr != NULL) {
	 /* Won't handle traces here. Use Tcl's slower getvar */
	 return Tcl_SetVar2 ((Tcl_Interp*) iptr, name, index, value,
			     TCL_LEAVE_ERR_MSG);
      }
      if (varptr->hPtr == NULL) {
	 sprintf (iptr->result, "Dangling upvar : %.50s", name);
	 return NULL;
      }
   }

   if (index != NULL) {
      if (varptr->flags & VAR_UNDEFINED) {
	 varptr->flags = VAR_ARRAY;
	 varptr->value.tablePtr = (Tcl_HashTable*) 
	    ckalloc (sizeof (Tcl_HashTable));
	 Tcl_InitHashTable (varptr->value.tablePtr, TCL_STRING_KEYS);
      } 
      else if (!(varptr->flags & VAR_ARRAY)) {
	 sprintf (iptr->result, "Not an array: %.50s", name);
	 return NULL;
      }
      hptr = Tcl_CreateHashEntry (varptr->value.tablePtr, index, &new);
      if (new) {
	 if (varptr->searchPtr != NULL) {
	    /* Won't handle this case here. */
	    return Tcl_SetVar2 ((Tcl_Interp*) iptr, name, index, value,
				TCL_LEAVE_ERR_MSG);
	 }
	 varptr = (Var*) ckalloc (sizeof (Var));
	 varptr->valueLength = 0;
	 varptr->valueSpace = 0;
	 varptr->value.string = NULL;
	 varptr->hPtr = hptr;
	 varptr->refCount = 0;
	 varptr->tracePtr = NULL;
	 varptr->searchPtr = NULL;
	 varptr->flags = VAR_UNDEFINED;
	 Tcl_SetHashValue (hptr, varptr);
      }
      else {
	 varptr = (Var*) Tcl_GetHashValue (hptr);
	 if (varptr->tracePtr != NULL) {
	    /* Won't handle traces here. Use Tcl's slower getvar */
	    return Tcl_SetVar2 ((Tcl_Interp*) iptr, name, index, value,
				TCL_LEAVE_ERR_MSG);
	 }
	 if (varptr->hPtr == NULL) {
	    sprintf (iptr->result, "Dangling upvar : %.50s(%.50s)", 
		     name, index);
	    return NULL;
	 }
      }
   }
   if (varptr->flags & VAR_ARRAY) {
      sprintf (iptr->result, "Not a scalar : %.50s %.50s", 
	       name, index);
      return NULL;
   }
   if (varptr->valueSpace <= len) {
      int newsize = len + 1;
      if (newsize < 24) newsize = 24;
      if (varptr->valueSpace == 0) ckfree (varptr->value.string);
      varptr->value.string = ckalloc (newsize);
   }
   strcpy (varptr->value.string, value);
   varptr->valueLength = len;
   varptr->flags &= ~VAR_UNDEFINED;
   return varptr->value.string;
}
				   
static int
SubstOneValue (Tcl_Interp *interp, 	   /* Interp for returning errors */
	       ValStruct *source,	   /* Source value */
	       register ParseBuf *destbuf) /* (Out) ptr to substituted word */
/*
 * Returns a fully substituted version of source in *destbuf.
 */
{
   Interp *iptr = (Interp*) interp;
   int result;
   int count;
   char *value;

   do { 
      switch (source->type) {
	 case LiteralValue: {
	    count = strlen (source->detail.literal);
	    ROOMFOR (count, *destbuf);
	    memcpy (destbuf->next, source->detail.literal, count);
	    destbuf->next += count;
	    break;
	 }
	 case CommandValue: {
	    result = ExecCommand (interp, source->detail.cmd);
	    if (result != TCL_OK) goto done;
	    count = strlen (interp->result);
	    ROOMFOR (count, *destbuf);
	    memcpy (destbuf->next, interp->result, count);
	    destbuf->next += count;
	    break;
	 }
	 case VarValue: {	    
	    if (source->detail.varvalue.index == NULL) {
	       value = MyGetVar (iptr, source->detail.varvalue.name, NULL);
	    } 
	    else {
	       char *first = destbuf->next;
	       result = SubstOneValue (interp,
				       source->detail.varvalue.index,
				       destbuf);
	       if (result != TCL_OK) {
		  goto done;
	       }
	       destbuf->next = first;
	       value = MyGetVar (iptr, source->detail.varvalue.name, first);
	    }
	    if (value == NULL) {
	       result = TCL_ERROR;
	       goto done;
	    }
	    count = strlen (value);
	    ROOMFOR (count, *destbuf);
	    memcpy (destbuf->next, value, count);
	    destbuf->next += count;
	    break;
	 }
	 default: {
	    printf ("Unknowkn value");
	    assert (0);
	 }
      }
      source = source->next;
   } while (source != NULL);

   result = TCL_OK;

done:
   ROOMFOR (1, *destbuf);
   *destbuf->next++ = '\0';
   return result;
}

int 
ExecCommand (Tcl_Interp * interp,    	/* Interp for returning result/errors*/
	     CmdStruct * cmdlist)	/* Command list to be executed */
/*
 * Executes command cmdlist.
 *
 * Returns (hopefully) whatever should be the result of evaluating
 * the string that was compiled into the 'cmd', including possible errors.
 */
{
   register int i;
   register Interp* iptr = (Interp*) interp;
   register CmdStruct* cmd = cmdlist;
   Tcl_HashEntry *hptr;
   Command *tclcmdptr;
   ParseBuf buf;
   int offsetStorage [PREALLOC_SIZE];
   int *offset = offsetStorage;   
   int noffset = PREALLOC_SIZE;
   int argc;
   int value;
   register int result;

   INITBUF (buf);

   /* Loop to execute all commands in the compiled command list */
   for (;;) {
      
      /* Eval each type appropriately ... */
      switch (cmd->type) {
	 case SetCmd: {
	    int indexoffset;
	    int valueoffset;
	    char *value;
	    char *index;
	    buf.next = buf.first;
	    result = SubstOneValue (interp, 
				    cmd->detail.setcmd.name,
				    &buf);
	    if (result != TCL_OK) goto done;
	    if (cmd->detail.setcmd.index != NULL) {
	       indexoffset = buf.next - buf.first;
	       result = SubstOneValue (interp, 
				       cmd->detail.setcmd.index,
				       &buf);
	       if (result != TCL_OK) goto done;
	    }
	    else {
	       indexoffset = -1;
	    }
	    if (cmd->detail.setcmd.value != NULL) {
	       valueoffset = buf.next - buf.first;
	       result = SubstOneValue (interp, 
				       cmd->detail.setcmd.value,
				       &buf);
	       if (result != TCL_OK) goto done;
	       index = indexoffset < 0 ? NULL : buf.first + indexoffset;
	       value = MySetVar (iptr, buf.first, index, 
				 buf.first + valueoffset);
	       if (value == NULL) {
		  result = TCL_ERROR;
		  goto done;
	       }	       
	    }
	    else {
	       index = indexoffset < 0 ? NULL : buf.first + indexoffset;
	       value = MyGetVar (iptr, buf.first, index);
	       if (value == NULL) {
		  result = TCL_ERROR;
		  goto done;
	       }
	    }
	    interp->result = value;
	    break;
	 }
	 case IncrCmd: {
	    char *name;
	    char *index;
	    char *value;
	    char *increment;
	    int oldvalue;
	    int	n;
	    buf.next = buf.first;
	    name = buf.first;
	    result = SubstOneValue (interp, 
				    cmd->detail.incrcmd.name,
				    &buf);
	    if (result != TCL_OK) goto done;
	    if (cmd->detail.incrcmd.index != NULL) {
	       index = buf.next;
	       result = SubstOneValue (interp, 
				       cmd->detail.incrcmd.index,
				       &buf);
	       if (result != TCL_OK) goto done;
	    }
	    else {
	       index = NULL;
	    }
	    if (cmd->detail.incrcmd.increment != NULL) {
	       increment = buf.next;
	       result = SubstOneValue (interp, 
				       cmd->detail.incrcmd.increment,
				       &buf);
	       if (result != TCL_OK) goto done;
	       result = Tcl_GetInt (interp, increment, &n);
	       if (result != TCL_OK) goto done;	    
	    }
	    else {
	       n = 1;
	    }
	    value = MyGetVar (iptr, name, index);
	    if (value == NULL) {
	       result = TCL_ERROR;
	       goto done;
	    }
	    result = Tcl_GetInt (interp, value, &oldvalue);
	    if (result != TCL_OK) goto done;
	    oldvalue += n;
	    sprintf (interp->result, "%d", oldvalue);
	    value = MySetVar (iptr, name, index, interp->result);
	    if (value == NULL) {
	       result = TCL_ERROR;
	       goto done;
	    }
	    interp->result = value;
	    break;
	 }
	 case IfCmd: {
	    for (i = 0; i < cmd->detail.ifcmd.ncond; i++) {
	       buf.next = buf.first;
	       result = SubstOneValue (interp, 
				       cmd->detail.ifcmd.cond [i],
				       &buf);
	       if (result != TCL_OK) goto done;
	       result = Tcl_ExprBoolean(interp, buf.first, &value);
	       if (result != TCL_OK) goto done;
	       if (value) {
		  result = ExecCommand (interp, 
					cmd->detail.ifcmd.body [i]);
		  if (result != TCL_OK) goto done;
		  break;
	       }
	    }
	    if (i == cmd->detail.ifcmd.ncond && 
		cmd->detail.ifcmd.elsebody != NULL) {
	       result = ExecCommand (interp, cmd->detail.ifcmd.elsebody);
	       if (result != TCL_OK) goto done;
	    }
	    break;
	 }  
	 case WhileCmd: {
	    buf.next = buf.first;
	    result = SubstOneValue (interp,
				    cmd->detail.whilecmd.cond,
				    &buf);
	    if (result != TCL_OK) goto done;
	    for (;;) {
	       result = Tcl_ExprBoolean(interp, buf.first, &value);
	       if (result != TCL_OK) goto done;
	       if (!value) break;
	       result = ExecCommand (interp, cmd->detail.whilecmd.body);
	       if (result != TCL_OK) {
		  if (result == TCL_BREAK) {
		     result = TCL_OK;
		     break;
		  }
		  if (result == TCL_ERROR) {
		     goto done;
		  }
	       }
	    }
	    break;
	 }
	 case RuntimeCmd: {
	    buf.next = buf.first;
	    result = SubstOneValue (interp,
				    cmd->detail.runtimecmd.body,
				    &buf);
	    if (result != TCL_OK) goto done;
	    result = Tcl_Eval (interp, buf.first);
	    if (result != TCL_OK) goto done;
	    break;
	 }
	 case OtherCmd: {
	    /* Build argv array */
	    char **argv;
	    argc = cmd->detail.othercmd.argc;
	    if (argc >= noffset) {
	       /* Not enough space in offset. Realloc offset.
		* Get one more entry in case we have to exec 'unknown' */
	       int oldnoffset = noffset;
	       noffset = cmd->detail.othercmd.argc+1;
	       if (offset == offsetStorage) {
		  offset = (int*) ckalloc (sizeof (int) * noffset);
		  memcpy (offset, offsetStorage, sizeof(int)*oldnoffset);
	       }
	       else {
		  offset = (int*) ckrealloc (offset, sizeof (int) * noffset);
	       }
	    }
	    buf.next = buf.first;
	    for (i = 0; i < argc; i++) {
	       offset [i] = buf.next - buf.first;
	       result = SubstOneValue (interp,
				       cmd->detail.othercmd.words [i],
				       &buf);
	       if (result != TCL_OK) goto done;
	    }
	    assert (sizeof (char*) == sizeof (int));
	    argv = (char**) offset;
	    for (i = 0; i < argc; ++i) {
	       argv [i] = offset [i] + buf.first;
	    }
	    /* Find out proc to call if not already known at compile time */
	    tclcmdptr = cmd->detail.othercmd.tclcmdptr;
	    if (tclcmdptr == NULL) {
	       /* Look it up in the command table */
	       hptr = Tcl_FindHashEntry (&iptr->commandTable, buf.first);
	       if (hptr == NULL) {
		  /* Not a known command. Try to exec command 'unknown' */
		  hptr = Tcl_FindHashEntry (&iptr->commandTable, buf.first);
		  if (hptr == NULL) {
		     /* No 'unknown' procedure defined */
		     Tcl_ResetResult (interp);
		     Tcl_AppendResult (interp, "invalid command name '",
				       buf.first, "'", (char*) NULL);
		     result = TCL_ERROR;
		     goto done;
		  }
		  /* Shift arguments so that 'unknown' can be first arg */
		  for (i = argc; i>= 0; i--) {
		     argv [i+1] = argv [i];
		  }
		  argv [0] = "unknown";
	       }
	       tclcmdptr = (Command*) Tcl_GetHashValue (hptr);
	    } 
	    /* Reset the result (could have been changed by previous cmd) */
	    Tcl_FreeResult (iptr);
	    iptr->result = iptr->resultSpace;
	    iptr->resultSpace [0] = '\0';     
	    /* Invoke command procedure */
	    result = (*tclcmdptr->proc) (tclcmdptr->clientData, 
					 interp, argc, argv);
	    /* 
	     * did not exactly understand what goes on below, but it was
	     * in the original tvlEval, so ...
	     */
	    if (tcl_AsyncReady) {
	       result = Tcl_AsyncInvoke (interp, result);
	    }
	    if (result != TCL_OK) goto done;
	    break;
	 }
	 case NullCmd: {
	    result = TCL_OK;
	    break;
	 }
	 default: {
	    printf ("unknown command\n");
	    assert (0);
	 }
      }
      cmd = cmd->next;
      if (cmd == NULL) break;

      /* Reset the result (could have been changed by previous cmd) */
      Tcl_FreeResult (iptr);
      iptr->result = iptr->resultSpace;
      iptr->resultSpace [0] = '\0'; 

   }

done:
   FREEBUF (buf);
   /* Free offset */
   if (offset != offsetStorage) {
      ckfree (offset);
   }
   if (result == TCL_ERROR) {
      Tcl_DString ds;
      Tcl_DStringInit (&ds);
      Tcl_DStringAppend (&ds, "Error while executing: ", -1);
      PrintCommand (&ds, cmdlist);
      if (Tcl_DStringLength (&ds) > 150) {
	 Tcl_DStringTrunc (&ds, 150);
	 Tcl_DStringAppend (&ds, " ... ", -1);
      }
      Tcl_AddErrorInfo (interp, Tcl_DStringValue (&ds));
      Tcl_DStringFree (&ds);
   }
   return result;
}

static int 
CopyBraces (Tcl_Interp *interp,		/* Interp for returning errors */
	    char **source, 		/* (In/Out) source string to parse */
	    char **dest)		/* (In/Out) dest string */
/* 
 * At the beginning, *source points to a the first char of a string in
 * braces (just after the left brace). At the end, *source is set to
 * the character just after the right brace. All characters between
 * the first (inclusive) and the last (exclusive) position of *source are
 * copied to *dest which is also updated to point to just after
 * the last copied character.
 * 
 * Returns a standard Tcl result.
 */
{
   char *src = *source;
   char *dst = *dest;
   int result = TCL_ERROR;
   char c;

   for (c = *src++; c != '\0' && c != '}'; c = *src++) {
      *dst++ = c;
      if (CHAR_TYPE (c) != TCL_NORMAL) {
	 if (c == '{') {
	    if (CopyBraces (interp, &src, &dst) != TCL_OK) goto error;
	 }
	 else if (c == '\\') {
	    if (*src == '\n') {
	       /* Just forget it */
	       dst--;
	       src++;
	    }
	    else {
	       /* Copy ipsis literis the backslash sequence */
	       int count;
	       (void) Tcl_Backslash (src-1, &count);
	       while ((count--)>1) { *dst++ = *src++; } 
	    }	    
	 }
      }
   }
   if (c != '}') {
      interp->result = "Missing end-brace";
      goto error;
   }
   *dst++ = c;
   result = TCL_OK;
error:
   *source = src;
   *dest = dst;
   return result;
}

static int 
CompileValue (Tcl_Interp * interp,	/* Interp for returning errors */
	      char* src,		/* string to parse */
	      ValStruct **valptr)	/* (Out) Where value code is returned*/
/*
 * Generates a chained list of ValStruct nodes that will
 * be processed at runtime to produce a string.
 */
{
   int result = TCL_ERROR;
   char c;
   ValStruct **headvalptr = valptr;
   ValStruct *index;
   ValStruct *value;
   CmdStruct *cmd;
   char *first;
   char *last;
   ParseBuf buf;

   INITBUF (buf);
   
   *headvalptr = NULL;

   while (*src != '\0') {
      c = *src;
      switch (c) {
	 case '[' : {
	    /*
	     * Copy everything between square brackets to a buffer and
	     * compile it.
	     */	    
	    int level = 1;
	    buf.next = buf.first;
	    for (c = *++src; c != '\0' && level > 0; c = *++src) {
	       ROOMFOR (1, buf);
	       if (c == '[') {
		  level++;
		  *buf.next++ = c;
	       }
	       else if (c == ']') {
		  level--;
		  if (level > 0) *buf.next++ = c;
	       }
	       else {
		  *buf.next++ = c;
	       }
	    }
	    if (level > 0) {
	       interp->result = "missing ']'";
	       goto done;
	    }
	    ROOMFOR (1, buf);
	    *buf.next = '\0';
	    result = CompileCommand (interp, buf.first, &cmd);
	    if (result != TCL_OK) goto done;
	    value = (ValStruct*) ckalloc (sizeof (ValStruct));
	    value->type = CommandValue;
	    value->next = NULL;
	    value->detail.cmd = cmd;
	    *valptr = value;
	    valptr = &(value->next);
	    break;
	 }
	 case '$' : {
	    c = *++src;
	    if (c == '{') {
	       first = ++src;
	       while (*src != '}') {
		  if (*src == '\0') {
		     interp->result = "missing '}'";
		     goto done;
		  }
		  src++;
	       }
	       last = src++;
	       index = NULL;
	    } 
	    else {
	       first = src;
	       while (isalnum (UCHAR(*src)) || (*src == '_')) {
		  src++;
	       }
	       last = src;
	       if (first == last) {
		  goto commonchar;
	       }
	       if (*src == '(') {
		  buf.next = buf.first;
		  for (c = *++src; c != ')'; c = *++src) {
		     if (*src == '\0') {
			interp->result = "missing ')'";
			goto done;
		     }
		     ROOMFOR (1, buf);
		     *buf.next++ = c;
		  }
		  src++;
		  ROOMFOR (1, buf);	
		  *buf.next = '\0';
		  result = CompileValue (interp, buf.first, &index);
		  if (result != TCL_OK) {
		     goto done;
		  }		  
	       }
	       else {
		  index = NULL;
	       }
	    }
	    value = (ValStruct*) ckalloc (sizeof (ValStruct));
	    value->type = VarValue;
	    value->next = NULL;
	    value->detail.varvalue.name = ckalloc (last-first+1);
	    strncpy (value->detail.varvalue.name, first, last-first);
	    value->detail.varvalue.name [last-first] = '\0';
	    value->detail.varvalue.index = index;
	    *valptr = value;
	    valptr = &(value->next);
	    break;
	 }
	 default: {
	    commonchar:
	    first = src;
	    do {
	       c = *++src;
	    } while (c != '\0' && c != '$' && c != '[');
	    last = src;
	    value = (ValStruct*) ckalloc (sizeof (ValStruct));
	    value->type = LiteralValue;
	    value->next = NULL;
	    value->detail.literal = ckalloc (last-first+1);
	    strncpy (value->detail.literal, first, last-first);
	    value->detail.literal [last-first] = '\0';
	    *valptr = value;
	    valptr = &(value->next);
	 }
      }
   }
   result = TCL_OK;
done:
   FREEBUF (buf);
   if (result != TCL_OK && *headvalptr != NULL) { 
      FreeValue (*headvalptr);
      *headvalptr = NULL;
   }
   return result;
}
   
static int
CompileOneWord (Tcl_Interp * interp,	/* Interp for returning errors */
		char **stringptr,	/* (In/Out) String to parse */
		ValStruct **valptr)	/* (Out) Where word code is returned*/
/*
 * Stores in *value one word of a command.
 * If no more words exist in string, returns a value of type NullValue.
 * Returns a standard Tcl result.
 */
{
   char *src = *stringptr;
   char *beginword;
   char c;
   int result = TCL_ERROR;
   int nchars;
   ValStruct *value;
   ParseBuf buf;

   INITBUF (buf);
   
   /* skip spaces */
   for (c = *src; c != '\0' && CHAR_TYPE (c) == TCL_SPACE; c = *++src);

   /* Return a NullValue if string is finished */
   if (c == '\0') {
      value = (ValStruct*) ckalloc (sizeof (ValStruct));
      value->type = NullValue;
      value->next = NULL;
      goto done;
   }

   /* Mark the beginning of the string */
   beginword = src;

   if (c == '{') {
      /*
       * A word beginning with a brace has to be a constant string 
       */
      char *dst = ++src;

      /* We know we can set src and dst to the same string because
       * CopyBraces can only shrink a string but not lengthen it 
       */
      if (CopyBraces (interp, &src, &dst) != TCL_OK) goto error;
      nchars = dst - beginword - 2;
      value = (ValStruct*) ckalloc (sizeof (ValStruct));
      value->type = LiteralValue;
      value->next = (ValStruct*) NULL;
      value->detail.literal = ckalloc (nchars+1);
      strncpy (value->detail.literal, beginword+1, nchars);
      value->detail.literal [nchars] = '\0';
   }
   else {
      /*
       * It is a word that eventually will contain substitutions that
       * will have to be resolved at runtime. Here, we identify the
       * end of the word and try to generate code for it
       * with a call to CompileValue.
       */

      if (c == '\"') {
	 /* Copy everything until but not including the endquote */
	 for (c = *++src; c != '\"'; c = *++src) {
	    ROOMFOR (1, buf);
	    if (c == '\0') {
	       interp->result = "Unbalanced quote";
	       goto error;
	    }
	    if (c == '\\') {
	       /* Substitute backslash sequence */
	       *buf.next++ = Tcl_Backslash (src, &nchars);
	       src += nchars-1;
	    }
	    else {
	       *buf.next++ = c;
	    }
	 }
	 ++src; /* Eat ending quote */
      }
      else {
	 /* Copy everything until finding a space */
	 int level = 0;
	 while (c != '\0' && (level>0 || CHAR_TYPE(c)!=TCL_SPACE)) {
	    ROOMFOR (1, buf);
	    if (c == '\\') {
	       /* Substitute backslash sequence */
	       *buf.next++ = Tcl_Backslash (src, &nchars);
	       src += nchars;
	       c = *src;
	    }
	    else {
	       *buf.next++ = c;
	       if (c == '[') {
		  level++;
	       }
	       else if (c == ']') {
		  level--;
	       }
	       c = *++src;
	    }
	 }
	 if (level>0) {
	    interp->result = "Missing ']'";
	    goto error;
	 }
      }
      ROOMFOR (1, buf);
      *buf.next = '\0';
      if (CompileValue (interp, buf.first, &value) != TCL_OK) {
	 goto error;
      }
   }

done:
   *valptr = value;
   *stringptr = src;
   result = TCL_OK;

error:
   FREEBUF (buf);
   return result;
}

static int 
CompileSubCommand (Tcl_Interp *interp,	/* Interp for returning errors */
		   ValStruct *word,	/* Where the subcommand is stored
					 * in string form */
		   CmdStruct **cmdptr)	/* Where the result of the compilation
					 * is stored */
/* 
 * Compiles the subcommand stored in 'word' if at all possible.
 *
 * Returns TCL_ERROR and an error message if compilation is not successful.
 */
{
   if (word->type == LiteralValue && 
       word->next == NULL) {
      /* Its a constant string, so try to compile it recursively */
      return CompileCommand (interp, word->detail.literal, cmdptr);
   }
   else {
      /* Command only known at run-time */
      *cmdptr = (CmdStruct*) ckalloc (sizeof (CmdStruct));
      (*cmdptr)->type = RuntimeCmd;
      (*cmdptr)->detail.runtimecmd.body = word;
      return TCL_OK;
   }
}

static int 
CompileIf (Tcl_Interp* interp, 	/* Interp for returning errors */
	   int nwords,		/* Number of arguments to command */
	   ValStruct **words,	/* Arguments to command */
	   CmdStruct **cmdptr)	/* Where the result of the compilation
				   is stored */
/*
 * Stores in *cmdptr a pointer to the compiled code of the 'if' command
 * as represented by the 'nwords' in 'words'. 
 *
 * If the compilation fails, interp is set to an error message,
 * *cmd is set to a NULL pointer, and TCL_ERROR is returned.
 */
{
   int iword = 1;
   int ncond = 0;
   int result = TCL_ERROR;
   int icond;
   ValStruct *cond [nwords];
   CmdStruct *body [nwords];
   CmdStruct *elsebody = (CmdStruct *) NULL;

   for (;;) {
      if (iword >= nwords) {
	 Tcl_AppendResult (interp, "wrong # args: missing condition after '",
			   words [iword-1]->detail.literal, "'", (char*) NULL);
	 goto error;
      }
      cond [ncond] = words [iword++];
      if (strcmp (words [iword]->detail.literal, "then") == 0) iword++;
      if (iword >= nwords) {
	 Tcl_AppendResult (interp, "wrong # args: missing script after '",
			   cond [ncond]->detail.literal, "'", (char*) NULL);
	 goto error;
      }

      if (CompileSubCommand (interp, words [iword++], &body [ncond])!= TCL_OK) {
	 goto error;
      }

      ncond++;

      if (iword >= nwords ||
	  strcmp (words [iword]->detail.literal, "elseif") != 0) break;

      iword++;
   }

   /* Compile else clause if it exists */
   if (iword < nwords) {
      if (strcmp (words [iword]->detail.literal, "else") == 0) iword++;
      if (iword == nwords) {
	 Tcl_AppendResult (interp, "wrong # args: missing script after 'else'",
			   (char*) NULL);
	 goto error;
      }
      if (CompileSubCommand (interp, words [iword], &elsebody) != TCL_OK) {
	 goto error;
      }      
   }

   (*cmdptr) = (CmdStruct*) ckalloc (sizeof (CmdStruct));
   (*cmdptr)->type = IfCmd;
   (*cmdptr)->next = (CmdStruct*) NULL;
   (*cmdptr)->detail.ifcmd.ncond = ncond;
   (*cmdptr)->detail.ifcmd.cond = (ValStruct**) ckalloc (sizeof (ValStruct*)*
							 ncond);
   (*cmdptr)->detail.ifcmd.body = (CmdStruct**) ckalloc (sizeof (CmdStruct*)*
							 ncond);
   (*cmdptr)->detail.ifcmd.elsebody = elsebody;
   for (icond = 0; icond < ncond; icond++) {
      (*cmdptr)->detail.ifcmd.cond [icond] = cond [icond];
      (*cmdptr)->detail.ifcmd.body [icond] = body [icond];
   }
   result = TCL_OK;

error:
   if (result != TCL_OK) {
      for (icond = 0; icond < ncond; icond++) {
	 ckfree (cond [icond]);
	 ckfree (body [icond]);
      }
   }
   return result;
}

static int 
CompileWhile (Tcl_Interp* interp, 	/* Interp for returning errors */
	      int nwords,		/* Number of arguments to command */
	      ValStruct **words,	/* Arguments to command */
	      CmdStruct **cmdptr)	/* Where the result of the compilation
					   is stored */
/*
 * Stores in *cmdptr a pointer to the compiled code of the 'while' command
 * as represented by the 'nwords' in 'words'. 
 *
 * If the compilation fails, interp is set to an error message,
 * *cmd is set to a NULL pointer, and TCL_ERROR is returned.
 */
{
   ValStruct *cond;
   CmdStruct *body = (CmdStruct *) NULL;
   *cmdptr = (CmdStruct *) NULL;

   if (nwords != 3) {
      Tcl_AppendResult (interp, "wrong # of args: should be",
			"'while <test> <command>'", (char*)NULL);
      return TCL_ERROR;
   }

   cond = words [1];
   if (CompileSubCommand (interp, words [2], &body)!= TCL_OK) {
      return TCL_ERROR;
   }

   (*cmdptr) = (CmdStruct*) ckalloc (sizeof (CmdStruct));
   (*cmdptr)->type = WhileCmd;
   (*cmdptr)->next = (CmdStruct*) NULL;
   (*cmdptr)->detail.whilecmd.cond = cond;
   (*cmdptr)->detail.whilecmd.body = body;

   return TCL_OK;
}

static void
SplitVarName (ValStruct **var,		/* (In/Out) variable name */
	      ValStruct **index)	/* (Out) array index val */
/*
 * Used for separating a ValStruct with a var specification into 
 * array name and index.
 */
{
   ValStruct **src = var;
   ValStruct **lparenpiece;
   ValStruct **rparenpiece;
   char *lparenchar;
   char *rparenchar;

   do {
      if ((*src)->type == LiteralValue) {
	 lparenchar = strchr ((*src)->detail.literal, '(');
	 if (lparenchar != NULL) {
	    lparenpiece = src;
	    while ((*src)->next != NULL) src = &((*src)->next);
	    rparenpiece = src;
	    if ((*src)->type != LiteralValue) {
	       break;
	    }
	    rparenchar = strrchr ((*src)->detail.literal, ')');
	    if (rparenchar == NULL || *(rparenchar+1) != '\0') break;
	    goto indexfound;	    
	 }
      }
      src = &((*src)->next);
   } while (*src != NULL);

   *index = (ValStruct *) NULL;
   return;

indexfound:
   if (lparenchar [1] == '\0') {
      *index = (*lparenpiece)->next;
   } 
   else {
      *index = (ValStruct *) ckalloc (sizeof (ValStruct));
      (*index)->type = LiteralValue;
      (*index)->next = (*lparenpiece)->next;
      (*index)->detail.literal = ckalloc (strlen (lparenchar+1)+1);
      strcpy ((*index)->detail.literal, lparenchar+1);
      if (lparenpiece == rparenpiece) {
	 rparenpiece = index;
	 rparenchar = (*index)->detail.literal + 
	    strlen ((*index)->detail.literal) - 1;	 
      }
   }
   (*lparenpiece)->next = (ValStruct*) NULL;
   *lparenchar = '\0';
   *rparenchar = '\0';
}
   
   
CompileSet (Tcl_Interp* interp, 	/* Interp for returning errors */
	    int nwords,			/* Number of arguments to command */
	    ValStruct **words,		/* Arguments to command */
	    CmdStruct **cmdptr)		/* Where the result of the compilation
					   is stored */
/*
 * Stores in *cmdptr a pointer to the compiled code of the 'set' command
 * as represented by the 'nwords' in 'words'. 
 *
 * If the compilation fails, interp is set to an error message,
 * *cmd is set to a NULL pointer, and TCL_ERROR is returned.
 */
{
   ValStruct * index;

   if (nwords != 2 && nwords != 3) {
      Tcl_AppendResult (interp, "wrong # of args: should be",
			"'set <var>' or 'set <var> <value>'", (char*)NULL);
      return TCL_ERROR;
   }

   SplitVarName (&(words [1]), &index);
   
   (*cmdptr) = (CmdStruct*) ckalloc (sizeof (CmdStruct));
   (*cmdptr)->type = SetCmd;
   (*cmdptr)->next = (CmdStruct*) NULL;
   (*cmdptr)->detail.setcmd.name = words [1];
   (*cmdptr)->detail.setcmd.index = index;
   
   if (nwords == 3) {
      (*cmdptr)->detail.setcmd.value = words [2];
   }	
   else {
      (*cmdptr)->detail.setcmd.value = (ValStruct *) NULL;
   }
   return TCL_OK;
}

            
CompileIncr (Tcl_Interp* interp, 	/* Interp for returning errors */
	     int nwords,		/* Number of arguments to command */
	     ValStruct **words,		/* Arguments to command */
	     CmdStruct **cmdptr)	/* Where the result of the compilation
					   is stored */
/*
 * Stores in *cmdptr a pointer to the compiled code of the 'incr' command
 * as represented by the 'nwords' in 'words'. 
 *
 * If the compilation fails, interp is set to an error message,
 * *cmd is set to a NULL pointer, and TCL_ERROR is returned.
 */
{
   ValStruct * index;

   if (nwords != 2 && nwords != 3) {
      Tcl_AppendResult (interp, "wrong # of args: should be",
			"'set <var>' or 'set <var> <value>'", (char*)NULL);
      return TCL_ERROR;
   }

   SplitVarName (&(words [1]), &index);
   
   (*cmdptr) = (CmdStruct*) ckalloc (sizeof (CmdStruct));
   (*cmdptr)->type = IncrCmd;
   (*cmdptr)->next = (CmdStruct*) NULL;
   (*cmdptr)->detail.incrcmd.name = words [1];
   (*cmdptr)->detail.incrcmd.index = index;
   
   if (nwords == 3) {
      (*cmdptr)->detail.incrcmd.increment = words [2];
   }	
   else {
      (*cmdptr)->detail.incrcmd.increment = (ValStruct *) NULL;
   }
   return TCL_OK;
}

            
static int
CompileOneCommand (Tcl_Interp *interp, 	/* Interp for returning errors */
		   char *string, 	/* String to parse */
		   CmdStruct **cmdptr)	/* Where the result of the compilation
					   is stored */
/*
 * Compiles one command putting the "code" in *cmdptr. If an error
 * occurs, *cmdptr is set to NULL and TCL_ERROR is returned.
 *
 * Returns a standard Tcl result.
 */

{
   char *src = string;
   int result = TCL_ERROR;
   int wordspace = 0;
   int nwords = 0;
   int iword;
   ValStruct **words = (ValStruct**) NULL;
   ValStruct *word;
   Tcl_HashEntry *hptr;
   Command* tclcmdptr;
   
   /* Get First word */
   if (CompileOneWord (interp, &src, &word) != TCL_OK) goto error;

   /* See if it is a null command */
   if (word->type == NullValue || word->detail.literal [0] == '#') {
      /* A comment or an empty string. Return a null command */
      FreeValue (word);
      *cmdptr = (CmdStruct*) ckalloc (sizeof (CmdStruct));
      (*cmdptr)->type = NullCmd;
      (*cmdptr)->next = (CmdStruct*) NULL;
      goto done;
   }

   /* Build the list of arguments for the command */
   nwords = 0;
   wordspace = PREALLOC_SIZE;
   words = (ValStruct**) ckalloc (sizeof(ValStruct*) * wordspace);
   do {      
      if (wordspace == nwords) {
	 wordspace *= 2;
	 words = (ValStruct**) ckrealloc (words, 
					  sizeof(ValStruct*) * wordspace);
      }
      words [nwords++] = word;
      if (CompileOneWord (interp, &src, &word) != TCL_OK) {
	 goto error;
      }
   } while (word->type != NullValue);
   
   /* Try to figure out a tcl "Command" to execute at run-time */
   if (words [0]->type == LiteralValue) {
      hptr = Tcl_FindHashEntry (&commandTable, words [0]->detail.literal);
      if (hptr == NULL) {
	 /* It is a command unknown at compile time */
	 tclcmdptr = (Command*) NULL;
      }
      else {
	 /* It is a builtin command */
	 tclcmdptr = (Command*) Tcl_GetHashValue(hptr);
      }
   }
   else {
      /* It is a command unknown at compile time */
      tclcmdptr = (Command*) NULL;
   }

   /* If it's a built in command, we should try to compile it further */
   if (tclcmdptr != (Command*) NULL) {
      *cmdptr = NULL;
      if (tclcmdptr->proc == Tcl_IfCmd) {
	 if (CompileIf (interp, nwords, words, cmdptr) != TCL_OK) {
	    goto error;
	 }
      }
      else if (tclcmdptr->proc == Tcl_WhileCmd) {
	 if (CompileWhile (interp, nwords, words, cmdptr) != TCL_OK) {
	    goto error;
	 }
      }
      else if (tclcmdptr->proc == Tcl_SetCmd) {
	 if (CompileSet (interp, nwords, words, cmdptr) != TCL_OK) {
	    goto error;
	 }
      }
      else if (tclcmdptr->proc == Tcl_IncrCmd) {
	 if (CompileIncr (interp, nwords, words, cmdptr) != TCL_OK) {
	    goto error;
	 }
      }

      if (*cmdptr != NULL) goto done;
   }

   /* At this point, a command is either not a built-in command or
    * we haven't created a specific compiled structure for it. Thus it
    * goes into the catch-all bag of 'OtherCmd'
    */
   
   *cmdptr = (CmdStruct*) ckalloc (sizeof (CmdStruct));
   (*cmdptr)->type = OtherCmd;
   (*cmdptr)->next = (CmdStruct*) NULL;
   (*cmdptr)->detail.othercmd.argc = nwords;
   (*cmdptr)->detail.othercmd.tclcmdptr = tclcmdptr;
   (*cmdptr)->detail.othercmd.words = ckalloc (sizeof(ValStruct*) * nwords);
   for (iword = 0; iword < nwords; ++iword) {
      (*cmdptr)->detail.othercmd.words [iword] = words [iword];
   }

done:
   result = TCL_OK;

error:
   if (result != TCL_OK) {
      Tcl_AddErrorInfo (interp, "\n while compiling ");
      Tcl_AddErrorInfo (interp, string);
      for (iword = 0; iword < nwords; ++iword) {
	 FreeValue (words [iword]);
      }
      *cmdptr = NULL;
   }

   if (words != NULL) {
      ckfree (words);
   }
      
   return result;
}


int
CompileCommand (Tcl_Interp* interp, 	/* Interp for returning errors */
		char * string, 	  	/* String to parse */
		CmdStruct ** cmd)	/* Where the result of the compilation
					   is stored */
/*
 * Breaks up string into separate commands and compiles them. The 
 * compiled code is put into *cmd. If a compilation error occurs,
 * *cmd is set to NULL and the result is TCL_ERROR.
 */
{
   char *copy = ckalloc (strlen (string)+1);
   char *dst = copy;
   char *src = string;
   char c;
   int result = TCL_ERROR;
   CmdStruct **cmdptr = cmd;
   
   *cmd = NULL;

   for (c = *src++; c != '\0'; c = *src++) {
      *dst++ = c;
      if (CHAR_TYPE (c) != TCL_NORMAL) {
	 if (c == '{') {
	    if (CopyBraces (interp, &src, &dst) != TCL_OK) goto error;
	 }
	 else if (c == '\\') {
	    if (*src == '\n') {
	       /* Just forget it */	       
	       dst--;
	       src++;
	    }
	    else {
	       /* Copy ipsis literis the backslash sequence */
	       int count;
	       (void) Tcl_Backslash (src-1, &count);
	       while ((count--)>1) { *dst++ = *src++; } 
	    }	    
	 }
	 else if (c == '\n' || c == ';') {
	    dst[-1] = '\0';
	    if (CompileOneCommand (interp, copy, cmdptr) != TCL_OK) {
	       return TCL_ERROR;
	    }
	    dst = copy;
	    if ((*cmdptr)->type == NullCmd) {
	       /* No need for a null command at this point*/
	       FreeCmd (*cmdptr);
	       *cmdptr = (CmdStruct*) NULL;
	    } else {
	       /* There's more than 1 command. Prepare for compiling the next*/
	       cmdptr = &((*cmdptr)->next);
	    }
	 }
      }
   }

   *dst = '\0';
   result = CompileOneCommand (interp, copy, cmdptr);
   if (result == TCL_OK && (*cmdptr)->type == NullCmd) {
      /* Don't generate code for superfluous null commands, but 
       * must generate code for at least one command. */
      if (cmdptr != cmd) {
	 FreeCmd (*cmdptr);
	 *cmdptr = (CmdStruct*) NULL;
      }
   }

error:
   if (result != TCL_OK && *cmd != NULL) {
      FreeCmd (*cmd);
      *cmd = (CmdStruct*) NULL;
   }
   ckfree (copy);
   return result;
}	       


int
Tcl_CompEvalFile (Tcl_Interp *interp,	/* Interpreter in which to 
					 * process file. */
		  char *fileName)   	/* Name of file to process.  
					 *  Tilde-substitution
					 * will be performed on this name. */
/*
 * Read in a file and process the entire file as one gigantic
 * Tcl command.
 * This proc is an almost exact copy of Tcl_EvalFile, except that
 * CompileCommand, ExecCommand and FreeCmd are used to process
 * the tcl string, instead of Tcl_Eval.
 */
{
    int fileId, result;
    struct stat statBuf;
    char *cmdBuffer, *oldScriptFile;
    Interp *iPtr = (Interp *) interp;
    Tcl_DString buffer;
    CmdStruct * code;
    char msg[200];

    Tcl_ResetResult(interp);
    oldScriptFile = iPtr->scriptFile;
    iPtr->scriptFile = fileName;
    fileName = Tcl_TildeSubst(interp, fileName, &buffer);
    if (fileName == NULL) {
	goto error;
    }
    fileId = open(fileName, O_RDONLY, 0);
    if (fileId < 0) {
	Tcl_AppendResult(interp, "couldn't read file \"", fileName,
		"\": ", Tcl_PosixError(interp), (char *) NULL);
	goto error;
    }
    if (fstat(fileId, &statBuf) == -1) {
	Tcl_AppendResult(interp, "couldn't stat file \"", fileName,
		"\": ", Tcl_PosixError(interp), (char *) NULL);
	close(fileId);
	goto error;
    }
    cmdBuffer = (char *) ckalloc((unsigned) statBuf.st_size+1);
    if (read(fileId, cmdBuffer, (size_t) statBuf.st_size) != statBuf.st_size) {
	Tcl_AppendResult(interp, "error in reading file \"", fileName,
		"\": ", Tcl_PosixError(interp), (char *) NULL);
	close(fileId);
	ckfree(cmdBuffer);
	goto error;
    }
    if (close(fileId) != 0) {
	Tcl_AppendResult(interp, "error closing file \"", fileName,
		"\": ", Tcl_PosixError(interp), (char *) NULL);
	ckfree(cmdBuffer);
	goto error;
    }
    cmdBuffer[statBuf.st_size] = 0;

    /* Compile and execute instead of eval */
    result = CompileCommand (interp, cmdBuffer, &code);
    ckfree(cmdBuffer);
    if (result == TCL_OK) {      
       result = ExecCommand (interp, code);
       if (result == TCL_RETURN) {
	  result = TCL_OK;
       }
       if (result == TCL_ERROR) {
	  sprintf (msg, "\nError executing code for file %.150s", fileName);
	  Tcl_AddErrorInfo(interp, msg);
       }       
       FreeCmd (code);
    }
    else {
       sprintf (msg, "\nError compiling file %.150s", fileName);
       Tcl_AddErrorInfo (interp, msg);
    }

    iPtr->scriptFile = oldScriptFile;
    Tcl_DStringFree(&buffer);
    return result;

    error:
    iPtr->scriptFile = oldScriptFile;
    Tcl_DStringFree(&buffer);
    return TCL_ERROR;
}
	
static int 
Tcl_Execute (ClientData data,		/* Pointer to compiled tcl code */
	     Tcl_Interp* interp,	/* Interp for returning result */
	     int argc,			/* arg count */
	     char *argv [])		/* arg array */
/*
 * Implements commands "<code> ? eval ?" "<code> free" and "<code> print"
 *
 * Returns result of the evaluation of <code> (for the eval option) 
 * or a null string otherwise.
 */
{
   int len = strlen (argv [0]);
   CmdStruct *code = (CmdStruct*) data;

   if (argc > 2) {
      Tcl_AppendResult (interp, "wrong # args. should be ",
			argv [0], " eval | free | print", (char*) NULL);
      return TCL_ERROR;
   }
  
   if (argc == 1 || strncmp (argv [1], "eval", len) == 0) {
      return ExecCommand (interp, code);
   }
   else if (strncmp (argv [1], "free", len) == 0) {
      char string [100];
      sprintf (string, "rename %s {}", argv [0]);
      return Tcl_Eval (interp, string);
   } 
   else if (strncmp (argv [1], "print", len) == 0) {
      Tcl_DString ds;
      Tcl_DStringInit (&ds);
      PrintCommand (&ds, code);
      Tcl_DStringResult (interp, &ds);
      return TCL_OK;
   }
   else {
      Tcl_AppendResult (interp, "illegal option: '", argv [1],
			"' should be 'eval', 'free' or 'print'", (char*) NULL);
      return TCL_ERROR;
   }
}

static int 
Tcl_Compile (ClientData dummy,		/* not used */
	     Tcl_Interp* interp,	/* Interp for returning errors */
	     int argc,			/* arg count */
	     char * argv []) 		/* arg array */
/*
 *
 * Implements command "compile <string>"
 *
 * Currently it calls CompileCommand, PrintCommand ExecCommand, FreeCmd
 */
{
   CmdStruct *code;
   static int codecount = 0;
   static char codeid [20];

   if (argc != 2) {
      Tcl_AppendResult (interp, "wrong # args: should be: ",
			argv [0], " <script> ", (char*)NULL);
      return TCL_ERROR;
   }

   if (CompileCommand (interp, argv [1], &code) != TCL_OK) {
      return TCL_ERROR;
   }

   sprintf (codeid, "code%d", codecount++);
   Tcl_CreateCommand (interp, codeid, Tcl_Execute, 
		      (ClientData) code, (Tcl_CmdDeleteProc*) FreeCmd);

   interp->result = codeid;   
   return TCL_OK;   
}

int 
Tcl_CompInit (Tcl_Interp* interp) 	/* Interp being initialized */
/*
 *
 * Inits the "compiler"
 */
{
   CmdInfo *cmdInfoPtr;
   Command *cmdPtr;
   int new;
   Tcl_HashEntry *hptr;

   /* Initialize the builtin command table */
   Tcl_InitHashTable (&commandTable, TCL_STRING_KEYS);
   for (cmdInfoPtr = builtInCmds; cmdInfoPtr->name != NULL; cmdInfoPtr++) {
      hptr = Tcl_CreateHashEntry (&commandTable, cmdInfoPtr->name, &new);
      if (new) {
	 cmdPtr = (Command*) ckalloc (sizeof (Command));
	 cmdPtr->proc = cmdInfoPtr->proc;
	 cmdPtr->clientData = (ClientData) NULL;
	 cmdPtr->deleteProc = NULL;
	 cmdPtr->deleteData = (ClientData) NULL;
	 Tcl_SetHashValue (hptr, cmdPtr);
      }
   }

   /* Create compiler-related commands */
   Tcl_CreateCommand (interp, "compile", Tcl_Compile, 
		      (ClientData) NULL, (Tcl_CmdDeleteProc*) NULL);
   Tcl_CreateCommand (interp, "csource", Tcl_CompSourceCmd, 
		      (ClientData) NULL, (Tcl_CmdDeleteProc*) NULL);
   return TCL_OK;
}


      
