/*
@(#)  FILE: tv_util.c  RELEASE: 1.3  DATE: 3/17/94, 10:43:07
*/
/*******************************************************************************

File:

    tv_util.c


Author:    Alex Measday, ISI


Purpose:

    These are utility functions used to manipulate UNIX "timeval"s.


Notes:

    These functions are reentrant under VxWorks.


Procedures:

    TV_ADD - adds two TIMEVALs.
    TV_COMPARE - compares two TIMEVALs.
    TV_CREATE - creates a TIMEVAL from a time expressed in seconds and
        microseconds.
    TV_CREATE_F - creates a TIMEVAL from a time expressed as a
        floating-point number of seconds.
    TV_DISPLAY - returns a printable representation of a TIMEVAL.
    TV_FLOAT - converts a TIMEVAL to a floating-point number of seconds.
    TV_SUBTRACT - subtracts one TIMEVAL from another.
    TV_TOD - returns the current time-of-day (GMT).

*******************************************************************************/


#include  <limits.h>			/* Maximum/minimum value definitions. */
#include  <stdio.h>			/* Standard I/O definitions. */
#include  <stdlib.h>			/* Standard C library definitions. */
#if defined(VMS)
#    include  <socket.h>		/* VMS "timeval" definitions. */
#    define  gmtime  localtime		/* VMS/UCX GMTIME() returns NULL. */
#elif defined(VXWORKS)
#    include  <time.h>			/* Time definitions. */
#    include  <timers.h>		/* POSIX timer definitions. */
#    include  <sys/times.h>		/* System time definitions. */
#else
#    include  <time.h>			/* Time definitions. */
#    include  <sys/time.h>		/* System time definitions. */
#endif
#include  "tv_util.h"			/* "timeval" manipulation functions. */

/*******************************************************************************

Procedure:

    tv_add ()


Purpose:

    Function TV_ADD returns the sum of two TIMEVALs.


    Invocation:

        result = tv_add (time_1, time_2) ;

    where

        <time_1>	- I
            is the first operand, a time represented by a UNIX TIMEVAL
            structure.
        <time_2>	- I
            is the second operand, a time represented by a UNIX TIMEVAL
            structure.
        <result>	- O
            returns, in a UNIX TIMEVAL structure, the sum of TIME_1 and
            TIME_2.

*******************************************************************************/


struct  timeval  tv_add (

#    if __STDC__ || defined(vaxc)
        struct  timeval  time_1,
        struct  timeval  time_2)
#    else
        time_1, time_2)

        struct  timeval  time_1 ;
        struct  timeval  time_2 ;
#    endif

{    /* Local variables. */
    struct  timeval  result ;


/* Add the two times together. */

    result.tv_sec = time_1.tv_sec + time_2.tv_sec ;
    result.tv_usec = time_1.tv_usec + time_2.tv_usec ;
    if (result.tv_usec > 1000000) {			/* Carry? */
        result.tv_sec++ ;  result.tv_usec = result.tv_usec - 1000000 ;
    }

    return (result) ;

}

/*******************************************************************************

Procedure:

    tv_compare ()


Purpose:

    Function TV_COMPARE compares two TIMEVALs.


    Invocation:

        comparison = tv_compare (time_1, time_2) ;

    where

        <time_1>	- I
            is a time represented by a UNIX TIMEVAL structure.
        <time_2>	- I
            is another time represented by a UNIX TIMEVAL structure.
        <comparison>	- O
            returns an integer indicating how the times compare:
                    -1  if TIME_1 < TIME_2
                     0  if TIME_1 = TIME_2
                    +1  if TIME_1 > TIME_2

*******************************************************************************/


int  tv_compare (

#    if __STDC__ || defined(vaxc)
        struct  timeval  time_1,
        struct  timeval  time_2)
#    else
        time_1, time_2)

        struct  timeval  time_1 ;
        struct  timeval  time_2 ;
#    endif

{

    if (time_1.tv_sec < time_2.tv_sec)
        return (-1) ;				/* Less than. */
    else if (time_1.tv_sec > time_2.tv_sec)
        return (1) ;				/* Greater than. */
    else if (time_1.tv_usec < time_2.tv_usec)
        return (-1) ;				/* Less than. */
    else if (time_1.tv_usec > time_2.tv_usec)
        return (1) ;				/* Greater than. */
    else
        return (0) ;				/* Equal. */

}

/*******************************************************************************

Procedure:

    tv_create ()


Purpose:

    Function TV_CREATE, given a time specified in seconds and microseconds,
    creates and returns a TIMEVAL representing that time.


    Invocation:

        result = tv_create (seconds, microseconds) ;

    where

        <seconds>	- I
            is the whole number of seconds in the time.
        <microseconds>	- I
            is the remaining microseconds in the time.
        <result>	- O
            returns a UNIX TIMEVAL structure representing the specified time.

*******************************************************************************/


struct  timeval  tv_create (

#    if __STDC__ || defined(vaxc)
        long  seconds,
        long  microseconds)
#    else
        seconds, microseconds)

        long  seconds ;
        long  microseconds ;
#    endif

{    /* Local variables. */
    struct  timeval  result ;



    seconds = (seconds < 0) ? 0 : seconds ;
    microseconds = (microseconds < 0) ? 0 : microseconds ;

/* "Normalize" the time so that the microseconds field is less than a million. */

    while (microseconds >= 1000000) {
        seconds++ ;  microseconds = microseconds - 1000000 ;
    }

/* Return the time in a TIMEVAL structure. */

    result.tv_sec = seconds ;
    result.tv_usec = microseconds ;

    return (result) ;

}

/*******************************************************************************

Procedure:

    tv_create_f ()


Purpose:

    Function TV_CREATE_F, given a floating-point number representing a
    time in seconds and fractional seconds, creates and returns a TIMEVAL
    representing that time.


    Invocation:

        result = tv_create_f (f_seconds) ;

    where

        <seconds>	- I
            is a floating-point number representing a time in seconds and
            fractional seconds.
        <result>	- O
            returns a UNIX TIMEVAL structure representing the specified time.

*******************************************************************************/


struct  timeval  tv_create_f (

#    if __STDC__ || defined(vaxc)
        double  f_seconds)
#    else
        f_seconds)

        double  f_seconds ;
#    endif

{    /* Local variables. */
    struct  timeval  result ;



    if (f_seconds < 0) {				/* Negative time? */
        result.tv_sec = 0 ;  result.tv_usec = 0 ;
    } else if (f_seconds > (double) LONG_MAX) {		/* Time too large? */
        result.tv_sec = LONG_MAX ;  result.tv_usec = 999999 ;
    } else {						/* Valid time. */
        result.tv_sec = f_seconds ;
        result.tv_usec = (f_seconds - (double) result.tv_sec) * 1000000.0 ;
    }

    return (result) ;

}

/*******************************************************************************

Procedure:

    tv_display ()


Purpose:

    Function TV_DISPLAY converts a binary TIMEVAL structure (seconds
    and microseconds) into an ASCII string ("YY-DOY-HR:MN:SC.MLS").


    Invocation:

        ASCII_time = tv_display (binary_time) ;

    where

        <binary_time>	- I
            is the binary TIMEVAL structure that will be converted to ASCII.
        <ASCII_time>	- O
            returns a pointer to an ASCII string, which contains the time
            formatted as "YY-DOY-HR:MN:SC.MLS".  In the year field, years
            '70' through '99' represent 1970 through 1999; years '00'
            through '69' represent 2000 through 2069.  Under VxWorks, the
            ASCII string is dynamically allocated; the caller is responsible
            for FREE(3)ing the string when it is no longer needed.  Under
            other operating systems, the ASCII string is stored in memory
            local to TV_DISPLAY(), so you should use or duplicate the string
            before calling TV_DISPLAY() again.

*******************************************************************************/


char  *tv_display (

#    if __STDC__ || defined(vaxc)
        struct  timeval  binary_time)
#    else
        binary_time)

        struct  timeval  binary_time ;
#    endif

{    /* Local variables. */
    long  clock_time ;
    struct  tm  utc ;
#ifdef VXWORKS
    char  *ASCII_time = malloc (32) ;
#else
    static  char  ASCII_time[32] ;
#endif



    clock_time = binary_time.tv_sec + (binary_time.tv_usec / 1000000) ;
#ifdef VXWORKS
    gmtime_r ((time_t *) &clock_time, &utc) ;
#else
    utc = *(gmtime ((time_t *) &clock_time)) ;
#endif

    sprintf (ASCII_time, "%02d-%03d-%02d:%02d:%02d.%03d",
             (utc.tm_year % 100), utc.tm_yday + 1,
             utc.tm_hour, utc.tm_min, utc.tm_sec,
             (binary_time.tv_usec % 1000000) / 1000) ;

    return (ASCII_time) ;

}

/*******************************************************************************

Procedure:

    tv_float ()


Purpose:

    Function TV_FLOAT converts a TIMEVAL structure (seconds and microseconds)
    into the equivalent, floating-point number of seconds.


    Invocation:

        real_time = tv_float (time) ;

    where

        <time>		- I
            is the TIMEVAL structure that is to be converted to floating point.
        <real_time>	- O
            returns the floating-point representation in seconds of the time.

*******************************************************************************/


double  tv_float (

#    if __STDC__ || defined(vaxc)
        struct  timeval  time)
#    else
        time)

        struct  timeval  time ;
#    endif

{
    return ((double) time.tv_sec + (time.tv_usec / 1000000.0)) ;
}

/*******************************************************************************

Procedure:

    tv_subtract ()


Purpose:

    Function TV_SUBTRACT subtracts one TIMEVAL from another TIMEVAL.


    Invocation:

        result = tv_subtract (time_1, time_2) ;

    where

        <time_1>	- I
            is the minuend (didn't you graduate from elementary school?),
            a time represented by a UNIX TIMEVAL structure.
        <time_2>	- I
            is the subtrahend (I repeat the question), a time represented
            by a UNIX TIMEVAL structure.
        <result>	- O
            returns, in a UNIX TIMEVAL structure, TIME_1 minus TIME_2.  If
            TIME_2 is greater than TIME_1, then a time of zero is returned.

*******************************************************************************/


struct  timeval  tv_subtract (

#    if __STDC__ || defined(vaxc)
        struct  timeval  time_1,
        struct  timeval  time_2)
#    else
        time_1, time_2)

        struct  timeval  time_1 ;
        struct  timeval  time_2 ;
#    endif

{    /* Local variables. */
    struct  timeval  result ;



/* Subtract the second time from the first. */

    if ((time_1.tv_sec < time_2.tv_sec) ||
        ((time_1.tv_sec == time_2.tv_sec) &&
         (time_1.tv_usec <= time_2.tv_usec))) {		/* TIME_1 <= TIME_2? */
        result.tv_sec = result.tv_usec = 0 ;
    } else {						/* TIME_1 > TIME_2 */
        result.tv_sec = time_1.tv_sec - time_2.tv_sec ;
        if (time_1.tv_usec < time_2.tv_usec) {
            result.tv_usec = time_1.tv_usec + 1000000 - time_2.tv_usec ;
            result.tv_sec-- ;				/* Borrow a second. */
        } else {
            result.tv_usec = time_1.tv_usec - time_2.tv_usec ;
        }
    }

    return (result) ;

}

/*******************************************************************************

Procedure:

    tv_tod ()


Purpose:

    Function TV_TOD returns the current time-of-day (GMT).  TV_TOD simply
    calls the UNIX system function, GETTIMEOFDAY(3), but it provides a
    simple means of using the current time in "timeval" calculations without
    having to allocate a special variable for it and making a separate call
    (i.e., you can "inline" a call to TV_TOD within a call to one of the
    other TV_UTIL functions).

      NOTE: Under VxWorks, TV_TOD() returns the current value of the
            real-time clock, not GMT.


    Invocation:

        current_GMT = tv_tod () ;

    where

        <result>	- O
            returns, in a UNIX TIMEVAL structure, the current GMT.  (Under
            VxWorks, the current value of the real-time clock is returned
            instead of GMT.)

*******************************************************************************/


struct  timeval  tv_tod (

#    if __STDC__ || defined(vaxc)
        void)
#    else
        )
#    endif

{    /* Local variables. */
    struct  timeval  current_GMT ;

#ifdef  VXWORKS
    struct  timespec  time_of_day ;

    clock_gettime (CLOCK_REALTIME, &time_of_day) ;
    current_GMT.tv_sec = time_of_day.tv_sec ;
    current_GMT.tv_usec = time_of_day.tv_nsec / 1000 ;
#else
    gettimeofday (&current_GMT, NULL) ;
#endif

    return (current_GMT) ;

}
