#############################################################################
# heirarchy.tcl,v 1.5 1995/01/12 17:46:36 drs1004 Exp
# 
# Copyright (C) 1994  Donald Syme
#
# Copyright 1992-1994 by Jay Sekora.  All rights reserved, except 
# that this file may be freely redistributed in whole or in part 
# for nonprofit, noncommercial use.
# 
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 1, or (at your option)
#    any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#    Contact Details:
#	Donald Syme
#	The Computer Laboratory
#	New Musuems Site
#	Pembroke St.
#	Cambridge U.K. CB2 3QG
#
#	email: Donald.Syme@cl.cam.ac.uk
#
#############################################################################




#----------------------------------------------------------------------------
# IMPORTANT:
#
#   To operate this package you should set the variable "heirarchy_library"
# to point to the directory where it is installed.
#
#
# WIDGET CLASS heirarchy
#
# To create a heirarchy widget:
#	heirarchy <window pathname> <options>
#
# Options	
#	(Any canvas option may be used with a heirarchy)
#
#	-rootnode
#	  	The name of the root node of the tree.  Each node
#		name must be unique amongst the children of each node.
#
#		Defaults to ".." for no good reason.
#
#	-nodechildren
#	    	A command which the widget will execute to retrieve the
#		children of a node.  The node is appended to
#		the command as a list of
#		node-names which form a path to the node from the root.
#		Thus the first element of this list will always be the
#		root node.
#
#		The default is a command to return information suitable for
#		displaying a directory heirarchy.
#
#	-nodelook
#		A command the widget will execute to get the llok of a node.
#		The node is appended to the command as a list of
#		node-names which form a path to the node from the root.
#		Thus the first element of this list will always be the
#		root node.  Also appended is a 
#		boolean value which indicates whether the node's children
#		are currently displayed.  This allows the node's
#		look to change if it is "opened" or "closed".
#
#		This command should return a 5-tuple containing:
#		 1. the text to display at the node
#		 2. the fill color to use for the text
#		 3. the font to use for the text
#                4. a bitmap to display 
#		 5. the foreground color for the bitmap.  
#		If no font (ie. {}) is specified then
#		the value from -font is used.  If no bitmap is specified
#		then no bitmap is displayed.
#			    
#		The default is a command to which produces a nice look
#		for a file manager.  
#
#	-font
#		The default font used for the text.
#
#	-textFill
#		The default fill color used for the text.
#
#	-selectTextFill
#		The default fill color used for the text of selected items.
#
#	-bitmapForeground
#		The default foreground color used for bitmaps.
#
#	-selectBitmapForeground
#		The default foreground color used for bitmaps of selected
#		items.
#
#	-selectBackground
#		The default background color used for both text and
# 		bitmaps of selected objects.
#
#	-itemBackground
#		The default background color used for both text and
# 		bitmaps of nonselected objects.
#
#	-command
#		A command to execute when a node is double-clicked/selected.
#		A list of paths-to-nodes giving all the selected nodes
#               at time of execution is appended as an argument.
#
#	-rootanchor
#		A location for the root of the tree to be anchored.  May
#		be any of n, ne, e, se, s, sw, w, nw.  The anchorings
#		ne,nw,se,sw,e and w all give "vertical stacking" in
#		that the children of each node are layed out in a 
#		vertical column.  The anchorings s and n give "horizontal"
#		stacking, so the children are layed out in a horizontal
#		row.
#
#	-padstack
#		The space added between two columns in the case of
#		anchorings that give vertical stacking, 
#		and between two rows in the case of row stacking.
#
#	-paddepth
#		The space added between two rows in the case of
#		vertical stacking, and between two columns in the case of
#		row stacking.
#
#	-padtextbitmap
#		The space added between the bitmap and the text for a given
#		node.
#
#	-multiselect
#		Whether multiple items in the tree may be selected at once.
#		Defaults to 1.
#
#	-commonselect
#		If this is true, when a node gets added to the selection,
#		all other nodes with the same *name* (regardless of
#		the path to the node) get selected as well.  The selction
#		is reported only as a set of node names, not a set of node 
#		paths.  Thus selection acts like an equivalence over nodes
#		of the same name.  Note this is not useful in
#		a directory heirarchy, but is useful in other heirarchies
#		where some nodes in the heirarchy really refer to the
#		same logical object.
#		Defaults to 0
#
# 	-expand
#		an integer value for an initial depth to expand to.
#
#		Defaults to "1"
#
# PUBLIC FUNCTIONS (operations on the widget)
#	heirarchy::expandnodeall
#	heirarchy::collapsenodeall
# 	heirarchy::expandnoden
# 	heirarchy::setselection
# 	heirarchy::addtoselection
# 	heirarchy::removefromselection
# 	heirarchy::toggleselection
# 	heirarchy::selection
#	heirarchy::bindnodes
#
# See the "documentation" at the head of each procedure for
# information on how to call the procedure and what it does.
# To be honest you will just have to use guess work really.
#
# NOTES
#
# One optimization assumes that the bitmap for a node with
# no children does not change depending whether it is open/closed.
# This seems reasonable.
#
# Heirarchies are not first class widgets.  They do
# not respond to the config command.
#
#----------------------------------------------------------------------------

option add *Heirarchy.font "-Adobe-Helvetica-Medium-R-Normal--*-100-*" widgetDefault
option add *Heirarchy.textFill black widgetDefault
option add *Heirarchy.selectTextFill black widgetDefault
option add *Heirarchy.bitmapForeground black widgetDefault
option add *Heirarchy.selectBitmapForeground black widgetDefault
option add *Heirarchy.selectBackground #b2dfee widgetDefault
option add *Heirarchy.itemBackground #ffe4c4 widgetDefault
option add *Heirarchy.command "" widgetDefault
option add *Heirarchy.nodechildren heirarchy::dircontents widgetDefault
option add *Heirarchy.nodelook heirarchy::filelook widgetDefault
option add *Heirarchy.rootnode "/." widgetDefault
option add *Heirarchy.rootanchor nw widgetDefault
option add *Heirarchy.paddepth 20 widgetDefault
option add *Heirarchy.padstack 6 widgetDefault
option add *Heirarchy.padtextbitmap 3 widgetDefault
option add *Heirarchy.expand 1 widgetDefault
option add *Heirarchy.debug 0 widgetDefault
option add *Heirarchy.multiselect 1 widgetDefault
option add *Heirarchy.commonselect 0 widgetDefault

proc heirarchy { w args} {
	global TkHolShell_flags
	global vals
	set passon_args ""
	for {set i 0} {$i<[llength $args]} {incr i} {
	    case [lindex $args $i]  -font {
	        incr i
	        set regroundfont [lindex $args $i]
	    } -selectTextFill {
	        incr i
	        set selectTextFill [lindex $args $i]
	    } -bitmapForeground    {
                incr i
	        set bitmapForeground [lindex $args $i]
	    } -selectBitmapForeground {
	        incr i
	        set selectBitmapForeground [lindex $args $i]
	    } -itemBackground {
	        incr i
	        set itemBackground [lindex $args $i]
	    } -selectBackground {
	        incr i
	        set selectBackground [lindex $args $i]
	    } -padstack {
	        incr i
	        set padstack [lindex $args $i]
	    } -textFill {
	        incr i
	        set textFill [lindex $args $i]
	    } -command {
	        incr i
	        set command [lindex $args $i]
	    } -rootnode {
	        incr i
	        set rootnode [lindex $args $i]
	    } -rootanchor {
	        incr i
	        set rootanchor [lindex $args $i]
	    } -nodechildren {
	        incr i
	        set nodechildren [lindex $args $i]
	    } -nodelook {
	        incr i
	        set nodelook [lindex $args $i]
	    } -expand {
	        incr i
	        set expand [lindex $args $i]
	    } -debug {
	        incr i
	        set debug [lindex $args $i]
	    } -paddepth {
	        incr i
	        set paddepth [lindex $args $i]
	    } -padtextbitmap {
	        incr i
	        set padtextbitmap [lindex $args $i]
	    } -multiselect {
	        incr i
	        set multiselect [lindex $args $i]
	    } -commonselect {
	        incr i
	        set commonselect [lindex $args $i]
	    } default { 
	        set arg1 [lindex $args $i]
	        incr i
	        set arg2 [lindex $args $i]
	    	lappend passon_args $arg1 $arg2
	    }
	}
	frame $w -class Heirarchy
	eval [list canvas $w.canv] $passon_args
	pack $w.canv -expand yes -fill both

	if ![info exists font] { set font [option get $w font Font] }
	if ![info exists command] { set command [option get $w command Command] }
	if ![info exists rootnode] { set rootnode [option get $w rootnode RootNode] }
	if ![info exists rootanchor] { set rootanchor [option get $w rootanchor RootAnchor] }
	if ![info exists nodechildren] { set nodechildren [option get $w nodechildren NodeRetriever] }
	if ![info exists nodelook] { set nodelook [option get $w nodelook NodeLook] }
	if ![info exists expand] { set expand [option get $w expand Expand] }
	if ![info exists paddepth] { set paddepth [option get $w paddepth PadDepth] }
	if ![info exists textFill] { set textFill [option get $w textFill TextFill] }
	if ![info exists selectTextFill] { set selectTextFill [option get $w selectTextFill SelectTextFill] }
	if ![info exists bitmapForeground] { set bitmapForeground [option get $w bitmapForeground BitmpForeground] }
	if ![info exists selectBitmapForeground] { set selectBitmapForeground [option get $w selectBitmapForeground SelectBitmapForeground] }
	if ![info exists itemBackground] { set itemBackground [option get $w itemBackground SelectBackground] }
	if ![info exists selectBackground] { set selectBackground [option get $w selectBackground SelectBackground] }
	if ![info exists padstack] { set padstack [option get $w padstack PadStack] }
	if ![info exists padtextbitmap] { set padtextbitmap [option get $w padtextbitmap PadTextBitmap] }
	if ![info exists debug] { set debug [option get $w debug Debug] }
	if ![info exists multiselect] { set multiselect [option get $w multiselect MultiSelect] }
	if ![info exists commonselect] { set commonselect [option get $w commonselect CommonSelect] }

	# transfer values into the storage array "vals"
	set vals($w,font) $font
	set vals($w,command) $command
	set vals($w,rootnode) $rootnode
	set vals($w,rootanchor) $rootanchor
	set vals($w,nodechildren) $nodechildren
	set vals($w,nodelook) $nodelook
	set vals($w,textFill) $textFill
	set vals($w,selectTextFill) $selectTextFill
	set vals($w,bitmapForeground) $bitmapForeground
	set vals($w,selectBitmapForeground) $selectBitmapForeground
	set vals($w,itemBackground) $itemBackground
	set vals($w,selectBackground) $selectBackground
	set vals($w,padstack) $padstack
	set vals($w,debug) $debug
	set vals($w,multiselect) $multiselect
	set vals($w,commonselect) $commonselect
	set vals($w,paddepth) $paddepth
	set vals($w,padtextbitmap) $padtextbitmap
	
	if $vals($w,debug) { puts "expand = $expand" }

	set vals($w,$vals($w,rootnode),showkids) 0
	set vals($w,selection) ""
	set vals($w,bindings) ""
	case $expand all {
            heirarchy::expandnodeall $w $vals($w,rootnode)
	} default {
            heirarchy::expandnoden $w $vals($w,rootnode) $expand
	}

	if {$command==""} {
	    heirarchy::bindnodes $w <Button-1> "heirarchy::setselection $w %n"
	} else {
	    heirarchy::bindnodes $w <Button-1> "heirarchy::setselection $w %n ; $command \[heirarchy::selection $w\]"
	}
        heirarchy::bindnodes $w <Double-Button-1> "heirarchy::togglenode $w %n"
        heirarchy::bindnodes $w <Shift-Button-1> "heirarchy::toggleselection $w %n"
        heirarchy::bindnodes $w <Control-Button-1> "heirarchy::toggleselection $w %n"

	bind $w.canv <Configure> "heirarchy::configure_notify $w %w %h %x Configure"
	bind $w.canv <ConfigureRequest> "heirarchy::configure_notify $w %w %h %x ConfigureRequest"
	bind $w.canv <ResizeRequest> "heirarchy::configure_notify $w %w %h %x ResizeRequest"
	bind $w <Destroy> "heirarchy::destroy_notify $w"
	return $w
}

#------------------------------------------------------------
# PUBLIC heirarchy::expandnodeall
#
# Expand all nodes from the given node then redraw
#
#------------------------------------------------------------

proc heirarchy::expandnodeall { w node_path } {
	global vals
	heirarchy::expandnoden $w $node_path 99999
}
				  
				       
#------------------------------------------------------------
# PUBLIC heirarchy::collapsenodeall
#
# Collapse the given node and all children then redraw.
#
#------------------------------------------------------------

proc heirarchy::collapsenodeall { w node_path } {
	if [heirarchy::collapsenode $w $node_path] {
	    heirarchy::redraw_after_node_change $w $node_path
	    heirarchy::discard_children $w $node_path
	}
}
				  
#------------------------------------------------------------
# PUBLIC heirarchy::expandnoden
#
# Expand the given node to the given depth
#------------------------------------------------------------

proc heirarchy::expandnoden { w node_path n } {
	global vals
	if [heirarchy::expandnoden_aux $w $node_path $n] {
	    heirarchy::redraw_after_node_change $w $node_path
        }
}

				  
proc heirarchy::expandnoden_aux { w node_path n } {
	global vals
	set returnval [heirarchy::expandnode $w $node_path]
        if {$returnval==0} { return 0 }
        if {$n==1} { return 1 }
	incr n -1
	foreach kid $vals($w,$node_path,kids) {
	    set kid_node_path $node_path		   
            lappend kid_node_path $kid
	    heirarchy::expandnoden_aux $w $kid_node_path $n
	}
	return 1
}

#------------------------------------------------------------
# heirarchy::togglenode
#
# Toggle the given node then redraw.
#
#------------------------------------------------------------

proc heirarchy::togglenode { w node_path } {
	global vals
	if $vals($w,$node_path,showkids) {
	    heirarchy::collapsenodeall $w $node_path
	} else {
	    heirarchy::expandnode1 $w $node_path
	}
}


#------------------------------------------------------------
# PUBLIC heirarchy::setselection
#
#
#------------------------------------------------------------

proc heirarchy::setselection { w node_path } {
	global vals
	foreach varname $vals($w,selection) {
	    set vals($w,$varname,select) 0
	}
	set vals($w,selection) ""
	    if $vals($w,commonselect) {
	        set varname [lindex $node_path [expr [llength $node_path]-1]]
	    } else {
	        set varname $node_path
	    }
 	    if {!$vals($w,$varname,select)} {
	        set vals($w,$varname,select) 1
	        lappend vals($w,selection) $varname
	    }
}

#------------------------------------------------------------
# PUBLIC heirarchy::addtoselection
#
#
#------------------------------------------------------------

proc heirarchy::addtoselection { w node_path } {
	global vals
	if $vals($w,debug) { puts "before addtoselection node_path = $node_path, vals($w,selection) = vals($w,selection)" }
	    if $vals($w,commonselect) {
	        set varname [lindex $node_path [expr [llength $node_path]-1]]
	    } else {
	        set varname $node_path
	    }
 	    if {!$vals($w,$varname,select)} {
	        set vals($w,$varname,select) 1
	        lappend vals($w,selection) $varname
	    }
	if $vals($w,debug) { puts "after addtoselection, vals($w,selection) = vals($w,selection)" }
}

#------------------------------------------------------------
# PUBLIC heirarchy::removefromselection
#
#
#------------------------------------------------------------

proc heirarchy::removefromselection { w node_path } {
	global vals
	if $vals($w,debug) { puts "before removefromselection node_path = $node_path, vals($w,selection) = vals($w,selection)" }
	    if $vals($w,commonselect) {
	        set varname [lindex $node_path [expr [llength $node_path]-1]]
	    } else {
	        set varname $node_path
	    }
 	    if {$vals($w,$varname,select)} {
	        set vals($w,$varname,select) 0
	        set index [lsearch $vals($w,selection) $varname]
		set vals($w,selection) [lreplace $vals($w,selection) $index $index]
	    }
	if $vals($w,debug) { puts "after removefromselection, vals($w,selection) = vals($w,selection)" }
}


#------------------------------------------------------------
# PUBLIC heirarchy::toggleselection
#
#
#------------------------------------------------------------

proc heirarchy::toggleselection { w node_path } {
	global vals
	if $vals($w,commonselect) {
	    set varname [lindex $node_path [expr [llength $node_path]-1]]
	} else {
	    set varname $node_path
	}
 	if {$vals($w,$varname,select)} {
	    heirarchy::removefromselection $w $node_path
        } else {
	    heirarchy::addtoselection $w $node_path
        }
}


#------------------------------------------------------------
# PUBLIC heirarchy::selection
#
# Retrive the currently selected set of node paths.
#------------------------------------------------------------

proc heirarchy::selection { w } {
	global vals
	return $vals($w,selection)
}

#------------------------------------------------------------
# PUBLIC heirarchy::bindnode
#
# Add an event binding toall nodes in the heirarchy.  
# In addition to normal substitutions, the following are available:
#	%n	- the path to the node at which the action takes place
#------------------------------------------------------------

proc heirarchy::bindnodes { w event command } {
	global vals
	heirarchy::bindnode_aux $w $vals($w,rootnode) $event $command
	lappend vals($w,bindings) [list $event $command]
}

proc heirarchy::bindnode_aux { w node_path event command } {
	global vals

	# this is dodgy - see regsub subspec info.

	regsub %n $command [list $node_path] subst_command
	if $vals($w,debug) { puts "command = $command, node_path = $node_path, subst_command = $subst_command" }
#	regsub %W $subst_command $w subst_command
	catch {$w.canv bind text:$node_path $event $subst_command}
	catch {$w.canv bind bitmap:$node_path $event $subst_command}
	if $vals($w,$node_path,showkids) {
	    foreach kid $vals($w,$node_path,kids) {
	        set kid_node_path $node_path		   
                lappend kid_node_path $kid
	        heirarchy::bindnode_aux $w $kid_node_path $event $command
	    }
        }
}



proc heirarchy::bindelem { w node_path elem } {
	global vals
	foreach binding $vals($w,bindings) {
	    regsub %n [lindex $binding 1] [list $node_path] subst_command
	    $w.canv bind [set elem]:$node_path [lindex $binding 0] $subst_command
	}
}


#------------------------------------------------------------
# UTILITY ROUTINES
#
# The remainder of these functions are utility functions
# only.
#------------------------------------------------------------

#------------------------------------------------------------
# heirarchy::destroy_notify
#
# Called just when the "heirarchy" widget is destroyed.
#------------------------------------------------------------

proc heirarchy::destroy_notify { w } {
 	global vals
	heirarchy::collapsenode $w $vals($w,rootnode)
	heirarchy::discard_children $w $vals($w,rootnode)
	heirarchy::discardnode $w $vals($w,rootnode)
	catch {unset vals($w,heightknown)}
	catch {unset vals($w,height)}
	catch {unset vals($w,width)}
	catch {unset vals($w,selection)}
}

#------------------------------------------------------------
#
#------------------------------------------------------------
proc heirarchy::setup_select_variable { w node_path } {
	global vals
	if $vals($w,commonselect) {
	    set varname [lindex $node_path [expr [llength $node_path]-1]]
	} else {
	    set varname $node_path
	}
        if ![info exists vals($w,$varname,select)] {
	    set vals($w,$varname,select) 0
	}
        if ![info exists vals($w,$node_path,traced)] {
	    set vals($w,$node_path,traced) 1
	    trace variable vals($w,$varname,select) w "heirarchy::select_change $w [list $node_path]"
	}
}

proc heirarchy::select_change { w node_path arg1 arg2 op } {
	global vals
	heirarchy::adjust_look $w $node_path
	heirarchy::remake_selection_box $w $node_path
}


#------------------------------------------------------------
# Utility routines to collapse and expand a single node
# without redrawing.
#
# Nb. doesn't reposition
#
# Routines often return 0/1 to indicate if any change
# has occurred in the tree.
#------------------------------------------------------------


proc heirarchy::expandnode { w node_path } {
	global vals
	if $vals($w,$node_path,showkids) { return 0 }
	set vals($w,$node_path,showkids) 1
	set vals($w,$node_path,kids) [eval $vals($w,nodechildren) [list $node_path]]
	if {[llength $vals($w,$node_path,kids)]==0} { return 0 }
	set vals($w,$node_path,look) [eval $vals($w,nodelook) [list $node_path] 1]
	heirarchy::setup_select_variable $w $node_path
	
	foreach kid $vals($w,$node_path,kids) {
	    set kid_node_path $node_path		   
            lappend kid_node_path $kid
	    set vals($w,$kid_node_path,look) [eval $vals($w,nodelook) [list $kid_node_path] 0]
	    heirarchy::setup_select_variable $w $kid_node_path
	    set vals($w,$kid_node_path,showkids) 0
	}
	return 1
}

proc heirarchy::collapsenode { w node_path } {
	global vals
	if {!$vals($w,$node_path,showkids)} { return 0 }
	set vals($w,$node_path,showkids) 0
	if {[llength $vals($w,$node_path,kids)]==0} { return 0}
	set vals($w,$node_path,look) [eval $vals($w,nodelook) [list $node_path] 0]
	foreach kid $vals($w,$node_path,kids) {
	    set kid_node_path $node_path		   
            lappend kid_node_path $kid
	    heirarchy::collapsenode $w $kid_node_path
	}
	return 1
}


proc heirarchy::discardnode { w node_path } {
	global vals
	if $vals($w,commonselect) {
	    set varname [lindex $node_path [expr [llength $node_path]-1]]
	} else {
	    set varname $node_path
	}
	trace vdelete vals($w,$varname,select) w "heirarchy::select_change $w [list $node_path]"
	if [llength [trace vinfo vals($w,$varname,select)]]==0 {
	    unset vals($w,$varname,select)
	}
        catch {$w.canv delete bitmap:$node_path}
        catch {$w.canv delete text:$node_path}
        catch {$w.canv delete rect:$node_path}
        catch {unset vals($w,$node_path,showkids)}
        catch {unset vals($w,$node_path,look)}
        catch {unset vals($w,$node_path,usages)}
        catch {unset vals($w,$node_path,my_depth_usage)}
        catch {unset vals($w,$node_path,traced)}
}


proc heirarchy::discard_children { w node_path } {
	global vals
	if [info exists vals($w,$node_path,kids)] {
	    foreach kid $vals($w,$node_path,kids) {
	        set kid_node_path $node_path		   
                lappend kid_node_path $kid
		heirarchy::discardnode $w $kid_node_path
	        heirarchy::discard_children $w $kid_node_path
	    }
	    unset vals($w,$node_path,kids) 
	}
}

				       
#------------------------------------------------------------
#
#
#------------------------------------------------------------

proc heirarchy::reevaluatenode { w node_path } {
	global vals
	heirarchy::expandnode $w $node_path
        if {$n==1} return
	if $vals($w,$node_path,showkids) {
	    foreach kid $vals($w,$node_path,kids) {
	        set kid_node_path $node_path		   
		lappend kid_node_path $kid
	        heirarchy::reevaluatenode $w $kid_node_path
	    }
	}
}
				       


#------------------------------------------------------------
#
#
#------------------------------------------------------------

proc heirarchy::expandnode1 { w node_path } {
	global vals
        heirarchy::expandnoden $w $node_path 1
}



				  
				  
#------------------------------------------------------------
# Reevaluate all expanded theories and redraw everything.
# Useful if something in the tree has changed.
#
#------------------------------------------------------------

proc heirarchy::complete_redraw { w } {
	global vals
	heirarchy::reevaluateenode $w $vals($w,rootnode)
	heirarchy::redraw_after_node_change $w $node_path
}


proc heirarchy::remake_selection_box { w node_path } {
	global vals
        if ![info exists vals($w,heightknown)] return
        catch {$w.canv delete rect:$node_path}
	if $vals($w,commonselect) {
	    set varname [lindex $node_path [expr [llength $node_path]-1]]
	} else {
	    set varname $node_path
	}
	if $vals($w,$varname,select) {
            set textBackground $vals($w,selectBackground)
	    # remake the selection box for the text if it has one
	    set text_dimensions [$w.canv bbox text:$node_path] 
	    eval [list $w.canv create rectangle] $text_dimensions [list -fill $textBackground -width 1 -tags [list rect:$node_path]]
	    $w.canv lower rect:$node_path text:$node_path
	}
}

proc heirarchy::adjust_look { w node_path } {
	global vals
	if $vals($w,commonselect) {
	    set varname [lindex $node_path [expr [llength $node_path]-1]]
	} else {
	    set varname $node_path
	}
	set text [lindex $vals($w,$node_path,look) 0]
	set bitmap [lindex $vals($w,$node_path,look) 3]
	if {$bitmap!=""} {
	    if $vals($w,$varname,select) {
        	set bitmapForeground $vals($w,selectBitmapForeground)
		set bitmapBackground $vals($w,selectBackground)
	    } else {
        	set bitmapForeground [lindex $vals($w,$node_path,look) 4]
		set bitmapBackground $vals($w,itemBackground)
		if {$bitmapForeground==""} { set bitmapForeground $vals($w,bitmapForeground) }
	    }
 	    $w.canv itemconfigure bitmap:$node_path -bitmap $bitmap -foreground $bitmapForeground  -background $bitmapBackground
	}
	if {$text!=""} {
	    set textFont [lindex $vals($w,$node_path,look) 2]
	    if $vals($w,$varname,select) {
        	set textFill $vals($w,selectTextFill)
	    } else {
	        set textFill [lindex $vals($w,$node_path,look) 1]
	        if {$textFill==""} { set textFill $vals($w,textFill) }
            }
	    if {$textFont==""} { set textFont $vals($w,font) }
	    $w.canv itemconfigure text:$node_path -text $text -fill $textFill -font $textFont 
	}		

}


#------------------------------------------------------------
# Redrawing apparatus
#
# recompute_positions recurses through the tree wokring
# out the relative offsets of children from their parents
# in terms of depth/stack(width) values.  
#
# "changed_node" is either empty or a node name which indicates
# where the only changes have occured in the heirarchy since the last
# call to rcompute_positions.  This is used because when a node is toggled
# on/off deep in the heirarchy then not all the positions of items
# need to be recomputed.  The only ones that do are everything below
# the changed node (of ocurse), and also everything which might depend on
# the stack usage of that node (i.e. everything above it).  Specifically
# the usages of the changed node's siblings do *not* need to be recomputed.
#------------------------------------------------------------

proc heirarchy::min {args} {
    set min 99999
    foreach a $args { if $a<$min {set min $a} } 
    return $min
}
proc heirarchy::max {args} {
    set max -99999
    foreach a $args { if $a>$max {set max $a} } 
    return $max
}


proc heirarchy::recompute_positions { w changed_node_path } {
	global vals
        return [heirarchy::recompute_positions_aux $w $vals($w,rootnode) $changed_node_path]
}

proc heirarchy::recompute_positions_aux { w node_path changed_node_path } {
	global vals
						     
	# If the changed_node_path now has only one element then
	# it must be one of the children of the current node.
	# We do not need to recompute the
	# usages of its siblings if it is.
	
	set changed_node_is_child [expr [llength $changed_node_path]==1]
	if $changed_node_is_child {
	    set changed_node [lindex $changed_node_path 0]
	} else {
	    set remaining_changed_node_path [lrange $changed_node_path 1 end]
	}
	if $vals($w,debug) { puts "$node_path: changed_node_path = $changed_node_path, changed_node_is_child = $changed_node_is_child" } 
	
	
	# Run through the children, recursively calculating their usage
	# of stack-depth real-estate, and allocating an intial placement
	# for each child in vals$w,$kid,
	#
	# Values do not need to be recompted for siblings of the changed
	# node and their descendants.  For the changed_node itself, in the
	# recursive call we set the value of changed_node to {} to prevent
	# any further changed_node checks.
	
	set children_stack_usage 0
	set children_depth_usage 0
	if $vals($w,$node_path,showkids) { 
	    foreach kid $vals($w,$node_path,kids) {
	        set kid_node_path $node_path		   
                lappend kid_node_path $kid
	        set vals($w,$kid_node_path,offset) $children_stack_usage
                if {$changed_node_is_child && $changed_node==$kid} {
		    set vals($w,$kid_node_path,usages) [heirarchy::recompute_positions_aux $w $kid_node_path {}]
		} else {
		    if {!$changed_node_is_child} {
		        set vals($w,$kid_node_path,usages) [heirarchy::recompute_positions_aux $w $kid_node_path $remaining_changed_node_path]
		    }
		}
		set child_stack_usage [lindex $vals($w,$kid_node_path,usages) 0]
	        if [regexp ^(n|s|e|w)$ $vals($w,rootanchor)] {
		    # these anchors are stacked centrally, and so
		    # we adjust each child back by half its stack
		    # usage to account for the centering.
	            incr vals($w,$kid_node_path,offset) [expr $child_stack_usage/2]
	        }
	        incr children_stack_usage $child_stack_usage
	        set children_depth_usage [heirarchy::max $children_depth_usage [lindex $vals($w,$kid_node_path,usages) 1]]
	        incr children_stack_usage $vals($w,padstack)
	    }
	}
	incr children_stack_usage -$vals($w,padstack)
		       
	# Make the items (if they do no already exist)
	# and place them any old place.  Adjust their look also.
	# The items get repositioned later.  
	
	set text [lindex $vals($w,$node_path,look) 0]
	set bitmap [lindex $vals($w,$node_path,look) 3]
	if {$bitmap!=""} {
	    if [llength [$w.canv find withtag bitmap:$node_path]]==0 {
	        $w.canv create bitmap 0 0 -anchor $vals($w,rootanchor) -tags [list bitmap:$node_path]
	        heirarchy::bindelem $w $node_path bitmap
	    }
	}
	if {$text!=""} {
	    if [llength [$w.canv find withtag text:$node_path]]==0 {
	        $w.canv create text 0 0 -anchor $vals($w,rootanchor) -tags [list text:$node_path]
	        heirarchy::bindelem $w $node_path text
	    }
	}		
	heirarchy::adjust_look $w $node_path


	# Now calculate the stack usage of our little piece
	# of the world.
	# We have to create the bitmap and text itams to get an idea
	# of their size
	
	set bitmap_height 0
	set bitmap_width 0
	set text_width 0				     
	set text_height 0				     
	
	if {$bitmap!=""} {
	    set bitmap_dimensions [$w.canv bbox bitmap:$node_path]
	    if $vals($w,debug) { puts "$node_path: bitmap_dimensions = $bitmap_dimensions" }
	    set bitmap_height [expr [lindex $bitmap_dimensions 3]-[lindex $bitmap_dimensions 1]]
	    set bitmap_width [expr [lindex $bitmap_dimensions 2]-[lindex $bitmap_dimensions 0]]
	    if $vals($w,debug) { puts "$node_path: bitmap_height     = $bitmap_height" }
	    if $vals($w,debug) { puts "$node_path: bitmap_width      = $bitmap_width" }
	}
	if {$text!=""} {
	    set text_dimensions [$w.canv bbox text:$node_path]
	    if $vals($w,debug) { puts "$node_path: text_dimensions = $text_dimensions" }
	    set text_height [expr [lindex $text_dimensions 3]-[lindex $text_dimensions 1]]
	    set text_width [expr [lindex $text_dimensions 2]-[lindex $text_dimensions 0]]
	    if $vals($w,debug) { puts "$node_path: text_height     = $text_height" }
	    if $vals($w,debug) { puts "$node_path: text_width      = $text_width" }
	}
	
	if [regexp ^(ne|se|e|nw|sw|w)$ $vals($w,rootanchor)] {
	    # these anchors are stacked vertically
	    set my_stack_usage [heirarchy::max $text_height $bitmap_height]
	    set my_depth_usage [expr {$text_width+$bitmap_width+$vals($w,padtextbitmap)}]
	} else {
	    # these anchors are stacked horizontally
	    set my_stack_usage [heirarchy::max $text_width $bitmap_width]
	    set my_depth_usage [expr {$text_height+$bitmap_height+$vals($w,padtextbitmap)}]
	}
	
	if $vals($w,debug) { puts "$node_path: my_stack_usage = $my_stack_usage, my_depth_usage = $my_depth_usage" }
	
	# Now reposition the children in the case of the centre
	# positioned items by half of $usage.  In the case
	# of the others position them downward by "my_stack_usage"
			
	if [regexp ^(n|s|e|w)$ $vals($w,rootanchor)] {
	    # these anchors are stacked centrally
	    set overall_stack_usage [heirarchy::max $children_stack_usage $my_stack_usage]
	    set overall_depth_usage [expr $children_depth_usage+$vals($w,paddepth)+$my_depth_usage]
	} else {
	    # these anchors are stacked on one side only
	    # the depth of the item itself does not effect the overall depth
	    # unless it is greater than all it children (e.g. if it has no children)
	    set overall_stack_usage [expr $children_stack_usage+$my_stack_usage+$vals($w,padstack)]
	    set overall_depth_usage [heirarchy::max [expr $children_depth_usage+$vals($w,paddepth)] $my_depth_usage]
	}
	if $vals($w,debug) { puts "$node_path: overall_stack_usage = $overall_stack_usage, overall_depth_usage = $overall_depth_usage" }
	if $vals($w,$node_path,showkids) { 
	    foreach kid $vals($w,$node_path,kids) {
	        set kid_node_path $node_path		   
                lappend kid_node_path $kid
	        if [regexp ^(n|s|e|w)$ $vals($w,rootanchor)] {
		    # these anchors are stacked centrally
	            incr vals($w,$kid_node_path,offset) [expr -$children_stack_usage/2]
		    # we need this to stop lone-children looking silly
		    if {abs($vals($w,$kid_node_path,offset)) < 2} {
		        set vals($w,$kid_node_path,offset) 0
		    }
	        } else {
		    # these anchors are stacked on one side only
		    
	            incr vals($w,$kid_node_path,offset) [expr $my_stack_usage+$vals($w,padstack)]
	        }       
		if $vals($w,debug) { puts "$node_path: vals($w,$kid_node_path,offset) = $vals($w,$kid_node_path,offset)" }
	    }
	}
	# remember some facts for locating the bitmap
	# and also for drawing decorations
	set vals($w,$node_path,my_stack_usage) $my_stack_usage 
	set vals($w,$node_path,my_depth_usage) $my_depth_usage 
	set vals($w,$node_path,bitmap_width) $bitmap_width
	set vals($w,$node_path,bitmap_height) $bitmap_height
	
	return [list $overall_stack_usage $overall_depth_usage]
}

proc heirarchy::configure_notify { w width height x y} {
	global vals	    
	if $vals($w,debug) { puts "heirarchy::configure_notify, w = $w, width = $width, height = $height" }
	set vals($w,width) $width
	set vals($w,height) $height
        if ![info exists vals($w,heightknown)] {
	    heirarchy::redraw_after_node_change $w {}
	    set vals($w,heightknown) 0
	}
	heirarchy::choose_initial_viewport $w
}							

proc heirarchy::choose_initial_viewport { w } {
	global vals
	case $vals($w,rootanchor) {
	    nw {
		set xview 0
		set yview 0
	    }
	    n {
		set xview [expr ($vals($w,stack_usage)/2-$vals($w,width)/2)/10]
		set yview 0
	    }
	    ne {
		set xview [expr ($vals($w,depth_usage)-$vals($w,width))/10]
		set yview 0
	    }
	    e {
		set xview [expr ($vals($w,depth_usage)-$vals($w,width))/10]
		set yview [expr ($vals($w,stack_usage)/2-$vals($w,height)/2)/10]
	    }
	    se {
		set xview [expr ($vals($w,depth_usage)-$vals($w,width))/10]
		set yview [expr ($vals($w,stack_usage)-$vals($w,height))/10]
	    }
	    s {
		set xview [expr ($vals($w,stack_usage)/2-$vals($w,width)/2)/10]
		set yview [expr ($vals($w,depth_usage)-$vals($w,height))/10]
	    }
	    sw {
		set xview 0
		set yview [expr ($vals($w,stack_usage)-$vals($w,height))/10]
	    }
	    w {
		set xview 0
		set yview [expr ($vals($w,stack_usage)/2-$vals($w,height)/2)/10]
	    }
	}
#	if {$xview < 0} { set xview 0 }
#	if {$yview < 0} { set yview 0 }
	$w.canv xview $xview
	$w.canv yview $yview

}


proc heirarchy::redraw_after_node_change { w changed_node_path } {
	global vals
	
	set remaining_changed_node_path [lrange $changed_node_path 1 end]
	
	# When a node changes, the positions of a whole lot of things
	# change.  The size of the scroll region also changes.
	# The heuristic we use to reposition the viewport is to
	# try to keep the node that was adjusted in the same
	# position on the screen.

	$w.canv delete decorations
		
	if ![info exists vals($w,height)] return
	
	# Calculate the screen location of the "still node".  This is
	# used later to relocate the still node back to the same
	# screen location
	
	if {$changed_node_path==""} { 
	    set still_node_path $vals($w,rootnode) 
	} else { 
	    set still_node_path $changed_node_path
	 }
  	set still_spot [$w.canv coords text:$still_node_path]
  	set still_spot_x [expr [lindex $still_spot 0]-[$w.canv canvasx 0]]
  	set still_spot_y [expr [lindex $still_spot 1]-[$w.canv canvasy 0]]
			   
	# Now calculate the new offset locations of everything
	
	set usages [heirarchy::recompute_positions $w $remaining_changed_node_path]
	set vals($w,stack_usage) [lindex $usages 0]
	set vals($w,depth_usage) [lindex $usages 1]
	
	# Now set the new scroll region
	
	if [regexp ^(ne|se|e|nw|sw|w)$ $vals($w,rootanchor)] {
	    # these anchors are stacked vertically
	    $w.canv config -scrollregion [list 0 0 $vals($w,depth_usage) $vals($w,stack_usage)]
	} else {
	    $w.canv config -scrollregion [list 0 0 $vals($w,stack_usage) $vals($w,depth_usage)]
	}
	
	# Next recursively move all the bits around to 
	# their correct positions.
	# We choose a point (start_depthpos,start_stackpos) to begin at.
	
	case $vals($w,rootanchor) {
	    nw {
	        set start_depthpos 0
		set start_stackpos 0
	    }
	    n {
	        set start_depthpos 0
		set start_stackpos [expr $vals($w,stack_usage)/2]
	    }
	    ne {
		set start_depthpos [expr $vals($w,depth_usage)]
	        set start_stackpos 0
	    }
	    e {
		set start_depthpos $vals($w,depth_usage)
	        set start_stackpos [expr $vals($w,stack_usage)/2]
	    }
	    se {
		set start_depthpos $vals($w,depth_usage)
	        set start_stackpos $vals($w,stack_usage)
	    }
	    s {
	        set start_depthpos $vals($w,depth_usage)
		set start_stackpos [expr $vals($w,stack_usage)/2]
	    }
	    sw {
	        set start_depthpos 0
		set start_stackpos $vals($w,stack_usage)
		# in this case we do not need to fiddle with the xview/yview
	    }
	    w {
	        set start_depthpos 0
		set start_stackpos [expr $vals($w,stack_usage)/2]
		# in this case we do not need to fiddle with the xview/yview
	    }
	}
        heirarchy::redraw_aux $w $vals($w,rootnode) $start_depthpos $start_stackpos
	
	# Calculate where the still node is now on the canvas and adjust
	# the viewport so it is located on the screen at the
	# same screen location.
	
  	set new_still_spot [$w.canv coords text:$still_node_path]
	# Nb. 10 is the scroll increment.  it should be calculated.
	
	$w.canv xview [expr ([lindex $new_still_spot 0]-$still_spot_x)/10]
	$w.canv yview [expr ([lindex $new_still_spot 1]-$still_spot_y)/10]
	
}

proc heirarchy::redraw_aux { w node_path depthpos stackpos } {
	global vals
	
	# now draw the bar line on which each child line sits,
	# and each child line as well.
	
	set iscentered [regexp ^(n|s|e|w)$ $vals($w,rootanchor)]
	set isvertical [regexp ^(ne|se|e|nw|sw|w)$ $vals($w,rootanchor)]
	case $vals($w,rootanchor) {
	    nw { set depthop +; set stackop + }
	    n { set depthop +; set stackop + }
	    ne { set depthop -; set stackop + }
	    e { set depthop -; set stackop + }
	    se { set depthop -; set stackop - }
	    s { set depthop -; set stackop + }
	    sw { set depthop +; set stackop - }
	    w { set depthop +; set stackop + }
	}
	# position the text and bitmap
	
	if $isvertical {
	    set bitmapx $depthpos
	    set bitmapy $stackpos
	    set textx [expr "$depthpos [set depthop] $vals($w,$node_path,bitmap_width) [set depthop] $vals($w,padtextbitmap)"]
	    set texty $stackpos
	} else {
	    set bitmapx $stackpos
	    set bitmapy [expr "$depthpos [set depthop] $vals($w,$node_path,bitmap_height) [set depthop] $vals($w,padtextbitmap)"]
	    set textx $stackpos
	    set texty $depthpos
	}
	if {[lindex $vals($w,$node_path,look) 2]!=""} {
	    $w.canv coords bitmap:$node_path $bitmapx $bitmapy
        }
	if {[lindex $vals($w,$node_path,look) 0]!=""} {
	    $w.canv coords text:$node_path $textx $texty
	    heirarchy::remake_selection_box $w $node_path
        }
	if !$vals($w,$node_path,showkids) return
	if {[llength $vals($w,$node_path,kids)]==0} return

			       
        # Note that below x/y's get swapped around for non-vertical stacking
		
	    set minkid_stackpos 99999
	    set maxkid_stackpos -99999
	    if $iscentered { 	     
		set top_depthpos [expr "$depthpos [set depthop] $vals($w,$node_path,my_depth_usage)"]
		set bar_depthpos [expr "$depthpos [set depthop] $vals($w,$node_path,my_depth_usage) [set depthop] $vals($w,paddepth)/2"]
	        set kid_depthpos [expr "$bar_depthpos [set depthop] $vals($w,paddepth)/2"]
		if $isvertical {
	            $w.canv create line $top_depthpos $stackpos $bar_depthpos $stackpos -width 1 -tags decorations
		} else {
	            $w.canv create line $stackpos $top_depthpos $stackpos $bar_depthpos -width 1 -tags decorations
		}
 	    } else {
		set bar_depthpos [expr "$depthpos [set depthop] $vals($w,paddepth)/2"]
	        set kid_depthpos [expr "$bar_depthpos [set depthop] $vals($w,paddepth)/2"]
	    }
	    foreach kid $vals($w,$node_path,kids) {
	        set kid_node_path $node_path		   
                lappend kid_node_path $kid
	    	set kid_stackpos [expr "$stackpos [set stackop] $vals($w,$kid_node_path,offset)"]
		heirarchy::redraw_aux $w $kid_node_path $kid_depthpos $kid_stackpos

		# adjust the bar to the kid to be in the center of the kid
		# this is a result of us using the root anchoring to anchor
		# the texts and bitmaps as well, which works well except
		# for here.
		#
		# Note this doesn't effect where the child is drawn - it only
		# effects the drawing of decorations.
		#
		# Checking for the maximum/minimum extents is used
		# to know how to draw the bar that connects all the little
		# bars together.  Doing the "is-it-greater/is-it-less"
		# checks both before and after the adjustment to kid_stackpos
		# is an effective way of getting the small extra part of
		# the bar needed to connect the parent to the rest.
		
		if {$kid_stackpos<$minkid_stackpos} { set minkid_stackpos $kid_stackpos}
		if {$kid_stackpos>$maxkid_stackpos} { set maxkid_stackpos $kid_stackpos}
		if {!$iscentered} {
	    	    set kid_stackpos [expr "$kid_stackpos [set stackop] ($vals($w,$kid_node_path,my_stack_usage)/2)"]
		}
		if {$kid_stackpos<$minkid_stackpos} { set minkid_stackpos $kid_stackpos}
		if {$kid_stackpos>$maxkid_stackpos} { set maxkid_stackpos $kid_stackpos}
		
		if $isvertical {
		    $w.canv create line $bar_depthpos $kid_stackpos $kid_depthpos $kid_stackpos -width 1 -tags decorations
		} else {
		    $w.canv create line $kid_stackpos $bar_depthpos $kid_stackpos $kid_depthpos -width 1 -tags decorations
		}
	    }
	    if $isvertical {
		$w.canv create line $bar_depthpos $minkid_stackpos $bar_depthpos $maxkid_stackpos -width 1 -tags decorations
	    } else {
		$w.canv create line $minkid_stackpos $bar_depthpos $maxkid_stackpos $bar_depthpos -width 1 -tags decorations
	    }
}



#------------------------------------------------------------
#
#
#------------------------------------------------------------

proc heirarchy::fileopen { dir } {
	puts "open dir $dir"
}

proc heirarchy::filelook { node_path showing_kids } {
	global heirarchy_library 
	set path [join $node_path /]
	set file [lindex $node_path [expr [llength $node_path]-1]]
	if [file readable $path] { 
	    if {[llength [heirarchy::dircontents $path]]!=0} {
	        if $showing_kids {
		    set bitmap @$heirarchy_library/folder_minus.xbm
		} else {
		    set bitmap @$heirarchy_library/folder_plus.xbm
		}
	    } else {
	        set bitmap @$heirarchy_library/folder.xbm
	    }
	    set textFill black
	    set bitmapColor black
	} else {
	    set textFill grey75
	    set bitmapColor grey75
	    set bitmap @$heirarchy_library/folder.xbm
	}
	set textFont "-Adobe-Helvetica-Bold-R-Normal--*-100-*"  
	return [list $file $textFill $textFont $bitmap $bitmapColor] 
}


proc heirarchy::dircontents { node_path } {
	set path [join $node_path /]
	
	# This would be the easiest way, but it uses features of find specific
	# to my version (GNU find version 3.8)
	
#        if [catch {set files [exec find $path/ -type d -printf "%f\\n" -mindepth 1 -maxdepth 1]}] { return "" }
#        if [catch {exec sh << "(cd $path; find . -type d \\! \\( -name . \\) -prune -print)"} files] { puts $files ; return "" }
#	return $files
        if [catch {exec ls $path} files] { return "" }
	set dirs ""
	foreach file $files {
	    if [file isdirectory $path/$file] { lappend dirs $file }
	}
	return $dirs
}


