########################################################################
#
# Copyright (c) 1994 John F. Sefler.
# All rights reserved.
#
########################################################################
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose, without fee, and without written
# agreement is hereby granted, provided the above copyright notice and
# the following two paragraphs appear in all copies of this software.
#
# IN NO EVENT SHALL JOHN F. SEFLER BE LIABLE TO ANY PARTY FOR DIRECT,
# INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
# OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF HE HAS
# BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# JOHN F. SEFLER SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING, BUT
# NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
# FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN
# "AS IS" BASIS, AND JOHN F. SEFLER HAS NO OBLIGATION TO PROVIDE
# MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
#
########################################################################

########################################################################
#
# MessageDialog Arguments:
#	parent  window name in which to center the dialog
#	dlg	window name for the toplevel dialog
#	title	window title decoration for window manager
#	msg	the message string to display
#	modal	should the dialog be modal? 1=yes 0=no
#	bitmap	an optional list containing a bitmap name, 
#		an optional foreground color, and
#		an optional background color
#	buttons a list of button labels
#	default which of the buttons should be bound to <Return>
#
# Return Value:
#	If the dialog is modal, then the button number pressed
#	is also returned.
#
# Usage Example:
#     MessageDialog . .info {Information Dialog} {This is an "information"
#               dialog box} 1 {info blue white} {"OK "Not OK"} 1
#
# Author: John Sefler
#
########################################################################
proc MessageDialog {parent dlg title msg modal bitmap buttons default} {

  global dialog_response

  # if the $dlg already exists, do nothing
  if [winfo exists $dlg] {
      wm deiconify $dlg
      raise $dlg
      return
  }

  # create a toplevel dialog, set its title
  toplevel    $dlg
  wm title    $dlg $title
  wm iconname $dlg $title
  wm withdraw $dlg
  if {$modal} {wm transient $dlg $parent}

  # create the frame widgets
  frame    $dlg.top    -borderwidth 1 -relief raised
  frame    $dlg.bottom -borderwidth 1 -relief raised
  pack     $dlg.top    -side top -fill both
  pack     $dlg.bottom -side bottom -fill both

  # create the bitmap (if supplied)
  set num_scan [scan $bitmap "%s %s %s" bitmap_file fg_color bg_color]
  if {$num_scan >= 1} {
    label $dlg.top.bitmap -bitmap $bitmap_file
    
    if {$num_scan >= 2} {
      $dlg.top.bitmap configure -foreground $fg_color
    }
    if {$num_scan >= 3} {
      $dlg.top.bitmap configure -background $bg_color \
				-relief ridge -borderwidth 2
    }

    pack $dlg.top.bitmap -side left -padx 5m -pady 5m
  }

  # create the message
  message $dlg.top.msg -text $msg -justify left -width 100m
  pack    $dlg.top.msg -side right -expand yes -fill both \
                       -pady 2m -padx 2m
  
  # determine a uniform width for the buttons
  set width 0
  foreach but $buttons {
    set butlen [string length $but]
    if {$butlen > $width} {set width $butlen}
  }

  # create the buttons
  set inx 1
  foreach but $buttons {

    if {$inx == $default} {
      frame  $dlg.default -relief sunken -borderwidth 1
      pack   $dlg.default -in $dlg.bottom -side left -expand yes \
                          -padx 3m -pady 2m

      button $dlg.but$inx -text $but -width $width -borderwidth 2
      pack   $dlg.but$inx -in $dlg.default -expand yes \
                          -padx 1m -pady 1m \
                          -ipadx 2m -ipady 1m
    } else {
      button $dlg.but$inx -text $but -width $width -borderwidth 2
      pack   $dlg.but$inx -in $dlg.bottom -side left -expand yes \
                          -padx 3m -pady 3m \
                          -ipadx 2m -ipady 1m
    }

    if {$modal} {
      $dlg.but$inx configure -command "set dialog_response $inx"
    } else {
      $dlg.but$inx configure -command "destroy $dlg"
    }

    incr inx
  }

  # set up a binding for <Return> (if there is a default button)
  if {$default > 0} {
    bind $dlg <Return> "set dialog_response $default"
  }

  # center the dialog to the parent
  CenterWindow $dlg $parent

  # If the dialog is modal, wait for the user to respond,
  # restore the focus, and return the index of the selected button
  if {$modal} {
    set old_focus [focus]
    update
    grab $dlg
    focus $dlg
    tkwait variable dialog_response
    destroy $dlg
    focus $old_focus
    return $dialog_response
  }
}

########################################################################
proc NotYetImplementedDialog {{parent .}} {
  set msg "Sorry!  This feature is not yet implemented."

  MessageDialog $parent .nyi {Warning Dialog} $msg 1 \
                             {warning red white} OK 1
}

########################################################################
proc QuitDialog {{parent .}} {

  # are you sure?
  set title   "Quit Dialog"
  set msg     "Are you sure?"
  set icon    "questhead red white"
  set buttons "Yes No"

  if {[MessageDialog $parent .quit_dlg $title $msg 1 $icon $buttons 1] == 1} {
    exit
  }
}

########################################################################
#
# ScrollDialog Arguments:
#	parent  window name in which to center the dialog
#	dlg	window name for the toplevel dialog
#	title	window title decoration for window manager
#	msg	the message string to display in the scroll
#	modal	should the dialog be modal?
#	bitmap	an optional list containing a bitmap name, 
#		an optional foreground color, and
#		an optional background color
#	buttons a list of button labels
#	default which of the buttons should be bound to <Return>
#
# Return Value:
#	If the dialog is modal, then the button number pressed
#	is also returned.
#
# Usage Example:
#     ScrollDialog . .dir "Directions Dialog" "Here are the Directions."
#               0 {info blue white} {Done} 0
#
# Author: John Sefler
#
########################################################################
proc ScrollDialog {parent dlg title msg modal bitmap buttons default} {

  global scroll_dialog_response

  # if the $dlg already exists, do nothing
  if [winfo exists $dlg] {
      wm deiconify $dlg
      raise $dlg
      return
  }

  # create a toplevel dialog, set its title
  toplevel    $dlg
  wm title    $dlg $title
  wm iconname $dlg $title
  wm withdraw $dlg
  wm minsize  $dlg 300 100
  if {$modal} {wm transient $dlg $parent}

  # create the frame widgets
  frame    $dlg.top    -borderwidth 1 -relief raised
  frame    $dlg.bottom -borderwidth 1 -relief raised
  pack     $dlg.top    -side top -fill both -expand yes
  pack     $dlg.bottom -side bottom -fill x

  # create the bitmap (if supplied)
  set num_scan [scan $bitmap "%s %s %s" bitmap_file fg_color bg_color]
  if {$num_scan >= 1} {
    label $dlg.top.bitmap -bitmap $bitmap_file
    
    if {$num_scan >= 2} {
      $dlg.top.bitmap configure -foreground $fg_color
    }
    if {$num_scan >= 3} {
      $dlg.top.bitmap configure -background $bg_color \
				-relief ridge -borderwidth 2
    }

    pack $dlg.top.bitmap -side left -padx 5m -pady 5m
  }

  # create the text message and scroll bar
  text  $dlg.text -wrap word -relief sunken -borderwidth 2 \
                       -width 50 -height 10\
		       -yscrollcommand "$dlg.scrl set" 
        $dlg.text insert current $msg 
        $dlg.text configure -state disabled
  scrollbar $dlg.scrl -borderwidth 2 -command "$dlg.text yview"
  pack $dlg.scrl -in $dlg.top -side right -fill y
  pack $dlg.text -in $dlg.top -side right -fill both -expand yes \
                     -padx 2m -pady 2m
  
  # determine a uniform width for the buttons
  set width 0
  foreach but $buttons {
    set butlen [string length $but]
    if {$butlen > $width} {set width $butlen}
  }

  # create the buttons
  set inx 1
  foreach but $buttons {

    if {$inx == $default} {
      frame  $dlg.default -relief sunken -borderwidth 1
      pack   $dlg.default -in $dlg.bottom -side left -expand yes \
                          -padx 3m -pady 2m

      button $dlg.but$inx -text $but -width $width -borderwidth 2
      pack   $dlg.but$inx -in $dlg.default -expand yes \
                          -padx 1m -pady 1m \
                          -ipadx 2m -ipady 1m
    } else {
      button $dlg.but$inx -text $but -width $width -borderwidth 2
      pack   $dlg.but$inx -in $dlg.bottom -side left -expand yes \
                          -padx 3m -pady 3m \
                          -ipadx 2m -ipady 1m
    }

    if {$modal} {
      $dlg.but$inx configure -command "set scroll_dialog_response $inx"
    } else {
      $dlg.but$inx configure -command "destroy $dlg"
    }

    incr inx
  }

  # set up a binding for <Return> (if there is a default button)
  if {$default > 0} {
    bind $dlg <Return> "set scroll_dialog_response $default"
  }

  # center the dialog to its parent
  CenterWindow $dlg $parent

  # If the dialog is modal, wait for the user to respond,
  # restore the focus, and return the index of the selected button
  if {$modal} {
    set old_focus [focus]
    update
    grab $dlg
    focus $dlg
    tkwait variable scroll_dialog_response
    destroy $dlg
    focus $old_focus
    return $dialog_response
  }
}

########################################################################
#
# EntryDialog Arguments:
#	parent  window name in which to center the dialog
#	dlg	window name for the toplevel dialog
#	title	window title decoration for window manager
#	msg	the message string to display
#	entry   a string to insert into the entry field
#	modal	should the dialog be modal? 1=yes 0=no
#	bitmap	an optional list containing a bitmap name, 
#		an optional foreground color, and
#		an optional background color
#	buttons a list of button labels
#	default which of the buttons should be bound to <Return>
#
# Return Value:
#	If the default button is pressed, then the entry string
#	is returned, otherwise the button index is returned
#
# Usage Example:
#       set lpr_cmd [EntryDialog . .printer_name "Printer" \
#                    "Printer command:" "lpr" 1 \
#                    {question red white} {Print Cancel} 1]
#
# Author: John Sefler
#
########################################################################
proc EntryDialog {parent dlg title msg entry modal bitmap buttons default} {

  global dialog_response

  # if the $dlg already exists, do nothing
  if [winfo exists $dlg] {
      wm deiconify $dlg
      raise $dlg
      return
  }

  # create a toplevel dialog, set its title
  toplevel    $dlg
  wm title    $dlg $title
  wm iconname $dlg $title
  wm withdraw $dlg
  if {$modal} {wm transient $dlg $parent}

  # create the frame widgets
  frame    $dlg.top    -borderwidth 1 -relief raised
  frame    $dlg.top.left  -borderwidth 0 -relief flat
  frame    $dlg.top.right -borderwidth 0 -relief flat
  frame    $dlg.bottom -borderwidth 1 -relief raised
  pack     $dlg.top    -side top -fill both
  pack     $dlg.top.left -side left 
  pack     $dlg.top.right -side right 
  pack     $dlg.bottom -side bottom -fill both

  # create the bitmap (if supplied)
  set num_scan [scan $bitmap "%s %s %s" bitmap_file fg_color bg_color]
  if {$num_scan >= 1} {
    label $dlg.top.left.bitmap -bitmap $bitmap_file
    
    if {$num_scan >= 2} {
      $dlg.top.left.bitmap configure -foreground $fg_color
    }
    if {$num_scan >= 3} {
      $dlg.top.left.bitmap configure -background $bg_color \
				-relief ridge -borderwidth 2
    }

    pack $dlg.top.left.bitmap -side left -padx 5m -pady 5m
  }

  # create the message
  message $dlg.top.right.msg -text $msg -justify left -width 100m
  pack    $dlg.top.right.msg -side top -anchor w \
                       -pady 2m -padx 2m

  # create the entry
  entry  $dlg.top.right.entry -relief sunken -borderwidth 2
         $dlg.top.right.entry insert 0 $entry
  pack   $dlg.top.right.entry -side bottom -fill x \
                       -pady 2m -padx 2m
  
  # determine a uniform width for the buttons
  set width 0
  foreach but $buttons {
    set butlen [string length $but]
    if {$butlen > $width} {set width $butlen}
  }

  # create the buttons
  set inx 1
  set dialog_response {}
  foreach but $buttons {

    if {$inx == $default} {
      frame  $dlg.default -relief sunken -borderwidth 1
      pack   $dlg.default -in $dlg.bottom -side left -expand yes \
                          -padx 3m -pady 2m

      button $dlg.but$inx -text $but -width $width -borderwidth 2
      pack   $dlg.but$inx -in $dlg.default -expand yes \
                          -padx 1m -pady 1m \
                          -ipadx 2m -ipady 1m
    } else {
      button $dlg.but$inx -text $but -width $width -borderwidth 2
      pack   $dlg.but$inx -in $dlg.bottom -side left -expand yes \
                          -padx 3m -pady 3m \
                          -ipadx 2m -ipady 1m
    }

    if {$modal} {
        $dlg.but$inx configure -command "set dialog_response $inx"
    } else {
      if {$inx == $default} {
        # I AM NOT SURE IF THIS WILL WORK
        $dlg.but$inx configure -command \
                     {set dialog_response "[$dlg.top.right.entry get]"; destroy $dlg; return $dialog_response}
      } else {
        $dlg.but$inx configure -command "destroy $dlg; return $inx"
      }
    }

     incr inx
   }

  # set up a binding for <Return> (if there is a default button)
  if {$default > 0} {
    bind $dlg <Return> {set dialog_response "[$dlg.top.right.entry get]"}
  }
  # set up a binding for <Return> on the entry widget
  bind $dlg.top.right.entry <Return> "$dlg.but$default invoke"

  # center the dialog to its parent
  CenterWindow $dlg $parent

  # If the dialog is modal, wait for the user to respond,
  # restore the focus, and return the index of the selected button
  if {$modal} {
    set old_focus [focus]
    update
    grab $dlg
    focus  $dlg.top.right.entry
    tkwait variable dialog_response
    if {$dialog_response == $default} {
      set dialog_response "[$dlg.top.right.entry get]"
    }
    destroy $dlg
    focus $old_focus
    return $dialog_response
  }
}


########################################################################
#
# Help Dialog Arguments:
#	parent  window name in which to center the dialog
#	dlg	window name for the toplevel dialog
#	title	window title decoration for window manager
#	tclfile name of a tcl file from which to load help
#               text and tcl commands for formatting the
#		appearance of the text.
#		The tclfile should refer to the text widget
#		as $w.  It can also add menu items to
#		a pulldown Topic pulldown menu called $m.
#		The intent of the Topic menu is for fast
#		topic indexing by scrolling automatically.
#	modal	should the dialog be modal?
#	bitmap	an optional list containing a bitmap name, 
#		an optional foreground color, and
#		an optional background color
#	buttons a list of button labels
#	default which of the buttons should be bound to <Return>
#
# Return Value:
#	If the dialog is modal, then the button number pressed
#	is also returned.
#
# Usage Example:
#     HelpDialog . .helpdialog "Help Dialog" help.tcl 0 \
#                {info blue white} {Done} 0
#
# Author: John Sefler
#
########################################################################
proc HelpDialog {parent dlg title tclfile modal bitmap buttons default} {

  global help_dialog_response

  # if the $dlg already exists, do nothing
  if [winfo exists $dlg] {
      wm deiconify $dlg
      raise $dlg
      return
  }

  # create a toplevel dialog, set its title
  toplevel    $dlg
  wm title    $dlg $title
  wm iconname $dlg $title
  wm withdraw $dlg
  wm minsize  $dlg 300 200
  if {$modal} {wm transient $dlg $parent}

  # menubar setup
  frame $dlg.menu -borderwidth 1 -relief raised
  pack  $dlg.menu -side top -anchor w -fill x
  menubutton $dlg.menu.topic -text "Topic" -menu $dlg.menu.topic.m 
  pack $dlg.menu.topic -side left

  menu $dlg.menu.topic.m

  # create the frame widgets
  frame    $dlg.top    -borderwidth 1 -relief raised
  frame    $dlg.bottom -borderwidth 1 -relief raised
  pack     $dlg.top    -side top -fill both -expand yes
  pack     $dlg.bottom -side bottom -fill x

  # create the bitmap (if supplied)
  set num_scan [scan $bitmap "%s %s %s" bitmap_file fg_color bg_color]
  if {$num_scan >= 1} {
    label $dlg.top.bitmap -bitmap $bitmap_file
    
    if {$num_scan >= 2} {
      $dlg.top.bitmap configure -foreground $fg_color
    }
    if {$num_scan >= 3} {
      $dlg.top.bitmap configure -background $bg_color \
				-relief ridge -borderwidth 2
    }

    pack $dlg.top.bitmap -side left -padx 5m -pady 5m
  }

  # create the text message and scroll bar
  text  $dlg.text -relief sunken -borderwidth 2 \
		       -yscrollcommand "$dlg.scrl set" 
  scrollbar $dlg.scrl -borderwidth 2 -command "$dlg.text yview"
  pack $dlg.scrl -in $dlg.top -side right -fill y
  pack $dlg.text -in $dlg.top -side right -fill both -expand yes \
                     -padx 2m -pady 2m
  
  # determine a uniform width for the buttons
  set width 0
  foreach but $buttons {
    set butlen [string length $but]
    if {$butlen > $width} {set width $butlen}
  }

  # create the buttons
  set inx 1
  foreach but $buttons {

    if {$inx == $default} {
      frame  $dlg.default -relief sunken -borderwidth 1
      pack   $dlg.default -in $dlg.bottom -side left -expand yes \
                          -padx 3m -pady 2m

      button $dlg.but$inx -text $but -width $width -borderwidth 2
      pack   $dlg.but$inx -in $dlg.default -expand yes \
                          -padx 1m -pady 1m \
                          -ipadx 2m -ipady 1m
    } else {
      button $dlg.but$inx -text $but -width $width -borderwidth 2
      pack   $dlg.but$inx -in $dlg.bottom -side left -expand yes \
                          -padx 3m -pady 3m \
                          -ipadx 2m -ipady 1m
    }

    if {$modal} {
      $dlg.but$inx configure -command "set help_dialog_response $inx"
    } else {
      $dlg.but$inx configure -command "destroy $dlg"
    }

    incr inx
  }

  # set up a binding for <Return> (if there is a default button)
  if {$default > 0} {
    bind $dlg <Return> "set help_dialog_response $default"
  }

  # is the tclfile present?
  if {[file exists $tclfile] == 0} {
    set title   "File Exists Dialog"
    set msg     "Cannot find help file \"$tclfile\"."
    set icon    "error red white"
    set buttons "OK"
    Dialog .error $title $msg 1 $icon $buttons 1
  } else {
  # source the tcl help file
  # note that $w in the source file refers to the text widget
  # note that $m in the source file refers to the topic menu
    set w $dlg.text
    set m $dlg.menu.topic.m
    source $tclfile
  }

  # center the dialog to its parent
  CenterWindow $dlg $parent

  # If the dialog is modal, wait for the user to respond,
  # restore the focus, and return the index of the selected button
  if {$modal} {
    set old_focus [focus]
    update
    grab $dlg
    focus $dlg
    tkwait variable help_dialog_response
    destroy $dlg
    focus $old_focus
    return $dialog_response
  }
}

########################################################################
#
# File Selection Dialog Arguments:
#	default_cmd	the procedure to which the file selection
#			string is returned
#	purpose		file entry label
#	default_name    default string to put into the file entry
#	parent  window name in which to center the dialog
#	dlg		name of the file selection dialog
#	oklabel		name to put in the bottom left default button
#	cancellabel	name to put in the bottom right button
#	errorHandler	procedure to call in case of an error in 
#                       file selection
#
# Return Value:
#	The selected file string
#
# Usage Example:
#     FileSelect PrintDrawing "Print Drawing To:" "file.ps"\
#                 . .printfile "Print" "Abort"
#
# Original Author:
# Mario Jorge Silva                            msilva@cs.Berkeley.EDU
# University of California Berkeley            Ph:    +1(510)642-8248
# Computer Science Division, 571 Evans Hall    Fax:   +1(510)642-5775
# Berkeley CA 94720                                 
#
# Modified By: John Sefler
#
#
# Copyright 1993, 1994 Regents of the University of California
# Permission to use, copy, modify, and distribute this
# software and its documentation for any purpose and without
# fee is hereby granted, provided that this copyright
# notice appears in all copies.  The University of California
# makes no representations about the suitability of this
# software for any purpose.  It is provided "as is" without
# express or implied warranty.
#
########################################################################


# names starting with "fileselect" are reserved by this module
# no other names used.


# this is the default proc  called when "OK" is pressed
# to indicate yours, give it as the first arg to "fileselect"

proc fileselect.default.cmd {f} {
  puts stderr "selected file $f"
}


proc fileselect.nullproc {{arg ""}} {
   # this proc intentionally left empty
}

# this is the default proc called when error is detected
# indicate your own proc as an argument to fileselect

proc fileselect.default.errorHandler {errorMessage} {
    puts stdout "error: $errorMessage"
    catch { cd ~ }
}

# this is a proc that returns a filename (or an empty string if cancelled)
# and offers a much simpler interface. It also preserves the cwdset before 
# invocation


proc FileSelectGet {
    {purpose "Open file:"} {defaultName ""}} {

    global fileselect
    set fileselect(pwd) [pwd]

    set w .fileSelect

    FileSelect fileselect.nullproc $purpose $defaultName $w \
               fileselect.default.errorHandler
     
    $fileselect(entry) delete 0 end
    $fileselect(entry) insert 0 $defaultName

    # Wait for button hits if no callbacks are defined
    # wait for the box to be destroyed
    update idletask
    tkwait window $w

    if {"$fileselect(result)" == "" } {
	cd $fileselect(pwd)
	return ""
    }

    set path [pwd]/$fileselect(result)

    cd $fileselect(pwd)
    return [string trimright [string trim $path] /]
}

# this is the proc that creates the file selector box

proc FileSelect {
    {cmd fileselect.default.cmd} 
    {purpose "Open file:"} 
    {defaultName ""} 
    {p .} 
    {w .fileSelect} 
    {oklabel "OK"} 
    {cancellabel "Cancel"} 
    {errorHandler fileselect.default.errorHandler}} {

    update
    catch {destroy $w}

    toplevel $w
    grab $w
    wm title $w "File Selection Dialog"
    wm widthdraw $w
    wm minsize $w 250 300

    global fileselect
    set fileselect(pwd) [pwd]

    # path independent names for the widgets
    set fileselect(entry) $w.eframe.entry
    set fileselect(list) $w.sframe.list
    set fileselect(scroll) $w.sframe.scroll
    set fileselect(ok) $w.okbut
    set fileselect(cancel) $w.cancelbut
    set fileselect(dirlabel) $w.dframe.dirlabel

    # widgets
    frame $w.topframepad -relief flat   -borderwidth 1
    frame $w.topframe    -relief flat   -borderwidth 10
    frame $w.botframe    -relief raised -borderwidth 1
    pack $w.botframe    -side bottom -fill x
    pack $w.topframepad -side top -fill both -expand yes 
    pack $w.topframe    -in $w.topframepad \
                        -side top -fill both -expand yes 

    frame $w.eframe
    frame $w.sframe
    frame $w.dframe

    pack $w.eframe -in $w.topframe -side top -fill x
    pack $w.sframe -in $w.topframe -side top -fill both -expand yes
    pack $w.dframe -in $w.topframe -side top -fill x



    label $w.eframe.label -text $purpose
    entry $w.eframe.entry -borderwidth 2 -relief sunken 
    $fileselect(entry) delete 0 end
    $fileselect(entry) insert 0 $defaultName

    pack $w.eframe.label -side top -anchor w
    pack $w.eframe.entry -side top -fill x


    scrollbar $w.sframe.yscroll -relief sunken -borderwidth 2\
	  -command "$w.sframe.list yview"
    listbox $w.sframe.list -relief sunken -borderwidth 2\
	  -yscroll "$w.sframe.yscroll set" 

    pack $w.sframe.yscroll -side right -fill y
    pack $w.sframe.list    -side left  -fill both -expand yes

    label $w.dframe.dir      -text "Current Directory:"
    label $w.dframe.dirlabel -text [pwd] 
    pack $w.dframe.dir      -side top -anchor w
    pack $w.dframe.dirlabel -side top -anchor w

    # buttons
    frame  $w.default -relief sunken -borderwidth 1
    pack   $w.default -in $w.botframe -side left -expand yes \
                        -padx 3m -pady 2m

    button $w.okbut -text $oklabel -borderwidth 2\
                    -command "fileselect.ok.cmd $w $cmd $errorHandler"
    pack   $w.okbut -in $w.default -expand yes \
                    -padx 1m -pady 1m \
                    -ipadx 2m -ipady 1m

    button $w.cancelbut -text $cancellabel -borderwidth 2\
                        -command "fileselect.cancel.cmd $w"
    pack   $w.cancelbut -in $w.botframe -side left -expand yes \
                        -padx 3m -pady 3m \
                        -ipadx 2m -ipady 1m

    # Fill the listbox with a list of the files in the directory 
 
    fileselect.getfiles

   # Set up bindings for the browser.
    bind $fileselect(entry) <Return> {eval $fileselect(ok) invoke}
    bind $fileselect(entry) <Control-c> {eval $fileselect(cancel) invoke}

    bind $w <Control-c> {eval $fileselect(cancel) invoke}
    bind $w <Return> {eval $fileselect(ok) invoke}


    tk_listboxSingleSelect $fileselect(list)


    bind $fileselect(list) <Button-1> {
        # puts stderr "button 1 release"
        %W select from [%W nearest %y]
	$fileselect(entry) delete 0 end
	$fileselect(entry) insert 0 [%W get [%W nearest %y]]
    }

    bind $fileselect(list) <Key> {
        %W select from [%W nearest %y]
        $fileselect(entry) delete 0 end
	$fileselect(entry) insert 0 [%W get [%W nearest %y]]
    }

    bind $fileselect(list) <Double-ButtonPress-1> {
        # puts stderr "double button 1"
        %W select from [%W nearest %y]
	$fileselect(entry) delete 0 end
	$fileselect(entry) insert 0 [%W get [%W nearest %y]]
	$fileselect(ok) invoke
    }

    bind $fileselect(list) <Return> {
        %W select from [%W nearest %y]
	$fileselect(entry) delete 0 end
	$fileselect(entry) insert 0 [%W get [%W nearest %y]]
	$fileselect(ok) invoke
    }

    # set kbd focus to entry widget
    focus $fileselect(entry)

  # center the dialog to its parent
  CenterWindow $dlg $parent

}


# auxiliary button procedures

proc fileselect.cancel.cmd {w} {
    # puts stderr "Cancel"

    global fileselect

    cd $fileselect(pwd)
    set fileselect(result) ""

    destroy $w
}

proc fileselect.ok.cmd {w cmd errorHandler} {
    global fileselect
    set selected [$fileselect(entry) get]

    # some nasty file names may cause "file isdirectory" to return an error
    if [catch {file isdirectory $selected} errorMessage] {
	$errorHandler $errorMessage
	destroy $w
	return
    }

    # clean the text entry and prepare the list
    $fileselect(entry) delete 0 end
    $fileselect(list) delete 0 end

    # perform globbing on the selection. 
    # If globing returns an error, destroy the browser and set the result

    if [catch {set globlist [glob [list $selected]]} errorMessage] {
	set fileselect(result) $selected
	destroy $w
	$cmd $selected
	return
    }

    # If resulting list length > 1, put the list on the file listbox and return
    # If globing expands to a list of filenames in multiple directories,
    # the indicated regexp is invalid and the error handler is called instead.

    if {[llength $globlist] > 1} {
	if {[regexp "/" $globlist] != 0} {
	    $errorHandler [list "Invalid regular expression, " $selected, "."]
	    destroy $w
	    return
	}
	fileselect.putfiles $globlist
	return
    }


    # selection may be a directory. Expand it.

    if {[file isdirectory $selected] != 0} {
	cd $selected
	set dir [pwd]
	$fileselect(dirlabel) configure -text $dir

	fileselect.getfiles
	return
    }

    # globlist contains only one entry. If selected is not equal to globlist,
    # the entry was specified with a wild card. We display the globlist entry
    # until it is specified by a unique selection.
   
    if {[string compare $selected $globlist] != 0} {
	fileselect.putfiles $globlist
	return
    }

    set fileselect(result) $selected
    destroy $w

    $cmd $selected
}

proc fileselect.getfiles {} {
    set files [lsort [concat [lrange [glob .*] 2 end] [glob -nocomplain *]]]

    # build list of the files: directories are
    # marked with a trailing "/"
    fileselect.putfiles $files
}

proc fileselect.putfiles {files} {
    global fileselect

    $fileselect(list) insert end "../"
    foreach i $files {
        if {[file isdirectory $i]} {
	    $fileselect(list) insert end $i/
	} else {
	    $fileselect(list) insert end $i
	}
    }
}


########################################################################
#
# KeyBindingDialog Arguments:
#	parent          window name in which to center the dialog
#	key_names_list  a list of strings to use as identifiers
#                       that descibe the function of the corresponding
#                       key binding in the global list of key_variables
#	key_variables_list  a list of variables containing the actual
#                       key bindings.  The list is assumed global.
#	ok_callback	procedure to call if ok has been pressed
#
# Return Value:
#	If the key bindings are accepted (user hit OK), then the global
#       variables in the key_variables_list are changed
#
# Usage Example:
#     KeyBindingDialog . $key_names_list $key_variables_list
#        where key_names_list = {{Move to the left} {Move to the right}}
#        where key_variables_list = {{key(Left)} {key(right}}
#        where key(left) = Left
#        where key(right) = Right
#
# Author: John Sefler
#
########################################################################
proc KeyBindingDialog {parent key_names_list key_variables_list {ok_callback NullProc}} {
  global dialog_response
# global $global_array_name
# puts $global_array_name
   
  # count how many keys
  set num_keys 0
  foreach key_variable $key_variables_list {incr num_keys}

  
  # If the window already exists, reset its key settings
  # and raise to the top
  set dlg .key_binding_dialog
  if [winfo exists $dlg] {
    for {set i 0} {$i < $num_keys} {incr i} {
      set key_variable [lindex $key_variables_list $i]
      upvar $key_variable key
      $dlg.f.f2.$i configure -text "$key"
    } 
    raise $dlg
    return
  }

  set prev_focus [focus]
  toplevel $dlg
  wm title $dlg "Key Bindings"
  wm withdraw $dlg
  wm transient $dlg $parent
   
  frame $dlg.f -relief raised -bd 1

  # create the left label frame and contents
  frame $dlg.f.f1 -relief flat -bd 1
  label $dlg.f.f1.title -relief flat -borderwidth 2 -anchor w \
                     -text "Key Functions:"
  pack $dlg.f.f1.title -fill x
  for {set i 0} {$i < $num_keys} {incr i} {
    label $dlg.f.f1.$i -relief flat -borderwidth 2 -anchor w \
                     -text "[expr $i + 1]. [lindex $key_names_list $i]"
    pack $dlg.f.f1.$i -fill x
  } 
  pack $dlg.f.f1 -side left -fill both -expand yes -padx 2m

  # create the right label frame and contents
  frame $dlg.f.f2 -relief flat -bd 1
  label $dlg.f.f2.title -relief flat -borderwidth 2 -anchor w \
                     -text "Keys:"
  pack $dlg.f.f2.title -fill x
  for {set i 0} {$i < $num_keys} {incr i} {
    set key_variable [lindex $key_variables_list $i]
    upvar #0 $key_variable key
    label $dlg.f.f2.$i -relief flat -bd 2 -text "$key"
    bind $dlg.f.f2.$i <Key> "%W configure -text %K; focus $dlg.f.f2.[expr $i + 1]"
    bind $dlg.f.f2.$i <FocusIn> "%W configure -relief sunken"
    bind $dlg.f.f2.$i <FocusOut> "%W configure -relief flat"
    bind $dlg.f.f2.$i <Button-1> "focus %W"
    pack $dlg.f.f2 $dlg.f.f2.$i -fill x
  }
  bind $dlg.f.f2.[expr $num_keys - 1] <Key> \
                               "%W configure -text %K; focus $dlg.f.f2.0"
  pack $dlg.f.f2 -side right -fill both -expand yes -padx 2m

  pack $dlg.f -fill both 
 
  # create the right label frame and contents
   frame $dlg.buttons -relief raised -bd 1
      # Ok button
      button $dlg.buttons.ok -text "Ok" \
			  -width 6 \
                          -relief raised -borderwidth 2 \
         -command "SetKeyBindingsFromDialog $dlg \"$key_variables_list\"; set dialog_response 1"
      pack $dlg.buttons.ok -side left -expand yes \
                          -padx 1m -pady 1m \
                          -ipadx 2m -ipady 1m
      # Reset button
      button $dlg.buttons.reset -text "Reset" \
			  -width 6 \
                          -relief raised -borderwidth 2 \
         -command "KeyBindingDialog $parent \"$key_names_list\" \"$key_variables_list\""
      pack $dlg.buttons.reset -side left -expand yes \
                          -padx 1m -pady 1m \
                          -ipadx 2m -ipady 1m 
      # Cancel button
      button $dlg.buttons.cancel -text "Cancel" \
			  -width 6 \
                          -relief raised -borderwidth 2 \
         -command "set dialog_response 0"
      pack $dlg.buttons.cancel -side left -expand yes \
                          -padx 1m -pady 1m \
                          -ipadx 2m -ipady 1m 
   pack $dlg.buttons -side bottom -fill x \
                          -ipadx 3m \
                          -ipady 2m

   CenterWindow $dlg .
   update
   grab $dlg
   focus $dlg.f.f2.0

   # wait until the user presses Ok or Cancel
   tkwait variable dialog_response

   if {$dialog_response} {$ok_callback}
   focus $prev_focus
   destroy $dlg
}

proc SetKeyBindingsFromDialog {dlg key_variables_list} {
  set i 0
  foreach key_variable $key_variables_list {
    set new_key [lindex [$dlg.f.f2.$i configure -text] 4]
    uplevel #0 set $key_variable $new_key
    incr i
  }
}

proc NullProc { } {
  # This is a dummy proceedure to call when you want to do nothing.
}





########################################################################
proc CenterWindow {w {p ""}} {
   update idletasks
   if {$p == ""} {
      set p [focus]
   }
     
   # Determine the toplevel window

   if {$p == "none"} {
      wm deiconify $w
      return
   }
   set p [winfo toplevel $p]
   if [winfo ismapped $w] {
      wm withdraw $w
   }
   update idletasks
   set winX [expr {(([winfo width  $p]-[winfo reqwidth  $w])/2)+[winfo x $p]}]
   if {$winX < 0} {set winX 0}
   set winY [expr {(([winfo height $p]-[winfo reqheight $w])/2)+[winfo y $p]}]
   if {$winY < 0} {set winY 0}
   wm geometry $w +${winX}+${winY}
   wm deiconify $w
}

########################################################################
proc ShowBusyCursor {{widget .} {update 1}} {
    SetWidgetsCursorTo $widget watch
    if $update {update idletasks}
}

########################################################################
proc RemoveBusyCursor {{widget .} {update 1}} {
    SetWidgetsPreviousCursor $widget
    if $update {update idletasks}
}

########################################################################
proc SetWidgetsCursorTo {widget cursor} {
   global prev_cursor

   set prev_cursor($widget) [lindex [$widget configure -cursor] 4]
#  puts "prev_cursor($widget) = $prev_cursor($widget)"

   # avoids unnessary resizing of the main window
   if {$widget != "."} {
     $widget configure -cursor $cursor
   }

   # set all the children's cursor
   foreach child [winfo children $widget] {
     SetWidgetsCursorTo $child $cursor
   }
}

########################################################################
proc SetWidgetsPreviousCursor {{widget .}} {
   global prev_cursor

   # avoids unnessary resizing of the main window
   if {$widget != "."} {
     $widget configure -cursor $prev_cursor($widget)
   }

   # set all the children's cursor
   foreach child [winfo children $widget] {
     SetWidgetsPreviousCursor $child
   }
}
