/* 
 * tclFrame.c --
 * 
 *	This file contains routines that provide for searching
 *	and manipulation of user created call frames.
 */

#include "tclInt.h"
     
 
/*
 *----------------------------------------------------------------------
 *
 * Tcl_GetFrameTable --
 *
 *	Return the table of named frames for the specified interpreter,
 *	creating one if necessary.  This routine exists to avoid adding
 *	a "userFrameTable" field to the Interp structure.
 *
 * Results:
 *	returns interpreter's named var frame table.
 *
 * Side effects:
 *	First call allocates table of frame tables.  Allocates new
 *	varfreme table if none exists for given interpreter.  Caches
 *	last frame table returned.
 *
 *----------------------------------------------------------------------
 */
   
Tcl_HashTable *userFrameTables 	= (Tcl_HashTable *) 0;
Tcl_Interp    *lastInterp 	= (Tcl_Interp *) 0;
Tcl_HashTable *lastFrameTable 	= (Tcl_HashTable *) 0;

Tcl_HashTable *
Tcl_GetFrameTable(interp)
    Tcl_Interp *interp;
{
    Tcl_HashEntry *hPtr;
    Tcl_HashTable *newFrameTable;
    int new;
    
    /* initialize frame tables if necessary */
    if (! userFrameTables) {
	userFrameTables = (Tcl_HashTable *) ckalloc(sizeof(Tcl_HashTable));
	Tcl_InitHashTable(userFrameTables, TCL_ONE_WORD_KEYS);
    }
    
    /* one bucket cache to reduce lookups */
    if (interp == lastInterp)
	return lastFrameTable;
    
    /* get/create new frame table */
    hPtr = Tcl_CreateHashEntry(userFrameTables, interp, &new);
    if (new) {
	newFrameTable = (Tcl_HashTable *) ckalloc(sizeof(Tcl_HashTable));
	Tcl_InitHashTable(newFrameTable, TCL_STRING_KEYS);
	Tcl_SetHashValue(hPtr, newFrameTable);
    } else {
	newFrameTable = (Tcl_HashTable *) Tcl_GetHashValue(hPtr);
    }
    
    /* cache new frame table */
    lastFrameTable = newFrameTable;
    lastInterp     = interp;
    
    return newFrameTable;
}
    
 
/*
 *----------------------------------------------------------------------
 *
 * TclCreateNamedFrame --
 *
 *	Create a new named variable frame and install it in
 *	in the named frame table for the interpreter.  
 *
 * Results:
 *	The return value is -1 if an error occurred in creating the
 *	frame (in this case an error message is left in interp->result).
 *	1 is returned if string was preceded by "#" and it specified a
 *	valid frame name.  The variable pointed to by framePtrPtr is
 *	filled in with the address of the desired frame
 *	(unless an error occurs, in which case it isn't modified).
 *
 * Side effects:
 *	Allocates new named frame and update's interp's userFrameTable
 *	if successful.
 *
 *----------------------------------------------------------------------
 */

int
TclCreateNamedFrame(interp, string, framePtrPtr)
    Tcl_Interp *interp;		/* Interpreter in which to find frame. */
    char *string;		/* String describing frame. */
    CallFrame **framePtrPtr;	/* Store pointer to frame here (or NULL
				 * if not desired) */
{
    CallFrame* framePtr;
    Tcl_HashEntry *hPtr;
    Tcl_HashTable *frameTable;
    int new;
    
    /* 
     * Get the frame table for this interp
     */
    frameTable = Tcl_GetFrameTable(interp);
    
    /* 
     * Check if named frame already exists in this interpreter
     */
    hPtr = Tcl_CreateHashEntry(frameTable, string, &new);    
    if (! new) {
	Tcl_AppendResult(interp, "named frame already exists: ", string, (char *) NULL);
	return TCL_ERROR;
    }
    
    /*
     * Set up and install a new named call frame.  Most of
     * the variables are never used, but perhaps they
     * could be set to more meaningful values.
     */
    framePtr = (CallFrame *) ckalloc(sizeof(CallFrame));
    Tcl_InitHashTable(&framePtr->varTable, TCL_STRING_KEYS);
    framePtr->level	   = 0;
    framePtr->argc 	   = 0;
    framePtr->argv 	   = NULL;
    framePtr->callerPtr    = NULL;
    framePtr->callerVarPtr = NULL;
    Tcl_SetHashValue(hPtr, framePtr);
}


 
/*
 *----------------------------------------------------------------------
 *
 * TclDeleteNamedFrame --
 *
 *	Deletes a named frame from the interpreter's 
 *	named frame table.
 *
 * Results:
 *	Uses TclDeleteVars to cleanup the contents of
 *	the frame and removes the named frame from the
 *	interpreter's named frame table.  There are possible
 *	problems here, as this does not attempt to handle
 *	the case in which other frames contain upvars which
 *	refer to this frame.
 *
 * Side effects:
 *	Hash entries are deleted and storage is deallocated.
 *
 *----------------------------------------------------------------------
 */

int
TclDeleteNamedFrame(interp, string)
    Tcl_Interp *interp;		/* Interpreter in which to find frame. */
    char *string;		/* String describing frame. */
{
    CallFrame* framePtr;
    Tcl_HashTable *frameTable;
    Tcl_HashEntry *hPtr;
    
    frameTable = Tcl_GetFrameTable(interp);
    hPtr = Tcl_FindHashEntry(frameTable, string);    
    if (! hPtr) {
	Tcl_AppendResult(interp, "no such frame: ", string, (char *) NULL);
	return TCL_ERROR;
    }

    /* cleanup the frame and the hash entry */
    framePtr = (CallFrame *) Tcl_GetHashValue(hPtr);
    TclDeleteVars(interp, framePtr);    
    ckfree((char*) framePtr);
    Tcl_DeleteHashEntry(hPtr);
    
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * TclFindNamedFrame --
 *
 *	Given a named frame expression, locate the frame
 *	in the named frame table.  
 *
 * Results:
 *	The return value is -1 if an error occurred in finding the
 *	frame (in this case an error message is left in interp->result).
 *	1 is returned if string was preceded by "#" and it specified a
 *	valid frame.  The variable pointed to by framePtrPtr is
 *	filled in with the address of the desired frame
 *	(unless an error occurs, in which case it isn't modified).
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
TclFindNamedFrame(interp, string, framePtrPtr)
    Tcl_Interp *interp;		/* Interpreter in which to find frame. */
    char *string;		/* String describing frame. */
    CallFrame **framePtrPtr;	/* Store pointer to frame here (or NULL
				 * if not desired) */
{
    register Interp *iPtr = (Interp *) interp;
    CallFrame *framePtr;
    Tcl_HashTable *frameTable;
    Tcl_HashEntry *hPtr;
    
    /*
     *	Handle the case of #?name in which we search the call frame
     *  stack for the first frame containing the named variable.
     *  We begin the search at the current call frame.
     */
    if (*string == '?') {
	string++;
	for (framePtr = iPtr->varFramePtr; framePtr != NULL;
	     framePtr = framePtr->callerVarPtr) {
	    hPtr = Tcl_FindHashEntry(&framePtr->varTable, string);
	    if (hPtr != NULL) {
		*framePtrPtr = framePtr;
		return 1;
	    }
	}    
	if (framePtr == NULL) {
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp, "can't find frame containing ", string, (char *) NULL);	    
	    return -1;
	}
    }
    
    /* 
     *  Name does not begin with a leading '?', so we presume its
     *  a user defined variable frame.
     */
    frameTable = Tcl_GetFrameTable(interp);
    hPtr = Tcl_FindHashEntry(frameTable, string);
    if (hPtr == NULL) {
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "can't find frame named ", string, (char *) NULL);	    
	return -1;
    }
    
    framePtr = (CallFrame *) Tcl_GetHashValue(hPtr);
    if (framePtrPtr) 
	*framePtrPtr = framePtr;
    return 1;
}


/*
 *----------------------------------------------------------------------
 *
 * Tcl_VarframeGetCmd --
 *
 *	This procedure is invoked to process the "varframe get" Tcl command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result value.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_VarframeGetCmd(interp, frame, argc, argv)    
    register Tcl_Interp *interp;	/* Current interpreter. */
    char *frame;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    register Interp *iPtr = (Interp *) interp;
    CallFrame *savedVarFramePtr, *framePtr;
    char *result;
    
    if (TclGetFrame(interp, frame, &framePtr) == -1)
	return TCL_ERROR;
    
    /*
     * Modify the interpreter state to execute in the given frame
     * and collect all the values of the listed variables.
     */
    
    savedVarFramePtr = iPtr->varFramePtr;
    iPtr->varFramePtr = framePtr;

    while (argc > 0) {
	result = Tcl_GetVar(interp, argv[0], TCL_LEAVE_ERR_MSG);
	if (result == NULL) {
	    iPtr->varFramePtr = savedVarFramePtr;
	    return TCL_ERROR;
	}
	Tcl_AppendElement(interp, result, 0);
	argc--;
	argv++;
    }

    iPtr->varFramePtr = savedVarFramePtr;
    return TCL_OK;    
}


/*
 *----------------------------------------------------------------------
 *
 * Tcl_VarframeSetCmd --
 *
 *	This procedure is invoked to process the "varframe set" Tcl command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result value.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_VarframeSetCmd(interp, frame, argc, argv)    
    register Tcl_Interp *interp;	/* Current interpreter. */
    char *frame;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    register Interp *iPtr = (Interp *) interp;
    CallFrame *savedVarFramePtr, *framePtr;
    char *result;
    
    if (TclGetFrame(interp, frame, &framePtr) == -1)
	return TCL_ERROR;
    
    /*
     * Modify the interpreter state to execute in the given frame
     * and set the variables to their specified values.
     */
    
    savedVarFramePtr = iPtr->varFramePtr;
    iPtr->varFramePtr = framePtr;

    while (argc > 0) {
	result = Tcl_SetVar(interp, argv[0], argv[1], TCL_LEAVE_ERR_MSG);
	if (result == NULL) {
	    iPtr->varFramePtr = savedVarFramePtr;
	    return TCL_ERROR;
	}
	argc -= 2;
	argv += 2;
    }

    iPtr->varFramePtr = savedVarFramePtr;
    return TCL_OK;    
}


/*
 *----------------------------------------------------------------------
 *
 * Tcl_VarframeCmd --
 *
 *	This procedure is invoked to process the "varframe" Tcl command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result value.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Tcl_VarframeCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    register Tcl_Interp *interp;	/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int length;
    char c;

    Tcl_HashTable *frameTable;
    Tcl_HashTable *varTablePtr;
    Tcl_HashSearch s;
    CallFrame* framePtr;
    Tcl_HashEntry *hPtr;
    Tcl_HashEntry *hPtr2;
    char *name;

    if (argc < 2) {
    usage:
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " option ?arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Dispatch based on the option.
     */
    c = argv[1][0];
    length = strlen(argv[1]);
    
    if ((c == 'c') && (strncmp(argv[1], "create", length) == 0)) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "usage: varframe create frame", (char *) NULL);
	    return TCL_ERROR;
	}
	TclCreateNamedFrame(interp, argv[2]+1);
	return TCL_OK;
	
    } else if ((c == 'd') && (strncmp(argv[1], "delete", length) == 0)) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "usage: varframe delete frame", (char *) NULL);
	    return TCL_ERROR;
	}
	TclDeleteNamedFrame(interp, argv[2]+1);
	return TCL_OK;
	
    } else if ((c == 'e') && (strncmp(argv[1], "exists", length) == 0)) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "usage: varframe exists frame", (char *) NULL);
	    return TCL_ERROR;
	}	
	
	if (TclGetFrame(interp, argv[2], &framePtr) == -1) {
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp, "0", (char *) NULL);
	} else {
	    Tcl_AppendResult(interp, "1", (char *) NULL);
	}
	return TCL_OK;
	
    } else if ((c == 'f') && (strncmp(argv[1], "find", length) == 0)) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "usage: varframe find var", (char *) NULL);
	    return TCL_ERROR;
	}
	
	frameTable = Tcl_GetFrameTable(interp);
	for (hPtr = Tcl_FirstHashEntry(frameTable, &s); hPtr != NULL;
	     hPtr = Tcl_NextHashEntry(&s)) {
	    
	    framePtr = (CallFrame *) Tcl_GetHashValue(hPtr);
	    varTablePtr = &framePtr->varTable;
	    hPtr2    = Tcl_FindHashEntry(varTablePtr, argv[2]);
	    
	    if (hPtr2 != NULL) {
		char *name = (char*) Tcl_GetHashKey(frameTable, hPtr);
		Tcl_AppendResult(interp, name, " ", (char *) NULL);
	    }
	}
	return TCL_OK;
	
    } else if ((c == 'g') && (strncmp(argv[1], "get", length) == 0)) {
	if (argc < 4) {
	    Tcl_AppendResult(interp, "usage: varframe get frame var ?var ...?", (char *) NULL);
	    return TCL_ERROR;
	}
	return Tcl_VarframeGetCmd(interp, argv[2], argc-3, argv+3);
	
    } else if ((c == 'l') && (strncmp(argv[1], "list", length) == 0)) {
	if (argc != 2) {
	    Tcl_AppendResult(interp, "usage: varframe list", (char *) NULL);
	    return TCL_ERROR;
	}
	
	frameTable = Tcl_GetFrameTable(interp);
	for (hPtr = Tcl_FirstHashEntry(frameTable, &s); hPtr != NULL;
	     hPtr = Tcl_NextHashEntry(&s)) {
	    
	    name = (char*) Tcl_GetHashKey(frameTable, hPtr);
	    Tcl_AppendResult(interp, name, " ", (char *) NULL);
	}
	return TCL_OK;

    } else if ((c == 's') && (strncmp(argv[1], "set", length) == 0)) {
	if (argc < 4 || (argc&1) != 1) {
	    Tcl_AppendResult(interp, "usage: varframe set frame var value ?var value ...?", (char *) NULL);
	    return TCL_ERROR;
	}
	return Tcl_VarframeSetCmd(interp, argv[2], argc-3, argv+3);
		
    } else if ((c == 'v') && (strncmp(argv[1], "vars", length) == 0)) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "usage: varframe vars frame", (char *) NULL);
	    return TCL_ERROR;
	}
	
	if (TclGetFrame(interp, argv[2], &framePtr) == -1)
	    return TCL_ERROR;
	
	if (framePtr == NULL) {
	    varTablePtr = &((Interp *)interp)->globalTable;
	} else {
	    varTablePtr = &framePtr->varTable;
	}
	for (hPtr = Tcl_FirstHashEntry(varTablePtr, &s); hPtr != NULL;
	     hPtr = Tcl_NextHashEntry(&s)) {
	    name = (char*) Tcl_GetHashKey(varTablePtr, hPtr);
	    Tcl_AppendResult(interp, name, " ", (char *) NULL);
	}
	return TCL_OK;
		
    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[1],
		"\": should be create, delete, exists, find, get, list, set or vars", (char *) NULL);
	return TCL_ERROR;
    }
}
